// File          : PaperDialogBox.java
// Description   : Dialog box to specify paper size
// Date          : 1st February 2006
// Last Modified : 9th June 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog box in which to specify paper size.
 * @author Nicola L C Talbot
 */
public class PaperDialogBox extends JDialog
   implements ActionListener
{
   public PaperDialogBox(JpgfDraw application, JDRPaper paper)
   {
      super(application, JDRResources.getString("paper.title"),
            true);
      application_ = application;

      JPanel panel = new JPanel();
      panel.setLayout(new BoxLayout(panel, BoxLayout.PAGE_AXIS));
      ButtonGroup group = new ButtonGroup();

      JPanel predefinedPanel = new JPanel(
         new FlowLayout(FlowLayout.LEFT));
      predefinedPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
      panel.add(predefinedPanel);
      predefined = new JRadioButton(
         JDRResources.getString("paper.predefined"));
      predefined.setMnemonic(
         JDRResources.getChar("paper.predefined.mnemonic"));
      predefined.addActionListener(this);
      predefined.setActionCommand("predefined");
      group.add(predefined);
      predefinedPanel.add(predefined);

      //predefinedBox = new JComboBox<String>(new String[]// Java7
      predefinedBox = new JComboBox(new String[]
      {
         JDRResources.getString("paper.a6"),
         JDRResources.getString("paper.a7"),
         JDRResources.getString("paper.a8"),
         JDRResources.getString("paper.a9"),
         JDRResources.getString("paper.a10"),
         JDRResources.getString("paper.b0"),
         JDRResources.getString("paper.b1"),
         JDRResources.getString("paper.b2"),
         JDRResources.getString("paper.b3"),
         JDRResources.getString("paper.b4"),
         JDRResources.getString("paper.b5"),
         JDRResources.getString("paper.b6"),
         JDRResources.getString("paper.b7"),
         JDRResources.getString("paper.b8"),
         JDRResources.getString("paper.b9"),
         JDRResources.getString("paper.b10"),
         JDRResources.getString("paper.c0"),
         JDRResources.getString("paper.c1"),
         JDRResources.getString("paper.c2"),
         JDRResources.getString("paper.c3"),
         JDRResources.getString("paper.c4"),
         JDRResources.getString("paper.c5"),
         JDRResources.getString("paper.c6"),
         JDRResources.getString("paper.c7"),
         JDRResources.getString("paper.c8"),
         JDRResources.getString("paper.c9"),
         JDRResources.getString("paper.c10")
      });
      predefinedPanel.add(predefinedBox);

      JPanel p1 = new JPanel(new FlowLayout(FlowLayout.LEFT));
      p1.setAlignmentX(Component.LEFT_ALIGNMENT);
      panel.add(p1);

      user = new JRadioButton(JDRResources.getString("paper.user"));
      user.setMnemonic(JDRResources.getChar("paper.user.mnemonic"));
      user.addActionListener(this);
      user.setActionCommand("user");
      group.add(user);
      p1.add(user);

      widthPanel = new NonNegativeLengthPanel(
         JDRResources.getString("paper.width"),
         JDRResources.getChar("paper.width.mnemonic"));
      widthPanel.getTextField().requestFocusInWindow();
      p1.add(widthPanel);

      heightPanel = new NonNegativeLengthPanel(
         JDRResources.getString("paper.height"),
         JDRResources.getChar("paper.height.mnemonic"));
      p1.add(heightPanel);

      getContentPane().add(panel, "Center");

      JPanel p2 = new JPanel();

      p2.add(JDRResources.createOkayButton(this));
      p2.add(JDRResources.createCancelButton(this));
      p2.add(JDRResources.createHelpButton("sec:paper"));

      getContentPane().add(p2, "South");
      pack();
      setLocationRelativeTo(application_);
      setPaper(paper);
   }

   public void initialise(boolean portrait)
   {
      mainPanel = application_.getCurrentFrame();
      portrait_ = portrait;
      setPaper(mainPanel.getPaper());
      setVisible(true);
   }

   public void setPaper(JDRPaper paper)
   {
      widthPanel.setValue(paper.getWidth());
      heightPanel.setValue(paper.getHeight());
      portrait_ = paper.isPortrait();
      int id = portrait_ ? paper.getID()-JDRPaper.ID_A6
                         : paper.getID()-JDRPaper.ID_A6R;

      if (id >= 0)
      {
         predefined.setSelected(true);
         predefinedBox.setSelectedIndex(id);
         widthPanel.setEnabled(false);
         heightPanel.setEnabled(false);
         predefinedBox.setEnabled(true);
      }
      else
      {
         user.setSelected(true);
         widthPanel.setEnabled(true);
         heightPanel.setEnabled(true);
         predefinedBox.setEnabled(false);
      }
   }

   public JDRPaper getPaper()
   {
      if (predefined.isSelected())
      {
         int id = predefinedBox.getSelectedIndex();
         JDRPaper paper = JDRPaper.getPredefinedPaper(
            portrait_ ? id+JDRPaper.ID_A6 : id+JDRPaper.ID_A6R);

         if (paper != null)
         {
            return paper;
         }
      }

      return new JDRPaper(widthPanel.getValue(),
                          heightPanel.getValue());
   }

   public void okay()
   {
      mainPanel.setPaperSize(getPaper());
      setVisible(false);
   }

   public void cancel()
   {
      JDRPaper oldPaper = mainPanel.getPaper();
      setVisible(false);
      application_.setPaperSize(oldPaper);
   }

   public void actionPerformed(ActionEvent e)
   {
      String action = e.getActionCommand();

      if (action == null) return;

      if (action.equals("okay"))
      {
         okay();
      } 
      else if (action.equals("cancel"))
      {
         cancel();
      }
      else if (action.equals("predefined"))
      {
         widthPanel.setEnabled(false);
         heightPanel.setEnabled(false);
         predefinedBox.setEnabled(true);
      }
      else if (action.equals("user"))
      {
         widthPanel.setEnabled(true);
         heightPanel.setEnabled(true);
         predefinedBox.setEnabled(false);
      }
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "";

      str += "PaperDialogBox:"+eol;
      str += "has focus: "+hasFocus()+eol;
      str += "widthPanel: "+widthPanel.info()+eol;
      str += "heightPanel: "+heightPanel.info()+eol;
      str += "predefined button has focus: "+predefined.hasFocus()+eol;
      str += "user button has focus: "+user.hasFocus()+eol;
      str += "predefined box has focus: "+predefinedBox.hasFocus()+eol;

      return str+eol;
   }

   private JpgfDraw application_;
   private NonNegativeLengthPanel widthPanel, heightPanel;
   private JRadioButton predefined, user;
   //private JComboBox<String> predefinedBox;// Java7
   private JComboBox predefinedBox;
   private static final int A6=0, A7=1, A8=2, A9=3, A10=4,
     B0=5, B1=6, B2=7, B3=8, B4=9, B5=10, C0=11, C1=12, C2=13,
     C3=14, C4=15, C5=16, C6=17, C7=18, C8=19, C9=20, C10=21;
   private boolean portrait_;

   private JDRFrame mainPanel = null;
}
