// File          : PaintPanel.java
// Description   : Panel for selecting paint
// Date          : 6th February 2006
// Last Modified : 13th August 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;

/**
 * Panel for selecting paint.
 * @author Nicola L C Talbot
 */

public class PaintPanel extends JPanel 
   implements AdjustmentListener,ActionListener
{
   public PaintPanel(JDRSelector selector)
   {
      super();

      selector_    = selector;

      setLayout(new BoxLayout(this, BoxLayout.X_AXIS));

      Box buttonPanel = Box.createVerticalBox();
      buttonPanel.setAlignmentY(Component.TOP_ALIGNMENT);

      cards = new JPanel(new CardLayout());
      cards.setAlignmentY(Component.TOP_ALIGNMENT);

      cards.setBorder(BorderFactory.createLoweredBevelBorder());

      ButtonGroup buttonGroup = new ButtonGroup();

      // Transparent
      transparentButton = new JRadioButton(
         JDRResources.getString("colour.transparent"));
      transparentButton.setMnemonic(
         JDRResources.getChar("colour.transparent.mnemonic"));

      transparentButton.addActionListener(this);
      transparentButton.setAlignmentX(Component.LEFT_ALIGNMENT);

      buttonGroup.add(transparentButton);

      buttonPanel.add(transparentButton);

      nonePanel = new JPanel();
      nonePanel.setAlignmentY(Component.TOP_ALIGNMENT);

      cards.add(nonePanel, NONEPANEL);

      // Single colour

      singleColourButton =
         new JRadioButton(JDRResources.getString("colour.single"));
      singleColourButton.setMnemonic(
         JDRResources.getChar("colour.single.mnemonic"));

      singleColourButton.addActionListener(this);
      singleColourButton.setAlignmentX(Component.LEFT_ALIGNMENT);

      buttonGroup.add(singleColourButton);
      buttonPanel.add(singleColourButton);

      singleColourPanel = new ColorPanel(this);
      singleColourPanel.setAlignmentY(Component.TOP_ALIGNMENT);
      singleColourPanel.setMnemonics(
         JDRResources.getChar("colour.rgb.mnemonic"),
         JDRResources.getChar("colour.cmyk.mnemonic"));

      JPanel singleColourCard = new JPanel();
      singleColourCard.setLayout(new BorderLayout());
      singleColourCard.add(singleColourPanel, BorderLayout.NORTH);

      cards.add(singleColourCard, COLOURPANEL);

      // Gradient Color

      gradientColourButton = new JRadioButton(
         JDRResources.getString("colour.gradient"));
      gradientColourButton.setMnemonic(
         JDRResources.getChar("colour.gradient.mnemonic"));

      gradientColourButton.addActionListener(this);
      gradientColourButton.setAlignmentX(Component.LEFT_ALIGNMENT);

      buttonGroup.add(gradientColourButton);
      buttonPanel.add(gradientColourButton);

      gradientPanel = new GradientPanel(selector);
      gradientPanel.setAlignmentY(Component.TOP_ALIGNMENT);

      cards.add(gradientPanel, GRADIENTPANEL);

      Dimension dim1 = singleColourPanel.getPreferredSize();
      Dimension dim2 = gradientPanel.getPreferredSize();

      dim2.height -= dim1.height;

      singleColourCard.add(Box.createRigidArea(dim2));

      add(buttonPanel);
      add(Box.createHorizontalStrut(10));

      add(new JScrollPane(cards));
   }

   public JDRPaint getPaint()
   {
      JDRPaint paint;

      if (singleColourButton.isSelected())
      {
         paint = singleColourPanel.getPaint();
      }
      else if (gradientColourButton.isSelected())
      {
         paint = gradientPanel.getPaint();
      }
      else
      {
         paint = new JDRTransparent();
      }

      return paint;
   }

   public void setPaint(JDRPaint paint)
   {
      if (paint instanceof JDRTransparent)
      {
         transparentButton.setSelected(true);
         selectTransparent();
      }
      else if (paint instanceof JDRGradient)
      {
         gradientColourButton.setSelected(true);
         selectGradientColour();
         gradientPanel.setPaint(paint);

      }
      else if (paint instanceof JDRRadial)
      {
         gradientColourButton.setSelected(true);
         selectGradientColour();
         gradientPanel.setPaint(paint);
      }
      else
      {
         singleColourPanel.setPaint(paint);
         selectSingleColour();
      }

      selector_.repaintSample();
   }

   public void selectTransparent()
   {
      if (!transparentButton.isSelected())
      {
         transparentButton.setSelected(true);
      }
      CardLayout layout = (CardLayout)cards.getLayout();
      layout.show(cards, NONEPANEL);
   }

   public void selectSingleColour()
   {
      if (!singleColourButton.isSelected())
      {
         singleColourButton.setSelected(true);
      }
      CardLayout layout = (CardLayout)cards.getLayout();
      layout.show(cards, COLOURPANEL);
   }

   public void selectGradientColour()
   {
      if (!gradientColourButton.isSelected())
      {
         gradientColourButton.setSelected(true);
      }
      CardLayout layout = (CardLayout)cards.getLayout();
      layout.show(cards, GRADIENTPANEL);
   }

   public void actionPerformed(ActionEvent e)
   {
      Object source = e.getSource();

      if (source == transparentButton)
      {
         selectTransparent();
      }
      else if (source == singleColourButton)
      {
         selectSingleColour();
      }
      else if (source == gradientColourButton)
      {
         selectGradientColour();
      }

      selector_.repaintSample();
   }

   public void adjustmentValueChanged(AdjustmentEvent evt)
   {
      selector_.repaintSample();
   }

   private JRadioButton transparentButton;
   private JRadioButton singleColourButton;
   private JRadioButton gradientColourButton;

   private ColorPanel singleColourPanel;
   private GradientPanel gradientPanel;
   private JPanel nonePanel;

   private JPanel cards;

   private static final String NONEPANEL="None Panel";
   private static final String COLOURPANEL="Colour Panel";
   private static final String GRADIENTPANEL="Gradient Panel";

//   private JButton selectSingleColourButton;
//   private SingleColourDialog singleColourDialog;
//   private ColorPanel startGradientPanel, endGradientPanel;

//   private JPanel singleColourSwatch, gradientSwatch;

//   private GradientDialog gradientDialog;
//   private JButton selectGradientButton;
/*
   private JLabel startLabel, endLabel;

   private LinearGradientDirectionPanel linearDirectionPanel;
   private RadialGradientDirectionPanel radialDirectionPanel;

   private JRadioButton linearButton, radialButton;
*/

   private JDRSelector selector_;
}
