// File          : MarkerPanel.java
// Description   : Panel for selecting marker type
// Date          : 6th February 2006
// Last Modified : 21st July 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.awt.image.*;
import java.beans.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

/**
 * Panel for selecting marker type.
 * @see ArrowStylePanel
 * @author Nicola L C Talbot
 */

public class MarkerPanel extends JPanel
   implements ActionListener,ListSelectionListener,
              ChangeListener
{
   public MarkerPanel(JDRSelector selector, ArrowStylePanel panel)
   {
      super();
      selector_ = selector;
      arrowStylePanel = panel;

      JPanel markerPanel = new JPanel();
      markerPanel.setLayout(new BoxLayout(markerPanel,
         BoxLayout.Y_AXIS));
      markerPanel.setAlignmentX(Component.LEFT_ALIGNMENT);

      ButtonGroup markerGroup = new ButtonGroup();

      // no marker

      noMarkerButton = new JRadioButton(
         JDRResources.getString("arrow.nomarker")+" ", true);
      noMarkerButton.setMnemonic(
         JDRResources.getChar("arrow.nomarker.mnemonic"));
      noMarkerButton.addActionListener(this);
      noMarkerButton.setAlignmentX(Component.LEFT_ALIGNMENT);
      markerGroup.add(noMarkerButton);

      markerPanel.add(noMarkerButton);

      // use a marker

      useMarkerButton = new JRadioButton(
         JDRResources.getString("arrow.usemarker")+" ");
      useMarkerButton.setMnemonic(
         JDRResources.getChar("arrow.usemarker.mnemonic"));
      useMarkerButton.setAlignmentX(Component.LEFT_ALIGNMENT);
      useMarkerButton.addActionListener(this);
      markerGroup.add(useMarkerButton);

      markerPanel.add(useMarkerButton);

      // marker tab pane

      markerTabPane = new JTabbedPane();
      markerTabPane.setAlignmentX(Component.LEFT_ALIGNMENT);
      markerTabPane.addChangeListener(this);

      markerTabPane.setToolTipText(
         JDRResources.getString("tooltip.arrow.type"));

      // arrow style markers

      //arrowStyleBox = new JList<MarkerItem>(arrowStyles);// Java7
      arrowStyleBox = new JList(arrowStyles);
      arrowStyleBox.setSelectionMode(
         ListSelectionModel.SINGLE_INTERVAL_SELECTION);
      arrowStyleBox.addListSelectionListener(this);
      arrowStyleBox.setCellRenderer(new MarkerItem());

      JScrollPane arrowSp = new JScrollPane(arrowStyleBox);

      markerTabPane.addTab(
         JDRResources.getString("arrow.tab.arrows")+" ", null, arrowSp,
         JDRResources.getString("tooltip.arrow.type.arrows"));

      // partial arrows

      //partialStyleBox = new JList<MarkerItem>(partialStyles);//Java7
      partialStyleBox = new JList(partialStyles);
      partialStyleBox.setSelectionMode(
         ListSelectionModel.SINGLE_INTERVAL_SELECTION);
      partialStyleBox.addListSelectionListener(this);
      partialStyleBox.setCellRenderer(new MarkerItem());

      JScrollPane partialSp = new JScrollPane(partialStyleBox);

      markerTabPane.addTab(
         JDRResources.getString("arrow.tab.partial")+" ", null, partialSp,
         JDRResources.getString("tooltip.arrow.type.partial"));

      // data point styles

      //dataStyleBox = new JList<MarkerItem>(dataStyles); // Java7
      dataStyleBox = new JList(dataStyles);
      dataStyleBox.setSelectionMode(
         ListSelectionModel.SINGLE_INTERVAL_SELECTION);
      dataStyleBox.addListSelectionListener(this);
      dataStyleBox.setCellRenderer(new MarkerItem());

      JScrollPane dataSp = new JScrollPane(dataStyleBox);

      markerTabPane.addTab(
         JDRResources.getString("arrow.tab.data")+" ", null, dataSp,
         JDRResources.getString("tooltip.arrow.type.data"));

      // bracket style markers

      //bracketStyleBox = new JList<MarkerItem>(bracketStyles); //Java7
      bracketStyleBox = new JList(bracketStyles);
      bracketStyleBox.setSelectionMode(
         ListSelectionModel.SINGLE_INTERVAL_SELECTION);
      bracketStyleBox.addListSelectionListener(this);
      bracketStyleBox.setCellRenderer(new MarkerItem());

      JScrollPane bracketSp = new JScrollPane(bracketStyleBox);

      markerTabPane.addTab(
         JDRResources.getString("arrow.tab.bracket")+" ", null, bracketSp,
         JDRResources.getString("tooltip.arrow.type.bracket"));

      // circle/diamond style markers

      //shapesStyleBox = new JList<MarkerItem>(shapesStyles);//Java7
      shapesStyleBox = new JList(shapesStyles);
      shapesStyleBox.setSelectionMode(
         ListSelectionModel.SINGLE_INTERVAL_SELECTION);
      shapesStyleBox.addListSelectionListener(this);
      shapesStyleBox.setCellRenderer(new MarkerItem());

      JScrollPane shapesSp = new JScrollPane(shapesStyleBox);

      markerTabPane.addTab(
         JDRResources.getString("arrow.tab.decorative")+" ", null, shapesSp,
         JDRResources.getString("tooltip.arrow.type.decorative"));

      // cap style markers

      //capStyleBox = new JList<MarkerItem>(capStyles);//Java7
      capStyleBox = new JList(capStyles);
      capStyleBox.setSelectionMode(
         ListSelectionModel.SINGLE_INTERVAL_SELECTION);
      capStyleBox.addListSelectionListener(this);
      capStyleBox.setCellRenderer(new MarkerItem());

      JScrollPane capSp = new JScrollPane(capStyleBox);

      markerTabPane.addTab(
         JDRResources.getString("arrow.tab.caps")+" ", null, capSp,
         JDRResources.getString("tooltip.arrow.type.caps"));

      markerPanel.add(markerTabPane);

      add(markerPanel, "Left");

      JPanel settingsPanel = new JPanel();
      settingsPanel.setLayout(new BoxLayout(settingsPanel,
         BoxLayout.Y_AXIS));

      JPanel repeatPanel = new JPanel(
         new FlowLayout(FlowLayout.LEADING));
      settingsPanel.add(repeatPanel);

      arrowSize = new NonNegativeLengthPanel(
         JDRResources.getString("size.label")+" ",
         JDRResources.getChar("size.mnemonic"),
         selector_.getSamplePathPanel());
      arrowSize.setEnabled(false);
      arrowSize.setValue(5.0f);
      repeatPanel.add(arrowSize);

      ButtonGroup repeatGroup = new ButtonGroup();

      // single arrow
      arrowSingle = new JRadioButton(
         JDRResources.getString("arrow.single")+" ",true);
      arrowSingle.setMnemonic(
         JDRResources.getChar("arrow.single.mnemonic"));
      arrowSingle.addActionListener(this);
      arrowSingle.setEnabled(false);
      repeatPanel.add(arrowSingle);
      repeatGroup.add(arrowSingle);

      // double arrow
      arrowDouble =
         new JRadioButton(JDRResources.getString("arrow.double")+" ");
      arrowDouble.setMnemonic(
         JDRResources.getChar("arrow.double.mnemonic"));
      arrowDouble.addActionListener(this);
      arrowDouble.setEnabled(false);
      repeatPanel.add(arrowDouble);
      repeatGroup.add(arrowDouble);

      // triple arrow
      arrowTriple =
         new JRadioButton(JDRResources.getString("arrow.triple")+" ");
      arrowTriple.setMnemonic(
         JDRResources.getChar("arrow.triple.mnemonic"));
      arrowTriple.addActionListener(this);
      arrowTriple.setEnabled(false);
      repeatPanel.add(arrowTriple);
      repeatGroup.add(arrowTriple);

      // reversed arrow
      arrowReverse =
         new JCheckBox(JDRResources.getString("arrow.reversed")+" ");
      arrowReverse.setMnemonic(
         JDRResources.getChar("arrow.reversed.mnemonic"));
      arrowReverse.addActionListener(this);
      arrowReverse.setEnabled(false);
      repeatPanel.add(arrowReverse);

      // orientation

      JPanel orientPanel = new JPanel();
      orientPanel.setLayout(new FlowLayout(FlowLayout.LEADING));
      settingsPanel.add(orientPanel);

      autoOrient = new JCheckBox(
         JDRResources.getString("arrow.autoorient")+" ",true);
      autoOrient.setMnemonic(
         JDRResources.getChar("arrow.autoorient.mnemonic"));
      autoOrient.setToolTipText(
         JDRResources.getString("tooltip.arrow.autoorient"));
      orientPanel.add(autoOrient);
      autoOrient.addActionListener(this);

      angleField = new DoubleField(0.0f);
      orientPanel.add(angleField);
      angleField.getDocument().addDocumentListener(
         new TextFieldSampleListener(selector_.getSamplePathPanel()));

      angleField.setToolTipText(
         JDRResources.getString("tooltip.arrow.angle"));

      autoOrient.setEnabled(false);
      angleField.setEnabled(false);

      // user offset

      JPanel autoOffsetPanel = new JPanel();
      autoOffsetPanel.setLayout(new FlowLayout(FlowLayout.LEADING));
      settingsPanel.add(autoOffsetPanel);

      autoOffset = new JCheckBox(
         JDRResources.getString("arrow.autooffset")+" ", true);
      autoOffset.setMnemonic(
         JDRResources.getChar("arrow.autooffset.mnemonic"));
      autoOffset.setToolTipText(
         JDRResources.getString("tooltip.arrow.autooffset"));
      autoOffsetPanel.add(autoOffset);
      autoOffset.addActionListener(this);

      offsetField = new LengthPanel(selector_.getSamplePathPanel());
      autoOffsetPanel.add(offsetField);

      offsetField.setToolTipText(
         JDRResources.getString("tooltip.arrow.offset"));

      autoOffset.setEnabled(false);
      offsetField.setEnabled(false);

      // repeat gap

      JPanel gapPanel = new JPanel();
      gapPanel.setLayout(new FlowLayout(FlowLayout.LEADING));
      settingsPanel.add(gapPanel);

      autoGap = new JCheckBox(
         JDRResources.getString("arrow.autogap")+" ", true);
      autoGap.setMnemonic(
         JDRResources.getChar("arrow.autogap.mnemonic"));
      autoGap.setToolTipText(
         JDRResources.getString("tooltip.arrow.autogap"));
      gapPanel.add(autoGap);
      autoGap.addActionListener(this);

      gapField = new LengthPanel(selector_.getSamplePathPanel());
      gapPanel.add(gapField);

      gapField.setToolTipText(
         JDRResources.getString("tooltip.arrow.gap"));

      autoGap.setEnabled(false);
      gapField.setEnabled(false);

      // Colour options

      JPanel colourOptionsPanel =
         new JPanel(new FlowLayout(FlowLayout.LEADING));
      settingsPanel.add(colourOptionsPanel);

      ButtonGroup colourGroup = new ButtonGroup();

      asLineButton = new JRadioButton(
         JDRResources.getString("arrow.colour.dependent")+" ",true);
      asLineButton.addActionListener(this);
      asLineButton.setToolTipText(
         JDRResources.getString("tooltip.arrow.colour.dependent"));

      //asLineButton.setAlignmentX(Component.LEFT_ALIGNMENT);
      colourOptionsPanel.add(asLineButton);
      colourGroup.add(asLineButton);

      notAsLineButton = new JRadioButton(
         JDRResources.getString("arrow.colour.independent")+" ");
      notAsLineButton.addActionListener(this);
      notAsLineButton.setToolTipText(
         JDRResources.getString("tooltip.arrow.colour.independent"));

      //notAsLineButton.setAlignmentX(Component.LEFT_ALIGNMENT);
      colourOptionsPanel.add(notAsLineButton);
      colourGroup.add(notAsLineButton);

      colourPanel = new ColorPanel();
      colourPanel.setMnemonics(
         JDRResources.getChar("colour.rgb.mnemonic"),
         JDRResources.getChar("colour.cmyk.mnemonic"));

      //colourPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
      settingsPanel.add(colourPanel);

      add(settingsPanel, "Center");

      setEnabled(true);
   }

   public void stateChanged(ChangeEvent e)
   {
      enableMarkerSettings();
      selector_.repaintSample();
   }

   public void addAdjustmentListener(AdjustmentListener al)
   {
      colourPanel.addAdjustmentListener(al);
   }

   public void addListSelectionListener(ListSelectionListener lis)
   {
      arrowStyleBox.addListSelectionListener(lis);
      partialStyleBox.addListSelectionListener(lis);
      dataStyleBox.addListSelectionListener(lis);
      bracketStyleBox.addListSelectionListener(lis);
      shapesStyleBox.addListSelectionListener(lis);
      capStyleBox.addListSelectionListener(lis);
   }

   public void adjustmentValueChanged(AdjustmentEvent evt)
   {
      selector_.repaintSample();
   }

   public void actionPerformed(ActionEvent e)
   {
      Object source = e.getSource();

      if (source == autoOrient)
      {
         enableAngleBoxes();

         if (!autoOrient.isSelected())
         {
            angleField.requestFocusInWindow();
         }
      }
      else if (source == autoOffset)
      {
         enableOffsetBoxes();

         if (!autoOffset.isSelected())
         {
            offsetField.requestFocusInWindow();
         }
      }
      else if (source == autoGap)
      {
         enableGapBoxes();

         if (!autoGap.isSelected())
         {
            gapField.requestFocusInWindow();
         }
      }
      else if (source == arrowSingle)
      {
         autoGap.setEnabled(false);
         enableGapBoxes();
      }
      else if (source == arrowDouble || source == arrowTriple)
      {
         int type = getArrowStyle();

         boolean noMarker = (type == JDRMarker.ARROW_NONE);

         autoGap.setEnabled(!noMarker);
         enableGapBoxes();
      }
      else if (source == asLineButton)
      {
         colourPanel.setEnabled(false);
      }
      else if (source == notAsLineButton)
      {
         colourPanel.setEnabled(true);
      }
      else if (source == noMarkerButton)
      {
         enableMarkerBoxes(false);
         enableMarkerSettings();
      }
      else if (source == useMarkerButton)
      {
         enableMarkerBoxes(true);
         enableMarkerSettings();

         getSelectedList().requestFocusInWindow();
      }

      selector_.repaintSample();
   }

   public void valueChanged(ListSelectionEvent evt)
   {
      enableMarkerSettings();
      selector_.repaintSample();
   }

   public void enableMarkerSettings()
   {
      MarkerItem item = getSelectedMarker();

      boolean noMarker = (item.getType() == JDRMarker.ARROW_NONE);
      boolean isResizable = item.isResizable();

      arrowStylePanel.updatePanel();
      enableMarkerBoxes();

      if (arrowSize == null) return;

      arrowSize.setEnabled(isResizable);

      arrowSingle.setEnabled(!noMarker);

      arrowDouble.setEnabled(!noMarker);

      arrowTriple.setEnabled(!noMarker);

      arrowReverse.setEnabled(!noMarker);

      autoOrient.setEnabled(!noMarker);
      enableAngleBoxes();

      autoOffset.setEnabled(!noMarker);
      enableOffsetBoxes();

      autoGap.setEnabled(!noMarker && !arrowSingle.isSelected());
      enableGapBoxes();

      asLineButton.setEnabled(!noMarker);
      notAsLineButton.setEnabled(!noMarker);

      colourPanel.setEnabled(
         !noMarker && notAsLineButton.isSelected());

   }

   private void colourButtonsSetEnabled(boolean flag)
   {
      asLineButton.setEnabled(flag);
      notAsLineButton.setEnabled(flag);
      colourPanel.setEnabled(flag && notAsLineButton.isSelected());
   }

   public void setEnabled(boolean flag)
   {
      MarkerItem marker = getSelectedMarker();

      int type =
         (marker == null ? JDRMarker.ARROW_NONE : marker.getType());

      boolean noMarker = (type == JDRMarker.ARROW_NONE);

      boolean isResizable =
         (marker == null ? false : marker.isResizable());

      useMarkerButton.setEnabled(flag);
      noMarkerButton.setEnabled(flag);
      enableMarkerBoxes(flag);

      arrowSize.setEnabled(flag && isResizable);
      arrowSingle.setEnabled(flag && !noMarker);
      arrowDouble.setEnabled(flag && !noMarker);
      arrowTriple.setEnabled(flag && !noMarker);
      arrowReverse.setEnabled(flag && !noMarker);
      autoOrient.setEnabled(flag && !noMarker);
      enableAngleBoxes();

      asLineButton.setEnabled(!noMarker);
      notAsLineButton.setEnabled(!noMarker);
      colourPanel.setEnabled(
         !noMarker && notAsLineButton.isSelected());
   }

   private void enableMarkerBoxes()
   {
      enableMarkerBoxes(true);
   }

   private void enableMarkerBoxes(boolean flag)
   {
      JScrollPane sp 
         = (JScrollPane)markerTabPane.getSelectedComponent();

      if (!flag)
      {
         sp.getViewport().getView().setEnabled(flag);
         sp.setEnabled(flag);
         markerTabPane.setEnabled(flag);
      }

      boolean selected = useMarkerButton.isSelected();

      //JList<MarkerItem> list = getSelectedList(sp);//Java7
      JList list = getSelectedList(sp);
      list.setEnabled(selected);
      sp.setEnabled(selected);
      markerTabPane.setEnabled(selected);

      if (selected && list.getSelectedIndex()==-1)
      {
         list.setSelectedIndex(0);
      }
   }

   private void enableAngleBoxes()
   {
      if (!autoOrient.isEnabled())
      {
         angleField.setEnabled(false);
         return;
      }

      boolean orient = autoOrient.isSelected();

      angleField.setEnabled(!orient);
   }

   private void enableOffsetBoxes()
   {
      if (!autoOffset.isEnabled())
      {
         offsetField.setEnabled(false);
         return;
      }

      boolean selected = autoOffset.isSelected();

      offsetField.setEnabled(!selected);
   }

   private void enableGapBoxes()
   {
      if (!autoGap.isEnabled())
      {
         gapField.setEnabled(false);
         return;
      }

      boolean selected = autoGap.isSelected();

      gapField.setEnabled(!selected);
   }

   public void setArrowColour(JDRPaint paint)
   {
      if (paint == null)
      {
         asLineButton.setSelected(true);
         colourPanel.setPaint(selector_.getLinePaint());
         colourPanel.setEnabled(false);
      }
      else
      {
         notAsLineButton.setSelected(true);
         colourPanel.setPaint(paint);
         colourPanel.setEnabled(true);
      }
   }

   public JDRPaint getArrowColour()
   {
      if (asLineButton.isSelected())
      {
         return null;
      }
      else
      {
         return colourPanel.getPaint();
      }
   }

   @SuppressWarnings("unchecked")
   //public JList<MarkerItem> getSelectedList(JScrollPane sp)//Java7
   public JList getSelectedList(JScrollPane sp)
   {
      //return (JList<MarkerItem>)sp.getViewport().getView();//Java7
      return (JList)sp.getViewport().getView();
   }


   //public JList<MarkerItem> getSelectedList()//Java7
   public JList getSelectedList()
   {
      return getSelectedList((JScrollPane)markerTabPane.getSelectedComponent());
   }

   public MarkerItem getSelectedMarker()
   {
      if (noMarkerButton.isSelected())
      {
         return noMarkerItem;
      }

      //JList<MarkerItem> list = getSelectedList();//Java7
      JList list = getSelectedList();

      MarkerItem item = (MarkerItem)list.getSelectedValue();

      if (item == null)
      {
         list.setSelectedIndex(0);
         //item = list.getSelectedValue();// Java7
         item = (MarkerItem)list.getSelectedValue();
      }

      return item;
   }

   public int getArrowStyle()
   {
      MarkerItem markerItem = getSelectedMarker();

      if (markerItem == null) return JDRMarker.ARROW_NONE;

      return markerItem.getType();
   }

   public void setArrowStyle(int style)
   {
      if (style == JDRMarker.ARROW_NONE)
      {
         noMarkerButton.setSelected(true);
         enableMarkerSettings();
         return;
      }

      useMarkerButton.setSelected(true);
      enableMarkerSettings();

      for (int i = 0; i < arrowStyles.length; i++)
      {
         MarkerItem item = arrowStyles[i];

         if (item.getType() == style)
         {
            arrowStyleBox.setSelectedIndex(i);
            markerTabPane.setSelectedIndex(ARROW_TAB);

            return;
         }
      }

      for (int i = 0; i < partialStyles.length; i++)
      {
         MarkerItem item = partialStyles[i];

         if (item.getType() == style)
         {
            partialStyleBox.setSelectedIndex(i);
            markerTabPane.setSelectedIndex(PARTIAL_TAB);

            return;
         }
      }

      for (int i = 0; i < dataStyles.length; i++)
      {
         MarkerItem item = dataStyles[i];

         if (item.getType() == style)
         {
            dataStyleBox.setSelectedIndex(i);
            markerTabPane.setSelectedIndex(DATA_TAB);

            return;
         }
      }

      for (int i = 0; i < bracketStyles.length; i++)
      {
         MarkerItem item = bracketStyles[i];

         if (item.getType() == style)
         {
            bracketStyleBox.setSelectedIndex(i);
            markerTabPane.setSelectedIndex(BRACKET_TAB);

            return;
         }
      }

      for (int i = 0; i < shapesStyles.length; i++)
      {
         MarkerItem item = shapesStyles[i];

         if (item.getType() == style)
         {
            shapesStyleBox.setSelectedIndex(i);
            markerTabPane.setSelectedIndex(SHAPES_TAB);

            return;
         }
      }

      for (int i = 0; i < capStyles.length; i++)
      {
         MarkerItem item = capStyles[i];

         if (item.getType() == style)
         {
            capStyleBox.setSelectedIndex(i);
            markerTabPane.setSelectedIndex(CAP_TAB);

            return;
         }
      }
   }

   public double getArrowSize()
   {
      return arrowSize.getValue();
   }

   public int getArrowRepeated()
   {
      if (arrowDouble.isSelected()) return 2;
      if (arrowTriple.isSelected()) return 3;

      return 1;
   }

   public void setArrowRepeated(int repeated)
   {
      if (repeated==3)
      {
         arrowTriple.setSelected(true);
      }
      else if (repeated==2)
      {
         arrowDouble.setSelected(true);
      }
      else
      {
         arrowSingle.setSelected(true);
      }
   }

   public boolean getArrowReverse()
   {
      return arrowReverse.isSelected();
   }

   public void setArrowReverse(boolean isReverseArrow)
   {
      arrowReverse.setSelected(isReverseArrow);
   }

   public void setArrowSize(double width)
   {
      arrowSize.setValue(width);
   }

   public JDRMarker getMarker()
   {
      JDRMarker marker=null;

      try
      {
         marker = JDRMarker.getPredefinedMarker(
         getArrowStyle(),1.0f,
         getArrowRepeated(),
         getArrowReverse(),
         getArrowSize());
         marker.fillPaint = getArrowColour();

         if (!autoOrient.isSelected())
         {
            marker.setOrient(false,
               Math.toRadians(angleField.getValue()));
         }

         if (!autoOffset.isSelected())
         {
            marker.enableUserOffset(true);
            marker.setOffset(offsetField.getValue());
         }

         if (!autoGap.isSelected())
         {
            marker.enableUserRepeatOffset(true);
            marker.setRepeatOffset(gapField.getValue());
         }
      }
      catch (InvalidMarkerTypeException e)
      {
         JDRResources.internalError(selector_, e);
      }
      catch (InvalidRepeatValueException e)
      {
         JDRResources.internalError(selector_, e);
      }

      return marker;
   }

   public void setMarker(JDRMarker marker)
   {
      boolean noMarker = (marker.getType() == JDRMarker.ARROW_NONE);

      setArrowStyle(marker.getType());
      setArrowRepeated(marker.getRepeated());
      setArrowReverse(marker.isReversed());
      if (marker.isResizable()) setArrowSize(marker.getSize());
      setArrowColour(marker.fillPaint);

      arrowSize.setEnabled(marker.isResizable());
      arrowSingle.setEnabled(!noMarker);
      arrowDouble.setEnabled(!noMarker);
      arrowTriple.setEnabled(!noMarker);
      arrowReverse.setEnabled(!noMarker);

      boolean orient = marker.getAutoOrient();

      autoOrient.setSelected(orient);
      autoOrient.setEnabled(!noMarker);
      enableAngleBoxes();

      if (!orient)
      {
         angleField.setValue(Math.toDegrees(marker.getAngle()));
      }

      boolean offsetEnabled = marker.isUserOffsetEnabled();
      autoOffset.setSelected(!offsetEnabled);
      autoOffset.setEnabled(!noMarker);
      enableOffsetBoxes();

      if (offsetEnabled)
      {
         offsetField.setValue(marker.getOffset());
      }

      boolean repeatEnabled = marker.isUserRepeatOffsetEnabled();
      autoGap.setSelected(!repeatEnabled);
      autoGap.setEnabled(!noMarker && !arrowSingle.isSelected());
      enableGapBoxes();
 
      if (repeatEnabled)
      {
         gapField.setValue(marker.getRepeatOffset());
      }
   }

   public String getDescription()
   {
      MarkerItem item = getSelectedMarker();

      String str = item.getLabel();

      if (item.getType() == JDRMarker.ARROW_NONE) return str;

      if (item.isResizable())
      {
         str += " "+getArrowSize();
      }

      if (getArrowRepeated()==2)
      {
         str += " "+JDRResources.getString("arrow.double");
      }
      else if (getArrowRepeated()==3)
      {
         str += " "+JDRResources.getString("arrow.triple");
      }


      if (getArrowReverse())
      {
         str += " "+JDRResources.getString("arrow.reversed");
      }

      return str;
   }

   private JDRSelector selector_;

   // styles
   //private JList<MarkerItem> arrowStyleBox, partialStyleBox, dataStyleBox,//Java7
   private JList arrowStyleBox, partialStyleBox, dataStyleBox,
      bracketStyleBox, shapesStyleBox, capStyleBox;

   private static final int ARROW_TAB=0, PARTIAL_TAB=1, DATA_TAB=2,
      BRACKET_TAB=3, SHAPES_TAB=4, CAP_TAB=5;

   private JTabbedPane markerTabPane;

   private JRadioButton noMarkerButton, useMarkerButton;

   // arrow size
   private NonNegativeLengthPanel arrowSize;

   // arrow single/double/triple?
   private JRadioButton arrowSingle, arrowDouble, arrowTriple;

   // arrow reversed?
   private JCheckBox arrowReverse;

   // auto orientation ?
   private JCheckBox autoOrient;
   private DoubleField angleField;

   // auto offset?
   private JCheckBox autoOffset;
   private LengthPanel offsetField;

   // auto repeat gap?
   private JCheckBox autoGap;
   private LengthPanel gapField;

   // does the arrow have a colour independent to the line colour?

   private JRadioButton asLineButton, notAsLineButton;

   private ColorPanel colourPanel;

   public static final MarkerItem noMarkerItem = 
      new MarkerItem(JDRMarker.ARROW_NONE,
         JDRResources.getString("arrow.none"));

   public static final MarkerItem[] arrowStyles = 
   {
      new MarkerItem(JDRMarker.ARROW_POINTED,
         JDRResources.getString("arrow.pointed")),
      new MarkerItem(JDRMarker.ARROW_POINTED60,
         JDRResources.getString("arrow.pointed60")),
      new MarkerItem(JDRMarker.ARROW_POINTED45,
         JDRResources.getString("arrow.pointed45")),
      new MarkerItem(JDRMarker.ARROW_CUSP,
         JDRResources.getString("arrow.cusp")),
      new MarkerItem(JDRMarker.ARROW_SINGLE,
         JDRResources.getString("arrow.latex")),
      new MarkerItem(JDRMarker.ARROW_ALT_SINGLE,
         JDRResources.getString("arrow.altlatex")),
      new MarkerItem(JDRMarker.ARROW_ALT_SINGLE_OPEN,
         JDRResources.getString("arrow.altlatex.open")),
      new MarkerItem(JDRMarker.ARROW_TRIANGLE,
         JDRResources.getString("arrow.triangle")),
      new MarkerItem(JDRMarker.ARROW_TRIANGLE_OPEN,
         JDRResources.getString("arrow.triangle.open")),
      new MarkerItem(JDRMarker.ARROW_EQUILATERAL_FILLED,
         JDRResources.getString("arrow.equilateral.filled")),
      new MarkerItem(JDRMarker.ARROW_EQUILATERAL_OPEN,
         JDRResources.getString("arrow.equilateral.open")),
      new MarkerItem(JDRMarker.ARROW_HOOKS,
         JDRResources.getString("arrow.hooks"))
   };

   public static final MarkerItem[] partialStyles =
   {
      new MarkerItem(JDRMarker.ARROW_HOOK_UP,
         JDRResources.getString("arrow.hook.up")),
      new MarkerItem(JDRMarker.ARROW_HOOK_DOWN,
         JDRResources.getString("arrow.hook.down")),
      new MarkerItem(JDRMarker.ARROW_HALF_POINTED_UP,
         JDRResources.getString("arrow.halfpointed.up")),
      new MarkerItem(JDRMarker.ARROW_HALF_POINTED_DOWN,
         JDRResources.getString("arrow.halfpointed.down")),
      new MarkerItem(JDRMarker.ARROW_HALF_POINTED60_UP,
         JDRResources.getString("arrow.halfpointed60.up")),
      new MarkerItem(JDRMarker.ARROW_HALF_POINTED60_DOWN,
         JDRResources.getString("arrow.halfpointed60.down")),
      new MarkerItem(JDRMarker.ARROW_HALF_POINTED45_UP,
         JDRResources.getString("arrow.halfpointed45.up")),
      new MarkerItem(JDRMarker.ARROW_HALF_POINTED45_DOWN,
         JDRResources.getString("arrow.halfpointed45.down")),
      new MarkerItem(JDRMarker.ARROW_HALF_CUSP_UP,
         JDRResources.getString("arrow.halfcusp.up")),
      new MarkerItem(JDRMarker.ARROW_HALF_CUSP_DOWN,
         JDRResources.getString("arrow.halfcusp.down"))
   };

   public static final MarkerItem[] bracketStyles =
   {
      new MarkerItem(JDRMarker.ARROW_SQUARE,
         JDRResources.getString("arrow.square")),
      new MarkerItem(JDRMarker.ARROW_BAR,
         JDRResources.getString("arrow.bar")),
      new MarkerItem(JDRMarker.ARROW_ROUND,
         JDRResources.getString("arrow.round")),
      new MarkerItem(JDRMarker.ARROW_BRACE,
         JDRResources.getString("arrow.brace")),
      new MarkerItem(JDRMarker.ARROW_ALT_SQUARE,
         JDRResources.getString("arrow.altsquare")),
      new MarkerItem(JDRMarker.ARROW_ALT_BAR,
         JDRResources.getString("arrow.altbar")),
      new MarkerItem(JDRMarker.ARROW_ALT_ROUND,
         JDRResources.getString("arrow.altround")),
      new MarkerItem(JDRMarker.ARROW_ALT_BRACE,
         JDRResources.getString("arrow.altbrace"))
   };

   public static final MarkerItem[] shapesStyles =
   {
      new MarkerItem(JDRMarker.ARROW_CIRCLE,
         JDRResources.getString("arrow.circle")),
      new MarkerItem(JDRMarker.ARROW_DIAMOND,
         JDRResources.getString("arrow.diamond")),
      new MarkerItem(JDRMarker.ARROW_CIRCLE_OPEN,
         JDRResources.getString("arrow.circle.open")),
      new MarkerItem(JDRMarker.ARROW_DIAMOND_OPEN,
         JDRResources.getString("arrow.diamond.open")),
      new MarkerItem(JDRMarker.ARROW_SCISSORS_UP_FILLED,
         JDRResources.getString("arrow.scissors.up.filled")),
      new MarkerItem(JDRMarker.ARROW_SCISSORS_DOWN_FILLED,
         JDRResources.getString("arrow.scissors.down.filled")),
      new MarkerItem(JDRMarker.ARROW_SCISSORS_UP_OPEN,
         JDRResources.getString("arrow.scissors.up.open")),
      new MarkerItem(JDRMarker.ARROW_SCISSORS_DOWN_OPEN,
         JDRResources.getString("arrow.scissors.down.open")),
      new MarkerItem(JDRMarker.ARROW_HEART_RIGHT_FILLED,
         JDRResources.getString("arrow.heart.right.filled")),
      new MarkerItem(JDRMarker.ARROW_HEART_RIGHT_OPEN,
         JDRResources.getString("arrow.heart.right.open")),
      new MarkerItem(JDRMarker.ARROW_HEART_FILLED,
         JDRResources.getString("arrow.heart.filled")),
      new MarkerItem(JDRMarker.ARROW_HEART_OPEN,
         JDRResources.getString("arrow.heart.open")),
      new MarkerItem(JDRMarker.ARROW_SNOWFLAKE,
         JDRResources.getString("arrow.snowflake")),
      new MarkerItem(JDRMarker.ARROW_STAR_CHEVRON_OPEN,
         JDRResources.getString("arrow.starchevron.open")),
      new MarkerItem(JDRMarker.ARROW_STAR_CHEVRON_FILLED,
         JDRResources.getString("arrow.starchevron.filled"))
   };

   public static final MarkerItem[] dataStyles =
   {
      new MarkerItem(JDRMarker.ARROW_DOTFILLED,
         JDRResources.getString("arrow.dotfilled")),
      new MarkerItem(JDRMarker.ARROW_DOTOPEN,
         JDRResources.getString("arrow.dotopen")),
      new MarkerItem(JDRMarker.ARROW_BOXFILLED,
         JDRResources.getString("arrow.boxfilled")),
      new MarkerItem(JDRMarker.ARROW_BOXOPEN,
         JDRResources.getString("arrow.boxopen")),
      new MarkerItem(JDRMarker.ARROW_CROSS,
         JDRResources.getString("arrow.cross")),
      new MarkerItem(JDRMarker.ARROW_PLUS,
         JDRResources.getString("arrow.plus")),
      new MarkerItem(JDRMarker.ARROW_STAR,
         JDRResources.getString("arrow.star")),
      new MarkerItem(JDRMarker.ARROW_ASTERISK,
         JDRResources.getString("arrow.asterisk")),
      new MarkerItem(JDRMarker.ARROW_STAR5_FILLED,
         JDRResources.getString("arrow.star5.filled")),
      new MarkerItem(JDRMarker.ARROW_STAR5_OPEN,
         JDRResources.getString("arrow.star5.open")),
      new MarkerItem(JDRMarker.ARROW_STAR6_FILLED,
         JDRResources.getString("arrow.star6.filled")),
      new MarkerItem(JDRMarker.ARROW_STAR6_OPEN,
         JDRResources.getString("arrow.star6.open")),
      new MarkerItem(JDRMarker.ARROW_TRIANGLE_UP_FILLED,
         JDRResources.getString("arrow.triangle.up.filled")),
      new MarkerItem(JDRMarker.ARROW_TRIANGLE_UP_OPEN,
         JDRResources.getString("arrow.triangle.up.open")),
      new MarkerItem(JDRMarker.ARROW_TRIANGLE_DOWN_FILLED,
         JDRResources.getString("arrow.triangle.down.filled")),
      new MarkerItem(JDRMarker.ARROW_TRIANGLE_DOWN_OPEN,
         JDRResources.getString("arrow.triangle.down.open")),
      new MarkerItem(JDRMarker.ARROW_RHOMBUS_FILLED,
         JDRResources.getString("arrow.rhombus.filled")),
      new MarkerItem(JDRMarker.ARROW_RHOMBUS_OPEN,
         JDRResources.getString("arrow.rhombus.open")),
      new MarkerItem(JDRMarker.ARROW_PENTAGON_FILLED,
         JDRResources.getString("arrow.pentagon.filled")),
      new MarkerItem(JDRMarker.ARROW_PENTAGON_OPEN,
         JDRResources.getString("arrow.pentagon.open")),
      new MarkerItem(JDRMarker.ARROW_HEXAGON_FILLED,
         JDRResources.getString("arrow.hexagon.filled")),
      new MarkerItem(JDRMarker.ARROW_HEXAGON_OPEN,
         JDRResources.getString("arrow.hexagon.open")),
      new MarkerItem(JDRMarker.ARROW_OCTAGON_FILLED,
         JDRResources.getString("arrow.octagon.filled")),
      new MarkerItem(JDRMarker.ARROW_OCTAGON_OPEN,
         JDRResources.getString("arrow.octagon.open")),
      new MarkerItem(JDRMarker.ARROW_SEMICIRCLE_FILLED,
         JDRResources.getString("arrow.semicircle.filled")),
      new MarkerItem(JDRMarker.ARROW_SEMICIRCLE_OPEN,
         JDRResources.getString("arrow.semicircle.open"))
   };

   public static final MarkerItem[] capStyles = 
   {
      new MarkerItem(JDRMarker.ARROW_RECTANGLE_CAP,
         JDRResources.getString("arrow.rectanglecap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_ROUND_CAP,
         JDRResources.getString("arrow.roundcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_TRIANGLE_CAP,
         JDRResources.getString("arrow.trianglecap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_INVERT_TRIANGLE_CAP,
         JDRResources.getString("arrow.inverttrianglecap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_CHEVRON_CAP,
         JDRResources.getString("arrow.chevroncap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_INVERT_CHEVRON_CAP,
         JDRResources.getString("arrow.invertchevroncap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_FAST_CAP,
         JDRResources.getString("arrow.fastcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_INVERT_FAST_CAP,
         JDRResources.getString("arrow.invertfastcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_BALL_CAP,
         JDRResources.getString("arrow.ballcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_LEAF_CAP,
         JDRResources.getString("arrow.leafcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_LEAF2_CAP,
         JDRResources.getString("arrow.doubleleafcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_LEAF3_CAP,
         JDRResources.getString("arrow.tripleleafcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_CLUB_CAP,
         JDRResources.getString("arrow.clubcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_LEAF3FOR_CAP,
         JDRResources.getString("arrow.tripleleafforwardcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_LEAF3BACK_CAP,
         JDRResources.getString("arrow.tripleleafbackcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_LEAF2FOR_CAP,
         JDRResources.getString("arrow.doubleleafforwardcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_LEAF2BACK_CAP,
         JDRResources.getString("arrow.doubleleafbackcap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_BULGE_CAP,
         JDRResources.getString("arrow.bulgecap"), 10, 50),
      new MarkerItem(JDRMarker.ARROW_CUTOUTBULGE_CAP,
         JDRResources.getString("arrow.cutoutbulgecap"), 10, 50),
   };

   private ArrowStylePanel arrowStylePanel;
}

//class MarkerItem implements ListCellRenderer<MarkerItem>// Java7
class MarkerItem implements ListCellRenderer
{
   public MarkerItem(int markerType, String markerLabel,
                     double penWidth)
   {
      this(markerType, markerLabel, penWidth, 24);
   }

   public MarkerItem(int markerType, String markerLabel,
                     double penWidth, int maxWidth)
   {
      type = markerType;
      label = markerLabel;

      try
      {
         JDRMarker marker = JDRMarker.getPredefinedMarker(markerType,
            penWidth, 1, false, 5.0);

         resizable = marker.isResizable();

         if (type != JDRMarker.ARROW_NONE)
         {
            GeneralPath path = marker.getGeneralPath();

            Rectangle bounds = path.getBounds();

            BufferedImage image 
               = new BufferedImage(maxWidth, (int)bounds.getHeight(),
                   BufferedImage.TYPE_INT_ARGB);

            Graphics2D g = image.createGraphics();

            RenderingHints renderHints =
               new RenderingHints(RenderingHints.KEY_ANTIALIASING,
                                  RenderingHints.VALUE_ANTIALIAS_ON);

            renderHints.add(new RenderingHints(
                              RenderingHints.KEY_RENDERING,
                              RenderingHints.VALUE_RENDER_QUALITY));

            double halfHeight = 0.5*bounds.getHeight();

            g.setRenderingHints(renderHints);
            g.translate(-bounds.getX(), halfHeight);

            g.setPaint(Color.black);
            g.fill(path);

            g.dispose();

            ic = new ImageIcon(image);

            image 
               = new BufferedImage(maxWidth,(int)bounds.getHeight(),
                    BufferedImage.TYPE_INT_ARGB);

            path = marker.getGeneralPath();

            g = image.createGraphics();

            g.translate(-bounds.getX(), halfHeight);

            g.setPaint(disabledForeground);
            g.fill(path);

            g.dispose();

            disabledIc = new ImageIcon(image);
         }
      }
      catch (InvalidMarkerTypeException e)
      {
         JDRResources.internalError(null,
            JDRResources.getString("error.invalid_arrow_type"), e);
      }
      catch (InvalidRepeatValueException e)
      {
      }
   }

   public MarkerItem(int markerType, String markerLabel)
   {
      this(markerType, markerLabel, 1.0);
   }

   public MarkerItem()
   {
      type = JDRMarker.ARROW_NONE;
      label = "";
      panel = new JPanel()
      {
         public void paintComponent(Graphics g)
         {
            super.paintComponent(g);

            if (cellHasFocus)
            {
               Rectangle bounds = getBounds();
               int w = (int)bounds.getWidth();
               int h = (int)bounds.getHeight();

               bounds.setBounds(1,1,w-2,h-2);

               Graphics2D g2 = (Graphics2D)g;

               Paint oldPaint = g2.getPaint();

               g2.setPaint(Color.darkGray);

               g2.draw(bounds);

               g2.setPaint(oldPaint);
            }
         }
      };

      panel.setLayout(new FlowLayout(FlowLayout.LEADING));
      iconComp = new JLabel();
      textComp = new JLabel();
      textComp.setOpaque(false);
      iconComp.setOpaque(false);
      panel.add(iconComp);
      panel.add(textComp);
   }

   //public Component getListCellRendererComponent(JList<? extends MarkerItem> list,//Java7
      //MarkerItem value, int index, boolean isSelected,//Java7
   public Component getListCellRendererComponent(JList list,
      Object value, int index, boolean isSelected,
      boolean hasCellFocus)
   {
      MarkerItem item = (MarkerItem)value;

      textComp.setText(item.getLabel()+" ");

      if (!list.isEnabled())
      {
         panel.setBackground(disabledBackground);
         panel.setForeground(disabledForeground);
         iconComp.setIcon(item.disabledIc);
      }
      else if (isSelected)
      {
         panel.setBackground(list.getSelectionBackground());
         panel.setForeground(list.getSelectionForeground());
         iconComp.setIcon(item.ic);
      }
      else
      {
         panel.setBackground(list.getBackground());
         panel.setForeground(list.getForeground());
         iconComp.setIcon(item.ic);
      }

      cellHasFocus = hasCellFocus;
      panel.setEnabled(list.isEnabled());
      panel.setFont(list.getFont());
      panel.setOpaque(true);

      return panel;
   }

   public int getType()
   {
      return type;
   }

   public String getLabel()
   {
      return label;
   }

   public boolean isResizable()
   {
      return resizable;
   }

   public String toString()
   {
      return "MarkerItem [type="+type+", label="+label+"]";
   }

   private int type;
   private String label;
   private Icon ic=null, disabledIc=null;
   private JLabel iconComp=null;
   private JLabel textComp=null;
   private JPanel panel=null;
   private boolean resizable = false;

   private boolean cellHasFocus=false;

   private static Color disabledBackground = Color.lightGray;
   private static Color disabledForeground = Color.gray;
}

