// File          : JDRSelector.java
// Description   : Dialog for setting path and/or text styles
// Date          : 6th February 2006
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog for setting marker style.
 * @see ArrowStyleSelector
 * @author Nicola L C Talbot
 */

public class JDRSelector extends JDialog 
   implements ActionListener
{
   public JDRSelector(JpgfDraw application,
                      String title, 
                      boolean showPaths, boolean showText,
                      boolean showAnchor)
   {
      super(application, true);
      application_ = application;
      setTitle(title);

      mainPanel = new JPanel();

      if (showPaths)
      {
         samplePathPanel = new SamplePathPanel(this, showText);

         Component comp;

         if (showText)
         {
            mainSplit = new JSplitPane(
               JSplitPane.HORIZONTAL_SPLIT,
               mainPanel,
               new JScrollPane(getSamplePathComp()));

            mainSplit.setResizeWeight(0.7);

            sampleTextPanel = new SampleTextPanel(this, showAnchor);

            JSplitPane split2 = new JSplitPane(
               JSplitPane.VERTICAL_SPLIT,
               new JScrollPane(getSampleTextComp()),
               mainSplit);

            split2.setResizeWeight(0.3);

            comp = split2;
         }
         else
         {
            mainSplit = new JSplitPane(
               JSplitPane.VERTICAL_SPLIT,
               new JScrollPane(getSamplePathComp()),
               mainPanel);

            mainSplit.setResizeWeight(0.3);

            comp = mainSplit;
         }

         add(comp);
      }
      else if (showText)
      {
         sampleTextPanel = new SampleTextPanel(this, showAnchor);

         getSampleTextComp().setPreferredSize(new Dimension(600,70));

         mainSplit = new JSplitPane(
            JSplitPane.VERTICAL_SPLIT,
            new JScrollPane(getSampleTextComp()),
            mainPanel);

         add(mainSplit);
      }

      // OK/Cancel etc panel

      actionPanel = new JPanel();
      add(actionPanel, "South");
      actionPanel.setBorder(BorderFactory.createEtchedBorder());

      actionPanel.add(JDRResources.createOkayButton(this));
      actionPanel.add(JDRResources.createCancelButton(this));

      help = JDRResources.createAppButton("help", null, null, 
         JDRResources.getString("label.help"));

      actionPanel.add(help);

      defaults = new JButton(JDRResources.getString("label.default"));
      defaults.setMnemonic(JDRResources.getChar("label.default.mnemonic"));
      defaults.setActionCommand("defaults");
      actionPanel.add(defaults);

      defaults.addActionListener(this);

      setSize(650, 650);
      setLocationRelativeTo(application_);
   }

   public JDRSelector(JpgfDraw application,
                      String title, 
                      boolean showPaths)
   {
      this(application, title, showPaths, false, false);
   }

   public JDRSelector(JpgfDraw application,
                      String title, 
                      boolean showPaths, boolean showText)
   {
      this(application, title, showPaths, showText, false);
   }

   public JDRSelector(JpgfDraw application,
                      String title)
   {
      this(application, title, true, true, true);
   }

   public void addToMain(JComponent comp)
   {
      mainPanel.add(comp);
   }

   public void setToMain(JComponent comp)
   {
      if (mainSplit.getTopComponent() == mainPanel)
      {
         mainSplit.setTopComponent(comp);
      }
      else
      {
         mainSplit.setBottomComponent(comp);
      }

      mainPanel = comp;
   }

   public void setMain(JComponent comp)
   {
   }

   public void repaintSample()
   {
      if (samplePathPanel != null)
      {
         samplePathPanel.updateSamples();
         samplePathPanel.repaint();
      }

      if (sampleTextPanel != null)
      {
         sampleTextPanel.updateSamples();
         sampleTextPanel.repaint();
      }
   }

   public void actionPerformed(ActionEvent e)
   {
      String action = e.getActionCommand();

      if (action == null) return;

      if (action.equals("cancel"))
      {
         cancel();
      }
      else if (action.equals("defaults"))
      {
         setDefaults();
      }
      else if (action.equals("okay"))
      {
         okay();
      }
   }

   public void initialise()
   {
      if (samplePathPanel != null)
      {
         samplePathPanel.updateSamples();
      }

      if (sampleTextPanel != null)
      {
         sampleTextPanel.updateSamples();
      }
 
      setVisible(true);
   }

   public void okay()
   {
      setVisible(false);
   }

   public void cancel()
   {
      setVisible(false);
   }

   public void setDefaults()
   {
   }

   public int getFontSeries()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         return 0;
      }

      return frame.getSelectedFontSeries();
   }

   public int getFontShape()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         return 0;
      }

      return frame.getSelectedFontShape();
   }

   public JDRPaint getLinePaint()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         try
         {
            return new JDRGray(100,0);
         }
         catch (InvalidFormatException e)
         {
            return new JDRTransparent();
         }
      }

      return frame.getSelectedLinePaint();
   }

   public JDRPaint getTextPaint()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         try
         {
            return new JDRGray(100, 0);
         }
         catch (InvalidFormatException e)
         {
            return new JDRTransparent();
         }
      }

      return frame.getSelectedTextPaint();
   }

   public JDRPaint getFillPaint()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         return new JDRTransparent();
      }

      return frame.getSelectedFillPaint();
   }

   public String getFontName()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         return "Serif";
      }

      return frame.getSelectedFontName();
   }

   public int getFontSize()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         return 10;
      }

      return frame.getSelectedFontSize();
   }

   public int getHalign()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         return 0;
      }

      return frame.getSelectedHalign();
   }

   public int getValign()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         return 0;
      }

      return frame.getSelectedValign();
   }

   public JDRStroke getStroke()
   {
      JDRFrame frame = application_.getCurrentFrame();

      if (frame == null)
      {
         return new JDRBasicStroke();
      }

      return frame.getSelectedStroke();
   }

   public String getSampleText()
   {
      return JDRResources.getString("font.sample");
   }

   public RenderingHints getRenderingHints()
   {
      return application_.getRenderingHints();
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str="";

      str += "is visible: "+isVisible()+eol;
      str += "has focus: "+hasFocus()+eol;
      str += "defaults has focus: "+defaults.hasFocus()+eol;
      str += "help has focus: "+help.hasFocus()+eol;

      ActionMap actionMap = getRootPane().getActionMap();
      str += "action map: "+eol;

      Object[] allKeys = actionMap.allKeys();

      for (int i = 0; i < allKeys.length; i++)
      {
         str += "Key: "+allKeys[i]+" Action: "+actionMap.get(allKeys[i])+eol;
      }

      return str+eol;
   }

   public SamplePanel getSamplePathPanel()
   {
      return samplePathPanel;
   }

   public SamplePanel getSampleTextPanel()
   {
      return sampleTextPanel;
   }

   public JComponent getSamplePathComp()
   {
      return (JComponent)samplePathPanel;
   }

   public JComponent getSampleTextComp()
   {
      return (JComponent)sampleTextPanel;
   }

   public JpgfDraw application_;
   private SamplePanel samplePathPanel, sampleTextPanel;
   public JButton defaults, help;
   protected JPanel actionPanel;

   protected JComponent mainPanel;

   protected JSplitPane mainSplit;
}
