// File          : GradientDialog.java
// Description   : Panel for selecting gradient paint
// Date          : 13th August 2008
// Last Modified : 13th August 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Panel for selecting gradient paint.
 * @see ColorPanel
 * @author Nicola L C Talbot
 */

public class GradientPanel extends JPanel
   implements ActionListener,AdjustmentListener
{
   public GradientPanel(JDRSelector selector)
   {
      super();
      selector_ = selector;

      setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));

      // start colour
      JLabel startLabel 
         = new JLabel(JDRResources.getString("colour.start"));
      startLabel.setAlignmentX(Component.LEFT_ALIGNMENT);

      add(startLabel);

      startPanel = new ColorPanel(this);

      startPanel.setMnemonics(
         JDRResources.getChar("colour.rgb.mnemonic"),
         JDRResources.getChar("colour.cmyk.mnemonic"));
      startPanel.setAlignmentX(Component.LEFT_ALIGNMENT);

      add(startPanel);

      // end colour
      JLabel endLabel 
         = new JLabel(JDRResources.getString("colour.end"));
      endLabel.setAlignmentX(Component.LEFT_ALIGNMENT);

      add(endLabel);

      endPanel = new ColorPanel(this);

      endPanel.setMnemonics(
         JDRResources.getChar("colour.rgb2.mnemonic"),
         JDRResources.getChar("colour.cmyk2.mnemonic"));
      endPanel.setAlignmentX(Component.LEFT_ALIGNMENT);

      add(endPanel);

      // type and direction
      JPanel gradientTypePanel = new JPanel();
      gradientTypePanel.setLayout(new BoxLayout(gradientTypePanel,
         BoxLayout.LINE_AXIS));
      gradientTypePanel.setAlignmentX(Component.LEFT_ALIGNMENT);

      ButtonGroup gradientTypeGroup = new ButtonGroup();

      linearButton = new JRadioButton(
         JDRResources.getString("colour.linear"), true);
      linearButton.setMnemonic(
         JDRResources.getChar("colour.linear.mnemonic"));
      linearButton.addActionListener(this);
      linearButton.setAlignmentX(Component.LEFT_ALIGNMENT);
      linearButton.setAlignmentY(Component.TOP_ALIGNMENT);
      gradientTypePanel.add(linearButton);
      gradientTypeGroup.add(linearButton);

      linearDirectionPanel = new LinearGradientDirectionPanel();
      linearDirectionPanel.addActionListener(this);
      linearDirectionPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
      linearDirectionPanel.setAlignmentY(Component.TOP_ALIGNMENT);
      gradientTypePanel.add(linearDirectionPanel);

      radialButton = new JRadioButton(
         JDRResources.getString("colour.radial"));
      radialButton.setMnemonic(
         JDRResources.getChar("colour.radial.mnemonic"));
      radialButton.addActionListener(this);
      radialButton.setAlignmentX(Component.LEFT_ALIGNMENT);
      radialButton.setAlignmentY(Component.TOP_ALIGNMENT);
      gradientTypePanel.add(radialButton);
      gradientTypeGroup.add(radialButton);

      radialDirectionPanel = new RadialGradientDirectionPanel();
      radialDirectionPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
      radialDirectionPanel.setAlignmentY(Component.TOP_ALIGNMENT);
      gradientTypePanel.add(radialDirectionPanel);

      add(gradientTypePanel);

      setDefaults();
   }

   public void actionPerformed(ActionEvent e)
   {
      Object source = e.getSource();

      if (source == linearButton)
      {
         linearDirectionPanel.setEnabled(true);
         radialDirectionPanel.setEnabled(false);
      }
      else if (source == radialButton)
      {
         linearDirectionPanel.setEnabled(false);
         radialDirectionPanel.setEnabled(true);
      }

      selector_.repaintSample();
   }

   public void adjustmentValueChanged(AdjustmentEvent evt)
   {
      selector_.repaintSample();
   }

   public void setPaint(JDRPaint paint)
   {
      startPanel.setPaint(((JDRShading)paint).getStartColor());

      endPanel.setPaint(((JDRShading)paint).getEndColor());

      if (paint instanceof JDRGradient)
      {
         linearDirectionPanel.setDirection(
            ((JDRGradient)paint).getDirection());
         linearButton.setSelected(true);

         linearDirectionPanel.setEnabled(true);
         radialDirectionPanel.setEnabled(false);
      }
      else
      {
         radialDirectionPanel.setDirection(
            ((JDRRadial)paint).getStartLocation());
         radialButton.setSelected(true);

         linearDirectionPanel.setEnabled(false);
         radialDirectionPanel.setEnabled(true);
      }
   }

   public JDRPaint getPaint()
   {
      JDRPaint paint=null;

      JDRPaint startPaint = startPanel.getPaint();
      JDRPaint endPaint = endPanel.getPaint();

      if (linearButton.isSelected())
      {
         try
         {
            paint = new JDRGradient(
               linearDirectionPanel.getDirection(),
               startPaint, endPaint);
         }
         catch (InvalidFormatException e)
         {
            JDRResources.internalError(this, e);

            paint = new JDRGradient();
         }
      }
      else
      {
         try
         {
            paint = new JDRRadial(
               radialDirectionPanel.getDirection(),
               startPaint, endPaint);
         }
         catch (InvalidFormatException e)
         {
            JDRResources.internalError(this, e);
            paint = new JDRRadial();
         }
      }

      return paint;
   }

   public void setDefaults()
   {
      setPaint(defaultPaint);
      selector_.repaint();
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "";

      str += "GradientPanel:"+eol;
      str += "old paint: "+oldPaint+eol;

      return str+eol;
   }

   private JDRSelector selector_;
   private ColorPanel startPanel, endPanel;

   private JDRPaint defaultPaint = new JDRGradient();

   private JDRPaint oldPaint = defaultPaint;

   private LinearGradientDirectionPanel linearDirectionPanel;
   private RadialGradientDirectionPanel radialDirectionPanel;

   private JRadioButton linearButton, radialButton;
}
