// File          : FLFSelector.java
// Description   : Dialog for setting flowframe data
// Date          : 6th February 2006
// Last Modified : 9th June 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog for setting flowframe data.
 * @see ArrowStyleSelector
 * @author Nicola L C Talbot
 */

public class FLFSelector extends JDialog
   implements ActionListener,ItemListener
{
   public FLFSelector(JpgfDraw application)
   {
      super(application,
         JDRResources.getString("flowframe.title"),true);
      application_ = application;

      int width = 400;
      int height = 250;
      setSize(width,height);
      Toolkit tk = Toolkit.getDefaultToolkit();
      Dimension d = tk.getScreenSize();
      int screenHeight = d.height;
      int screenWidth  = d.width;
      setLocation((screenWidth-width)/2, (screenHeight-height)/2);

      JPanel p1 = new JPanel();

      p1.setLayout(new GridBagLayout());

      GridBagConstraints gbc = new GridBagConstraints();

      gbc.weightx    = 100;
      gbc.weighty    = 100;
      gbc.gridwidth  = 1;
      gbc.gridheight = 1;
      gbc.gridx      = 0;
      gbc.gridy      = 0;
      gbc.anchor = GridBagConstraints.WEST;

      JLabel typeLabel = new JLabel(JDRResources.getString("flowframe.type"));
      typeLabel.setDisplayedMnemonic(
         JDRResources.getChar("flowframe.type.mnemonic"));
      p1.add(typeLabel, gbc);

      gbc.gridx      = 1;
      //type = new JComboBox<String>( // Java7
      type = new JComboBox(
         new String[]
         {
            JDRResources.getString("flowframe.static"),
            JDRResources.getString("flowframe.flow"),
            JDRResources.getString("flowframe.dynamic"),
            JDRResources.getString("flowframe.none")
         }
      );
      type.addItemListener(this);
      typeLabel.setLabelFor(type);
      p1.add(type, gbc);

      gbc.gridx      = 2;
      labelLabel = new JLabel(JDRResources.getString("flowframe.label"));
      labelLabel.setDisplayedMnemonic(
         JDRResources.getChar("flowframe.label.mnemonic"));
      p1.add(labelLabel, gbc);

      gbc.gridx      = 3;
      labelText = new JTextField(8);
      labelLabel.setLabelFor(labelText);
      p1.add(labelText, gbc);

      gbc.gridx      = 0;
      gbc.gridy      ++;
      borderLabel = new JLabel(JDRResources.getString("flowframe.border"));
      borderLabel.setDisplayedMnemonic(
         JDRResources.getChar("flowframe.border.mnemonic"));
      p1.add(borderLabel, gbc);

      gbc.gridx      = 1;
      //border = new JComboBox<String>(// Java7
      border = new JComboBox(
         new String[]
         {
             JDRResources.getString("flowframe.border_none"),
             JDRResources.getString("flowframe.border_as_shown")
         }
      );
      border.setSelectedItem(
         JDRResources.getString("flowframe.border_as_shown"));
      borderLabel.setLabelFor(border);
      p1.add(border, gbc);

      gbc.gridx = 2;
      pagesLabel = new JLabel(JDRResources.getString("flowframe.pages"));
      pagesLabel.setDisplayedMnemonic(
         JDRResources.getChar("flowframe.pages.mnemonic"));
      p1.add(pagesLabel, gbc);

      gbc.gridx = 3;
      //pageList = new JComboBox<String>(// Java7
      pageList = new JComboBox(
         new String[]
         {
            JDRResources.getString("flowframe.pages_all"),
            JDRResources.getString("flowframe.pages_none"),
            JDRResources.getString("flowframe.pages_odd"),
            JDRResources.getString("flowframe.pages_even")
         }
      );
      pageList.setEditable(true);
      pagesLabel.setLabelFor(pageList);
      p1.add(pageList, gbc);

      gbc.gridy++;
      gbc.gridx=0;
      gbc.gridwidth=1;
      gbc.gridheight=1;

      shapeLabel = new JLabel(
         JDRResources.getString("flowframe.shape"));
      shapeLabel.setDisplayedMnemonic(
         JDRResources.getChar("flowframe.shape.mnemonic"));
      p1.add(shapeLabel, gbc);

      gbc.gridx++;

      //shapeBox = new JComboBox<String>(// Java7
      shapeBox = new JComboBox(
         new String[]
         {
            JDRResources.getString("flowframe.shape_standard"),
            JDRResources.getString("flowframe.shape_parshape"),
            JDRResources.getString("flowframe.shape_shapepar")
          }
      );
      shapeBox.addItemListener(this);
      shapeLabel.setLabelFor(shapeBox);
      p1.add(shapeBox, gbc);

      gbc.gridx++;

      alignLabel = new JLabel(
         JDRResources.getString("flowframe.align"));
      alignLabel.setDisplayedMnemonic(
         JDRResources.getChar("flowframe.align.mnemonic"));
      p1.add(alignLabel, gbc);

      gbc.gridx++;

      //alignBox = new JComboBox<String>(// Java7
      alignBox = new JComboBox(
         new String[]
         {
            JDRResources.getString("flowframe.align_top"),
            JDRResources.getString("flowframe.align_middle"),
            JDRResources.getString("flowframe.align_bottom")
          }
      );
      alignLabel.setLabelFor(alignBox);
      p1.add(alignBox, gbc);

      gbc.gridy++;
      gbc.gridx = 0;
      gbc.gridwidth  = 4;
      gbc.gridheight = 2;
      margins = new MarginPanel();
      p1.add(margins, gbc);

      getContentPane().add(p1, "Center");

      JPanel p2 = new JPanel();

      p2.add(JDRResources.createOkayButton(this));
      p2.add(JDRResources.createCancelButton(this));
      p2.add(JDRResources.createHelpButton("sec:framedef"));

      getContentPane().add(p2, "South");
   }

   public void initialise()
   {
      JDRFrame mainPanel = application_.getCurrentFrame();
      object = mainPanel.getSelectedObject();

      if (object != null) setValues(object.flowframe);

      setVisible(true);
   }

   public void okay()
   {
      FlowFrame flowframe=null;
      JDRFrame mainPanel = application_.getCurrentFrame();

      int idx = type.getSelectedIndex();
      if (idx != NONE)
      {
         // check label supplied
         if (labelText.getText().equals(""))
         {
            JDRResources.error(this,
               JDRResources.getString("error.no_idl"));
            return;
         }

         // check unique label
         if (!mainPanel.isUniqueLabel(idx,object,labelText.getText()))
         {
            JDRResources.error(this,
               JDRResources.getString("error.idl_exists"));
            return;
         }

         String pages = (String)pageList.getSelectedItem();

         if (pages.equals(
              JDRResources.getString("flowframe.pages_all")))
         {
            pages = "all";
         }
         else if (pages.equals(
            JDRResources.getString("flowframe.pages_odd")))
         {
            pages = "odd";
         }
         else if (pages.equals(
            JDRResources.getString("flowframe.pages_even")))
         {
            pages = "even";
         }
         else if (pages.equals(
            JDRResources.getString("flowframe.pages_none")))
         {
            pages = "none";
         }

         // check page list is valid
         if (!FlowFrame.isValidPageList(pages))
         {
            JDRResources.error(this,
               JDRResources.getString("error.invalid_pagelist"));
            return;
         }
         
         flowframe  = new FlowFrame(idx, 
            border.getSelectedIndex()==1, labelText.getText(),
            pages);

         if (margins.isEnabled())
         {
            flowframe.left = margins.left();
            flowframe.right = margins.right();
            flowframe.top = margins.top();
            flowframe.bottom = margins.bottom();
         }
         else
         {
            flowframe.left   = 0;
            flowframe.right  = 0;
            flowframe.top    = 0;
            flowframe.bottom = 0;
         }

         if ((idx == FlowFrame.STATIC || idx == FlowFrame.DYNAMIC))
         {
            if (object instanceof JDRPath)
            {
               try
               {
                  flowframe.setShape(shapeBox.getSelectedIndex());
               }
               catch (InvalidFormatException e)
               {
                  JDRResources.internalError(this, e);
                  return;
               }
            }

            try
            {
               flowframe.setVAlign(alignBox.getSelectedIndex());
            }
            catch (InvalidFormatException e)
            {
               JDRResources.internalError(this, e);
               return;
            }
         }
      }

      setVisible(false);
      if (object != null) mainPanel.setFlowFrame(object, flowframe);
      object=null;
   }

   public void actionPerformed(ActionEvent evt)
   {
      String action = evt.getActionCommand();

      if (action == null) return;

      if (action.equals("okay"))
      {
         okay();
      }
      else if (action.equals("cancel"))
      {
         setVisible(false);
         object=null;
      }
   }

   public void itemStateChanged(ItemEvent evt)
   {
      Object source = evt.getSource();

      if (evt.getStateChange() == ItemEvent.SELECTED)
      {
         if (source == type)
         {
            int idx = type.getSelectedIndex();

            boolean isStaticOrDynamic = 
              (idx == STATIC || idx == DYNAMIC);

            boolean enableShape = (isStaticOrDynamic
               && (object instanceof JDRPath));

            shapeLabel.setEnabled(enableShape);
            shapeBox.setEnabled(enableShape);

            if (!alignBox.isEnabled() && isStaticOrDynamic)
            {
               alignBox.setSelectedIndex(
                  idx == STATIC ? FlowFrame.CENTER : FlowFrame.TOP);
            }

            alignLabel.setEnabled(isStaticOrDynamic);
            alignBox.setEnabled(isStaticOrDynamic);

            if (idx == NONE)
            {
               border.setEnabled(false);
               borderLabel.setEnabled(false);
               labelText.setEnabled(false);
               labelLabel.setEnabled(false);
               pageList.setEnabled(false);
               pagesLabel.setEnabled(false);
               margins.setEnabled(false);
            }
            else
            {
               border.setEnabled(true);
               borderLabel.setEnabled(true);
               labelText.setEnabled(true);
               labelLabel.setEnabled(true);
               pageList.setEnabled(true);
               pagesLabel.setEnabled(true);
               margins.setEnabled(
                  (shapeBox.getSelectedIndex()==FlowFrame.STANDARD)
                   || !enableShape);
            }
         }
         else if (source == shapeBox)
         {
            margins.setEnabled(
              (shapeBox.getSelectedIndex()==FlowFrame.STANDARD));
         }
      }
   }

   public void setValues(FlowFrame flowframe)
   {
      if (flowframe == null)
      {
         type.setSelectedIndex(NONE);
         labelText.setText("");
      }
      else
      {
         int idx = flowframe.getType();
         type.setSelectedIndex(idx);
         border.setSelectedIndex(flowframe.border?1:0);

         if (flowframe.pages.equals("all"))
         {
            pageList.setSelectedItem(
               JDRResources.getString("flowframe.pages_all"));
         }
         else if (flowframe.pages.equals("odd"))
         {
            pageList.setSelectedItem(
               JDRResources.getString("flowframe.pages_odd"));
         }
         else if (flowframe.pages.equals("even"))
         {
            pageList.setSelectedItem(
               JDRResources.getString("flowframe.pages_even"));
         }
         else if (flowframe.pages.equals("none"))
         {
            pageList.setSelectedItem(
               JDRResources.getString("flowframe.pages_none"));
         }
         else
         {
            pageList.setSelectedItem(flowframe.pages);
         }

         labelText.setText(flowframe.label);

         margins.setMargins(flowframe.left, flowframe.right,
                            flowframe.top, flowframe.bottom);

         shapeBox.setSelectedIndex(flowframe.getShape());
         alignBox.setSelectedIndex(flowframe.getVAlign());

         boolean isStaticOrDynamic = 
           (idx == STATIC || idx == DYNAMIC);

         boolean enableShape = (isStaticOrDynamic
            && (object instanceof JDRPath));

         shapeLabel.setEnabled(enableShape);
         shapeBox.setEnabled(enableShape);

         alignLabel.setEnabled(isStaticOrDynamic);
         alignBox.setEnabled(isStaticOrDynamic);
      }
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "";

      str += "FLFSelector:"+eol;
      str += "has focus: "+hasFocus()+eol;
      str += "type box has focus: "+type.hasFocus()+eol;
      str += "border box has focus: "+border.hasFocus()+eol;
      str += "page list box has focus: "+pageList.hasFocus()+eol;
      str += "label field has focus: "+labelText.hasFocus()+eol;

      return str+eol;
   }

   private JpgfDraw application_;
   private JDRCompleteObject object=null;
   //private JComboBox<String> type, border, pageList; //Java7
   private JComboBox type, border, pageList;
   private JTextField labelText;
   private JLabel labelLabel, borderLabel, pagesLabel; 
   private MarginPanel margins;
   public static final int STATIC=0, FLOW=1, DYNAMIC=2, NONE=3;

   private JComboBox shapeBox;
   private JLabel shapeLabel;

   private JComboBox alignBox;
   private JLabel alignLabel;
}
