// File          : DashPatternBoxPanel.java
// Description   : Panel for selecting dash pattern
// Date          : 6th February 2006
// Last Modified : 9th June 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;

import javax.swing.*;
import javax.swing.event.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

/**
 * Panel for selecting dash pattern.
 * @author Nicola L C Talbot
 */

class DashPatternBox extends JPanel implements ActionListener
{
   public DashPatternBox(SamplePanel panel)
   {
      super();
      samplePanel = panel;

      setLayout(new GridLayout(2, 3));

      offsetPanel = new NonNegativeLengthPanel(
         JDRResources.getString("dash.offset")+" ");
      offsetPanel.getDocument().addDocumentListener(
         new TextFieldSampleListener(panel));
      offsetPanel.getTextField().setColumns(2);
      add(offsetPanel);

      dashPanel = new NonNegativeLengthPanel(
         JDRResources.getString("dash.label")+" ");
      dashPanel.getDocument().addDocumentListener(
         new TextFieldSampleListener(panel));
      dashPanel.getTextField().setColumns(2);
      dashPanel.setValue(10.0f);
      add(dashPanel);

      gapPanel = new NonNegativeLengthPanel(
         JDRResources.getString("dash.gap")+" ");
      gapPanel.getDocument().addDocumentListener(
         new TextFieldSampleListener(panel));
      gapPanel.getTextField().setColumns(2);
      gapPanel.setValue(5.0f);
      add(gapPanel);

      secondary = new JCheckBox(JDRResources.getString("dash.secondary")+" ");
      secondary.addActionListener(this);
      add(secondary);

      dash2Panel = new NonNegativeLengthPanel(
         JDRResources.getString("dash.label")+" ");
      dash2Panel.getDocument().addDocumentListener(
         new TextFieldSampleListener(panel));
      dash2Panel.getTextField().setColumns(2);
      dash2Panel.setValue(1.0f);
      add(dash2Panel);

      gap2Panel = new NonNegativeLengthPanel(
         JDRResources.getString("dash.gap")+" ");
      gap2Panel.getDocument().addDocumentListener(
         new TextFieldSampleListener(panel));
      gap2Panel.getTextField().setColumns(2);
      gap2Panel.setValue(5.0f);
      add(gap2Panel);

      setSecondaryDash(false);
   }

   public void actionPerformed(ActionEvent evt)
   {
      Object source = evt.getSource();

      if (source == secondary)
      {
         enableSecondaryDash(secondary.isSelected());
         samplePanel.updateSamples();
      }
   }

   public void setEnabled(boolean flag)
   {
      dashPanel.setEnabled(flag);
      gapPanel.setEnabled(flag);
      offsetPanel.setEnabled(flag);
      secondary.setEnabled(flag);

      if (flag == false)
      {
         secondary.setSelected(false);
         enableSecondaryDash(false);
      }
      else
      {
         dashPanel.getTextField().requestFocusInWindow();
      }
   }

   public DashPattern getValue()
   {
      float dash = (float)dashPanel.getValue();
      float gap  = (float)gapPanel.getValue();

      if (dash == 0.0f && gap == 0.0f)
      {
         dash = 1.0f;
      }

      if (secondary.isSelected())
      {
         float dash2 = (float)dash2Panel.getValue();
         float gap2  = (float)gap2Panel.getValue();

         if (dash2 == 0.0f && gap2 == 0.0f)
         {
            dash2 = 1.0f;
         }

         return new DashPattern(new float[] {dash, gap, dash2, gap2},
                                (float)offsetPanel.getValue());
      }
      else
      {
         return new DashPattern(new float[] {dash, gap},
                                (float)offsetPanel.getValue());
      }
   }

   public void setValue(DashPattern dp)
   {
      setSecondaryDash(false);

      if (dp.pattern == null)
      {
         dashPanel.setValue(1.0);
         gapPanel.setValue(0.0);
      }
      else if (dp.pattern.length == 1)
      {
         dashPanel.setValue(dp.pattern[0]);
         gapPanel.setValue(0.0F);
      }
      else
      {
         dashPanel.setValue(dp.pattern[0]);
         gapPanel.setValue(dp.pattern[1]);

         if (dp.pattern.length >= 4)
         {
            dash2Panel.setValue(dp.pattern[2]);
            gap2Panel.setValue(dp.pattern[3]);
            setSecondaryDash(true);
         }
      }

      offsetPanel.setValue(dp.offset);
   }

   public void setSecondaryDash(boolean flag)
   {
      secondary.setSelected(flag);
      enableSecondaryDash(flag);
   }

   public void enableSecondaryDash(boolean flag)
   {
      dash2Panel.setEnabled(flag);
      gap2Panel.setEnabled(flag);
   }

   private NonNegativeLengthPanel dashPanel, gapPanel, offsetPanel,
      dash2Panel, gap2Panel;
   private JCheckBox secondary;
   private SamplePanel samplePanel;
}
