// File          : ConfigSettingsDialog.java
// Description   : Dialog for configuring application settings
// Date          : 1st February 2006
// Last Modified : 10th April 2011
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.text.DecimalFormat;
import java.io.*;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.awt.image.*;
import java.beans.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.border.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.filter.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog for configuring application settings.
 */

public class ConfigSettingsDialog extends JDialog
   implements ActionListener
{
   public ConfigSettingsDialog(JpgfDraw application)
   {
      super(application,
         JDRResources.getString("config.title"), true);
      application_ = application;

      JTabbedPane tabbedPane = new JTabbedPane();
      getContentPane().add(tabbedPane, "Center");

      int idx=0;

      dragScale = new DragScalePanel();

      tabbedPane.addTab(JDRResources.getString("hotspots.title"),
         null, dragScale,
         JDRResources.getString("tooltip.hotspots"));
      tabbedPane.setMnemonicAt(idx++,
         JDRResources.getChar("hotspots.mnemonic"));

      renderPanel = new RenderPanel();

      tabbedPane.addTab(JDRResources.getString("render.title"),
         null, renderPanel,
         JDRResources.getString("tooltip.render"));
      tabbedPane.setMnemonicAt(idx++,
         JDRResources.getChar("render.mnemonic"));

      dirPanel = new DirPanel();

      tabbedPane.addTab(JDRResources.getString("startdir.title"),
         null, dirPanel, JDRResources.getString("tooltip.startdir"));
      tabbedPane.setMnemonicAt(idx++,
         JDRResources.getChar("startdir.mnemonic"));

      jdrPanel = new JDRSettingsPanel();

      tabbedPane.addTab(JDRResources.getString("jdr.title"), null,
        jdrPanel, JDRResources.getString("tooltip.jdr"));
      tabbedPane.setMnemonicAt(idx++,
        JDRResources.getChar("jdr.mnemonic"));

      initAppSettingsPanel = new InitAppSettingsPanel();

      tabbedPane.addTab(JDRResources.getString("initsettings.title"),
         null, initAppSettingsPanel,
         JDRResources.getString("tooltip.initsettings"));
      tabbedPane.setMnemonicAt(idx++,
         JDRResources.getChar("initsettings.mnemonic"));

      langPanel = new LanguagePanel(application.appName.toLowerCase());

      tabbedPane.addTab(JDRResources.getString("lang.title"),
         null, langPanel,
         JDRResources.getString("lang.tooltip"));

      // OK/Cancel Button panel
      JPanel p = new JPanel();
      getContentPane().add(p, "South");

      p.add(JDRResources.createOkayButton(this));
      p.add(JDRResources.createCancelButton(this));
      p.add(JDRResources.createHelpButton("sec:configure"));

      pack();
      Dimension dim = getSize();
      dim.height = 400;
      setSize(dim);
      setLocationRelativeTo(application_);
   }

   public void actionPerformed(ActionEvent evt)
   {
      String action = evt.getActionCommand();

      if (action == null) return;

      if (action.equals("okay"))
      {
         okay();
      }
      else if (action.equals("cancel"))
      {
         setVisible(false);
      }
   }

   public void display()
   {
      JDRFrame frame = application_.getCurrentFrame();

      dragScale.initialise(application_);
      renderPanel.initialise(application_);
      dirPanel.initialise(application_);
      jdrPanel.initialise(application_);
      initAppSettingsPanel.initialise(application_);
      setVisible(true);
   }

   public void okay()
   {
      JDRFrame frame = application_.getCurrentFrame();

      dragScale.okay(application_);
      renderPanel.okay(application_);
      dirPanel.okay(application_);
      jdrPanel.okay(application_);
      initAppSettingsPanel.okay(application_);
      langPanel.okay();

      setVisible(false);
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "";

      str += "ConfigSettingsDialog:"+eol;
      str += "has focus: "+hasFocus()+eol;

      return str+eol;
   }

   private DragScalePanel dragScale;
   private RenderPanel renderPanel;
   private DirPanel dirPanel;
   private JDRSettingsPanel jdrPanel;
   private InitAppSettingsPanel initAppSettingsPanel;
   private LanguagePanel langPanel;
   private JpgfDraw application_;
}

class JDRSettingsPanel extends JPanel
{
   public JDRSettingsPanel()
   {
      super();

      Box b = Box.createVerticalBox();

      add(b);

      ButtonGroup saveGroup = new ButtonGroup();

      save = new JRadioButton(JDRResources.getString("jdr.save"));
      save.setMnemonic(JDRResources.getChar("jdr.save.mnemonic"));
      b.add(save);
      saveGroup.add(save);

      noSave = new JRadioButton(JDRResources.getString("jdr.no_save"));
      noSave.setMnemonic(JDRResources.getChar("jdr.no_save.mnemonic"));
      b.add(noSave);
      saveGroup.add(noSave);

      ButtonGroup useGroup = new ButtonGroup();

      use = new JRadioButton(JDRResources.getString("jdr.use"));
      use.setMnemonic(JDRResources.getChar("jdr.use.mnemonic"));
      b.add(use);
      useGroup.add(use);

      noUse = new JRadioButton(JDRResources.getString("jdr.no_use"));
      noUse.setMnemonic(JDRResources.getChar("jdr.no_use.mnemonic"));
      b.add(noUse);
      useGroup.add(noUse);

      warnOld = new JCheckBox(JDRResources.getString("jdr.warn_load_old"));
      warnOld.setMnemonic(JDRResources.getChar("jdr.warn_load_old.mnemonic"));
      b.add(warnOld);
   }

   public void initialise(JpgfDraw application)
   {
      if (application.saveJDRsettings())
      {
         save.setSelected(true);
      }
      else
      {
         noSave.setSelected(true);
      }

      if (application.useJDRsettings())
      {
         use.setSelected(true);
      }
      else
      {
         noUse.setSelected(true);
      }

      warnOld.setSelected(application.warnOnOldJdr());
   }

   public void okay(JpgfDraw application)
   {
      application.setJDRsettings(save.isSelected(), use.isSelected(),
         warnOld.isSelected());
   }

   private JRadioButton save, noSave, use, noUse;
   private JCheckBox warnOld;
}

class InitAppSettingsPanel extends JPanel
{
   public InitAppSettingsPanel()
   {
      super();

      Box b = Box.createVerticalBox();

      add(b);

      ButtonGroup useGroup = new ButtonGroup();

      useLast = new JRadioButton(
         JDRResources.getString("initsettings.last")+" ");
      useLast.setMnemonic(
         JDRResources.getChar("initsettings.last.mnemonic"));
      useLast.setAlignmentX(Component.LEFT_ALIGNMENT);
      b.add(useLast);
      useGroup.add(useLast);

      useCurrent = new JRadioButton(
         JDRResources.getString("initsettings.current")+" ");
      useCurrent.setMnemonic(
         JDRResources.getChar("initsettings.current.mnemonic"));
      useCurrent.setAlignmentX(Component.LEFT_ALIGNMENT);
      b.add(useCurrent);
      useGroup.add(useCurrent);

      useDefault = new JRadioButton(
         JDRResources.getString("initsettings.default")+" ");
      useDefault.setMnemonic(
         JDRResources.getChar("initsettings.default.mnemonic"));
      useDefault.setAlignmentX(Component.LEFT_ALIGNMENT);
      b.add(useDefault);
      useGroup.add(useDefault);
   }

   public void initialise(JpgfDraw application)
   {
      switch (application.initSettings())
      {
         case JpgfDrawSettings.INIT_LAST :
            useLast.setSelected(true);
         break;
         case JpgfDrawSettings.INIT_DEFAULT :
            useDefault.setSelected(true);
         break;
         case JpgfDrawSettings.INIT_USER :
            useCurrent.setSelected(true);
         break;
      }
   }

   public void okay(JpgfDraw application)
   {
      if (useLast.isSelected())
      {
         application.setInitSettings(JpgfDrawSettings.INIT_LAST);
      }
      else if (useDefault.isSelected())
      {
         application.setInitSettings(JpgfDrawSettings.INIT_DEFAULT);
      }
      else if (useCurrent.isSelected())
      {
         application.setInitSettings(JpgfDrawSettings.INIT_USER);
      }
   }

   private JRadioButton useLast, useDefault, useCurrent;
}

class DirPanel extends JPanel
   implements ActionListener
{
   public DirPanel()
   {
      super();

      ButtonGroup group = new ButtonGroup();

      Box box = Box.createVerticalBox();

      cwd = new JRadioButton(JDRResources.getString("startdir.cwd")+" ");
      cwd.setMnemonic(JDRResources.getChar("startdir.cwd.mnemonic"));
      cwd.setAlignmentX(Component.LEFT_ALIGNMENT);
      group.add(cwd);

      cwd.addActionListener(this);

      box.add(cwd);

      lastDir = new JRadioButton(
         JDRResources.getString("startdir.last")+" ");
      lastDir.setMnemonic(
         JDRResources.getChar("startdir.last.mnemonic"));
      lastDir.setAlignmentX(Component.LEFT_ALIGNMENT);
      group.add(lastDir);

      lastDir.addActionListener(this);

      box.add(lastDir);

      Box panel = Box.createHorizontalBox();

      panel.setAlignmentX(Component.LEFT_ALIGNMENT);

      named = new JRadioButton(
         JDRResources.getString("startdir.named")+" ");
      named.setMnemonic(JDRResources.getChar("startdir.named.mnemonic"));
      group.add(named);

      named.addActionListener(this);

      panel.add(named);

      directory = new JTextField(10);
      panel.add(directory);

      browse = new JButton(JDRResources.getString("startdir.browse"));
      browse.setMnemonic(JDRResources.getChar("startdir.browse.mnemonic"));
      browse.addActionListener(this);

      panel.add(browse);

      box.add(panel);

      add(box);

      fc = new JFileChooser();
      fc.setCurrentDirectory(new File("."));
      fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
      fc.setApproveButtonText(JDRResources.getString("label.okay"));
      fc.setApproveButtonMnemonic(JDRResources.getChar("label.okay.mnemonic"));
   }

   public void actionPerformed(ActionEvent evt)
   {
      Object source = evt.getSource();

      if (source == cwd || source == lastDir)
      {
         directory.setEnabled(false);
         browse.setEnabled(false);
      }
      else if (source == named)
      {
         directory.setEnabled(true);
         browse.setEnabled(true);
      }
      else if (source == browse)
      {
         int result = fc.showOpenDialog(this);

         if (result == JFileChooser.APPROVE_OPTION)
         {
            directory.setText(fc.getSelectedFile().getAbsolutePath());
         }
      }
   }

   public void initialise(JpgfDraw application)
   {
      int type = application.getStartDirType();

      switch (type)
      {
         case JpgfDrawSettings.STARTDIR_CWD:
            cwd.setSelected(true);
            directory.setEnabled(false);
            browse.setEnabled(false);
         break;
         case JpgfDrawSettings.STARTDIR_LAST:
            lastDir.setSelected(true);
            directory.setEnabled(false);
            browse.setEnabled(false);
         break;
         case JpgfDrawSettings.STARTDIR_NAMED:
            named.setSelected(true);
            directory.setText(application.getStartDirectory());
            directory.setEnabled(true);
            browse.setEnabled(true);
         break;
         default :
            JDRResources.internalError(this,
               JDRResources.getString("internal_error.invalid_startdir")+
               ": " +type);
      }
   }

   public void okay(JpgfDraw application)
   {
      int type;

      if (cwd.isSelected())
      {
         type = JpgfDrawSettings.STARTDIR_CWD;
      }
      else if (lastDir.isSelected())
      {
         type = JpgfDrawSettings.STARTDIR_LAST;
      }
      else
      {
         type = JpgfDrawSettings.STARTDIR_NAMED;
      }

      application.setStartDirectory(type, directory.getText());
   }

   private JRadioButton cwd, lastDir, named;
   private JButton browse;
   private JTextField directory;
   private JFileChooser fc;
}

class RenderPanel extends JPanel
{
   public RenderPanel()
   {
      super();

      setLayout(new BorderLayout());

      JPanel rowPanel = new JPanel();

      rowPanel.setBorder(BorderFactory.createLoweredBevelBorder());

      JPanel panel = new JPanel(new GridLayout(2, 3));

      panel.add(
        new JLabel(JDRResources.getString("render.anti_alias")+" "));

      ButtonGroup antialiasGroup = new ButtonGroup();

      antialiasOn = new JRadioButton(
         JDRResources.getString("render.on")+" ", false);
      antialiasOn.setMnemonic(JDRResources.getChar("render.on.mnemonic"));
      antialiasGroup.add(antialiasOn);

      panel.add(antialiasOn);

      antialiasOff = new JRadioButton(
         JDRResources.getString("render.off")+" ", true);
      antialiasOff.setMnemonic(
         JDRResources.getChar("render.off.mnemonic"));
      antialiasGroup.add(antialiasOff);

      panel.add(antialiasOff);

      panel.add(
         new JLabel(JDRResources.getString("render.rendering")+" "));

      ButtonGroup renderGroup = new ButtonGroup();

      quality = new JRadioButton(
         JDRResources.getString("render.quality")+" ", false);
      quality.setMnemonic(JDRResources.getChar("render.quality.mnemonic"));
      renderGroup.add(quality);

      panel.add(quality);

      speed = new JRadioButton(
         JDRResources.getString("render.speed")+" ", true);
      speed.setMnemonic(JDRResources.getChar("render.speed.mnemonic"));
      renderGroup.add(speed);

      panel.add(speed);

      rowPanel.add(panel);

      panel = new JPanel();

      JLabel label = new JLabel(
         JDRResources.getString("render.control_size")+" ");
      label.setDisplayedMnemonic(
         JDRResources.getChar("render.control_size.mnemonic"));
      panel.add(label);

      controlSize = new NonNegativeLengthPanel();
      label.setLabelFor(controlSize.getTextField());
      controlSize.setValue(10);

      panel.add(controlSize);

      rowPanel.add(panel);

      add(rowPanel, "North");

      panel = new JPanel(new GridLayout(2,2));

      panel.setBorder(BorderFactory.createLoweredBevelBorder());

      controlPaintPanels = new ControlPaintPanel[8];

      // Standard control colour

      addControlPaintPanels(panel, 0, "standard", new JDRPoint());

      // Symmetry line control colour

      addControlPaintPanels(panel, 2, "symmetry",
         new JDRSymmetryLinePoint());

      add(panel);

      // Pattern anchor control colour

      addControlPaintPanels(panel, 4, "patternanchor",
         new JDRPatternAnchorPoint());

      // Pattern adjust control colour

      addControlPaintPanels(panel, 6, "patternadjust",
         new JDRPatternAdjustPoint());

      add(panel, "Center");

      // Adjust label dimension so they are all the same width

      Dimension dim = controlPaintPanels[0].getLabel().getPreferredSize();

      double labelWidth = dim.getWidth();
      double labelHeight = dim.getHeight();

      dim = controlPaintPanels[1].getLabel().getPreferredSize();

      if (dim.getWidth() > labelWidth)
      {
         labelWidth = dim.getWidth();
      }

      if (dim.getHeight() > labelHeight)
      {
         labelHeight = dim.getHeight();
      }

      dim.setSize(labelWidth, labelHeight);

      for (int i = 0; i < controlPaintPanels.length; i++)
      {
         controlPaintPanels[i].getLabel().setPreferredSize(dim);
      }
   }

   private void addControlPaintPanels(JComponent parent,
      int index, String tag, JDRPoint point)
   {
      Box box = Box.createVerticalBox();

      box.setBorder(BorderFactory.createTitledBorder(
        BorderFactory.createEtchedBorder(),
        JDRResources.getString("render.control."+tag)));

      controlPaintPanels[index] = new ControlPaintPanel(point,
         JDRResources.getStringWithValue("render.control.choosertitle",
         new String[] {JDRResources.getString("render.control."+tag),
                       JDRResources.getString("render.control.unselected")}));

      box.add(controlPaintPanels[index]);

      JDRPoint selectedPoint = (JDRPoint)point.clone();
      selectedPoint.setSelected(true);

      controlPaintPanels[index+1] = new ControlPaintPanel(selectedPoint,
         JDRResources.getStringWithValue("render.control.choosertitle",
         new String[] {JDRResources.getString("render.control."+tag),
                       JDRResources.getString("render.control.selected")}));

      box.add(controlPaintPanels[index+1]);

      parent.add(box);
   }

   public void initialise(JpgfDraw application)
   {
      if (application.getAntiAlias())
      {
         antialiasOn.setSelected(true);
      }
      else
      {
         antialiasOff.setSelected(true);
      }

      if (application.getRenderQuality())
      {
         quality.setSelected(true);
      }
      else
      {
         speed.setSelected(true);
      }

      controlSize.setValue(JDRPoint.pointSize);
   }

   public void okay(JpgfDraw application)
   {
      application.setRendering(antialiasOn.isSelected(),
                               quality.isSelected());
      JDRPoint.pointSize = controlSize.getValue();

      for (int i = 0; i < controlPaintPanels.length; i++)
      {
         controlPaintPanels[i].updatePaint();
      }

      application.repaint();
   }

   private JRadioButton antialiasOn, antialiasOff;
   private JRadioButton quality, speed;
   private NonNegativeLengthPanel controlSize;

   private ControlPaintPanel[] controlPaintPanels;
}

class ControlPaintPanel extends JPanel implements ActionListener
{
   public ControlPaintPanel(JDRPoint p, String chooserTitle)
   {
      this(p, chooserTitle, 10.0);
   }

   public ControlPaintPanel(JDRPoint p, String chooserTitle,
      double ptSize)
   {
      point = p;

      pointSize = ptSize;

      point.x = pointSize*0.5+1;
      point.y = point.x;

      label = new JLabel(point.isSelected() ?
                     JDRResources.getString("render.control.selected") :
                     JDRResources.getString("render.control.unselected"));

      paint = point.isSelected() ? point.getSelectedPaint()
                                 : point.getUnselectedPaint();

      samplePanelSize = new Dimension((int)pointSize+2, (int)pointSize+2);

      samplePanel = new JPanel()
         {
            public void paintComponent(Graphics g)
            {
               super.paintComponent(g);

               point.draw(g, 1.0, paint, pointSize);
            }

            public Color getBackground()
            {
               return Color.white;
            }

            public Dimension getPreferredSize()
            {
               return samplePanelSize;
            }

            public Dimension getMinimumSize()
            {
               return samplePanelSize;
            }
         };

      add(samplePanel);

      add(label);

      selectButton = new JButton(JDRResources.getString("label.select"));

      selectButton.addActionListener(this);

      add(selectButton);

      colorChooser = new JColorChooser((Color)paint);

      colorChooser.setPreviewPanel
      (
         new JPanel()
         {
            public void paintComponent(Graphics g)
            {
               super.paintComponent(g);

               point.draw(g, 1.0, colorChooser.getColor(), pointSize);
            }

            public Color getBackground()
            {
               return Color.white;
            }

            public Dimension getPreferredSize()
            {
               return samplePanelSize;
            }

            public Dimension getMinimumSize()
            {
               return samplePanelSize;
            }
         }
      );

      colorDialog = JColorChooser.createDialog
      (
         this, chooserTitle, true, colorChooser,
         new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               paint = colorChooser.getColor();
               samplePanel.repaint();
            }
         },
         null
      );
   }

   public void updatePaint()
   {
      if (point.isSelected())
      {
         point.setSelectedPaint(paint);
      }
      else
      {
         point.setUnselectedPaint(paint);
      }
   }

   public void actionPerformed(ActionEvent evt)
   {
      Object source = evt.getSource();

      if (source == selectButton)
      {
         colorDialog.setVisible(true);
      }
   }

   public JLabel getLabel()
   {
      return label;
   }

   public String getLabelText()
   {
      return label.getText();
   }

   private JDRPoint point;
   private Paint paint;
   private double pointSize;

   private JPanel samplePanel;
   private JButton selectButton;
   private JColorChooser colorChooser;
   private JDialog colorDialog;

   private Dimension samplePanelSize;

   private JLabel label;
}

class DragScalePanel extends JPanel
{
   public DragScalePanel()
   {
      super();

      ButtonGroup group = new ButtonGroup();

      enable = new JRadioButton(
         JDRResources.getString("hotspots.enable"));
      enable.setMnemonic(
         JDRResources.getChar("hotspots.enable.mnemonic"));

      group.add(enable);
      add(enable);

      disable = new JRadioButton(
         JDRResources.getString("hotspots.disable"));
      disable.setMnemonic(
         JDRResources.getChar("hotspots.disable.mnemonic"));

      group.add(disable);
      add(disable);
   }

   public void initialise(JpgfDraw application)
   {
      boolean flag = application.dragScaleEnabled();

      if (flag)
      {
         enable.setSelected(flag);
      }
      else
      {
         disable.setSelected(!flag);
      }
   }

   public void okay(JpgfDraw application)
   {
      application.setDragScale(enable.isSelected());
   }

   private JRadioButton enable, disable;
}

class LanguagePanel extends JPanel
{
   public LanguagePanel(String appName)
   {
      super(new BorderLayout());

      JPanel p = new JPanel();

      JLabel dictLabel = new JLabel(JDRResources.getString("lang.dict"));
      dictLabel.setDisplayedMnemonic(JDRResources.getChar("lang.dict.mnemonic"));
      dictLabel.setToolTipText(JDRResources.getString("lang.dict.tooltip"));

      p.add(dictLabel);

      //dictLangBox = new JComboBox<String>(JDRResources.getAvailableDictLanguages());// Java7
      dictLangBox = new JComboBox(JDRResources.getAvailableDictLanguages());
      dictLabel.setLabelFor(dictLangBox);

      p.add(dictLangBox);

      String id = JDRResources.dictLocaleId;

      if (id != null)
      {
         dictLangBox.setSelectedItem(id);
      }

      JLabel helpLabel = new JLabel(JDRResources.getString("lang.help"));
      helpLabel.setDisplayedMnemonic(JDRResources.getChar("lang.help.mnemonic"));
      helpLabel.setToolTipText(JDRResources.getString("lang.help.tooltip"));

      p.add(helpLabel);

      //helpLangBox = new JComboBox<String>(// Java 7
      helpLangBox = new JComboBox(
         JDRResources.getAvailableHelpLanguages(appName));
      helpLabel.setLabelFor(helpLangBox);

      p.add(helpLangBox);

      id = JDRResources.helpLocaleId;

      if (id != null)
      {
         helpLangBox.setSelectedItem(id);
      }

      add(p, "Center");

      JTextArea info = new JTextArea(JDRResources.getString("lang.info"));
      info.setEditable(false);
      info.setOpaque(false);
      info.setLineWrap(true);
      info.setWrapStyleWord(true);

      add(info, "South");
   }

   public void okay()
   {
      JDRResources.dictLocaleId = (String)dictLangBox.getSelectedItem();
      JDRResources.helpLocaleId = (String)helpLangBox.getSelectedItem();
   }

   //private JComboBox<String> dictLangBox, helpLangBox; // Java7
   private JComboBox dictLangBox, helpLangBox;
}
