// File          : ColorPanel.java
// Description   : Panel for selecting non-gradient paint
// Date          : 6th February 2006
// Last Modified : 9th June 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;

/**
 * Panel for selecting non-gradient paint.
 * @author Nicola L C Talbot
 */

public class ColorPanel extends JPanel
   implements ActionListener,ChangeListener
{
   public ColorPanel()
   {
      rgbPanel = new ColorRGBPanel();
      cmykPanel = new ColorCMYKPanel();
      hsbPanel = new ColorHSBPanel();
      greyPanel = new GreyPanel();
      initialise();
   }

   public ColorPanel(AdjustmentListener al)
   {
      rgbPanel = new ColorRGBPanel(al);
      cmykPanel = new ColorCMYKPanel(al);
      hsbPanel = new ColorHSBPanel(al);
      greyPanel = new GreyPanel(al);
      initialise();
   }

   public void addAdjustmentListener(AdjustmentListener al)
   {
      rgbPanel.addAdjustmentListener(al);
      cmykPanel.addAdjustmentListener(al);
      hsbPanel.addAdjustmentListener(al);
      greyPanel.addAdjustmentListener(al);
   }

   public void initialise()
   {
      setLayout(new GridBagLayout());

      GridBagConstraints constraints = new GridBagConstraints();

      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.gridwidth = 2;
      constraints.gridheight = 1;
      constraints.weightx = 0;
      constraints.weighty = 100;
      constraints.anchor= GridBagConstraints.WEST;
      constraints.fill = GridBagConstraints.NONE;

      tabbedPane = new JTabbedPane();
      tabbedPane.setTabPlacement(JTabbedPane.RIGHT);
      tabbedPane.setAlignmentY(Component.TOP_ALIGNMENT);
      add(tabbedPane, constraints);

      // RGB selector panel

      tabbedPane.addTab(JDRResources.getString("colour.rgb"),
                        null, rgbPanel,
                        JDRResources.getString("tooltip.rgb"));
      tabbedPane.addChangeListener(this);
      currentPanel = rgbPanel;

      // CMYK selector panel

      tabbedPane.addTab(JDRResources.getString("colour.cmyk"),
                        null, cmykPanel,
                        JDRResources.getString("tooltip.cmyk"));

      // HSB selector panel

      tabbedPane.addTab(JDRResources.getString("colour.hsb"),
                        null, hsbPanel,
                        JDRResources.getString("tooltip.hsb"));

      // Grey selector panel

      tabbedPane.addTab(JDRResources.getString("colour.grey"),
                        null, greyPanel,
                        JDRResources.getString("tooltip.grey"));

      // predefined colour panel
      constraints.gridx = 2;
      constraints.gridy = 0;
      constraints.gridwidth = 1;
      constraints.gridheight = 1;
      constraints.weightx = 0;
      constraints.anchor= GridBagConstraints.EAST;
      predefinedPanel  = new JPanel();
      predefinedPanel.setAlignmentY(Component.TOP_ALIGNMENT);
      add(predefinedPanel, constraints);
      predefinedPanel.setLayout(new GridLayout(4,5));
      predefinedButtons = new ColorButton[MAX_PREDEFINED];
      predefined_n = 0;

      try
      {
         addPredefinedColor(Color.black,
            JDRResources.getString("tooltip.swatch.black"));
         addPredefinedColor(Color.darkGray,
            JDRResources.getString("tooltip.swatch.darkgrey"));
         addPredefinedColor(Color.gray,
            JDRResources.getString("tooltip.swatch.grey"));
         addPredefinedColor(Color.lightGray,
            JDRResources.getString("tooltip.swatch.lightgrey"));
         addPredefinedColor(Color.white,
            JDRResources.getString("tooltip.swatch.white"));

         addPredefinedColor(new Color(0,0,128),
            JDRResources.getString("tooltip.swatch.darkblue"));
         addPredefinedColor(Color.blue,
            JDRResources.getString("tooltip.swatch.blue"));
         addPredefinedColor(Color.cyan,
            JDRResources.getString("tooltip.swatch.cyan"));
         addPredefinedColor(Color.green,
            JDRResources.getString("tooltip.swatch.green"));
         addPredefinedColor(new Color(0,128,0),
            JDRResources.getString("tooltip.swatch.darkgreen"));

         addPredefinedColor(Color.yellow,
            JDRResources.getString("tooltip.swatch.yellow"));
         addPredefinedColor(new JDRColorCMYK(0.0,0.1,0.84,0),
            JDRResources.getString("tooltip.swatch.goldenrod"));
         addPredefinedColor(Color.orange,
            JDRResources.getString("tooltip.swatch.orange"));
         addPredefinedColor(new JDRColorCMYK(0.14,0.42,0.56,0.0),
            JDRResources.getString("tooltip.swatch.tan"));
         addPredefinedColor(new JDRColorCMYK(0.0,0.72,1.0,0.45),
            JDRResources.getString("tooltip.swatch.rawsienna"));

         addPredefinedColor(Color.red,
            JDRResources.getString("tooltip.swatch.red"));
         addPredefinedColor(new Color(128,0,0),
            JDRResources.getString("tooltip.swatch.darkred"));
         addPredefinedColor(new JDRColorCMYK(0.32,0.64,0.0,0.0),
            JDRResources.getString("tooltip.swatch.orchid"));
         addPredefinedColor(Color.magenta,
            JDRResources.getString("tooltip.swatch.magenta"));
         addPredefinedColor(Color.pink,
            JDRResources.getString("tooltip.swatch.pink"));
      }
      catch (InvalidFormatException ignore)
      {
      }
   }

   public void addPredefinedColor(Color c, String name)
   {
      predefinedButtons[predefined_n] = new ColorButton(c);
      predefinedPanel.add(predefinedButtons[predefined_n]);
      predefinedButtons[predefined_n].addActionListener(this);
      predefinedButtons[predefined_n].setToolTipText(name);
      predefined_n++;
   }

   public void addPredefinedColor(JDRPaint c, String name)
   {
      predefinedButtons[predefined_n] = new ColorButton(c);
      predefinedPanel.add(predefinedButtons[predefined_n]);
      predefinedButtons[predefined_n].addActionListener(this);
      predefinedButtons[predefined_n].setToolTipText(name);
      predefined_n++;
   }

   public void setMnemonics(char rgbMnemonic, char cmykMnemonic)
   {
      // set mnemonics
      tabbedPane.setMnemonicAt(0, rgbMnemonic);
      tabbedPane.setMnemonicAt(1, cmykMnemonic);
   }

   public void setMnemonics(char rgbMnemonic, char cmykMnemonic,
      char hsbMnemonic, char greyMnemonic)
   {
      // set mnemonics
      tabbedPane.setMnemonicAt(0, rgbMnemonic);
      tabbedPane.setMnemonicAt(1, cmykMnemonic);
      tabbedPane.setMnemonicAt(2, hsbMnemonic);
      tabbedPane.setMnemonicAt(3, greyMnemonic);
   }

   public boolean requestDefaultColourFocus()
   {
      currentPanel = (SingleColourSelector)
         tabbedPane.getSelectedComponent();

      return currentPanel.requestDefaultColourFocus();
   }

   public void actionPerformed(ActionEvent evt)
   {
      Object source = evt.getSource();

      if (source instanceof ColorButton)
      {
         currentPanel = (SingleColourSelector)
            tabbedPane.getSelectedComponent();

         currentPanel.setPaint(((ColorButton)source).getPaint());
      }
   }

   public void stateChanged(ChangeEvent e)
   {
      JDRPaint paint = currentPanel.getPaint();

      currentPanel = (SingleColourSelector)
         tabbedPane.getSelectedComponent();

      currentPanel.setPaint(paint);
      currentPanel.requestDefaultColourFocus();
   }

   public JDRPaint getPaint()
   {
      currentPanel = (SingleColourSelector)
         tabbedPane.getSelectedComponent();

      return currentPanel.getPaint();
   }

   public void setPaint(JDRPaint paint)
   {
      if (paint instanceof JDRColor)
      {
         tabbedPane.setSelectedComponent(rgbPanel);
         currentPanel = rgbPanel;
      }
      else if (paint instanceof JDRColorCMYK)
      {
         tabbedPane.setSelectedComponent(cmykPanel);
         currentPanel = cmykPanel;
      }
      else if (paint instanceof JDRColorHSB)
      {
         tabbedPane.setSelectedComponent(hsbPanel);
         currentPanel = hsbPanel;
      }
      else if (paint instanceof JDRGray)
      {
         tabbedPane.setSelectedComponent(greyPanel);
         currentPanel = greyPanel;
      }
      else if (paint instanceof JDRShading)
      {
         setPaint(((JDRShading)paint).getStartColor());
         return;
      }
      else
      {
         setPaint(paint.getJDRColor());
         return;
      }

      currentPanel.setPaint(paint);
   }

   public void setEnabled(boolean flag)
   {
      rgbPanel.setEnabled(flag);
      cmykPanel.setEnabled(flag);
      hsbPanel.setEnabled(flag);
      greyPanel.setEnabled(flag);
      tabbedPane.setEnabled(flag);

      for (int i = 0; i < predefined_n; i++)
      {
         predefinedButtons[i].setVisible(flag);
      }
   }

   private ColorRGBPanel rgbPanel;
   private ColorCMYKPanel cmykPanel;
   private ColorHSBPanel hsbPanel;
   private GreyPanel greyPanel;

   private JTabbedPane tabbedPane;

   private SingleColourSelector currentPanel;

   // predefined colour buttons
   private ColorButton[] predefinedButtons;
   private static final int MAX_PREDEFINED=20; 

   private JPanel predefinedPanel;
   private int predefined_n=0;
}
