// File          : CharacterSelector.java
// Description   : Dialog for selecting characters
// Date          : 6th February 2006
// Last Modified : 9th June 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.beans.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog for selecting characters.
 * @author Nicola L C Talbot
 */

public class CharacterSelector extends JDialog
   implements ActionListener,PropertyChangeListener
{
   public CharacterSelector(JFrame parent,
      SymbolSelectorListener symbolListener)
   {
      super(parent, JDRResources.getString("symbol.title"), true);

      init(symbolListener);
   }

   public CharacterSelector(Dialog parent,
      SymbolSelectorListener symbolListener)
   {
      super(parent, JDRResources.getString("symbol.title"), true);

      init(symbolListener);
   }

   private void init(SymbolSelectorListener symbolListener)
   {
      symbolListener_ = symbolListener;

      Box topPanel = Box.createHorizontalBox();

      JLabel label = new JLabel(JDRResources.getString("symbol.text"));
      label.setDisplayedMnemonic(JDRResources.getChar("symbol.text.mnemonic"));

      topPanel.add(label);

      textfield = new JTextField()
      {
         public void paintComponent(Graphics g)
         {
            Graphics2D g2 = (Graphics2D)g;
            g2.setRenderingHints(symbolListener_.getRenderingHints());
            super.paintComponent(g);
         }
      };

      label.setLabelFor(textfield);

      topPanel.add(textfield);

      getContentPane().add(topPanel, "North");

      unicodePanel = new UnicodePanel(symbolListener_, this);

      getContentPane().add(unicodePanel, "East");

      JPanel charPanel = new JPanel();

      int cols = 14;
      int rows = (int)Math.ceil(((double)MAX_BUTTONS)/cols);

      GridLayout layout = new GridLayout(rows, cols);
      charPanel.setLayout(layout);

      buttons = new JButton[MAX_BUTTONS];

      for (int i = 0; i < MAX_BUTTONS; i++)
      {
         buttons[i] = new CharacterButton(i+32, symbolListener);
         charPanel.add(buttons[i]);
         buttons[i].addActionListener(this);
      }

      Dimension dim = buttons[0].getPreferredSize();

      int buttonWidth = dim.width;
      int buttonHeight = dim.height;

      dim.width = dim.width*(cols+1)+layout.getHgap()*(cols-1);
      dim.height = dim.height*rows+layout.getVgap()*(rows-1);

      charPanel.setPreferredSize(dim);

      scrollPane = new JScrollPane(charPanel);

      getContentPane().add(scrollPane, "Center");

      JPanel p2 = new JPanel();

      p2.add(JDRResources.createOkayButton(this));
      p2.add(JDRResources.createCancelButton(this));
      p2.add(JDRResources.createHelpButton("mi:insertsymbol"));

      getContentPane().add(p2, "South");

      dim = scrollPane.getPreferredSize();
      dim.height=400;
      dim.width+=10;
      scrollPane.setSize(dim);

      dim.height = CharacterButton.getMaxHeight();
      textfield.setPreferredSize(dim);

      pack();

      dim = getSize();
      dim.height=400;
      setSize(dim);

      Toolkit tk = Toolkit.getDefaultToolkit();
      Dimension d = tk.getScreenSize();
      int screenHeight = d.height;
      int screenWidth  = d.width;
      setLocation((screenWidth-dim.width)/2,
                  (screenHeight-dim.height)/2);

      JScrollBar vBar = scrollPane.getVerticalScrollBar();
      vBar.setUnitIncrement(buttonHeight);

      JScrollBar hBar = scrollPane.getHorizontalScrollBar();
      hBar.setUnitIncrement(buttonWidth);

      addWindowListener(new WindowAdapter()
         {
            public void windowActivated(WindowEvent e)
            {
               textfield.requestFocusInWindow();
            }
         });

      KeyboardFocusManager focusManager =
         KeyboardFocusManager.getCurrentKeyboardFocusManager();
      focusManager.addPropertyChangeListener(this);
   }

   public void display()
   {
      Font font = symbolListener_.getSymbolButtonFont();

      textfield.setFont(font);
      textfield.setText(symbolListener_.getSymbolText());
      textfield.setCaretPosition(
         symbolListener_.getSymbolCaretPosition());
      textfield.requestFocusInWindow();

      Dimension dim = textfield.getSize();

      setVisible(true);
   }

   public void propertyChange(PropertyChangeEvent e)
   {
      String propertyName = e.getPropertyName();

      if (propertyName.equals("focusOwner")
        && e.getNewValue() instanceof CharacterButton)
      {
         CharacterButton button = (CharacterButton)e.getNewValue();

         unicodePanel.setHexString(button.getHexString());
      }
   }

   public void okay()
   {
      setVisible(false);
      symbolListener_.setText(textfield.getText());
      symbolListener_.setSymbolCaretPosition(
         textfield.getCaretPosition());
      symbolListener_.requestSymbolFocus();
   }

   public void actionPerformed(ActionEvent e)
   {
      Object source = e.getSource();
      String action = e.getActionCommand();

      if (action == null)
      {
         action = "";
      }

      if (action.equals("okay"))
      {
         okay();
      } 
      else if (action.equals("cancel"))
      {
         setVisible(false);
      }
      else if (source instanceof CharacterButton)
      {
         CharacterButton charButton = (CharacterButton)source;

         insertChar(charButton.getChar());

         unicodePanel.setHexString(charButton.getHexString());

         textfield.requestFocusInWindow();
      }
   }

   public void insertChar(char c)
   {
      int pos = textfield.getCaretPosition();
      String str = textfield.getText();
      textfield.setText(str.substring(0,pos)+c+str.substring(pos));
      textfield.setCaretPosition(pos+1);
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "";

      str += "CharacterSelector:"+eol;
      str += "has focus: "+hasFocus()+eol;
      str += "textfield has focus: "+textfield.hasFocus()+eol;

      return str+eol;
   }


   private SymbolSelectorListener symbolListener_;
   private JButton[] buttons;
   private JTextField textfield;

   private UnicodePanel unicodePanel;

   private final int MAX_BUTTONS = 2002;
   private JScrollPane scrollPane;
}
