// File          : BitmapProperties.java
// Description   : Dialog to set bitmap properties
// Date          : 6th February 2006
// Last Modified : 9th June 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;

import java.io.*;
import java.util.*;

import javax.swing.*;
import javax.swing.event.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.filter.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Dialog box to set bitmap properties.
 * @author Nicola L C Talbot
 */

public class BitmapProperties extends JDialog
   implements ActionListener
{
   public BitmapProperties(JpgfDraw application, JFileChooser fc)
   {
      super(application,JDRResources.getString("bitmap.properties.label"),
            true);
      setSize(550,260);
      setLocationRelativeTo(application);
      application_ = application;
      fc_ = fc;

      JPanel p1 = new JPanel();

      p1.setLayout(new GridBagLayout());

      GridBagConstraints gbc = new GridBagConstraints();
      gbc.weightx    = 100;
      gbc.weighty    = 100;
      gbc.gridx      = 0;
      gbc.gridy      = 0;
      gbc.gridwidth  = 1;
      gbc.gridheight = 1;
      gbc.anchor     = GridBagConstraints.EAST;
      gbc.fill       = GridBagConstraints.NONE;

      JLabel filenameLabel = new JLabel(
         JDRResources.getString("label.filename"));
      filenameLabel.setDisplayedMnemonic(
         JDRResources.getChar("label.filename.mnemonic"));

      p1.add(filenameLabel,gbc);

      filename = new JTextField(JDRResources.getString("error.no_filename"));
      filenameLabel.setLabelFor(filename);

      gbc.gridx      = 1;
      gbc.gridwidth  = 3;
      gbc.fill       = GridBagConstraints.HORIZONTAL;
      gbc.anchor     = GridBagConstraints.WEST;
      p1.add(filename,gbc);

      browse = new JButton(JDRResources.getString("browse.label"));
      browse.setMnemonic(JDRResources.getChar("browse.label.mnemonic"));
      browse.addActionListener(this);
      browse.setActionCommand("browse");
      gbc.gridx      = 4;
      gbc.gridwidth  = 1;
      gbc.fill       = GridBagConstraints.NONE;
      p1.add(browse,gbc);

      gbc.gridx      = 0;
      gbc.gridy      = 1;
      gbc.gridwidth  = 1;
      gbc.fill       = GridBagConstraints.NONE;
      gbc.anchor     = GridBagConstraints.EAST;

      JLabel latexlinkLabel = new JLabel(
         JDRResources.getString("bitmap.properties.latexfilename"));
      latexlinkLabel.setDisplayedMnemonic(
         JDRResources.getChar("bitmap.properties.latexfilename.mnemonic"));

      p1.add(latexlinkLabel, gbc);

      latexlinkText =
         new JTextField(JDRResources.getString("error.no_filename"));

      latexlinkLabel.setLabelFor(latexlinkText);

      gbc.gridx      = 1;
      gbc.gridwidth  = 3;
      gbc.fill       = GridBagConstraints.HORIZONTAL;
      gbc.anchor     = GridBagConstraints.WEST;

      p1.add(latexlinkText,gbc);

      gbc.gridx      = 0;
      gbc.gridy      = 2;
      gbc.gridwidth  = 1;
      gbc.anchor     = GridBagConstraints.EAST;
      gbc.fill       = GridBagConstraints.NONE;

      JLabel commandLabel = new JLabel(
         JDRResources.getString("bitmap.properties.command"));
      commandLabel.setDisplayedMnemonic(
         JDRResources.getChar("bitmap.properties.command.mnemonic"));

      p1.add(commandLabel, gbc);

      gbc.gridx      = 1;
      gbc.gridwidth  = 3;
      gbc.fill       = GridBagConstraints.HORIZONTAL;
      gbc.anchor     = GridBagConstraints.WEST;
      latexCommand = new JTextField("\\pgfimage");

      commandLabel.setLabelFor(latexCommand);

      p1.add(latexCommand, gbc);

      gbc.gridy++;
      gbc.gridx      = 0;
      gbc.gridwidth  = 1;
      p1.add(Box.createGlue(), gbc);
      gbc.gridx++;
      p1.add(Box.createGlue(), gbc);
      gbc.gridx++;
      p1.add(Box.createGlue(), gbc);
      gbc.gridx++;
      p1.add(Box.createGlue(), gbc);

      gbc.gridy++;
      gbc.gridx      = 0;
      gbc.gridwidth  = 1;
      gbc.anchor     = GridBagConstraints.NORTHEAST;
      gbc.fill       = GridBagConstraints.NONE;

      JLabel matrixLabel = new JLabel(
         JDRResources.getString("bitmap.properties.matrix"));
      matrixLabel.setDisplayedMnemonic(
         JDRResources.getChar("bitmap.properties.matrix.mnemonic"));

      p1.add(matrixLabel, gbc);

      gbc.gridx++;
      gbc.gridwidth  = 3;
      gbc.gridheight = 2;
      gbc.fill       = GridBagConstraints.HORIZONTAL;
      gbc.anchor     = GridBagConstraints.WEST;

      JPanel matrixPanel = new JPanel();

      matrixPanel.setLayout(new GridLayout(3,3));

      field = new DoubleField[6];

      field[0] = new DoubleField(1);
      matrixPanel.add(field[0]);

      matrixLabel.setLabelFor(field[0]);

      field[2] = new DoubleField(0);
      matrixPanel.add(field[2]);

      field[4] = new DoubleField(0);
      matrixPanel.add(field[4]);

      field[1] = new DoubleField(0);
      matrixPanel.add(field[1]);

      field[3] = new DoubleField(1);
      matrixPanel.add(field[3]);

      field[5] = new DoubleField(0);
      matrixPanel.add(field[5]);

      matrixPanel.add(new JLabel("0"));

      matrixPanel.add(new JLabel("0"));

      matrixPanel.add(new JLabel("1"));

      for (int i = 0; i < 6; i++)
      {
         field[i].setHorizontalAlignment(JTextField.LEFT);
      }

      p1.add(matrixPanel, gbc);

      getContentPane().add(p1, "Center");

      JPanel p2 = new JPanel();

      p2.add(JDRResources.createOkayButton(this));
      p2.add(JDRResources.createCancelButton(this));
      p2.add(JDRResources.createHelpButton("sec:bitmapproperties"));

      getContentPane().add(p2, "South");
   }

   public void initialise()
   {
      JDRFrame frame = application_.getCurrentFrame();
      setProperties(frame.getSelectedBitmap());
      filename.requestFocusInWindow();
      setVisible(true);
   }

   public void browseAction()
   {
      int result = fc_.showOpenDialog(this);

      if (result == JFileChooser.APPROVE_OPTION)
      {
         String newfilename = fc_.getSelectedFile().getAbsolutePath();
         String linkname;
         if (File.separator.equals("\\"))
         {
            StringTokenizer t = new StringTokenizer(newfilename, "\\");
            linkname = t.nextToken();

            while (t.hasMoreTokens())
            {
               linkname += File.separator + t.nextToken();
            }
         }
         else
         {
            linkname = newfilename;
         }

         filename.setText(newfilename);
         latexlinkText.setText(linkname);
      }
   }

   public void setProperties(JDRBitmap db)
   {
      if (db == null)
      {
         JDRResources.internalError(this,
            JDRResources.getString("internal_error.no_bitmap"));
         return;
      }

      bitmap = db;
      filename.setText(bitmap.getFilename());
      latexlinkText.setText(bitmap.getLaTeXLinkName());
      latexCommand.setText(bitmap.latexCommand);

      double[] matrix = new double[6];

      db.getTransformation(matrix);

      for (int i = 0; i < 6; i++)
      {
         field[i].setValue(matrix[i]);
         field[i].setCaretPosition(0);
      }
   }

   public void okay()
   {
      String newfilename = filename.getText();
      String newlatexname = latexlinkText.getText();
      String command = latexCommand.getText();

      double[] matrix = new double[6];

      for (int i = 0; i < 6; i++)
      {
         matrix[i] = field[i].getValue();
      }

      File file = new File(newfilename);

      if (!file.exists())
      {
         JDRResources.error(this, 
                       new String[]
                       {"'"+newfilename+"'",
                       JDRResources.getString("error.not_found")});
      }
      else
      {
         setVisible(false);
         JDRFrame frame = application_.getCurrentFrame();
         frame.setBitmapProperties(bitmap,
            newfilename, newlatexname, command, matrix);
         bitmap=null;
      }
   }

   public void actionPerformed(ActionEvent e)
   {
      String action = e.getActionCommand();

      if (action == null) return;

      if (action.equals("okay"))
      {
         okay();
      } 
      else if (action.equals("cancel"))
      {
         bitmap=null;
         setVisible(false);
      }
      else if (action.equals("browse"))
      {
         browseAction();
      }
   }

   private JpgfDraw application_;
   private DoubleField[] field;
   private JButton browse;
   private JTextField filename, latexlinkText, latexCommand;
   private JDRBitmap bitmap=null;
   private JFileChooser fc_;
}
