// File          : AnglePanel.java
// Description   : Panel for specifying angles
// Date          : 2012-03-06
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw.dialog;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.*;

/**
 * Panel for specifying angles.
 */

public class AnglePanel extends JPanel
   implements ItemListener
{
   public AnglePanel(String label, char mnemonic)
   {
      JLabel jlabel = new JLabel(label);
      jlabel.setDisplayedMnemonic(mnemonic);

      add(jlabel);

      valueField = new DoubleField(0.0F);
      jlabel.setLabelFor(valueField);
      add(valueField);

      jlabel.setLabelFor(valueField);

      String[] units = new String[2];

      units[DEGREES] = JDRResources.getString("rotate.degrees");
      units[RADIANS] = JDRResources.getString("rotate.radians");

      //unitBox = new JComboBox<String>(units); // Java7
      unitBox = new JComboBox(units);

      unitBox.addItemListener(this);

      currentUnit = DEGREES;

      unitBox.setSelectedIndex(currentUnit);

      add(unitBox);
   }

   public void requestValueFocus()
   {
      valueField.requestFocusInWindow();
   }

   public void itemStateChanged(ItemEvent evt)
   {
      Object source = evt.getSource();

      if (evt.getStateChange() == ItemEvent.SELECTED
       && source == unitBox)
      {
         int oldUnit = currentUnit;

         currentUnit = unitBox.getSelectedIndex();

         if (oldUnit != currentUnit)
         {
            double angle = valueField.getValue();

            switch (currentUnit)
            {
               case DEGREES :
                  valueField.setValue(Math.toDegrees(angle));
                  break;
               case RADIANS :
                  valueField.setValue(Math.toRadians(angle));
                  break;
            }
         }
      }
   }

   public double getRadians()
   {
      double angle = valueField.getValue();

      if (unitBox.getSelectedIndex() == DEGREES)
      {
         angle = Math.toRadians(angle);
      }

      return angle;
   }

   public void setValue(double rad)
   {
      double angle = rad;

      if (unitBox.getSelectedIndex() == DEGREES)
      {
         angle = Math.toDegrees(angle);

         // round to 3 decimal places since conversion is likely to
         // be inexact

         double truncated = Math.floor(angle);
         double remainder = 0.001*Math.round((angle - truncated)*1000);

         angle = truncated + remainder;
      }

      valueField.setValue(angle);
   }

   public void setDegrees(double deg)
   {
      unitBox.setSelectedIndex(DEGREES);

      valueField.setValue(deg);
   }

   public void setRadians(double rad)
   {
      unitBox.setSelectedIndex(RADIANS);

      valueField.setValue(rad);
   }

   private DoubleField valueField;
   //private JComboBox<String> unitBox; // Java 7
   private JComboBox unitBox;

   private static final int DEGREES=0, RADIANS=1;

   private int currentUnit;
}
