// File          : SaveJdr.java
// Description   : Runnable to save image in thread.
// Date          : 14th July 2008
// Last Modified : 1 October 2009
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw;

import java.io.*;
import java.awt.*;
import javax.swing.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;
import uk.ac.uea.cmp.nlct.jdrresources.*;

/**
 * Runnable to save image in JDR format.
 */
public class SaveJdr implements Runnable
{
   private SaveJdr()
   {
   }

   public SaveJdr(JDRFrame frame, String filename, float version,
      JDRGroup image)
   {
      this(frame, filename, version, image, null);
   }

   public SaveJdr(JDRFrame frame, String filename, float version,
      JDRGroup image, CanvasSettings settings)
   {
      jdrFrame = frame;
      fileName = filename;
      jdrVersion = version;
      jdrImage = image;
      canvasSettings = settings;
   }

   public void run()
   {
      jdrFrame.setIoInProgress(true);

      boolean success = false;
      DataOutputStream out;

      Cursor oldCursor = jdrFrame.getCursor();
      jdrFrame.setCursor(
         Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

      JpgfDraw app = jdrFrame.getApplication();

      app.showMessageFrame(
         JDRResources.getStringWithValue("info.saving", fileName));
      app.setStatusInfo(
         JDRResources.getStringWithValue("info.saving", fileName));
      try
      {
         out = new DataOutputStream(new FileOutputStream(fileName));
      }
      catch (IOException e)
      {
         JDRResources.error(jdrFrame, new String[]
         {JDRResources.getStringWithValue("error.io.open", fileName),
          e.getMessage()});

         jdrFrame.setIoInProgress(false);
         app.hideMessageFrame();

         jdrFrame.setCursor(oldCursor);
         app.setTool(jdrFrame.currentTool());
         return;
      }

      try
      {
         if (canvasSettings != null)
         {
            JDR.save(jdrImage, canvasSettings, out, jdrVersion);
         }
         else
         {
            JDR.save(jdrImage, out, jdrVersion);
         }

         success = true;
      }
      catch (IOException e)
      {
         JDRResources.error(jdrFrame, new String[]
         {JDRResources.getString("error.io.save"),
          e.getMessage()});
      }
      catch (Exception e)
      {
         JDRResources.internalError(jdrFrame, e);
      }

      jdrFrame.setIoInProgress(false);
      app.hideMessageFrame();

      try
      {
         out.close();
      }
      catch (IOException e)
      {
         JDRResources.error(jdrFrame, new String[]
         {JDRResources.getString("error.io.close"),
          e.getMessage()});
      }

      if (success)
      {
         jdrFrame.setFilename(fileName);
         jdrFrame.markAsSaved();

         jdrFrame.addRecentFile(fileName);
      }

      jdrFrame.setCursor(oldCursor);
      app.setTool(jdrFrame.currentTool());
   }

   private String fileName;
   private float jdrVersion;
   private JDRGroup jdrImage;
   private JDRFrame jdrFrame;
   private CanvasSettings canvasSettings;
}
