// File        : RulerPanel.java
// Description : Provides ruler panel for canvas
// Date        : 5th June 2008
// Author      : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import uk.ac.uea.cmp.nlct.jdr.*;

/**
 * Provides ruler for canvas.
 */
public class RulerPanel extends JPanel
   implements SwingConstants,MouseMotionListener
{
   public RulerPanel(int dir, int w, int h, JDRCanvas canvas)
   {
      direction = dir;
      canvas_ = canvas;

      setPreferredSize(new Dimension(w, h));

      setBorder(BorderFactory.createEtchedBorder());
      addMouseListener(new MouseAdapter()
         {
            public void mousePressed(MouseEvent evt)
            {
               int x = evt.getX();
               int y = evt.getY();
            }

            public void mouseClicked(MouseEvent evt)
            {
               int x = evt.getX();
               int y = evt.getY();
            }
         });
      addMouseMotionListener(this);

      oldP = new Point2D.Double(-MARKER_WIDTH, -MARKER_HEIGHT);

      mag_ = 1.0;
   }

   public void update(double x, double y)
   {
      double mag = getNormalizedMagnification();
      oldP.setLocation(mag*x, mag*y);
      repaint();
   }

   public void mouseMoved(MouseEvent e)
   {
      double mag = getNormalizedMagnification();
      Point2D currentPos = canvas_.getNearestTic(e.getX(),e.getY());

      oldP.setLocation(mag*currentPos.getX(), mag*currentPos.getY());
      repaint();
   }

   public void mouseEntered(MouseEvent e)
   {
      double mag = getNormalizedMagnification();
      Point2D currentPos = canvas_.getNearestTic(e.getX(),e.getY());

      oldP.setLocation(mag*currentPos.getX(), mag*currentPos.getY());
      repaint();
   }

   public void mouseDragged(MouseEvent e)
   {
      double mag = getNormalizedMagnification();
      Point2D currentPos = canvas_.getNearestTic(e.getX(),e.getY());

      oldP.setLocation(mag*currentPos.getX(), mag*currentPos.getY());
      repaint();
   }

   public void paintComponent(Graphics g)
   {
      super.paintComponent(g);
      double mag = getNormalizedMagnification();

      Dimension d = getPreferredSize();

      JDRGrid grid = canvas_.getGrid();

      Point2D major = grid.getMajorTicDistance();
      Point2D minor = grid.getMinorTicDistance();

      JDRUnit unit = grid.getMainUnit();

      double label;

      if (direction == HORIZONTAL)
      {
         label = unit.fromBp(major.getX());

         d.width = (int)(d.width/mag);

         int i = 0;
         int x = 0;

         if (minor.getX() > 0)
         {
            while (x <= d.width)
            {
               g.drawLine((int)(mag*x), 0, 
                          (int)(mag*x), (d.height*4)/10);
               i++;
               x = (int)(i*minor.getX());
            }
         }

         i = 0;
         x = 0;

         while (x <= d.width)
         {
            g.drawLine((int)(mag*x), 0, (int)(mag*x), (d.height*8)/10);
            g.drawString(""+i*label, (int)(mag*x)+2, (d.height*8)/10);
            i++;

            x = (int)(i*major.getX());
         }
      }
      else
      {
         label = unit.fromBp(major.getY());

         d.height = (int)(d.height/mag);
         int i = 0;
         int y = 0;

         if (minor.getY() > 0)
         {
            while (y <= d.height)
            {
               g.drawLine(0, (int)(mag*y), 
                         (d.width*4)/10, (int)(mag*y));
               i++;
               y = (int)(i*minor.getY());
            }
         }

         i = 0;
         y = 0;

         while (y <= d.height)
         {
            g.drawLine(0, (int)(mag*y), (d.width*8)/10, (int)(mag*y));
            g.drawString(""+i*label, 2, (int)(mag*y));
            i++;

            y = (int)(i*major.getY());
         }
      }

      drawMarker(g, oldP.getX(), oldP.getY());
   }

   public void drawMarker(Graphics g, double x, double y)
   {
      if (direction == SwingConstants.HORIZONTAL)
      {
         Polygon p = new Polygon();
         p.addPoint((int)Math.round(x-MARKER_WIDTH/2),0);
         p.addPoint((int)Math.round(x+MARKER_WIDTH/2),0);
         p.addPoint((int)Math.round(x), MARKER_HEIGHT);
         g.fillPolygon(p);
      }
      else
      {
         Polygon p = new Polygon();
         p.addPoint(0,(int)Math.round(y-MARKER_WIDTH/2));
         p.addPoint(0,(int)Math.round(y+MARKER_WIDTH/2));
         p.addPoint(MARKER_HEIGHT, (int)Math.round(y));
         g.fillPolygon(p);
      }
   }

   public double getMagnification()
   {
      return mag_;
   }

   public double getNormalizedMagnification()
   {
      return mag_*JDRUnit.getNormalizingFactor();
   }

   public void setMagnification(double value)
   {
      mag_ = value;
   }

   private int direction;
   public final static int MARKER_WIDTH=10; 
   public final static int MARKER_HEIGHT=10; 

   private Point2D oldP;
   private JDRCanvas canvas_;

   private double mag_;
}
