// File          : LoadEps.java
// Description   : Runnable to load image in thread.
// Date          : 14th July 2008
// Last Modified : 14th July 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import javax.swing.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;
import uk.ac.uea.cmp.nlct.jdrresources.*;

/**
 * Runnable to load image in EPS format. (This is still
 * experimental.)
 */
public class LoadEps implements Runnable
{
   private LoadEps()
   {
   }

   public LoadEps(JDRFrame frame, File file, double normalsize)
   {
      jdrFrame = frame;
      normalSize = normalsize;
      epsFile = file;
   }

   public void run()
   {
      jdrFrame.setIoInProgress(true);
      JDRGroup image=null;

      JDRGuiMessage mess = null;

      if (JDR.messageSystem instanceof JDRGuiMessage)
      {
         mess = (JDRGuiMessage)JDR.messageSystem;

         mess.displayMessages();
      }

      JpgfDraw app = jdrFrame.getApplication();

      Cursor oldCursor = jdrFrame.getCursor();
      jdrFrame.setCursor(
         Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

      app.setStatusInfo(
         JDRResources.getStringWithValue("info.loading",
            epsFile.getAbsolutePath()));

      BufferedReader in;

      try
      {
         in = new BufferedReader(new FileReader(epsFile));
      }
      catch (IOException e)
      {
         JDRResources.error(jdrFrame, new String[]
            {JDRResources.getStringWithValue("error.io.open",
                epsFile.getAbsolutePath()),
            e.getMessage()});

         jdrFrame.setCursor(oldCursor);
         app.setTool(jdrFrame.currentTool());
         jdrFrame.setIoInProgress(false);

         if (mess != null)
         {
            mess.hideMessages();
         }

         return;
      }

      String fileName = epsFile.getAbsolutePath();

      int index = fileName.lastIndexOf(".");

      if (index != -1)
      {
         fileName = fileName.substring(0, index);
      }

      try
      {
         image = EPS.load(in, fileName+"Img", 1, normalSize);

         JDR.messageSystem.messageln("Image loaded");

         if (image.anyDraftBitmaps())
         {
            JDRResources.warning(jdrFrame,
               JDRResources.getString("warning.draft_bitmaps"));
         }
      }
      catch (IOException e)
      {
         JDR.messageSystem.messageln("Failed to load EPS image.");
         JDRResources.error(jdrFrame, new String[]
         {JDRResources.getString("error.io.load"),e.getMessage()});
      }
      catch (InvalidFormatException e)
      {
         JDR.messageSystem.messageln("Failed to load EPS image. (This function is still experimental)");
         JDRResources.error(jdrFrame, e);
      }
      catch (NoninvertibleTransformException e)
      {
         JDR.messageSystem.messageln("Failed to load EPS image.");
         JDRResources.error(jdrFrame, e);
      }
      catch (OutOfMemoryError e)
      {
         JDR.messageSystem.messageln("Failed to load EPS image: out of memory");
         JDRResources.error(jdrFrame,
            JDRResources.getString("error.out_of_memory"));
      }

      try
      {
         in.close();
      }
      catch (IOException e)
      {
         JDRResources.error(jdrFrame, new String[]
         {JDRResources.getString("error.io.close"),
         e.getMessage()});
      }

      if (image != null)
      {
         jdrFrame.setImage(image);
         jdrFrame.setFilename(fileName);
         jdrFrame.markAsModified();
      }

      jdrFrame.setCursor(oldCursor);
      app.setTool(jdrFrame.currentTool());
      jdrFrame.setIoInProgress(false);

      if (mess != null)
      {
         mess.hideMessages();
      }
   }

   private File epsFile;
   private JDRFrame jdrFrame;
   private double normalSize;
}
