// File          : LoadAjr.java
// Description   : Runnable to load image in thread.
// Date          : 14th July 2008
// Last Modified : 1 October 2009
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw;

import java.io.*;
import java.awt.*;
import javax.swing.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;
import uk.ac.uea.cmp.nlct.jdrresources.*;

/**
 * Runnable to load image in JDR format.
 */
public class LoadAjr implements Runnable
{
   private LoadAjr()
   {
   }

   public LoadAjr(JDRFrame frame, String filename)
   {
      jdrFrame = frame;
      fileName = filename;
   }

   public void run()
   {
      jdrFrame.setIoInProgress(true);
      JDRGroup image=null;

      JpgfDraw app = jdrFrame.getApplication();

      Cursor oldCursor = jdrFrame.getCursor();
      jdrFrame.setCursor(
         Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

      app.showMessageFrame(
        JDRResources.getStringWithValue("info.loading", fileName));

      BufferedReader in;

      try
      {
         in = new BufferedReader(new FileReader(fileName));
      }
      catch (IOException e)
      {
         JDRResources.error(jdrFrame, new String[]
            {JDRResources.getStringWithValue("error.io.open",
                fileName),
            e.getMessage()});

         jdrFrame.setIoInProgress(false);
         app.hideMessageFrame();

         return;
      }

      try
      {
         CanvasSettings s = (CanvasSettings)jdrFrame.settings.clone();

         image = AJR.load(in, s);
         jdrFrame.applySettingsIfRequired(s);

         float versionNum = AJR.getLastLoadedVersion();

         if (jdrFrame.warnOnOldJdr())
         {
            if (versionNum < AJR.currentVersion)
            {
               JDRResources.warning(jdrFrame,
                  JDRResources.getStringWithValue("warning.load.ajr",
                     ""+versionNum));
            }
         }

         if (image.anyDraftBitmaps())
         {
            JDRResources.warning(jdrFrame,
               JDRResources.getString("warning.draft_bitmaps"));
         }

      }
      catch (IOException e)
      {
         JDRResources.error(jdrFrame, new String[]
         {JDRResources.getString("error.io.load"),e.getMessage()});
      }
      catch (InvalidFormatException e)
      {
         JDRResources.error(jdrFrame, e);
      }
      catch (Exception e)
      {
         JDRResources.internalError(jdrFrame, e);
      }

      jdrFrame.setCursor(oldCursor);
      jdrFrame.setIoInProgress(false);
      app.hideMessageFrame();

      try
      {
         in.close();
      }
      catch (IOException e)
      {
         JDRResources.error(jdrFrame, new String[]
         {JDRResources.getString("error.io.close"),
         e.getMessage()});
      }

      if (image != null)
      {
         jdrFrame.setImage(image);
         jdrFrame.setFilename(fileName);
         jdrFrame.addRecentFile(fileName);

         if (JDR.bitmapReplaced)
         {
            jdrFrame.markAsModified();
         }
      }

      app.setTool(jdrFrame.currentTool());
   }

   private String fileName;
   private JDRFrame jdrFrame;
}
