// File          : JpgfDraw.java
// Purpose       : Drawing application
// Date          : 1st February 2006
// Last Modified : 25 July 2010
// Version       : 0.5.6b
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw;

import java.io.*;
import java.util.*;
import java.text.*;
import java.net.*;  

import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*;
import java.awt.geom.*;
import java.awt.print.*;
import java.awt.image.*;
import java.awt.font.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.border.*;
import javax.swing.undo.*;
import javax.swing.filechooser.FileFilter;
import javax.swing.TransferHandler;
import javax.swing.plaf.basic.*;
import javax.print.*;
import javax.print.attribute.*;
import javax.print.attribute.standard.*;
import javax.imageio.*;
import javax.help.*;  

import java.beans.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.filter.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.dialog.*;

public class JpgfDraw extends JFrame
   implements ActionListener,
              InternalFrameListener,
              UndoableEditListener,
              JDRApp
{
   // set up GUI
   public JpgfDraw(CanvasSettings settings, String filename)
   {
      super(JpgfDraw.appName);

      setIconImage(JDRResources.appIcon("jdrlogosmall.png").getImage());

      setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

      addWindowListener(new WindowAdapter()
         {
            public void windowActivated(WindowEvent evt)
            {
               JDRFrame frame =
                  (JDRFrame)theDesktop.getSelectedFrame();

               if (frame != null) frame.requestFocusInWindow();
            }

            public void windowClosing(WindowEvent evt)
            {
               quit();
            }
         });

      // set dimensions and location
      Toolkit tk = Toolkit.getDefaultToolkit();
      Dimension d = tk.getScreenSize();
      int screenHeight = d.height;
      int screenWidth  = d.width;
      int width  = 3*screenHeight/4;
      int height = screenWidth/2;
      setSize(width,height);
      setLocation((screenWidth-width)/2, (screenHeight-height)/2);

      // set up available font families

      GraphicsEnvironment env = 
         GraphicsEnvironment.getLocalGraphicsEnvironment();
      availableFontFamilies = env.getAvailableFontFamilyNames();

      // set up annotation font

      try
      {
         JDRCompleteObject.annoteFont = new Font(
            JDRResources.getString("font.annote.name"), 0,
            JDRResources.getInt("font.annote.size"));
      }
      catch (NumberFormatException e)
      {
         JDRResources.internalError(this,
            JDRResources.getStringWithValue(
               "internal_error.integer_key",
               "font.annote.size"));
      }

      if (!disablePrint)
      {
         // set up print request attribute set
         printRequestAttributeSet = new HashPrintRequestAttributeSet();

         getPrintService(settings.paper);
      }
      else
      {
         printRequestAttributeSet = null;
      }

      // initialise help set
      initializeHelp(this);

      // load list of recent files
      loadRecentFiles();

      Container contentPane = getContentPane();

      // create and add status bar
      statusBar = new StatusBar(getPreferredSize().width,20);
      contentPane.add(statusBar, "South");

      // create and add horizontal tool bar
      toolBar = Box.createHorizontalBox();
      toolBar.setSize(width, 100);

      SlidingToolBar hSlidingBar 
         = new SlidingToolBar(toolBar, SlidingToolBar.HORIZONTAL);

      contentPane.add(hSlidingBar, "North");

      // create and add vertical tool bar

      sidePanel = Box.createVerticalBox();
      SlidingToolBar vSlidingBar = 
         new SlidingToolBar(sidePanel, SlidingToolBar.VERTICAL);

      contentPane.add(vSlidingBar, "West");

      sidePanel.setPreferredSize(new Dimension(32, height));

      startup.setInfo(JDRResources.getString("message.init_menus"));

      startup.setDeterminate(162);

      writeLogAction = new WriteLogAction();
      dumpAllAction = new DumpAllAction();

      addGeneralAction(this);

      // create menu bar, menu and menu item
      JMenuBar mbar = new JMenuBar();
      setJMenuBar(mbar);

      // File Menu

      fileM = new JMenu(JDRResources.getString("file.label"));
      fileM.setMnemonic(JDRResources.getChar("file.mnemonic"));
      mbar.add(fileM);

      startup.incProgress();
      startup.setInfo(fileM.getText());

      // New

      newButtonItem = new JDRButtonItem("file.new", "new",
         KeyStroke.getKeyStroke(KeyEvent.VK_N, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.new"),
         toolBar, fileM);

      startup.incProgress();
      startup.setInfo(fileM.getText()+"->"+newButtonItem.getText());

      hSlidingBar.setUnitIncrement( 
         newButtonItem.getButtonPreferredWidth());

      // Open

      openButtonItem = new JDRButtonItem("file.open", "open",
         KeyStroke.getKeyStroke(KeyEvent.VK_O, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.open"),
         toolBar, fileM);

      startup.setInfo(fileM.getText()+"->"+openButtonItem.getText());

      // open dialog box

      openjdrFC = new JDRFileChooser();
      addGeneralAction(openjdrFC);

      openjdrFC.setCurrentDirectory(new File(appSettings.startDir));

      jdrAjrFileFilter = new JdrAjrFileFilter(
         JDRResources.getString("filter.jdrajr"));

      openjdrFC.addChoosableFileFilter(jdrAjrFileFilter);
      openjdrFC.setFileFilter(jdrAjrFileFilter);

      // save as dialog box

      savejdrFC = new JDRFileChooser();
      addGeneralAction(savejdrFC);

      savejdrFC.setCurrentDirectory(new File(appSettings.startDir));
      savejdrFC.setAcceptAllFileFilterUsed(false);

      jdrFileFilter = new JdrFileFilter(
         JDRResources.getString("filter.jdr"));
      savejdrFC.addChoosableFileFilter(jdrFileFilter);

      ajrFileFilter = new AjrFileFilter(
         JDRResources.getString("filter.ajr"));
      savejdrFC.addChoosableFileFilter(ajrFileFilter);

      int version = (int)Math.round(10.0*(JDR.currentVersion-0.1f))-10;

      int numOldVersions = version+1;

      oldJdrFileFilter = new JdrFileFilter[numOldVersions];
      oldAjrFileFilter = new AjrFileFilter[numOldVersions];

      for (int i = numOldVersions-1; i >= 0; i--)
      {
         float thisVersion = (float)(i*0.1+1.0);

         oldJdrFileFilter[i] = new JdrFileFilter(
         JDRResources.getStringWithValue(
            "filter.version.jdr", ""+thisVersion), thisVersion);
         savejdrFC.addChoosableFileFilter(oldJdrFileFilter[i]);

         oldAjrFileFilter[i] = new AjrFileFilter(
         JDRResources.getStringWithValue(
            "filter.version.ajr", ""+thisVersion), thisVersion);
         savejdrFC.addChoosableFileFilter(oldAjrFileFilter[i]);
      }

      savejdrFC.setFileFilter(jdrFileFilter);

      jdrFCAddExt = new JCheckBox(
         JDRResources.getString("fc.append_ext"));
      jdrFCAddExt.setSelected(true);
      savejdrFC.setAccessory(jdrFCAddExt);

      startup.incProgress();


      // Recent Files

      recentM = new JMenu(JDRResources.getString("file.recent"));
      recentM.setMnemonic(
         JDRResources.getChar("file.recent.mnemonic"));
      fileM.add(recentM);

      recentM.addMenuListener(new MenuListener()
         {
            public void menuSelected(MenuEvent evt)
            {
               setRecentFiles();
            }
            public void menuDeselected(MenuEvent evt)
            {
            }
            public void menuCanceled(MenuEvent evt)
            {
            }
         });

      startup.incProgress();
      startup.setInfo(fileM.getText()+"->"+recentM.getText());

      // Image Description

      imageDescriptionItem = new JMenuItem(
         JDRResources.getString("file.description"),
         JDRResources.getChar("file.description.mnemonic"));
      imageDescriptionItem.addActionListener(this);
      fileM.add(imageDescriptionItem);

      imageDescriptionDialog
         = new DescriptionDialogBox(this, "sec:imagedescription");

      addGeneralAction(imageDescriptionDialog);

      startup.incProgress();
      startup.setInfo(fileM.getText()+"->"+imageDescriptionItem.getText());

      // Save

      saveButtonItem = new JDRButtonItem("file.save", "save",
         KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.save"),
         toolBar, fileM);

      startup.incProgress();
      startup.setInfo(fileM.getText()+"->"+saveButtonItem.getText());

      // Save As

      saveAsItem =
         new JMenuItem(JDRResources.getString("file.save_as"),
         JDRResources.getChar("file.save_as.mnemonic"));
      saveAsItem.addActionListener(this);
      fileM.add(saveAsItem);

      startup.incProgress();
      startup.setInfo(fileM.getText()+"->"+saveAsItem.getText());

      // Export

      exportItem = new JMenuItem(
         JDRResources.getString("file.export"),
         JDRResources.getChar("file.export.mnemonic"));
      exportItem.addActionListener(this);
      fileM.add(exportItem);

      startup.setInfo(fileM.getText()+"->"+exportItem.getText());

      exportFC = new JDRFileChooser();
      addGeneralAction(exportFC);

      exportFC.setDialogTitle(
         JDRResources.getString("export.title"));

      exportFC.setCurrentDirectory(new File(appSettings.startDir));
      exportFC.setAcceptAllFileFilterUsed(false);

      pgfFileFilter = new TeXFileFilter(
         JDRResources.getString("filter.pgf"));
      exportFC.addChoosableFileFilter(pgfFileFilter);

      flfFileFilter = new StyFileFilter(
         JDRResources.getString("filter.flf"));
      exportFC.addChoosableFileFilter(flfFileFilter);

      pgfDocFileFilter = new TeXFileFilter(
         JDRResources.getString("filter.pgfdoc"));
      exportFC.addChoosableFileFilter(pgfDocFileFilter);

      pngFileFilter = new PngFileFilter(
         JDRResources.getString("filter.png"));
      exportFC.addChoosableFileFilter(pngFileFilter);

      epsFileFilter = new EpsFileFilter(
         JDRResources.getString("filter.eps"));
      exportFC.addChoosableFileFilter(epsFileFilter);

      svgFileFilter = new SvgFileFilter(
         JDRResources.getString("filter.svg"));
      exportFC.addChoosableFileFilter(svgFileFilter);

      exportFCAddExt = new JCheckBox(
         JDRResources.getString("fc.append_ext"));
      exportFCAddExt.setSelected(true);
      exportFC.setAccessory(exportFCAddExt);

      exportFC.setFileFilter(pgfFileFilter);

      texFC = new JFileChooser();

      texFC.setCurrentDirectory(new File(appSettings.startDir));
      texFC.setAcceptAllFileFilterUsed(false);
      texFileFilter = new TeXFileFilter(
         JDRResources.getString("filter.tex"));
      texFC.addChoosableFileFilter(texFileFilter);
      texFCAddExt = new JCheckBox(
         JDRResources.getString("fc.append_ext"));
      texFCAddExt.setSelected(true);
      texFC.setAccessory(texFCAddExt);

      addGeneralAction(texFC);

      startup.incProgress();

      // Import

      // not fully implemented

      if (experimentalMode || JDRResources.debugMode)
      {
         importItem = new JMenuItem(
            JDRResources.getString("file.import"),
            JDRResources.getChar("file.import.mnemonic"));
         importItem.addActionListener(this);

         startup.setInfo(fileM.getText()+"->"+importItem.getText());

         fileM.add(importItem);

         importFC = new JDRFileChooser();
         importFC.setDialogTitle(
            JDRResources.getString("import.title"));

         importFC.setCurrentDirectory(
            new File(appSettings.startDir));
         importFC.setAcceptAllFileFilterUsed(false);

         importFC.addChoosableFileFilter(epsFileFilter);

         addGeneralAction(importFC);
      }

      startup.incProgress();

      // Printer Page Setup dialog

      pageDialogItem = new JMenuItem(
         JDRResources.getString("file.page"),
         JDRResources.getChar("file.page.mnemonic"));
      pageDialogItem.addActionListener(this);

      startup.setInfo(fileM.getText()+"->"+pageDialogItem.getText());

      fileM.add(pageDialogItem);

      // Print

      printItem = new JMenuItem(
         JDRResources.getString("file.print"),
         JDRResources.getChar("file.print.mnemonic"));
      printItem.addActionListener(this);

      startup.setInfo(fileM.getText()+"->"+printItem.getText());
      fileM.add(printItem);

      if (disablePrint)
      {
         printItem.setEnabled(false);
      }

      fileM.addSeparator();

      startup.incProgress();

      // Close

      closeItem = new JMenuItem(
         JDRResources.getString("file.close"), 
         JDRResources.getChar("file.close.mnemonic"));
      closeItem.addActionListener(this);
      fileM.add(closeItem);

      startup.setInfo(fileM.getText()+"->"+closeItem.getText());

      discardDB = new DiscardDialogBox(this);

      addGeneralAction(discardDB);

      startup.incProgress();

      // Quit ^Q

      quitItem =
         new JMenuItem(JDRResources.getString("file.quit"),
                       JDRResources.getChar("file.quit.mnemonic"));
      quitItem.addActionListener(this);
      quitItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q,
         InputEvent.CTRL_MASK));
      fileM.add(quitItem);

      startup.setInfo(fileM.getText()+"->"+quitItem.getText());
      startup.incProgress();

      // Edit Menu

      editM = new JMenu(JDRResources.getString("edit.label"));
      editM.setMnemonic(JDRResources.getChar("edit.mnemonic"));
      mbar.add(editM);
      startup.setInfo(editM.getText());

      //initialise undo/redo system

      undoManager = new UndoManager();
      undoSupport = new UndoableEditSupport();
      undoSupport.addUndoableEditListener(this);

      startup.incProgress();

      // Undo

      undoItem = new JMenuItem(JDRResources.getString("edit.undo"),
                               JDRResources.getChar("edit.undo.mnemonic"));
      undoItem.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               Cursor oldCursor = getCursor();
               setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
               undoManager.undo();
               refreshUndoRedo();
               setCursor(oldCursor);
            }
         });
      undoItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F8,0));
      editM.add(undoItem);
      startup.setInfo(editM.getText()+"->"+undoItem.getText());

      startup.incProgress();

      // Redo

      redoItem = new JMenuItem(JDRResources.getString("edit.redo"),
                               JDRResources.getChar("edit.redo.mnemonic"));
      redoItem.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               Cursor oldCursor = getCursor();
               setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
               undoManager.redo();
               refreshUndoRedo();
               setCursor(oldCursor);
            }
         });
      redoItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F9,0));
      editM.add(redoItem);
      startup.setInfo(editM.getText()+"->"+redoItem.getText());

      refreshUndoRedo();

      editM.addSeparator();

      startup.incProgress();

      // Select All

      selectAllButtonItem = new JDRButtonItem("edit.select_all",
         "selectAll",
         GeneralAction.KEY_SELECT_ALL,
         this, JDRResources.getString("tooltip.select_all"),
         toolBar, editM);

      startup.setInfo(editM.getText()+"->"+selectAllButtonItem.getText());
      startup.incProgress();

      // Deselect All

      deselectAllItem = new JMenuItem(
         JDRResources.getString("edit.deselect_all"),
         JDRResources.getChar("edit.deselect_all.mnemonic"));
      deselectAllItem.addActionListener(this);
      deselectAllItem.setAccelerator(GeneralAction.KEY_DESELECT_ALL);
      editM.add(deselectAllItem);

      startup.incProgress();
      startup.setInfo(editM.getText()+"->"+deselectAllItem.getText());

      // Move By

      moveByItem = new JMenuItem(
         JDRResources.getString("edit.moveby"),
         JDRResources.getChar("edit.moveby.mnemonic"));
      moveByItem.addActionListener(this);
      moveByItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_F7, 0));
      editM.add(moveByItem);

      startup.setInfo(editM.getText()+"->"+moveByItem.getText());

      moveByDialog = new MoveByDialogBox(this);

      addGeneralAction(moveByDialog);

      startup.incProgress();

      // Cut

      cutButtonItem = new JDRButtonItem("edit.cut", "cut",
         GeneralAction.KEY_CUT,
         this, JDRResources.getString("tooltip.cut"),
         toolBar, editM);

      startup.setInfo(editM.getText()+"->"+cutButtonItem.getText());
      startup.incProgress();

      // Copy

      copyButtonItem = new JDRButtonItem("edit.copy", "copy",
         GeneralAction.KEY_COPY,
         this, JDRResources.getString("tooltip.copy"),
         toolBar, editM);

      startup.setInfo(editM.getText()+"->"+copyButtonItem.getText());
      startup.incProgress();

      // Paste

      pasteButtonItem = new JDRButtonItem("edit.paste", "paste",
         GeneralAction.KEY_PASTE,
         this, JDRResources.getString("tooltip.paste"),
         toolBar, editM);

      startup.setInfo(editM.getText()+"->"+pasteButtonItem.getText());
      startup.incProgress();

      // Object Description

      objectDescriptionItem = new JMenuItem(
         JDRResources.getString("edit.description"),
         JDRResources.getChar("edit.description.mnemonic"));
      objectDescriptionItem.addActionListener(this);
      editM.add(objectDescriptionItem);

      objectDescriptionDialog
         = new DescriptionDialogBox(this, "sec:objectDesc");

      addGeneralAction(imageDescriptionDialog);

      editM.addSeparator();

      startup.setInfo(editM.getText()+"->"+objectDescriptionItem.getText());
      startup.incProgress();

      // JDRPath submenu

      pathM = new JMenu(JDRResources.getString("edit.path"));
      pathM.setMnemonic(JDRResources.getChar("edit.path.mnemonic"));
      editM.add(pathM);

      startup.setInfo(editM.getText()+"->"+pathM.getText());
      startup.incProgress();

      // Edit JDRPath

      editPathButtonItem = new JDRToggleButtonItem("edit.path.edit",
         "editPath",
         KeyStroke.getKeyStroke(KeyEvent.VK_I, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.edit_path"), false,
         toolBar, pathM);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"+editPathButtonItem.getText());
      startup.incProgress();

      // JDRLine colour
      linePaintItem = new JMenuItem(
         JDRResources.getString("edit.path.line_colour"),
         JDRResources.getChar("edit.path.line_colour.mnemonic"));

      linePaintChooserBox = new LinePaintSelector(this);
      addGeneralAction(linePaintChooserBox);

      linePaintItem.addActionListener(this);
      pathM.add(linePaintItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"+linePaintItem.getText());
      startup.incProgress();

      // Fill colour
      fillPaintItem = new JMenuItem(
         JDRResources.getString("edit.path.fill_colour"),
         JDRResources.getChar("edit.path.fill_colour.mnemonic"));

      fillPaintChooserBox = new FillPaintSelector(this);
      addGeneralAction(fillPaintChooserBox);

      fillPaintItem.addActionListener(this);
      pathM.add(fillPaintItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"+fillPaintItem.getText());
      startup.incProgress();

      // Edit JDRLine Style sub menu
      lineStyleM = new JMenu(
         JDRResources.getString("edit.path.style"));
      lineStyleM.setMnemonic(
         JDRResources.getChar("edit.path.style.mnemonic"));
      pathM.add(lineStyleM);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"+lineStyleM.getText());
      startup.incProgress();

      // all styles
      lineStyleItem = new JMenuItem(
         JDRResources.getString("edit.path.style.all"),
         JDRResources.getChar("edit.path.style.all.mnemonic"));

      lineStyleChooserBox = new LineStyleSelector(this);
      addGeneralAction(lineStyleChooserBox);

      lineStyleItem.addActionListener(this);
      lineStyleItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_Y,
         InputEvent.CTRL_MASK));
      lineStyleM.add(lineStyleItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+lineStyleItem.getText());
      startup.incProgress();

      // all markers
      allMarkersItem = new JMenuItem(
         JDRResources.getString("edit.path.style.all_markers"),
         JDRResources.getChar("edit.path.style.all_markers.mnemonic"));
      allMarkersItem.setToolTipText(
         JDRResources.getString("tooltip.all_markers"));

      allMarkersChooserBox = new ArrowStyleSelector(this,
         ArrowStylePanel.ALL);
      addGeneralAction(allMarkersChooserBox);

      allMarkersItem.addActionListener(this);
      lineStyleM.add(allMarkersItem);

      lineStyleM.addSeparator();

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+allMarkersItem.getText());
      startup.incProgress();

      // line width

      lineWidthItem = new JMenuItem(
         JDRResources.getString("edit.path.style.linewidth"),
         JDRResources.getChar("edit.path.style.linewidth.mnemonic"));

      lineWidthItem.addActionListener(this);
      lineStyleM.add(lineWidthItem);
      lineWidthChooserBox = new LineWidthSelector(this);
      addGeneralAction(lineWidthChooserBox);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+lineWidthItem.getText());
      startup.incProgress();

      // dash pattern

      dashItem = new JMenuItem(
         JDRResources.getString("edit.path.style.dashpattern"),
         JDRResources.getChar("edit.path.style.dashpattern.mnemonic"));

      dashItem.addActionListener(this);
      dashPatternChooserBox = new DashPatternSelector(this);
      addGeneralAction(dashPatternChooserBox);

      lineStyleM.add(dashItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+dashItem.getText());
      startup.incProgress();

      // Cap style sub menu

      JMenu capM = new JMenu(
         JDRResources.getString("edit.path.style.capstyle"));
      capM.setMnemonic(
         JDRResources.getChar("edit.path.style.capstyle.mnemonic"));

      lineStyleM.add(capM);

      capM.addMenuListener(new MenuListener()
         {
            public void menuSelected(MenuEvent evt)
            {
               JDRFrame frame = getCurrentFrame();
               if (frame==null) return;

               JDRBasicStroke stroke = frame.getSelectedStroke();

               if (stroke != null)
               {
                  switch (stroke.getCapStyle())
                  {
                     case BasicStroke.CAP_BUTT:
                        capButtItem.setSelected(true);
                     break;
                     case BasicStroke.CAP_ROUND:
                        capRoundItem.setSelected(true);
                     break;
                     case BasicStroke.CAP_SQUARE:
                        capSquareItem.setSelected(true);
                     break;
                  }
               }
            }
            public void menuDeselected(MenuEvent evt)
            {
            }
            public void menuCanceled(MenuEvent evt)
            {
            }
         });


      ButtonGroup capGroup = new ButtonGroup();

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+capM.getText());
      startup.incProgress();

      // butt cap

      capButtItem = new JRadioButtonMenuItem(
         JDRResources.getString("edit.path.style.capstyle.butt"));
      capButtItem.setMnemonic(
         JDRResources.getChar("edit.path.style.capstyle.butt.mnemonic"));

      capButtItem.addActionListener(this);
      capGroup.add(capButtItem);
      capM.add(capButtItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+capM.getText()+"->"+capButtItem.getText());
      startup.incProgress();

      // round cap

      capRoundItem = new JRadioButtonMenuItem(
         JDRResources.getString("edit.path.style.capstyle.round"));
      capRoundItem.setMnemonic(
         JDRResources.getChar("edit.path.style.capstyle.round.mnemonic"));

      capRoundItem.addActionListener(this);
      capGroup.add(capRoundItem);
      capM.add(capRoundItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+capM.getText()+"->"+capRoundItem.getText());
      startup.incProgress();

      // square cap

      capSquareItem = new JRadioButtonMenuItem(
         JDRResources.getString("edit.path.style.capstyle.square"));
      capSquareItem.setMnemonic(
         JDRResources.getChar("edit.path.style.capstyle.square.mnemonic"));

      capSquareItem.addActionListener(this);
      capGroup.add(capSquareItem);
      capM.add(capSquareItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+capM.getText()+"->"+capSquareItem.getText());
      startup.incProgress();

      // Join Style

      joinItem = new JMenuItem(
          JDRResources.getString("edit.path.style.joinstyle"),
          JDRResources.getChar("edit.path.style.joinstyle.mnemonic"));

      lineStyleM.add(joinItem);

      joinStyleChooserBox = new JoinStyleSelector(this);
      addGeneralAction(joinStyleChooserBox);

      joinItem.addActionListener(this);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+joinItem.getText());
      startup.incProgress();

      // start arrow style
      startArrowItem = new JMenuItem(
         JDRResources.getString("edit.path.style.startarrow"),
         JDRResources.getChar("edit.path.style.startarrow.mnemonic"));

      startArrowChooserBox = new ArrowStyleSelector(this,
         ArrowStylePanel.START);
      addGeneralAction(startArrowChooserBox);

      startArrowItem.addActionListener(this);
      lineStyleM.add(startArrowItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+startArrowItem.getText());
      startup.incProgress();

      // mid arrow style
      midArrowItem = new JMenuItem(
         JDRResources.getString("edit.path.style.midarrow"),
         JDRResources.getChar("edit.path.style.midarrow.mnemonic"));

      midArrowChooserBox = new ArrowStyleSelector(this,
         ArrowStylePanel.MID);
      addGeneralAction(midArrowChooserBox);

      midArrowItem.addActionListener(this);
      lineStyleM.add(midArrowItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+midArrowItem.getText());
      startup.incProgress();

      // end arrow style

      endArrowItem = new JMenuItem(
         JDRResources.getString("edit.path.style.endarrow"),
         JDRResources.getChar("edit.path.style.endarrow.mnemonic"));

      endArrowChooserBox = new ArrowStyleSelector(this,
         ArrowStylePanel.END);
      addGeneralAction(endArrowChooserBox);

      endArrowItem.addActionListener(this);
      lineStyleM.add(endArrowItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+endArrowItem.getText());
      startup.incProgress();

      // winding rule sub menu

      JMenu windingM = new JMenu(
         JDRResources.getString("edit.path.style.windingrule"));
      windingM.setMnemonic(
         JDRResources.getChar("edit.path.style.windingrule.mnemonic"));

      lineStyleM.add(windingM);

      ButtonGroup windingGroup = new ButtonGroup();

      windingM.addMenuListener(new MenuListener()
         {
            public void menuSelected(MenuEvent evt)
            {
               JDRFrame frame = getCurrentFrame();
               if (frame==null) return;

               JDRStroke stroke = frame.getSelectedStroke();

               if (stroke != null)
               {
                  switch (stroke.getWindingRule())
                  {
                     case GeneralPath.WIND_EVEN_ODD:
                        windingEvenOddItem.setSelected(true);
                     break;
                     case GeneralPath.WIND_NON_ZERO:
                        windingNonZeroItem.setSelected(true);
                     break;
                  }
               }
            }
            public void menuDeselected(MenuEvent evt)
            {
            }
            public void menuCanceled(MenuEvent evt)
            {
            }
         });

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+windingM.getText());
      startup.incProgress();

      // even-odd

      windingEvenOddItem = new JRadioButtonMenuItem(
         JDRResources.getString("edit.path.style.windingrule.evenodd"));
      windingEvenOddItem.setMnemonic(
         JDRResources.getChar("edit.path.style.windingrule.evenodd.mnemonic"));

      windingGroup.add(windingEvenOddItem);
      windingM.add(windingEvenOddItem);
      windingEvenOddItem.addActionListener(this);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+windingM.getText()+"->"
        +windingEvenOddItem.getText());
      startup.incProgress();

      // non zero

      windingNonZeroItem = new JRadioButtonMenuItem(
         JDRResources.getString("edit.path.style.windingrule.nonzero"));
      windingNonZeroItem.setMnemonic(
         JDRResources.getChar("edit.path.style.windingrule.nonzero.mnemonic"));

      windingGroup.add(windingNonZeroItem);
      windingM.add(windingNonZeroItem);
      windingNonZeroItem.addActionListener(this);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +lineStyleM.getText()+"->"+windingM.getText()+"->"
        +windingNonZeroItem.getText());
      startup.incProgress();

      // JDRText submenu

      textM = new JMenu(JDRResources.getString("edit.text"));
      textM.setMnemonic(JDRResources.getChar("edit.text.mnemonic"));
      editM.add(textM);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText());
      startup.incProgress();

      // Edit JDRText

      editTextItem = new JMenuItem(
         JDRResources.getString("edit.text.edit"), 
         JDRResources.getChar("edit.text.edit.mnemonic"));

      editTextBox = new TextSelector(this);
      addGeneralAction(editTextBox);

      editTextItem.addActionListener(this);
      editTextItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_I,
         InputEvent.CTRL_MASK+InputEvent.SHIFT_MASK));
      textM.add(editTextItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+editTextItem.getText());
      startup.incProgress();

      // JDRText colour

      textPaintItem = new JMenuItem(
         JDRResources.getString("edit.text.colour"),
         JDRResources.getChar("edit.text.colour.mnemonic"));

      textPaintChooserBox = new TextPaintSelector(this);
      addGeneralAction(textPaintChooserBox);

      textPaintItem.addActionListener(this);
      textM.add(textPaintItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+textPaintItem.getText());
      startup.incProgress();

      // JDRText transformation matrix

      textMatrixItem = new JMenuItem(
         JDRResources.getString("edit.text.matrix"),
         JDRResources.getChar("edit.text.matrix.mnemonic"));

      textMatrixDialog = new SetTransformDialogBox(this);
      addGeneralAction(textMatrixDialog);

      textMatrixItem.addActionListener(this);
      textM.add(textMatrixItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+textMatrixItem.getText());
      startup.incProgress();

      // Change Font

      fontStyleM = new JMenu(
         JDRResources.getString("edit.text.font"));
      fontStyleM.setMnemonic(
         JDRResources.getChar("edit.text.font.mnemonic"));

      textM.add(fontStyleM);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText());
      startup.incProgress();

      // All font styles

      textStyleItem = new JMenuItem(
         JDRResources.getString("edit.text.font.all_styles"),
         JDRResources.getChar("edit.text.font.all_styles.mnemonic"));

      textStyleChooserBox = new FontSelector(this);
      addGeneralAction(textStyleChooserBox);

      textStyleItem.addActionListener(this);
      fontStyleM.add(textStyleItem);

      fontStyleM.addSeparator();

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +textStyleItem.getText());
      startup.incProgress();

      // font family

      fontFamilyItem = new JMenuItem(
         JDRResources.getString("edit.text.font.family"),
         JDRResources.getChar("edit.text.font.family.mnemonic"));

      fontFamilyChooserBox = new FontFamilySelector(this);
      addGeneralAction(fontFamilyChooserBox);

      fontFamilyItem.addActionListener(this);
      fontStyleM.add(fontFamilyItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +fontFamilyItem.getText());
      startup.incProgress();

      // font size

      fontSizeItem = new JMenuItem(
         JDRResources.getString("edit.text.font.size"),
         JDRResources.getChar("edit.text.font.size.mnemonic"));

      fontSizeChooserBox = new FontSizeSelector(this);
      addGeneralAction(fontSizeChooserBox);

      fontSizeItem.addActionListener(this);
      fontStyleM.add(fontSizeItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +fontSizeItem.getText());
      startup.incProgress();

      // font shape

      fontShapeItem = new JMenuItem(
         JDRResources.getString("edit.text.font.shape"),
         JDRResources.getChar("edit.text.font.shape.mnemonic"));

      fontShapeChooserBox = new FontShapeSelector(this);
      addGeneralAction(fontShapeChooserBox);

      fontShapeItem.addActionListener(this);
      fontStyleM.add(fontShapeItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +fontShapeItem.getText());
      startup.incProgress();

      // font series

      fontSeriesItem = new JMenuItem(
         JDRResources.getString("edit.text.font.series"),
         JDRResources.getChar("edit.text.font.series.mnemonic"));

      fontSeriesChooserBox = new FontSeriesSelector(this);
      addGeneralAction(fontSeriesChooserBox);

      fontSeriesItem.addActionListener(this);
      fontStyleM.add(fontSeriesItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +fontSeriesItem.getText());
      startup.incProgress();

      // text anchor

      JMenu fontAnchorM = new JMenu(
         JDRResources.getString("edit.text.font.anchor"));
      fontAnchorM.setMnemonic(
         JDRResources.getChar("edit.text.font.anchor.mnemonic"));

      fontStyleM.add(fontAnchorM);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +fontAnchorM.getText());
      startup.incProgress();

      // both

      fontAnchorItem = new JMenuItem(
         JDRResources.getString("edit.text.font.anchor.both"),
         JDRResources.getChar("edit.text.font.anchor.both.mnemonic"));

      fontAnchorChooserBox = new FontAnchorSelector(this);
      addGeneralAction(fontAnchorChooserBox);

      fontAnchorItem.addActionListener(this);
      fontAnchorM.add(fontAnchorItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +fontAnchorM.getText()+"->"+fontAnchorItem.getText());
      startup.incProgress();

      // horizontal anchor setting

      fontHAnchorItem = new JMenuItem(
         JDRResources.getString("edit.text.font.anchor.horizontal"),
         JDRResources.getChar(
            "edit.text.font.anchor.horizontal.mnemonic"));

      fontHAnchorChooserBox = new FontHAnchorSelector(this);
      addGeneralAction(fontHAnchorChooserBox);

      fontHAnchorItem.addActionListener(this);
      fontAnchorM.add(fontHAnchorItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +fontAnchorM.getText()+"->"+fontHAnchorItem.getText());
      startup.incProgress();

      // vertical anchor setting

      fontVAnchorItem = new JMenuItem(
         JDRResources.getString("edit.text.font.anchor.vertical"),
         JDRResources.getChar(
            "edit.text.font.anchor.vertical.mnemonic"));

      fontVAnchorChooserBox = new FontVAnchorSelector(this);
      addGeneralAction(fontVAnchorChooserBox);

      fontVAnchorItem.addActionListener(this);
      fontAnchorM.add(fontVAnchorItem);

      startup.setInfo(editM.getText()+"->"+pathM.getText()+"->"
        +textM.getText()+fontStyleM.getText()
        +fontAnchorM.getText()+"->"+fontVAnchorItem.getText());
      startup.incProgress();

      // Reduce to grey scale

      reduceToGreyItem = new JMenuItem(
         JDRResources.getString("edit.togrey"),
         JDRResources.getChar("edit.togrey.mnemonic"));
      reduceToGreyItem.addActionListener(this);
      editM.add(reduceToGreyItem);

      startup.setInfo(editM.getText()+"->"+reduceToGreyItem.getText());

      // Fade

      fadeItem = new JMenuItem(
         JDRResources.getString("edit.fade"),
         JDRResources.getChar("edit.fade.mnemonic"));
      fadeItem.addActionListener(this);
      fadeItem.setActionCommand("fade");
      editM.add(fadeItem);

      fadeDialog = new FadeDialogBox(this);

      editM.addSeparator();

      startup.setInfo(editM.getText()+"->"+fadeItem.getText());
      startup.incProgress();

      // Move To Front

      moveToFrontButtonItem = new JDRButtonItem("edit.front",
         "movetofront",
         KeyStroke.getKeyStroke(KeyEvent.VK_F, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.front"),
         toolBar, editM);

      startup.setInfo(editM.getText()+"->"+moveToFrontButtonItem.getText());
      startup.incProgress();

      // Move To Back

      moveToBackButtonItem = new JDRButtonItem("edit.back",
         "movetoback",
         KeyStroke.getKeyStroke(KeyEvent.VK_B, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.back"),
         toolBar, editM);

      startup.setInfo(editM.getText()+"->"+moveToBackButtonItem.getText());
      startup.incProgress();

      // Transform Menu

      transformM = new JMenu(
         JDRResources.getString("transform.label"));
      transformM.setMnemonic(
         JDRResources.getChar("transform.mnemonic"));
      mbar.add(transformM);

      startup.setInfo(editM.getText()+"->"+transformM.getText());
      startup.incProgress();

      // Rotate

      rotateButtonItem = new JDRButtonItem("transform.rotate",
         "rotate",
         KeyStroke.getKeyStroke(KeyEvent.VK_W, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.rotate"),
         toolBar, transformM);

      rotateDB = new RotateDialogBox(this);
      addGeneralAction(rotateDB);

      startup.setInfo(transformM.getText() +"->"+rotateButtonItem.getText());
      startup.incProgress();

      // Scale

      scaleButtonItem = new JDRButtonItem("transform.scale",
         "scale",
         KeyStroke.getKeyStroke(KeyEvent.VK_Z, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.scale"),
         toolBar, transformM);

      scaleDB = new ScaleDialogBox(this);
      addGeneralAction(scaleDB);

      startup.setInfo(transformM.getText() +"->"+scaleButtonItem.getText());
      startup.incProgress();

      // Shear

      shearButtonItem = new JDRButtonItem("transform.shear",
         "shear",
         KeyStroke.getKeyStroke(KeyEvent.VK_H, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.shear"),
         toolBar, transformM);

      shearDB = new ShearDialogBox(this);
      addGeneralAction(shearDB);

      startup.setInfo(transformM.getText()+"->"+shearButtonItem.getText());
      startup.incProgress();

      transformM.addSeparator();

      // Reverse path

      reverseItem = new JMenuItem(
         JDRResources.getString("transform.reverse"),
         JDRResources.getChar("transform.reverse.mnemonic"));
      reverseItem.addActionListener(this);
      transformM.add(reverseItem);

      startup.setInfo(transformM.getText()+"->"+reverseItem.getText());
      startup.incProgress();

      // Merge Paths

      mergePathsItem = new JMenuItem(
         JDRResources.getString("transform.merge"),
         JDRResources.getChar("transform.merge.mnemonic"));

      mergePathsItem.addActionListener(this);
      mergePathsItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_J,
         InputEvent.CTRL_MASK));
      transformM.add(mergePathsItem);

      startup.setInfo(transformM.getText()+"->"+mergePathsItem.getText());
      startup.incProgress();

      // Path Union

      pathUnionItem = new JMenuItem(
         JDRResources.getString("transform.union"),
         JDRResources.getChar("transform.union.mnemonic"));

      pathUnionItem.addActionListener(this);
      transformM.add(pathUnionItem);

      startup.setInfo(transformM.getText()+"->"+pathUnionItem.getText());
      startup.incProgress();

      // XOR Paths

      xorPathsItem = new JMenuItem(
         JDRResources.getString("transform.xor"),
         JDRResources.getChar("transform.xor.mnemonic"));

      xorPathsItem.addActionListener(this);
      transformM.add(xorPathsItem);

      startup.setInfo(transformM.getText()+"->"+xorPathsItem.getText());
      startup.incProgress();

      // Path Intersection

      intersectPathsItem = new JMenuItem(
         JDRResources.getString("transform.intersect"),
         JDRResources.getChar("transform.intersect.mnemonic"));

      intersectPathsItem.addActionListener(this);
      transformM.add(intersectPathsItem);

      startup.setInfo(transformM.getText()+"->"+intersectPathsItem.getText());
      startup.incProgress();

      // Subtract Paths

      subtractPathsItem = new JMenuItem(
         JDRResources.getString("transform.subtract"),
         JDRResources.getChar("transform.subtract.mnemonic"));

      subtractPathsItem.addActionListener(this);
      transformM.add(subtractPathsItem);

      startup.setInfo(transformM.getText()+"->"+subtractPathsItem.getText());
      startup.incProgress();

      transformM.addSeparator();

      patternM = new JMenu(JDRResources.getString("transform.pattern"));

      patternM.setMnemonic(JDRResources.getChar("transform.pattern.mnemonic"));

      startup.setInfo(transformM.getText()+"->"+patternM.getText());
      transformM.add(patternM);

      transformM.addSeparator();

      // Set Pattern

      setPatternButtonItem = new JDRButtonItem("transform.pattern.set",
         "pattern", null,
         this, JDRResources.getString("tooltip.pattern"),
         toolBar, patternM);

      patternBox = new PatternDialogBox(this);

      startup.setInfo(transformM.getText()+"->"+patternM.getText()
         +"->"+setPatternButtonItem.getText());
      startup.incProgress();

      // Edit Pattern

      editPatternItem = new JMenuItem(
         JDRResources.getString("transform.pattern.edit"),
         JDRResources.getChar("transform.pattern.edit.mnemonic"));

      editPatternItem.addActionListener(this);
      patternM.add(editPatternItem);

      startup.setInfo(transformM.getText()+"->"+patternM.getText()
         +"->"+editPatternItem.getText());
      startup.incProgress();

      // Remove Pattern

      removePatternItem = new JMenuItem(
         JDRResources.getString("transform.pattern.remove"),
         JDRResources.getChar("transform.pattern.remove.mnemonic"));

      removePatternItem.addActionListener(this);
      patternM.add(removePatternItem);

      startup.setInfo(transformM.getText()+"->"+patternM.getText()
         +"->"+removePatternItem.getText());
      startup.incProgress();

      // Convert to JDRPath

      convertToPathItem = new JMenuItem(
         JDRResources.getString("transform.convert"),
         JDRResources.getChar("transform.convert.mnemonic"));

      convertToPathItem.addActionListener(this);
      convertToPathItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_D,
         InputEvent.CTRL_MASK));
      transformM.add(convertToPathItem);

      startup.setInfo(transformM.getText() +"->"+convertToPathItem.getText());
      startup.incProgress();

      // Convert pattern or symmetric path to full path

      convertToFullPathItem = new JMenuItem(
         JDRResources.getString("transform.convert_to_full"),
         JDRResources.getChar("transform.convert_to_full.mnemonic"));

      convertToFullPathItem.addActionListener(this);
      transformM.add(convertToFullPathItem);

      startup.setInfo(transformM.getText() +"->"+convertToFullPathItem.getText());
      startup.incProgress();

      // split text

      splitTextItem = new JMenuItem(
         JDRResources.getString("transform.split"),
         JDRResources.getChar("transform.split.mnemonic"));
      splitTextItem.addActionListener(this);
      transformM.add(splitTextItem);

      startup.setInfo(transformM.getText() +"->"+splitTextItem.getText());
      startup.incProgress();

      // Convert to JDRTextPath

      convertToTextPathButtonItem = new JDRButtonItem("transform.textpath",
         "textpath", null,
         this, JDRResources.getString("tooltip.textpath"),
         toolBar, transformM);

      startup.setInfo(transformM.getText() +"->"+convertToTextPathButtonItem.getText());
      startup.incProgress();

      // Separate compound shape

      separateItem = new JMenuItem(
         JDRResources.getString("transform.separate"),
         JDRResources.getChar("transform.separate.mnemonic"));

      separateItem.addActionListener(this);
      transformM.add(separateItem);

      startup.setInfo(transformM.getText() +"->"+separateItem.getText());
      startup.incProgress();

      transformM.addSeparator();

      // Group

      groupButtonItem = new JDRButtonItem("transform.group",
         "group",
         GeneralAction.KEY_GROUP,
         this, JDRResources.getString("tooltip.group"),
         toolBar, transformM);

      startup.setInfo(transformM.getText() +"->"+groupButtonItem.getText());
      startup.incProgress();

      // Ungroup

      ungroupButtonItem = new JDRButtonItem("transform.ungroup",
         "ungroup",
         GeneralAction.KEY_UNGROUP,
         this, JDRResources.getString("tooltip.ungroup"),
         toolBar, transformM);

      startup.setInfo(transformM.getText() +"->"+ungroupButtonItem.getText());
      startup.incProgress();

      // Justify Sub Menu

      justifyM = new JMenu(
         JDRResources.getString("transform.justify"));
      justifyM.setMnemonic(
         JDRResources.getChar("transform.justify.mnemonic"));
      transformM.add(justifyM);

      startup.setInfo(transformM.getText() +"->"+justifyM.getText());
      startup.incProgress();

      // Left Align

      leftAlignItem = new JMenuItem(
         JDRResources.getString("transform.justify.left"),
         JDRResources.getChar("transform.justify.left.mnemonic"));
      leftAlignItem.addActionListener(this);
      justifyM.add(leftAlignItem);

      startup.setInfo(transformM.getText() +"->"+justifyM.getText()
         +"->"+leftAlignItem.getText());
      startup.incProgress();

      // Centre Align

      centreAlignItem = new JMenuItem(
         JDRResources.getString("transform.justify.centre"),
         JDRResources.getChar("transform.justify.centre.mnemonic"));
      centreAlignItem.addActionListener(this);
      justifyM.add(centreAlignItem);

      startup.setInfo(transformM.getText() +"->"+justifyM.getText()
         +"->"+centreAlignItem.getText());
      startup.incProgress();

      // Right Align

      rightAlignItem = new JMenuItem(
         JDRResources.getString("transform.justify.right"),
         JDRResources.getChar("transform.justify.right.mnemonic"));
      rightAlignItem.addActionListener(this);
      justifyM.add(rightAlignItem);

      justifyM.addSeparator();

      startup.setInfo(transformM.getText() +"->"+justifyM.getText()
         +"->"+rightAlignItem.getText());
      startup.incProgress();

      // Top Align

      topAlignItem = new JMenuItem(
         JDRResources.getString("transform.justify.top"),
         JDRResources.getChar("transform.justify.top.mnemonic"));
      topAlignItem.addActionListener(this);
      justifyM.add(topAlignItem);

      startup.setInfo(transformM.getText() +"->"+justifyM.getText()
         +"->"+topAlignItem.getText());
      startup.incProgress();

      // Middle Align

      middleAlignItem = new JMenuItem(
         JDRResources.getString("transform.justify.middle"),
         JDRResources.getChar("transform.justify.middle.mnemonic"));
      middleAlignItem.addActionListener(this);
      justifyM.add(middleAlignItem);

      startup.setInfo(transformM.getText() +"->"+justifyM.getText()
         +"->"+middleAlignItem.getText());
      startup.incProgress();

      // Bottom Align

      bottomAlignItem = new JMenuItem(
         JDRResources.getString("transform.justify.bottom"),
         JDRResources.getChar("transform.justify.bottom.mnemonic"));
      bottomAlignItem.addActionListener(this);
      justifyM.add(bottomAlignItem);

      startup.setInfo(transformM.getText() +"->"+justifyM.getText()
         +"->"+bottomAlignItem.getText());
      startup.incProgress();

      // Tools menu

      toolsM = new JMenu(JDRResources.getString("tools.label"));
      toolsM.setMnemonic(JDRResources.getChar("tools.mnemonic"));
      mbar.add(toolsM);

      ButtonGroup toolGroup = new ButtonGroup();
      ButtonGroup toolMenuGroup = new ButtonGroup();

      startup.setInfo(toolsM.getText());
      startup.incProgress();

      // Select

      selectButtonItem = new JDRToolButtonItem("tools.select",
         "select",
         KeyStroke.getKeyStroke(KeyEvent.VK_P, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.select"),
         toolMenuGroup, toolGroup, sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+selectButtonItem.getText());
      startup.incProgress();

      vSlidingBar.setUnitIncrement(
         selectButtonItem.getButtonPreferredHeight());

      // Open line

      openLineButtonItem = new JDRToolButtonItem("tools.open_line",
         "openline",
         KeyStroke.getKeyStroke(KeyEvent.VK_L, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.open_line"),
         toolMenuGroup, toolGroup, sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+openLineButtonItem.getText());
      startup.incProgress();

      // Closed Line

      closedLineButtonItem = new JDRToolButtonItem("tools.closed_line",
         "closedline",
         KeyStroke.getKeyStroke(KeyEvent.VK_L,
            InputEvent.CTRL_MASK + InputEvent.SHIFT_MASK),
         this, JDRResources.getString("tooltip.closed_line"),
         toolMenuGroup, toolGroup, sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+closedLineButtonItem.getText());
      startup.incProgress();

      // Open curve

      openCurveButtonItem = new JDRToolButtonItem("tools.open_curve",
         "opencurve",
         KeyStroke.getKeyStroke(KeyEvent.VK_K, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.open_curve"),
         toolMenuGroup, toolGroup, sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+openCurveButtonItem.getText());
      startup.incProgress();

      // Closed curve

      closedCurveButtonItem = new JDRToolButtonItem("tools.closed_curve",
         "closedcurve",
         KeyStroke.getKeyStroke(KeyEvent.VK_K,
            InputEvent.CTRL_MASK + InputEvent.SHIFT_MASK),
         this, JDRResources.getString("tooltip.closed_curve"),
         toolMenuGroup, toolGroup, sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+closedCurveButtonItem.getText());
      startup.incProgress();

      // Rectangle

      rectangleButtonItem = new JDRToolButtonItem("tools.rectangle",
         "rectangle",
         KeyStroke.getKeyStroke(KeyEvent.VK_R, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.rectangle"),
         toolMenuGroup, toolGroup, sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+rectangleButtonItem.getText());
      startup.incProgress();

      // Ellipse

      ellipseButtonItem = new JDRToolButtonItem("tools.ellipse",
         "ellipse",
         KeyStroke.getKeyStroke(KeyEvent.VK_E, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.ellipse"),
         toolMenuGroup, toolGroup, sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+ellipseButtonItem.getText());
      startup.incProgress();

      // Text

      textButtonItem = new JDRToolButtonItem("tools.text",
         "text",
         KeyStroke.getKeyStroke(KeyEvent.VK_T, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.text"),
         toolMenuGroup, toolGroup, sidePanel, toolsM);

      toolsM.addSeparator();

      startup.setInfo(toolsM.getText()+"->"+textButtonItem.getText());
      startup.incProgress();

      // Gap

      gapButtonItem = new JDRButtonItem("tools.gap", "gap",
         KeyStroke.getKeyStroke(KeyEvent.VK_M, InputEvent.CTRL_MASK),
         this, JDRResources.getString("tooltip.gap"),
         sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+gapButtonItem.getText());
      startup.incProgress();

      // Abandon

      abandonButtonItem = new JDRButtonItem("tools.abandon", "abandon",
         KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
         this, JDRResources.getString("tooltip.abandon"),
         sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+abandonButtonItem.getText());
      startup.incProgress();

      // Finish

      finishButtonItem = new JDRButtonItem("tools.finish", "finish",
         KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
         this, JDRResources.getString("tooltip.finish"),
         sidePanel, toolsM);

      startup.setInfo(toolsM.getText()+"->"+finishButtonItem.getText());
      startup.incProgress();

      Dimension prefSize = sidePanel.getPreferredSize();
      prefSize.width += 4;
      sidePanel.setPreferredSize(prefSize);

      // navigate menu

      navigateM = new JMenu( 
         JDRResources.getString("navigate.label"));
      navigateM.setMnemonic(
         JDRResources.getChar("navigate.mnemonic"));
      mbar.add(navigateM);

      startup.setInfo(navigateM.getText());
      startup.incProgress();

      // go to co-ordinate

      gotoItem = new JMenuItem(
         JDRResources.getString("navigate.goto"),
         JDRResources.getChar("navigate.goto.mnemonic"));
      gotoItem.addActionListener(this);
      gotoItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_F5, 0));
      navigateM.add(gotoItem);

      gotoDialog = new GoToDialogBox(this);
      addGeneralAction(gotoDialog);

      startup.setInfo(navigateM.getText()+"->"+gotoItem.getText());
      startup.incProgress();

      // select next object

      nextItem = new JMenuItem(
         JDRResources.getString("navigate.select"),
         JDRResources.getChar("navigate.select.mnemonic"));
      nextItem.addActionListener(this);
      nextItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_F5, KeyEvent.SHIFT_MASK));
      navigateM.add(nextItem);

      startup.setInfo(navigateM.getText()+"->"+nextItem.getText());
      startup.incProgress();

      // add next object to selection

      addNextItem = new JMenuItem(
         JDRResources.getString("navigate.add_next"),
         JDRResources.getChar("navigate.add_next.mnemonic"));
      addNextItem.addActionListener(this);
      addNextItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_F6, KeyEvent.SHIFT_MASK));
      navigateM.add(addNextItem);

      startup.setInfo(navigateM.getText()+"->"+addNextItem.getText());
      startup.incProgress();

      // deselect lowest object and add next object to selection

      skipItem = new JMenuItem(
         JDRResources.getString("navigate.skip"),
         JDRResources.getChar("navigate.skip.mnemonic"));
      skipItem.addActionListener(this);
      skipItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_F6, 0));
      navigateM.add(skipItem);

      startup.setInfo(navigateM.getText()+"->"+skipItem.getText());
      startup.incProgress();

      // scroll to selected objects.

      findSelectedItem = new JMenuItem(
         JDRResources.getString("navigate.find"),
         JDRResources.getChar("navigate.find.mnemonic"));
      findSelectedItem.addActionListener(this);
      findSelectedItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_F7, KeyEvent.SHIFT_MASK));
      navigateM.add(findSelectedItem);

      startup.setInfo(navigateM.getText()+"->"+findSelectedItem.getText());
      startup.incProgress();

      // find object with given description

      findByDescriptionItem = new JMenuItem(
         JDRResources.getString("navigate.description"),
         JDRResources.getChar("navigate.description.mnemonic"));
      findByDescriptionItem.addActionListener(this);
      navigateM.add(findByDescriptionItem);

      findByDescDialog = new FindByDescriptionDialogBox(this);
      addGeneralAction(findByDescDialog);

      startup.setInfo(navigateM.getText()+"->"+findByDescriptionItem.getText());
      startup.incProgress();

      // find object with given description and add to current 
      // selection

      addByDescriptionItem = new JMenuItem(
         JDRResources.getString("navigate.add_description"),
         JDRResources.getChar("navigate.add_description.mnemonic"));
      addByDescriptionItem.addActionListener(this);
      navigateM.add(addByDescriptionItem);

      startup.setInfo(navigateM.getText()+"->"+addByDescriptionItem.getText());
      startup.incProgress();

      // Bitmap menu

      bitmapM = new JMenu(JDRResources.getString("bitmap.label"));
      bitmapM.setMnemonic(JDRResources.getChar("bitmap.mnemonic"));
      mbar.add(bitmapM);

      startup.setInfo(bitmapM.getText());
      startup.incProgress();

      // Insert Bitmap

      insertBitmapItem = new JMenuItem(
         JDRResources.getString("bitmap.insert"),
         JDRResources.getChar("bitmap.insert.mnemonic"));
      insertBitmapItem.addActionListener(this);

      bitmapM.add(insertBitmapItem);

      bitmapFC = new JFileChooser();
      bitmapFC.setCurrentDirectory(new File(appSettings.startDir));
      bitmapFC.setAcceptAllFileFilterUsed(false);
      BitmapFileFilter bitmapFileFilter = new BitmapFileFilter(
         JDRResources.getString("filter.bitmap"));
      bitmapFC.addChoosableFileFilter(bitmapFileFilter);

      imagePreviewPanel = new ImagePreview(bitmapFC);
      bitmapFC.setAccessory(imagePreviewPanel);

      addGeneralAction(bitmapFC);

      imagePreviewPanel.setPreviewSelected(getPreviewBitmaps());

      imagePreviewPanel.addPreviewBoxActionListener(
         new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               setPreviewBitmaps(
                  imagePreviewPanel.isPreviewSelected());
            }
         });

      startup.setInfo(bitmapM.getText()+"->"+insertBitmapItem.getText());
      startup.incProgress();

      // Refresh

      refreshItem = new JMenuItem(
         JDRResources.getString("bitmap.refresh"),
         JDRResources.getChar("bitmap.refresh.mnemonic"));
      refreshItem.addActionListener(this);
      bitmapM.add(refreshItem);

      startup.setInfo(bitmapM.getText()+"->"+refreshItem.getText());
      startup.incProgress();

      // Properties

      bitmapPropItem = new JMenuItem(
          JDRResources.getString("bitmap.properties"),
          JDRResources.getChar("bitmap.properties.mnemonic"));

      bitmapPropChooserBox = new BitmapProperties(this, bitmapFC);
      addGeneralAction(bitmapPropChooserBox);

      bitmapPropItem.addActionListener(this);
      bitmapM.add(bitmapPropItem);

      startup.setInfo(bitmapM.getText()+"->"+bitmapPropItem.getText());
      startup.incProgress();

      // Vectorize

      if (experimentalMode || JDRResources.debugMode)
      {
         vectorizeItem = new JMenuItem(
            JDRResources.getString("bitmap.vectorize"),
            JDRResources.getChar("bitmap.vectorize.mnemonic"));
         vectorizeItem.addActionListener(this);
         vectorizeItem.setActionCommand("vectorize");

         startup.setInfo(bitmapM.getText()+"->"+vectorizeItem.getText());
         bitmapM.add(vectorizeItem);
      }

      vectorizeBitmapDialog = new VectorizeBitmapDialog(this);

      startup.incProgress();

      // TeX/LaTeX
      texM = new JMenu(JDRResources.getString("tex.label"));
      texM.setMnemonic(JDRResources.getChar("tex.mnemonic"));
      mbar.add(texM);

      startup.setInfo(texM.getText());
      startup.incProgress();

      // Parshape

      parshapeItem = new JMenuItem(
         JDRResources.getString("tex.parshape"),
         JDRResources.getChar("tex.parshape.mnemonic"));
      TeXAction parshapeAction = 
         new TeXAction(this, TeXAction.PARSHAPE);
      addGeneralAction(parshapeAction.getDialog());

      parshapeItem.addActionListener(parshapeAction);
      parshapeItem.setActionCommand("showparshapedialog");
      texM.add(parshapeItem);

      startup.setInfo(texM.getText()+"->"+parshapeItem.getText());
      startup.incProgress();

      // Shapepar

      shapeparItem = new JMenuItem(
         JDRResources.getString("tex.shapepar"),
         JDRResources.getChar("tex.shapepar.mnemonic"));
      TeXAction shapeparAction = 
         new TeXAction(this, TeXAction.SHAPEPAR);
      addGeneralAction(shapeparAction.getDialog());
      shapeparItem.setActionCommand("showparshapedialog");

      shapeparItem.addActionListener(shapeparAction);
      texM.add(shapeparItem);

      startup.setInfo(texM.getText()+"->"+shapeparItem.getText());
      startup.incProgress();

      // Flow Frame Menu

      JMenu flowframeM = new JMenu(
         JDRResources.getString("tex.flowframe"));
      flowframeM.setMnemonic(
         JDRResources.getChar("tex.flowframe.mnemonic"));
      texM.add(flowframeM);

      startup.setInfo(texM.getText()+"->"+flowframeM.getText());
      startup.incProgress();

      // Clear All

      clearAllItem = new JMenuItem(
         JDRResources.getString("tex.flowframe.clear"));
      clearAllItem.setMnemonic(
         JDRResources.getChar("tex.flowframe.clear.mnemonic"));
      clearAllItem.addActionListener(this);
      flowframeM.add(clearAllItem);

      startup.setInfo(texM.getText()+"->"+flowframeM.getText()
         +"->"+clearAllItem.getText());
      startup.incProgress();

      // Set Frame

      setFrameItem = new JMenuItem(
          JDRResources.getString("tex.flowframe.set_frame"));
      setFrameItem.setMnemonic(
          JDRResources.getChar("tex.flowframe.set_frame.mnemonic"));
      setFrameItem.addActionListener(this);
      flowframeM.add(setFrameItem);

      flfSelector = new FLFSelector(this);
      addGeneralAction(flfSelector);

      startup.setInfo(texM.getText()+"->"+flowframeM.getText()
         +"->"+setFrameItem.getText());
      startup.incProgress();

      // Set Typeblock

      setTypeblockItem = new JMenuItem(
         JDRResources.getString("tex.flowframe.set_typeblock"));
      setTypeblockItem.setMnemonic(
         JDRResources.getChar("tex.flowframe.set_typeblock.mnemonic"));
      setTypeblockItem.addActionListener(this);
      flowframeM.add(setTypeblockItem);

      setTypeblockSelector = new FLFSetTypeblock(this);
      addGeneralAction(setTypeblockSelector);

      startup.setInfo(texM.getText()+"->"+flowframeM.getText()
         +"->"+setTypeblockItem.getText());

      // Display pages

      displayPageDialog = new DisplayPageDialog(this);
      addGeneralAction(displayPageDialog);

      displayPageItem = new JMenuItem(
         JDRResources.getString("tex.flowframe.display_page"));
      displayPageItem.setMnemonic(
         JDRResources.getChar("tex.flowframe.display_page.mnemonic"));
      displayPageItem.addActionListener(this);
      flowframeM.add(displayPageItem);

      startup.setInfo(texM.getText()+"->"+flowframeM.getText()
         +"->"+displayPageItem.getText());

      startup.incProgress();

      // TeX settings

      JMenu texSettingsM = new JMenu(
         JDRResources.getString("tex.config"));
      texSettingsM.setMnemonic(
         JDRResources.getChar("tex.config.mnemonic"));
      texM.add(texSettingsM);

      startup.setInfo(texM.getText()+"->"+texSettingsM.getText());
      // set normalsize

      setNormalSizeItem = new JMenuItem(
         JDRResources.getString("tex.config.normalsize"));
      setNormalSizeItem.setMnemonic(
         JDRResources.getChar("tex.config.normalsize.mnemonic"));
      texSettingsM.add(setNormalSizeItem);

      setNormalSizeItem.addActionListener(this);
      setNormalSizeItem.setToolTipText(
         JDRResources.getString("tooltip.texsettings"));

      texSettingsDialog = new TeXSettingsDialog(this);
      addGeneralAction(texSettingsDialog);

      startup.setInfo(texM.getText()+"->"+texSettingsM.getText()
         +"->"+setNormalSizeItem.getText());
      startup.incProgress();

      // auto adjust anchor

      autoAdjustAnchorItem = new JCheckBoxMenuItem(
         JDRResources.getString("tex.config.anchor"),
         isAutoAnchorEnabled());
      autoAdjustAnchorItem.setMnemonic(
         JDRResources.getChar("tex.config.anchor.mnemonic"));

      autoAdjustAnchorItem.addActionListener(this);
      autoAdjustAnchorItem.setToolTipText(
         JDRResources.getString("tooltip.autoanchor"));

      texSettingsM.add(autoAdjustAnchorItem);

      startup.setInfo(texM.getText()+"->"+texSettingsM.getText()
         +"->"+autoAdjustAnchorItem.getText());
      startup.incProgress();

      // auto escape special characters

      autoEscapeSpCharsItem = new JCheckBoxMenuItem(
         JDRResources.getString("tex.config.escape"),
         isAutoEscapeSpCharsEnabled());
      autoEscapeSpCharsItem.setMnemonic(
         JDRResources.getChar("tex.config.escape.mnemonic"));

      autoEscapeSpCharsItem.addActionListener(this);
      autoEscapeSpCharsItem.setToolTipText(
         JDRResources.getString("tooltip.escape_special_chars"));

      texSettingsM.add(autoEscapeSpCharsItem);

      startup.setInfo(texM.getText()+"->"+texSettingsM.getText()
         +"->"+autoEscapeSpCharsItem.getText());
      startup.incProgress();

      // Settings
      settingsM = new JMenu(
         JDRResources.getString("settings.label"));
      settingsM.setMnemonic(
         JDRResources.getChar("settings.mnemonic"));
      mbar.add(settingsM);

      startup.setInfo(settingsM.getText());
      startup.incProgress();

      // Styles
      stylesItem = new JMenuItem(
         JDRResources.getString("settings.styles"),
         JDRResources.getChar("settings.styles.mnemonic"));

      startup.setInfo("Style Selector");
      styleChooserBox = new StyleSelector(this);
      addGeneralAction(styleChooserBox);

      stylesItem.addActionListener(this);
      settingsM.add(stylesItem);

      startup.setInfo(settingsM.getText()+"->"+stylesItem.getText());
      startup.incProgress();

      // set style settings
      styleChooserBox.setLinePaint(appSettings.linePaint);
      styleChooserBox.setFillPaint(appSettings.fillPaint);
      styleChooserBox.setTextPaint(appSettings.textPaint);
      styleChooserBox.setFontName(appSettings.fontFamily);
      styleChooserBox.setFontSize(appSettings.fontSize);
      styleChooserBox.setFontSeries(appSettings.fontSeries);
      styleChooserBox.setFontShape(appSettings.fontShape);
      styleChooserBox.setLaTeXFontFamily(appSettings.latexFontFamily);
      styleChooserBox.setLaTeXFontShape(appSettings.latexFontShape);
      styleChooserBox.setLaTeXFontSeries(appSettings.latexFontSeries);
      styleChooserBox.setLaTeXFontSize(appSettings.latexFontSize);
      styleChooserBox.setHalign(appSettings.pgfHalign);
      styleChooserBox.setValign(appSettings.pgfValign);
      styleChooserBox.setStroke(appSettings.stroke);

      startup.incProgress();

      // Show tools

      showToolsItem = new JCheckBoxMenuItem(
         JDRResources.getString("settings.tools"), true);
      showToolsItem.setMnemonic(
         JDRResources.getChar("settings.tools.mnemonic"));
      showToolsItem.addActionListener(this);
      settingsM.add(showToolsItem);
      toolBar.setVisible(appSettings.showToolBar);  
      sidePanel.setVisible(appSettings.showToolBar);  
      showToolsItem.setSelected(appSettings.showToolBar);

      startup.setInfo(settingsM.getText()+"->"+showToolsItem.getText());
      startup.incProgress();

      // Show both rulers

      showRulersItem = new JCheckBoxMenuItem(
         JDRResources.getString("settings.rulers"), true);
      showRulersItem.addActionListener(this);
      showRulersItem.setMnemonic(
         JDRResources.getChar("settings.rulers.mnemonic"));
      settingsM.add(showRulersItem);

      startup.setInfo(settingsM.getText()+"->"+showRulersItem.getText());
      startup.incProgress();

      // Show status bar

      showStatusBarItem = new JCheckBoxMenuItem(
         JDRResources.getString("settings.status"),
         appSettings.showStatus);
      showStatusBarItem.setMnemonic(
         JDRResources.getChar("settings.status.mnemonic"));
      showStatusBarItem.addActionListener(this);
      settingsM.add(showStatusBarItem);
      statusBar.setVisible(appSettings.showStatus);  

      startup.setInfo(settingsM.getText()+"->"+showStatusBarItem.getText());
      startup.incProgress();

      // Grid Sub Menu

      gridM = new JMenu(JDRResources.getString("settings.grid"));
      gridM.setMnemonic(
         JDRResources.getChar("settings.grid.mnemonic"));
      settingsM.add(gridM);

      startup.setInfo(settingsM.getText()+"->"+gridM.getText());
      startup.incProgress();

      // Show Grid

      showGridButtonItem = new JDRToggleButtonItem(
         "settings.grid.show", "showgrid",
         KeyStroke.getKeyStroke(KeyEvent.VK_F2, 0),
         this, JDRResources.getString("tooltip.show_grid"),
         toolBar, gridM);

      startup.setInfo(settingsM.getText()+"->"+gridM.getText()
         +"->"+showGridButtonItem.getText());
      startup.incProgress();

      // Lock

      lockGridButtonItem = new JDRToggleButtonItem(
         "settings.grid.lock", "lockgrid",
         KeyStroke.getKeyStroke(KeyEvent.VK_F2, InputEvent.SHIFT_MASK),
         this, JDRResources.getString("tooltip.lock_grid"),
         toolBar, gridM);

      startup.setInfo(settingsM.getText()+"->"+gridM.getText()
         +"->"+lockGridButtonItem.getText());
      startup.incProgress();

      // Grid Settings

      gridSettingsItem = new JMenuItem(
         JDRResources.getString("settings.grid.settings"),
         JDRResources.getChar("settings.grid.settings.mnemonic"));

      gridSettingsChooserBox = new GridSettings(this);
      addGeneralAction(gridSettingsChooserBox);

      gridSettingsItem.addActionListener(this);
      gridM.add(gridSettingsItem);

      startup.setInfo(settingsM.getText()+"->"+gridM.getText()
         +"->"+gridSettingsItem.getText());
      startup.incProgress();

      // Zoom sub menu

      zoomM = new JMenu(JDRResources.getString("settings.zoom"));
      zoomM.setMnemonic(
         JDRResources.getChar("settings.zoom.mnemonic"));
      settingsM.add(zoomM);

      ButtonGroup zoomGroup = new ButtonGroup();
      
      startup.setInfo(settingsM.getText()+"->"+zoomM.getText());
      startup.incProgress();

      // Fit Width

      zoomWidthItem = new JMenuItem(
         JDRResources.getString("settings.zoom.width"),
         JDRResources.getChar("settings.zoom.width.mnemonic"));
      zoomWidthItem.addActionListener(this);
      zoomM.add(zoomWidthItem);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoomWidthItem.getText());
      startup.incProgress();

      // Fit Height

      zoomHeightItem = new JMenuItem(
         JDRResources.getString("settings.zoom.height"),
         JDRResources.getChar("settings.zoom.height.mnemonic"));
      zoomHeightItem.addActionListener(this);
      zoomM.add(zoomHeightItem);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoomHeightItem.getText());
      startup.incProgress();

      // Fit Page

      zoomPageItem = new JMenuItem(
         JDRResources.getString("settings.zoom.page"),
         JDRResources.getChar("settings.zoom.page.mnemonic"));
      zoomPageItem.addActionListener(this);
      zoomM.add(zoomPageItem);

      zoomM.addSeparator();

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoomPageItem.getText());
      startup.incProgress();

      // User defined zoom

      zoomSettingsItem = new JRadioButtonMenuItem(
         JDRResources.getString("settings.zoom.user"), false);
      zoomSettingsItem.setMnemonic(
         JDRResources.getChar("settings.zoom.user.mnemonic"));

      zoomSettingsChooserBox = new ZoomSettings(this, this);
      addGeneralAction(zoomSettingsChooserBox);

      zoomSettingsItem.addActionListener(this);
      zoomM.add(zoomSettingsItem);
      zoomGroup.add(zoomSettingsItem);


      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoomSettingsItem.getText());
      startup.incProgress();

      // 25% Magnification

      zoom25Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.zoom.25"), false);
      zoom25Item.setMnemonic(
         JDRResources.getChar("settings.zoom.25.mnemonic"));
      zoom25Item.addActionListener(this);
      zoomM.add(zoom25Item);
      zoomGroup.add(zoom25Item);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoom25Item.getText());
      startup.incProgress();

      // 50% Magnification

      zoom50Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.zoom.50"), false);
      zoom50Item.setMnemonic(
         JDRResources.getChar("settings.zoom.50.mnemonic"));
      zoom50Item.addActionListener(this);
      zoomM.add(zoom50Item);
      zoomGroup.add(zoom50Item);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoom50Item.getText());
      startup.incProgress();

      // 75% Magnification

      zoom75Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.zoom.75"), false);
      zoom75Item.setMnemonic(
         JDRResources.getChar("settings.zoom.75.mnemonic"));
      zoom75Item.addActionListener(this);
      zoomM.add(zoom75Item);
      zoomGroup.add(zoom75Item);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoom75Item.getText());
      startup.incProgress();

      // 100% Magnification

      zoom100Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.zoom.100"), true);
      zoom100Item.setMnemonic(
         JDRResources.getChar("settings.zoom.100.mnemonic"));
      zoom100Item.addActionListener(this);
      zoomM.add(zoom100Item);
      zoomGroup.add(zoom100Item);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoom100Item.getText());
      startup.incProgress();

      // 200% Magnification

      zoom200Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.zoom.200"), false);
      zoom200Item.setMnemonic(
         JDRResources.getChar("settings.zoom.200.mnemonic"));
      zoom200Item.addActionListener(this);
      zoomM.add(zoom200Item);
      zoomGroup.add(zoom200Item);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoom200Item.getText());
      startup.incProgress();

      // 400% Magnification

      zoom400Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.zoom.400"), false);
      zoom400Item.setMnemonic(
         JDRResources.getChar("settings.zoom.400.mnemonic"));
      zoom400Item.addActionListener(this);
      zoomM.add(zoom400Item);
      zoomGroup.add(zoom400Item);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoom400Item.getText());
      startup.incProgress();

      // 800% Magnification

      zoom800Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.zoom.800"), false);
      zoom800Item.setMnemonic(
         JDRResources.getChar("settings.zoom.800.mnemonic"));
      zoom800Item.addActionListener(this);
      zoomM.add(zoom800Item);
      zoomGroup.add(zoom800Item);

      startup.setInfo(settingsM.getText()+"->"+zoomM.getText()
        +"->"+zoom800Item.getText());
      startup.incProgress();

      // Paper size menu

      paperM = new JMenu(JDRResources.getString("settings.paper"));
      paperM.setMnemonic(
         JDRResources.getChar("settings.paper.mnemonic"));
      settingsM.add(paperM);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText());
      startup.incProgress();

      // Show printer margins

      showPrinterMarginsItem = new JCheckBoxMenuItem(
         JDRResources.getString("settings.paper.margins"), true);
      showPrinterMarginsItem.setMnemonic(
         JDRResources.getChar("settings.paper.margins.mnemonic"));
      showPrinterMarginsItem.addActionListener(this);
      paperM.add(showPrinterMarginsItem);
      showPrinterMarginsItem.setSelected(false);

      if (disablePrint)
      {
         showPrinterMarginsItem.setEnabled(false);
      }

      ButtonGroup orientationGroup = new ButtonGroup();

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+showPrinterMarginsItem.getText());
      startup.incProgress();

      // Portrait

      portraitItem = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.portrait"), true);
      portraitItem.setMnemonic(
         JDRResources.getChar("settings.paper.portrait.mnemonic"));
      portraitItem.addActionListener(this);
      paperM.add(portraitItem);
      orientationGroup.add(portraitItem);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+portraitItem.getText());
      startup.incProgress();

      // Landscape

      landscapeItem = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.landscape"), false);
      landscapeItem.setMnemonic(
         JDRResources.getChar("settings.paper.landscape.mnemonic"));
      landscapeItem.addActionListener(this);
      paperM.add(landscapeItem);
      orientationGroup.add(landscapeItem);

      paperM.addSeparator();

      ButtonGroup paperGroup = new ButtonGroup();

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+landscapeItem.getText());
      startup.incProgress();

      // A5 paper

      paperA5Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.A5"), false);
      paperA5Item.setMnemonic(
         JDRResources.getChar("settings.paper.A5.mnemonic"));
      paperA5Item.addActionListener(this);
      paperM.add(paperA5Item);
      paperGroup.add(paperA5Item);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperA5Item.getText());
      startup.incProgress();

      // A4 paper

      paperA4Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.A4"), false);
      paperA4Item.setMnemonic(
         JDRResources.getChar("settings.paper.A4.mnemonic"));
      paperA4Item.addActionListener(this);
      paperM.add(paperA4Item);
      paperGroup.add(paperA4Item);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperA4Item.getText());
      startup.incProgress();

      // A3 paper

      paperA3Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.A3"), false);
      paperA3Item.setMnemonic(
         JDRResources.getChar("settings.paper.A3.mnemonic"));
      paperA3Item.addActionListener(this);
      paperM.add(paperA3Item);
      paperGroup.add(paperA3Item);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperA3Item.getText());
      startup.incProgress();

      // A2 paper

      paperA2Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.A2"), false);
      paperA2Item.setMnemonic(
         JDRResources.getChar("settings.paper.A2.mnemonic"));
      paperA2Item.addActionListener(this);
      paperM.add(paperA2Item);
      paperGroup.add(paperA2Item);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperA2Item.getText());
      startup.incProgress();

      // A1 paper

      paperA1Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.A1"), false);
      paperA1Item.setMnemonic(
         JDRResources.getChar("settings.paper.A1.mnemonic"));
      paperA1Item.addActionListener(this);
      paperM.add(paperA1Item);
      paperGroup.add(paperA1Item);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperA1Item.getText());
      startup.incProgress();

      // A0 paper

      paperA0Item = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.A0"), false);
      paperA0Item.setMnemonic(
         JDRResources.getChar("settings.paper.A0.mnemonic"));
      paperA0Item.addActionListener(this);
      paperM.add(paperA0Item);
      paperGroup.add(paperA0Item);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperA0Item.getText());
      startup.incProgress();

      // Letter paper

      paperLetterItem = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.letter"), false);
      paperLetterItem.setMnemonic(
         JDRResources.getChar("settings.paper.letter.mnemonic"));
      paperLetterItem.addActionListener(this);
      paperM.add(paperLetterItem);
      paperGroup.add(paperLetterItem);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperLetterItem.getText());
      startup.incProgress();

      // Legal paper

      paperLegalItem = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.legal"), false);
      paperLegalItem.setMnemonic(
         JDRResources.getChar("settings.paper.legal.mnemonic"));
      paperLegalItem.addActionListener(this);
      paperM.add(paperLegalItem);
      paperGroup.add(paperLegalItem);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperLegalItem.getText());
      startup.incProgress();

      // Executive paper

      paperExecutiveItem = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.executive"), false);
      paperExecutiveItem.setMnemonic(
         JDRResources.getChar("settings.paper.executive.mnemonic"));
      paperExecutiveItem.addActionListener(this);
      paperM.add(paperExecutiveItem);
      paperGroup.add(paperExecutiveItem);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperExecutiveItem.getText());
      startup.incProgress();

      // other paper

      paperOtherItem = new JRadioButtonMenuItem(
         JDRResources.getString("settings.paper.other"), false);
      paperOtherItem.setMnemonic(
         JDRResources.getChar("settings.paper.other.mnemonic"));
      paperOtherItem.addActionListener(this);
      paperM.add(paperOtherItem);
      paperGroup.add(paperOtherItem);

      paperDialog = new PaperDialogBox(this, settings.paper);
      addGeneralAction(paperDialog);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+paperOtherItem.getText());
      startup.incProgress();

      // configure settings

      configSettingsItem = new JMenuItem(
         JDRResources.getString("settings.config"),
         JDRResources.getChar("settings.config.mnemonic"));

      configDialog = new ConfigSettingsDialog(this);
      addGeneralAction(configDialog);

      configSettingsItem.addActionListener(this);
      settingsM.add(configSettingsItem);

      startup.setInfo(settingsM.getText()+"->"+paperM.getText()
        +"->"+configSettingsItem.getText());
      startup.incProgress();

      // Window menu

      windowM = new JMenu(JDRResources.getString("window.label"));
      windowM.setMnemonic(
         JDRResources.getChar("window.label.mnemonic"));
      mbar.add(windowM);

      windowButtonGroup = new ButtonGroup();
      windowM.addSeparator();

      startup.setInfo(windowM.getText());
      startup.incProgress();

      // Tile frames

      tileItem = new JMenuItem(
         JDRResources.getString("window.tile"));
      tileItem.setMnemonic(
         JDRResources.getChar("window.tile.mnemonic"));
      windowM.add(tileItem);
      tileItem.addActionListener(this);

      startup.setInfo(windowM.getText()+"->"+tileItem.getText());
      startup.incProgress();

      // Arrange frames vertically

      verticallyItem = new JMenuItem(
         JDRResources.getString("window.vertically"));
      verticallyItem.setMnemonic(
         JDRResources.getChar("window.vertically.mnemonic"));
      windowM.add(verticallyItem);
      verticallyItem.addActionListener(this);

      startup.setInfo(windowM.getText()+"->"+verticallyItem.getText());
      startup.incProgress();

      // Arrange frames horizontally

      horizontallyItem = new JMenuItem(
         JDRResources.getString("window.horizontally"));
      horizontallyItem.setMnemonic(
         JDRResources.getChar("window.horizontally.mnemonic"));
      windowM.add(horizontallyItem);

      horizontallyItem.addActionListener(this);

      startup.setInfo(windowM.getText()+"->"+horizontallyItem.getText());
      startup.incProgress();

      // minimize all frames

      minimizeItem = new JMenuItem(
         JDRResources.getString("window.minimize"));
      minimizeItem.setMnemonic(
         JDRResources.getChar("window.minimize.mnemonic"));
      windowM.add(minimizeItem);
      minimizeItem.addActionListener(this);

      startup.setInfo(windowM.getText()+"->"+minimizeItem.getText());
      startup.incProgress();

      // maximize all frames

      maximizeItem = new JMenuItem(
         JDRResources.getString("window.maximize"));
      maximizeItem.setMnemonic(
         JDRResources.getChar("window.maximize.mnemonic"));

      windowM.add(maximizeItem);
      maximizeItem.addActionListener(this);

      startup.setInfo(windowM.getText()+"->"+maximizeItem.getText());
      startup.incProgress();

      // Help menu

      JMenu helpM = new JMenu(JDRResources.getString("help.label"));
      helpM.setMnemonic(JDRResources.getChar("help.mnemonic"));
      mbar.add(helpM);

      startup.setInfo(helpM.getText());
      startup.incProgress();

      // manual

      JMenuItem item = addHelpItem(helpM);

      toolBar.add(JDRResources.createMainHelpButton());

      startup.setInfo(helpM.getText()+"->"+item.getText());
      startup.incProgress();

      // Licence dialog

      licenceDialog = new LicenceDialog(this);
      addGeneralAction(licenceDialog);

      licenceItem = new JMenuItem(
         JDRResources.getString("help.licence"),
         JDRResources.getChar("help.licence.mnemonic"));
      licenceItem.addActionListener(this);
      helpM.add(licenceItem);

      startup.setInfo(helpM.getText()+"->"+licenceItem.getText());
      startup.incProgress();

      // About dialog

      aboutDialog = new AboutDialog(this, JpgfDraw.appName,
         JpgfDraw.version);
      addGeneralAction(aboutDialog);

      aboutItem = new JMenuItem(
         JDRResources.getString("help.about"),
         JDRResources.getChar("help.about.mnemonic"));
      aboutItem.addActionListener(this);
      helpM.add(aboutItem);

      startup.setInfo(helpM.getText()+"->"+aboutItem.getText());
      // debug menu

      if (JDRResources.debugMode)
      {
         debugM = new JMenu("Debug");
         debugM.setMnemonic('D');
         mbar.add(debugM);

         startup.setInfo(debugM.getText());
 
         JMenuItem infoItem = new JMenuItem("Object info", 'I');
         infoItem.addActionListener(new ObjectInfoAction());
         debugM.add(infoItem);

         startup.setInfo(debugM.getText()+"->"+infoItem.getText());

         JMenuItem writeLogItem = new JMenuItem("Write Log", 'L');
         writeLogItem.addActionListener(writeLogAction);
         writeLogItem.setAccelerator(writeLogKeyStroke);
         debugM.add(writeLogItem);

         startup.setInfo(debugM.getText()+"->"+writeLogItem.getText());

         JMenuItem dumpAllItem = new JMenuItem("Dump All", 'A');
         dumpAllItem.addActionListener(dumpAllAction);
         dumpAllItem.setAccelerator(dumpAllKeyStroke);
         debugM.add(dumpAllItem);

         startup.setInfo(debugM.getText()+"->"+dumpAllItem.getText());
      }

      // set the browse utility for bitmaps

      JDRBitmap.setBrowseUtil(new BrowseUtil(
         JDRResources.getString("browse.label"),
         JDRResources.getString("browse.not_found"),
         JDRResources.getString("browse.invalid_format"),
         JDRResources.getString("browse.cant_refresh"),
         JDRResources.getString("browse.title"),
         JDRResources.getString("browse.invalid_title"),
         JDRResources.getString("browse.discard")));
      JDRBitmap.setFileChooser(bitmapFC);

      startup.setInfo(JDRResources.getString("message.init_desktop"));
      startup.setIndeterminate();

      // set up the desktop

      theDesktop = new JDesktopPane();
      theDesktop.setBackground(Color.lightGray);
      getContentPane().add(theDesktop);

      GraphicsConfiguration gc = getGraphicsConfiguration();
      JDRUnit.setNormalizingFactor(gc);

      setSize(600,460);
      setVisible(true);

      startup.toFront();

      // add initial frame
      if (filename == null) addFrame(settings);
      else addFrame(settings, filename);

      discardAllEdits();

      disposeStartup();
   }

   class WriteLogAction extends AbstractAction
   {
      public WriteLogAction()
      {
         super("write log");
      }

      public void actionPerformed(ActionEvent evt)
      {
         writeLog();
      }
   }

   class ObjectInfoAction implements ActionListener
   {
      public void actionPerformed(ActionEvent evt)
      {
         objectInfo();
      }
   }

   class DumpAllAction extends AbstractAction
   {
      public void actionPerformed(ActionEvent evt)
      {
         dumpAll();
      }
   }

   public static void debugMessage(String message)
   {
      JDRResources.debugMessage(message);
   }

   public void addGeneralAction(JFrame frame)
   {
      addGeneralAction(frame.getRootPane());
   }

   public void addGeneralAction(JDialog dialog)
   {
      addGeneralAction(dialog.getRootPane());
   }

   public void addGeneralAction(JComponent comp)
   {
      comp.getActionMap().put(
         "write log", writeLogAction);
      comp.getInputMap(
         JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(
            writeLogKeyStroke, "write log");

      comp.getActionMap().put(
         "dump all", dumpAllAction);
      comp.getInputMap(
         JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(
            dumpAllKeyStroke, "dump all");
   }

   public void actionPerformed(ActionEvent evt)
   {
      Object source = evt.getSource();
      String action = evt.getActionCommand();

      if (action == null)
      {
         action = "";
      }

      JDRFrame frame = (JDRFrame)theDesktop.getSelectedFrame();

      if (action.equals("new"))
      {
         addFrame();
      }
      else if (action.equals("open"))
      {
         openImage();
      }
      else if (source == importItem)
      {
         importImage();
      }
      else if (source == imageDescriptionItem)
      {
         if (frame == null) return;
         displayImageDescriptionDialog(frame.getAllPaths());
      }
      else if (action.equals("save"))
      {
         saveImage();
      }
      else if (source == saveAsItem)
      {
         saveAs(true);
      }
      else if (source == exportItem)
      {
         exportImage();
      }
      else if (source == printItem)
      {
         if (frame == null) return;

         frame.print();
      }
      else if (source == pageDialogItem)
      {
         if (frame == null) return;

         pageDialog(frame);
      }
      else if (source == closeItem)
      {
         if (frame == null) return;

         close();
      }
      else if (source == quitItem)
      {
         quit();
      }
      else if (action.equals("selectAll"))
      {
         if (frame == null) return;
         frame.selectAll();
      }
      else if (source == deselectAllItem)
      {
         if (frame == null) return;
         frame.deselectAll();
      }
      else if (source == moveByItem)
      {
         if (frame == null) return;
         showMoveByDialog();
      }
      else if (action.equals("cut"))
      {
         if (frame == null) return;
         frame.cutSelectedPaths();
      }
      else if (action.equals("copy"))
      {
         if (frame == null) return;
         frame.copySelectedPaths();
      }
      else if (action.equals("paste"))
      {
         if (frame == null) return;
         frame.pasteSelectedPaths();
      }
      else if (source == objectDescriptionItem)
      {
         if (frame == null) return;
         JDRCompleteObject object = frame.getSelectedObject();
         displayObjectDescriptionDialog(object);
      }
      else if (action.equals("editPath"))
      {
         if (frame == null) return;

         setEditPathState(frame, editPathButtonItem.isSelected());
      }
      else if (source == linePaintItem)
      {
         if (frame == null) return;
         displayLinePaintChooser();
      }
      else if (source == fillPaintItem)
      {
         if (frame == null) return;
         displayFillPaintChooser();
      }
      else if (source == lineStyleItem)
      {
         if (frame == null) return;
         lineStyleChooserBox.initialise();
      }
      else if (source == lineWidthItem)
      {
         if (frame == null) return;
         displayLineWidthChooser();
      }
      else if (source == dashItem)
      {
         if (frame == null) return;
         displayDashPatternChooser();
      }
      else if (source == capButtItem)
      {
         if (frame == null) return;
         frame.setSelectedCapStyle(BasicStroke.CAP_BUTT);
      }
      else if (source == capRoundItem)
      {
         if (frame == null) return;
         frame.setSelectedCapStyle(BasicStroke.CAP_ROUND);
      }
      else if (source == capSquareItem)
      {
         if (frame == null) return;
         frame.setSelectedCapStyle(BasicStroke.CAP_SQUARE);
      }
      else if (source == joinItem)
      {
         if (frame == null) return;

         displayJoinStyleChooser();
      }
      else if (source == startArrowItem)
      {
         if (frame == null) return;
         displayStartMarkerChooser();
      }
      else if (source == midArrowItem)
      {
         if (frame == null) return;
         displayMidMarkerChooser();
      }
      else if (source == endArrowItem)
      {
         if (frame == null) return;
         displayEndMarkerChooser();
      }
      else if (source == allMarkersItem)
      {
         if (frame == null) return;
         displayAllMarkerChooser();
      }
      else if (source == windingEvenOddItem)
      {
         if (frame == null) return;
         frame.setSelectedWindingRule(GeneralPath.WIND_EVEN_ODD);
      }
      else if (source == windingNonZeroItem)
      {
         if (frame == null) return;
         frame.setSelectedWindingRule(GeneralPath.WIND_NON_ZERO);
      }
      else if (source == editTextItem)
      {
         if (frame == null) return;
         displayEditTextDialog();
      }
      else if (source == textPaintItem)
      {
         if (frame == null) return;
         displayTextPaintChooser();
      }
      else if (source == textMatrixItem)
      {
         if (frame == null) return;
         textMatrixDialog.initialise();
      }
      else if (source == textStyleItem)
      {
         if (frame == null) return;
         textStyleChooserBox.initialise();
      }
      else if (source == fontFamilyItem)
      {
         if (frame == null) return;
         displayFontFamilyChooser();
      }
      else if (source == fontSizeItem)
      {
         if (frame == null) return;
         displayFontSizeChooser();
      }
      else if (source == fontShapeItem)
      {
         if (frame == null) return;
         displayFontShapeChooser();
      }
      else if (source == fontSeriesItem)
      {
         if (frame == null) return;
         displayFontSeriesChooser();
      }
      else if (source == fontAnchorItem)
      {
         if (frame == null) return;
         displayFontAnchorChooser();
      }
      else if (source == fontHAnchorItem)
      {
         if (frame == null) return;
         displayFontHAnchorChooser();
      }
      else if (source == fontVAnchorItem)
      {
         if (frame == null) return;
         displayFontVAnchorChooser();
      }
      else if (source == reduceToGreyItem)
      {
         if (frame == null) return;
         frame.reduceToGrey();
      }
      else if (source == fadeItem)
      {
         if (frame == null) return;

         fadeDialog.display();
      }
      else if (action.equals("movetofront"))
      {
         if (frame == null) return;
         frame.moveDrawObjectToFront();
      }
      else if (action.equals("movetoback"))
      {
         if (frame == null) return;
         frame.moveDrawObjectToBack();
      }
      else if (action.equals("rotate"))
      {
         if (frame == null) return;
         rotateDB.display();
      }
      else if (action.equals("scale"))
      {
         if (frame == null) return;
         scaleDB.display();
      }
      else if (action.equals("shear"))
      {
         if (frame == null) return;
         shearDB.display();
      }
      else if (source == reverseItem)
      {
         if (frame == null) return;
         frame.reverseSelectedPaths();
      }
      else if (source == convertToPathItem)
      {
         if (frame == null) return;
         frame.convertToPath();
      }
      else if (source == convertToFullPathItem)
      {
         if (frame == null) return;
         frame.convertToFullPath();
      }
      else if (action.equals("pattern"))
      {
         if (frame == null) return;

         patternBox.display();  
      }
      else if (source == editPatternItem)
      {
         if (frame == null) return;

         patternBox.display(frame.getSelectedPatternIndex());  
      }
      else if (source == removePatternItem)
      {
         if (frame == null) return;

         frame.removePattern();
      }
      else if (action.equals("textpath"))
      {
         if (frame == null) return;
         frame.convertToTextPath();
      }
      else if (source == separateItem)
      {
         if (frame == null) return;
         frame.separate();
      }
      else if (source == mergePathsItem)
      {
         if (frame == null) return;
         frame.mergePaths();
      }
      else if (source == pathUnionItem)
      {
         if (frame == null) return;
         frame.pathUnion();
      }
      else if (source == xorPathsItem)
      {
         if (frame == null) return;
         frame.xorPaths();
      }
      else if (source == intersectPathsItem)
      {
         if (frame == null) return;
         frame.pathIntersect();
      }
      else if (source == subtractPathsItem)
      {
         if (frame == null) return;
         frame.subtractPaths();
      }
      else if (source == splitTextItem)
      {
         if (frame == null) return;
         frame.splitText();
      }
      else if (action.equals("group"))
      {
         if (frame == null) return;
         frame.group();
      }
      else if (action.equals("ungroup"))
      {
         if (frame == null) return;
         frame.ungroup();
      }
      else if (source == leftAlignItem)
      {
         if (frame == null) return;
         frame.leftAlign();
      }
      else if (source == centreAlignItem)
      {
         if (frame == null) return;
         frame.centreAlign();
      }
      else if (source == rightAlignItem)
      {
         if (frame == null) return;
         frame.rightAlign();
      }
      else if (source == topAlignItem)
      {
         if (frame == null) return;
         frame.topAlign();
      }
      else if (source == middleAlignItem)
      {
         if (frame == null) return;
         frame.middleAlign();
      }
      else if (source == bottomAlignItem)
      {
         if (frame == null) return;
         frame.bottomAlign();
      }
      else if (action.equals("select"))
      {
         if (frame == null) return;
         frame.setAction(ACTION_SELECT);
      }
      else if (action.equals("openline"))
      {
         if (frame == null) return;
         frame.setAction(ACTION_OPEN_LINE);
      }
      else if (action.equals("closedline"))
      {
         if (frame == null) return;
         frame.setAction(ACTION_CLOSED_LINE);
      }
      else if (action.equals("opencurve"))
      {
         if (frame == null) return;
         frame.setAction(ACTION_OPEN_CURVE);
      }
      else if (action.equals("closedcurve"))
      {
         if (frame == null) return;
         frame.setAction(ACTION_CLOSED_CURVE);
      }
      else if (action.equals("rectangle"))
      {
         if (frame == null) return;
         frame.setAction(ACTION_RECTANGLE);
      }
      else if (action.equals("ellipse"))
      {
         if (frame == null) return;
         frame.setAction(ACTION_ELLIPSE);
      }
      else if (action.equals("text"))
      {
         if (frame == null) return;
         frame.setAction(ACTION_TEXT);
      }
      else if (action.equals("gap"))
      {
         if (frame == null) return;
         frame.gap();
      }
      else if (action.equals("abandon"))
      {
         if (frame == null) return;
         frame.abandonPath();
      }
      else if (action.equals("finish"))
      {
         if (frame == null) return;
         frame.finishPath();
      }
      else if (source == insertBitmapItem)
      {
         if (frame == null) return;

         displayInsertBitmapChooser(frame);
      }
      else if (source == refreshItem)
      {
         GraphicsConfiguration gc = getGraphicsConfiguration();
         JDRUnit.setNormalizingFactor(gc);

         if (frame == null) return;
         frame.refresh();
      }
      else if (source == bitmapPropItem)
      {
         if (frame == null) return;
         displayBitmapPropertiesChooser();
      }
      else if (action.equals("vectorize"))
      {
         if (frame == null) return;
         displayVectorizeBitmapDialog();
      }
      else if (source == clearAllItem)
      {
         if (frame == null) return;
         frame.unsetAllFlowFrames();
      }
      else if (source == setFrameItem)
      {
         if (frame == null) return;
         flfSelector.initialise();
      }
      else if (source == setTypeblockItem)
      {
         if (frame == null) return;
         setTypeblockSelector.initialise();
      }
      else if (source == stylesItem)
      {
         if (frame == null) return;
         styleChooserBox.display();
      }
      else if (source == showToolsItem)
      {
         boolean flag = showToolsItem.isSelected();
         toolBar.setVisible(flag);
         sidePanel.setVisible(flag);
         appSettings.showToolBar=flag;
      }
      else if (source == showRulersItem)
      {
         if (frame == null) return;
         frame.showRulers(showRulersItem.isSelected());
      }
      else if (source == showStatusBarItem)
      {
         boolean flag = showStatusBarItem.isSelected();
         statusBar.setVisible(flag);
         appSettings.showStatus=flag;
      }
      else if (action.equals("showgrid"))
      {
         if (frame == null) return;

         setGridDisplay(showGridButtonItem.isSelected());
      }
      else if (action.equals("lockgrid"))
      {
         if (frame == null) return;

         frame.lockGrid(lockGridButtonItem.isSelected());

         statusBar.setLock(lockGridButtonItem.isSelected());
      }
      else if (source == gridSettingsItem)
      {
         if (frame == null) return;
         gridSettingsChooserBox.initialise();
      }
      else if (source == zoomSettingsItem)
      {
         if (frame == null) return;
         zoomSettingsChooserBox.display();
      }
      else if (source == zoomWidthItem)
      {
         if (frame == null) return;
         frame.zoomWidth();
         zoomSettingsChooserBox.setMag(frame.getMagnification());
         zoomSettingsItem.setSelected(true);
      }
      else if (source == zoomHeightItem)
      {
         if (frame == null) return;
         frame.zoomHeight();
         zoomSettingsChooserBox.setMag(frame.getMagnification());
         zoomSettingsItem.setSelected(true);
      }
      else if (source == zoomPageItem)
      {
         if (frame == null) return;
         frame.zoomPage();
         zoomSettingsChooserBox.setMag(frame.getMagnification());
         zoomSettingsItem.setSelected(true);
      }
      else if (source == zoom25Item)
      {
         if (frame == null) return;
         frame.setMagnification(0.25);
      }
      else if (source == zoom50Item)
      {
         if (frame == null) return;
         frame.setMagnification(0.5);
      }
      else if (source == zoom75Item)
      {
         if (frame == null) return;
         frame.setMagnification(0.75);
      }
      else if (source == zoom100Item)
      {
         if (frame == null) return;
         frame.setMagnification(1.0);
      }
      else if (source == zoom200Item)
      {
         if (frame == null) return;
         frame.setMagnification(2.0);
      }
      else if (source == zoom400Item)
      {
         if (frame == null) return;
         frame.setMagnification(4.0);
      }
      else if (source == zoom800Item)
      {
         if (frame == null) return;
         frame.setMagnification(8.0);
      }
      else if (source == showPrinterMarginsItem)
      {
         if (frame == null) return;
         frame.setMargins();
      }
      else if (source == portraitItem)
      {
         if (frame == null) return;
         frame.setOrientation(true);
      }
      else if (source == landscapeItem)
      {
         if (frame == null) return;
         frame.setOrientation(false);
      }
      else if (source == paperA5Item)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.A5 : JDRPaper.A5R);
      }
      else if (source == paperA4Item)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.A4 : JDRPaper.A4R);
      }
      else if (source == paperA3Item)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.A3 : JDRPaper.A3R);
      }
      else if (source == paperA2Item)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.A2 : JDRPaper.A2R);
      }
      else if (source == paperA1Item)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.A1 : JDRPaper.A1R);
      }
      else if (source == paperA0Item)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.A0 : JDRPaper.A0R);
      }
      else if (source == paperLetterItem)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.LETTER : JDRPaper.LETTERR);
      }
      else if (source == paperLegalItem)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.LEGAL : JDRPaper.LEGALR);
      }
      else if (source == paperExecutiveItem)
      {
         if (frame == null) return;
         frame.setPaperSize(portraitItem.isSelected() ?
                            JDRPaper.EXECUTIVE : JDRPaper.EXECUTIVER);
      }
      else if (source == paperOtherItem)
      {
         if (frame == null) return;
         paperDialog.initialise(portraitItem.isSelected());
      }
      else if (source == configSettingsItem)
      {
         configDialog.display();
      }
      else if (source == setNormalSizeItem)
      {
         if (frame == null) return;
         texSettingsDialog.display(frame);
      }
      else if (source == autoAdjustAnchorItem)
      {
         setAutoAnchor(autoAdjustAnchorItem.isSelected());
      }
      else if (source == autoEscapeSpCharsItem)
      {
         setAutoEscapeSpChars(autoEscapeSpCharsItem.isSelected());
      }
      else if (source == licenceItem)
      {
         licenceDialog.setVisible(true);
      }
      else if (source == aboutItem)
      {
         aboutDialog.setVisible(true);
      }
      else if (source == tileItem)
      {
         tileFrames();
         frame = (JDRFrame)theDesktop.getSelectedFrame();
      }
      else if (source == verticallyItem)
      {
         tileFramesVertically();
         frame = (JDRFrame)theDesktop.getSelectedFrame();
      }
      else if (source == horizontallyItem)
      {
         tileFramesHorizontally();
         frame = (JDRFrame)theDesktop.getSelectedFrame();
      }
      else if (source == minimizeItem)
      {
         minimizeAll();
         frame = (JDRFrame)theDesktop.getSelectedFrame();
      }
      else if (source == maximizeItem)
      {
         maximizeAll();
         frame = (JDRFrame)theDesktop.getSelectedFrame();
      }
      else if (source == gotoItem)
      {
         if (frame == null) return;
         gotoDialog.initialise();
      }
      else if (source == nextItem)
      {
         if (frame == null) return;
         frame.getCanvas().selectNextObject();
      }
      else if (source == addNextItem)
      {
         if (frame == null) return;
         frame.getCanvas().addNextObject();
      }
      else if (source == skipItem)
      {
         if (frame == null) return;
         frame.getCanvas().skipObject();
      }
      else if (source == findSelectedItem)
      {
         if (frame == null) return;
         frame.getCanvas().findSelectedObjects();
      }
      else if (source == findByDescriptionItem)
      {
         if (frame == null) return;
         displayFindByDescriptionDialog();
      }
      else if (source == addByDescriptionItem)
      {
         if (frame == null) return;
         findByDescDialog.initialise(false);
      }
      else if (source == displayPageItem)
      {
         if (frame == null) return;
         displayPageDialog.display();
      }

      if (frame != null)
      {
         frame.requestFocusInWindow();
      }
   }

   public void pageDialog(JDRFrame frame)
   {
      if (disablePrint)
      {
         JDRResources.error(this,
            JDRResources.getString("error.printing.no_service"));

         return;
      }

      PrinterJob printJob = PrinterJob.getPrinterJob();

      Rectangle2D rect = getPrintableArea();

      MediaPrintableArea mpa = new MediaPrintableArea(
          (float)(rect.getX()/72), (float)(rect.getY()/72),
          (float)(rect.getWidth()/72), (float)(rect.getHeight()/72),
          MediaPrintableArea.INCH);

      if (printJob != null)
      {
         printRequestAttributeSet.add(mpa);

         PageFormat pf
            = printJob.pageDialog(printRequestAttributeSet);

         if (pf != null)
         {
            frame.setMargins(pf);
         }
      }
      else
      {
          JDRResources.error(this,
             JDRResources.getString("error.printing.no_service"));
      }
   }

   public PrintService getPrintService(JDRPaper p)
   {
      if (printRequestAttributeSet == null)
      {
         return null;
      }

      // set paper size
      printRequestAttributeSet.add(p.isPortrait() ?
               OrientationRequested.PORTRAIT :
               OrientationRequested.LANDSCAPE);
      printRequestAttributeSet.add(p.getMediaSizeName());

      // set paper margins

      printRequestAttributeSet.add(p.getMediaPrintableArea());

      // find print services
      PrintService[] services = PrinterJob.lookupPrintServices();

      if (services.length > 0)
      {
         return services[0];
      }

      return null;
   }

   public Rectangle2D getPrintableArea()
   {
      return getCurrentSettings().getImageableArea();
   }

   public void doPrintJob(PrinterJob printJob)
      throws PrinterException
   {
      if (disablePrint)
      {
          JDRResources.error(this,
             JDRResources.getString("error.printing.no_service"));

          return;
      }

      if (printJob.printDialog(printRequestAttributeSet))
      {
         printJob.print(printRequestAttributeSet);
      }
   }

   public void displayObjectDescriptionDialog(JDRCompleteObject object)
   {
      if (object != null)
      {
         objectDescriptionDialog.initialise(object);
      }
   }

   public void displayImageDescriptionDialog(JDRCompleteObject object)
   {
      if (object != null)
      {
         imageDescriptionDialog.initialise(object);
      }
   }

   public void displayFindByDescriptionDialog()
   {
      findByDescDialog.initialise(true);
   }

   public void displayEditTextDialog()
   {
      editTextBox.display();
   }

   public void setEditPathState(JDRFrame frame, boolean state)
   {
      if (editPathButtonItem.isSelected() != state)
      {
         editPathButtonItem.setSelected(state);
      }

      if (state)
      {
         frame.editPath();
         statusBar.setInfo(JDRResources.getString("info.edit_path"));
      }
      else
      {
         frame.finishEditPath();
         statusBar.setInfo(JDRResources.getString("info.select"));
      }
   }

   public void displayInsertBitmapChooser(JDRFrame frame)
   {
      int result = bitmapFC.showOpenDialog(this);

      if (result == JFileChooser.APPROVE_OPTION)
      {
         String filename = bitmapFC.getSelectedFile().getAbsolutePath();

         frame.insertBitmap(filename);
      }
   }

   public void displayBitmapPropertiesChooser()
   {
      bitmapPropChooserBox.initialise();
   }

   public void displayVectorizeBitmapDialog()
   {
      vectorizeBitmapDialog.display();
   }

   public void displayTextPaintChooser()
   {
      textPaintChooserBox.initialise();
   }

   public void displayFontFamilyChooser()
   {
      fontFamilyChooserBox.initialise();
   }

   public void displayFontSizeChooser()
   {
      fontSizeChooserBox.initialise();
   }

   public void displayFontShapeChooser()
   {
      fontShapeChooserBox.initialise();
   }

   public void displayFontSeriesChooser()
   {
      fontSeriesChooserBox.initialise();
   }

   public void displayFontAnchorChooser()
   {
      fontAnchorChooserBox.initialise();
   }

   public void displayFontHAnchorChooser()
   {
      fontHAnchorChooserBox.initialise();
   }

   public void displayFontVAnchorChooser()
   {
      fontVAnchorChooserBox.initialise();
   }

   public void displayLinePaintChooser()
   {
      linePaintChooserBox.initialise();
   }

   public void displayFillPaintChooser()
   {
      fillPaintChooserBox.initialise();
   }

   public void displayLineWidthChooser()
   {
      lineWidthChooserBox.initialise();
   }

   public void displayDashPatternChooser()
   {
      dashPatternChooserBox.initialise();
   }

   public void displayJoinStyleChooser()
   {
      joinStyleChooserBox.initialise();
   }

   public void displayStartMarkerChooser()
   {
      startArrowChooserBox.initialise();
   }

   public void displayMidMarkerChooser()
   {
      midArrowChooserBox.initialise();
   }

   public void displayEndMarkerChooser()
   {
      endArrowChooserBox.initialise();
   }

   public void displayAllMarkerChooser()
   {
      allMarkersChooserBox.initialise();
   }

   public void internalFrameClosing(InternalFrameEvent e)
   {
   }

   public void internalFrameClosed(InternalFrameEvent e)
   {
   }

   public void internalFrameOpened(InternalFrameEvent e)
   {
   }

   public void internalFrameIconified(InternalFrameEvent e)
   {
      if (!iconifyFrameUndoRedo)
      {
         UndoableEdit edit = new IconifyFrame(e.getInternalFrame(),
            true);
         postEdit(edit);
      }
   }
 
   public void internalFrameDeiconified(InternalFrameEvent e)
   {
      if (!iconifyFrameUndoRedo)
      {
         UndoableEdit edit = new IconifyFrame(e.getInternalFrame(),
            false);
         postEdit(edit);
      }
   }

   public void internalFrameActivated(InternalFrameEvent e)
   {
      if (!activateFrameUndoRedo)
      {
         UndoableEdit edit = new ActivateFrame(e.getInternalFrame(),
            true);
         postEdit(edit);
      }
   }

   public void internalFrameDeactivated(InternalFrameEvent e)
   {
      setCurrentFile("");

      if (!activateFrameUndoRedo)
      {
         UndoableEdit edit = new ActivateFrame(e.getInternalFrame(),
            false);
         postEdit(edit);
      }
   }

   class ActivateFrame extends AbstractUndoableEdit
   {
      private JInternalFrame frame_;
      private boolean activate_;

      public ActivateFrame(JInternalFrame frame, boolean activate)
      {
         frame_ = frame;
         activate_ = activate;
      }

      public void redo() throws CannotRedoException
      {
         if (frame_ == null)
         {
            throw new CannotRedoException();
         }

         activateFrameUndoRedo=true;

         try
         {
            frame_.setSelected(activate_);
         }
         catch (java.beans.PropertyVetoException e)
         {
            throw new CannotRedoException();
         }

         activateFrameUndoRedo=false;
      }

      public void undo() throws CannotUndoException
      {
         if (frame_ == null)
         {
            throw new CannotUndoException();
         }

         activateFrameUndoRedo=true;

         try
         {
            frame_.setSelected(!activate_);
         }
         catch (java.beans.PropertyVetoException e)
         {
            throw new CannotUndoException();
         }

         activateFrameUndoRedo=false;
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return activate_ 
                ? JDRResources.getString("undo.select_child_window")
                : JDRResources.getString("undo.deselect_child_window");
      }
   }

   class IconifyFrame extends AbstractUndoableEdit
   {
      private boolean iconify_;
      private JInternalFrame frame_;

      public IconifyFrame(JInternalFrame frame, boolean iconify)
      {
         frame_ = frame;
         iconify_ = iconify;
      }

      public void redo() throws CannotRedoException
      {
         iconifyFrameUndoRedo=true;

         try
         {
            frame_.setIcon(iconify_);
         }
         catch (java.beans.PropertyVetoException e)
         {
            throw new CannotRedoException();
         }

         iconifyFrameUndoRedo=false;
      }

      public void undo() throws CannotUndoException
      {
         iconifyFrameUndoRedo=true;

         try
         {
            frame_.setIcon(!iconify_);
         }
         catch (java.beans.PropertyVetoException e)
         {
            throw new CannotUndoException();
         }

         iconifyFrameUndoRedo=false;
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return iconify_ 
                ? JDRResources.getString("undo.iconify_child_window")
                : JDRResources.getString("undo.deiconify_child_window");
      }
   }

   public static boolean getPreviewBitmaps()
   {
      return appSettings.previewBitmaps;
   }

   public static void setPreviewBitmaps(boolean selected)
   {
      appSettings.previewBitmaps = selected;
   }

   public static boolean isRobotEnabled()
   {
      return (appSettings.robot != null);
   }

   public static boolean isLaTeXFontUpdateEnabled()
   {
      return appSettings.updateLaTeXFonts;
   }

   public static void setLaTeXFontUpdate(boolean update)
   {
      appSettings.updateLaTeXFonts = update;
   }

   public static void setAutoAnchor(boolean update)
   {
      appSettings.autoUpdateAnchors = update;
   }

   public static boolean isAutoAnchorEnabled()
   {
      return appSettings.autoUpdateAnchors;
   }

   public static void setAutoEscapeSpChars(boolean enabled)
   {
      appSettings.autoEscapeSpChars = enabled;
   }

   public static boolean isAutoEscapeSpCharsEnabled()
   {
      return appSettings.autoEscapeSpChars;
   }

   public static void moveMouse(int screenX, int screenY)
   {
      if (appSettings.robot != null)
      {
         appSettings.robot.mouseMove(screenX, screenY);
      }
   }

   public String[] getFontFamilies()
   {
      return availableFontFamilies;
   }

   public int initSettings()
   {
      return appSettings.initSettings;
   }

   public void setInitSettings(int i)
   {
      appSettings.initSettings=i;

      if (appSettings.initSettings==JpgfDrawSettings.INIT_USER)
      {
         saveUserSettings();
      }
   }

   public boolean saveJDRsettings()
   {
      return appSettings.saveJDRsettings;
   }

   public boolean useJDRsettings()
   {
      return appSettings.useJDRsettings;
   }

   public boolean warnOnOldJdr()
   {
      return appSettings.warnOnOldJdr;
   }

   public void setJDRsettings(boolean saveJDRSettings, 
                              boolean useJDRSettings,
                              boolean warnOnOld)
   {
      appSettings.saveJDRsettings=saveJDRSettings;
      appSettings.useJDRsettings=useJDRSettings;
      appSettings.warnOnOldJdr = warnOnOld;
   }

   public int getStartDirType()
   {
      return appSettings.startDirType;
   }

   public String getStartDirectory()
   {
      return appSettings.startDir;
   }

   public void setStartDirectory(int startDirType, String dir)
   {
      appSettings.startDirType = startDirType;

      switch (startDirType)
      {
         case JpgfDrawSettings.STARTDIR_CWD :
            appSettings.startDir = ".";
         break;
         case JpgfDrawSettings.STARTDIR_NAMED :
            appSettings.startDir = dir;
         break;
      }
   }

   public void setCurrentFile(String name)
   {
      setCurrentFile(name, false);
   }

   public void setCurrentFile(String name, boolean modified)
   {
      if (name == null)
      {
         return;
      }

      currentFile = name;

      setModified(modified);

      File file = new File(currentFile);

      File parent = file.getParentFile();

      if (parent != null)
      {
         savejdrFC.setCurrentDirectory(parent);
         openjdrFC.setCurrentDirectory(parent);
      }
   }

   public void setModified(boolean modified)
   {
      statusBar.setModified(modified);
      if (modified)
      {
         if (currentFile.equals(""))
         {
            setTitle(JpgfDraw.appName+" "
                     +"["+JDRResources.getString("info.modified")+"]");
         }
         else
         {
            setTitle(JpgfDraw.appName+" - "+currentFile
                     +"["+JDRResources.getString("info.modified")+"]");
         }
      }
      else
      {
         if (currentFile == null || currentFile.equals(""))
         {
            setTitle(JpgfDraw.appName);
         }
         else
         {
            setTitle(JpgfDraw.appName+" - "+currentFile);
         }
      }
   }

   public double getCurrentMagnification()
   {
      return getCurrentFrame().getMagnification();
   }

   public void setCurrentMagnification(double factor)
   {
      if (factor <= 0)
      {
         return;
      }

      JDRFrame frame = getCurrentFrame();

      if (frame != null)
      {
         frame.setMagnification(factor);
      }
   }

   /**
    * Gets the magnification specified by the zoom menu.
    */
   public double getMagnification()
   {
      if (zoom25Item.isSelected())
      {
         return 0.25;
      }
      else if (zoom50Item.isSelected())
      {
         return 0.5;
      }
      else if (zoom75Item.isSelected())
      {
         return 0.75;
      }
      else if (zoom100Item.isSelected())
      {
         return 1;
      }
      else if (zoom200Item.isSelected())
      {
         return 2;
      }
      else if (zoom400Item.isSelected())
      {
         return 4;
      }
      else if (zoom800Item.isSelected())
      {
         return 8;
      }
      else
      {
         return zoomSettingsChooserBox.getMag();
      }
   }

   public boolean showMargins()
   {
      return showPrinterMarginsItem.isSelected();
   }

   public static boolean dragScaleEnabled()
   {
      return appSettings.enableDragScale;
   }

   public static void setDragScale(boolean isEnabled)
   {
      appSettings.enableDragScale = isEnabled;
   }

   public void setCurrentPosition(String position)
   {
      statusBar.setPosition(position);
   }

   public boolean isPathEdited()
   {
      return editPathButtonItem.isSelected();
   }

   public void setEditPathButton(boolean flag)
   {
      editPathButtonItem.setSelected(flag);

      if (flag)
      {
         statusBar.setInfo(JDRResources.getString("info.edit_path"));
      }
      else
      {
         statusBar.setInfo(JDRResources.getString("info.select"));
      }
   }

   public void resetGapButton()
   {
      gapButtonItem.setSelected(false);
   }

   public void setTool(int tool)
   {
      String info = "";
      JDRToolButtonItem toolButtonItem = null;

      switch (tool)
      {
         case ACTION_SELECT :
            info = JDRResources.getString("info.select");
            toolButtonItem = selectButtonItem;
         break;
         case ACTION_OPEN_LINE :
            info = JDRResources.getString("info.open_path");
            toolButtonItem = openLineButtonItem;
         break;
         case ACTION_CLOSED_LINE :
            info = JDRResources.getString("info.closed_path");
            toolButtonItem = closedLineButtonItem;
         break;
         case ACTION_OPEN_CURVE :
            info = JDRResources.getString("info.open_path");
            toolButtonItem = openCurveButtonItem;
         break;
         case ACTION_CLOSED_CURVE :
            info = JDRResources.getString("info.closed_path");
            toolButtonItem = closedCurveButtonItem;
         break;
         case ACTION_RECTANGLE :
            info = JDRResources.getString("info.rectangle");
            toolButtonItem = rectangleButtonItem;
         break;
         case ACTION_ELLIPSE :
            info = JDRResources.getString("info.ellipse");
            toolButtonItem = ellipseButtonItem;
         break;
         case ACTION_TEXT :
            info = JDRResources.getString("info.text");
            toolButtonItem = textButtonItem;
         break;
      }

      if (toolButtonItem != null & !toolButtonItem.isSelected())
      {
         toolButtonItem.setSelected(true);
      }

      statusBar.setInfo(info);
      JDRFrame frame = getSelectedFrame();
      if (frame != null) frame.enableTools();
   }

   public String getStatusInfo()
   {
      return statusBar.getInfo();
   }

   public void setStatusInfo(String info)
   {
      statusBar.setInfo(info);
   }

   public void updateStatus(int n)
   {
      statusBar.noFramesSelected(n);
   }

   public void updateStatus()
   {
      JDRFrame frame = getSelectedFrame();

      if (frame == null)
      {
         JInternalFrame[] f = theDesktop.getAllFrames();

         updateStatus(f.length);
      }
   }

   public void setPaperSize(JDRPaper paper)
   {
      if (paper == JDRPaper.A0)
      {
         paperA0Item.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.A1)
      {
         paperA1Item.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.A2)
      {
         paperA2Item.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.A3)
      {
         paperA3Item.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.A4)
      {
         paperA4Item.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.A5)
      {
         paperA5Item.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.LETTER)
      {
         paperLetterItem.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.LEGAL)
      {
         paperLegalItem.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.EXECUTIVE)
      {
         paperExecutiveItem.setSelected(true);
         portraitItem.setSelected(true);
      }
      else if (paper == JDRPaper.A0R)
      {
         paperA0Item.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else if (paper == JDRPaper.A1R)
      {
         paperA1Item.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else if (paper == JDRPaper.A2R)
      {
         paperA2Item.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else if (paper == JDRPaper.A3R)
      {
         paperA3Item.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else if (paper == JDRPaper.A4R)
      {
         paperA4Item.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else if (paper == JDRPaper.A5R)
      {
         paperA5Item.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else if (paper == JDRPaper.LETTERR)
      {
         paperLetterItem.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else if (paper == JDRPaper.LEGALR)
      {
         paperLegalItem.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else if (paper == JDRPaper.EXECUTIVER)
      {
         paperExecutiveItem.setSelected(true);
         landscapeItem.setSelected(true);
      }
      else
      {
         paperOtherItem.setSelected(true);
         paperDialog.setPaper(paper);
         if (paper.isPortrait())
         {
            portraitItem.setSelected(true);
         }
         else
         {
            landscapeItem.setSelected(true);
         }
      }

      getPrintService(paper);
   }

   public void showMoveByDialog()
   {
      if (moveByItem.isEnabled())
      {
         moveByDialog.initialise();
      }
   }

   public void setRulers(boolean flag)
   {
      showRulersItem.setSelected(flag);
   }

   public void setGridDisplay(boolean flag)
   {
      JDRFrame frame = (JDRFrame)theDesktop.getSelectedFrame();

      showGridButtonItem.setSelected(flag);

      if (frame != null) frame.showGrid(flag);
   }

   public void setLockGrid(boolean flag)
   {
      lockGridButtonItem.setSelected(flag);
      statusBar.setLock(flag);
   }

   public void applySettings(CanvasSettings s)
   {
      setRulers(s.showRulers);
      setGridDisplay(s.isGridDisplayed());
      setTool(s.getTool());
      setPaperSize(s.paper);
   }

   public boolean getAntiAlias()
   {
      return appSettings.antialias;
   }

   public boolean getRenderQuality()
   {
      return appSettings.renderquality;
   }

   public RenderingHints getRenderingHints()
   {
      return renderHints;
   }

   public static void setRendering(boolean antialias, boolean renderquality)
   {
      appSettings.antialias     = antialias;
      appSettings.renderquality = renderquality;

      if (antialias)
      {
         renderHints = 
            new RenderingHints(RenderingHints.KEY_ANTIALIASING,
                               RenderingHints.VALUE_ANTIALIAS_ON);
      }
      else
      {
         renderHints = 
            new RenderingHints(RenderingHints.KEY_ANTIALIASING,
                               RenderingHints.VALUE_ANTIALIAS_OFF);
      }

      if (renderquality)
      {
         renderHints.add(new RenderingHints(
                           RenderingHints.KEY_RENDERING,
                           RenderingHints.VALUE_RENDER_QUALITY));
      }
      else
      {
         renderHints.add(new RenderingHints(
                           RenderingHints.KEY_RENDERING,
                           RenderingHints.VALUE_RENDER_SPEED));
      }
   }

   public JDRPaper getCurrentPaper()
   {
      boolean isPortrait = portraitItem.isSelected();

      if (paperA0Item.isSelected())
      {
         return isPortrait ? JDRPaper.A0 : JDRPaper.A0R;
      }
      else if (paperA1Item.isSelected())
      {
         return isPortrait ? JDRPaper.A1 : JDRPaper.A1R;
      }
      else if (paperA2Item.isSelected())
      {
         return isPortrait ? JDRPaper.A2 : JDRPaper.A2R;
      }
      else if (paperA3Item.isSelected())
      {
         return isPortrait ? JDRPaper.A3 : JDRPaper.A3R;
      }
      else if (paperA4Item.isSelected())
      {
         return isPortrait ? JDRPaper.A4 : JDRPaper.A4R;
      }
      else if (paperA5Item.isSelected())
      {
         return isPortrait ? JDRPaper.A5 : JDRPaper.A5R;
      }
      else if (paperLetterItem.isSelected())
      {
         return isPortrait ? JDRPaper.LETTER : JDRPaper.LETTERR;
      }
      else if (paperLegalItem.isSelected())
      {
         return isPortrait ? JDRPaper.LEGAL : JDRPaper.LEGALR;
      }
      else if (paperExecutiveItem.isSelected())
      {
         return isPortrait ? JDRPaper.EXECUTIVE : JDRPaper.EXECUTIVER;
      }

      return paperDialog.getPaper();
   }

   public void setCurrentSettings(JDRPaint linePaint,
      JDRPaint fillPaint, JDRPaint textPaint, JDRBasicStroke stroke,
      String fontFamilyName, int fontSize, int fontSeries,
      int fontShape, String latexFamily, String latexFontSize,
      String latexSeries, String latexFontShape, int pgfHalign,
      int pgfValign)
   {
      UndoableEdit edit = new SetCurrentSettings(linePaint,
         fillPaint, textPaint, stroke, fontFamilyName, fontSize,
         fontSeries, fontShape, latexFamily, latexFontSize,
         latexSeries, latexFontShape, pgfHalign, pgfValign);

      postEdit(edit);
   }

   class SetCurrentSettings extends AbstractUndoableEdit
   {
      private String string_=JDRResources.getString("undo.set_current_settings");
      private JDRPaint oldLinePaint_, linePaint_, oldFillPaint_,
         fillPaint_, oldTextPaint_, textPaint_;
      private JDRBasicStroke oldStroke_, stroke_;
      private String oldFamilyName_, familyName_, oldLatexFamily_,
        latexFamily_, oldLatexSize_, latexSize_, oldLatexSeries_,
        latexSeries_, oldLatexShape_, latexShape_;
      private int oldSize_, size_, oldSeries_, series_, oldShape_,
         shape_, oldHalign_, hAlign_, oldValign_, vAlign_;

      public SetCurrentSettings(JDRPaint linePaint,
         JDRPaint fillPaint, JDRPaint textPaint, JDRBasicStroke stroke,
         String fontFamilyName, int fontSize, int fontSeries,
         int fontShape, String latexFamily, String latexFontSize,
         String latexFontSeries, String latexFontShape, int pgfHalign,
         int pgfValign)
      {
         oldLinePaint_ = appSettings.linePaint;
         linePaint_    = linePaint;
         oldFillPaint_ = appSettings.fillPaint;
         fillPaint_    = fillPaint;
         oldTextPaint_ = appSettings.textPaint;
         textPaint_    = textPaint;
         oldStroke_    = appSettings.stroke;
         stroke_       = stroke;
         oldFamilyName_ = appSettings.fontFamily;
         familyName_    = fontFamilyName;
         oldSize_       = appSettings.fontSize;
         size_          = fontSize;
         oldShape_      = appSettings.fontShape;
         shape_         = fontShape;
         oldSeries_     = appSettings.fontSeries;
         series_        = fontSeries;
         oldHalign_     = appSettings.pgfHalign;
         hAlign_        = pgfHalign;
         oldValign_     = appSettings.pgfValign;
         vAlign_        = pgfValign;
         oldLatexFamily_ = appSettings.latexFontFamily;
         latexFamily_    = latexFamily;
         oldLatexShape_  = appSettings.latexFontShape;
         latexShape_     = latexFontShape;
         oldLatexSize_   = appSettings.latexFontSize;
         latexSize_      = latexFontSize;
         oldLatexSeries_ = appSettings.latexFontSeries;
         latexSeries_    = latexFontSeries;

         appSettings.linePaint  = linePaint_;
         appSettings.fillPaint  = fillPaint_;
         appSettings.textPaint  = textPaint_;
         appSettings.stroke     = stroke_;
         appSettings.fontFamily = familyName_;
         appSettings.fontSize   = size_;
         appSettings.fontSeries = series_;
         appSettings.fontShape  = shape_;
         appSettings.pgfHalign  = hAlign_;
         appSettings.pgfValign  = vAlign_;
         appSettings.latexFontFamily = latexFamily_;
         appSettings.latexFontSeries = latexSeries_;
         appSettings.latexFontSize   = latexSize_;
         appSettings.latexFontShape  = latexShape_;
         JDRFrame frame = (JDRFrame)theDesktop.getSelectedFrame();
         if (frame!=null) frame.resetTextField();
      }

      public void redo() throws CannotRedoException
      {
         appSettings.linePaint  = linePaint_;
         appSettings.fillPaint  = fillPaint_;
         appSettings.textPaint  = textPaint_;
         appSettings.stroke     = stroke_;
         appSettings.fontFamily = familyName_;
         appSettings.fontSize   = size_;
         appSettings.fontSeries = series_;
         appSettings.fontShape  = shape_;
         appSettings.pgfHalign  = hAlign_;
         appSettings.pgfValign  = vAlign_;
         appSettings.latexFontFamily = latexFamily_;
         appSettings.latexFontSeries = latexSeries_;
         appSettings.latexFontSize   = latexSize_;
         appSettings.latexFontShape  = latexShape_;
         JDRFrame frame = (JDRFrame)theDesktop.getSelectedFrame();
         if (frame!=null) frame.resetTextField();
      }

      public void undo() throws CannotUndoException
      {
         appSettings.linePaint  = oldLinePaint_;
         appSettings.fillPaint  = oldFillPaint_;
         appSettings.textPaint  = oldTextPaint_;
         appSettings.stroke     = oldStroke_;
         appSettings.fontFamily = oldFamilyName_;
         appSettings.fontSize   = oldSize_;
         appSettings.fontSeries = oldSeries_;
         appSettings.fontShape  = oldShape_;
         appSettings.pgfHalign  = oldHalign_;
         appSettings.pgfValign  = oldValign_;
         appSettings.latexFontFamily = oldLatexFamily_;
         appSettings.latexFontSeries = oldLatexSeries_;
         appSettings.latexFontSize   = oldLatexSize_;
         appSettings.latexFontShape  = oldLatexShape_;
         JDRFrame frame = (JDRFrame)theDesktop.getSelectedFrame();
         if (frame!=null) frame.resetTextField();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string_;
      }
   }

   public JDRBasicStroke getCurrentStroke()
   {
      return appSettings.stroke;
   }

   public JDRPaint getCurrentLinePaint()
   {
      return appSettings.linePaint;
   }

   public JDRPaint getCurrentFillPaint()
   {
      return appSettings.fillPaint;
   }

   public JDRPaint getCurrentTextPaint()
   {
      return appSettings.textPaint;
   }

   public String getCurrentFontFamily()
   {
      return appSettings.fontFamily;
   }

   public String getCurrentLaTeXFontFamily()
   {
      return appSettings.latexFontFamily;
   }

   public int getCurrentFontSize()
   {
      return appSettings.fontSize;
   }

   public int getCurrentFontSeries()
   {
      return appSettings.fontSeries;
   }

   public int getCurrentFontShape()
   {
      return appSettings.fontShape;
   }

   public int getCurrentPGFHAlign()
   {
      return appSettings.pgfHalign;
   }

   public int getCurrentPGFVAlign()
   {
      return appSettings.pgfValign;
   }

   public String getCurrentLaTeXFontSeries()
   {
      return appSettings.latexFontSeries;
   }

   public LaTeXFontBase getCurrentLaTeXFontBase()
   {
      JDRFrame frame = getCurrentFrame();

      if (frame != null)
      {
         return frame.latexFonts;
      }

      return null;
   }

   public String getCurrentLaTeXFontSize()
   {
      JDRFrame frame = getCurrentFrame();

      if (frame != null && frame.latexFonts != null)
      {
         return frame.latexFonts.getLaTeXCmd(
            getCurrentFontSize());
      }

      return appSettings.latexFontSize;
   }

   public String getCurrentLaTeXFontShape()
   {
      return appSettings.latexFontShape;
   }

   public int getCurrentFontWeight()
   {
      int fontSeries = getCurrentFontSeries();
      int fontShape  = getCurrentFontShape();

      int weight = 0;
      weight += (fontSeries == JDRFont.SERIES_MEDIUM?
                 Font.PLAIN : Font.BOLD);
      weight += (fontShape == JDRFont.SHAPE_UPRIGHT?
                 0 : Font.ITALIC);

      return weight;
   }

   private Font requestNewFontFamily()
   {
      //JComboBox<String> box = new JComboBox<String>(availableFontFamilies);//Java7
      JComboBox box = new JComboBox(availableFontFamilies);

      JOptionPane.showMessageDialog(this,
         new Object[]{
            JDRResources.getString("error.unknown_font")+": "
               +getCurrentFontFamily(),
            JDRResources.getString("error.unknown_font_select"),
            box},
            JDRResources.getString("error.unknown_font"),
            JOptionPane.ERROR_MESSAGE
         );

      appSettings.fontFamily = (String)box.getSelectedItem();
      appSettings.fontShape = JDRFont.SHAPE_UPRIGHT;
      appSettings.fontSeries = JDRFont.SERIES_MEDIUM;

      styleChooserBox.setFontName(appSettings.fontFamily);

      return new Font(getCurrentFontFamily(),
         getCurrentFontWeight(), getCurrentFontSize());
   }

   public Font getCurrentFont()
   {
      Font f;

      try
      {
         f = new Font(getCurrentFontFamily(),
         getCurrentFontWeight(), getCurrentFontSize());

         if (f == null)
         {
            f = requestNewFontFamily();
         }
      }
      catch (Exception e)
      {
         f = requestNewFontFamily();
      }

      return f;
   }

   public int getCurrentTool()
   {
      if (selectButtonItem.isSelected())
      {
         return ACTION_SELECT;
      }
      else if (openLineButtonItem.isSelected())
      {
         return ACTION_OPEN_LINE;
      }
      else if (closedLineButtonItem.isSelected())
      {
         return ACTION_CLOSED_LINE;
      }
      else if (openCurveButtonItem.isSelected())
      {
         return ACTION_OPEN_CURVE;
      }
      else if (closedCurveButtonItem.isSelected())
      {
         return ACTION_CLOSED_CURVE;
      }
      else if (rectangleButtonItem.isSelected())
      {
         return ACTION_RECTANGLE;
      }
      else if (ellipseButtonItem.isSelected())
      {
         return ACTION_ELLIPSE;
      }

      return ACTION_TEXT;
   }

   public void saveString(String str)
   {
      int result = texFC.showSaveDialog(this);

      if (result == JFileChooser.APPROVE_OPTION)
      {
         String filename = texFC.getSelectedFile().getAbsolutePath();

         if (texFCAddExt.isSelected())
         {
            if (!(filename.toLowerCase().endsWith(".tex")
                ||filename.toLowerCase().endsWith(".ltx")))
            {
               filename = filename + ".tex";
            }
         }

         // does the file already exist?
         File file = new File(filename);
         if (file.exists())
         {
            int selection = JOptionPane.showConfirmDialog(this,
               new String[] {filename,
               JDRResources.getString("warning.file_exists")},
               JDRResources.getString("warning.title"),
               JOptionPane.YES_NO_OPTION,
               JOptionPane.WARNING_MESSAGE);

               if (selection == JOptionPane.NO_OPTION) return;
         }

         // save string to file

         PrintWriter out;

         try
         {
            out = new PrintWriter(new FileWriter(filename));
         }
         catch (IOException e)
         {
            JDRResources.error(this, new String[]
            {JDRResources.getStringWithValue("error.io.open",
               filename),
            e.getMessage()});

            return;
         }
         catch (Exception e)
         {
            JDRResources.error(this, e);
            return;
         }

         out.println(str);

         out.close();
      }
   }

   public void undoableEditHappened(UndoableEditEvent evt)
   {
      UndoableEdit edit = evt.getEdit();
      undoManager.addEdit(edit);
      refreshUndoRedo();
   }

   private class UndoAction extends AbstractAction
   {
      public void actionPerformed(ActionEvent evt)
      {
         undoManager.undo();
         refreshUndoRedo();
      }
   }

   private class RedoAction extends AbstractAction
   {
      public void actionPerformed(ActionEvent evt)
      {
         undoManager.redo();
         refreshUndoRedo();
      }
   }

   public void refreshUndoRedo()
   {
      undoItem.setText(undoManager.getUndoPresentationName());
      undoItem.setEnabled(undoManager.canUndo());

      redoItem.setText(undoManager.getRedoPresentationName());
      redoItem.setEnabled(undoManager.canRedo());
   }

   public void postEdit(UndoableEdit edit)
   {
      undoSupport.postEdit(edit);
   }

   public void discardAllEdits()
   {
      undoManager.discardAllEdits();
      refreshUndoRedo();
   }

   public void disableUndoRedo()
   {
      undoItem.setEnabled(false);
      redoItem.setEnabled(false);
   }

   public void enableFrameRelatedItems(boolean flag)
   {
      imageDescriptionItem.setEnabled(flag);
      saveButtonItem.setEnabled(flag);
      saveAsItem.setEnabled(flag);
      exportItem.setEnabled(flag);
      printItem.setEnabled(flag && !disablePrint);
      //pageDialogItem.setEnabled(flag);
      closeItem.setEnabled(flag);
      selectAllButtonItem.setEnabled(flag);
      deselectAllItem.setEnabled(flag);
      moveByItem.setEnabled(flag);
      cutButtonItem.setEnabled(flag);
      copyButtonItem.setEnabled(flag);
      pasteButtonItem.setEnabled(flag);
      objectDescriptionItem.setEnabled(flag);
      pathM.setEnabled(flag);
      textM.setEnabled(flag);
      reduceToGreyItem.setEnabled(flag);
      fadeItem.setEnabled(flag);
      moveToFrontButtonItem.setEnabled(flag);
      moveToBackButtonItem.setEnabled(flag);
      navigateM.setEnabled(flag);
      transformM.setEnabled(flag);
      toolsM.setEnabled(flag);
      bitmapM.setEnabled(flag);
      texM.setEnabled(flag);
      showToolsItem.setEnabled(flag);
      showRulersItem.setEnabled(flag);
      gridM.setEnabled(flag);
      zoomM.setEnabled(flag);
      paperM.setEnabled(flag);

      editPathButtonItem.setEnabled(flag);
      rotateButtonItem.setEnabled(flag);
      scaleButtonItem.setEnabled(flag);
      shearButtonItem.setEnabled(flag);
      groupButtonItem.setEnabled(flag);
      ungroupButtonItem.setEnabled(flag);
      showGridButtonItem.setEnabled(flag);
      lockGridButtonItem.setEnabled(flag);
      setPatternButtonItem.setEnabled(flag);

      selectButtonItem.setEnabled(flag);
      openLineButtonItem.setEnabled(flag);
      closedLineButtonItem.setEnabled(flag);
      openCurveButtonItem.setEnabled(flag);
      closedCurveButtonItem.setEnabled(flag);
      rectangleButtonItem.setEnabled(flag);
      ellipseButtonItem.setEnabled(flag);
      textButtonItem.setEnabled(flag);
      gapButtonItem.setEnabled(flag);
      abandonButtonItem.setEnabled(flag);
      finishButtonItem.setEnabled(flag);

      if (flag)
      {
         JDRFrame frame = (JDRFrame)theDesktop.getSelectedFrame();
         if (frame != null) frame.enableTools();

         statusBar.enableStatus();
      }
   }

   public void enableEditTools(boolean enable)
   {
      enableEditTools(enable, enable);
   }

   public void enableEditTools(boolean enable, boolean enableGap)
   {
      gapButtonItem.setEnabled(enableGap);
      abandonButtonItem.setEnabled(enable);
      finishButtonItem.setEnabled(enable);
   }

   public void enableTools(int currentTool, int totalObjects, int[] number, boolean[] any)
   {
      if (currentTool == ACTION_SELECT && !editPathButtonItem.isSelected())
      {
         int n = number[JDRGroup.ANY];

         boolean anyNonTextShapes = any[JDRGroup.NON_TEXTUAL_SHAPE];

         boolean anyShapes = any[JDRGroup.SHAPE];

         boolean anyTextual = any[JDRGroup.TEXTUAL];

         int numPatterns = number[JDRGroup.PATTERN];

         int numNonTextShapes = number[JDRGroup.NON_TEXTUAL_SHAPE];

         int numShapes = number[JDRGroup.SHAPE];

         int numCompoundShapes = number[JDRGroup.COMPOUND];
   
         int numTextual = number[JDRGroup.TEXTUAL];

         insertBitmapItem.setEnabled(true);
         selectAllButtonItem.setEnabled(totalObjects > 0);     
         deselectAllItem.setEnabled(n > 0);     
         moveByItem.setEnabled(n > 0);     
         cutButtonItem.setEnabled(n > 0);
         copyButtonItem.setEnabled(n > 0);
         pasteButtonItem.setEnabled(true);
         objectDescriptionItem.setEnabled(n==1);
         pathM.setEnabled(anyShapes);

         editPathButtonItem.setEnabled(numShapes == 1 && n == 1);

         linePaintItem.setEnabled(anyNonTextShapes);
         fillPaintItem.setEnabled(anyNonTextShapes);
         lineStyleM.setEnabled(anyNonTextShapes);

         textM.setEnabled(anyTextual);
         textMatrixItem.setEnabled(numTextual == 1);
         editTextItem.setEnabled(numTextual == 1 && n == 1);

         reduceToGreyItem.setEnabled(anyShapes || anyTextual);
         fadeItem.setEnabled(reduceToGreyItem.isEnabled());

         moveToFrontButtonItem.setEnabled(totalObjects > 1 && n > 0);
         moveToBackButtonItem.setEnabled(moveToFrontButtonItem.isEnabled());

         rotateButtonItem.setEnabled(n > 0);
         scaleButtonItem.setEnabled(n > 0);
         shearButtonItem.setEnabled(n > 0);
         reverseItem.setEnabled(numShapes > 0);

         convertToPathItem.setEnabled(
           (anyTextual || anyShapes)
           && !any[JDRGroup.GROUP] && !any[JDRGroup.BITMAP]);

         patternM.setEnabled(numShapes > 0 && !any[JDRGroup.TEXT]
            && !any[JDRGroup.BITMAP] && !any[JDRGroup.GROUP]);

         setPatternButtonItem.setEnabled(
            numShapes > 0 && !any[JDRGroup.PATTERN] 
           && !any[JDRGroup.TEXT] && !any[JDRGroup.BITMAP]
           && !any[JDRGroup.GROUP]);

         editPatternItem.setEnabled(numPatterns == 1);

         removePatternItem.setEnabled(numPatterns > 0);

         convertToFullPathItem.setEnabled((any[JDRGroup.SYMMETRIC]
           || any[JDRGroup.PATTERN])
            && !any[JDRGroup.GROUP]);

         mergePathsItem.setEnabled(
               (numShapes > 1)
            && !any[JDRGroup.GROUP] && !any[JDRGroup.BITMAP]
            && !any[JDRGroup.TEXT]);

         convertToTextPathButtonItem.setEnabled(
           numShapes == 1 && number[JDRGroup.TEXT] == 1 && n == 2);

         separateItem.setEnabled(numCompoundShapes > 0);

         pathUnionItem.setEnabled((numShapes > 1)
           && !any[JDRGroup.GROUP] && !any[JDRGroup.BITMAP]
           && !any[JDRGroup.TEXT]);
         xorPathsItem.setEnabled(pathUnionItem.isEnabled());
         intersectPathsItem.setEnabled(pathUnionItem.isEnabled());
         subtractPathsItem.setEnabled(pathUnionItem.isEnabled());

         splitTextItem.setEnabled(numTextual > 0);

         groupButtonItem.setEnabled(n > 1);
         ungroupButtonItem.setEnabled(any[JDRGroup.GROUP]);
         justifyM.setEnabled(any[JDRGroup.GROUP]);
         setFrameItem.setEnabled(n > 0);

         parshapeItem.setEnabled(
              (numNonTextShapes == 1)
           && !anyTextual && !any[JDRGroup.GROUP] && !any[JDRGroup.BITMAP]);

         shapeparItem.setEnabled(parshapeItem.isEnabled());

         bitmapPropItem.setEnabled(any[JDRGroup.BITMAP]);

         if (vectorizeItem != null)
         {
            vectorizeItem.setEnabled(number[JDRGroup.BITMAP]==1);
         }

         nextItem.setEnabled(totalObjects > 0);
         addNextItem.setEnabled(totalObjects > 0);
         skipItem.setEnabled(totalObjects > 0);
         findSelectedItem.setEnabled(n > 0);
         findByDescriptionItem.setEnabled(totalObjects > 0);
         addByDescriptionItem.setEnabled(totalObjects > 0);
         enableEditTools(false);
      }
      else
      {
         disableAllTools();
      }
   }

   public void disableAllTools()
   {
      selectAllButtonItem.setEnabled(false);     
      deselectAllItem.setEnabled(false);     
      moveByItem.setEnabled(false);     
      cutButtonItem.setEnabled(false);
      copyButtonItem.setEnabled(false);
      pasteButtonItem.setEnabled(false);
      objectDescriptionItem.setEnabled(false);
      pathM.setEnabled(false);
      textM.setEnabled(false);
      reduceToGreyItem.setEnabled(false);
      fadeItem.setEnabled(false);
      moveToFrontButtonItem.setEnabled(false);
      moveToBackButtonItem.setEnabled(false);
      rotateButtonItem.setEnabled(false);
      scaleButtonItem.setEnabled(false);
      shearButtonItem.setEnabled(false);
      reverseItem.setEnabled(false);
      convertToPathItem.setEnabled(false);
      convertToFullPathItem.setEnabled(false);
      setPatternButtonItem.setEnabled(false);
      patternM.setEnabled(false);
      convertToTextPathButtonItem.setEnabled(false);
      separateItem.setEnabled(false);
      mergePathsItem.setEnabled(false);
      pathUnionItem.setEnabled(false);
      xorPathsItem.setEnabled(false);
      intersectPathsItem.setEnabled(false);
      subtractPathsItem.setEnabled(false);
      splitTextItem.setEnabled(false);
      groupButtonItem.setEnabled(false);
      ungroupButtonItem.setEnabled(false);
      justifyM.setEnabled(false);
      enableEditTools(false);
      setFrameItem.setEnabled(false);
      parshapeItem.setEnabled(false);
      shapeparItem.setEnabled(false);
      insertBitmapItem.setEnabled(false);
      bitmapPropItem.setEnabled(false);

      if (vectorizeItem != null)
      {
         vectorizeItem.setEnabled(false);
      }

      nextItem.setEnabled(false);
      addNextItem.setEnabled(false);
      skipItem.setEnabled(false);
      findSelectedItem.setEnabled(false);
      findByDescriptionItem.setEnabled(false);
      addByDescriptionItem.setEnabled(false);

      if (!editPathButtonItem.isSelected())
      {
         editPathButtonItem.setEnabled(false);
      }
   }

   public JDRFrame getSelectedFrame()
   {
      if (theDesktop==null) return null;
      return (JDRFrame)theDesktop.getSelectedFrame();
   }

   // returns currently selected frame or last frame to have focus
   public JDRFrame getCurrentFrame()
   {
      if (theDesktop==null) return null;
      JDRFrame frame = getSelectedFrame();

      if (frame == null)
      {
         JInternalFrame[] f = theDesktop.getAllFrames();
         if (f.length > 0)
         {
            frame = (JDRFrame)f[0];
         }
      }

      return frame;
   }

   public void tileFrames()
   {
      JInternalFrame[] frames = theDesktop.getAllFrames();

      Dimension dim = theDesktop.getSize();

      int n = frames.length;

      if (n == 0) return;

      if (n == 1)
      {
         try
         {
            frames[0].setMaximum(true);
            frames[0].setIcon(false);
         }
         catch (PropertyVetoException ignore)
         {
         }

         return;
      }

      int cols = (int)Math.ceil(Math.sqrt(n));

      int rows = (int)Math.ceil((double)n/(double)cols);

      dim.width  /= cols;
      dim.height /= rows;

      for (int i = 0, j=0, x=0, y=0; i < n; i++)
      {
         try
         {
            frames[i].setMaximum(false);
            frames[i].setIcon(false);
         }
         catch (PropertyVetoException ignore)
         {
         }
         frames[i].reshape(x, y, dim.width, dim.height);

         y += dim.height;
         j++;
         if (j >= rows)
         {
            y = 0;
            x += dim.width;
            j=0;
         }
      }
   }

   public void tileFramesHorizontally()
   {
      JInternalFrame[] frames = theDesktop.getAllFrames();

      Dimension dim = theDesktop.getSize();

      int n = frames.length;

      if (n == 0) return;

      if (n == 1)
      {
         try
         {
            frames[0].setMaximum(true);
            frames[0].setIcon(false);
         }
         catch (PropertyVetoException ignore)
         {
         }

         return;
      }

      dim.height = (int)Math.ceil((double)dim.height/(double)n);

      for (int i = 0, x=0, y=0; i < n; i++)
      {
         try
         {
            frames[i].setMaximum(false);
            frames[i].setIcon(false);
         }
         catch (PropertyVetoException ignore)
         {
         }
         frames[i].reshape(x, y, dim.width, dim.height);

         y += dim.height;
      }
   }

   public void tileFramesVertically()
   {
      JInternalFrame[] frames = theDesktop.getAllFrames();

      Dimension dim = theDesktop.getSize();

      int n = frames.length;

      if (n == 0) return;

      if (n == 1)
      {
         try
         {
            frames[0].setMaximum(true);
            frames[0].setIcon(false);
         }
         catch (PropertyVetoException ignore)
         {
         }

         return;
      }

      dim.width = (int)Math.ceil((double)dim.width/(double)n);

      for (int i = 0, x=0, y=0; i < n; i++)
      {
         try
         {
            frames[i].setMaximum(false);
            frames[i].setIcon(false);
         }
         catch (PropertyVetoException ignore)
         {
         }
         frames[i].reshape(x, y, dim.width, dim.height);

         x += dim.width;
      }
   }

   public void minimizeAll()
   {
      JInternalFrame[] frames = theDesktop.getAllFrames();

      for (int i = 0, n=frames.length; i < n; i++)
      {
         try
         {
            frames[i].setIcon(true);
         }
         catch (PropertyVetoException ignore)
         {
         }
      }
   }

   public void maximizeAll()
   {
      JInternalFrame[] frames = theDesktop.getAllFrames();

      for (int i = 0, n=frames.length; i < n; i++)
      {
         try
         {
            frames[i].setMaximum(true);
            if (frames[i].isIcon()) frames[i].setIcon(false);
         }
         catch (PropertyVetoException ignore)
         {
         }
      }
   }

   public CanvasSettings getCurrentSettings()
   {
      JDRFrame currentFrame = (JDRFrame)theDesktop.getSelectedFrame();

      CanvasSettings settings;

      if (currentFrame == null)
      {
         settings = new CanvasSettings();
      }
      else
      {
         settings = (CanvasSettings)currentFrame.getSettings().clone();
      }

      settings.setDisplayGrid(showGridButtonItem.isSelected());
      settings.setGridLock(lockGridButtonItem.isSelected());
      settings.showRulers = showRulersItem.isSelected();
      try
      {
         settings.setTool(getCurrentTool());
      }
      catch (InvalidToolException ignore)
      {
      }
      settings.paper = getCurrentPaper();

      return settings;
   }

   public JDRFrame addFrame()
   {
      return addFrame(getCurrentSettings());
   }

   public JDRFrame addFrame(String filename)
   {
      return addFrame(getCurrentSettings(), filename);
   }

   public JDRFrame addFrame(CanvasSettings settings)
   {
      return addFrame(settings, null);
   }

   public JDRFrame addFrame(CanvasSettings settings, String filename)
   {
      JDRFrame frame = new JDRFrame(filename, settings, 
         windowM, windowButtonGroup, this);

      frame.addInternalFrameListener(this);

      Dimension dim = theDesktop.getSize();
      dim.width /= 2;
      dim.height /= 2;
      frame.setSize(dim);

      // attach internal frame to desktop and show it
      theDesktop.add(frame);

      try
      {
         frame.setMaximum(true);
      }
      catch (PropertyVetoException ignore)
      {
      }

      frame.setVisible(true);
      updateWindowMenu();
      frame.requestFocusInWindow();
      frame.updateTextAreaBounds();

      return frame;
   }

   public void removeFrame(JInternalFrame frame)
   {
      theDesktop.remove(frame);
      frame.dispose();

      JInternalFrame[] frames = theDesktop.getAllFrames();
      if (frames.length > 0)
      {
         try
         {
            frames[0].setSelected(true);
         }
         catch (PropertyVetoException ignore)
         {
         }
      }
      else
      {
         updateStatus(0);
      }
   }

   public void openImage()
   {
      CanvasSettings settings = getCurrentSettings();

      JDRFrame currentFrame = getSelectedFrame();

      int result = openjdrFC.showOpenDialog(this);

      if (result == JFileChooser.APPROVE_OPTION)
      {
         if (currentFrame == null)
         {
            addFrame(settings,
               openjdrFC.getSelectedFile().getAbsolutePath());
         }
         else if (currentFrame.newImage)
         {
            if (currentFrame.isIcon())
            {
               currentFrame.selectThisFrame();
            }

            currentFrame.load(
               openjdrFC.getSelectedFile().getAbsolutePath());
         }
         else
         {
            addFrame(settings,
                     openjdrFC.getSelectedFile().getAbsolutePath());
         }
      }
   }

   public void importImage()
   {
      int result = importFC.showOpenDialog(this);

      if (result == JFileChooser.APPROVE_OPTION)
      {
         JDRFrame currentFrame = getSelectedFrame();

         double normalsize = 10;

         if (currentFrame != null)
         {
            normalsize = currentFrame.getLaTeXFonts().getNormalSize();
         }

         File file = new File(
            importFC.getSelectedFile().getAbsolutePath());

         Thread thread = new Thread(
            new LoadEps(currentFrame, file, normalsize));

         thread.start();

         thread = null;
      }
   }

   public void saveImage()
   {
      JDRFrame frame = getCurrentFrame();
      if (frame == null) return;

      if (frame.isIcon())
      {
         frame.selectThisFrame();
      }

      if (frame.hasFileName())
      {
         String filenameLC = frame.getFilename().toLowerCase();

         if (filenameLC.endsWith(".ajr"))
         {
            frame.saveAJR(true);
         }
         else
         {
            frame.save(true);
         }
      }
      else
      {
         promptAndSave(frame, true);
      }
   }

   public boolean promptAndSave(JDRFrame frame, boolean useThread)
   {
      int result = savejdrFC.showSaveDialog(frame);

      if (result == JFileChooser.APPROVE_OPTION)
      { 
         saveAs(frame, useThread);

         return true;
      }

      return false;
   }

   public void saveAs(JDRFrame frame, boolean useThread)
   {
      String filename = savejdrFC.getSelectedFile().getAbsolutePath();

      JDRFileFilterInterface filter 
         = (JDRFileFilterInterface)savejdrFC.getFileFilter();

      String filenameLC = filename.toLowerCase();

      int index = filenameLC.lastIndexOf('.');

      String extension;

      if (index == -1)
      {
         extension = "";
      }
      else
      {
         extension = filenameLC.substring(index);
      }

      if (jdrFCAddExt.isSelected()
       && !(extension.equals(".jdr") || extension.equals(".ajr")))
      {
         extension = "." + filter.getDefaultExtension();

         filename = filename + extension;
      }

      // does the file already exist?
      File file = new File(filename);

      if (file.exists())
      {
         int selection = JOptionPane.showConfirmDialog(frame,
            new String[]
            {filename,
            JDRResources.getString("warning.file_exists")},
            JDRResources.getString("warning.title"),
            JOptionPane.YES_NO_OPTION,
            JOptionPane.WARNING_MESSAGE);

            if (selection != JOptionPane.YES_OPTION) return;
      }

      float version = filter.getVersion();

      if (extension.equals(".jdr"))
      {
         frame.save(filename, version, useThread);
      }
      else if (extension.equals(".ajr"))
      {
         frame.saveAJR(filename, version, useThread);
      }
      else if (filter instanceof JdrFileFilter
             || filter instanceof JdrAjrFileFilter)
      {
         frame.save(filename, version, useThread);
      }
      else if (filter instanceof AjrFileFilter)
      {
         frame.save(filename, version, useThread);
      }
      else
      {
         JDRResources.internalError("Unknown file filter");
      }
   }

   public void saveAs(boolean useThread)
   {
      JDRFrame frame = getCurrentFrame();
      if (frame == null) return;

      if (frame.isIcon())
      {
         frame.selectThisFrame();
      }

      int result = savejdrFC.showSaveDialog(frame);

      if (result == JFileChooser.APPROVE_OPTION)
      {
         saveAs(frame, useThread);
      }
   }

   public void exportImage()
   {
      JDRFrame frame = getCurrentFrame();
      if (frame == null) return;

      if (frame.isIcon())
      {
         frame.selectThisFrame();
      }

      if (!frame.getFilename().equals(""))
      {
         File file = new File(frame.getFilename());

         String name = file.getName();

         int i = name.lastIndexOf(".jdr");

         if (i == -1) i = name.lastIndexOf(".ajr");

         if (i != -1)
         {
            name = name.substring(0, i);
         }

         File currentDir = exportFC.getCurrentDirectory();

         exportFC.setSelectedFile(new File(currentDir, name));
      }

      int result = exportFC.showSaveDialog(frame);

      if (result == JFileChooser.APPROVE_OPTION)
      {
         String filename = exportFC.getSelectedFile().getAbsolutePath();

         FileFilter filter = exportFC.getFileFilter();

         if (exportFCAddExt.isSelected())
         {
            if (filter == pgfFileFilter &&
                    !(filename.toLowerCase().endsWith(".tex")
                    ||filename.toLowerCase().endsWith(".ltx")))
            {
               filename = filename + ".tex";
            }
            else if (filter == pgfDocFileFilter &&
                    !(filename.toLowerCase().endsWith(".tex")
                    ||filename.toLowerCase().endsWith(".ltx")))
            {
               filename = filename + ".tex";
            }
            else if (filter == flfFileFilter &&
                    !filename.toLowerCase().endsWith(".sty"))
            {
               filename = filename + ".sty";
            }
            else if (filter == pngFileFilter &&
                    !filename.toLowerCase().endsWith(".png"))
            {
               filename = filename + ".png";
            }
            else if (filter == epsFileFilter &&
                    !filename.toLowerCase().endsWith(".eps"))
            {
               filename = filename + ".eps";
            }
            else if (filter == svgFileFilter &&
                    !filename.toLowerCase().endsWith(".svg"))
            {
               filename = filename + ".svg";
            }
         }

         // does the file already exist?
         File file = new File(filename);
         if (file.exists())
         {
            int selection = JOptionPane.showConfirmDialog(frame,
               new String[]
               {filename,
               JDRResources.getString("warning.file_exists")},
               JDRResources.getString("warning.title"),
               JOptionPane.YES_NO_OPTION,
               JOptionPane.WARNING_MESSAGE);

               if (selection != JOptionPane.YES_OPTION) return;
         }

         if (filter == pgfFileFilter)
         {         
            frame.savePGF(filename);
         }
         else if (filter == pgfDocFileFilter)
         {         
            frame.savePGFDoc(filename);
         }
         else if (filter == pngFileFilter)
         {         
            frame.savePNG(filename);
         }
         else if (filter == epsFileFilter)
         {         
            frame.saveEPS(filename);
         }
         else if (filter == svgFileFilter)
         {
            frame.saveSVG(filename);
         }
         else
         {
            frame.saveFlowFrame(filename);
         }
      }
   }

   public void close()
   {
      JDRFrame frame = (JDRFrame)theDesktop.getSelectedFrame();

      frame.discard();
      frame = null;
      System.gc();
      updateWindowMenu();
   }

   public DiscardDialogBox getDiscardDialogBox()
   {
      return discardDB;
   }

   public void updateWindowMenu()
   {
      JInternalFrame[] frames = theDesktop.getAllFrames();

      int n = frames.length;

      for (int i = 0; i < n; i++)
      {
         JDRFrame f = (JDRFrame)frames[i];

         String id = new String((i < 9 ? " ":"")+(i+1));
         f.menuItem.setText(id+" "+f.getTitle());
         f.menuItem.setMnemonic(id.charAt(1));
      }

      enableFrameRelatedItems(n > 0);
   }

   public void quit()
   {
      CanvasSettings settings = getCurrentSettings();

      maximizeAll();
      JInternalFrame[] frames = theDesktop.getAllFrames();

      for (int i = 0, n=frames.length; i < n; i++)
      {
         JDRFrame f = (JDRFrame)frames[i];

         if (!f.discard())
         {
            // user cancelled quit or I/O operation in progress
            return;
         }

      }

      saveRecentFiles();

      if (initSettings()!=JpgfDrawSettings.INIT_USER)
      {
         saveUserSettings();
      }

      System.exit(0);
   }

   public void saveUserSettings()
   {
      try
      {
         if (usersettings == null)
         {
            JDRResources.error(this,
               JDRResources.getString("error.config_save_failed"));
         }
         else
         {
            File file = new File(usersettings, "jpgfdraw.conf");
            saveConfig(file, getCurrentSettings());
         }
      }
      catch (Exception e)
      {
         JDRResources.error(this, new String[]
            {JDRResources.getString("error.io.save"),
            e.getMessage()});
      }
   }

   public void addRecentFile(String filename)
   {
      for (int i = recentFiles.size()-1; i >= 0; i--)
      {
         String str = (String)recentFiles.get(i);

         if (filename.equals(str))
         {
            recentFiles.remove(i);
         }
      }

      recentFiles.add(0, filename);
   }

   public void setRecentFiles()
   {
      if (recentFiles == null) return;

      if (recentM.getMenuComponentCount() > 0)
      {
         recentM.removeAll();
      }

      for (int i = 0, n = recentFiles.size(); i < n && i < 10; i++)
      {
         String file = (String)recentFiles.get(i);
         String num = new String(i==9?"10":" "+(i+1));
         JMenuItem item = new JMenuItem(num+" "+file);
         item.setMnemonic(num.charAt(1));
         item.addActionListener(new LoadAction(this, file, i, recentFiles));
         recentM.add(item);
      }
   }

   public void saveRecentFiles()
   {
      if (recentFiles == null || usersettings == null) return;

      int n = recentFiles.size();

      if (n == 0) return;

      String filename = usersettings+File.separator+"recentfiles";

      try
      {
         PrintWriter out = new PrintWriter(new FileWriter(filename));

         for (int i = 0; i < n && i < 10; i++)
         {
            String file = (String)recentFiles.get(i);
            out.println(file);
         }

         out.close();
      }
      catch (Exception e)
      {
         JDRResources.error(null,e);
      }
   }

   public void loadRecentFiles()
   {
      if (recentFiles == null && usersettings != null)
      {
         String filename = usersettings+File.separator+"recentfiles";

         recentFiles = new Vector<String>(10);

         try
         {
            BufferedReader in = new BufferedReader(
               new FileReader(filename));

            String s;

            while ((s=in.readLine()) != null)
            {
               if (!s.equals(""))
               {
                  // check not already added

                  boolean found=false;

                  for (int i = 0, n=recentFiles.size(); i < n; i++)
                  {
                     String str = (String)recentFiles.get(i);
                     if (s.equals(str))
                     {
                        found = true;
                        break;
                     }
                  }

                  if (!found) recentFiles.add(s);
               }
            }

            in.close();
         }
         catch (IOException ignore)
         {
         }
         catch (Exception e)
         {
            JDRResources.error(null,e);
         }
      }
   }

   public static JMenuItem addHelpItem(JMenu helpM)
   {
      return JDRResources.addHelpItem(helpM, JpgfDraw.appName);
   }

   public static void initializeHelp(JFrame parent)
   {
      JDRResources.initialiseHelp(parent, appName.toLowerCase());
   }

   public static void enableHelpOnButton(AbstractButton comp,
      String id)
   {
      JDRResources.enableHelpOnButton(comp, id);
   }

   public static void version()
   {
      disposeStartup();
      System.out.println(
         JDRResources.getStringWithValue("about.version",
         new String[]{JpgfDraw.appName, JpgfDraw.version},
         JpgfDraw.appName+" version "+JpgfDraw.version));
      System.exit(0);
   }

   public static void syntax()
   {
      disposeStartup();
      System.out.println(
         JDRResources.getStringWithValue("about.version",
         new String[]{JpgfDraw.appName, JpgfDraw.version},
         JpgfDraw.appName+" version "+JpgfDraw.version));

      System.out.println();
      System.out.println("Syntax: jpgfdraw [options] [filename]");
      System.out.println();
      System.out.println("Options:");
      System.out.println();
      System.out.println("-(no)show_grid");
      System.out.println("-(no)grid_lock");
      System.out.println("-(no)toolbar");
      System.out.println("-(no)statusbar");
      System.out.println("-(no)rulers");
      System.out.println("-paper <type>");
      System.out.println("-(no)debug");
      System.out.println("-(no)experimental");
      System.out.println("-version");

      System.exit(0);
   }

   public void writeLog()
   {
      if (logFile == null)
      {
         JDRResources.error(this, "no log file");
         return;
      }

      try
      {
         PrintWriter out = new PrintWriter(new FileWriter(logFile));

         out.println("jpgfdraw v"+version);

         DateFormat dateFormat = DateFormat.getDateTimeInstance();

         out.println("Log file created "
            +dateFormat.format(new Date()));

         out.println("main window:");
         out.println(this);

         out.println();
         out.println("desktop:");
         out.println(theDesktop);

         JInternalFrame[] allFrames = theDesktop.getAllFrames();

         out.println();
         out.println(allFrames.length+" internal frame(s)");
         out.println();

         for (int i = 0; i < allFrames.length; i++)
         {
            out.println("Frame "+i);

            JDRFrame f = (JDRFrame)allFrames[i];

            f.printInfo(out);

            out.println();
         }

         out.println(linePaintChooserBox.info());
         out.println(fillPaintChooserBox.info());
         out.println(lineStyleChooserBox.info());
         out.println(lineWidthChooserBox.info());
         out.println(dashPatternChooserBox.info());
         out.println(joinStyleChooserBox.info());
         out.println(startArrowChooserBox.info());
         out.println(midArrowChooserBox.info());
         out.println(endArrowChooserBox.info());
         out.println(allMarkersChooserBox.info());
         out.println(editTextBox.info());
         out.println(textPaintChooserBox.info());
         out.println(textStyleChooserBox.info());
         out.println(fontFamilyChooserBox.info());
         out.println(fontSizeChooserBox.info());
         out.println(fontShapeChooserBox.info());
         out.println(fontSeriesChooserBox.info());
         out.println(fontAnchorChooserBox.info());
         out.println(fontHAnchorChooserBox.info());
         out.println(fontVAnchorChooserBox.info());
         out.println(styleChooserBox.info());
         out.println(configDialog.info());
         out.println(texSettingsDialog.info());
         out.println(rotateDB.info());
         out.println(scaleDB.info());
         out.println(shearDB.info());
         out.println(flfSelector.info());
         out.println(discardDB.info());
         out.println(gotoDialog.info());

         out.close();

         JOptionPane.showMessageDialog(null,
         "Information written to '"+logFile.getAbsolutePath()+"'",
         "Write Log",
         JOptionPane.INFORMATION_MESSAGE);
      }
      catch (IOException e)
      {
         JDRResources.error(this, e);
      }
   }

   public void objectInfo()
   {
      JDRFrame frame = (JDRFrame)theDesktop.getSelectedFrame();

      String message;

      if (frame == null)
      {
         message = "Can't access any data";
      }
      else
      {
         message = frame.getSelectedInfo();
      }

      JTextArea textArea = new JTextArea(message);
      textArea.setEditable(false);

      JScrollPane sp = new JScrollPane(textArea);

      sp.setPreferredSize(new Dimension(500,400));

      JOptionPane.showMessageDialog(null, sp, "Object Info",
         JOptionPane.PLAIN_MESSAGE);
   }

   public void dumpAll()
   {
      if (usersettings == null)
      {
         JDRResources.error(null, 
            "Can't dump all images - no configuration directory");
         return;
      }

      JInternalFrame[] allFrames = theDesktop.getAllFrames();

      if (allFrames.length == 0)
      {
         return;
      }

      Calendar cal = new GregorianCalendar();

      int month = cal.get(Calendar.MONTH)+1;
      int day = cal.get(Calendar.DAY_OF_MONTH);
      int hour = cal.get(Calendar.HOUR_OF_DAY);
      int minute = cal.get(Calendar.MINUTE);
      int second = cal.get(Calendar.SECOND);

      String dir = 
         cal.get(Calendar.YEAR)
         + (month < 10 ? ("0"+month) : ""+month)
         + (day < 10 ? ("0"+day) : ""+day)
         + (hour < 10 ? ("0"+hour) : ""+hour)
         + (minute < 10 ? ("0"+minute) : ""+minute)
         + (second < 10 ? ("0"+second) : ""+second);

      File parent = new File(usersettings, dir);

      if (!parent.mkdir())
      {
         JDRResources.error(null, "Unable to create directory '"
          +parent.getAbsolutePath()+"'");

         return;
      }

      for (int i = 0; i < allFrames.length; i++)
      {
         JDRFrame frame = (JDRFrame)allFrames[i];

         File file = new File(parent, "image"+(i+1)+".jdr");

         frame.save(file, JDR.currentVersion, true);
      }
   }

   public static void loadConfig(File file, CanvasSettings settings)
      throws IOException,InvalidFormatException
   {
      // reads in settings from config file, only sets the settings
      // that are specified in file.

      BufferedReader in = new BufferedReader(
         new FileReader(file));

      String s;
      int line = 0;
      boolean doneRobot=false;
      boolean diffVersion=true;

      JDRMarker startComposite = null;
      JDRMarker midComposite   = null;
      JDRMarker endComposite   = null;

      while ((s=in.readLine()) != null)
      {
         line++;
         if (s.charAt(0) == '#') continue;

         StringTokenizer t = new StringTokenizer(s, "=");

         if (!t.hasMoreTokens())
         {
            throw new InvalidFormatException(
               JDRResources.getStringWithValue(
                  "error.invalid_format", line));
         }

         String key = t.nextToken();

         String value = "";

         if (t.hasMoreTokens())
         {
            value = t.nextToken();
         }

         if (key.equals("initsettings"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.three_value", line));
            }

            if (i < 0 || i > 2)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.three_value", line));
            }

            appSettings.initSettings=i;

            if (appSettings.initSettings
                   == JpgfDrawSettings.INIT_DEFAULT)
            {
               break;
            }

         }
         else if (key.equals("version"))
         {
            diffVersion = !value.equals(version);

            if (value.compareTo("0.5.6b") < 0)
            {
               try
               {
                  JDRResources.initialiseDictionary();
               }
               catch (IOException ioe)
               {
                  JDRResources.internalError(null, ioe);
               }
            }

         }
         else if (key.equals("dict_lang"))
         {
            JDRResources.dictLocaleId = value;

            try
            {
               JDRResources.initialiseDictionary();
            }
            catch (IOException ioe)
            {
               JDRResources.internalError(null, ioe);
            }
         }
         else if (key.equals("help_lang"))
         {
            JDRResources.helpLocaleId = value;
         }
         else if (key.equals("preview_bitmaps"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.previewBitmaps = (i == 1);
         }
         else if (key.equals("show_grid"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            settings.setDisplayGrid(i == 1);
         }
         else if (key.equals("lock_grid"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            settings.setGridLock(i == 1);
         }
         else if (key.equals("show_tools"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.showToolBar = (i == 1);
         }
         else if (key.equals("show_rulers"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            settings.showRulers = (i == 1);
         }
         else if (key.equals("show_status"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.showStatus = (i == 1);
         }
         else if (key.equals("dragscale"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.enableDragScale = (i == 1);
         }
         else if (key.equals("antialias"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.antialias = (i == 1);
         }
         else if (key.equals("render_quality"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.renderquality = (i == 1);
         }
         else if (key.equals("startdir_type"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.three_value", line));
            }

            if (i < 0 || i > 2)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.three_value", line));
            }

            appSettings.startDirType = i;
         }
         else if (key.equals("startdir"))
         {
            appSettings.startDir=value;
         }
         else if (key.equals("normalsize"))
         {
            int i = 10;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            settings.normalsize = i;
         }
         else if (key.equals("latexfontupdate"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.updateLaTeXFonts = (i == 1);
         }
         else if (key.equals("autoupdateanchors"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.autoUpdateAnchors = (i == 1);
         }
         else if (key.equals("autoescapespchars"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.autoEscapeSpChars = (i == 1);
         }
         else if (key.equals("save_jdrsettings"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.saveJDRsettings = (i == 1);
         }
         else if (key.equals("use_jdrsettings"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.useJDRsettings = (i == 1);
         }
         else if (key.equals("warn_load_old"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.warnOnOldJdr = (i == 1);
         }
         else if (key.equals("tool"))
         {
            settings.setTool(value);
         }
         else if (key.equals("paper"))
         {
            StringTokenizer pt = new StringTokenizer(value, " ");

            String pvalue = pt.nextToken();

            if (pvalue.equals("user"))
            {
               try
               {
                  pvalue = pt.nextToken();
                  double w = Double.parseDouble(pvalue);
                  pvalue = pt.nextToken();
                  double h = Double.parseDouble(pvalue);
                  settings.paper=new JDRPaper(w, h);
               }
               catch (Exception e)
               {
                  throw new InvalidFormatException(
                     JDRResources.getStringWithValue(
                        "error.invalid_paper_dimension", value));
               }
            }
            else
            {
               JDRPaper paper=JDRPaper.getPredefinedPaper(value);
               if (paper != null)
               {
                  settings.paper=paper;
               }
               else
               {
                  throw new InvalidFormatException(
                    JDRResources.getStringWithValue(
                       "error.unknown_papersize", value));
               }
            }
         }
         else if (key.equals("fontname"))
         {
            appSettings.fontFamily=value;
         }
         else if (key.equals("fontsize"))
         {
            int i = 10;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            appSettings.fontSize = i;
         }
         else if (key.equals("fontshape"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.fontShape = i;
         }
         else if (key.equals("fontseries"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.fontSeries = i;
         }
         else if (key.equals("latexfontname"))
         {
            appSettings.latexFontFamily=value;
         }
         else if (key.equals("latexfontseries"))
         {
            appSettings.latexFontSeries=value;
         }
         else if (key.equals("latexfontshape"))
         {
            appSettings.latexFontShape=value;
         }
         else if (key.equals("latexfontsize"))
         {
            appSettings.latexFontSize=value;
         }
         else if (key.equals("capstyle"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.three_value", line));
            }

            if (i < 0 || i > 2)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.three_value", line));
            }

            appSettings.stroke.setCapStyle(i);
         }
         else if (key.equals("joinstyle"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.three_value", line));
            }

            if (i < 0 || i > 2)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.three_value", line));
            }

            appSettings.stroke.setJoinStyle(i);
         }
         else if (key.equals("windingrule"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.stroke.setWindingRule(i);
         }
         else if (key.equals("penwidth"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            appSettings.stroke.setPenWidth(i);
         }
         else if (key.equals("mitrelimit"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            appSettings.stroke.setMitreLimit(i);
         }
         else if (key.equals("startarrow"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            if (i < 0 || i >= JDRMarker.maxMarkers())
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number_badrange", line));
            }

            try
            {
               appSettings.stroke.setStartArrow(i);
            }
            catch (InvalidMarkerTypeException e)
            {
               // this shouldn't happen, as it should already
               // have been caught
               JDRResources.internalError(null, 
                  JDRResources.getString("error.invalid_arrow_type")
                  +" "+e.getInvalidID(), e);
            }
         }
         else if (key.equals("secondarystartarrow"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            if (i < 0 || i >= JDRMarker.maxMarkers())
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number_badrange", line));
            }

            try
            {
               startComposite = JDRMarker.getPredefinedMarker(i);
            }
            catch (InvalidMarkerTypeException e)
            {
               // this shouldn't happen, as it should already
               // have been caught
               JDRResources.internalError(null, 
                  JDRResources.getString("error.invalid_arrow_type")
                  +" "+e.getInvalidID(), e);
            }
         }
         else if (key.equals("startarrowuseroffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setStartUserOffsetEnabled(i==1);
         }
         else if (key.equals("startoverlaid"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setStartOverlay(i==1);
         }
         else if (key.equals("startarrowuseroffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setStartOffset(i);
         }
         else if (key.equals("midarrowuseroffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setMidUserOffsetEnabled(i==1);
         }
         else if (key.equals("midoverlaid"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setMidOverlay(i==1);
         }
         else if (key.equals("midarrowuseroffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setMidOffset(i);
         }
         else if (key.equals("endarrowuseroffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setEndUserOffsetEnabled(i==1);
         }
         else if (key.equals("endoverlaid"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setEndOverlay(i==1);
         }
         else if (key.equals("endarrowuseroffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setEndOffset(i);
         }
         else if (key.equals("startarrowrepeatoffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setStartRepeatOffsetEnabled(i==1);
         }
         else if (key.equals("startarrowrepeatoffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setStartRepeatOffset(i);
         }
         else if (key.equals("midarrowrepeatoffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setMidRepeatOffsetEnabled(i==1);
         }
         else if (key.equals("midarrowrepeatoffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setMidRepeatOffset(i);
         }
         else if (key.equals("endarrowrepeatoffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            appSettings.stroke.setEndRepeatOffsetEnabled(i==1);
         }
         else if (key.equals("endarrowrepeatoffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setEndRepeatOffset(i);
         }
         else if (key.equals("secondarystartarrowuseroffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            startComposite.enableUserOffset(i==1);
         }
         else if (key.equals("secondarystartarrowuseroffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            if (i < 0)
            startComposite.setOffset(i);
         }
         else if (key.equals("secondarymidarrowuseroffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            midComposite.enableUserOffset(i==1);
         }
         else if (key.equals("secondarymidarrowuseroffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            midComposite.setOffset(i);
         }
         else if (key.equals("secondaryendarrowuseroffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            endComposite.enableUserOffset(i==1);
         }
         else if (key.equals("secondaryendarrowuseroffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            endComposite.setOffset(i);
         }
         else if (key.equals("secondarystartarrowrepeatoffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            startComposite.enableUserRepeatOffset(i==1);
         }
         else if (key.equals("secondarystartarrowrepeatoffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            startComposite.setRepeatOffset(i);
         }
         else if (key.equals("secondarymidarrowrepeatoffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            midComposite.enableUserRepeatOffset(i==1);
         }
         else if (key.equals("secondarymidarrowrepeatoffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            midComposite.setRepeatOffset(i);
         }
         else if (key.equals("secondaryendarrowrepeatoffsetenabled"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            
            endComposite.enableUserRepeatOffset(i==1);
         }
         else if (key.equals("secondaryendarrowrepeatoffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            endComposite.setRepeatOffset(i);
         }
         else if (key.equals("midarrow"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            if (i < 0 || i >= JDRMarker.maxMarkers())
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number_badrange", line));
            }

            try
            {
               appSettings.stroke.setMidArrow(i);
            }
            catch (InvalidMarkerTypeException e)
            {
               // this shouldn't happen, as it should already
               // have been caught
               JDRResources.internalError(null,
                  JDRResources.getString("error.invalid_arrow_type")
                  +" "+e.getInvalidID(), e);
            }
         }
         else if (key.equals("secondarymidarrow"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            if (i < 0 || i >= JDRMarker.maxMarkers())
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number_badrange", line));
            }

            try
            {
               midComposite = JDRMarker.getPredefinedMarker(i);
            }
            catch (InvalidMarkerTypeException e)
            {
               // this shouldn't happen, as it should already
               // have been caught
               JDRResources.internalError(null, 
                  JDRResources.getString("error.invalid_arrow_type")
                  +" "+e.getInvalidID(), e);
            }
         }
         else if (key.equals("endarrow"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            if (i < 0 || i >= JDRMarker.maxMarkers())
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number_badrange", line));
            }

            try
            {
               appSettings.stroke.setEndArrow(i);
            }
            catch (InvalidMarkerTypeException e)
            {
               // this shouldn't happen, as it should already
               // have been caught
               JDRResources.internalError(null,
                 JDRResources.getString("error.invalid_arrow_type")
                 +" "+e.getInvalidID(), e);
            }
         }
         else if (key.equals("secondaryendarrow"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            if (i < 0 || i >= JDRMarker.maxMarkers())
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number_badrange", line));
            }

            try
            {
               endComposite = JDRMarker.getPredefinedMarker(i);
            }
            catch (InvalidMarkerTypeException e)
            {
               // this shouldn't happen, as it should already
               // have been caught
               JDRResources.internalError(null, 
                  JDRResources.getString("error.invalid_arrow_type")
                  +" "+e.getInvalidID(), e);
            }
         }
         else if (key.equals("startarrowsize"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            appSettings.stroke.setStartArrowSize(i);
         }
         else if (key.equals("secondarystartarrowsize"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.no_negative", line));
            }

            startComposite.setSize(i);
         }
         else if (key.equals("midarrowsize"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.no_negative", line));
            }

            appSettings.stroke.setMidArrowSize(i);
         }
         else if (key.equals("secondarymidarrowsize"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.no_negative", line));
            }

            midComposite.setSize(i);
         }
         else if (key.equals("endarrowsize"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.no_negative", line));
            }

            appSettings.stroke.setEndArrowSize(i);
         }
         else if (key.equals("secondaryendarrowsize"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.no_negative", line));
            }

            endComposite.setSize(i);
         }
         else if (key.equals("startarrowdouble"))
         {
            // provide compatibility with older versions
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                  "error.conf.boolean_value", line));
            }

            appSettings.stroke.setStartArrowRepeat(i==1?2:1);
         }
         else if (key.equals("endarrowdouble"))
         {
            // provide compatibility with older versions
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.stroke.setEndArrowRepeat(i==1?2:1);
         }
         else if (key.equals("startarrowrepeat"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i <= 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.positive_only", line));
            }

            appSettings.stroke.setStartArrowRepeat(i);
         }
         else if (key.equals("secondarystartarrowrepeat"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i <= 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.positive_only", line));
            }

            startComposite.setRepeated(i);
         }
         else if (key.equals("midarrowrepeat"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i <= 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.positive_only", line));
            }

            appSettings.stroke.setMidArrowRepeat(i);
         }
         else if (key.equals("secondarymidarrowrepeat"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i <= 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.positive_only", line));
            }

            midComposite.setRepeated(i);
         }
         else if (key.equals("endarrowrepeat"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i <= 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.positive_only", line));
            }

            appSettings.stroke.setEndArrowRepeat(i);
         }
         else if (key.equals("secondaryendarrowrepeat"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i <= 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.positive_only", line));
            }

            endComposite.setRepeated(i);
         }
         else if (key.equals("startarrowreverse"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.stroke.setStartArrowReverse(i==1);
         }
         else if (key.equals("secondarystartarrowreverse"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            startComposite.setReversed(i==1);
         }
         else if (key.equals("midarrowreverse"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.stroke.setMidArrowReverse(i==1);
         }
         else if (key.equals("secondarymidarrowreverse"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            midComposite.setReversed(i==1);
         }
         else if (key.equals("endarrowreverse"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.stroke.setEndArrowReverse(i==1);
         }
         else if (key.equals("secondaryendarrowreverse"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            endComposite.setReversed(i==1);
         }
         else if (key.equals("startarroworient"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.stroke.setStartArrowAutoOrient(i==1);
         }
         else if (key.equals("secondarystartarroworient"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            startComposite.setOrient(i==1);
         }
         else if (key.equals("midarroworient"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.stroke.setMidArrowAutoOrient(i==1);
         }
         else if (key.equals("secondarymidarroworient"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            midComposite.setOrient(i==1);
         }
         else if (key.equals("endarroworient"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            appSettings.stroke.setEndArrowAutoOrient(i==1);
         }
         else if (key.equals("secondaryendarroworient"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.boolean_value", line));
            }

            endComposite.setOrient(i==1);
         }
         else if (key.equals("startarrowpaint"))
         {
            appSettings.stroke.setStartArrowColour(
               loadPaintConfig(value,line));
         }
         else if (key.equals("secondarystartarrowpaint"))
         {
            startComposite.setFillPaint(
               loadPaintConfig(value,line));
         }
         else if (key.equals("midarrowpaint"))
         {
            appSettings.stroke.setMidArrowColour(
               loadPaintConfig(value,line));
         }
         else if (key.equals("secondarymidarrowpaint"))
         {
            midComposite.setFillPaint(loadPaintConfig(value,line));
         }
         else if (key.equals("endarrowpaint"))
         {
            appSettings.stroke.setEndArrowColour(
               loadPaintConfig(value,line));
         }
         else if (key.equals("secondaryendarrowpaint"))
         {
            endComposite.setFillPaint(loadPaintConfig(value,line));
         }
         else if (key.equals("startarrowangle"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setStartArrowAngle(i);
         }
         else if (key.equals("secondarystartarrowangle"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            startComposite.setAngle(i);
         }
         else if (key.equals("midarrowangle"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setMidArrowAngle(i);
         }
         else if (key.equals("secondarymidarrowangle"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            midComposite.setAngle(i);
         }
         else if (key.equals("endarrowangle"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            appSettings.stroke.setEndArrowAngle(i);
         }
         else if (key.equals("secondaryendarrowangle"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            endComposite.setAngle(i);
         }
         else if (key.equals("dashoffset"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            appSettings.stroke.dashPattern.offset = (float)i;
         }
         else if (key.equals("dash"))
         {
            t = new StringTokenizer(value, ",");

            value = t.nextToken();

            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            int n = i;

            if (n == 0)
            {
               appSettings.stroke.dashPattern.pattern=null;
            }
            else
            {
               appSettings.stroke.dashPattern.pattern = new float[n];

               value = "";

               for (int j = 0; j < n; j++)
               {
                  if (!t.hasMoreTokens())
                  {
                     throw new InvalidFormatException(
                        JDRResources.getStringWithValue(
                     "error.conf.no_more_tokens", line));
                  }
                  value = t.nextToken();

                  double f = 0;
                  try
                  {
                     f = Double.parseDouble(value);
                  }
                  catch (NumberFormatException e)
                  {
                     throw new InvalidFormatException(
                     JDRResources.getStringWithValue(
                     "error.conf.number", line));
                  }

                  if (f < 0)
                  {
                     throw new InvalidFormatException(
                        JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
                  }

                  appSettings.stroke.dashPattern.pattern[j] = (float)f;
               }

               if (t.hasMoreTokens())
               {
                  throw new InvalidFormatException(
                     JDRResources.getStringWithValue(
                     "error.conf.too_many_tokens", line));
               }
            }
         }
         else if (key.equals("linepaint"))
         {
            appSettings.linePaint = loadPaintConfig(value,line);
         }
         else if (key.equals("fillpaint"))
         {
            appSettings.fillPaint = loadPaintConfig(value,line);
         }
         else if (key.equals("textpaint"))
         {
            appSettings.textPaint = loadPaintConfig(value,line);
         }
         else if (key.equals("unit"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.integer", line));
            }

            JDRUnit unit;

            switch (i)
            {
               case JDRUnit.PT :
                  unit = JDRUnit.pt;
               break;
               case JDRUnit.IN :
                  unit = JDRUnit.in;
               break;
               case JDRUnit.CM :
                  unit = JDRUnit.cm;
               break;
               case JDRUnit.BP :
                  unit = JDRUnit.bp;
               break;
               default :
                 throw new InvalidFormatException(
                    JDRResources.getStringWithValue(
                     "error.conf.number_badrange", line));
            }
// Fix this
         }
         else if (key.equals("majordivisions"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            int majorDivisions = i;
// Fix this
         }
         else if (key.equals("subdivisions"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            int subDivisions = i;
// Fix this
         }
         else if (key.equals("controlsize"))
         {
            double i = 0;
            try
            {
               i = Double.parseDouble(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.number", line));
            }

            if (i < 0)
            {
               throw new InvalidFormatException(
                  JDRResources.getStringWithValue(
                     "error.conf.no_negative", line));
            }

            JDRPoint.pointSize = i;
         }
         else if (key.equals("widest_char"))
         {
            CanvasTextField.widestChar = value;
         }
         else if (key.equals("robot"))
         {
            int i = 0;
            try
            {
               i = Integer.parseInt(value);
            }
            catch (NumberFormatException e)
            {
               throw new InvalidFormatException(
                  JDRResources.getString("error.conf.boolean_value")
                  +" "+line);
            }

            if (i < 0 || i > 1)
            {
               throw new InvalidFormatException(
                  JDRResources.getString("error.conf.boolean_value")
                  +" "+line);
            }

            if (i == 1)
            {
               try
               {
                  appSettings.robot = new Robot();
               }
               catch (AWTException awte)
               {
                  JDRResources.warning(null, new String[] {
                     JDRResources.getString("warning.no_robot"),
                     awte.getMessage()});
                  appSettings.robot = null;
               }
               catch (SecurityException sexc)
               {
                  JDRResources.warning(null, new String[] {
                     JDRResources.getString("warning.no_robot"),
                     sexc.getMessage()});
                  appSettings.robot = null;
               }
            }
            else
            {
               appSettings.robot=null;
            }

            doneRobot=true;
         }
         else
         {
            throw new InvalidFormatException(
               JDRResources.getStringWithValue(
                  "error.conf.unknown_key",
                  new String[]{key, ""+line}));
         }
      }

      in.close();

      appSettings.stroke.getStartArrow().setCompositeMarker(
         startComposite);
      appSettings.stroke.getMidArrow().setCompositeMarker(
         midComposite);
      appSettings.stroke.getEndArrow().setCompositeMarker(
         endComposite);

      setRendering(appSettings.antialias, appSettings.renderquality);

      if (!doneRobot || diffVersion)
      {
         // no robot setting found, try initialising robot
         try
         {
            appSettings.robot = new Robot();
         }
         catch (AWTException awte)
         {
            JDRResources.warning(null, new String[] {
               JDRResources.getString("warning.no_robot"),
               awte.getMessage()});
            appSettings.robot = null;
         }
         catch (SecurityException sexc)
         {
            JDRResources.warning(null, new String[] {
               JDRResources.getString("warning.no_robot"),
               sexc.getMessage()});
            appSettings.robot = null;
         }
      }
   }

   public static void saveConfig(File file, CanvasSettings s)
      throws IOException
   {
      // saves settings to config file

      PrintWriter out = new PrintWriter(
         new FileWriter(file));

      if (JDRResources.dictLocaleId != null)
      {
         out.println("dict_lang="+JDRResources.dictLocaleId);
      }

      if (JDRResources.helpLocaleId != null)
      {
         out.println("help_lang="+JDRResources.helpLocaleId);
      }

      out.println("version="+version);
      out.println("robot="+(appSettings.robot==null?0:1));
      out.println("initsettings="+appSettings.initSettings);
      if (appSettings.initSettings==JpgfDrawSettings.INIT_DEFAULT)
      {
         out.close();
         return;
      }

      out.println("preview_bitmaps="+(appSettings.previewBitmaps?1:0));
      out.println("show_grid="+(s.isGridDisplayed()?1:0));
      out.println("lock_grid="+(s.isGridLocked()?1:0));
      out.println("show_tools="+(appSettings.showToolBar?1:0));
      out.println("show_rulers="+(s.showRulers?1:0));
      out.println("show_status="+(appSettings.showStatus?1:0));
      out.println("dragscale="+(appSettings.enableDragScale?1:0));
      out.println("antialias="+(appSettings.antialias?1:0));
      out.println("render_quality="+(appSettings.renderquality?1:0));
      out.println("startdir_type="+appSettings.startDirType);
      if (appSettings.startDirType == JpgfDrawSettings.STARTDIR_LAST)
      {
         out.println("startdir="+savejdrFC.getCurrentDirectory());
      }
      else
      {
         out.println("startdir="+appSettings.startDir);
      }

      out.println("save_jdrsettings="
         +(appSettings.saveJDRsettings?1:0));
      out.println("use_jdrsettings="
         +(appSettings.useJDRsettings?1:0));
      out.println("warn_load_old="+(appSettings.warnOnOldJdr?1:0));
      out.println("normalsize="+s.normalsize);

      out.println("latexfontupdate="
         +(appSettings.updateLaTeXFonts?1:0));
      out.println("autoupdateanchors="
         +(appSettings.autoUpdateAnchors?1:0));
      out.println("autoescapespchars="
         +(appSettings.autoEscapeSpChars?1:0));

      out.println("tool="+s.getToolString());

      if (s.paper.getID() == JDRPaper.ID_USER)
      {
         out.println("paper=user "+s.paper.getWidth()+" "
            +s.paper.getHeight());
      }
      else
      {
         out.println("paper="+s.paper.getName());
      }

      JDRMarker startComposite 
         = appSettings.stroke.getStartArrow().getCompositeMarker();
      JDRMarker midComposite 
         = appSettings.stroke.getMidArrow().getCompositeMarker();
      JDRMarker endComposite 
         = appSettings.stroke.getEndArrow().getCompositeMarker();

      out.println("fontname="+appSettings.fontFamily);
      out.println("fontshape="+appSettings.fontShape);
      out.println("fontseries="+appSettings.fontSeries);
      out.println("fontsize="+appSettings.fontSize);
      out.println("latexfontname="+appSettings.latexFontFamily);
      out.println("latexfontshape="+appSettings.latexFontShape);
      out.println("latexfontseries="+appSettings.latexFontSeries);
      out.println("latexfontsize="+appSettings.latexFontSize);

      out.println("capstyle="+appSettings.stroke.getCapStyle());
      out.println("joinstyle="+appSettings.stroke.getJoinStyle());
      out.println("windingrule="+appSettings.stroke.getWindingRule());
      out.println("penwidth="+appSettings.stroke.getPenWidth());
      out.println("mitrelimit="+appSettings.stroke.getMitreLimit());
      out.println("startarrow="+appSettings.stroke.getStartArrowType());
      out.println("startarrowsize="+appSettings.stroke.getStartArrowSize());
      out.println("midarrow="+appSettings.stroke.getMidArrowType());
      out.println("midarrowsize="+appSettings.stroke.getMidArrowSize());
      out.println("endarrow="+appSettings.stroke.getEndArrowType());
      out.println("endarrowsize="+appSettings.stroke.getEndArrowSize());
      out.println("startarrowrepeat="
         + (appSettings.stroke.getStartArrowRepeated()));
      out.println("midarrowrepeat="
         + (appSettings.stroke.getMidArrowRepeated()));
      out.println("endarrowrepeat="
         + (appSettings.stroke.getEndArrowRepeated()));

      out.println("startarrowreverse="
         + (appSettings.stroke.getStartArrowReverse() ? 1 : 0));
      out.println("midarrowreverse="
         + (appSettings.stroke.getMidArrowReverse() ? 1 : 0));
      out.println("endarrowreverse="
         + (appSettings.stroke.getEndArrowReverse() ? 1 : 0));

      out.println("startarroworient="
         + (appSettings.stroke.getStartArrowAutoOrient() ? 1 : 0));

      out.println("midarroworient="
         + (appSettings.stroke.getMidArrowAutoOrient() ? 1 : 0));

      out.println("endarroworient="
         + (appSettings.stroke.getEndArrowAutoOrient() ? 1 : 0));

      out.println("startarrowangle="
         + appSettings.stroke.getStartArrowAngle());

      out.println("midarrowangle="
         + appSettings.stroke.getMidArrowAngle());

      out.println("endarrowangle="
         + appSettings.stroke.getEndArrowAngle());

      out.println("startarrowuseroffsetenabled="
         + (appSettings.stroke.getStartUserOffsetEnabled() ? 1 : 0));
      out.println("midarrowuseroffsetenabled="
         + (appSettings.stroke.getMidUserOffsetEnabled() ? 1 : 0));
      out.println("endarrowuseroffsetenabled="
         + (appSettings.stroke.getEndUserOffsetEnabled() ? 1 : 0));

      out.println("startarrowuseroffset="
         + appSettings.stroke.getStartOffset());
      out.println("midarrowuseroffset="
         + appSettings.stroke.getMidOffset());
      out.println("endarrowuseroffset="
         + appSettings.stroke.getEndOffset());

      out.println("startarrowrepeatoffsetenabled="
         + (appSettings.stroke.getStartRepeatOffsetEnabled() ? 1 : 0));
      out.println("midarrowrepeatoffsetenabled="
         + (appSettings.stroke.getMidRepeatOffsetEnabled() ? 1 : 0));
      out.println("endarrowrepeatoffsetenabled="
         + (appSettings.stroke.getEndRepeatOffsetEnabled() ? 1 : 0));

      out.println("startarrowrepeatoffset="
         + appSettings.stroke.getStartRepeatOffset());
      out.println("midarrowrepeatoffset="
         + appSettings.stroke.getMidRepeatOffset());
      out.println("endarrowrepeatoffset="
         + appSettings.stroke.getEndRepeatOffset());

      JDRMarker startMarker = appSettings.stroke.getStartArrow();

      JDRPaint paint = startMarker.fillPaint;

      if (paint!=null)
      {
         out.println("startarrowpaint="+savePaintConfig(paint));
      }

      JDRMarker midMarker = appSettings.stroke.getMidArrow();

      paint = midMarker.fillPaint;

      if (paint!=null)
      {
         out.println("midarrowpaint="+savePaintConfig(paint));
      }

      JDRMarker endMarker = appSettings.stroke.getEndArrow();

      paint = endMarker.fillPaint;

      if (paint!=null)
      {
         out.println("endarrowpaint="+savePaintConfig(paint));
      }

      if (startComposite != null)
      {
         out.println("secondarystartarrow="+startComposite.getType());
         out.println("secondarystartarrowsize="
            +startComposite.getSize());
         out.println("secondarystartarrowrepeat="
            +startComposite.getRepeated());
         out.println("secondarystartarrowreverse="
            +(startComposite.isReversed() ? 1 : 0));
         out.println("secondarystartarroworient="
            +(startComposite.getAutoOrient() ? 1 : 0));
         out.println("secondarystartarrowangle="
            +startComposite.getAngle());
         if (startComposite.getFillPaint() != null)
         {
            out.println("secondarystartarrowpaint="
               + savePaintConfig(startComposite.getFillPaint()));
         }

         out.println("secondarystartarrowuseroffsetenabled="
            + (startComposite.isUserOffsetEnabled() ? 1 : 0));

         out.println("secondarystartarrowuseroffset="
            + startComposite.getOffset());

         out.println("secondarystartarrowrepeatoffsetenabled="
            + (startComposite.isUserRepeatOffsetEnabled() ? 1 : 0));

         out.println("secondarystartarrowrepeatoffset="
            + startComposite.getRepeatOffset());

         out.println("startoverlaid="
            + (startMarker.isOverlaid() ? 1 : 0));
      }

      if (midComposite != null)
      {
         out.println("secondarymidarrow="+midComposite.getType());
         out.println("secondarymidarrowsize="
            +midComposite.getSize());
         out.println("secondarymidarrowrepeat="
            +midComposite.getRepeated());
         out.println("secondarymidarrowreverse="
            +(midComposite.isReversed() ? 1 : 0));
         out.println("secondarymidarroworient="
            +(midComposite.getAutoOrient() ? 1 : 0));
         out.println("secondarymidarrowangle="
            +midComposite.getAngle());

         if (midComposite.getFillPaint() != null)
         {
            out.println("secondarymidarrowpaint="
               + savePaintConfig(midComposite.getFillPaint()));
         }

         out.println("secondarymidarrowuseroffsetenabled="
            + (midComposite.isUserOffsetEnabled() ? 1 : 0));

         out.println("secondarymidarrowuseroffset="
            + midComposite.getOffset());

         out.println("secondarymidarrowrepeatoffsetenabled="
            + (midComposite.isUserRepeatOffsetEnabled() ? 1 : 0));

         out.println("secondarymidarrowrepeatoffset="
            + midComposite.getRepeatOffset());

         out.println("midoverlaid="
            + (midMarker.isOverlaid() ? 1 : 0));
      }

      if (endComposite != null)
      {
         out.println("secondaryendarrow="+endComposite.getType());
         out.println("secondaryendarrowsize="
            +endComposite.getSize());
         out.println("secondaryendarrowrepeat="
            +endComposite.getRepeated());
         out.println("secondaryendarrowreverse="
            +(endComposite.isReversed() ? 1 : 0));
         out.println("secondaryendarroworient="
            +(endComposite.getAutoOrient() ? 1 : 0));
         out.println("secondaryendarrowangle="
            +endComposite.getAngle());

         if (endComposite.getFillPaint() != null)
         {
            out.println("secondaryendarrowpaint="
               + savePaintConfig(endComposite.getFillPaint()));
         }

         out.println("secondaryendarrowuseroffsetenabled="
            + (endComposite.isUserOffsetEnabled() ? 1 : 0));

         out.println("secondaryendarrowuseroffset="
            + endComposite.getOffset());

         out.println("secondaryendarrowrepeatoffsetenabled="
            + (endComposite.isUserRepeatOffsetEnabled() ? 1 : 0));

         out.println("secondaryendarrowrepeatoffset="
            + endComposite.getRepeatOffset());

         out.println("endoverlaid="
            + (endMarker.isOverlaid() ? 1 : 0));
      }

      out.println("dashoffset="
         +appSettings.stroke.dashPattern.offset);

      if (appSettings.stroke.dashPattern.pattern==null)
      {
         out.println("dash=0");
      }
      else
      {
         int n = appSettings.stroke.dashPattern.pattern.length;
         out.print("dash="+n);
         for (int i = 0; i < n; i++)
         {
            out.print(","+appSettings.stroke.dashPattern.pattern[i]);
         }
         out.println();
      }

      out.println("linepaint="
         +savePaintConfig(appSettings.linePaint));
      out.println("fillpaint="
         +savePaintConfig(appSettings.fillPaint));
      out.println("textpaint="
         +savePaintConfig(appSettings.textPaint));
/*
      out.print("unit=");
      if (s.unit == JDRUnit.pt)
      {
         out.println(CanvasSettings.PT);
      }
      else if (s.unit == JDRUnit.in)
      {
         out.println(CanvasSettings.IN);
      }
      else if (s.unit == JDRUnit.cm)
      {
         out.println(CanvasSettings.CM);
      }
      else
      {
         out.println(CanvasSettings.BP);
      }
      out.println("majordivisions="+s.majorDivisions);
      out.println("subdivisions="+s.subDivisions);
*/
      out.println("controlsize="+JDRPoint.pointSize);
      out.println("widest_char="+CanvasTextField.widestChar);

      out.close();
   }

   private static String savePaintConfig(JDRPaint paint)
   {
      return JDR.getPaintLoader().getConfigString(paint);
   }

   private static JDRPaint loadPaintConfig(String str, int line)
      throws InvalidFormatException
   {
      JDRPaint paint=null;

      try
      {
         paint = JDR.getPaintLoader().parseConfig(str);
      }
      catch (InvalidFormatException e)
      {
         throw new InvalidFormatException(
            JDRResources.getStringWithValue(
               "error.bad_paint_format", line)
               +"\n"+e.getMessage());
      }

      return paint;
   }

   public static void showMessageFrame()
   {
      messageSystem.displayMessages();
   }

   public static void showMessageFrame(String text)
   {
      messageSystem.messageln(text);
      messageSystem.displayMessages();
   }

   public static void addMessage(String text)
   {
      messageSystem.messageln(text);
   }

   public static void hideMessageFrame()
   {
      messageSystem.hideMessages();
   }

   public static void initialiseStartup()
   {
      startup = new StartupProgress();
   }

   public static void disposeStartup()
   {
      if (startup == null) return;

      startup.dismiss();
      startup = null;
   }

   public static void main(String[] args)
   {
      if (args.length == 1)
      {
         if (args[0].equals("-version"))
         {
            version();
         }

         if (args[0].equals("-help") || args[0].equals("-h"))
         {
            syntax();
         }
      }

      messageSystem = new JDRGuiMessage();
      JDR.messageSystem = messageSystem;

      CanvasSettings settings = new CanvasSettings();
      String filename = null;

      File file;

      renderHints = new RenderingHints(
         RenderingHints.KEY_ANTIALIASING,
         RenderingHints.VALUE_ANTIALIAS_OFF);


      usersettings=JDRResources.getUserConfigDir();


      if (usersettings == null)
      {
         JDRResources.error(
            JDRResources.getString("error.config_failed",
              "Can't determine configuration directory"));
      }

      // is there a config file in usersettings?

      String userconfname=null;

      if (usersettings != null)
      {
         file = new File(usersettings, "jpgfdraw.conf");

         try
         {
            userconfname = file.getCanonicalPath();
         }
         catch (IOException ignore)
         {
         }

         if (userconfname != null && file.exists() && file.isFile())
         {
            try
            {
               loadConfig(file, settings);
            }
            catch (Exception e)
            {
               e.printStackTrace();
               JDRResources.error(new String[]{"'"+userconfname+"':",
                               e.getMessage()});
            }

            if (!JDRResources.isDictInitialised())
            {
               try
               {
                  JDRResources.initialiseDictionary();
               }
               catch (IOException ioe)
               {
                  JDRResources.internalError(null, ioe);
               }
            }
         }
         else
         {
            try
            {
               JDRResources.initialiseDictionary();
            }
            catch (IOException ioe)
            {
               JDRResources.internalError(null, ioe);
            }

            // no setting found, try initialising robot
            try
            {
               appSettings.robot = new Robot();
            }
            catch (AWTException awte)
            {
               JDRResources.warning(null, new String[] {
                  JDRResources.getString("warning.no_robot"),
                  awte.getMessage()});
               appSettings.robot = null;
            }
            catch (SecurityException se)
            {
               JDRResources.warning(null, new String[] {
                  JDRResources.getString("warning.no_robot"),
                  se.getMessage()});
               appSettings.robot = null;
            }
         }
      }

      initialiseStartup();

      // load Java to LaTeX font family mappings

      String latexmappings=null;

      if (usersettings != null)
      {
         file = new File(usersettings, "latexfontmap");

         try
         {
            latexmappings = file.getCanonicalPath();

            if (latexmappings != null && file.exists()
                && file.isFile())
            {
               try
               {
                  LaTeXFont.loadJavaMappings(file);
               }
               catch (InvalidFormatException e)
               {
                  JDRResources.error(
                     new String[]{"'"+latexmappings+"':",
                                  e.getMessage()});
               }
            }
         }
         catch (IOException e)
         {
            JDRResources.error(new String[]{"'"+latexmappings+"':",
                            e.getMessage()});
         }
      }

      // process command line arguments

      for (int i = 0; i < args.length; i++)
      {
         if (args[i].equals("-disable_print"))
         {
            disablePrint = true;
         }
         else if (args[i].equals("-show_grid"))
         {
            settings.setDisplayGrid(true);
         }
         else if (args[i].equals("-noshow_grid"))
         {
            settings.setDisplayGrid(false);
         }
         else if (args[i].equals("-grid_lock"))
         {
            settings.setGridLock(true);
         }
         else if (args[i].equals("-nogrid_lock"))
         {
            settings.setGridLock(false);
         }
         else if (args[i].equals("-toolbar"))
         {
            appSettings.showToolBar = true;
         }
         else if (args[i].equals("-notoolbar"))
         {
            appSettings.showToolBar = false;
         }
         else if (args[i].equals("-statusbar"))
         {
            appSettings.showStatus = true;
         }
         else if (args[i].equals("-nostatusbar"))
         {
            appSettings.showStatus = false;
         }
         else if (args[i].equals("-rulers"))
         {
            settings.showRulers = true;
         }
         else if (args[i].equals("-norulers"))
         {
            settings.showRulers = false;
         }
         else if (args[i].equals("-paper"))
         {
            if (args.length == i+1)
            {
               JDRResources.error(
                  JDRResources.getString("error.missing_papersize"));
               System.exit(0);
            }
            i++;

            if (args[i].equals("user"))
            {
               if (args.length == i+1)
               {
                  JDRResources.error(JDRResources.getString(
                     "error.missing_paper_width"));
                  System.exit(0);
               }
               i++;
               try
               {
                  double w = JDRUnit.parseUnit(args[i]);
                  if (args.length == i+1)
                  {
                     JDRResources.error(
                        JDRResources.getString(
                           "error.missing_paper_height"));
                     System.exit(0);
                  }
                  if (w <= 0)
                  {
                     JDRResources.error(
                        JDRResources.getString(
                           "error.invalid_dimension") +": "+args[i]);
                     System.exit(0);
                  }

                  i++;
                  double h = JDRUnit.parseUnit(args[i]);

                  if (h <= 0)
                  {
                     JDRResources.error(JDRResources.getString(
                        "error.invalid_dimension")
                        +": "+args[i]);
                     System.exit(0);
                  }
                  settings.paper = new JDRPaper(w, h);
               }
               catch (InvalidUnitException e)
               {
                  JDRResources.error(
                     JDRResources.getString("error.invalid_unit")
                     +": "+e.getInvalidUnit());
                  System.exit(0);
               }
               catch (InvalidDimensionException e)
               {
                  JDRResources.error(
                     JDRResources.getString("error.paper_dimension"));
                  System.exit(0);
               }
            }
            else
            {
               JDRPaper paper = JDRPaper.getPredefinedPaper(args[i]);

               if (paper == null)
               {
                  JDRResources.error(
                    JDRResources.getStringWithValue(
                       "error.unknown_papersize", args[i]));
                  System.exit(0);
               }
               else
               {
                  settings.paper = paper;
               }
            }
         }
         else if (args[i].equals("-debug"))
         {
            JDRResources.debugMode = true;
         }
         else if (args[i].equals("-nodebug"))
         {
            JDRResources.debugMode = false;
         }
         else if (args[i].equals("-version"))
         {
            version();
         }
         else if (args[i].equals("-experimental"))
         {
            experimentalMode = true;
         }
         else if (args[i].equals("-noexperimental"))
         {
            experimentalMode = false;
         }
         else if (args[i].equals("-help") || args[i].equals("-h"))
         {
            syntax();
         }
         else if (args[i].substring(0,1).equals("-"))
         {
            JDRResources.error(JDRResources.getStringWithValue(
               "error.unknown_option", args[i]));
            System.exit(0);
         }
         else
         {
            if (filename == null)
            {
               filename = args[i];
            }
            else
            {
               JDRResources.error(JDRResources.getString(
                  "error.one_filename"));
               System.exit(0);
            }
         }
      }

      if (filename != null)
      {
         try
         {
            filename = (new File(filename)).getCanonicalPath();
         }
         catch (IOException e)
         {
            JDRResources.error(e.getMessage());
            filename = null;
         }
         catch (Exception e)
         {
            JDRResources.error(null,e);
            filename = null;
         }
      }

      if (usersettings != null)
      {
         logFile = new File(usersettings, "jpgfdraw.log");
      }
      
      try
      {
         JpgfDraw application = new JpgfDraw(settings,filename);
      }
      catch (RuntimeException e)
      {
         JDRResources.internalError(null, e);
      }
   }

   public static final String version = "0.5.6b";
   public static String usersettings;
   public static JDRGuiMessage messageSystem;
   public static final String appName = "Jpgfdraw";

   private static boolean experimentalMode = false;

   private static StartupProgress startup = null;

   private Vector<String> recentFiles=null;
   // panels

   private JComponent toolBar, sidePanel;
   private StatusBar statusBar;

   // dialog boxes

   private AboutDialog aboutDialog;
   private LicenceDialog licenceDialog;
   private LinePaintSelector linePaintChooserBox;
   private FillPaintSelector fillPaintChooserBox;
   private LineStyleSelector lineStyleChooserBox;
   private LineWidthSelector lineWidthChooserBox;
   private DashPatternSelector dashPatternChooserBox;
   private JoinStyleSelector joinStyleChooserBox;
   private ArrowStyleSelector startArrowChooserBox;
   private ArrowStyleSelector midArrowChooserBox;
   private ArrowStyleSelector endArrowChooserBox;
   private ArrowStyleSelector allMarkersChooserBox;
   private TextSelector editTextBox;
   private TextPaintSelector textPaintChooserBox;
   private FontSelector textStyleChooserBox;
   private FontFamilySelector fontFamilyChooserBox;
   private FontSizeSelector fontSizeChooserBox;
   private FontShapeSelector fontShapeChooserBox;
   private FontSeriesSelector fontSeriesChooserBox;
   private FontAnchorSelector fontAnchorChooserBox;
   private FontHAnchorSelector fontHAnchorChooserBox;
   private FontVAnchorSelector fontVAnchorChooserBox;
   private BitmapProperties bitmapPropChooserBox;
   private StyleSelector styleChooserBox;
   private GridSettings gridSettingsChooserBox;
   private ZoomSettings zoomSettingsChooserBox;
   private ConfigSettingsDialog configDialog;
   private TeXSettingsDialog texSettingsDialog;
   private RotateDialogBox rotateDB;
   private ScaleDialogBox scaleDB;
   private ShearDialogBox shearDB;
   private FLFSelector flfSelector;
   private FLFSetTypeblock setTypeblockSelector;
   private DiscardDialogBox discardDB;
   private GoToDialogBox gotoDialog;
   private MoveByDialogBox moveByDialog;
   private DisplayPageDialog displayPageDialog;
   private DescriptionDialogBox imageDescriptionDialog;
   private DescriptionDialogBox objectDescriptionDialog;
   private FindByDescriptionDialogBox findByDescDialog;
   private PaperDialogBox paperDialog;
   private SetTransformDialogBox textMatrixDialog;
   private PatternDialogBox patternBox;
   private VectorizeBitmapDialog vectorizeBitmapDialog;
   private FadeDialogBox fadeDialog;

   // file choosers
   private static JFileChooser savejdrFC, openjdrFC,
      exportFC, bitmapFC, texFC, importFC;

   // thumbnail image panel
   private static ImagePreview imagePreviewPanel;

   // file filters

   private JdrAjrFileFilter jdrAjrFileFilter;
   private JdrFileFilter jdrFileFilter;
   private JdrFileFilter[] oldJdrFileFilter;
   private AjrFileFilter ajrFileFilter;
   private AjrFileFilter[] oldAjrFileFilter;
   private TeXFileFilter pgfFileFilter;
   private TeXFileFilter pgfDocFileFilter;
   private StyFileFilter flfFileFilter;
   private PngFileFilter pngFileFilter;
   private EpsFileFilter epsFileFilter;
   private TeXFileFilter texFileFilter;
   private SvgFileFilter svgFileFilter;

   // undo/redo stuff
   private UndoManager undoManager;
   public UndoableEditSupport undoSupport;
   private JMenuItem undoItem, redoItem;

   // button and menu items

   private JDRButtonItem newButtonItem, openButtonItem, saveButtonItem,
      selectAllButtonItem, cutButtonItem, copyButtonItem, pasteButtonItem,
      moveToFrontButtonItem, moveToBackButtonItem, rotateButtonItem,
      scaleButtonItem, shearButtonItem, groupButtonItem,
      ungroupButtonItem, gapButtonItem, abandonButtonItem,
      finishButtonItem, setPatternButtonItem, convertToTextPathButtonItem;

   private JDRToggleButtonItem editPathButtonItem, showGridButtonItem,
      lockGridButtonItem;

   private JDRToolButtonItem selectButtonItem, openLineButtonItem,
      closedLineButtonItem, openCurveButtonItem, closedCurveButtonItem,
      rectangleButtonItem, ellipseButtonItem, textButtonItem;

   // menus
   private JMenuItem saveAsItem,
                     exportItem, importItem, printItem, 
                     pageDialogItem,
                     closeItem, quitItem,
                     deselectAllItem, linePaintItem, fillPaintItem,
                     lineStyleItem, lineWidthItem, dashItem,
                     startArrowItem, midArrowItem, endArrowItem,
                     allMarkersItem, editTextItem, textPaintItem,
                     textStyleItem, fontFamilyItem, fontSizeItem,
                     fontShapeItem, fontSeriesItem,
                     textMatrixItem, fontAnchorItem,
                     fontVAnchorItem, fontHAnchorItem,
                     reverseItem,
                     convertToPathItem, mergePathsItem,
                     convertToFullPathItem,
                     removePatternItem,
                     editPatternItem,
                     separateItem,
                     xorPathsItem, intersectPathsItem,
                     subtractPathsItem, pathUnionItem, 
                     leftAlignItem, centreAlignItem,
                     rightAlignItem, topAlignItem, middleAlignItem,
                     bottomAlignItem, splitTextItem, 
                     insertBitmapItem,
                     refreshItem, bitmapPropItem, parshapeItem,
                     shapeparItem, clearAllItem, setFrameItem,
                     setTypeblockItem, stylesItem, joinItem,
                     gridSettingsItem, setNormalSizeItem,
                     configSettingsItem, licenceItem, aboutItem,
                     tileItem, verticallyItem, horizontallyItem,
                     minimizeItem, maximizeItem, gotoItem, nextItem,
                     addNextItem, skipItem, moveByItem,
                     findSelectedItem, displayPageItem,
                     imageDescriptionItem, objectDescriptionItem,
                     findByDescriptionItem, addByDescriptionItem,
                     zoomWidthItem, zoomHeightItem, zoomPageItem,
                     reduceToGreyItem, vectorizeItem, fadeItem;

   private JRadioButtonMenuItem 
      zoomSettingsItem, zoom25Item,
      zoom50Item, zoom75Item, zoom100Item, zoom200Item, zoom400Item,
      zoom800Item, portraitItem, landscapeItem, paperA5Item,
      paperA4Item, paperA3Item, paperA2Item, paperA1Item, paperA0Item,
      paperLegalItem, paperLetterItem, paperExecutiveItem,
      paperOtherItem, capButtItem, capRoundItem, capSquareItem,
      windingEvenOddItem, windingNonZeroItem;

   private JCheckBoxMenuItem showToolsItem, showRulersItem,
      showStatusBarItem, 
      showPrinterMarginsItem, autoAdjustAnchorItem, 
      autoEscapeSpCharsItem;

   private JMenu fileM, editM, pathM, textM, transformM, justifyM,
                 toolsM, bitmapM, texM, settingsM, windowM, recentM,
                 gridM, zoomM, paperM, navigateM, fontStyleM,
                 lineStyleM, patternM;
   private JCheckBox jdrFCAddExt, texFCAddExt, exportFCAddExt;

   // Clipboard
   public final Clipboard clipboard = getToolkit().getSystemClipboard();

   // constants
   public static final int ACTION_SELECT=0;
   public static final int ACTION_OPEN_LINE=1;
   public static final int ACTION_CLOSED_LINE=2;
   public static final int ACTION_OPEN_CURVE=3;
   public static final int ACTION_CLOSED_CURVE=4;
   public static final int ACTION_RECTANGLE=5;
   public static final int ACTION_ELLIPSE=6;
   public static final int ACTION_TEXT=7;
   public static final int MAX_TOOLS=8;

   // desktop
   private JDesktopPane theDesktop;
   private ButtonGroup windowButtonGroup;
   private String currentFile="";
   private static JpgfDrawSettings appSettings = new JpgfDrawSettings();
   private static RenderingHints renderHints;
   private boolean activateFrameUndoRedo=false;
   private boolean iconifyFrameUndoRedo=false;

   // available fonts
   private String[] availableFontFamilies;

   // printer settings
   private HashPrintRequestAttributeSet printRequestAttributeSet;

   public static boolean disablePrint=false;

   private static File logFile;

   private JMenu debugM;

   public static KeyStroke writeLogKeyStroke
      = KeyStroke.getKeyStroke(KeyEvent.VK_F10, 0);

   public static KeyStroke dumpAllKeyStroke
      = KeyStroke.getKeyStroke(KeyEvent.VK_F11, 0);

   private WriteLogAction writeLogAction;

   private DumpAllAction dumpAllAction;

}

class StartupProgress extends Thread
{
   public StartupProgress()
   {
      super();

      initialiseStartup();
      start();
   }

   public void run()
   {
      running = true;

      try
      {
         while (running)
         {
            synchronized(this)
            {
               this.wait(100);
            }
         }
      }
      catch (InterruptedException e)
      {
      }
   }

   public synchronized void toFront()
   {
      startup.toFront();
   }

   public synchronized void dismiss()
   {
      startup.setVisible(false);
      running = false;
   }

   private synchronized void initialiseStartup()
   {
      startup = new JFrame();
      startup.setUndecorated(true);

      Toolkit tk = Toolkit.getDefaultToolkit();
      Dimension d = tk.getScreenSize();
      int screenHeight = d.height;
      int screenWidth  = d.width;
      int width = 550;
      int height = 200;
      startup.setIconImage(
         JDRResources.appIcon("jdrlogosmall.png").getImage());
      startup.setTitle(JDRResources.getString("startup.title"));

      JPanel panel = new JPanel();
      panel.setLayout(new BorderLayout());
      panel.setBorder(BorderFactory.createRaisedBevelBorder());
      startup.getContentPane().add(panel);

      Box b = Box.createVerticalBox();
      b.add(new JLabel(JDRResources.getStringWithValue(
            "about.version",
            new String[]{JpgfDraw.appName, JpgfDraw.version})+" "));

      b.add(Box.createGlue());
      b.add(new JLabel(JDRResources.getString("about.copyright")
                       +" 2006 Nicola L.C. Talbot "));
      b.add(Box.createGlue());
      b.add(new JLabel(JDRResources.getStringWithValue(
         "about.disclaimer", JpgfDraw.appName)+" "));

      b.add(new JLabel(JDRResources.getString("about.see_licence")+" "));
      b.add(Box.createGlue());
      b.add(new JLabel("http://www.dickimaw-books.com/ "));
      b.add(Box.createGlue());
      String translator=JDRResources.getString("about.translator");

      if (translator.length() > 0 && !translator.equals("?unknown?"))
      {
         b.add(new JLabel(JDRResources.getString("about.translated_by")+" "));
         b.add(Box.createGlue());
         b.add(new JLabel(translator));
         b.add(Box.createGlue());
         String url = JDRResources.getString("about.translator_url");
         if (url.length() > 0 && !url.equals("?unknown?"))
         {
            b.add(new JLabel(url+" "));
            b.add(Box.createGlue());
         }
         String info = JDRResources.getString("about.translator_info");
         if (info.length() > 0 && !info.equals("?unknown?"))
         {
            b.add(new JLabel(info+" "));
            b.add(Box.createGlue());
         }
         height += 200;
      }

      startup.setSize(width,height);
      startup.setLocation((screenWidth-width)/2,
                          (screenHeight-height)/2);

      b.add(Box.createGlue());
      startupInfo = new JLabel("");
      b.add(startupInfo);

      startupProgress = new JProgressBar();
      b.add(startupProgress);

      panel.add(b, "Center");

      Box b2 = Box.createVerticalBox();
      b2.add(new JLabel(JDRResources.appIcon("jdrlogolarge.png")));
      b2.add(Box.createGlue());
      panel.add(b2, "East");

      startup.setVisible(true);
   }

   public synchronized void setInfo(String info)
   {
      startupInfo.setText(info);
   }

   public synchronized String getInfo()
   {
      return startupInfo.getText();
   }

   public synchronized void setDeterminate(int n)
   {
      startupProgress.setIndeterminate(false);
      startupProgress.setMaximum(n);
      startupProgress.setValue(0);
   }

   public synchronized void incProgress()
   {
      startupProgress.setValue(startupProgress.getValue()+1);
   }

   public synchronized void setIndeterminate()
   {
      startupProgress.setIndeterminate(true);
   }

   private JFrame startup=null;
   private JLabel startupInfo;
   private JProgressBar startupProgress;

   private boolean running;
}
