// File          : JDRFrame.java
// Description   : Internal frame used by JpgfDraw.
// Date          : 6th February 2006
// Last Modified : 25th July 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.awt.image.*;
import java.awt.datatransfer.*;
import java.awt.print.PageFormat;

import java.io.*;

import java.beans.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.border.*;
import javax.swing.undo.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.dialog.*;

/**
 * Internal frame used by JpgfDraw.
 * @author Nicola L C Talbot
 */

public class JDRFrame extends JInternalFrame
   implements ActionListener,InternalFrameListener,FocusListener
{
   public JDRFrame(String filename, CanvasSettings jpdsettings, 
      JMenu menu, ButtonGroup buttonGroup, JpgfDraw application)
   {
      super("JpgfDraw", true, true, true, true);
      settings = jpdsettings;
      application_ = application;

      addInternalFrameListener(this);
      setBackground(Color.white);
      setAutoscrolls(true);
      newImage = true;

      menu_ = menu;
      menuItem = new JRadioButtonMenuItem(currentFilename);
      menuItem.addActionListener(this);
      menu_.insert(menuItem, 0);

      buttonGroup.add(menuItem);
      setDefaultCloseOperation(JInternalFrame.DO_NOTHING_ON_CLOSE);

      canvas = new JDRCanvas(this);
      canvas.setTransferHandler(new JDRTransferHandler());

      scrollPane = new JScrollPane(canvas);

      hRuler = 
         new RulerPanel(SwingConstants.HORIZONTAL,
                        canvas.getPreferredSize().width,
                        25, canvas);
      vRuler =
         new RulerPanel(SwingConstants.VERTICAL,
                        25, canvas.getPreferredSize().height,
                        canvas);

      unitLabel = new JLabel("pt");

      scrollPane.setColumnHeaderView(hRuler);
      scrollPane.setRowHeaderView(vRuler);
      scrollPane.setCorner(JScrollPane.UPPER_LEFT_CORNER, unitLabel);
      scrollPane.getViewport().addChangeListener(canvas);

      getContentPane().add(scrollPane, "Center");

      canvas.addMouseMotionListener(hRuler);
      canvas.addMouseMotionListener(vRuler);

      addFocusListener(this);

      // deal with page up/down etc in canvas keylistener
      scrollPane.getActionMap().getParent().clear();

      latexFonts = new LaTeXFontBase(10);

      if (filename != null)
      {
         load(filename);
      }

      if (filename == null)
      {
         String label="";
         count++;
         label=JDRResources.getString("label.untitled");
         if (count > 1) label += count;
         setFilename(label);
      }

      applySettings(settings);
   }

   public void setImage(JDRGroup image)
   {
      canvas.setImage(image);
      newImage = false;
   }

   public void selectThisFrame()
   {
      try
      {
         setSelected(true);

         if (isIcon())
         {
            setIcon(false);
            setMaximum(true);
         }
      }
      catch (PropertyVetoException ignore)
      {
      }
      moveToFront();
   }

   public void updateTextAreaBounds()
   {
      canvas.updateTextAreaBounds();
   }

   public void setFilename(String name)
   {
      currentFilename = name;
      int idx = name.lastIndexOf(File.separator);
      if (idx != -1)
      {
         name = name.substring(idx+1);
      }
      updateTitle();
      application_.setCurrentFile(currentFilename, !saved);
      application_.updateWindowMenu();
   }

   public void updateTitle()
   {
      int numHidden = canvas.getNumberOfHiddenObjects();

      if (numHidden == 0)
      {
         setTitle(currentFilename);
      }
      else if (numHidden == 1)
      {
         setTitle(currentFilename+" ("
           +JDRResources.getStringWithValue("label.hidden_object",
            "1")+")");
      }
      else
      {
         setTitle(currentFilename+" ("
           +JDRResources.getStringWithValue("label.hidden_objects",
             numHidden)+")");
      }
   }

   public String getFilename()
   {
      return currentFilename;
   }

   public boolean hasFileName()
   {
      return !currentFilename.matches(
         JDRResources.getString("label.untitled", "Untitled")
         +"\\d*");
   }

   public void addRecentFile(String filename)
   {
      application_.addRecentFile(filename);
   }

   public void saveString(String str)
   {
      application_.saveString(str);
   }

   public void applySettings(CanvasSettings s)
   {
      settings = s;
      application_.setTool(s.getTool());
      setAction(s.getTool());
      setNormalSize(s.normalsize);
      showRulers(s.showRulers);
      application_.setLockGrid(s.grid_lock);
      application_.setGridDisplay(s.display_grid);
      setGrid(s.getGrid(), false);
      setPaperSize(s.paper, false);
      setMagnification(application_.getMagnification());
   }

   public void setGrid(JDRGrid grid)
   {
      setGrid(grid, true);
   }

   public void setGrid(JDRGrid grid, boolean updateBackground)
   {
      settings.setGrid(grid);

      unitLabel.setText(grid.getUnitLabel());

      if (updateBackground) canvas.setBackgroundImage(true);

      vRuler.repaint();
      hRuler.repaint();
      if (updateBackground) canvas.repaint();
   }

   public void showGrid(boolean show)
   {
      settings.setDisplayGrid(show);
      canvas.setBackgroundImage(true);
      canvas.repaint();
   }

   public void setMagnification(double mag)
   {
      setMagnification(mag, true);
   }

   public void setMagnification(double mag, boolean updateBackground)
   {
      hRuler.setMagnification(mag);
      vRuler.setMagnification(mag);

      Dimension dim = new Dimension(
         (int)(mag*JDRUnit.getNormalizingFactor()
               *settings.paper.getWidth()),
         (int)(mag*JDRUnit.getNormalizingFactor()
               *settings.paper.getHeight()));

      setPreferredSize(dim);
      canvas.setPreferredSize(dim);
      revalidate();
      scrollPane.setPreferredSize(dim);
      scrollPane.revalidate();
      Dimension rulerDim = hRuler.getPreferredSize();
      rulerDim.width = dim.width;
      hRuler.setPreferredSize(rulerDim);
      hRuler.revalidate();
      rulerDim = vRuler.getPreferredSize();
      rulerDim.height = dim.height;
      vRuler.setPreferredSize(rulerDim);
      vRuler.revalidate();

      vRuler.repaint();
      hRuler.repaint();

      if (updateBackground) canvas.setBackgroundImage(true);

      canvas.updateTextFieldBounds();
      if (updateBackground) canvas.repaint();
   }

   public void zoomWidth()
   {
      double paperWidth = JDRUnit.getNormalizingFactor()
                        * settings.paper.getWidth();
      double paperHeight = JDRUnit.getNormalizingFactor()
                        * settings.paper.getHeight();

      Dimension dim = scrollPane.getViewport().getExtentSize();

      int viewWidth = dim.width;

      JScrollBar vBar = scrollPane.getVerticalScrollBar();

      if (!vBar.isVisible())
      {
         // Vertical scroll bar is currently not visible.
         // Will scaling cause it to reappear?

         int viewHeight = dim.height;
         dim = vBar.getSize();

         if (paperHeight*(viewWidth-dim.width)/paperWidth > viewHeight)
         {
            viewWidth -= dim.width;
         }
      }
      else if (scrollPane.getVerticalScrollBarPolicy()
            == ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED)
      {
         // Vertical scroll bar is currently visible.
         // Will scaling cause it to disappear?

         int viewHeight = dim.height;
         dim = vBar.getSize();

         if (paperHeight*(viewWidth+dim.width)/paperWidth < viewHeight)
         {
            viewWidth += dim.width;
         }
      }

      double mag = viewWidth/paperWidth;

      if (mag != 0.0) setMagnification(mag);
   }

   public void zoomHeight()
   {
      double paperWidth = JDRUnit.getNormalizingFactor()
                        * settings.paper.getWidth();
      double paperHeight = JDRUnit.getNormalizingFactor()
                        * settings.paper.getHeight();

      Dimension dim = scrollPane.getViewport().getExtentSize();

      int viewWidth = dim.width;
      int viewHeight = dim.height;

      JScrollBar hBar = scrollPane.getHorizontalScrollBar();

      if (!hBar.isVisible())
      {
         // Horizontal scroll bar is currently not visible.
         // Will scaling cause it to reappear?

         dim = hBar.getSize();

         if (paperWidth*(viewHeight-dim.height)/paperHeight > viewWidth)
         {
            viewHeight -= dim.height;
         }
      }
      else if (scrollPane.getHorizontalScrollBarPolicy()
            == ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED)
      {
         // Horizontal scroll bar is currently visible.
         // Will scaling cause it to disappear?

         dim = hBar.getSize();

         if (paperWidth*(viewHeight+dim.height)/paperHeight < viewWidth)
         {
            viewHeight += dim.height;
         }
      }

      double mag = viewHeight/paperHeight;

      if (mag != 0.0) setMagnification(mag);
   }

   public void zoomPage()
   {
      double paperWidth = JDRUnit.getNormalizingFactor()
                        * settings.paper.getWidth();
      double paperHeight = JDRUnit.getNormalizingFactor()
                        * settings.paper.getHeight();

      Dimension dim = scrollPane.getViewport().getExtentSize();

      int viewWidth = dim.width;
      int viewHeight = dim.height;

      double magX = viewWidth/paperWidth;
      double magY = viewHeight/paperHeight;

      if (magX < magY)
      {
         JScrollBar vBar = scrollPane.getVerticalScrollBar();

         if (vBar.isVisible() && 
            scrollPane.getVerticalScrollBarPolicy()
               == ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED)
         {
            // Vertical scroll bar is currently visible.
            // Will scaling cause it to disappear?

            dim = vBar.getSize();

            if (paperHeight*(viewWidth+dim.width)/paperWidth 
                  < viewHeight)
            {
               viewWidth += dim.width;

               magX = viewWidth/paperWidth;
            }
         }
      }
      else
      {
         JScrollBar hBar = scrollPane.getHorizontalScrollBar();

         if (hBar.isVisible() && 
            scrollPane.getHorizontalScrollBarPolicy()
               == ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED)
         {
            // Horizontal scroll bar is currently visible.
            // Will scaling cause it to disappear?

            dim = hBar.getSize();

            if (paperWidth*(viewHeight+dim.height)/paperHeight 
                  < viewWidth)
            {
               viewHeight += dim.height;

               magY = viewHeight/paperHeight;
            }
         }
      }

      double mag = (magX < magY ? magX : magY);

      if (mag != 0.0) setMagnification(mag);
   }

   public void setOrientation(boolean isPortrait)
   {
      JDRPaper paper = settings.paper.getPaper(isPortrait);

      if (paper != settings.paper)
      {
         setPaperSize(paper);
         canvas.setBackgroundImage(true);
      }
   }

   public void setPaperSize(JDRPaper p)
   {
      setPaperSize(p, true);
   }

   public void setPaperSize(JDRPaper p, boolean updateBackground)
   {
      settings.paper = p;
      double mag = getMagnification();

      Dimension dim = new Dimension(
         (int)(mag*JDRUnit.getNormalizingFactor()
               *settings.paper.getWidth()),
         (int)(mag*JDRUnit.getNormalizingFactor()
               *settings.paper.getHeight()));

      canvas.setPreferredSize(dim);
      canvas.revalidate();
      scrollPane.setPreferredSize(dim);
      scrollPane.revalidate();
      Dimension rulerDim = hRuler.getPreferredSize();
      rulerDim.width = dim.width;
      hRuler.setPreferredSize(rulerDim);
      hRuler.revalidate();
      rulerDim = vRuler.getPreferredSize();
      rulerDim.height = dim.height;
      vRuler.setPreferredSize(rulerDim);
      vRuler.revalidate();

      vRuler.repaint();
      hRuler.repaint();

      if (updateBackground) canvas.setBackgroundImage(true);
      canvas.resetTextField();

      if (updateBackground) repaint();
      application_.setPaperSize(p);
   }

   public void setMargins(PageFormat pf)
   {
      settings.setMargins(pf);
      application_.setPaperSize(getPaper());
      canvas.setBackgroundImage(true);
      repaint();
   }

   public void setMargins()
   {
      application_.setPaperSize(getPaper());
      canvas.setBackgroundImage(true);
      repaint();
   }

   public void updateBackground()
   {
      canvas.setBackgroundImage(true);
      repaint();
   }

   public boolean showMargins()
   {
      return application_.showMargins();
   }

   public void showRulers(boolean flag)
   {
      settings.showRulers=flag;
      scrollPane.getColumnHeader().setVisible(flag);
      scrollPane.getRowHeader().setVisible(flag);
      application_.setRulers(flag);
   }

   public CanvasSettings getSettings()
   {
      return settings;
   }

   public void goToCoordinate(double x, double y)
   {
      canvas.goToCoordinate(x, y);
   }

   public RenderingHints getRenderingHints()
   {
      return application_.getRenderingHints();
   }

   public Font getCurrentFont()
   {
      return application_.getCurrentFont();
   }

   public JDRBasicStroke getCurrentStroke()
   {
      return application_.getCurrentStroke();
   }

   public JDRPaint getCurrentLinePaint()
   {
      return application_.getCurrentLinePaint();
   }

   public JDRPaint getCurrentFillPaint()
   {
      return application_.getCurrentFillPaint();
   }

   public JDRPaint getCurrentTextPaint()
   {
      return application_.getCurrentTextPaint();
   }

   public String getCurrentFontFamily()
   {
      return application_.getCurrentFontFamily();
   }

   public String getCurrentLaTeXFontFamily()
   {
      return application_.getCurrentLaTeXFontFamily();
   }

   public int getCurrentFontSize()
   {
      return application_.getCurrentFontSize();
   }

   public int getCurrentFontSeries()
   {
      return application_.getCurrentFontSeries();
   }

   public int getCurrentFontShape()
   {
      return application_.getCurrentFontShape();
   }

   public int getCurrentPGFHAlign()
   {
      return application_.getCurrentPGFHAlign();
   }

   public int getCurrentPGFVAlign()
   {
      return application_.getCurrentPGFVAlign();
   }

   public String getCurrentLaTeXFontSize()
   {
      return application_.getCurrentLaTeXFontSize();
   }

   public String getCurrentLaTeXFontSeries()
   {
      return application_.getCurrentLaTeXFontSeries();
   }

   public String getCurrentLaTeXFontShape()
   {
      return application_.getCurrentLaTeXFontShape();
   }

   public int getSelectedFontSeries()
   {
      return canvas.getSelectedFontSeries();
   }

   public int getSelectedFontSize()
   {
      return canvas.getSelectedFontSize();
   }

   public int getSelectedFontShape()
   {
      return canvas.getSelectedFontShape();
   }

   public int getSelectedHalign()
   {
      return canvas.getSelectedHalign();
   }

   public int getSelectedValign()
   {
      return canvas.getSelectedValign();
   }

   public String getSelectedFontName()
   {
      return canvas.getSelectedFontName();
   }

   public JDRPaint getSelectedLinePaint()
   {
      return canvas.getSelectedLinePaint();
   }

   public JDRPaint getSelectedFillPaint()
   {
      return canvas.getSelectedFillPaint();
   }

   public JDRPaint getSelectedTextPaint()
   {
      return canvas.getSelectedTextPaint();
   }

   public Font getSymbolButtonFont()
   {
      return canvas.getSymbolButtonFont();
   }

   public JDRBasicStroke getSelectedStroke()
   {
      return canvas.getSelectedBasicStroke();
   }

   public JDRTextual getSelectedText()
   {
      return canvas.getSelectedText();
   }

   public JDRTextual getSelectedFont()
   {
      return canvas.getSelectedFont();
   }

   public JDRBitmap getSelectedBitmap()
   {
      return canvas.getSelectedBitmap();
   }

   public JDRCompleteObject getSelectedObject()
   {
      return canvas.getSelectedObject();
   }

   public JDRCompleteObject getObject(int index)
   {
      return canvas.getObject(index);
   }

   public JDRPath getSelectedPath()
   {
      return canvas.getSelectedPath();
   }

   public int getSelectedPatternIndex()
   {
      return canvas.getSelectedPatternIndex();
   }

   public void reduceToGrey()
   {
      canvas.reduceToGrey();
   }

   public void fade(double value)
   {
      canvas.fade(value);
   }

   public void setSelectedTextPaint(JDRPaint paint)
   {
      canvas.setSelectedTextPaint(paint);
   }

   public void setSelectedLinePaint(JDRPaint paint)
   {
      canvas.setSelectedLinePaint(paint);
   }

   public void setSelectedFillPaint(JDRPaint paint)
   {
      canvas.setSelectedFillPaint(paint);
   }

   public void setSelectedStroke(JDRBasicStroke stroke)
   {
      canvas.setSelectedStroke(stroke);
   }

   public void setSelectedLineWidth(double lineWidth)
   {
      canvas.setSelectedLineWidth(lineWidth);
   }

   public void setSelectedDashPattern(DashPattern pattern)
   {
      canvas.setSelectedDashPattern(pattern);
   }

   public void setSelectedCapStyle(int capStyle)
   {
      canvas.setSelectedCapStyle(capStyle);
   }

   public void setSelectedJoinStyle(int joinStyle)
   {
      canvas.setSelectedJoinStyle(joinStyle);
   }

   public void setSelectedJoinStyle(int joinStyle, double mitreLimit)
   {
      canvas.setSelectedJoinStyle(joinStyle, mitreLimit);
   }

   public void setSelectedStartArrow(JDRMarker marker)
   {
      canvas.setSelectedStartArrow(marker);
   }

   public void setSelectedMidArrow(JDRMarker marker)
   {
      canvas.setSelectedMidArrow(marker);
   }

   public void setSelectedEndArrow(JDRMarker marker)
   {
      canvas.setSelectedEndArrow(marker);
   }

   public void setSelectedMarkers(JDRMarker marker)
   {
      canvas.setSelectedMarkers(marker);
   }

   public void setSelectedWindingRule(int rule)
   {
      canvas.setSelectedWindingRule(rule);
   }

   public void setSelectedHalign(int align)
   {
      canvas.setSelectedHalign(align);
   }

   public void setSelectedValign(int align)
   {
      canvas.setSelectedValign(align);
   }

   public void setSelectedAnchor(int halign, int valign)
   {
      canvas.setSelectedAnchor(halign, valign);
   }

   public void setSelectedText(String text)
   {
      canvas.setSelectedText(text);
   }

   public void setSelectedText(String text, String ltxText)
   {
      canvas.setSelectedText(text, ltxText);
   }

   public void setSelectedFont(JDRText text)
   {
      canvas.setSelectedFont(text);
   }

   public void setSelectedFontFamily(String family, String latexFam)
   {
      canvas.setSelectedFontFamily(family, latexFam);
   }

   public void setSelectedFontSize(int size, String latexSize)
   {
      canvas.setSelectedFontSize(size, latexSize);
   }

   public void setSelectedFontShape(int shape, String latexShape)
   {
      canvas.setSelectedFontShape(shape, latexShape);
   }

   public void setSelectedFontSeries(int series, String latexSeries)
   {
      canvas.setSelectedFontSeries(series, latexSeries);
   }

   public void setBitmapProperties(JDRBitmap bitmap,
      String newfilename, String newlatexfilename,
      String command, double[] matrix)
   {
      canvas.setBitmapProperties(bitmap, newfilename, newlatexfilename,
         command, matrix);
   }

   public LaTeXFontBase getLaTeXFonts()
   {
      return latexFonts;
   }

   public void setNormalSize(int normalsize)
   {
      settings.normalsize = normalsize;
      latexFonts.setNormalSize(normalsize);
   }

   public int getNormalSize()
   {
      return settings.normalsize;
   }

   public void postEdit(UndoableEdit edit)
   {
      application_.postEdit(edit);
   }

   public void disableUndoRedo()
   {
      application_.disableUndoRedo();
   }

   public void refreshUndoRedo()
   {
      application_.refreshUndoRedo();
   }

   public void enableTools()
   {
      canvas.enableTools();
   }

   public void enableTools(int size, int[] number, boolean[] any)
   {
      application_.enableTools(currentTool(), size, number, any);
   }

   public void enableEditTools(boolean enable, boolean enableGap)
   {
      application_.enableEditTools(enable, enableGap);
   }

   public void enableEditTools(boolean enable)
   {
      application_.enableEditTools(enable);
   }

   public int currentTool()
   {
      return settings.getTool();
   }

   public void setTool(int tool)
   {
      try
      {
         settings.setTool(tool);
      }
      catch (InvalidToolException e)
      {
         JDRResources.internalError(this, e);
      }
      application_.setTool(tool);
      canvas.enableTools();
   }

   public void lockGrid(boolean lock)
   {
      settings.setGridLock(lock);
   }

   public boolean isPathEdited()
   {
      return application_.isPathEdited();
   }

   public void setAction(int tool)
   {
      canvas.setAction(tool);
   }

   public void refresh()
   {
      double factor = JDRUnit.getNormalizingFactor();

      JDR.bitmapReplaced = false;

      canvas.refresh();

      if (JDR.bitmapReplaced)
      {
         markAsModified();
      }

      if (factor != JDRUnit.getNormalizingFactor())
      {
         // screen mode has changed
         setMagnification(getMagnification());
      }
   }

   public void insertBitmap(String filename)
   {
      canvas.insertBitmap(filename);
   }

   public void finishPath()
   {
      canvas.finishPath();
   }

   public void abandonPath()
   {
      canvas.abandonPath();
   }

   public void gap()
   {
      canvas.gap();
   }

   public void resetGapButton()
   {
      application_.resetGapButton();
   }

   public void bottomAlign()
   {
      canvas.bottomAlign();
   }

   public void middleAlign()
   {
      canvas.middleAlign();
   }

   public void topAlign()
   {
      canvas.topAlign();
   }

   public void rightAlign()
   {
      canvas.rightAlign();
   }

   public void centreAlign()
   {
      canvas.centreAlign();
   }

   public void leftAlign()
   {
      canvas.leftAlign();
   }

   public void group()
   {
      canvas.group();
   }

   public void ungroup()
   {
      canvas.ungroup();
   }

   public void mergePaths()
   {
      canvas.mergePaths();
   }

   public void xorPaths()
   {
      canvas.xorPaths();
   }

   public void pathIntersect()
   {
      canvas.pathIntersect();
   }

   public void subtractPaths()
   {
      canvas.subtractPaths();
   }

   public void pathUnion()
   {
      canvas.pathUnion();
   }

   public void splitText()
   {
      canvas.splitText();
   }

   public void convertToPath()
   {
      canvas.convertToPath();
   }

   public void convertToFullPath()
   {
      canvas.convertToFullPath();
   }

   public void convertToTextPath()
   {
      canvas.convertToTextPath();
   }

   public void separate()
   {
      canvas.separate();
   }

   public void reverseSelectedPaths()
   {
      canvas.reverseSelectedPaths();
   }

   public void unsetAllFlowFrames()
   {
      canvas.unsetAllFlowFrames();
   }

   public void rotateSelectedPaths(double angle)
   {
      canvas.rotateSelectedPaths(angle);
   }

   public void scaleXSelectedPaths(double factor)
   {
      canvas.scaleXSelectedPaths(factor);
   }

   public void scaleYSelectedPaths(double factor)
   {
      canvas.scaleYSelectedPaths(factor);
   }

   public void scaleSelectedPaths(double factor)
   {
      canvas.scaleSelectedPaths(factor);
   }

   public void shearSelectedPaths(double factorX, double factorY)
   {
      canvas.shearSelectedPaths(factorX, factorY);
   }

   public void shapepar(boolean useOutline)
   {
      canvas.shapepar(useOutline);
   }

   public void parshape(boolean useOutline)
   {
      canvas.parshape(useOutline);
   }

   public void resetTextField()
   {
      canvas.resetTextField();
   }

   public void setCurrentPosition(String position)
   {
      application_.setCurrentPosition(position);
   }

   public boolean isUniqueLabel(int frameType, JDRCompleteObject object,
                                   String label)
   {
      return canvas.isUniqueLabel(frameType, object, label);
   }

   public void setFlowFrame(JDRCompleteObject object, FlowFrame f)
   {
      canvas.setFlowFrame(object, f);
   }

   public void setTypeblock(double left, double right,
                            double top, double bottom)
   {
      canvas.setTypeblock(left, right, top, bottom);
   }

   public FlowFrame getTypeblock()
   {
      return canvas.getTypeblock();
   }

   public void editPath()
   {
      canvas.editPath();
   }

   public void finishEditPath()
   {
      canvas.finishEditPath();
   }

   public void selectAll()
   {
      canvas.selectAll();
   }

   public void deselectAll()
   {
      canvas.deselectAll();
   }

   public void markAsSaved()
   {
      saved = true;
      newImage = false;
      application_.setModified(false);
   }

   public void markAsModified()
   {
      saved = false;
      newImage = false;
      application_.setModified(true);
   }

   public boolean isModified()
   {
      return !saved;
   }

   public boolean isSaved()
   {
      return saved;
   }

   public void print()
   {
      canvas.print();
   }

   public void load(String filename)
   {
      canvas.load(filename);
   }

   public void save(boolean useThread)
   {
      save(currentFilename, JDR.currentVersion, useThread);
   }

   public void save(String filename, float jdrversion, 
      boolean useThread)
   {
      canvas.save(filename, jdrversion, useThread);
   }

   public void save(File file, float jdrversion, boolean useThread)
   {
      canvas.save(file.getAbsolutePath(), jdrversion, useThread);
   }

   public void saveAJR(boolean useThread)
   {
      canvas.saveAJR(currentFilename, AJR.currentVersion, useThread);
   }

   public void saveAJR(String filename, float jdrversion, 
      boolean useThread)
   {
      canvas.saveAJR(filename, jdrversion, useThread);
   }

   public void savePGF(String filename)
   {
      canvas.savePGF(filename);
   }

   public void savePGFDoc(String filename)
   {
      canvas.savePGFDoc(filename);
   }

   public void saveFlowFrame(String filename)
   {
      canvas.saveFlowFrame(filename);
   }

   public void savePNG(String filename)
   {
      canvas.savePNG(filename);
   }

   public void saveEPS(String filename)
   {
      Cursor oldCursor = getCursor();
      setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      canvas.saveEPS(filename);
      setCursor(oldCursor);
   }

   public void saveSVG(String filename)
   {
      canvas.saveSVG(filename);
   }

   public DiscardDialogBox getDiscardDialogBox()
   {
      return application_.getDiscardDialogBox();
   }

   public boolean discard()
   {
      if (isIoInProgress())
      {
         JDRResources.error(this,
            JDRResources.getString("error.io.in_progress"));

         return false;
      }

      boolean isDiscarded = canvas.discard();

      if (isDiscarded)
      {
         discardWithoutSaving();
      }

      newImage = false;

      return isDiscarded;
   }

   public boolean saveAndDiscard()
   {
      if (hasFileName())
      {
         save(false);
      }
      else
      {
         if (!application_.promptAndSave(this, false))
         {
            return false;
         }
      }

      discardWithoutSaving();
      return true;
   }

   public void discardWithoutSaving()
   {
      setVisible(false);
      application_.removeFrame(this);
      menu_.remove(menuItem);
      application_.updateWindowMenu();
      newImage = true;
      dispose();
      application_.discardAllEdits();
      canvas.discardImage();
   }

   public void internalFrameClosing(InternalFrameEvent e)
   {
      discard();
   }

   public void internalFrameClosed(InternalFrameEvent e)
   {
   }

   public void internalFrameOpened(InternalFrameEvent e)
   {
   }

   public void internalFrameIconified(InternalFrameEvent e)
   {
      application_.updateStatus();
   }
 
   public void internalFrameDeiconified(InternalFrameEvent e)
   {
   }

   public void internalFrameActivated(InternalFrameEvent e)
   {
      menuItem.setSelected(true);
      applySettings(settings);
      application_.setCurrentFile(currentFilename, !saved);
      application_.enableFrameRelatedItems(true);
   }

   public void internalFrameDeactivated(InternalFrameEvent e)
   {
      abandonPath();
      application_.enableFrameRelatedItems(
         application_.getSelectedFrame()!=null);
   }

   public void focusLost(FocusEvent evt)
   {
   }

   public void focusGained(FocusEvent evt)
   {
      if (canvas == null) return;

      if (canvas.isTextFieldVisible() && !evt.isTemporary())
      {
         canvas.requestSymbolFocus();
      }
   }

   public void actionPerformed(ActionEvent evt)
   {
      Object source = evt.getSource();
      if (source == menuItem)
      {
         selectThisFrame();
      }
   }

   public void moveDrawObjectToFront()
   {
      canvas.moveToFront();
   }

   public void moveDrawObjectToBack()
   {
      canvas.moveToBack();
   }

   public void convertSelectedToPattern(JDRPattern pattern)
   {
      canvas.convertSelectedToPattern(pattern);
   }

   public void updatePattern(int index, JDRPattern pattern)
   {
      canvas.updatePattern(index, pattern);
   }

   public void removePattern()
   {
      canvas.removeSelectedPattern();
   }

   public double getMagnification()
   {
      if (hRuler == null) return 1.0;
      return hRuler.getMagnification();
   }

   public JDRGrid getGrid()
   {
      return settings.getGrid();
   }

   public JDRUnit getUnit()
   {
      return settings.getGrid().getMainUnit();
   }

   public double getAbsoluteMinorDistance(int major, int minor,
                       JDRUnit unit)
   {
      double majorH = unit.toBp(major);
      double minorH = minor;
      if (minorH > 0)
      {
         minorH = majorH/minorH;
      }
      else
      {
         minorH = 0;
      }

      return minorH*getMagnification();
   }

   public void updateRulers(double x, double y)
   {
      vRuler.update(x, y);
      hRuler.update(x, y);
   }

   public double getPaperWidth()
   {
      return settings.paper.getWidth();
   }

   public double getPaperHeight()
   {
      return settings.paper.getHeight();
   }

   public boolean getGridLock()
   {
      return settings.isGridLocked();
   }

   public void setGridLock(boolean lock)
   {
      settings.setGridLock(lock);
   }

   public boolean showGrid()
   {
      return settings.isGridDisplayed();
   }

   public boolean saveJDRsettings()
   {
      return application_.saveJDRsettings();
   }

   public void applySettingsIfRequired(CanvasSettings s)
   {
      if (application_.useJDRsettings()) applySettings(s);
   }

   public boolean warnOnOldJdr()
   {
      return application_.warnOnOldJdr();
   }

   public JDRPaper getPaper()
   {
      return settings.paper;
   }

   public void pasteSelectedPaths()
   {
      Clipboard clipboard = getClipboard();

      Transferable clipData = clipboard.getContents(clipboard);
      if (clipData != null)
      {
         if (clipData.isDataFlavorSupported(
           new DataFlavor((new JDRGroup()).getClass(), "JDRGroup")))
         {
            TransferHandler handler = canvas.getTransferHandler();
            handler.importData(canvas, clipData);
            repaint();
         }
      }
   }

   public void copySelectedPaths()
   {
      TransferHandler handler = canvas.getTransferHandler();
      handler.exportToClipboard(canvas,
         getClipboard(),TransferHandler.COPY);
   }

   public void cutSelectedPaths()
   {
      TransferHandler handler = canvas.getTransferHandler();
      handler.exportToClipboard(canvas,
         getClipboard(),TransferHandler.MOVE);
      canvas.deleteSelection();
      canvas.repaint();
   }

   public Clipboard getClipboard()
   {
      return application_.clipboard;
   }

   public JDRCanvas getCanvas()
   {
      return canvas;
   }

   public JpgfDraw getApplication()
   {
      return application_;
   }

   public void setEditPathButton(boolean flag)
   {
      application_.setEditPathButton(flag);
      canvas.enableTools();
   }

   public void setDisplayPage(int page)
   {
      canvas.setDisplayPage(page);
   }

   public int getDisplayPage()
   {
      return canvas.getDisplayPage();
   }

   public JDRGroup getAllPaths()
   {
      return canvas.getAllPaths();
   }

   public void selectObjectAndScroll(JDRCompleteObject object)
   {
      canvas.selectObjectAndScroll(object);
   }

   public void updateLaTeXFontSize()
   {
      canvas.updateLaTeXFontSize(latexFonts);
   }

   public void addObject(JDRCompleteObject object, String undoText)
   {
      canvas.addObject(object, undoText);
   }

   public void printInfo(PrintWriter out) throws IOException
   {
      out.println("is saved: "+saved);
      out.println("new image: "+newImage);
      out.println("current filename: "+currentFilename);
      out.println(latexFonts);
      out.println("Image:");
      canvas.printInfo(out);

      out.println(this);
   }

   /**
    * Gets information on selected objects.
    * @return string containing information about selected objects
    */
   public String getSelectedInfo()
   {
      return canvas.getSelectedInfo();
   }

   public void setIoInProgress(boolean flag)
   {
      ioInProgress = flag;
   }

   public boolean isIoInProgress()
   {
      return ioInProgress;
   }

   public JScrollPane scrollPane;

   private RulerPanel vRuler, hRuler;

   private JpgfDraw application_;

   private String currentFilename;
   public CanvasSettings settings;

   public boolean newImage;
   private boolean saved=true;

   private JDRCanvas canvas;

   public static int count=0;

   private JMenu menu_;
   public JMenuItem menuItem;

   private JLabel unitLabel;

   public LaTeXFontBase latexFonts;

   private boolean ioInProgress=false;
}
