// File          : JDRCanvas.java
// Description   : Panel on which to draw JDR images
// Date          : 5th June 2008
// Last Modified : 18th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.awt.image.*;
import java.awt.print.*;

import java.io.*;
import java.util.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.border.*;
import javax.swing.undo.*;

import javax.print.*;
import javax.print.attribute.*;
import javax.print.attribute.standard.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;
import uk.ac.uea.cmp.nlct.jdrresources.filter.*;
import uk.ac.uea.cmp.nlct.jdrresources.numfield.*;

import uk.ac.uea.cmp.nlct.jpgfdraw.dialog.*;

public class JDRCanvas extends JPanel
   implements MouseMotionListener,MouseListener,
              Scrollable,SymbolSelectorListener,Printable,
              ChangeListener,JDRImage
{
   public JDRCanvas(JDRFrame frame)
   {
      super();
      frame_ = frame;

      initCanvas();
   }

   private void initCanvas()
   {
      setBackground(Color.white);
      mouse = new Point2D.Double(0,0);

      paths = new JDRGroup();
      anchor = null;
      currentPath = null;
      currentSegment = null;
      selectedIndex = -1;
      scanshape = null;
      mouseDown = false;
      displayPage = PAGES_ALL;

      setLayout(null);

      textField = new CanvasTextField(this, frame_.getApplication());
      add(textField);
      setTextFieldFont(frame_.getCurrentFont());
      textField.setVisible(false);
      updateTextFieldBounds();
      //textField.addActionListener(this);

      symbolSelector
         = new CharacterSelector(frame_.getApplication(), this);
      frame_.getApplication().addGeneralAction(symbolSelector);

      CanvasAction.getFinishAction(this);
      CanvasAction.getAbandonAction(this);
      CanvasAction.getShowPopupAction(this);
      CanvasAction.getConstructClickAction(this);
      CanvasAction.getMoveLeftAction(this);
      CanvasAction.getMoveRightAction(this);
      CanvasAction.getMoveUpAction(this);
      CanvasAction.getMoveDownAction(this);
      CanvasAction.getScrollHomeUpAction(this);
      CanvasAction.getScrollHomeLeftAction(this);
      CanvasAction.getScrollEndDownAction(this);
      CanvasAction.getScrollEndRightAction(this);
      CanvasAction.getBlockScrollDownAction(this);
      CanvasAction.getBlockScrollRightAction(this);
      CanvasAction.getBlockScrollUpAction(this);
      CanvasAction.getBlockScrollLeftAction(this);
      CanvasAction.getDeleteLastAction(this);

      CanvasAction nextAction = CanvasAction.getNextAction(this);
      CanvasAction prevAction = CanvasAction.getPrevAction(this);
      CanvasAction delPointAction = CanvasAction.getDeletePointAction(this);
      CanvasAction insertAction = CanvasAction.getInsertAction(this);
      CanvasAction moveAction = CanvasAction.getMoveAction(this);

      // construct/edit text area popup menu

      textpopupMenu = new JPopupMenu();

      copyText = new JMenuItem(
          JDRResources.getString("text.copy"),
          JDRResources.getChar("text.copy.mnemonic"));
      textpopupMenu.add(copyText);
      copyText.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C,
         InputEvent.CTRL_MASK));
      copyText.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               textField.copy();
            }
         });

      cutText = new JMenuItem(
          JDRResources.getString("text.cut"),
          JDRResources.getChar("text.cut.mnemonic"));
      textpopupMenu.add(cutText);
      cutText.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,
         InputEvent.CTRL_MASK));
      cutText.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               textField.cut();
            }
         });

      JMenuItem pasteText = new JMenuItem(
          JDRResources.getString("text.paste"),
          JDRResources.getChar("text.paste.mnemonic"));
      textpopupMenu.add(pasteText);
      pasteText.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V,
         InputEvent.CTRL_MASK));
      pasteText.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               textField.paste();
            }
         });

      JMenuItem select_allText = new JMenuItem(
          JDRResources.getString("text.select_all"),
          JDRResources.getChar("text.select_all.mnemonic"));
      textpopupMenu.add(select_allText);
      select_allText.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_A,
         InputEvent.CTRL_MASK));
      select_allText.addActionListener(new ActionListener()
         {
            public void actionPerformed(ActionEvent evt)
            {
               textField.selectAll();
            }
         });

      JMenuItem insertSymbol = new JMenuItem(
         JDRResources.getString("text.insert_symbol"),
         JDRResources.getChar("text.insert_symbol.mnemonic"));
      textpopupMenu.add(insertSymbol);
      insertSymbol.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_INSERT, 0));

      insertSymbol.setActionCommand("insert");

      textField.addMouseListener(new MouseAdapter()
        {
           public void mousePressed(MouseEvent evt)
           {
              checkForPopupTrigger(evt);
           }

           public void mouseReleased(MouseEvent evt)
           {
              checkForPopupTrigger(evt);
           }
        });

      // edit path menu

      popupMenu = new JPopupMenu();

      // Next control

      popupMenu.add(nextAction.createNextPointItem());

      // Previous Control

      popupMenu.add(prevAction.createPrevPointItem());

      // Delete control

      deletePointItem = delPointAction.createDelPointItem();

      popupMenu.add(deletePointItem);

      // Insert Control

      addPointItem = insertAction.createAddPointItem();

      popupMenu.add(addPointItem);

      // Convert to line

      CanvasAction canvasAction = new CanvasAction(this);

      convertToLineItem = canvasAction.createConvertToLineItem();

      popupMenu.add(convertToLineItem);

      // Convert To Curve

      convertToCurveItem = canvasAction.createConvertToCurveItem();

      popupMenu.add(convertToCurveItem);

      // Convert To Move

      convertToMoveItem = canvasAction.createConvertToMoveItem();

      popupMenu.add(convertToMoveItem);

      // Symmetry submenu

      symmetryMenu = new JMenu(
         JDRResources.getString("editpath.symmetry"));
      popupMenu.add(symmetryMenu);

      // Toggle symmetry

      hasSymmetryItem = canvasAction.createHasSymmetryItem();

      symmetryMenu.add(hasSymmetryItem);

      // Toggle join anchor

      anchorSymmetryItem = canvasAction.createSymmetryAnchorItem();

      symmetryMenu.add(anchorSymmetryItem);

      // Toggle close anchor 

      closeAnchorSymmetryItem = canvasAction.createSymmetryCloseItem();

      symmetryMenu.add(closeAnchorSymmetryItem);

      // Continuous 

      continuousItem = canvasAction.createContinuousItem();

      popupMenu.add(continuousItem);

      // Open path submenu

      openPathMenu = new JMenu(JDRResources.getString("editpath.open"));

      openPathMenu.setMnemonic(JDRResources.getChar("editpath.open.mnemonic"));

      openPathMenu.setToolTipText(
         JDRResources.getString("tooltip.open_path"));

      popupMenu.add(openPathMenu);

      // Open (remove last segment)

      openPathRemoveItem = canvasAction.createOpenRemoveItem();

      openPathMenu.add(openPathRemoveItem);

      // Open (remove keep segment)

      openPathKeepItem = canvasAction.createOpenKeepItem();

      openPathMenu.add(openPathKeepItem);

      // Close path submenu

      closePathMenu = new JMenu(
         JDRResources.getString("editpath.close"));
      closePathMenu.setMnemonic(
         JDRResources.getChar("editpath.close.mnemonic"));
      closePathMenu.setToolTipText(
         JDRResources.getString("tooltip.close_path"));

      popupMenu.add(closePathMenu);

      // Close with line

      JDRMenuItem closePathLineItem = canvasAction.createCloseLineItem();

      closePathMenu.add(closePathLineItem);

      // Close path with continuous curve

      JDRMenuItem closePathContItem = canvasAction.createCloseContItem();

      closePathMenu.add(closePathContItem);

      // Close path, merging end points

      JDRMenuItem closePathMergeItem = canvasAction.createCloseMergeItem();

      closePathMenu.add(closePathMergeItem);

      // Move point

      movePtDialog = new MovePointDialog(frame_);
      frame_.getApplication().addGeneralAction(movePtDialog);

      JDRMenuItem coordinatesItem = moveAction.createMovePointItem();
      popupMenu.add(coordinatesItem);

      // Snap to grid

      JDRMenuItem snapToGridItem = canvasAction.createSnapItem();
      popupMenu.add(snapToGridItem);

      // Break path

      breakPathItem = canvasAction.createBreakPathItem();
      popupMenu.add(breakPathItem);

      TextAction textAction = new TextAction(frame_.getApplication());
      GeneralAction generalAction = new GeneralAction(frame_);

      // only text selected popup menu

      selectTextPopupMenu = new JPopupMenu();

      // Description

      textDescriptionItem = generalAction.createDescriptionItem(
         "selectedtext.description");
      selectTextPopupMenu.add(textDescriptionItem);

      // Edit text

      editTextItem = textAction.createEditItem();
      selectTextPopupMenu.add(editTextItem);

      selectTextPopupMenu.add(new JPopupMenu.Separator());

      // Cut

      JDRMenuItem textCutItem = generalAction.createCutItem();
      selectTextPopupMenu.add(textCutItem);

      // Copy

      JDRMenuItem textCopyItem = generalAction.createCopyItem();
      selectTextPopupMenu.add(textCopyItem);

      // Paste

      JDRMenuItem textPasteItem = generalAction.createPasteItem();
      selectTextPopupMenu.add(textPasteItem);

      selectTextPopupMenu.add(new JPopupMenu.Separator());

      // Select all

      JDRMenuItem textSelectAllItem = generalAction.createSelectAllItem();
      selectTextPopupMenu.add(textSelectAllItem);

      // Deselect all

      JDRMenuItem textDeselectAllItem = generalAction.createDeselectAllItem();
      selectTextPopupMenu.add(textDeselectAllItem);

      // Group

      textGroupItem = generalAction.createGroupItem();
      selectTextPopupMenu.add(textGroupItem);

      selectTextPopupMenu.add(new JPopupMenu.Separator());

      // Text paint

      JDRMenuItem textPaintItem = textAction.createPaintItem();
      selectTextPopupMenu.add(textPaintItem);

      // Family

      JDRMenuItem textFamilyItem = textAction.createFamilyItem();
      selectTextPopupMenu.add(textFamilyItem);

      // Size

      JDRMenuItem textSizeItem = textAction.createSizeItem();
      selectTextPopupMenu.add(textSizeItem);

      // Shape

      JDRMenuItem textShapeItem = textAction.createShapeItem();
      selectTextPopupMenu.add(textShapeItem);

      // Series

      JDRMenuItem textSeriesItem = textAction.createSeriesItem();
      selectTextPopupMenu.add(textSeriesItem);

      // anchor

      JMenu sTextAnchorM = new JMenu(
         JDRResources.getString("selectedtext.anchor"));
      sTextAnchorM.setMnemonic(
         JDRResources.getChar("selectedtext.anchor.mnemonic"));

      selectTextPopupMenu.add(sTextAnchorM);

      // both

      JDRMenuItem textAnchorItem = textAction.createAnchorItem();
      sTextAnchorM.add(textAnchorItem);

      // horizontal

      JDRMenuItem textHAnchorItem = textAction.createHAnchorItem();
      sTextAnchorM.add(textHAnchorItem);

      // vertical

      JDRMenuItem textVAnchorItem = textAction.createVAnchorItem();
      sTextAnchorM.add(textVAnchorItem);

      PathAction pathAction = new PathAction(frame_);

      // only path selected popup menu

      selectPathPopupMenu = new JPopupMenu();

      pathDescriptionItem = generalAction.createDescriptionItem
         ("selectedpath.description");
      selectPathPopupMenu.add(pathDescriptionItem);

      // Edit path

      editPathItem = pathAction.createEditItem();
      selectPathPopupMenu.add(editPathItem);

      selectPathPopupMenu.add(new JPopupMenu.Separator());

      // Cut

      JDRMenuItem pathCutItem = generalAction.createCutItem();
      selectPathPopupMenu.add(pathCutItem);

      // Copy

      JDRMenuItem pathCopyItem = generalAction.createCopyItem();
      selectPathPopupMenu.add(pathCopyItem);

      // Paste

      JDRMenuItem pathPasteItem = generalAction.createPasteItem();
      selectPathPopupMenu.add(pathPasteItem);

      selectPathPopupMenu.add(new JPopupMenu.Separator());

      // Select all

      JDRMenuItem pathSelectAllItem = generalAction.createSelectAllItem();
      selectPathPopupMenu.add(pathSelectAllItem);

      // Deselect all

      JDRMenuItem pathDeselectAllItem = generalAction.createDeselectAllItem();
      selectPathPopupMenu.add(pathDeselectAllItem);

      // Group

      pathGroupItem = generalAction.createGroupItem();
      selectPathPopupMenu.add(pathGroupItem);

      selectPathPopupMenu.add(new JPopupMenu.Separator());

      // Line paint

      JDRMenuItem linePaintItem = pathAction.createLinePaintItem();
      selectPathPopupMenu.add(linePaintItem);

      // Fill paint

      JDRMenuItem fillPaintItem = pathAction.createFillPaintItem();
      selectPathPopupMenu.add(fillPaintItem);

      // Pen width

      JDRMenuItem lineWidthItem = pathAction.createPenWidthItem();
      selectPathPopupMenu.add(lineWidthItem);

      // Dash

      JDRMenuItem dashItem = pathAction.createDashItem();
      selectPathPopupMenu.add(dashItem);

      // Cap submenu

      JMenu capMenu = new JMenu(
         JDRResources.getString("selectedpath.capstyle"));
      capMenu.setMnemonic(
         JDRResources.getChar("selectedpath.capstyle.mnemonic"));

      selectPathPopupMenu.add(capMenu);

      ButtonGroup capGroup = new ButtonGroup();

      // Butt cap

      capButtItem = pathAction.createButtCapItem();

      capMenu.add(capButtItem);
      capGroup.add(capButtItem);

      // Round cap

      capRoundItem = pathAction.createRoundCapItem();;

      capMenu.add(capRoundItem);
      capGroup.add(capRoundItem);

      // Square cap

      capSquareItem = pathAction.createSquareCapItem();

      capMenu.add(capSquareItem);
      capGroup.add(capSquareItem);

      // Join

      JDRMenuItem joinItem = pathAction.createJoinItem();
      selectPathPopupMenu.add(joinItem);

      // Markers

      JMenu markerMenu = new JMenu(JDRResources.getString("selectedpath.marker"));
      markerMenu.setMnemonic(JDRResources.getChar("selectedpath.marker.mnemonic"));

      selectPathPopupMenu.add(markerMenu);

      // All markers

      JDRMenuItem allMarkerItem = pathAction.createMarkerAllItem();
      markerMenu.add(allMarkerItem);

      markerMenu.addSeparator();

      // Start Marker

      JDRMenuItem startMarkerItem = pathAction.createMarkerStartItem();
      markerMenu.add(startMarkerItem);

      // Mid Marker

      JDRMenuItem midMarkerItem = pathAction.createMarkerMidItem();
      markerMenu.add(midMarkerItem);

      // End Marker

      JDRMenuItem endMarkerItem = pathAction.createMarkerEndItem();
      markerMenu.add(endMarkerItem);

      // Winding rule sub menu

      JMenu windingMenu = new JMenu(
         JDRResources.getString("selectedpath.windingrule"));
      windingMenu.setMnemonic(
         JDRResources.getChar("selectedpath.windingrule.mnemonic"));

      selectPathPopupMenu.add(windingMenu);

      ButtonGroup windingGroup = new ButtonGroup();

      // Even odd

      windingEvenOddItem = pathAction.createWindEvenOddItem();

      windingMenu.add(windingEvenOddItem);
      windingGroup.add(windingEvenOddItem);

      // Non zero

      windingNonZeroItem = pathAction.createWindNonZeroItem();

      windingMenu.add(windingNonZeroItem);
      windingGroup.add(windingNonZeroItem);

      // only textpaths selected popup menu

      selectTextPathPopupMenu = new JPopupMenu();

      textPathDescriptionItem = generalAction.createDescriptionItem(
         "selectedtextpath.description");
      selectTextPathPopupMenu.add(textPathDescriptionItem);

      // Edit text

      editTPTextItem = textAction.createEditItem();
      selectTextPathPopupMenu.add(editTPTextItem);

      // Edit Path

      editTPPathItem = pathAction.createEditItem();
      selectTextPathPopupMenu.add(editTPPathItem);

      selectTextPathPopupMenu.add(new JPopupMenu.Separator());

      // Cut

      JDRMenuItem textPathCutItem = generalAction.createCutItem();
      selectTextPathPopupMenu.add(textPathCutItem);

      // Copy

      JDRMenuItem textPathCopyItem = generalAction.createCopyItem();
      selectTextPathPopupMenu.add(textPathCopyItem);

      // Paste

      JDRMenuItem textPathPasteItem = generalAction.createPasteItem();
      selectTextPathPopupMenu.add(textPathPasteItem);

      selectTextPathPopupMenu.add(new JPopupMenu.Separator());

      // Select all

      JDRMenuItem textPathSelectAllItem = generalAction.createSelectAllItem();
      selectTextPathPopupMenu.add(textPathSelectAllItem);

      // Deselect all

      JDRMenuItem textPathDeselectAllItem = generalAction.createDeselectAllItem();
      selectTextPathPopupMenu.add(textPathDeselectAllItem);

      // Group

      textPathGroupItem = generalAction.createGroupItem();
      selectTextPathPopupMenu.add(textPathGroupItem);

      selectTextPathPopupMenu.add(new JPopupMenu.Separator());

      // Paint

      JDRMenuItem textPathPaintItem = textAction.createPaintItem();
      selectTextPathPopupMenu.add(textPathPaintItem);

      // Family

      JDRMenuItem textPathFamilyItem = textAction.createFamilyItem();
      selectTextPathPopupMenu.add(textPathFamilyItem);

      // Size

      JDRMenuItem textPathSizeItem = textAction.createSizeItem();
      selectTextPathPopupMenu.add(textPathSizeItem);

      // Shape

      JDRMenuItem textPathShapeItem = textAction.createShapeItem();
      selectTextPathPopupMenu.add(textPathShapeItem);

      // Series

      JDRMenuItem textPathSeriesItem = textAction.createSeriesItem();
      selectTextPathPopupMenu.add(textPathSeriesItem);

      // anchor

      JMenu textPathAnchorM = new JMenu(
         JDRResources.getString("selectedtext.anchor"));
      textPathAnchorM.setMnemonic(
         JDRResources.getChar("selectedtext.anchor.mnemonic"));
      selectTextPathPopupMenu.add(textPathAnchorM);

      // both

      JDRMenuItem textPathAnchorItem = textAction.createAnchorItem();
      textPathAnchorM.add(textPathAnchorItem);

      // horizontal

      JDRMenuItem textPathHAnchorItem = textAction.createHAnchorItem();
      textPathAnchorM.add(textPathHAnchorItem);

      // vertical

      JDRMenuItem textPathVAnchorItem = textAction.createVAnchorItem();
      textPathAnchorM.add(textPathVAnchorItem);

      // only bitmaps selected popup menu

      BitmapAction bitmapAction = new BitmapAction(frame_);

      selectBitmapPopupMenu = new JPopupMenu();

      // Description

      bitmapDescriptionItem = generalAction.createDescriptionItem(
         "selectedbitmap.description");
      selectBitmapPopupMenu.add(bitmapDescriptionItem);

      // Properties

      bitmapPropItem = bitmapAction.createBitmapPropertiesItem();
      selectBitmapPopupMenu.add(bitmapPropItem);

      selectBitmapPopupMenu.add(new JPopupMenu.Separator());

      // Cut

      bitmapCutItem = generalAction.createCutItem();
      selectBitmapPopupMenu.add(bitmapCutItem);

      // Copy

      bitmapCopyItem = generalAction.createCopyItem();
      selectBitmapPopupMenu.add(bitmapCopyItem);

      // Paste

      JDRMenuItem bitmapPasteItem = generalAction.createPasteItem();
      selectBitmapPopupMenu.add(bitmapPasteItem);

      selectBitmapPopupMenu.add(new JPopupMenu.Separator());

      // Select all

      JDRMenuItem bitmapSelectAllItem = generalAction.createSelectAllItem();
      selectBitmapPopupMenu.add(bitmapSelectAllItem);

      // Deselect all

      JDRMenuItem bitmapDeselectAllItem = generalAction.createDeselectAllItem();
      selectBitmapPopupMenu.add(bitmapDeselectAllItem);

      // Group

      bitmapGroupItem = generalAction.createGroupItem();
      selectBitmapPopupMenu.add(bitmapGroupItem);

      // selected objects popup menu

      selectPopupMenu = new JPopupMenu();

      // Description

      descriptionItem = generalAction.createDescriptionItem(
         "selected.description");
      selectPopupMenu.add(descriptionItem);

      selectPopupMenu.add(new JPopupMenu.Separator());

      // Cut

      cutItem = generalAction.createCutItem();
      selectPopupMenu.add(cutItem);

      // Copy

      copyItem = generalAction.createCopyItem();
      selectPopupMenu.add(copyItem);

      // Paste

      pasteItem = generalAction.createPasteItem();
      selectPopupMenu.add(pasteItem);

      selectPopupMenu.add(new JPopupMenu.Separator());

      // Select All

      generalSelectAllItem = generalAction.createSelectAllItem();
      selectPopupMenu.add(generalSelectAllItem);

      // Deselect All

      generalDeselectAllItem = generalAction.createDeselectAllItem();
      selectPopupMenu.add(generalDeselectAllItem);

      // Group

      generalGroupItem = generalAction.createGroupItem();
      selectPopupMenu.add(generalGroupItem);

      // Ungroup

      ungroupItem = generalAction.createUngroupItem();
      selectPopupMenu.add(ungroupItem);

      selectPopupMenu.add(new JPopupMenu.Separator());

      // Paths sub menu

      pathMenu = new JMenu(JDRResources.getString("selected.path"));
      pathMenu.setMnemonic(JDRResources.getChar("selected.path.mnemonic"));
      selectPopupMenu.add(pathMenu);

      // Line paint

      linePaintItem = pathAction.createLinePaintItem();
      pathMenu.add(linePaintItem);

      // Fill paint

      fillPaintItem = pathAction.createFillPaintItem();
      pathMenu.add(fillPaintItem);

      // Pen width

      lineWidthItem = pathAction.createPenWidthItem();
      pathMenu.add(lineWidthItem);

      // Dash

      dashItem = pathAction.createDashItem();
      pathMenu.add(dashItem);

      // Cap submenu

      capMenu = new JMenu(
         JDRResources.getString("edit.path.style.capstyle"));
      capMenu.setMnemonic(
         JDRResources.getChar("edit.path.style.capstyle.mnemonic"));

      pathMenu.add(capMenu);

      capGroup = new ButtonGroup();

      // Butt cap

      generalCapButtItem = pathAction.createButtCapItem();

      capMenu.add(generalCapButtItem);
      capGroup.add(generalCapButtItem);

      // Round cap

      generalCapRoundItem = pathAction.createRoundCapItem();

      capMenu.add(generalCapRoundItem);
      capGroup.add(generalCapRoundItem);

      // Square cap

      generalCapSquareItem = pathAction.createSquareCapItem();

      capMenu.add(generalCapSquareItem);
      capGroup.add(generalCapSquareItem);

      // Join

      joinItem = pathAction.createJoinItem();

      pathMenu.add(joinItem);

      // Marker submenu

      markerMenu = new JMenu(JDRResources.getString("selectedpath.marker"));
      markerMenu.setMnemonic(JDRResources.getChar("selectedpath.marker.mnemonic"));

      pathMenu.add(markerMenu);

      // All markers

      allMarkerItem = pathAction.createMarkerAllItem();
      markerMenu.add(allMarkerItem);

      markerMenu.addSeparator();

      // Start marker

      startMarkerItem = pathAction.createMarkerStartItem();
      markerMenu.add(startMarkerItem);

      // Mid marker

      midMarkerItem = pathAction.createMarkerMidItem();
      markerMenu.add(midMarkerItem);

      // End marker

      endMarkerItem = pathAction.createMarkerEndItem();
      markerMenu.add(endMarkerItem);

      // Winding rule submenu

      windingMenu = new JMenu(
         JDRResources.getString("edit.path.style.windingrule"));
      windingMenu.setMnemonic(
         JDRResources.getChar("edit.path.style.windingrule.mnemonic"));

      pathMenu.add(windingMenu);

      windingGroup = new ButtonGroup();

      // Even odd

      generalWindingEvenOddItem = pathAction.createWindEvenOddItem();

      windingMenu.add(generalWindingEvenOddItem);
      windingGroup.add(generalWindingEvenOddItem);

      // Non-zero

      generalWindingNonZeroItem = pathAction.createWindNonZeroItem();

      windingMenu.add(generalWindingNonZeroItem);
      windingGroup.add(generalWindingNonZeroItem);

      // Text sub menu

      textMenu = new JMenu(JDRResources.getString("selected.text"));
      textMenu.setMnemonic(
         JDRResources.getChar("selected.text.mnemonic"));
      selectPopupMenu.add(textMenu);

      // Text paint

      JDRMenuItem generalTextPaintItem = textAction.createPaintItem();
      textMenu.add(generalTextPaintItem);

      // Family

      textFamilyItem = textAction.createFamilyItem();
      textMenu.add(textFamilyItem);

      // Size

      textSizeItem = textAction.createSizeItem();
      textMenu.add(textSizeItem);

      // Shape

      textShapeItem = textAction.createShapeItem();
      textMenu.add(textShapeItem);

      // Series

      textSeriesItem = textAction.createSeriesItem();
      textMenu.add(textSeriesItem);

      // anchor

      JMenu generalTextAnchorM = new JMenu(
         JDRResources.getString("edit.text.font.anchor"));
      generalTextAnchorM.setMnemonic(
         JDRResources.getChar("edit.text.font.anchor.mnemonic"));

      textMenu.add(generalTextAnchorM);

      // both

      textAnchorItem = textAction.createAnchorItem();
      generalTextAnchorM.add(textAnchorItem);

      // horizontal

      textHAnchorItem = textAction.createHAnchorItem();
      generalTextAnchorM.add(textHAnchorItem);

      // vertical

      textVAnchorItem = textAction.createVAnchorItem();
      generalTextAnchorM.add(textVAnchorItem);

      // bitmap sub menu

      bitmapMenu = new JMenu(
         JDRResources.getString("selected.bitmap"));
      bitmapMenu.setMnemonic(
         JDRResources.getChar("selected.bitmap.mnemonic"));

      selectPopupMenu.add(bitmapMenu);

      // Insert Bitmap 

      generalInsertBitmapItem = bitmapAction.createInsertBitmapItem();
      bitmapMenu.add(generalInsertBitmapItem);

      // Bitmap Properties

      generalBitmapPropItem = bitmapAction.createBitmapPropertiesItem();
      bitmapMenu.add(generalBitmapPropItem);

      // Justify sub menu

      justifyMenu = new JMenu(
         JDRResources.getString("selected.justify"));
      justifyMenu.setMnemonic(
         JDRResources.getChar("selected.justify.mnemonic"));

      selectPopupMenu.add(justifyMenu);

      // Left align

      JDRMenuItem leftAlignItem = generalAction.createLeftAlignItem();
      justifyMenu.add(leftAlignItem);

      // Centre align

      JDRMenuItem centreAlignItem = generalAction.createCentreAlignItem();
      justifyMenu.add(centreAlignItem);

      // Right align

      JDRMenuItem rightAlignItem = generalAction.createRightAlignItem();
      justifyMenu.add(rightAlignItem);

      justifyMenu.add(new JPopupMenu.Separator());

      // Top align

      JDRMenuItem topAlignItem = generalAction.createTopAlignItem();
      justifyMenu.add(topAlignItem);

      // middle align

      JDRMenuItem middleAlignItem = generalAction.createMiddleAlignItem();
      justifyMenu.add(middleAlignItem);

      // bottom align

      JDRMenuItem bottomAlignItem = generalAction.createBottomAlignItem();
      justifyMenu.add(bottomAlignItem);

      // none selected popup menu

      noneSelectedPopupMenu = new JPopupMenu();

      // Image description

      JDRMenuItem imageDescriptionItem = generalAction.createImageDescriptionItem();
      noneSelectedPopupMenu.add(imageDescriptionItem);

      // Select all

      selectAllItem = generalAction.createSelectAllItem();
      noneSelectedPopupMenu.add(selectAllItem);

      // Find by description

      findByDescriptionItem = generalAction.createFindItem();
      noneSelectedPopupMenu.add(findByDescriptionItem);

      // Paste

      nonePasteItem = generalAction.createPasteItem();
      noneSelectedPopupMenu.add(nonePasteItem);

      // Insert bitmap

      insertBitmapItem = bitmapAction.createInsertBitmapItem();
      noneSelectedPopupMenu.add(insertBitmapItem);

      addMouseListener(this);
      addMouseMotionListener(this);

      setPreferredSize(new Dimension((int)frame_.getPaperWidth(),
                                     (int)frame_.getPaperHeight()));
   }

   public void markAsModified()
   {
      frame_.markAsModified();
   }

   public JDRGroup getAllPaths()
   {
      return paths;
   }

   public void deleteCurrentControlPoint()
   {
      int tool = frame_.currentTool();
      if (tool != JpgfDraw.ACTION_SELECT) return;

      if (editedPath == null) return;

      if (editedPath.getSelectedSegment() instanceof JDRBezier)
      {
         JDRBezier curve = (JDRBezier)editedPath.getSelectedSegment();

         JDRPoint editedPt = editedPath.getSelectedControl();

         if (editedPt == curve.getControl1()
          || editedPt == curve.getControl2())
         {
            return;
         }
      }

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      int n = editedPath.size();

      if (n == 1 || (n == 2 && editedPath.isClosed()))
      {
         JDRCompleteObject object = null;

         // Remove path

         for (int i = 0, m = paths.size(); i < m; i++)
         {
            object = paths.get(i);

            if (object == editedPath)
            {
               UndoableEdit edit = new EditPath(editedPath,false);
               ce.addEdit(edit);
               edit = new RemoveObject(object, i);
               ce.addEdit(edit);
               break;
            }
         }
      }
      else
      {
         UndoableEdit edit = new DeletePoint(editedPath);
         ce.addEdit(edit);
      }

      ce.end();
      frame_.postEdit(ce);
   }

   public void addControlPoint()
   {
      if (editedPath == null) return;

      UndoableEdit edit = new AddPoint(editedPath);
      frame_.postEdit(edit);
   }

   public void convertToLine()
   {
      if (editedPath == null) return;

      // If the current path is a symmetric path and the
      // selected segment is the line of symmetry, do
      // nothing. (Menu item should be disabled to prevent
      // this.)

      JDRPathSegment segment = editedPath.getSelectedSegment();

      if (segment == null) return;

      UndoableEdit edit;

      try
      {
         if (editedPath instanceof JDRSymmetricPath)
         {
            JDRSymmetricPath path = (JDRSymmetricPath)editedPath;

            if (segment == path.getSymmetry())
            {
               return;
            }
            else if (segment == path.getJoin())
            {
               edit = new JoinToLine(path);
            }
            else if (segment == path.getClosingSegment())
            {
               edit = new ClosingToLine(path);
            }
            else
            {
               edit = new ConvertToLine(editedPath, segment);
            }
         }
         else
         {
            edit = new ConvertToLine(editedPath, segment);
         }

         frame_.postEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_, e);
      }
   }


   public void convertToCurve()
   {
      if (editedPath == null) return;

      // If the current path is a symmetric path and the
      // selected segment is the line of symmetry, do
      // nothing. (Menu item should be disabled to prevent
      // this.)

      JDRPathSegment segment = editedPath.getSelectedSegment();

      if (segment == null) return;

      UndoableEdit edit;

      try
      {
         if (editedPath instanceof JDRSymmetricPath)
         {
            JDRSymmetricPath path = (JDRSymmetricPath)editedPath;

            if (segment == path.getSymmetry())
            {
               return;
            }
            else if (segment == path.getJoin())
            {
               edit = new JoinToCurve(path);
            }
            else if (segment == path.getClosingSegment())
            {
               edit = new ClosingToCurve(path);
            }
            else
            {
               edit = new ConvertToCurve(editedPath, segment);
            }
         }
         else
         {
            edit = new ConvertToCurve(editedPath, segment);
         }

         frame_.postEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void convertToMove()
   {
      if (editedPath == null) return;

      // If the current path is a symmetric path and the
      // selected segment is the line of symmetry, do
      // nothing. (Menu item should be disabled to prevent
      // this.)

      JDRPathSegment segment = editedPath.getSelectedSegment();

      if (segment == null) return;

      UndoableEdit edit;

      try
      {
         if (editedPath instanceof JDRSymmetricPath)
         {
            JDRSymmetricPath path = (JDRSymmetricPath)editedPath;

            if (segment == path.getSymmetry())
            {
               return;
            }
            else if (segment == path.getJoin())
            {
               edit = new JoinToMove(path);
            }
            else if (segment == path.getClosingSegment())
            {
               edit = new ClosingToMove(path);
            }
            else
            {
               edit = new ConvertToSegment(editedPath, segment);
            }
         }
         else
         {
            edit = new ConvertToSegment(editedPath, segment);
         }

         frame_.postEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void anchorSymmetry()
   {
      if (editedPath == null
       || !(editedPath instanceof JDRSymmetricPath)) return;

      boolean flag = !((JDRSymmetricPath)editedPath).isAnchored();

      try
      {
         UndoableEdit edit = new SetSymmetryJoinAnchor(flag);
         anchorSymmetryItem.setSelected(flag);

         frame_.postEdit(edit);
      }
      catch (InvalidClassException e)
      {
         // This shouldn't happen
      }
   }

   public void closeAnchorSymmetry()
   {
      if (editedPath == null
      || !(editedPath instanceof JDRSymmetricPath)) return;

      boolean flag = ((JDRSymmetricPath)editedPath).getClosingSegment() != null;

      try
      {
         UndoableEdit edit = new SetSymmetryCloseAnchor(flag);

         closeAnchorSymmetryItem.setSelected(flag);

         frame_.postEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void snapToGrid()
   {
      JDRPoint selectedPoint = getSelectedPoint();

      if (selectedPoint == null) return;

      boolean lock = frame_.getGridLock();
      frame_.setGridLock(true);
      double mag = getMagnification();
      Point2D p = getNearestTic(selectedPoint.x*mag,
                                selectedPoint.y*mag);
      frame_.setGridLock(lock);

      setSelectedPoint(p.getX(),p.getY());
   }

   public void breakPath()
   {
      if (editedPath == null || getSelectedPoint()==null)
      {
         return;
      }

      try
      {
         UndoableEdit edit = new BreakPath(editedPath);
         frame_.postEdit(edit);
      }
      catch (EmptyPathException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString(
               "internal_error.empty_path"), e);
      }
      catch (NoSegmentSelectedException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString(
               "internal_error.no_segment_selected"), e);
      }
      catch (SelectedSegmentNotFoundException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString(
               "internal_error.cant_find_selected_segment"),
               e);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_,e);
      }
   }

   public void makeContinuous()
   {
      if (editedPath == null) return;

      try
      {
         UndoableEdit edit = new MakeContinuous();
         frame_.postEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void openPath(boolean removeSeg)
   {
      if (editedPath == null) return;

      try
      {
         UndoableEdit edit = new OpenPath(editedPath, removeSeg);
         frame_.postEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void closePath(int closeType)
   {
      if (editedPath == null) return;

      try
      {
         UndoableEdit edit = new ClosePath(editedPath, closeType);
         frame_.postEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.error(frame_, e);
      }
   }

   public void moveControlOrObject()
   {
      if (editedPath != null && getSelectedPoint() != null)
      {
         movePtDialog.display();
      }
      else
      {
         frame_.getApplication().showMoveByDialog();
      }
   }

   public void showPopup()
   {
      if (textField.isVisible())
      {
         String selectedText = textField.getSelectedText();

         copyText.setEnabled(selectedText != null);
         cutText.setEnabled(selectedText != null);

         textpopupMenu.show(this, (int)mouse.getX(),
            (int)mouse.getY());

      }
      else if (editedPath != null)
      {
         showEditPathPopup(this, (int)mouse.getX(),
            (int)mouse.getY());
      }
      else if (getCurrentTool() == JpgfDraw.ACTION_SELECT)
      {
         showSelectPopup(this, (int)mouse.getX(),
            (int)mouse.getY());
      }
   }

   public void moveLeft(int modifiers)
   {
      if ((modifiers & KeyEvent.BUTTON1_MASK)
            == KeyEvent.BUTTON1_MASK)
      {
         if (!mouseDown || !JpgfDraw.isRobotEnabled()) return;

         Point pt = getMousePosition(true);
         if (pt == null) return;
         Point location = getLocationOnScreen();

         JpgfDraw.moveMouse((int)Math.round(pt.x-1+location.getX()),
                            (int)Math.round(pt.y+location.getY()));
      }
      else if (!textField.isVisible())
      {
         unitScrollLeft();
      }
   }

   public void moveRight(int modifiers)
   {
      if ((modifiers & KeyEvent.BUTTON1_MASK)
            == KeyEvent.BUTTON1_MASK)
      {
         if (!mouseDown || !JpgfDraw.isRobotEnabled()) return;

         Point pt = getMousePosition(true);
         if (pt == null) return;
         Point location = getLocationOnScreen();

         JpgfDraw.moveMouse((int)Math.round(pt.x+1+location.getX()),
                            (int)Math.round(pt.y+location.getY()));
      }
      else if (!textField.isVisible())
      {
         unitScrollRight();
      }
   }

   public void moveUp(int modifiers)
   {
      if ((modifiers & KeyEvent.BUTTON1_MASK)
            == KeyEvent.BUTTON1_MASK)
      {
         if (!mouseDown || !JpgfDraw.isRobotEnabled()) return;

         Point pt = getMousePosition(true);
         if (pt == null) return;
         Point location = getLocationOnScreen();

         JpgfDraw.moveMouse((int)Math.round(pt.x+location.getX()),
                            (int)Math.round(pt.y-1+location.getY()));
      }
      else if (!textField.isVisible())
      {
         unitScrollUp();
      }
   }

   public void moveDown(int modifiers)
   {
      if ((modifiers & KeyEvent.BUTTON1_MASK)
            == KeyEvent.BUTTON1_MASK)
      {
         if (!mouseDown || !JpgfDraw.isRobotEnabled()) return;

         Point pt = getMousePosition(true);
         if (pt == null) return;
         Point location = getLocationOnScreen();

         JpgfDraw.moveMouse((int)Math.round(pt.x+location.getX()),
                            (int)Math.round(pt.y+1+location.getY()));
      }
      else if (!textField.isVisible())
      {
         unitScrollDown();
      }
   }

   public void goToCoordinate(double x, double y)
   {
      // check not outside boundary
      double w = frame_.getPaperWidth();
      if (x > w)
      {
         x = w;
      }

      double h = frame_.getPaperHeight();
      if (y > h)
      {
         y = h;
      }

      Point2D location = scrollToLocation(x, y);
      double scale = getMagnification();
      Point pt = new Point((int)Math.round(location.getX()*scale),
                           (int)Math.round(location.getY()*scale));
      SwingUtilities.convertPointToScreen(pt, this);
      JpgfDraw.moveMouse(pt.x, pt.y);
      moveTo(location);
      frame_.updateRulers(location.getX(), location.getY());
   }

   public void deleteLast()
   {
      if (currentPath == null || currentSegment == null) return;

      int n = currentPath.size();

      if (n == 1)
      {
         abandonPath();
         return;
      }

      BBox box = currentPath.getControlBBox();
      currentSegment.mergeControlBBox(box);

      JDRPathSegment seg = currentPath.removeSegment(n-1);

      currentSegment.setStart(seg.getStartX(), seg.getStartY());

      repaint(box.getRectangle(getMagnification()));
   }

   public void findSelectedObjects()
   {
      boolean done = false;

      double minX = frame_.getPaperWidth();
      double minY = frame_.getPaperHeight();

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            done = true;
            BBox box = object.getBBox();

            if (box.getMinX() < minX) minX = box.getMinX();
            if (box.getMinY() < minY) minY = box.getMinY();
         }
      }

      if (done) goToCoordinate(minX, minY);
   }

   public void showEditPathPopup(Component component, int x, int y)
   {
       JDRPathSegment editedSegment = editedPath.getSelectedSegment();
       JDRPoint editedControl = editedPath.getSelectedControl();

       boolean isLineOfSymmetry = false;
       boolean isPartialSegment = false;

       if (editedPath.hasSymmetricPath())
       {
          JDRSymmetricPath p = editedPath.getSymmetricPath();

          if (editedSegment == p.getSymmetry())
          {
             isLineOfSymmetry = true;
          }
          else if (editedSegment instanceof JDRPartialSegment)
          {
             isPartialSegment = true;
          }

          anchorSymmetryItem.setEnabled(true);
          anchorSymmetryItem.setSelected(p.isAnchored());
          hasSymmetryItem.setSelected(true);

          if (p.isClosed())
          {
             boolean hasCloseAnchor = p.getClosingSegment() == null;

             closeAnchorSymmetryItem.setEnabled(true);
             closeAnchorSymmetryItem.setSelected(hasCloseAnchor);

             openPathRemoveItem.setEnabled(!hasCloseAnchor);
             openPathKeepItem.setEnabled(hasCloseAnchor);
          }
          else
          {
             closeAnchorSymmetryItem.setEnabled(false);
          }
       }
       else
       {
          anchorSymmetryItem.setEnabled(false);
          closeAnchorSymmetryItem.setEnabled(false);
          hasSymmetryItem.setSelected(false);
          openPathRemoveItem.setEnabled(editedPath.isClosed());
          openPathKeepItem.setEnabled(editedPath.isClosed());
       }

       boolean enableContinuous;

       if (editedSegment != null)
       {
          convertToLineItem.setEnabled(
             !(editedSegment instanceof JDRLine)
           &&!(editedSegment instanceof JDRPartialLine));

          convertToCurveItem.setEnabled(
             !(editedSegment instanceof JDRBezier) && !isLineOfSymmetry
             );

          convertToMoveItem.setEnabled(
              !editedSegment.isGap() && !isLineOfSymmetry);

          deletePointItem.setEnabled(
                !isLineOfSymmetry
             && !isPartialSegment
             && (editedControl == editedSegment.getStart()
              || editedControl == editedSegment.getEnd()));

          enableContinuous =
             (editedSegment instanceof JDRBezier
           || editedSegment instanceof JDRPartialBezier);

          addPointItem.setEnabled(!isLineOfSymmetry && !isPartialSegment);
          breakPathItem.setEnabled(!isLineOfSymmetry && !isPartialSegment);

       }
       else
       {
          enableContinuous = false;
          convertToLineItem.setEnabled(false);
          convertToMoveItem.setEnabled(false);
          convertToCurveItem.setEnabled(false);
          addPointItem.setEnabled(false);
          deletePointItem.setEnabled(false);
          breakPathItem.setEnabled(false);
       }

       openPathMenu.setEnabled(editedPath.isClosed());
       closePathMenu.setEnabled(!editedPath.isClosed());

       if (enableContinuous)
       {
          boolean hasEnd = editedPath.segmentHasEnd(editedSegment);

          int index;

          try
          {
             index = editedSegment.getControlIndex(editedControl);
          }
          catch (NoSuchElementException e)
          {
             index = editedSegment.controlCount();
          }

          int segmentIndex = editedPath.getSelectedIndex();

          if ((segmentIndex == 0 && index < 2 && !editedPath.isClosed())
           || index == 0 
           || (hasEnd && index > 1 
               && (!editedPath.hasSymmetricPath()
                 ||!(editedPath.hasSymmetricPath()
                     && index < editedSegment.controlCount()
                     && (segmentIndex == editedPath.getSymmetricPath()
                          .getUnderlyingShape().size()-1)))
              )
             )
          {
             enableContinuous = false;
          }
       }

       continuousItem.setEnabled(enableContinuous);

       popupMenu.show(component, x, y);
   }

   public void showSelectPopup(Component component, int x, int y)
   {
      int[] numSelected = paths.numberSelected(null);

      int total = numSelected[JDRGroup.ANY];

      int numTextPaths = numSelected[JDRGroup.TEXTPATH];

      int numShapes = numSelected[JDRGroup.SHAPE];
      int numGroups = numSelected[JDRGroup.GROUP];
      int numBitmaps = numSelected[JDRGroup.BITMAP];
      int numText = numSelected[JDRGroup.TEXT];
      int numPaths = numSelected[JDRGroup.PATH];

      if (total == 0)
      {
         boolean emptyImage = (paths.size() == 0);
         selectAllItem.setEnabled(!emptyImage);
         findByDescriptionItem.setEnabled(!emptyImage);

         noneSelectedPopupMenu.show(component, x, y);
      }
      else if (numText == 0 && numPaths == 0 &&  
               numTextPaths > 0
               && numBitmaps == 0 && numGroups == 0)
      {
         textPathDescriptionItem.setEnabled(
            numTextPaths==1);
         editTPTextItem.setEnabled(numTextPaths==1);
         editTPPathItem.setEnabled(numTextPaths==1);
         textPathGroupItem.setEnabled(numTextPaths > 1);

         selectTextPathPopupMenu.show(component, x, y);
      }
      else if (numShapes > 0
       && numText == 0 && numBitmaps == 0
       && numGroups == 0 && numTextPaths == 0)
      {
         pathDescriptionItem.setEnabled(numShapes==1);
         editPathItem.setEnabled(numShapes==1);
         pathGroupItem.setEnabled(numShapes > 1);

         JDRBasicStroke stroke = getSelectedBasicStroke();

         try
         {
            capButtItem.setSelected(false);
            capSquareItem.setSelected(false);
            capRoundItem.setSelected(false);

            switch (stroke.getCapStyle())
            {
               case BasicStroke.CAP_BUTT:
                  capButtItem.setSelected(true);
               break;
               case BasicStroke.CAP_SQUARE:
                  capSquareItem.setSelected(true);
               break;
               case BasicStroke.CAP_ROUND:
                  capRoundItem.setSelected(true);
               break;
            }

            windingEvenOddItem.setSelected(false);
            windingNonZeroItem.setSelected(false);

            switch (stroke.getWindingRule())
            {
               case GeneralPath.WIND_EVEN_ODD:
                  windingEvenOddItem.setSelected(true);
               break;
               case GeneralPath.WIND_NON_ZERO:
                  windingNonZeroItem.setSelected(true);
               break;
            }

            selectPathPopupMenu.show(component, x, y);
         }
         catch (NullPointerException e)
         {
            // Something's gone wrong

            JDRResources.internalError(frame_, e);
         }
      }
      else if (numText > 0 && numShapes == 0 && numBitmaps == 0
        && numGroups == 0)
      {
         textDescriptionItem.setEnabled(numText==1);
         editTextItem.setEnabled(numText==1);
         textGroupItem.setEnabled(numText > 1);

         selectTextPopupMenu.show(component, x, y);
      }
      else if (numText == 0 && numShapes == 0 && numBitmaps > 0
        && numGroups == 0)
      {
         bitmapDescriptionItem.setEnabled(numBitmaps==1);
         bitmapPropItem.setEnabled(numBitmaps==1);
         bitmapGroupItem.setEnabled(numBitmaps > 1);

         selectBitmapPopupMenu.show(component, x, y);
      }
      else
      {
         descriptionItem.setEnabled(total==1);

         pathMenu.setEnabled(paths.anyPathsSelected());
         textMenu.setEnabled(paths.anyTextSelected());
         generalBitmapPropItem.setEnabled(numBitmaps==1);
         generalGroupItem.setEnabled(total > 1);
         ungroupItem.setEnabled(numGroups > 0);
         justifyMenu.setEnabled(numGroups > 0);

         if (pathMenu.isEnabled())
         {
            JDRShape shape = getSelectedNonTextShape();
            JDRStroke stroke = shape.getStroke();

            if (stroke instanceof JDRBasicStroke)
            {
               switch (((JDRBasicStroke)stroke).getCapStyle())
               {
                  case BasicStroke.CAP_BUTT:
                     generalCapButtItem.setSelected(true);
                  break;
                  case BasicStroke.CAP_SQUARE:
                     generalCapSquareItem.setSelected(true);
                  break;
                  case BasicStroke.CAP_ROUND:
                     generalCapRoundItem.setSelected(true);
                  break;
               }
            }

            switch (stroke.getWindingRule())
            {
               case GeneralPath.WIND_EVEN_ODD:
                  generalWindingEvenOddItem.setSelected(true);
               break;
               case GeneralPath.WIND_NON_ZERO:
                  generalWindingNonZeroItem.setSelected(true);
               break;
            }
         }

         selectPopupMenu.show(component, x, y);
      }
   }

   public void doConstructMouseClick()
   {
      Point2D location
         = new Point2D.Double(mouse.getX(), mouse.getY());

      doConstructMouseClick(location);
   }

   public void doConstructMouseClick(Point2D currentPos)
   {
      JDRPaint currentLinePaint = frame_.getCurrentLinePaint();
      JDRPaint currentFillPaint = frame_.getCurrentFillPaint();
      JDRBasicStroke currentStroke = frame_.getCurrentStroke();
      int tool = frame_.currentTool();

      try
      {
         switch (tool)
         {
            case JpgfDraw.ACTION_SELECT :
               return;
            case JpgfDraw.ACTION_TEXT :
               startTextAndPostEdit(currentPos);
               break;
            case JpgfDraw.ACTION_OPEN_LINE :
            case JpgfDraw.ACTION_CLOSED_LINE :
               if (anchor != null && currentSegment!=null
                  && currentPath != null)
               {
                  currentPath.add(currentSegment);

                  if (!(currentSegment instanceof JDRLine
                      || currentSegment instanceof JDRBezier))
                  {
                     frame_.resetGapButton();
                  }
               }
               else
               {
                  currentPath = new JDRPath(currentLinePaint,
                                         currentFillPaint,
                                         (JDRBasicStroke)currentStroke.clone());
                  frame_.enableEditTools(true);
                  frame_.disableUndoRedo();
               }
               currentSegment = new JDRLine(currentPos,
                                         currentPos);
               anchor = currentPos;
               break;
            case JpgfDraw.ACTION_OPEN_CURVE :
            case JpgfDraw.ACTION_CLOSED_CURVE :
               if (anchor != null && currentSegment != null
                   && currentPath!=null)
               {
                  currentPath.add(currentSegment);

                  if (!(currentSegment instanceof JDRLine
                      || currentSegment instanceof JDRBezier))
                  {
                     frame_.resetGapButton();
                  }
               }
               else
               {
                  currentPath = new JDRPath(currentLinePaint,
                                         currentFillPaint,
                                         (JDRBasicStroke)currentStroke.clone());
                  frame_.enableEditTools(true);
                  frame_.disableUndoRedo();
               }
               currentSegment = new JDRBezier(currentPos,
                                           currentPos);
               anchor = currentPos;
               break;
         case JpgfDraw.ACTION_RECTANGLE :
         case JpgfDraw.ACTION_ELLIPSE :
            if (anchor == null)
            {
               frame_.disableUndoRedo();
               frame_.enableEditTools(true, false);
               anchor = currentPos;
            }
            else
            {
               finishPath();
            }
            break;
         }
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_,e);
      }
   }

   public void moveSelectedObjects(double shift_left, double shift_up)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            edit = new MoveObject(object, -shift_left, -shift_up);
            ce.addEdit(edit);
         }
      }

      if (edit != null)
      {
         ce.end();
         frame_.postEdit(ce);
      }
   }

   public JDRGroup getVisibleObjects()
   {
      if (displayPage == PAGES_ALL)
      {
         return paths;
      }

      int n = paths.size();

      JDRGroup visibleObjects;

      if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
      {
         visibleObjects = new JDRGroup(n);
      }
      else
      {
         visibleObjects = new JDRGroup();
      }

      for (int i = 0; i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (isObjectVisible(object))
         {
            visibleObjects.add(object);
         }
      }

      return visibleObjects;
   }

   public void selectNextObject()
   {
      // select next object in stack

      boolean done=false;
      JDRCompleteObject prevobject = null;
      JDRCompleteObject object = null;

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      JDRGroup visible = getVisibleObjects();

      for (int i = 0, n = visible.size(); i < n; i++)
      {
         JDRCompleteObject thisobject = visible.get(i);

         if (thisobject.isSelected())
         {
             if (object == null) object = prevobject;

             UndoableEdit edit = new SelectObject(thisobject,false,
                 JDRResources.getString("undo.deselect"));
             ce.addEdit(edit);
             done = true;
         }

         prevobject = thisobject;
      }

      if (object == null)
      {
         object = prevobject;
      }

      if (object != null)
      {
         UndoableEdit edit = new SelectObject(object,true,
            JDRResources.getString("undo.select"));

         ce.addEdit(edit);
         done = true;
      }

      if (done)
      {
         ce.end();
         frame_.postEdit(ce);
      }
   }

   public void addNextObject()
   {
      // select next object in stack (without deselecting others)

      JDRCompleteObject prevobject = null;
      JDRCompleteObject object = null;

      JDRGroup visible = getVisibleObjects();

      for (int i = 0, n = visible.size(); i < n; i++)
      {
         JDRCompleteObject thisobject = visible.get(i);

         if (thisobject.isSelected())
         {
             if (object == null) object = prevobject;
         }

         prevobject = thisobject;
      }

      if (object == null)
      {
         object = prevobject;
      }

      if (object != null)
      {
         UndoableEdit edit = new SelectObject(object,true,
            JDRResources.getString("undo.select"));

         frame_.postEdit(edit);
      }
   }

   public void skipObject()
   {
      // deselects lowest selected object, and selects object behind

      JDRCompleteObject prevobject = null;
      JDRCompleteObject object = null;
      JDRCompleteObject oldObject=null; // object marked for deselection

      JDRGroup visible = getVisibleObjects();

      for (int i = 0, n = visible.size(); i < n; i++)
      {
         JDRCompleteObject thisobject = visible.get(i);

         if (thisobject.isSelected())
         {
             if (object == null)
             {
                object = prevobject;
                oldObject = thisobject;
             }
         }

         prevobject = thisobject;
      }

      if (object == null)
      {
         object = prevobject;
      }

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      boolean done=false;

      if (oldObject != null)
      {
         UndoableEdit edit = new SelectObject(oldObject,false,
            JDRResources.getString("undo.deselect"));

         ce.addEdit(edit);
         done = true;
      }

      if (object != null)
      {
         UndoableEdit edit = new SelectObject(object,true,
            JDRResources.getString("undo.select"));

         ce.addEdit(edit);
         done = true;
      }

      if (done)
      {
         ce.end();
         frame_.postEdit(ce);
      }
   }

   public void selectNextControl()
   {
      if (editedPath == null) return;

      UndoableEdit edit = new SelectNextControl();
      frame_.postEdit(edit);
   }

   public void selectPrevControl()
   {
      if (editedPath == null) return;

      UndoableEdit edit = new SelectPrevControl();
      frame_.postEdit(edit);
   }

   public void skipObjOrNextPt()
   {
      if (editedPath == null)
      {
         skipObject();
      }
      else
      {
         selectNextControl();
      }
   }

   public void addNextObjOrPrevPt()
   {
      if (editedPath == null)
      {
         addNextObject();
      }
      else
      {
         selectPrevControl();
      }
   }

   public void insertControlOrSymbol()
   {
      if (editedPath != null)
      {
         addControlPoint();
      }
      else if (textField.isVisible())
      {
         symbolSelector.display();
      }
   }

   public JDRGrid getGrid()
   {
      return frame_.getGrid();
   }

   public JDRUnit getUnit()
   {
      return frame_.getUnit();
   }

   private boolean checkForPopupTrigger(MouseEvent evt)
   {
      if (evt.isPopupTrigger())
      {
         if (textField.isVisible())
         {
            textpopupMenu.show(evt.getComponent(),
                               evt.getX(), evt.getY());

            return true;
         }
         else if (frame_.currentTool() == JpgfDraw.ACTION_SELECT)
         {
            if (editedPath != null)
            {
               showEditPathPopup(evt.getComponent(),
                                 evt.getX(),evt.getY());
            }
            else
            {
               showSelectPopup(evt.getComponent(),
                  evt.getX(), evt.getY());
            }

            return true;
         }
      }

      return false;
   }

   public void refresh()
   {
      paths.refresh();
      repaint();
   }

   public void insertBitmap(String imagefilename)
   {
      JDRBitmap bitmap = null;

      try
      {
         bitmap = new JDRBitmap(imagefilename);
      }
      catch (InvalidLinkNameException e)
      {
         JDRResources.error(frame_,
            JDRResources.getStringWithValue("error.invalid_linkname",
            e.getInvalidFilename()));
         return;
      }
      catch (InvalidImageFormatException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_imageformat"));
         return;
      }

      if (bitmap.isDraft())
      {
         JDRResources.warning(frame_,
                          JDRResources.getString("warning.draft_bitmap"));
      }

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      UndoableEdit edit;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         edit = new SelectObject(
            paths.get(i), false,
            JDRResources.getString("undo.deselect_all"));
         ce.addEdit(edit);
      }

      edit = new AddObject(bitmap,
         JDRResources.getString("undo.new_bitmap"));
      ce.addEdit(edit);

      edit = new SelectObject(
         bitmap, true, JDRResources.getString("undo.new_bitmap"));
      ce.addEdit(edit);

      ce.end();
      frame_.postEdit(ce);
   }

   public void parshape(boolean useOutline)
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRPath)
         {
            Graphics2D g2 = (Graphics2D)getGraphics();
            JDRPath p = (JDRPath)object;
            double scale = getMagnification();
            g2.scale(scale,scale);

            try
            {
               scanshape = p.parshape(g2, 
                  frame_.getLaTeXFonts().getBaselineskip(LaTeXFontBase.NORMALSIZE),
                  useOutline);

               frame_.saveString(scanshape.string);
            }
            catch (InvalidShapeException e)
            {
               JOptionPane.showMessageDialog(frame_,
               new String[]
               {JDRResources.getString("error.parshape.convert"),
                e.getMessage()},
               JDRResources.getString("error.shape.incompatible"),
               JOptionPane.ERROR_MESSAGE);
            }
            g2.dispose();
            scanshape = null;
            repaint(p.getBBox().getRectangle(scale));
            return;
         }
      }
   }

   public void shapepar(boolean useOutline)
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRPath)
         {
            Graphics2D g2 = (Graphics2D)getGraphics();
            JDRPath p = (JDRPath)object;
            double scale = getMagnification();
            g2.scale(scale,scale);

            try
            {
               scanshape = p.shapepar(g2, 
                  frame_.getLaTeXFonts().getBaselineskip(LaTeXFontBase.NORMALSIZE),
                  useOutline);

               frame_.saveString(scanshape.string);
            }
            catch (InvalidShapeException e)
            {
               JOptionPane.showMessageDialog(frame_,
               new String[]
               {JDRResources.getString("error.shapepar.convert"),
                e.getMessage()},
               JDRResources.getString("error.shape.incompatible"),
               JOptionPane.ERROR_MESSAGE);
            }
            g2.dispose();
            
            scanshape = null;
            repaint(p.getBBox().getRectangle(scale));
            return;
         }
      }
   }

   public void updatePattern(int index, JDRPattern pattern)
   {
      try
      {
         frame_.postEdit(new UpdatePattern(index, pattern));
      }
      catch (Exception e)
      {
         JDRResources.internalError(this, e);
      }
   }

   public void convertSelectedToPattern(JDRPattern pattern)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected() && object instanceof JDRShape)
            {
               UndoableEdit edit =
                  new ConvertToPattern((JDRShape)object, pattern, i);
               ce.addEdit(edit);

               flag = true;
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidReplicaException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public void removeSelectedPattern()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRPattern)
         {
            UndoableEdit edit =
               new RemovePattern((JDRPattern)object, i);
            ce.addEdit(edit);

            flag = true;
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public void leftAlign()
   {
      alignSelectedGroups(AlignGroup.LEFT);
   }

   public void centreAlign()
   {
      alignSelectedGroups(AlignGroup.CENTRE);
   }

   public void rightAlign()
   {
      alignSelectedGroups(AlignGroup.RIGHT);
   }

   public void topAlign()
   {
      alignSelectedGroups(AlignGroup.TOP);
   }

   public void middleAlign()
   {
      alignSelectedGroups(AlignGroup.MIDDLE);
   }

   public void bottomAlign()
   {
      alignSelectedGroups(AlignGroup.BOTTOM);
   }

   public void alignSelectedGroups(int align)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRGroup)
         {
            edit = new AlignGroup((JDRGroup)object,i, align);
            ce.addEdit(edit);
         }
      }

      ce.end();
      if (edit != null) frame_.postEdit(ce);
   }

   public void setSelectedLineWidth(double lineWidth)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setLineWidth(lineWidth, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidPenWidthException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setLineWidth(double width, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
      throws InvalidPenWidthException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setLineWidth(width, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape
       && !(object instanceof JDRTextual))
      {
         UndoableEdit edit = new SetLineWidth((JDRShape)object, width);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedDashPattern(DashPattern pattern)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setDashPattern(pattern, object, ce);
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public boolean setDashPattern(DashPattern pattern,
                            JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setDashPattern(pattern, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if ((object instanceof JDRShape)
        && !(object instanceof JDRTextual))
      {
         UndoableEdit edit = new SetDashPattern((JDRShape)object, pattern);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedCapStyle(int capStyle)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setCapStyle(capStyle, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidCapStyleException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setCapStyle(int style, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
      throws InvalidCapStyleException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setCapStyle(style, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape
       && !(object instanceof JDRTextual))
      {
         UndoableEdit edit = new SetCapStyle((JDRShape)object, style);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedJoinStyle(int joinStyle)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setJoinStyle(joinStyle, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidJoinStyleException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setJoinStyle(int style, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
      throws InvalidJoinStyleException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setJoinStyle(style, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape
        && !(object instanceof JDRTextual))
      {
         UndoableEdit edit = new SetJoinStyle((JDRShape)object, style);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedJoinStyle(int joinStyle, double mitreLimit)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setJoinStyle(joinStyle, mitreLimit, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidJoinStyleException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
      catch (InvalidMitreLimitException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setJoinStyle(int style, double mitreLimit,
      JDRCompleteObject object, JDRCanvasCompoundEdit ce)
      throws InvalidJoinStyleException,
             InvalidMitreLimitException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setJoinStyle(style, mitreLimit, grp.get(i), ce)
                 || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape
        && !(object instanceof JDRTextual))
      {
         UndoableEdit edit = new SetJoinStyle((JDRShape)object, style);
         ce.addEdit(edit);
         edit = new SetMitreLimit((JDRPath)object, mitreLimit);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedMitreLimit(double limit)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setMitreLimit(limit, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidMitreLimitException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setMitreLimit(double limit, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
      throws InvalidMitreLimitException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setMitreLimit(limit, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape
       && !(object instanceof JDRTextual))
      {
         UndoableEdit edit = new SetMitreLimit((JDRShape)object, limit);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedStartArrow(JDRMarker marker)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setStartArrow(marker, object, ce) || flag;
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public boolean setStartArrow(JDRMarker marker,
       JDRCompleteObject object, JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setStartArrow(marker, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape && !object.hasTextual())
      {
         try
         {
            UndoableEdit edit = new SetStartArrow((JDRShape)object, marker);
            ce.addEdit(edit);
            return true;
         }
         catch (InvalidMarkerTypeException e)
         {
            JDRResources.internalError(frame_,
               JDRResources.getString("error.invalid_arrow_type")
               +" "+e.getInvalidID(), e);
         }
      }

      return false;
   }

   public void setSelectedMidArrow(JDRMarker marker)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setMidArrow(marker, object, ce) || flag;
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public boolean setMidArrow(JDRMarker marker,
       JDRCompleteObject object, JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setMidArrow(marker, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape && !object.hasTextual())
      {
         try
         {
            UndoableEdit edit = new SetMidArrow((JDRShape)object, marker);
            ce.addEdit(edit);
            return true;
         }
         catch (InvalidMarkerTypeException e)
         {
            JDRResources.internalError(frame_,
               JDRResources.getString("error.invalid_arrow.type")
               +" "+e.getInvalidID(), e);
         }
      }

      return false;
   }

   public void setSelectedEndArrow(JDRMarker marker)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setEndArrow(marker, object, ce) || flag;
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public boolean setEndArrow(JDRMarker marker,
       JDRCompleteObject object, JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setEndArrow(marker, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape && !object.hasTextual())
      {
         try
         {
            UndoableEdit edit = new SetEndArrow((JDRShape)object, marker);
            ce.addEdit(edit);
            return true;
         }
         catch (InvalidMarkerTypeException e)
         {
            JDRResources.internalError(frame_,
               JDRResources.getString("error.invalid_arrow_type")
               +" "+e.getInvalidID(), e);
         }
      }

      return false;
   }

   public void setSelectedMarkers(JDRMarker marker)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setMarkers(marker, object, ce) || flag;
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public boolean setMarkers(JDRMarker marker,
       JDRCompleteObject object, JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setMarkers(marker, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape && !object.hasTextual())
      {
         try
         {
            UndoableEdit edit = new SetMarkers((JDRShape)object, marker);
            ce.addEdit(edit);
            return true;
         }
         catch (InvalidMarkerTypeException e)
         {
            JDRResources.internalError(frame_,
               JDRResources.getString("error.invalid_arrow_type")
               +" "+e.getInvalidID(), e);
         }
      }

      return false;
   }

   public void setSelectedWindingRule(int rule)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setWindingRule(rule, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidWindingRuleException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setWindingRule(int style, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
      throws InvalidWindingRuleException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setWindingRule(style, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape && !object.hasTextual())
      {
         UndoableEdit edit = new SetWindingRule((JDRShape)object, style);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedHalign(int align)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setHalign(align, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidHAlignException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setHalign(int align, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
      throws InvalidHAlignException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setHalign(align, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object.hasTextual())
      {
         UndoableEdit edit
            = new SetHAlign(object.getTextual(), align);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedValign(int align)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setValign(align, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidVAlignException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setValign(int align, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
      throws InvalidVAlignException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setValign(align, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object.hasTextual())
      {
         UndoableEdit edit
            = new SetVAlign(object.getTextual(), align);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedAnchor(int halign, int valign)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

         boolean flag = false;

         for (int i = 0, n = paths.size(); i < n; i++)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               flag = setAnchor(halign, valign, object, ce);
            }
         }

         ce.end();
         if (flag) frame_.postEdit(ce);
      }
      catch (InvalidHAlignException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
      catch (InvalidVAlignException e)
      {
         // this shouldn't happen
         JDRResources.internalError(this, e);
      }
   }

   public boolean setAnchor(int halign, int valign, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
      throws InvalidVAlignException,InvalidHAlignException
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setAnchor(halign, valign, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object.hasTextual())
      {
         UndoableEdit edit
            = new SetAnchor(object.getTextual(), halign, valign);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedStroke(JDRBasicStroke s)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setStroke(s, object, ce);
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public boolean setStroke(JDRBasicStroke s, JDRCompleteObject object, 
                            JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setStroke(s, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape)
      {
         UndoableEdit edit = new SetLineStyle((JDRShape)object, s);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void unsetFlowFrame(JDRCompleteObject object, JDRCanvasCompoundEdit ce)
   {
      FlowFrame f = null;

      if (object instanceof JDRGroup)
      {
         JDRGroup group = (JDRGroup)object;

         for (int i = 0, n = group.size(); i < n; i++)
         {
            unsetFlowFrame(group.get(i), ce);
         }
      }

      UndoableEdit edit = new SetFlowFrame(object, f,
         JDRResources.getString("undo.clear_flowframes"));

      ce.addEdit(edit);
   }

   public void unsetAllFlowFrames()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      unsetFlowFrame(paths, ce);

      ce.end();
      frame_.postEdit(ce);
      repaint();
      setBackgroundImage(true);
   }

   public FlowFrame getTypeblock()
   {
      return paths.flowframe;
   }

   public void setTypeblock(double left, double right,
                            double top, double bottom)
   {
      UndoableEdit edit = new SetTypeblock(left, right, top, bottom);
      frame_.postEdit(edit);
   }

   public boolean isUniqueLabel(JDRGroup group, int frameType,
      JDRCompleteObject object, String label)
   {
      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject o = group.get(i);

         if (o.flowframe != null && o != object)
         {
            if (o.flowframe.getType() == frameType 
              && o.flowframe.label.equals(label))
            {
               return false;
            }
         }

         if (o instanceof JDRGroup)
         {
            if (!isUniqueLabel((JDRGroup)o,frameType,object,label))
            {
               return false;
            }
         }
      }

      return true;
   }

   public boolean isUniqueLabel(int frameType, JDRCompleteObject object,
                                   String label)
   {
      return isUniqueLabel(paths, frameType, object, label);
   }

   public static boolean containsFlowFrameData(JDRGroup group)
   {
      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object.flowframe != null)
         {
            return true;
         }
      }

      return false;
   }

   public boolean setFlowFrame(JDRCompleteObject object, FlowFrame f)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      if (object instanceof JDRGroup)
      {
         JDRGroup group = (JDRGroup)object;

         if (f != null && containsFlowFrameData(group))
         {
            int response = JOptionPane.showConfirmDialog(frame_,
               JDRResources.getString("flowframe.confirm.group"),
               JDRResources.getString("flowframe.confirm.group.title"),
               JOptionPane.YES_NO_OPTION,
               JOptionPane.QUESTION_MESSAGE);

            if (response == JOptionPane.YES_OPTION)
            {
               for (int i = 0, n = group.size(); i < n; i++)
               {
                  JDRCompleteObject o = group.get(i);
                  UndoableEdit edit = new SetFlowFrame(o, null);
                  ce.addEdit(edit);
               }
            }
            else
            {
               return false;
            }
         }
      }

      UndoableEdit edit = new SetFlowFrame(object, f);
      ce.addEdit(edit);
      ce.end();
      frame_.postEdit(ce);

      return true;
   }

   public JDRCompleteObject getSelectedObject()
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            return object;
         }
      }

      return null;
   }

   public JDRCompleteObject getObject(int index)
   {
      JDRCompleteObject object = null;

      try
      {
         object = paths.get(index);
      }
      catch (ArrayIndexOutOfBoundsException e)
      {
         JDRResources.internalError(frame_, "Can't located object "+index, e);
      }

      return object;
   }

   public int getSelectedPatternIndex()
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.hasPattern() && object.isSelected())
         {
            return i;
         }
      }

      return -1;
   }


   public void setBitmapProperties(JDRBitmap bitmap,
      String newfilename, String newlatexfilename,
      String command, double[] matrix)
   {
      UndoableEdit edit = new SetBitmapProperties(
                             bitmap, newfilename,
                             newlatexfilename,
                             command, matrix);
      frame_.postEdit(edit);

      if (bitmap.isDraft())
      {
         JDRResources.warning(frame_,
                          JDRResources.getString("warning.draft_bitmap"));
      }
   }

   public void setSymmetry()
   {
      if (editedPath == null) return;

      boolean flag = !editedPath.hasSymmetricPath();

      hasSymmetryItem.setSelected(flag);
      setSymmetry(flag);
   }

   private void setSymmetry(boolean addSymmetry)
   {
      try
      {
         for (int i = 0; i < paths.size(); i++)
         {
            if (paths.get(i) == editedPath)
            {
               UndoableEdit edit = new SetSymmetricPath(editedPath, addSymmetry, i);

               frame_.postEdit(edit);

               return;
            }
         }
      }
      catch (uk.ac.uea.cmp.nlct.jdr.InvalidObjectException e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public JDRBasicStroke getSelectedBasicStroke()
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRShape)
         {
            JDRShape shape = (JDRShape)object;
            JDRStroke stroke = shape.getStroke();

            if (stroke instanceof JDRBasicStroke)
            {
               return (JDRBasicStroke) stroke;
            }
         }
      }

      return frame_.getCurrentStroke();
   }

   public boolean setLinePaint(JDRPaint paint, JDRCompleteObject object, 
                               JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setLinePaint(paint, grp.get(i), ce) || flag;
         }
         return flag;
      }
      else if (object instanceof JDRShape && !object.hasTextual())
      {
         UndoableEdit edit = new SetLinePaint((JDRShape)object, paint);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public boolean setTextPaint(JDRPaint paint, JDRCompleteObject object, 
                               JDRCanvasCompoundEdit ce)
   {
      boolean flag=false;

      if (object instanceof JDRGroup)
      {
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setTextPaint(paint, grp.get(i), ce) || flag;
         }
      }
      else if (object.hasTextual())
      {
         UndoableEdit edit
            = new SetTextPaint(object.getTextual(), paint);
         ce.addEdit(edit);
         return true;
      }

      return flag;
   }

   public boolean setFillPaint(JDRPaint paint, JDRCompleteObject object, 
                               JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = setFillPaint(paint, grp.get(i), ce) || flag;
         }

         return flag;
      }
      else if (object instanceof JDRShape && !object.hasTextual())
      {
         UndoableEdit edit = new SetFillPaint((JDRShape)object, paint);
         ce.addEdit(edit);
         return true;
      }

      return false;
   }

   public void setSelectedLinePaint(JDRPaint paint)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setLinePaint(paint, object, ce);
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public void setSelectedTextPaint(JDRPaint paint)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setTextPaint(paint, object, ce);
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public JDRPaint getSelectedLinePaint()
   {
      JDRShape shape = getSelectedNonTextShape();

      if (shape != null)
      {
         return shape.getLinePaint();
      }

      return frame_.getCurrentLinePaint();
   }

   public JDRPaint getSelectedTextPaint()
   {
      JDRTextual txt = getSelectedText();

      if (txt != null)
      {
         return txt.getTextPaint();
      }

      return frame_.getCurrentTextPaint();
   }

   public void setSelectedFillPaint(JDRPaint paint)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = setFillPaint(paint, object, ce);
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public void reduceToGrey()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      boolean flag = false;
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            flag = reduceToGrey(object, ce);
         }
      }

      ce.end();
      if (flag) frame_.postEdit(ce);
   }

   public boolean reduceToGrey(JDRCompleteObject object, 
                               JDRCanvasCompoundEdit ce)
   {
      if (object instanceof JDRGroup)
      {
         boolean flag=false;
         JDRGroup grp = (JDRGroup)object;

         for (int i = 0, n = grp.size(); i < n; i++)
         {
            flag = reduceToGrey(grp.get(i), ce) || flag;
         }

         return flag;
      }
      else if (object instanceof JDRShape && !object.hasTextual())
      {
         JDRShape path = (JDRShape)object;

         JDRPaint linePaint = path.getLinePaint();

         if (linePaint instanceof JDRShading)
         {
            linePaint = (JDRPaint)linePaint.clone();
            ((JDRShading)linePaint).reduceToGreyScale();
         }
         else if (!(linePaint instanceof JDRTransparent))
         {
            linePaint = linePaint.getJDRGray();
         }

         JDRPaint fillPaint = path.getFillPaint();

         if (fillPaint instanceof JDRShading)
         {
            fillPaint = (JDRPaint)fillPaint.clone();
            ((JDRShading)fillPaint).reduceToGreyScale();
         }
         else if (!(fillPaint instanceof JDRTransparent))
         {
            fillPaint = fillPaint.getJDRGray();
         }

         UndoableEdit edit;

         edit = new SetFillPaint(path, fillPaint);
         ce.addEdit(edit);
         edit = new SetLinePaint(path, linePaint);
         ce.addEdit(edit);

         JDRBasicStroke stroke = (JDRBasicStroke)path.getStroke();

         JDRMarker marker = stroke.getStartArrow();

         if (marker.fillPaint != null)
         {
            marker = (JDRMarker)marker.clone();

            if (marker.fillPaint instanceof JDRShading)
            {
               // shaded marker fill paint isn't implemented
               // yet, but add the code in case it is implemented
               // in future
               ((JDRShading)marker.fillPaint).reduceToGreyScale();
            }
            else
            {
               marker.fillPaint = marker.fillPaint.getJDRGray();
            }

            try
            {
               edit = new SetStartArrow(path, marker);
               ce.addEdit(edit);
            }
            catch (InvalidMarkerTypeException e)
            {
            }
         }

         marker = stroke.getMidArrow();

         if (marker.fillPaint != null)
         {
            marker = (JDRMarker)marker.clone();

            if (marker.fillPaint instanceof JDRShading)
            {
               ((JDRShading)marker.fillPaint).reduceToGreyScale();
            }
            else
            {
               marker.fillPaint = marker.fillPaint.getJDRGray();
            }

            try
            {
               edit = new SetMidArrow(path, marker);
               ce.addEdit(edit);
            }
            catch (InvalidMarkerTypeException e)
            {
            }
         }

         marker = stroke.getEndArrow();

         if (marker.fillPaint != null)
         {
            marker = (JDRMarker)marker.clone();

            if (marker.fillPaint instanceof JDRShading)
            {
               ((JDRShading)marker.fillPaint).reduceToGreyScale();
            }
            else
            {
               marker.fillPaint = marker.fillPaint.getJDRGray();
            }

            try
            {
               edit = new SetEndArrow(path, marker);
               ce.addEdit(edit);
            }
            catch (InvalidMarkerTypeException e)
            {
            }
         }

         return true;
      }
      else if (object.hasTextual())
      {
         JDRTextual text = object.getTextual();

         JDRPaint textPaint = text.getTextPaint();

         if (textPaint instanceof JDRShading)
         {
            textPaint = (JDRPaint)textPaint.clone();
            ((JDRShading)textPaint).reduceToGreyScale();
         }
         else if (!(textPaint instanceof JDRTransparent))
         {
            textPaint = textPaint.getJDRGray();
         }

         UndoableEdit edit;

         edit = new SetTextPaint(text, textPaint);
         ce.addEdit(edit);

         return true;
      }

      return false;
   }

   public void fade(double value)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      for (int i = paths.size()-1; i >= 0; i--)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            edit = new FadeObject(object, value, i);
            ce.addEdit(edit);
         }
      }

      ce.end();
      frame_.postEdit(ce);
   }

   public JDRPaint getSelectedFillPaint()
   {
      JDRShape shape = getSelectedNonTextShape();

      if (shape != null)
      {
         return shape.getFillPaint();
      }

      return frame_.getCurrentFillPaint();
   }

   public JDRTextual getSelectedFont()
   {
      JDRTextual text = getSelectedText();

      if (text == null)
      {
         text = new JDRText();
      }

      return text;
   }

   public String getSelectedFontName()
   {
      return getSelectedFont().getFontFamily();
   }

   public int getSelectedFontShape()
   {
      return getSelectedFont().getFontShape();
   }

   public int getSelectedFontSeries()
   {
      return getSelectedFont().getFontSeries();
   }

   public int getSelectedFontSize()
   {
      return getSelectedFont().getFontSize();
   }

   public int getSelectedHalign()
   {
      return getSelectedText().getHAlign();
   }

   public int getSelectedValign()
   {
      return getSelectedText().getVAlign();
   }

   public JDRTextual getSelectedText()
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRGroup)
            {
               JDRTextual txt = ((JDRGroup)object).getText();

               if (txt != null)
               {
                  return txt;
               }
            }
            else if (object.hasTextual())
            {
               return object.getTextual();
            }
         }
      }

      return null; 
   }

   public JDRShape getSelectedShape()
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRShape)
            {
               return (JDRShape)object;
            }

            if (object instanceof JDRGroup)
            {
               JDRShape path = ((JDRGroup)object).getShape();

               if (path != null)
               {
                  return path;
               }
            }
         }
      }

      return null; 
   }

   public JDRPath getSelectedPath()
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRPath)
            {
               return (JDRPath)object;
            }

            if (object instanceof JDRGroup)
            {
               JDRPath path = ((JDRGroup)object).getPath();

               if (path != null)
               {
                  return path;
               }
            }
         }
      }

      return null; 
   }

   public JDRShape getSelectedNonTextShape()
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRShape
            && !(object.hasTextual()))
            {
               return (JDRShape)object;
            }

            if (object instanceof JDRGroup)
            {
               JDRShape shape = ((JDRGroup)object).getNonTextShape();

               if (shape != null)
               {
                  return shape;
               }
            }
         }
      }

      return null; 
   }

   public JDRBitmap getSelectedBitmap()
   {
      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRBitmap)
            {
               return (JDRBitmap)object;
            }

            if (object instanceof JDRGroup)
            {
               JDRBitmap bitmap = ((JDRGroup)object).getBitmap();

               if (bitmap != null)
               {
                  return bitmap;
               }
            }
         }
      }

      return null; 
   }

   public boolean isTextFieldVisible()
   {
      return textField.isVisible();
   }

   public String getSymbolText()
   {
      return textField.getText();
   }

   public int getSymbolCaretPosition()
   {
      return textField.getCaretPosition();
   }

   public void setSymbolCaretPosition(int position)
   {
      textField.setCaretPosition(position);
   }

   public void requestSymbolFocus()
   {
      textField.requestFocusInWindow();
   }

   public void setSelectedText(String text)
   {
      setSelectedText(text, text);
   }

   public void setSelectedText(String text, String ltxText)
   {
      JDRTextual object = getSelectedText();

      if (object != null)
      {
         if (text.equals(""))
         {
             JDRResources.error(frame_,
                 JDRResources.getString("error.empty_string"));
         }
         else
         {
            UndoableEdit edit = new SetText(object, text, ltxText);
            frame_.postEdit(edit);
         }
      }
   }

   public boolean setSelectedFont(JDRGroup group, JDRTextual text,
      JDRCanvasCompoundEdit ce)
   throws InvalidFontWeightException,
          InvalidFontShapeException,
          InvalidFontSizeException,
          InvalidHAlignException,
          InvalidVAlignException
   {
      boolean done = false;

      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRGroup)
            {
               done = setSelectedFont(((JDRGroup)object), text, ce);
            }
            else if (object.hasTextual())
            {
               UndoableEdit edit = new SetFont(object.getTextual(),
                                       text.getFontFamily(),
                                       text.getFontSeries(),
                                       text.getFontShape(),
                                       text.getFontSize(),
                                       text.getLaTeXFamily(), 
                                       text.getLaTeXSize(), 
                                       text.getLaTeXSeries(),
                                       text.getLaTeXShape(),
                                       text.getHAlign(),
                                       text.getVAlign());

               ce.addEdit(edit);
               done = true;
            }
         }
      }

      return done;
   }

   public void setSelectedFont(JDRTextual text)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
         boolean done=false;

         done = setSelectedFont(paths, text, ce);

         ce.end();
         if (done) frame_.postEdit(ce);
      }
      catch (InvalidFormatException e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public boolean setSelectedFontFamily(JDRGroup group,
      String family, String latexFam, JDRCanvasCompoundEdit ce)
   {
      boolean done = false;

      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRGroup)
            {
               done = setSelectedFontFamily(((JDRGroup)object), 
                  family, latexFam, ce);
            }
            else if (object.hasTextual())
            {
               UndoableEdit edit
                  = new SetFontFamily(object.getTextual(),
                                      family, latexFam);

               ce.addEdit(edit);
               done = true;
            }
         }
      }

      return done;
   }

   public void setSelectedFontFamily(String family, String latexFam)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      boolean done=false;

      done = setSelectedFontFamily(paths, family, latexFam, ce);

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public boolean setSelectedFontSize(JDRGroup group, int size,
      String latexSize, JDRCanvasCompoundEdit ce)
   throws InvalidFontSizeException
   {
      boolean done = false;

      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRGroup)
            {
               done = setSelectedFontSize(((JDRGroup)object), size,
                         latexSize, ce);
            }
            else if (object.hasTextual())
            {
               UndoableEdit edit
                  = new SetFontSize(object.getTextual(),
                                    size, latexSize);

               ce.addEdit(edit);
               done = true;
            }
         }
      }

      return done;
   }

   public void setSelectedFontSize(int size, String latexSize)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
         boolean done=false;

         done = setSelectedFontSize(paths, size, latexSize, ce);

         ce.end();
         if (done) frame_.postEdit(ce);
      }
      catch (InvalidFontSizeException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("error.invalid_size"), e);
      }
   }

   public boolean setSelectedFontShape(JDRGroup group, int shape,
      String latexShape, JDRCanvasCompoundEdit ce)
   throws InvalidFontShapeException
   {
      boolean done = false;

      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRGroup)
            {
               done = setSelectedFontShape(((JDRGroup)object), shape,
                         latexShape, ce);
            }
            else if (object.hasTextual())
            {
               UndoableEdit edit
                  = new SetFontShape(object.getTextual(),
                                     shape, latexShape);

               ce.addEdit(edit);
               done = true;
            }
         }
      }

      return done;
   }

   public void setSelectedFontShape(int shape, String latexShape)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
         boolean done=false;

         done = setSelectedFontShape(paths, shape, latexShape, ce);

         ce.end();
         if (done) frame_.postEdit(ce);
      }
      catch (InvalidFontShapeException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("error.invalid_shape"), e);
      }
   }

   public boolean setSelectedFontSeries(JDRGroup group, int series,
      String latexSeries, JDRCanvasCompoundEdit ce)
   throws InvalidFontWeightException
   {
      boolean done = false;

      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRGroup)
            {
               done = setSelectedFontSeries(((JDRGroup)object), series,
                         latexSeries, ce);
            }
            else if (object.hasTextual())
            {
               UndoableEdit edit
                  = new SetFontSeries(object.getTextual(),
                                       series, latexSeries);

               ce.addEdit(edit);
               done = true;
            }
         }
      }

      return done;
   }

   public void setSelectedFontSeries(int series, String latexSeries)
   {
      try
      {
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
         boolean done=false;

         done = setSelectedFontSeries(paths, series, latexSeries, ce);

         ce.end();
         if (done) frame_.postEdit(ce);
      }
      catch (InvalidFontWeightException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("error.invalid_series"), e);
      }
   }

   public boolean setSelectedTextTransform(
      JDRGroup group, double[] matrix, JDRCanvasCompoundEdit ce)
   {
      boolean done = false;

      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object.isSelected())
         {
            if (object instanceof JDRGroup)
            {
               done = setSelectedTextTransform(((JDRGroup)object), 
                  matrix, ce);
            }
            else if (object.hasTextual())
            {
               UndoableEdit edit
                  = new SetTextTransform(object.getTextual(), matrix);

               ce.addEdit(edit);
               done = true;
            }
         }
      }

      return done;
   }

   public void setSelectedTextTransform(double[] matrix)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      boolean done=false;

      done = setSelectedTextTransform(paths, matrix, ce);

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public void reverseSelectedPaths()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRShape)
         {
            try
            {
               edit = new ReversePath((JDRShape)object,i);
               ce.addEdit(edit);
            }
            catch (Exception excp)
            {
               JDRResources.internalError(frame_, excp);
            }
         }
      }

      ce.end();
      if (edit != null) frame_.postEdit(ce);
   }

   public boolean updateLaTeXFontSize(JDRGroup group, 
      LaTeXFontBase latexFonts, JDRCanvasCompoundEdit ce)
   {
      boolean done = false;

      for (int i = 0, n = group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object instanceof JDRGroup)
         {
            done = updateLaTeXFontSize(((JDRGroup)object), 
                      latexFonts, ce);
         }
         else if (object instanceof JDRText)
         {
            JDRText t = (JDRText)object;

            UndoableEdit edit = new SetLaTeXFontSize(
               t, latexFonts.getLaTeXCmd(t.getFontSize()));

            ce.addEdit(edit);
            done = true;
         }
      }

      return done;
   }

   public void updateLaTeXFontSize(LaTeXFontBase latexFonts)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      boolean done=false;

      done = updateLaTeXFontSize(paths, latexFonts, ce);

      ce.end();

      if (done) frame_.postEdit(ce);
   }


   public void selectObjectAndScroll(JDRCompleteObject object)
   {
      UndoableEdit edit = new SelectObject(
         object, true, JDRResources.getString("undo.select"));
      frame_.postEdit(edit);
      BBox box = object.getBBox();

      scrollToLocation(box.getMinX(), box.getMinY());
   }

   public void selectAll()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      JDRGroup visible = getVisibleObjects();

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         edit = new SelectObject(
            paths.get(i),true,
            JDRResources.getString("undo.select_all"));
         ce.addEdit(edit);
      }

      ce.end();
      if (edit != null) frame_.postEdit(ce);
   }

   public void deleteSelection()
   {
      int n = paths.size();
      if (n == 0) return;

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      for (int i = paths.size()-1; i >= 0; i--)
      {
         JDRCompleteObject object = paths.get(i);
         if (object.isSelected())
         {
            //paths.remove(i);
            UndoableEdit edit = new RemoveObject(object, i);
            ce.addEdit(edit);
         }
      }

      ce.end();
      frame_.postEdit(ce);
   }

   public JDRGroup getSelection()
   {
      int n = paths.size();

      JDRGroup g;

      if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
      {
         g = new JDRGroup(n);
      }
      else
      {
         g = new JDRGroup();
      }

      for (int i = 0; i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            g.add((JDRCompleteObject)object.clone());
         }
      }

      return g;
   }

   public void copySelection(JDRGroup grp)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      deselectAll();

      JDRGrid grid = frame_.getGrid();

      Point2D offset = grid.getMinorTicDistance();

      BBox box = grp.getBBox();

      if (box.getMinX() < offset.getX())
      {
         offset.setLocation(-offset.getX(), offset.getY());
      }

      if (box.getMinY() < offset.getY())
      {
         offset.setLocation(offset.getX(), -offset.getY());
      }

      for (int i = 0, n = grp.size(); i < n; i++)
      {
         JDRCompleteObject object = (JDRCompleteObject)grp.get(i).clone();
         grp.setSelected(true);
         object.translate(-offset.getX(),-offset.getY());
         UndoableEdit edit = new AddObject(object, 
            JDRResources.getString("undo.paste"));
         ce.addEdit(edit);
      }

      ce.end();
      if (ce.canUndo()) frame_.postEdit(ce);
   }

   public void setImage(JDRGroup image)
   {
      paths = image;
      updateBounds();
      repaint();
   }

   public void updateBounds()
   {
      Graphics2D g = (Graphics2D)getGraphics();

      if (g != null)
      {
         g.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g.scale(norm,norm);

         paths.updateBounds(g);
         g.dispose();
      }
   }

   public int getCurrentTool()
   {
      return frame_.currentTool();
   }

   private JScrollBar getHorizontalScrollBar()
   {
      return frame_.scrollPane.getHorizontalScrollBar();
   }

   private JScrollBar getVerticalScrollBar()
   {
      return frame_.scrollPane.getVerticalScrollBar();
   }

   private void unitScrollLeft()
   {
      JScrollBar scrollBar = getHorizontalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      int increment = scrollBar.getUnitIncrement(-1);

      scrollBar.setValue(scrollBar.getValue()-increment);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   private void unitScrollRight()
   {
      JScrollBar scrollBar = getHorizontalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      int increment = scrollBar.getUnitIncrement(1);

      scrollBar.setValue(scrollBar.getValue()+increment);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   private void unitScrollDown()
   {
      JScrollBar scrollBar = getVerticalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      int increment = scrollBar.getUnitIncrement(1);

      scrollBar.setValue(scrollBar.getValue()+increment);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   private void unitScrollUp()
   {
      JScrollBar scrollBar = getVerticalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      int increment = scrollBar.getUnitIncrement(-1);

      scrollBar.setValue(scrollBar.getValue()-increment);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   public void blockScrollDown()
   {
      JScrollBar scrollBar = getVerticalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      int increment = scrollBar.getBlockIncrement(1);

      scrollBar.setValue(scrollBar.getValue()+increment);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }

   }

   public void blockScrollUp()
   {
      JScrollBar scrollBar = getVerticalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      int increment = scrollBar.getBlockIncrement(-1);

      scrollBar.setValue(scrollBar.getValue()-increment);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   public void blockScrollLeft()
   {
      JScrollBar scrollBar = getHorizontalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      int increment = scrollBar.getBlockIncrement(-1);

      scrollBar.setValue(scrollBar.getValue()-increment);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   public void blockScrollRight()
   {
      JScrollBar scrollBar = getHorizontalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      int increment = scrollBar.getBlockIncrement(1);

      scrollBar.setValue(scrollBar.getValue()+increment);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   public void scrollHomeUp()
   {
      JScrollBar scrollBar = getVerticalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      scrollBar.setValue(0);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   public void scrollHomeLeft()
   {
      JScrollBar scrollBar = getHorizontalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      scrollBar.setValue(0);

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   public void scrollEndDown()
   {
      JScrollBar scrollBar = getVerticalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      scrollBar.setValue(scrollBar.getMaximum());

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   public void scrollEndRight()
   {
      JScrollBar scrollBar = getHorizontalScrollBar();

      if (scrollBar == null || !scrollBar.isVisible()) return;

      scrollBar.setValue(scrollBar.getMaximum());

      Point pt = getMousePosition(true);

      if (pt != null)
      {
         setStatusCoords(pt.x, pt.y);
         frame_.updateRulers(pt.x, pt.y);
      }
   }

   public void scrollToPixel(double px, double py)
   {
      double scale = getMagnification();

      scrollToLocation(px/scale, py/scale);
   }

   public Point2D scrollToLocation(double x, double y)
   {
      JScrollBar hScrollBar = getHorizontalScrollBar();
      JScrollBar vScrollBar = getVerticalScrollBar();

      double paperW = frame_.getPaperWidth();
      double paperH = frame_.getPaperHeight();

      if (x < 0)
      {
         x = 0;
      }
      else if (x > paperW)
      {
         x = paperW;
      }

      if (y < 0)
      {
         y = 0;
      }
      else if (y > paperH)
      {
         y = paperH;
      }

      Point2D location = new Point2D.Double(x, y);

      double ratioX = x/paperW;
      double ratioY = y/paperH;

      int maxWidth = hScrollBar.getMaximum();
      int maxHeight = vScrollBar.getMaximum();

      int hValue = (int)Math.round(maxWidth*ratioX);
      int vValue = (int)Math.round(maxHeight*ratioY);

      int hIncrement = hScrollBar.getBlockIncrement(1);
      int vIncrement = vScrollBar.getBlockIncrement(1);

      hValue -= hIncrement/2;
      vValue -= vIncrement/2;

      if (hValue < 0) hValue=0;
      if (vValue < 0) vValue=0;

      hScrollBar.setValue(hValue);
      vScrollBar.setValue(vValue);

      return location;
   }

   public void stateChanged(ChangeEvent e)
   {
      if (JDR.getOptimize() == JDR.OPTIMIZE_MEMORY)
      {
         setBackgroundImage();
         repaint();
      }
   }

   public boolean getScrollableTracksViewportWidth() {return false;}
   public boolean getScrollableTracksViewportHeight(){return false;}

   public Dimension getPreferredScrollableViewportSize()
   {
      return getPreferredSize();
   }

   public int getUnitIncrement()
   {
      Point2D minor = frame_.getGrid().getMinorTicDistance();
      Point2D major = frame_.getGrid().getMinorTicDistance();

      double inc = (minor.getX() > 0 ? minor.getX() : major.getX());

      return (int)Math.ceil(inc*getMagnification());
   }

   public int getScrollableUnitIncrement(Rectangle visibleRect,
                                         int orientation,
                                         int direction)
   {
      int currentPos = 0;
      if (orientation == SwingConstants.HORIZONTAL)
      {
         currentPos = visibleRect.x;
      }
      else
      {
         currentPos = visibleRect.y;
      }

      int unitIncrement = getUnitIncrement();

      if (direction < 0)
      {
         int newPosition = currentPos - (currentPos/unitIncrement)
                                       * unitIncrement;

         return newPosition==0 ? unitIncrement : newPosition;
      }
      else
      {
         return ((currentPos/unitIncrement)+1)*unitIncrement
              - currentPos;
      }
   }

   public int getScrollableBlockIncrement(Rectangle visibleRect,
                                          int orientation,
                                          int direction)
   {
      int unitIncrement = getUnitIncrement();

      if (orientation == SwingConstants.HORIZONTAL)
         return (unitIncrement >= visibleRect.width) ?
           visibleRect.width :
           visibleRect.width - unitIncrement;
      else
         return (unitIncrement >= visibleRect.height) ?
           visibleRect.height :
           visibleRect.height - unitIncrement;
   }

   public void enableTools()
   {
      int[] number = new int[JDRGroup.MAX_OBJECT_TYPES];
      boolean[] any = new boolean[JDRGroup.MAX_OBJECT_TYPES];
      int size;

      if (paths == null)
      {
         for (int i = 0; i < JDRGroup.MAX_OBJECT_TYPES; i++)
         {
            number[i] = 0;
            any[i] = false;
         }

         size = 0;
      }
      else
      {
         paths.numberAnySelected(number, any);

         size = paths.size();
      }

      frame_.enableTools(size, number, any);
   }

   public JDRPoint getSelectedPoint()
   {
      if (editedPath == null) return null;

      return editedPath.getSelectedControl();
   }

   public void setSelectedPoint(double x, double y)
   {
      JDRPoint selectedPoint = getSelectedPoint();

      if (selectedPoint != null && editedPath != null)
      {
         double oldx = selectedPoint.x;
         double oldy = selectedPoint.y;

         UndoableEdit edit = new MovePoint(editedPath,
                                           editedPath.getSelectedSegment(),
                                           selectedPoint,
                                           x-selectedPoint.x,
                                           y-selectedPoint.y);
         frame_.postEdit(edit);
      }
   }

   public void setText(String str)
   {
      textField.setText(str);
      textField.requestFocusInWindow();
   }

   public Font getSymbolFont()
   {
      return textFieldFont;
   }

   public Font getSymbolButtonFont()
   {
      return symbolButtonFont;
   }

   public void updateTextFieldBounds()
   {
      double factor = getMagnification();

      AffineTransform af = new AffineTransform();
      af.scale(factor, factor);
      //textField.setFont(textFieldFont.deriveFont(af));
      textField.setFont(textFieldFont);
      textField.requestFocusInWindow();
      textField.repaint();
   }

   public void resetTextField()
   {
      Point p;
      JDRPaint currentTextPaint = frame_.getCurrentTextPaint();

      if (currentText == null)
      {
         p = textField.getPosition();
      }
      else
      {
         p = currentText.getStart().getPoint();
         currentText.setTextPaint(currentTextPaint);
      }

      setTextFieldFont(frame_.getCurrentFont());

      resetTextField(currentTextPaint, p);
   }

   public void setTextFieldFont(Font f)
   {
      textFieldFont = f;
      symbolButtonFont = new Font(textFieldFont.getName(),
                                  textFieldFont.getStyle(), 14);
   }

   public void resetTextField(JDRPaint foreground, Point location)
   {
      textField.setPosition(location.x, location.y);
      textField.setTextPaint(foreground);
      updateTextFieldBounds();
      repaint();
      textField.requestFocusInWindow();
   }

   public void startTextAndPostEdit(Point2D currentPos)
   {
      UndoableEdit edit = startText(currentPos);
      if (edit != null) frame_.postEdit(edit);
   }

   public UndoableEdit startText(Point2D currentPos)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      if (currentText != null)
      {
         ce.addEdit(finishText());
      }
      UndoableEdit edit = new ShowTextField(currentPos);
      ce.addEdit(edit);
      ce.end();
      return ce;
   }

   public void abandonText()
   {
      textField.setVisible(false);
      currentText = null;
   }

   public void finishTextAndPostEdit()
   {
      UndoableEdit edit = finishText();
      if (edit != null) frame_.postEdit(edit);
   }

   public UndoableEdit finishText()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      UndoableEdit edit = new HideTextField();

      if (currentText == null)
      {
         return edit;
      }

      if (textField.getText().equals("")) return edit;

      ce.addEdit(edit);

      Graphics2D g2 = (Graphics2D)getGraphics();
      g2.setRenderingHints(frame_.getRenderingHints());
      double norm = JDRUnit.getNormalizingFactor();
      g2.scale(norm,norm);

      try
      {
         currentText.setFont(g2, frame_.getCurrentFontFamily(),
            frame_.getCurrentFontSeries(),
            frame_.getCurrentFontShape(),
            frame_.getCurrentFontSize());
      }
      catch (InvalidFontWeightException we)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("error.invalid_series"), we);
      }
      catch (InvalidFontShapeException spe)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("error.invalid_shape"), spe);
      }
      catch (InvalidFontSizeException sze)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("error.invalid_size"), sze);
      }

      currentText.setText(g2,textField.getText());
      currentText.setLaTeXFont(frame_.getCurrentLaTeXFontFamily(),
                           frame_.getCurrentLaTeXFontSize(),
                           frame_.getCurrentLaTeXFontSeries(),
                           frame_.getCurrentLaTeXFontShape());

      try
      {
         currentText.setHAlign(frame_.getCurrentPGFHAlign());
      }
      catch (InvalidHAlignException e)
      {
         JDRResources.internalError(frame_, e);
      }

      try
      {
         currentText.setVAlign(frame_.getCurrentPGFVAlign());
      }
      catch (InvalidVAlignException e)
      {
         JDRResources.internalError(frame_, e);
      }

      g2.dispose();

      if (JpgfDraw.isAutoEscapeSpCharsEnabled())
      {
         String newText = currentText.getText().replaceAll(
            "\\\\", "\\\\textbackslash ");

         newText = newText.replaceAll("#", "\\\\#");
         newText = newText.replaceAll("%", "\\\\%");
         newText = newText.replaceAll("_", "\\\\_");
         newText = newText.replaceAll("\\$", "\\\\\\$");
         newText = newText.replaceAll("\\&", "\\\\&");
         newText = newText.replaceAll("\\{", "\\\\{");
         newText = newText.replaceAll("\\}", "\\\\}");
         newText = newText.replaceAll("~", "\\\\textasciitilde{}");
         newText = newText.replaceAll("\\^", "\\\\textasciicircum{}");

         currentText.latexText = newText;
      }

      try
      {
         edit = new AddObject(currentText, 
            JDRResources.getString("undo.new_text"));
         ce.addEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.internalError(this,e);
      }
      currentText = null;
      frame_.newImage = false;

      ce.end();
      return ce;
   }

   public void finish()
   {
      if (currentPath != null)
      {
         finishPath();
      }
      else if (textField.isVisible() && currentText != null)
      {
         JDRPoint dp = currentText.getStart();
         Point p = new Point((int)Math.round(dp.x),
                             (int)Math.round(dp.y));
         p.y += textField.getHeight()/getMagnification();
         JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
         ce.addEdit(finishText());
         ce.addEdit(startText(p));
         ce.end();
         frame_.postEdit(ce);
      }
   }

   public void abandonPath()
   {
      if (currentPath == null) return;

      currentPath    = null;
      currentSegment = null;
      anchor         = null;
      repaint();
      frame_.enableEditTools(false);
      frame_.refreshUndoRedo();
   }

   public void finishPath()
   {
      if (currentPath == null) return;

      int tool = frame_.currentTool();

      JDRPaint currentLinePaint = frame_.getCurrentLinePaint();
      JDRPaint currentFillPaint = frame_.getCurrentFillPaint();
      JDRBasicStroke currentStroke = (JDRBasicStroke)frame_.getCurrentStroke().clone();

      frame_.refreshUndoRedo();

      if (anchor != null &&
          (tool == JpgfDraw.ACTION_RECTANGLE ||
           tool == JpgfDraw.ACTION_ELLIPSE))
      {
         currentPath = new JDRPath(currentLinePaint,
                                currentFillPaint,
                                currentStroke);
      }

      BBox box = null;

      if (currentPath != null)
      {
         box = currentPath.getBBox();
      }

      if (tool != JpgfDraw.ACTION_RECTANGLE &&
          tool != JpgfDraw.ACTION_ELLIPSE)
      {
         if (box == null)
         {
            abandonPath();
            return;
         }

         if (box.getWidth() <= 1.002 && box.getHeight() <= 1.002)
         {
            abandonPath();
            return;
         }
      }

      try
      {
         switch (tool)
         {
            case JpgfDraw.ACTION_SELECT :
               break;
            case JpgfDraw.ACTION_CLOSED_LINE :
               currentPath.close(JDRShape.CLOSE_LINE);
            case JpgfDraw.ACTION_OPEN_LINE :
               UndoableEdit edit = new AddObject(currentPath,
                  JDRResources.getString("undo.new_line"));
               frame_.postEdit(edit);
               break;
            case JpgfDraw.ACTION_CLOSED_CURVE :
               currentPath.close(JDRShape.CLOSE_CONT);
            case JpgfDraw.ACTION_OPEN_CURVE :
               edit = new AddObject(currentPath, 
                  JDRResources.getString("undo.new_curve"));
               frame_.postEdit(edit);
               break;
            case JpgfDraw.ACTION_RECTANGLE :
               if (Math.abs(anchor.getX()-mouse.getX()) <= 1.002
                && Math.abs(anchor.getY()-mouse.getY()) <= 1.002)
               {
                  abandonPath();
                  return;
               }

               currentPath = JDRPath.constructRectangle(anchor, mouse);

               ((JDRPath)currentPath).setStyle(currentLinePaint,
                                    currentFillPaint,
                                    currentStroke);
               edit = new AddObject(currentPath, 
                  JDRResources.getString("undo.new_rectangle"));
               frame_.postEdit(edit);
               break;
            case JpgfDraw.ACTION_ELLIPSE :
               double w = Math.abs(mouse.getX()-anchor.getX());
               double h = Math.abs(mouse.getY()-anchor.getY());

               if (w <= 1.002 && h <= 1.002)
               {
                  abandonPath();
                  return;
               }

               currentPath = JDRPath.constructEllipse(anchor, w, h);
               ((JDRPath)currentPath).setStyle(currentLinePaint,
                                    currentFillPaint,
                                    currentStroke);
               edit = new AddObject(currentPath,
                  JDRResources.getString("undo.new_ellipse"));
               frame_.postEdit(edit);
               break;
         }
      }
      catch (Exception e)
      {
         JDRResources.internalError(this,e);
      }

      if (currentPath != null)
      {
         box = currentPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));
      }

      if (currentSegment != null)
      {
         box = currentSegment.getControlBBox();
         repaint(box.getRectangle(getMagnification()));
      }
      currentSegment = null;
      currentPath = null;
      anchor = null;
      frame_.enableEditTools(false);
   }

   public void addObject(JDRCompleteObject object, String undoText)
   {
      frame_.postEdit(new AddObject(object, undoText));
   }

   public void drawPrinterMargins(Graphics2D g)
   {
      if (!frame_.showMargins()) return;

      Rectangle2D printableArea 
         = frame_.getApplication().getPrintableArea();

      if (printableArea == null)
      {
         return;
      }

      GeneralPath area = new GeneralPath(GeneralPath.WIND_EVEN_ODD);

      area.moveTo(0f, 0f);
      area.lineTo(0f, (float)frame_.getPaperHeight());
      area.lineTo((float)frame_.getPaperWidth(),
                  (float)frame_.getPaperHeight());
      area.lineTo((float)frame_.getPaperWidth(), 0f);
      area.closePath();
      area.append(printableArea, false);
      area.closePath();

      g.fill(area);
   }

   public void print()
   {
      // set paper size
      JDRPaper p = frame_.getPaper();

      PrintService service 
         = frame_.getApplication().getPrintService(p);

      if (service != null)
      {
         // obtain printer job
         PrinterJob printJob = PrinterJob.getPrinterJob();

         printJob.setPrintable(this);

         try
         {
            printJob.setPrintService(service);

            frame_.getApplication().doPrintJob(printJob);
         }
         catch (PrinterException pe)
         {
            JDRResources.error(frame_, new String[]
               {JDRResources.getString("error.printing"),
               pe.getMessage()});
         }
         catch (Exception e)
         {
            JDRResources.internalError(frame_,e);
         }
      }
      else
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.printing.no_service"));
      }
   }

   public int print(Graphics g, PageFormat pageFormat, int pageIndex)
   {
      if (pageIndex > 0)
      {
         return Printable.NO_SUCH_PAGE;
      }
      else
      {
         Graphics2D g2 = (Graphics2D)g;

         RepaintManager currentManager 
            = RepaintManager.currentManager(this);
         currentManager.setDoubleBufferingEnabled(false);

         AffineTransform oldAf = g2.getTransform();

         for (int i = 0, n = paths.size(); i < n; i++)
         {
             paths.get(i).draw(g,false,this);
         }

         g2.setTransform(oldAf);

         currentManager.setDoubleBufferingEnabled(true);
         return Printable.PAGE_EXISTS;
      }
   }

   public double getMagnification()
   {
      return frame_.getMagnification()*JDRUnit.getNormalizingFactor();
   }

   public boolean isObjectVisible(JDRCompleteObject object)
   {
      return isObjectVisible(object, displayPage);
   }

   public boolean isObjectVisible(JDRCompleteObject object, int page)
   {
       if (object.flowframe != null)
       {
          switch (page)
          {
             case PAGES_ALL:
                return true;
             case PAGES_EVEN:
                return object.flowframe.isDefinedOnEvenPages();
             case PAGES_ODD:
                return object.flowframe.isDefinedOnOddPages();
             default :
                return object.flowframe.isDefinedOnPage(page);
          }
       }

      return true;
   }

   public int getNumberOfHiddenObjects()
   {
      if (paths == null)
      {
         return 0;
      }

      int numHidden=0;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (!isObjectVisible(object))
         {
            numHidden++;
         }
      }

      return numHidden;
   }

   public void setDisplayPage(int page)
   {
      if (paths == null)
      {
         return;
      }

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (!isObjectVisible(object, page))
         {
            UndoableEdit edit = new SelectObject(
               object,false,
               JDRResources.getString("undo.deselect"));
            ce.addEdit(edit);
         }
      }

      UndoableEdit edit = new DisplayPageEdit(page);
      ce.addEdit(edit);

      ce.end();
      frame_.postEdit(ce);
   }

   public int getDisplayPage()
   {
      return displayPage;
   }

   public RenderingHints getRenderingHints()
   {
      return frame_.getRenderingHints();
   }

   public void paintComponent(Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;

      AffineTransform oldAf = g2.getTransform();
      Stroke oldStroke = g2.getStroke();

      super.paintComponent(g);

      if (paths == null)
      {
         return;
      }

      RenderingHints oldHints = g2.getRenderingHints();

      double scale = getMagnification();

      Rectangle rect = getBounds(null);

      g.getClipBounds(rect);

      Rectangle2D.Double clipBounds 
         = new Rectangle2D.Double(
               rect.getX()/scale, rect.getY()/scale,
               rect.getWidth()/scale, rect.getHeight()/scale);

      if (backgroundImage != null)
      {
         g2.drawImage(backgroundImage,
                     (int)(backgroundImageX*scale),
                     (int)(backgroundImageY*scale), this);
         g2.setRenderingHints(frame_.getRenderingHints());
         g2.scale(scale,scale);
      }
      else
      {
         if (frame_.showGrid())
         {
            drawGrid(g2, scale);
         }

         g2.scale(scale,scale);

         g2.setPaint(marginColor);

         drawPrinterMargins(g2);

         g2.setPaint(typeblockColor);

         if (paths.flowframe != null)
         {
            paths.flowframe.draw(g,
               new BBox(0,0,frame_.getPaperWidth(),
                            frame_.getPaperHeight()));
         }

         g2.setRenderingHints(frame_.getRenderingHints());
      }

      Vector<BBox> selectedBBoxes = null;

      if ((backgroundImage == null)
        ||(backgroundImage != null && editedPath == null))
      {
         int n = paths.size();

         if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
         {
            selectedBBoxes = new Vector<BBox>(n);
         }
         else
         {
            selectedBBoxes = new Vector<BBox>();
         }

         for (int i = 0; i < n; i++)
         {
             JDRCompleteObject object = paths.get(i);

             if (isObjectVisible(object))
             {
                if (!object.isEdited())
                {
                   BBox box = object.getBBox();
                   boolean isShowing;

                   if (JpgfDraw.dragScaleEnabled())
                   {
                      JDRPoint topLeft = box.getTopLeft();
                      JDRPoint bottomRight = box.getBottomRight();

                      BBox extendedBox = box.add(topLeft.getBBox());
                      extendedBox.encompass(bottomRight.getBBox());

                      isShowing = extendedBox.intersects(clipBounds);
                   }
                   else
                   {
                      isShowing = box.intersects(clipBounds);
                   }

                   if (isShowing)
                   {
                      object.draw(g,false,this);

                      if (object.isSelected())
                      {
                         selectedBBoxes.add(box);
                      }
                   }
                }
             }
         }
      }

      g2.setRenderingHints(oldHints);
      g2.setStroke(oldStroke);
      g2.setTransform(oldAf);

      if (selectedBBoxes != null)
      {
         // Draw the bounding boxes for the selected objects

         for (int i = 0, n=selectedBBoxes.size(); i < n; i++)
         {
            BBox box = selectedBBoxes.get(i);

            box.draw(g, scale,
              JpgfDraw.dragScaleEnabled() ? hotspotFlags : 0);
         }

         selectedBBoxes = null;
      }

      if (currentPath != null)
      {
         // Draw the path under construction

         BBox box = currentPath.getControlBBox();

         if (box != null && box.intersects(clipBounds))
         {
            currentPath.drawDraft(g, scale);
         }

         if (currentSegment != null) 
         {
            currentSegment.drawDraft(g,scale,true);
         }
      }
      else if (editedPath != null)
      {
         BBox box = editedPath.getControlBBox();

         if (box.intersects(clipBounds))
         {
            g2.setStroke(oldStroke);
            //g2.setTransform(oldAf);
            editedPath.drawDraft(g, scale);
         }

/*
         if (editedPath.getSelectedSegment() != null)
         {
            ((Graphics2D) g).setPaint(JDRCompleteObject.selectColor);
            g2.setStroke(editedSegmentDash);
            editedPath.getSelectedSegment().draw(g, scale);
         }
*/
      }

      if (scanshape != null)
      {
         scanshape.draw(g2);
      }

      if (dragBBox != null)
      {
         g2.setXORMode(getBackground());

         dragBBox.draw(g2, scale);

         g2.setPaintMode();
      }

      //g2.setTransform(oldAf);
   }

   public void deselectAll()
   {
      if (paths == null)
      {
         return;
      }

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      boolean done=false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         UndoableEdit edit = new SelectObject(
            paths.get(i), false,
            JDRResources.getString("undo.deselect_all"));
         ce.addEdit(edit);
         done=true;
      }

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public boolean deselectAll(JDRCanvasCompoundEdit ce)
   {
      boolean done=false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         UndoableEdit edit = new SelectObject(
            paths.get(i), false,
            JDRResources.getString("undo.deselect_all"));
         ce.addEdit(edit);
         done=true;
      }

      return done;
   }

   public void drawRectangle(Graphics g, Point p1, Point p2)
   {
      int x      = (int)p1.x;
      int y      = (int)p1.y;
      int width  = (int)(p2.x-p1.x);
      int height = (int)(p2.y-p1.y);

      if (p1.x > p2.x)
      {
         width = -width;
         x = p2.x;
      }
      if (p1.y > p2.y)
      {
         height = -height;
         y = p2.y;
      }

      g.drawRect(x, y, width, height);
   }

   public void drawEllipse(Graphics g, Point p1, Point p2)
   {
      int x      = p1.x;
      int y      = p1.y;
      int width  = 2*Math.abs(p2.x-p1.x);
      int height = 2*Math.abs(p2.y-p1.y);

      g.drawOval(x-width/2, y-height/2, width, height);
   }

   public void setStatusCoords(double x, double y)
   {
      JDRGrid grid = frame_.getGrid();

      Point2D point = grid.getFromCartesian(frame_.getPaper(), x, y);

      frame_.setCurrentPosition(grid.formatLocation(point));
   }

   public void mouseMoved(MouseEvent evt)
   {
      Point2D currentPos = getNearestTic(evt.getX(), evt.getY());
      moveTo(currentPos);
   }

   public void mousePressed(MouseEvent evt)
   {
      Point2D currentPos = getNearestTic(evt.getX(),evt.getY());
      mouseDown = true;
      int tool = frame_.currentTool();

      if (checkForPopupTrigger(evt)) return;

      if (evt.isAltDown()) return;

      if (tool == JpgfDraw.ACTION_SELECT)
      {
         if (paths == null) return;

         dragBBox = null;

         if (dragScaleHotspot != BBox.HOTSPOT_NONE)
         {
            cpedit = new JDRCanvasCompoundEdit();

            switch (dragScaleHotspot)
            {
               case BBox.HOTSPOT_S :
               case BBox.HOTSPOT_E :
               case BBox.HOTSPOT_SE :
                  dragScaleAnchor = 
                    dragScaleObject.getBBox().getTopLeft();
               break;
               case BBox.HOTSPOT_NE :
               case BBox.HOTSPOT_NW :
                  dragScaleAnchor = 
                    dragScaleObject.getBBox().getBottomLeft();
               break;
               case BBox.HOTSPOT_SW :
                  dragScaleAnchor =
                     dragScaleObject.getBBox().getCentre();
               break;
            }

         }
         else if (frame_.isPathEdited())
         {
            if (editedPath == null) return;

            if (cpedit==null)
               cpedit = new JDRCanvasCompoundEdit();

            try
            {
               UndoableEdit edit =
                  new SelectControl(currentPos);
               frame_.postEdit(edit);
               repaint(editedPath.getControlBBox().
                  getRectangle(getMagnification()));
               setCursor(Cursor.getPredefinedCursor(
                            Cursor.MOVE_CURSOR));
            }
            catch (NullPointerException e)
            {
               // no control selected, stop editing

/*
               finishEditPath();
               anchor = null;
               setCursor(Cursor.getPredefinedCursor(
                            Cursor.DEFAULT_CURSOR));
*/
            }
         }
         else if (!paths.anySelected(currentPos))
         {
            boolean selected=false;
            JDRCompleteObject thisPath=null;
            selectedIndex = -1;
            for (int i = paths.size()-1; i >= 0; i--)
            {
               thisPath = paths.get(i);
               BBox bbox = thisPath.getBBox();
               if (bbox.contains(currentPos)
                 &&isObjectVisible(thisPath))
               {
                  selected = true;
                  selectedIndex = i;
                  break;
               }
            }

            anchor = (Point2D)currentPos.clone();

            if (selected && !evt.isShiftDown())
            {
               JDRCanvasCompoundEdit ce 
                  = new JDRCanvasCompoundEdit();
               if (!evt.isControlDown()) deselectAll(ce);
               if (!isObjectVisible(thisPath)) return;

               UndoableEdit edit = new SelectObject(
                  thisPath, true);
               ce.addEdit(edit);
               ce.end();
               frame_.postEdit(ce);
               cpedit = new JDRCanvasCompoundEdit();
               setCursor(
                  Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
            }
            else
            {
               dragBBox = new BBox(anchor.getX(), anchor.getY(),
                  anchor.getX(), anchor.getY());
               deselectAll();
               setCursor(
                  Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
            }
         }
         else
         {
            cpedit = new JDRCanvasCompoundEdit();
            setCursor(
               Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
         }
      }
   } 

   public void mouseReleased(MouseEvent evt)
   {
      Point2D currentPos = getNearestTic(evt.getX(),evt.getY());
      mouseDown = false;
      int tool = frame_.currentTool();

      if (evt.isMetaDown() || evt.isAltDown()) return;

      if (tool == JpgfDraw.ACTION_SELECT)
      {
         setToolCursor();

         dragBBox = null;

         if (dragScaleHotspot != BBox.HOTSPOT_NONE)
         {
            dragScaleHotspot = BBox.HOTSPOT_NONE;
            dragScaleObject = null;

            cpedit.end();
            frame_.postEdit(cpedit);
            cpedit = null;
            anchor=null;
         }
         else if (!frame_.isPathEdited())
         {
            if (!paths.anySelected() && anchor != null)
            {
               // end of drag selection
               double minx = (anchor.getX() < mouse.getX() ?
                              anchor.getX() : mouse.getX());
               double miny = (anchor.getY() < mouse.getY() ?
                              anchor.getY() : mouse.getY());
               double maxx = (anchor.getX() > mouse.getX() ?
                              anchor.getX() : mouse.getX());
               double maxy = (anchor.getY() > mouse.getY() ?
                              anchor.getY() : mouse.getY());
               BBox box = new BBox(minx,miny,maxx,maxy);

               JDRGroup grp;

               if (evt.isShiftDown())
               {
                  grp = paths.getAllInside(box);
               }
               else
               {
                  grp = paths.getAllIntersects(box);
               }

               repaint(box.getRectangle(getMagnification()));

               int n=0;
               JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
               for (int i = 0, m = grp.size(); i < m; i++)
               {
                  JDRCompleteObject object = grp.get(i);
                  if (!isObjectVisible(object))
                  {
                     continue;
                  }

                  UndoableEdit edit = new SelectObject(
                     object, true);
                  ce.addEdit(edit);
                  n++;
               }
               ce.end();
               if (n > 0) frame_.postEdit(ce);
               anchor = null;
            }
            else if (cpedit != null)
            {
               // end of move
               cpedit.end();
               frame_.postEdit(cpedit);
               cpedit = null;
            }
         }
         else if (cpedit != null)
         {
            cpedit.end();
            frame_.postEdit(cpedit);
            cpedit = null;
         }
      }
   }

   public void mouseClicked(MouseEvent evt)
   {
      if (frame_.isIoInProgress())
      {
         return;
      }

      Point2D currentPos = getNearestTic(evt.getX(),evt.getY());
      int tool = frame_.currentTool();

      if (evt.isMetaDown() || evt.isAltDown()) return;

      if (frame_.isPathEdited())
      {
         try
         {
            UndoableEdit edit =
               new SelectControl(currentPos);
            frame_.postEdit(edit);
            repaint(editedPath.getControlBBox().
               getRectangle(getMagnification()));
            setCursor(Cursor.getPredefinedCursor(
                         Cursor.MOVE_CURSOR));
            frame_.postEdit(edit);
         }
         catch (NullPointerException e)
         {
               // no control selected, stop editing

               finishEditPath();
               anchor = null;
               setCursor(Cursor.getPredefinedCursor(
                            Cursor.DEFAULT_CURSOR));
         }
      }
      else if (evt.getClickCount() >= 2 && currentPath != null)
      {
         finishPath();
      }
      else if (dragScaleHotspot != BBox.HOTSPOT_NONE)
      {
         anchor.setLocation(currentPos.getX(),
                            currentPos.getY());
      }
      else if (tool == JpgfDraw.ACTION_SELECT)
      {
         if (evt.getClickCount() == 1 && evt.isShiftDown())
         {
            if (paths == null) return;
   
            for (int i = paths.size()-1; i >= 0; i--)
            {
               JDRCompleteObject object = paths.get(i);
               BBox bbox = object.getBBox();
               if (object.isSelected()
                  && bbox.contains(currentPos)
                  && isObjectVisible(object))
               {
                  UndoableEdit edit = new SelectObject(
                     object, false,
                     JDRResources.getString("undo.deselect"));
                  frame_.postEdit(edit);
                  break;
               }
            }
         }
         else if (evt.getClickCount() >= 2)
         {
            if (paths == null) return;

            boolean selected=false;
            JDRCompleteObject thisPath=null;
            int index=-1;
            for (int i = selectedIndex-1; i >= 0; i--)
            {
               thisPath = paths.get(i);
               BBox bbox = thisPath.getBBox();
               if (bbox.contains(currentPos)
                 && isObjectVisible(thisPath))
               {
                  index = i;
                  selected = true;
                  break;
               }
            }
            if (!selected)
            {
               for (int i = paths.size()-1; i > selectedIndex; i--)
               {
                  index = i;
                  thisPath = paths.get(i);
                  BBox bbox = thisPath.getBBox();
                  if (bbox.contains(currentPos)
                    && isObjectVisible(thisPath))
                  {
                     selected = true;
                     break;
                  }
               }
            }
            if (selected)
            {
               if (thisPath.isSelected())
               {
                  // already selected
               }
               else
               {
                  JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
                  boolean done = false;
                  if (!evt.isControlDown())
                  {
                     done = deselectAll(ce);
                  }
                  if (isObjectVisible(thisPath))
                  {
                     UndoableEdit edit = new SelectObject(
                        thisPath, true,
                        JDRResources.getString("undo.select"));
                     ce.addEdit(edit);
                     done = true;
                  }
                  ce.end();
                  if (done) frame_.postEdit(ce);
               }
               repaint();
            }
            anchor = (Point2D)currentPos.clone();
            selectedIndex=index;
         }
      }
      else
      {
         doConstructMouseClick(currentPos);
      }
   }

   public void mouseEntered(MouseEvent evt)
   {
      Point2D currentPos = getNearestTic(evt.getX(), evt.getY());
      mouse.setLocation(currentPos.getX(),currentPos.getY());
      setStatusCoords(mouse.getX(), mouse.getY());
      setToolCursor();
   }

   public void mouseExited(MouseEvent evt)
   {
   }

   public void moveTo(Point2D currentPos)
   {
      int tool = frame_.currentTool();

      double x = currentPos.getX();
      double y = currentPos.getY();

      setStatusCoords(x, y);

      if (anchor != null)
      {
         if ((tool == JpgfDraw.ACTION_OPEN_LINE || 
             tool == JpgfDraw.ACTION_CLOSED_LINE) &&
             currentSegment != null)
         {
            currentSegment.setEnd(currentPos.getX(),currentPos.getY());
            repaint();
         }
         else if ((tool == JpgfDraw.ACTION_OPEN_CURVE ||
                  tool == JpgfDraw.ACTION_CLOSED_CURVE) &&
                  currentSegment != null)
         {
            if (currentSegment instanceof JDRBezier)
            {
               JDRBezier seg;
               JDRPathSegment prev = currentPath.getLastSegment();
               if (currentPath.size() == 0 || 
                   !(prev instanceof JDRBezier))
               {
                  seg = new JDRBezier(anchor, currentPos);
               }
               else
               {
                  seg = JDRBezier.constructBezier((JDRSegment)prev,currentPos);
                  if (currentPath.size() > 1 && prev instanceof JDRBezier)
                  {
                     JDRPathSegment beforePrev = 
                        currentPath.get(currentPath.size()-2);
                     if (beforePrev instanceof JDRBezier)
                     {
                        JDRBezier.makeContinuous((JDRBezier)beforePrev,
                                              (JDRBezier)prev);
                     }
                  }
               }
               currentSegment = seg;
               repaint();
            }
            else
            {
               Graphics g = getGraphics();
               g.setXORMode(getBackground());
               currentSegment.draw(g,true);
               currentSegment.setEnd(currentPos.getX(),currentPos.getY());
               currentSegment.draw(g,true);
               g.dispose();
            }
         }
         else if (tool == JpgfDraw.ACTION_RECTANGLE)
         {
            currentPath = JDRPath.constructRectangle(
               anchor, new Point2D.Double(x,y));
            repaint();
         }
         else if (tool == JpgfDraw.ACTION_ELLIPSE)
         {
            double w = Math.abs(mouse.getX()-anchor.getX());
            double h = Math.abs(mouse.getY()-anchor.getY());
            currentPath = JDRPath.constructEllipse(anchor, w, h);
            repaint();
         }
      }

      dragScaleHotspot=BBox.HOTSPOT_NONE;
      dragScaleObject = null;
      dragScaleIndex = -1;

      if (tool == JpgfDraw.ACTION_SELECT && editedPath == null
         && JpgfDraw.dragScaleEnabled())
      {
         //BBox.hotspotD = getUnitIncrement()/2;

         for (int i = paths.size()-1; i >= 0; i--)
         {
            JDRCompleteObject object = paths.get(i);

            if (object.isSelected())
            {
               dragScaleHotspot = object.getBBox().getHotspot(currentPos);

               if (dragScaleHotspot != BBox.HOTSPOT_NONE)
               {
                  dragScaleObject = object;
                  dragScaleIndex = i;
                  break;
               }
            }
         }

         switch (dragScaleHotspot)
         {
            case BBox.HOTSPOT_NONE :
               setToolCursor();
            break;
            case BBox.HOTSPOT_E :
               setCursor(Cursor.getPredefinedCursor(
                  Cursor.E_RESIZE_CURSOR));
            break;
            case BBox.HOTSPOT_SE :
               setCursor(Cursor.getPredefinedCursor(
                  Cursor.SE_RESIZE_CURSOR));
            break;
            case BBox.HOTSPOT_S :
               setCursor(Cursor.getPredefinedCursor(
                  Cursor.S_RESIZE_CURSOR));
            break;
            case BBox.HOTSPOT_SW :
               setCursor(Cursor.getPredefinedCursor(
                  Cursor.HAND_CURSOR));
            break;
            case BBox.HOTSPOT_W :
               setCursor(Cursor.getPredefinedCursor(
                  Cursor.DEFAULT_CURSOR));
            break;
            case BBox.HOTSPOT_NE :
               setCursor(Cursor.getPredefinedCursor(
                  Cursor.N_RESIZE_CURSOR));
            break;
            case BBox.HOTSPOT_N :
               setCursor(Cursor.getPredefinedCursor(
                  Cursor.DEFAULT_CURSOR));
            break;
            case BBox.HOTSPOT_NW :
               setCursor(Cursor.getPredefinedCursor(
                  Cursor.E_RESIZE_CURSOR));
            break;
         }
      }
      mouse.setLocation(x, y);
   }

   public void mouseDragged(MouseEvent evt)
   {
      Point2D currentPos = getNearestTic(evt.getX(), evt.getY());
      int tool = frame_.currentTool();

      if (evt.isAltDown() || evt.isMetaDown()) return;

      double x = currentPos.getX();
      double y = currentPos.getY();

      scrollRectToVisible(new Rectangle(evt.getX(),evt.getY(),1,1));

      if (tool == JpgfDraw.ACTION_SELECT)
      {
         if (dragScaleObject != null)
         {
            BBox bbox = dragScaleObject.getBBox();
            double minX = dragScaleAnchor.x;
            double minY = dragScaleAnchor.y;

            double dx = x < minX ? mouse.getX()-x : x-mouse.getX();
            double dy = y < minY ? mouse.getY()-y : y-mouse.getY();

            switch (dragScaleHotspot)
            {
               case BBox.HOTSPOT_E :
               // scale horizontally
                  if (dx == 0) break;
                  double width = bbox.getWidth();
                  if (width == 0) return;
                  double newWidth = width + dx;
                  double factor = newWidth/width;
                  if (Math.abs(factor) >= 1e-6)
                  {
                     if ((x < minX && mouse.getX() > minX)
                      || (x > minX && mouse.getX() < minX))
                     {
                        // flipped
                        factor = -factor;
                     }

                     UndoableEdit edit = new ScaleX(dragScaleObject,
                        factor, dragScaleIndex, dragScaleAnchor);
                     cpedit.addEdit(edit);
                  }
                  else
                  {
                     x = mouse.getX();
                  }
               break;
               case BBox.HOTSPOT_S :
               // scale vertically
                  if (dy == 0) break;
                  double height = bbox.getHeight();
                  if (height == 0) return;
                  double newHeight = height + dy;
                  factor = newHeight/height;
                  if (Math.abs(factor) >= 1e-6)
                  {
                     if ((y < minY && mouse.getY() > minY)
                      || (y > minY && mouse.getY() < minY))
                     {
                        // flipped
                        factor = -factor;
                     }

                     UndoableEdit edit = new ScaleY(dragScaleObject,
                        factor, dragScaleIndex, dragScaleAnchor);
                     cpedit.addEdit(edit);
                  }
                  else
                  {
                     y = mouse.getY();
                  }
               break;
               case BBox.HOTSPOT_SE :
               // scale in both directions
                  if (dx == 0 && dy == 0) break;
                  height = bbox.getHeight();
                  width = bbox.getWidth();
                  if (height == 0 || width == 0) return;
                  newHeight = height + dy;
                  newWidth = width + dx;
                  double factorY = newHeight/height;
                  double factorX = newWidth/width;

                  if (Math.abs(factorY) >= 1e-6
                    && Math.abs(factorX) >= 1e-6)
                  {
                     if ((y < minY && mouse.getY() > minY)
                      || (y > minY && mouse.getY() < minY))
                     {
                        // flipped
                        factorY = -factorY;
                     }

                     if ((x < minX && mouse.getX() > minX)
                      || (x > minX && mouse.getX() < minX))
                     {
                        // flipped
                        factorX = -factorX;
                     }

                     UndoableEdit edit = new Scale(dragScaleObject,
                        factorX, factorY, dragScaleIndex,
                        dragScaleAnchor);
                     cpedit.addEdit(edit);
                  }
                  else
                  {
                     x = mouse.getX();
                     y = mouse.getY();
                  }
               break;
               case BBox.HOTSPOT_SW :
               // rotate
               // (ax,ay) is vector from centre to old position
               // (bx,by) is vector from centre to new position
               double ax = mouse.getX()-dragScaleAnchor.x;
               double ay = mouse.getY()-dragScaleAnchor.y;
               double bx = x - dragScaleAnchor.x;
               double by = y - dragScaleAnchor.y;

               // compute angle

               double norma = Math.sqrt(ax*ax+ay*ay);
               double normb = Math.sqrt(bx*bx+by*by);

               if (norma != 0 && normb != 0)
               {
                  double cosA = ax/norma;
                  double cosB = bx/normb;

                  double theta = Math.acos(cosA)-Math.acos(cosB);

                  if (by > 0) theta = -theta;

                  UndoableEdit edit = new Rotate(dragScaleObject,
                     theta, dragScaleIndex, dragScaleAnchor);
                  cpedit.addEdit(edit);
               }
               break; 
               case BBox.HOTSPOT_NW :
               // shear horizontally
               double sx = ((double)(x-mouse.getX()))
                         / ((double)mouse.getY());

               if (sx != 0)
               {
                  UndoableEdit edit = new Shear(dragScaleObject,
                     sx, 0.0, dragScaleIndex, dragScaleAnchor);
                  cpedit.addEdit(edit);
               }
               break;
               case BBox.HOTSPOT_NE :
               // shear vertically
               double sy = ((double)(mouse.getY()-y))
                         / ((double)mouse.getX());

               if (sy != 0)
               {
                  UndoableEdit edit = new Shear(dragScaleObject,
                     0.0, sy, dragScaleIndex, dragScaleAnchor);
                  cpedit.addEdit(edit);
               }
               break;
            }
         }
         else if (frame_.isPathEdited())
         {
            JDRPoint p = getSelectedPoint();

            if (p != null && editedPath != null)
            {
               UndoableEdit edit = new MovePoint(
                  editedPath,
                  editedPath.getSelectedSegment(),
                  p,
                  x-mouse.getX(),
                  y-mouse.getY());

               if (cpedit!=null) cpedit.addEdit(edit);
            }
         }
         else
         {
            if (paths.anySelected())
            {
               dragBBox = null;
               double shift_left = mouse.getX()-x;
               double shift_up   = mouse.getY()-y;

               for (int i = 0, n=paths.size(); i < n; i++)
               {
                  JDRCompleteObject object = paths.get(i);

                  if (object.isSelected())
                  {
                     UndoableEdit edit = new MoveObject(
                        object, shift_left, shift_up);

                     if (cpedit != null) cpedit.addEdit(edit);
                  }
               }
            }
            else if (anchor != null && dragBBox != null)
            {
               double minx = Math.min(anchor.getX(), x);
               double miny = Math.min(anchor.getY(), y);
               double maxx = Math.max(anchor.getX(), x);
               double maxy = Math.max(anchor.getY(), y);

               repaint(dragBBox.getRectangle(getMagnification()));

               dragBBox.reset(minx, miny, maxx, maxy);

               repaint(dragBBox.getRectangle(getMagnification()));
            }
         }
      }

      mouse.setLocation(x, y);

      setStatusCoords(x, y);
   }

   public void gap()
   {
      if (currentSegment != null)
      {
         JDRSegment oldSegment = currentSegment;
         currentSegment = new JDRSegment(oldSegment.getStart(),
                                      oldSegment.getEnd());
       
         repaint();
      }
   }

   public void finishEditPath()
   {
      try
      {
         UndoableEdit edit = new EditPath(editedPath,false);
         frame_.postEdit(edit);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_,
         JDRResources.getString("internal_error.finish_edit_path_failed"),
         e);
      }

      frame_.setEditPathButton(false);
   }

   public void editPath()
   {
      //setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
      setToolCursor();
      JDRCompleteObject object = paths.getSelected();

      if (!(object instanceof JDRShape) || object == null)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.no_path"));
      }
      else
      {
         UndoableEdit edit = new EditPath((JDRShape)object,
                                          !object.isEdited());
         frame_.postEdit(edit);
      }
   }

   public void setToolCursor()
   {
      setToolCursor(frame_.currentTool());
   }

   public void setToolCursor(int tool)
   {
      switch (tool)
      {
         case JpgfDraw.ACTION_SELECT :
            setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
            break;
         case JpgfDraw.ACTION_TEXT :
            setCursor(Cursor.getPredefinedCursor(Cursor.TEXT_CURSOR));
            break;
         default :
            setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
      }
   }

   public void setAction(int tool)
   {
      setToolCursor(tool);
      if (tool != JpgfDraw.ACTION_TEXT) requestFocusInWindow();

      int oldTool = frame_.currentTool();
      if (oldTool == tool) return;

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      if (oldTool == JpgfDraw.ACTION_SELECT)
      {
         if (editedPath != null)
         {
            try
            {
               UndoableEdit edit = new EditPath(editedPath,false);
               ce.addEdit(edit);
            }
            catch (Exception e)
            {
               JDRResources.internalError(frame_,
               "setAction("+tool+")\n"+
               JDRResources.getString("internal_error.finish_edit_path_failed"),
               e);
            }
         }
   
         deselectAll(ce);
      }
      UndoableEdit edit = new SetTool(tool);
      ce.addEdit(edit);

      ce.end();
      frame_.postEdit(ce);
   }

   public void finishEditingPath()
   {
      //setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
      setToolCursor();
      if (editedPath != null)
      {
         editedPath.setEditMode(false);
      }
      editedPath = null;
      frame_.setEditPathButton(false);
   }

   public void rotateSelectedPaths(double angle)
   {
      if (angle == 0.0F) return;

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            edit = new Rotate(object, angle, i);
            ce.addEdit(edit);
         }
      }

      ce.end();
      if (edit != null) frame_.postEdit(ce);
   }

   public void scaleSelectedPaths(double factor)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            edit = new Scale(object, factor, i);
            ce.addEdit(edit);
         }
      }

      ce.end();
      if (edit != null) frame_.postEdit(ce);
   }

   public void scaleXSelectedPaths(double factor)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit=null;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            edit = new ScaleX(object, factor, i);
            ce.addEdit(edit);
         }
      }

      ce.end();
      if (edit != null) frame_.postEdit(ce);
   }

   public void scaleYSelectedPaths(double factor)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            edit = new ScaleY(object, factor, i);
            ce.addEdit(edit);
         }
      }

      ce.end();
      if (edit != null) frame_.postEdit(ce);
   }

   public void shearSelectedPaths(double factorX, double factorY)
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit=null;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            edit = new Shear(object, factorX, factorY, i);
            ce.addEdit(edit);
         }
      }

      ce.end();
      if (edit != null) frame_.postEdit(ce);
   }

   public void moveToFront()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      boolean done=false;

      for (int i = paths.size()-1; i >= 0; i--)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            UndoableEdit edit = new MoveToFront(object);
            ce.addEdit(edit);
            done = true;
         }
      }

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public void moveToBack()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      boolean done=false;

      for (int i = 0, n=paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            UndoableEdit edit = new MoveToBack(object);
            ce.addEdit(edit);
            done = true;
         }
      }

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public void splitText()
   {
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      boolean done=false;

      for (int i = 0, n=paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object.hasTextual())
         {
            try
            {
               UndoableEdit edit = new SplitText(object.getTextual(),i);
               ce.addEdit(edit);
               done = true;
            }
            catch (EmptyGroupException e)
            {
               JDRResources.error(frame_,
                  JDRResources.getString("error.split_failed"));
            }
         }
      }

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public void convertToPath()
   {
      boolean done = false;
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      for (int i = 0, n=paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);
;
         if (object.isSelected())
         {
            try
            {
               if (object instanceof JDRText)
               {
                  UndoableEdit edit
                     = new ConvertTextToPath((JDRText)object, i);
                  ce.addEdit(edit);
                  done = true;
               }
               else if (object instanceof JDRTextPath)
               {
                  UndoableEdit edit
                     = new ConvertTextPathToPath((JDRTextPath)object, i);
                  ce.addEdit(edit);
                  done = true;
               }
               else if (object instanceof JDRPath)
               {
                  UndoableEdit edit 
                     = new ConvertOutlineToPath((JDRPath)object, i);
                  ce.addEdit(edit);
                  done = true;
               }
            }
            catch (EmptyGroupException e)
            {
               JDRResources.error(frame_,
                  JDRResources.getString("error.convert_to_path_failed"));
            }
            catch (Exception e)
            {
               JDRResources.internalError(frame_,
                JDRResources.getString("internal_error.convert_to_path")
                +"\n"+e.getMessage(),
                 e);
            }
         }
      }

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public void convertToFullPath()
   {
      boolean done = false;
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      for (int i = 0, n=paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);
;
         if (object.isSelected())
         {
            try
            {
               if (object instanceof JDRPattern)
               {
                  UndoableEdit edit 
                     = new ConvertToFullPath((JDRPattern)object, i);
                  ce.addEdit(edit);
                  done = true;
               }
            }
            catch (Exception e)
            {
               JDRResources.internalError(frame_, e);
            }
         }
      }

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public void convertToTextPath()
   {
      JDRShape path = null;
      JDRText text = null;
      int pathIndex = -1;
      int textIndex = -1;

      for (int i = 0, n=paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);
;
         if (object.isSelected())
         {
            if (object instanceof JDRShape
           && !(object.hasTextual()))
            {
               path = (JDRShape)object;
               pathIndex = i;
            }
            else if (object instanceof JDRText)
            {
               text = (JDRText)object;
               textIndex = i;
            }

            if (text != null && path != null)
            {
               break;
            }
         }
      }

      if (text == null || path == null)
      {
         JDRResources.internalError(frame_,
          JDRResources.getString(
            "internal_error.convert_to_textpath"));
      }
      else
      {
         UndoableEdit edit = new ConvertToTextPath(path, text,
            pathIndex, textIndex);

         frame_.postEdit(edit);
      }
   }

   public void separate()
   {
      boolean done = false;
      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();

      for (int i = 0, n=paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);
;
         if (object.isSelected())
         {
            if (object instanceof JDRCompoundShape)
            {
               try
               {
                  UndoableEdit edit =
                    new Separate((JDRCompoundShape)object, i);
                  ce.addEdit(edit);

                  done = true;
               }
               catch (InvalidPathException e)
               {
                  JDRResources.internalError(frame_, e);
               }
            }
         }
      }

      ce.end();
      if (done) frame_.postEdit(ce);
   }

   public void mergePaths()
   {
      int total = paths.size();

      Vector<JDRShape> list;

      if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
      {
         list = new Vector<JDRShape>(total);
      }
      else
      {
         list = new Vector<JDRShape>();
      }

      int n = 0;

      for (int i = 0; i < total; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRShape)
         {
            list.add((JDRShape)object);
            n++;
         }
      }

      if (n < 2) return;

      try
      {
         UndoableEdit edit = new MergePaths(list);
         frame_.postEdit(edit);
      }
      catch (EmptyPathException epe)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.empty_path"),
            epe);
      }
      catch (NoPathException npe)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.no_path"),
            npe);
      }
      catch (CanOnlyMergePathsException compe)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.can_only_merge_paths"),
            compe);
      }
      catch (IllFittingPathException ifpe)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.ill_fitting"),
            ifpe);
      }
      catch (Exception e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void xorPaths()
   {
      int total = paths.size();

      Vector<JDRShape> list;

      if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
      {
         list = new Vector<JDRShape>(total);
      }
      else
      {
         list = new Vector<JDRShape>();
      }

      int n = 0;

      for (int i = 0; i < total; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRShape)
         {
            list.add((JDRShape)object);
            n++;
         }
      }

      if (n < 2) return;

      try
      {
         UndoableEdit edit = new XORPaths(list);
         frame_.postEdit(edit);
      }
      catch (EmptyPathException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.empty_path"), e);
      }
      catch (MissingMoveException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.missing_move"), e);
      }
      catch (InvalidPathException e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void pathIntersect()
   {
      int total = paths.size();

      Vector<JDRShape> list;

      if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
      {
         list = new Vector<JDRShape>(total);
      }
      else
      {
         list = new Vector<JDRShape>();
      }

      int n = 0;

      for (int i = 0; i < total; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRShape)
         {
            list.add((JDRShape)object);
            n++;
         }
      }

      if (n < 2) return;

      try
      {
         UndoableEdit edit = new PathIntersect(list);
         frame_.postEdit(edit);
      }
      catch (EmptyPathException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.cant_create_intersect"));
      }
      catch (MissingMoveException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.missing_move"), e);
      }
      catch (InvalidPathException e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void subtractPaths()
   {
      int total = paths.size();

      Vector<JDRShape> list;

      if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
      {
         list = new Vector<JDRShape>(total);
      }
      else
      {
         list = new Vector<JDRShape>();
      }

      int n = 0;

      for (int i = 0; i < total; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRShape)
         {
            list.add((JDRShape)object);
            n++;
         }
      }

      if (n < 2) return;

      try
      {
         UndoableEdit edit = new SubtractPaths(list);
         frame_.postEdit(edit);
      }
      catch (EmptyPathException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.empty_path"), e);
      }
      catch (MissingMoveException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.missing_move"), e);
      }
      catch (InvalidPathException e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void pathUnion()
   {
      int total = paths.size();

      Vector<JDRShape> list;

      if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
      {
         list = new Vector<JDRShape>(total);
      }
      else
      {
         list = new Vector<JDRShape>();
      }

      int n = 0;

      for (int i = 0; i < total; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected() && object instanceof JDRShape)
         {
            list.add((JDRShape)object);
            n++;
         }
      }

      if (n < 2) return;

      try
      {
         UndoableEdit edit = new PathUnion(list);
         frame_.postEdit(edit);
      }
      catch (EmptyPathException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.empty_path"), e);
      }
      catch (MissingMoveException e)
      {
         JDRResources.internalError(frame_,
            JDRResources.getString("internal_error.missing_move"), e);
      }
      catch (InvalidPathException e)
      {
         JDRResources.internalError(frame_, e);
      }
   }

   public void group()
   {
      int total = paths.size();

      Vector<JDRCompleteObject> list;

      if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
      {
         list = new Vector<JDRCompleteObject>(total);
      }
      else
      {
         list = new Vector<JDRCompleteObject>();
      }

      int n = 0;

      for (int i = 0; i < total; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            list.add(object);
            n++;
         }
      }

      if (n < 2) return;

      UndoableEdit edit = new GroupObjects(list);
      frame_.postEdit(edit);
   }

   public void ungroup()
   {
      int response=JOptionPane.NO_OPTION;

      JDRCanvasCompoundEdit ce = new JDRCanvasCompoundEdit();
      UndoableEdit edit = null;

      for (int i = paths.size()-1; i >= 0; i--)
      {
         JDRCompleteObject object = paths.get(i);
         if (object.isSelected() && object instanceof JDRGroup)
         {
            if (object.flowframe != null && response==JOptionPane.NO_OPTION)
            {
               response = JOptionPane.showConfirmDialog(frame_,
                  JDRResources.getString("flowframe.confirm.ungroup"),
                  JDRResources.getString("flowframe.confirm.ungroup.title"),
                  JOptionPane.YES_NO_OPTION,
                  JOptionPane.QUESTION_MESSAGE);

               if (response != JOptionPane.YES_OPTION) return;
            }

            edit = new UngroupObjects(i);
            ce.addEdit(edit);
         }
      }

      if (edit != null)
      {
         ce.end();
         frame_.postEdit(ce);
      }
   }

   // x0 and y0 are in terms of the mouse position relative to the
   // top corner of the canvas, so they need to be divided by the
   // magnification to obtain the grid co-ordinate.
   
   public Point2D getNearestTic(double x0, double y0)
   {
      double mag = getMagnification();
      double x = x0/mag;
      double y = y0/mag;

      if (!frame_.getGridLock()) return new Point2D.Double(x, y);

      return frame_.getGrid().getClosestTic(frame_.getPaper(), x, y);
   }

   private void drawGrid(Graphics2D g, double scale)
   {
      frame_.getGrid().drawGrid(g, frame_.getPaper(), scale);
   }

   public void setBackgroundImage()
   {
      setBackgroundImage(false);
   }

   public void setBackgroundImage(boolean forceUpdate)
   {
      if (paths == null)
      {
         return;
      }

      if ((frame_.showGrid()
           || paths.flowframe != null
           || editedPath != null))
      {
         double scale = getMagnification();
         double originX = 0;
         double originY = 0;

         if (JDR.getOptimize() == JDR.OPTIMIZE_SPEED)
         {
            try
            {
               backgroundImage = 
                  new BufferedImage(
                         (int)(frame_.getPaperWidth()*scale),
                         (int)(frame_.getPaperHeight()*scale),
                         BufferedImage.TYPE_INT_ARGB);
            }
            catch (OutOfMemoryError e)
            {
               backgroundImage = null;
               return;
            }
         }
         else
         {
            Rectangle visibleRect = getVisibleRect();

            if (visibleRect.getWidth() == 0 ||
                visibleRect.getHeight() == 0)
            {
               backgroundImage = null;
               return;
            }

            if (backgroundImage != null && !forceUpdate)
            {
               Rectangle r = 
                  new Rectangle((int)(backgroundImageX/scale),
                                (int)(backgroundImageY/scale),
                                backgroundImage.getWidth(),
                                backgroundImage.getHeight());

               if (r.contains(visibleRect))
               {
                  return;
               }
            }

            double w = visibleRect.getWidth()/scale;
            double h = visibleRect.getHeight()/scale;

            backgroundImageX = visibleRect.getX()/scale;
            backgroundImageY = visibleRect.getY()/scale;

            if (backgroundImageX - w < 0)
            {
               if (backgroundImageX + w <= frame_.getPaperWidth())
               {
                  w = 2*w;
               }
            }
            else
            {
               backgroundImageX -= w;

               if (backgroundImageX + w > frame_.getPaperWidth())
               {
                  w = 2*w;
               }
               else
               {
                  w = 2*w;
               }
            }

            if (backgroundImageY - h < 0)
            {
               if (backgroundImageY + h <= frame_.getPaperHeight())
               {
                  h = 2*h;
               }
            }
            else
            {
               backgroundImageY -= h;

               if (backgroundImageY + h > frame_.getPaperHeight())
               {
                  h = 2*h;
               }
               else
               {
                  h = 2*h;
               }
            }

            int buffImageW = (int)Math.ceil(w*scale);
            int buffImageH = (int)Math.ceil(h*scale);

            try
            {
               if (backgroundImage == null ||
                   (backgroundImage.getWidth() != buffImageW &&
                   backgroundImage.getHeight() != buffImageH))
               {
                  backgroundImage = 
                     new BufferedImage(
                            buffImageW,
                            buffImageH,
                            BufferedImage.TYPE_INT_ARGB);
               }
               else
               {
                  for (int i=0, n=backgroundImage.getWidth();i < n;i++)
                  {
                     for (int j=0,m=backgroundImage.getHeight();j < m;j++)
                     {
                        backgroundImage.setRGB(i, j, 0);
                     }
                  }
               }
            }
            catch (OutOfMemoryError e)
            {
               backgroundImage = null;
               return;
            }

            originX = backgroundImageX/scale;
            originY = backgroundImageY/scale;
         }

         Graphics2D g = backgroundImage.createGraphics();

         RenderingHints renderHints = 
            new RenderingHints(RenderingHints.KEY_ANTIALIASING,
                               RenderingHints.VALUE_ANTIALIAS_OFF);

         renderHints.add(new RenderingHints(
                           RenderingHints.KEY_RENDERING,
                           RenderingHints.VALUE_RENDER_SPEED));

         g.setRenderingHints(renderHints);
         BasicStroke stroke = new BasicStroke(1);
         g.setStroke(stroke);

         g.translate(-originX,-originY);

         if (frame_.showGrid())
         {
            drawGrid(g, scale);
         }

         g.scale(scale,scale);

         g.setPaint(marginColor);

         drawPrinterMargins(g);

         g.setPaint(typeblockColor);

         if (paths.flowframe != null)
         {
            paths.flowframe.draw(g,
               new BBox(0,0,frame_.getPaperWidth(),
                        frame_.getPaperHeight()));
         }

         if (editedPath != null)
         {
            g.setRenderingHints(frame_.getRenderingHints());

            JDRCompleteObject object;
            int n = paths.size();

            for (int i = 0; i < n; i++)
            {
                object = paths.get(i);

                if (isObjectVisible(object))
                {
                   if (!object.isEdited())
                   {
                      object.draw(g,false,this);
                   }
                }
            }
         }

         g.dispose();
      }
      else
      {
         backgroundImage = null;
      }
   }

   public void save(String filename, boolean useThread)
   {
      if (filename.toLowerCase().endsWith(".ajr"))
      {
         saveAJR(filename, AJR.currentVersion, useThread);
      }
      else
      {
         save(filename, JDR.currentVersion, useThread);
      }
   }

   public void save(String filename, float jdrversion, 
     boolean useThread)
   {
      if (currentText != null) finishTextAndPostEdit();

      if (jdrversion < JDR.currentVersion)
      {
         if (JOptionPane.showConfirmDialog(frame_,
                JDRResources.getStringWithValue("warning.save.jdr",
                   ""+jdrversion),
                JDRResources.getString("warning.title"),
                JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION)
         {
            return;
         }
      }

      SaveJdr sj;

      if (frame_.saveJDRsettings())
      {
         sj = new SaveJdr(frame_, filename,
              jdrversion, paths, frame_.settings);
      }
      else
      {
         sj = new SaveJdr(frame_, filename,
            jdrversion, paths);
      }

      if (useThread)
      {
         Thread thread = new Thread(sj);

         thread.start();

         thread = null;
      }
      else
      {
         sj.run();
      }
   }

   public void saveAJR(String filename, float ajrversion,
      boolean useThread)
   {
      if (currentText != null) finishTextAndPostEdit();

      if (ajrversion < AJR.currentVersion)
      {
         if (JOptionPane.showConfirmDialog(frame_,
             JDRResources.getStringWithValue("warning.save.ajr",
                ""+ajrversion),
             JDRResources.getString("warning.title"),
             JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION)
         {
            return;
         }
      }

      SaveAjr sa;

      if (frame_.saveJDRsettings())
      {
         sa = new SaveAjr(frame_, filename,
            ajrversion, paths, frame_.settings);
      }
      else
      {
         sa = new SaveAjr(frame_, filename,
            ajrversion, paths);
      }

      if (useThread)
      {
         Thread thread = new Thread(sa);

         thread.start();

         thread = null;
      }
      else
      {
         sa.run();
      }
   }

   public void savePGF(String filename)
   {
      if (currentText != null) finishTextAndPostEdit();

      Thread thread = new Thread(new SavePgf(frame_, filename, paths));

      thread.start();

      thread = null;
   }

   public void savePGFDoc(String filename)
   {
      if (currentText != null) finishTextAndPostEdit();

      Thread thread = new Thread(new SavePgfDoc(frame_, filename, paths));

      thread.start();

      thread = null;
   }

   public void saveFlowFrame(String filename)
   {
      if (currentText != null) finishTextAndPostEdit();

      Thread thread = new Thread(new SaveFlf(frame_, filename, paths));

      thread.start();

      thread = null;
   }

   public void savePNG(String filename)
   {
      if (currentText != null) finishTextAndPostEdit();

      Thread thread = new Thread(new SavePng(frame_, filename, paths));

      thread.start();

      thread = null;
   }

   public void saveEPS(String filename)
   {
      if (currentText != null) finishTextAndPostEdit();

      Thread thread = new Thread(new SaveEps(frame_, filename, paths));

      thread.start();

      thread = null;
   }

   public void saveSVG(String filename)
   {
      if (currentText != null) finishTextAndPostEdit();

      Thread thread = new Thread(new SaveSvg(frame_, filename, paths));

      thread.start();

      thread = null;
   }

   public void updateTextAreaBounds()
   {
      Graphics2D g = (Graphics2D)getGraphics();

      if (g != null)
      {
         g.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g.scale(norm,norm);

         paths.updateBounds(g);
         g.dispose();
      }
   }

   public void load(String filename)
   {
      if (filename.endsWith(".jdr"))
      {
         loadJDR(filename);
      }
      else if (filename.endsWith(".ajr"))
      {
         loadAJR(filename);
      }
      else
      {
         BufferedReader in;

         try
         {
            in = new BufferedReader(new FileReader(filename));
         }
         catch (IOException e)
         {
            JDRResources.error(frame_, new String[]
               {JDRResources.getStringWithValue("error.io.open",
                  filename),
               e.getMessage()});

            return;
         }

         boolean isAJR = false;

         try
         {
            String string = AJR.readString(in, 3);
            isAJR = string.equals("AJR");
         }
         catch (Exception e)
         {
            JDRResources.error(frame_, e);
         }

         try
         {
            in.close();
         }
         catch (IOException e)
         {
            JDRResources.error(frame_, e);
         }

         if (isAJR)
         {
            loadAJR(filename);
         }
         else
         {
            loadJDR(filename);
         }
      }
   }

   public void loadJDR(String filename)
   {
      Thread thread = new Thread(new LoadJdr(frame_, filename));

      thread.start();

      thread = null;
/*
      DataInputStream in;

      try
      {
         in = new DataInputStream(new FileInputStream(filename));
      }
      catch (IOException e)
      {
         JDRResources.error(frame_, new String[]
            {JDRResources.getStringWithValue("error.io.open",
                filename),
            e.getMessage()});

         return false;
      }
      catch (Exception e)
      {
         JDRResources.error(frame_, e);
         return false;
      }

      boolean success = false;
      try
      {
         CanvasSettings s = (CanvasSettings)frame_.settings.clone();

         paths = JDR.load(in, s);
         frame_.applySettingsIfRequired(s);

         float versionNum = JDR.getLastLoadedVersion();

         if (frame_.warnOnOldJdr())
         {
            if (versionNum < JDR.currentVersion)
            {
               JDRResources.warning(frame_,
                  JDRResources.getStringWithValue("warning.load.jdr",
                     ""+versionNum));
            }
         }

         if (paths.anyDraftBitmaps())
         {
            JDRResources.warning(frame_,
               JDRResources.getString("warning.draft_bitmaps"));
         }

         frame_.setFilename(filename);
         frame_.newImage = false;
         repaint();
         frame_.addRecentFile(filename);
         success = true;
      }
      catch (IOException e)
      {
         JDRResources.error(frame_, new String[]
         {JDRResources.getString("error.io.load"),e.getMessage()});
      }
      catch (InvalidMarkerTypeException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_arrow_type")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidPaintIDException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.id.colour")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidCompositeMarkerException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_arrow_composite"), e);
      }
      catch (InvalidPenWidthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_penwidth")
            +": "+e.getInvalidWidth(), e);
      }
      catch (InvalidCapStyleException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_cap")
            +": "+e.getInvalidStyle(), e);
      }
      catch (InvalidJoinStyleException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_join")
            +": "+e.getInvalidStyle(), e);
      }
      catch (InvalidWindingRuleException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_winding_rule")
            +": "+e.getInvalidStyle(), e);
      }
      catch (InvalidRedValueException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_red")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidGreenValueException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_green")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidBlueValueException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_blue")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidAlphaValueException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_alpha")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidCyanValueException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_cyan")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidMagentaValueException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_magenta")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidYellowValueException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_yellow")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidBlackValueException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_black")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidStartColourIDException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.id.startcolour")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidEndColourIDException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.id.endcolour")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidGradientDirectionException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_direction")
            +": "+e.getInvalidValue(), e);
      }
      catch (InvalidOpenCloseIDException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.id.open_close")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidSegmentIDException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.id.segment")
            +": "+e.getInvalidType(), e);
      }
      catch (EmptyPathException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("internal_error.empty_path"), e);
      }
      catch (InvalidVAlignException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_latex_valign")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidHAlignException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_latex_halign")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidTextLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_text_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidFontFamilyLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_family_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidLatexFontFamilyLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_latex_family_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidLatexFontSeriesLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_latex_series_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidLatexFontShapeLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_latex_shape_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidLatexFontSizeLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_latex_size_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidDescriptionLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_description_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidFilenameLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_filename_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidLatexFilenameLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_latexfilename_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidLatexImagCmdLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_latex_imagecmd_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidObjectIDException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.id.object")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidFrameTypeException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_frame_type")
            +": "+e.getInvalidID(), e);
      }
      catch (InvalidIdlLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_idl_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidPageListLengthException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.invalid_pagelist_length")
            +": "+e.getInvalidLength(), e);
      }
      catch (InvalidShapeIDException e)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.id.shape")
            +": "+e.getInvalidID(), e);
      }
      catch (Exception e)
      {
         JDRResources.error(frame_,e);
      }

      if (success)
      {
         Graphics2D g = (Graphics2D)getGraphics();

         if (g != null)
         {
            g.setRenderingHints(frame_.getRenderingHints());
            double norm = JDRUnit.getNormalizingFactor();
            g.scale(norm,norm);

            paths.updateBounds(g);
            g.dispose();
         }
      }

      try
      {
         in.close();
      }
      catch (IOException e)
      {
         JDRResources.error(frame_, new String[]
         {JDRResources.getString("error.io.close"),
         e.getMessage()});
      }
      catch (Exception e)
      {
         JDRResources.error(frame_,e);
      }

      return success;
*/
   }

   public void loadAJR(String filename)
   {
      Thread thread = new Thread(new LoadAjr(frame_, filename));

      thread.start();

      thread = null;
   }

   public boolean discard()
   {
      boolean done = true;

      if (currentPath != null)
      {
         JDRResources.error(frame_,
            JDRResources.getString("error.finish_or_discard"));
         done = false;
      }
      else if (frame_.isSaved())
      {
         paths = new JDRGroup();
         String filename = JDRResources.getString("label.untitled");
         count++;
         if (count > 1) filename += count;
         frame_.setFilename(filename);
         frame_.newImage = true;
         if (editedPath != null) finishEditPath();
      }
      else
      {
         boolean old = frame_.newImage;
         frame_.newImage = false;

         DiscardDialogBox db = frame_.getDiscardDialogBox();

         db.initialise();

         done = frame_.newImage;
         frame_.newImage = old;

         if (editedPath != null && done) finishEditPath();
      }

      return done;
   }

   public void repaintObject(JDRCompleteObject object)
   {
      JDRCompleteObject parent = object.getParent();

      if (parent == null || parent == paths)
      {
         repaint(object.getExtent().getRectangle(getMagnification()));
      }
      else if (parent == object)
      {
         JDRResources.internalError(frame_, 
            "Something has gone wrong.\n"
           +"Object can't be its own parent!");
      }
      else
      {
         repaintObject(parent);
      }
   }

   class JDRCanvasCompoundEdit extends CompoundEdit
   {
      public JDRCanvasCompoundEdit()
      {
         super();
      }

      public void undo () throws CannotUndoException
      {
         Cursor oldCursor = getCursor();
         setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
         super.undo();
         setCursor(oldCursor);
      }

      public void redo () throws CannotRedoException
      {
         Cursor oldCursor = getCursor();
         setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
         super.redo();
         setCursor(oldCursor);
      }
   }

   class SetTool extends AbstractUndoableEdit
   {
      private int _newTool, _oldTool;

      public SetTool(int tool)
      {
         _oldTool = frame_.currentTool();
         _newTool = tool;

         setTool(_oldTool, _newTool);
      }

      public void undo() throws CannotUndoException
      {
         if (_newTool == JpgfDraw.ACTION_TEXT && currentText != null)
         {
            abandonText();
         }

         setTool(_newTool, _oldTool);
      }

      public void redo() throws CannotRedoException
      {
         if (_oldTool == JpgfDraw.ACTION_TEXT && currentText != null)
         {
            abandonText();
         }

         setTool(_oldTool, _newTool);
      }

      public void setTool(int oldTool, int newTool)
      {
         setToolCursor(newTool);
   
         if (oldTool == newTool) return;
   
         dragScaleHotspot=BBox.HOTSPOT_NONE;
         dragScaleIndex = -1;
         dragScaleObject = null;
         dragScaleAnchor=null;
   
         if (currentPath != null)
         {
            if (oldTool == JpgfDraw.ACTION_RECTANGLE || 
                oldTool == JpgfDraw.ACTION_ELLIPSE)
            {
               abandonPath();
            }
            else
            {
               finishPath();
            }
         }
         else if (oldTool == JpgfDraw.ACTION_TEXT
               && currentText != null)
         {
            finishTextAndPostEdit();
         }
   
         anchor = null;
   
         if (newTool != JpgfDraw.ACTION_SELECT)
         {
            enableTools();
         }
   
         frame_.setTool(newTool);
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.set_tool");
      }
   }

   class SetTypeblock extends AbstractUndoableEdit
   {
      private FlowFrame oldTypeblock, newTypeblock;
      private String string=JDRResources.getString("undo.set_typeblock");

      public SetTypeblock(double left, double right,
                          double top, double bottom)
      {
         oldTypeblock = paths.flowframe;

         newTypeblock = new FlowFrame(FlowFrame.TYPEBLOCK);

         newTypeblock.left = left;
         newTypeblock.right = right;
         newTypeblock.top = top;
         newTypeblock.bottom = bottom;

         paths.flowframe = newTypeblock;

         frame_.markAsModified();
         setBackgroundImage();
         repaint();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.flowframe = oldTypeblock;
         frame_.markAsModified();
         setBackgroundImage();
         repaint();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.flowframe = newTypeblock;
         frame_.markAsModified();
         setBackgroundImage();
         repaint();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string;
      }
   }

   class SetFlowFrame extends AbstractUndoableEdit
   {
      private FlowFrame oldFrame, newFrame;
      private JDRCompleteObject object_;
      private String string=JDRResources.getString("undo.set_frame");

      public SetFlowFrame(JDRCompleteObject object, FlowFrame f,
                          String presentation)
      {
         string = presentation;
         object_ = object;
         oldFrame = object_.flowframe;
         newFrame = f;

         object_.flowframe = newFrame;
         frame_.markAsModified();
         BBox box = object_.getExtent();
         frame_.updateTitle();
         repaint(box.getRectangle(getMagnification()));
      }

      public SetFlowFrame(JDRCompleteObject object, FlowFrame f)
      {
         this(object, f, JDRResources.getString("undo.set_frame"));
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         object_.flowframe = oldFrame;
         frame_.markAsModified();
         BBox box = object_.getExtent();
         frame_.updateTitle();
         repaint(box.getRectangle(getMagnification()));
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         object_.flowframe = newFrame;
         frame_.markAsModified();
         BBox box = object_.getExtent();
         frame_.updateTitle();
         repaint(box.getRectangle(getMagnification()));
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string;
      }
   }

   class AddObject extends AbstractUndoableEdit
   {
      private int index_;
      private JDRCompleteObject object_;
      private String string=JDRResources.getString("undo.new_object");

      public AddObject(JDRCompleteObject newObject)
      {
         object_ = newObject;
         paths.add(newObject);
         index_ = paths.size()-1;
         BBox box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public AddObject(JDRCompleteObject newObject, String presentationString)
      {
         this(newObject);
         string = presentationString;
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(index_);
         BBox box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         object_.reset();
         paths.add(index_, object_);
         BBox box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string;
      }
   }

   class RemoveObject extends AbstractUndoableEdit
   {
      private int index_;
      private JDRCompleteObject object_;

      public RemoveObject(JDRCompleteObject object, int i)
      {
         object_ = object;
         index_ = i;
         paths.remove(index_);
         repaint(object_.getExtent().getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.add(index_, object_);
         repaint(object_.getExtent().getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.remove(index_);
         repaint(object_.getExtent().getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.delete");
      }
   }

   class SplitText extends AbstractUndoableEdit
   {
      private int index_;
      private JDRTextual text_;
      private JDRCompleteObject object_;
      private Rectangle repaintExtent;

      public SplitText(JDRTextual text, int i)
      throws EmptyGroupException
      {
         text_  = text;
         index_ = i;
         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         JDRGroup group_ = text_.split(g2);
         if (group_.size() == 1)
         {
            object_ = group_.get(0);
         }
         else
         {
            object_ = group_;
         }
         g2.dispose();

         if (group_.size() == 0)
         {
            throw new EmptyGroupException();
         }

         paths.set(index_, object_);
         BBox oldBox = text_.getExtent();
         BBox newBox = object_.getExtent();

         repaintExtent
            = oldBox.add(newBox).getRectangle(getMagnification());
         repaint(repaintExtent);
         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, (JDRCompleteObject)text_);
         repaint(repaintExtent);
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, object_);
         repaint(repaintExtent);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.splittext");
      }
   }

   class ConvertTextToPath extends AbstractUndoableEdit
   {
      private int index_;
      private JDRText text_;
      private JDRCompleteObject object_;

      public ConvertTextToPath(JDRText text, int i)
         throws MissingMoveException,EmptyGroupException
      {
         text_  = text;
         index_ = i;
         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
            JDRGroup group_ = text_.convertToPath(g2);
            if (group_.size() == 1)
            {
               object_ = group_.get(0);
            }
            else
            {
               object_ = group_;
            }
         }
         catch (MissingMoveException e)
         {
            g2.dispose();
            throw e;
         }
         catch (EmptyGroupException e)
         {
            g2.dispose();
            throw e;
         }
         g2.dispose();
         paths.set(index_, object_);
         BBox oldBox = text_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, text_);
         BBox oldBox = text_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, object_);
         BBox oldBox = text_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_path");
      }
   }

   class ConvertOutlineToPath extends AbstractUndoableEdit
   {
      private int index_;
      private JDRShape oldObject_, object_;

      public ConvertOutlineToPath(JDRShape path, int i)
         throws InvalidPathException
      {
         oldObject_ = path;
         index_ = i;

         object_ = path.outlineToPath();
         paths.set(index_, object_);
         object_.setSelected(true);

         BBox oldBox = oldObject_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, oldObject_);
         BBox oldBox = oldObject_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, object_);
         BBox oldBox = oldObject_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_path");
      }
   }

   class ConvertToFullPath extends AbstractUndoableEdit
   {
      private int index_;
      private JDRCompleteObject oldObject_, object_;

      public ConvertToFullPath(JDRPattern path, int i)
        throws EmptyPathException,IllFittingPathException
      {
         oldObject_ = (JDRCompleteObject)path;
         index_ = i;

         object_ = path.getFullObject();
         paths.set(index_, object_);
         object_.setSelected(true);

         BBox oldBox = oldObject_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, oldObject_);
         BBox oldBox = oldObject_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, object_);
         BBox oldBox = oldObject_.getExtent();
         BBox newBox = object_.getExtent();
         repaint(oldBox.add(newBox).getRectangle(getMagnification()));
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_path");
      }
   }

   class ConvertTextPathToPath extends AbstractUndoableEdit
   {
      private int index_;
      private JDRTextPath textPath_;
      private JDRCompleteObject object_;
      private Rectangle dirtyRegion;

      public ConvertTextPathToPath(JDRTextPath textPath, int i)
         throws MissingMoveException,EmptyGroupException
      {
         textPath_  = textPath;
         index_ = i;
         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         JDRGroup group = textPath.split(g2);

         try
         {
            for (int j = 0; j < group.size(); j++)
            {
               JDRGroup grp 
                  = ((JDRText)group.get(j)).convertToPath(g2);
               group.set(j, grp.get(0));
            }

            if (group.size() == 1)
            {
               object_ = group.get(0);
            }
            else
            {
               object_ = group;
            }
         }
         catch (MissingMoveException e)
         {
            g2.dispose();
            throw e;
         }
         catch (EmptyGroupException e)
         {
            g2.dispose();
            throw e;
         }

         g2.dispose();
         paths.set(index_, object_);
         BBox oldBox = textPath_.getExtent();
         BBox newBox = object_.getExtent();
         dirtyRegion 
            = oldBox.add(newBox).getRectangle(getMagnification());

         repaint(dirtyRegion);
         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, textPath_);

         repaint(dirtyRegion);
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, object_);

         repaint(dirtyRegion);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_path");
      }
   }

   class RemovePattern extends AbstractUndoableEdit
   {
      private int index_;
      private JDRShape shape;
      private JDRPattern pattern;

      private BBox oldBox, newBox;

      public  RemovePattern(JDRPattern path, int index)
      {
         index_ = index;

         pattern = path;
         shape = pattern.getUnderlyingShape();

         paths.set(index_, shape);

         oldBox = shape.getExtent();
         newBox = ((JDRCompleteObject)pattern).getExtent();

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         paths.set(index_, (JDRCompleteObject)pattern);

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         paths.set(index_, shape);

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.pattern.remove");
      }
   }

   class ConvertToPattern extends AbstractUndoableEdit
   {
      private int index_;
      private JDRShape oldShape;
      private JDRPattern newShape;

      private BBox oldBox, newBox;

      public ConvertToPattern(JDRShape path, JDRPattern pattern, int index)
      throws InvalidReplicaException
      {
         index_ = index;
         oldShape = path;

         BBox box = oldShape.getBBox();

         newShape = (JDRPattern)pattern.clone();

         newShape.setPatternAnchor(box.getMidX(), box.getMidY());

         newShape.setUnderlyingShape(oldShape);

         newShape.setDefaultPatternAdjust();

         paths.set(index_, (JDRCompleteObject)newShape);

         oldBox = oldShape.getExtent();
         newBox = newShape.getExtent();

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         paths.set(index_, oldShape);

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         paths.set(index_, (JDRCompleteObject)newShape);

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_pattern");
      }
   }

   class UpdatePattern extends AbstractUndoableEdit
   {
      private JDRCompoundShape newShape, oldShape;

      private int index_;

      private BBox box;

      public UpdatePattern(int index, JDRPattern pattern)
      throws InvalidReplicaException
      {
         oldShape = (JDRCompoundShape)paths.get(index);

         if (!oldShape.hasPattern())
         {
            throw new ClassCastException("Not a pattern");
         }

         if (oldShape instanceof JDRPattern)
         {
            JDRPattern oldPattern = (JDRPattern)oldShape;

            pattern.setUnderlyingShape(oldPattern.getUnderlyingShape());
            pattern.setPatternAnchor(oldPattern.getPatternAnchor());

            if (oldPattern.getPatternAdjust() != null
             && pattern.getClass() == oldShape.getClass())
            {
               pattern.setPatternAdjust(oldPattern.getPatternAdjust());
            }
            else
            {
               pattern.setDefaultPatternAdjust();
            }

            newShape = pattern;
         }
         else
         {
            newShape = (JDRCompoundShape)oldShape.clone();
            setUnderPattern(newShape, pattern);
         }

         index_ = index;

         box = oldShape.getExtent();

         paths.set(index_, (JDRCompleteObject)newShape);

         box.encompass(newShape.getExtent());

         repaint(box.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      private void setUnderPattern(JDRCompoundShape shape,
         JDRPattern pattern)
      {
         JDRShape underlyingShape = shape.getUnderlyingShape();

         if (underlyingShape instanceof JDRPattern)
         {
            JDRPattern oldPattern = (JDRPattern)underlyingShape;

            pattern.setUnderlyingShape(oldPattern.getUnderlyingShape());
            pattern.setPatternAnchor(oldPattern.getPatternAnchor());

            if (oldPattern.getPatternAdjust() != null
             && pattern.getClass() == oldPattern.getClass())
            {
               pattern.setPatternAdjust(oldPattern.getPatternAdjust());
            }
            else
            {
               pattern.setDefaultPatternAdjust();
            }

            shape.setUnderlyingShape(pattern);

            return;
         }

         setUnderPattern((JDRCompoundShape)underlyingShape, pattern);
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         paths.set(index_, oldShape);

         repaint(box.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         paths.set(index_, newShape);

         repaint(box.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.edit_pattern");
      }
   }

   class ConvertToTextPath extends AbstractUndoableEdit
   {
      private int lowerIndex, higherIndex;
      private JDRCompleteObject lowerObject;
      private JDRCompleteObject higherObject;
      private JDRTextPath textPath;
      private JDRStroke oldStroke, newStroke;
      private JDRShape oldPath;

      private BBox oldBox, newBox;

      public ConvertToTextPath(JDRShape path, JDRText text,
         int pathIndex, int textIndex)
      {
         if (pathIndex < textIndex)
         {
            lowerIndex = pathIndex;
            lowerObject = path;
            higherIndex = textIndex;
            higherObject = text;
         }
         else
         {
            lowerIndex = textIndex;
            lowerObject = text;
            higherIndex = pathIndex;
            higherObject = path;
         }

         oldStroke = path.getStroke();
         oldPath = path;

         oldBox = lowerObject.getExtent().add(higherObject.getExtent());

         textPath = new JDRTextPath(path, text);
         textPath.setSelected(true);

         newStroke = textPath.getStroke();

         paths.remove(higherIndex);
         paths.set(lowerIndex, textPath);

         newBox = textPath.getExtent();

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         oldPath.setStroke(oldStroke);

         paths.add(higherIndex, higherObject);
         paths.set(lowerIndex, lowerObject);

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         textPath.setStroke(newStroke);

         paths.remove(higherIndex);
         paths.set(lowerIndex, textPath);

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_textpath");
      }
   }

   class Separate extends AbstractUndoableEdit
   {
      private int index_;
      private JDRGroup group;
      private JDRCompoundShape original;

      private BBox oldBox, newBox;

      public Separate(JDRCompoundShape shape, int index)
        throws InvalidPathException
      {
         original = shape;
         index_ = index;

         Graphics2D g = (Graphics2D)getGraphics();
         group = shape.separate(g);
         group.setSelected(true);
         g.dispose();

         paths.set(index, group);

         oldBox = original.getExtent();
         newBox = group.getExtent();

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         paths.set(index_, original);

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         paths.set(index_, group);

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}

      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.separate");
      }
   }

   class MergePaths extends AbstractUndoableEdit
   {
      private int index_;
      private int[] indexArray;
      private Vector<JDRShape> list_;
      private JDRShape newPath;
      private int n;

      public MergePaths(Vector<JDRShape> list) 
         throws NoPathException,
                EmptyPathException,
                CanOnlyMergePathsException,
                IllFittingPathException
      {
         list_  = list;
         n      = list.size();

         indexArray = new int[n];

         for (int i = 0; i < n; i++)
         {
            JDRShape object = list.get(i);

            indexArray[i] = paths.indexOf(object);
         }

         Arrays.sort(indexArray);

         index_ = indexArray[0];

         newPath = paths.mergePaths(indexArray);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(index_);

         for (int i = 0; i < n; i++)
         {
            paths.add(indexArray[i], (JDRPath)list_.get(i));
         }

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.merge_paths");
      }
   }

   class XORPaths extends AbstractUndoableEdit
   {
      private int index_;
      private int[] indexArray;
      private Vector<JDRShape> list_;
      private JDRShape newPath;
      private int n;

      public XORPaths(Vector<JDRShape> list) 
         throws InvalidPathException
      {
         list_  = list;
         n      = list.size();

         indexArray = new int[n];

         for (int i = 0; i < n; i++)
         {
            JDRShape object = list.get(i);

            indexArray[i] = paths.indexOf(object);
         }

         Arrays.sort(indexArray);

         index_ = indexArray[0];

         newPath = list.get(0).exclusiveOr(list.get(1));

         for (int i = 2; i < n; i++)
         {
            newPath = newPath.exclusiveOr(list.get(i));
         }

         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(index_);

         for (int i = 0; i < n; i++)
         {
            paths.add(indexArray[i], list_.get(i));
         }

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.xor_paths");
      }
   }

   class PathIntersect extends AbstractUndoableEdit
   {
      private int index_;
      private int[] indexArray;
      private Vector<JDRShape> list_;
      private JDRShape newPath;
      private int n;

      public PathIntersect(Vector<JDRShape> list) 
         throws InvalidPathException
      {
         list_  = list;
         n      = list.size();

         indexArray = new int[n];

         for (int i = 0; i < n; i++)
         {
            JDRShape object = list.get(i);

            indexArray[i] = paths.indexOf(object);
         }

         Arrays.sort(indexArray);

         index_ = indexArray[0];

         newPath = list.get(0).intersect(list.get(1));

         for (int i = 2; i < n; i++)
         {
            newPath = newPath.intersect(list.get(i));
         }

         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(index_);

         for (int i = 0; i < n; i++)
         {
            paths.add(indexArray[i], list_.get(i));
         }

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.path_intersect");
      }
   }

   class SubtractPaths extends AbstractUndoableEdit
   {
      private int index_;
      private int[] indexArray;
      private Vector<JDRShape> list_;
      private JDRShape newPath;
      private int n;

      public SubtractPaths(Vector<JDRShape> list) 
         throws InvalidPathException
      {
         list_  = list;
         n      = list.size();

         indexArray = new int[n];

         for (int i = 0; i < n; i++)
         {
            JDRShape object = list.get(i);

            indexArray[i] = paths.indexOf(object);
         }

         Arrays.sort(indexArray);

         index_ = indexArray[0];

         newPath = list.get(0).subtract(list.get(1));

         for (int i = 2; i < n; i++)
         {
            newPath = newPath.subtract(list.get(i));
         }

         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(index_);

         for (int i = 0; i < n; i++)
         {
            paths.add(indexArray[i], list_.get(i));
         }

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.subtract_paths");
      }
   }

   class PathUnion extends AbstractUndoableEdit
   {
      private int index_;
      private int[] indexArray;
      private Vector<JDRShape> list_;
      private JDRShape newPath;
      private int n;

      public PathUnion(Vector<JDRShape> list) 
         throws InvalidPathException
      {
         list_  = list;
         n      = list.size();

         indexArray = new int[n];

         for (int i = 0; i < n; i++)
         {
            JDRShape object = list.get(i);

            indexArray[i] = paths.indexOf(object);
         }

         Arrays.sort(indexArray);

         index_ = indexArray[0];

         newPath = list.get(0).pathUnion(list.get(1));

         for (int i = 2; i < n; i++)
         {
            newPath = newPath.pathUnion(list.get(i));
         }

         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(index_);

         for (int i = 0; i < n; i++)
         {
            paths.add(indexArray[i], list_.get(i));
         }

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();

         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indexArray[i]);
         }

         paths.add(index_, newPath);

         double scale = getMagnification();
         Rectangle rect = newPath.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.path_union");
      }
   }

   class GroupObjects extends AbstractUndoableEdit
   {
      private int grpIndex_=0;
      private Vector<JDRCompleteObject> list_;
      private JDRGroup group_;
      private int[] indices;
      private int n;

      public GroupObjects(Vector<JDRCompleteObject> list)
      {
         // list should be sorted before passed to here
         list_  = list;
         n      = list.size();

         indices = new int[n];

         for (int i = 0; i < n; i++)
         {
            indices[i] = paths.indexOf(list_.get(i));
         }

         grpIndex_ = indices[0];

         group_ = new JDRGroup(n);

         for (int i = 0; i < n; i++)
         {
            JDRCompleteObject object = list_.get(i);
            group_.add(object);
         }

         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indices[i]);
         }

         paths.add(grpIndex_, group_);
         group_.setSelected(true);

         double scale = getMagnification();
         Rectangle rect = group_.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         double scale = getMagnification();
         Rectangle rect = group_.getExtent().getRectangle(scale);

         paths.remove(grpIndex_);

         for (int i = 0; i < n; i++)
         {
            JDRCompleteObject object = (JDRCompleteObject)list_.get(i);
            paths.add(indices[i], object);
         }

         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indices[i]);
         }
         paths.add(grpIndex_, group_);

         double scale = getMagnification();
         Rectangle rect = group_.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.group");
      }
   }

   class UngroupObjects extends AbstractUndoableEdit
   {
      private int grpIndex_;
      private Vector<JDRCompleteObject> list_;
      private JDRGroup group_;
      private int[] indices;
      private int n;

      public UngroupObjects(int index)
      {
         grpIndex_ = index;

         group_ = (JDRGroup)paths.get(grpIndex_);

         n = group_.size();

         list_ = new Vector<JDRCompleteObject>(n);
         indices = new int[n];

         paths.remove(grpIndex_);

         for (int i = 0; i < n; i++)
         {
            JDRCompleteObject object = group_.get(i);
            list_.add(i, object);
            indices[i] = grpIndex_+i;
            paths.add(indices[i], object);
         }

         double scale = getMagnification();
         Rectangle rect = group_.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         double scale = getMagnification();
         Rectangle rect = group_.getExtent().getRectangle(scale);

         paths.remove(grpIndex_);
         for (int i = 0; i < n; i++)
         {
            JDRCompleteObject object = (JDRCompleteObject)list_.get(i);
            paths.add(indices[i], object);
         }

         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         for (int i = n-1; i >= 0; i--)
         {
            paths.remove(indices[i]);
         }
         paths.add(grpIndex_, group_);

         double scale = getMagnification();
         Rectangle rect = group_.getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
         enableTools();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.ungroup");
      }
   }

   class MoveToFront extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_;
      private int oldIndex;

      public MoveToFront(JDRCompleteObject object)
      {
         object_ = object;
         oldIndex = paths.indexOf(object_);

         double scale = getMagnification();
         Rectangle rect = 
            paths.get(oldIndex).getExtent().getRectangle(scale);

         paths.moveToFront(object_);

         repaint(rect);
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         double scale = getMagnification();
         Rectangle rect = 
            paths.get(oldIndex).getExtent().getRectangle(scale);

         paths.moveToFront(object_);

         repaint(rect);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(object_);
         paths.add(oldIndex, object_);

         double scale = getMagnification();
         Rectangle rect = 
            paths.get(oldIndex).getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.move_to_front");
      }
   }

   class MoveToBack extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_;
      private int oldIndex;

      public MoveToBack(JDRCompleteObject object)
      {
         object_ = object;
         oldIndex = paths.indexOf(object_);

         double scale = getMagnification();
         Rectangle rect = 
            paths.get(oldIndex).getExtent().getRectangle(scale);

         paths.moveToBack(object_);

         repaint(rect);
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         double scale = getMagnification();
         Rectangle rect = 
            paths.get(oldIndex).getExtent().getRectangle(scale);

         paths.moveToBack(object_);

         repaint(rect);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(object_);
         paths.add(oldIndex, object_);

         double scale = getMagnification();
         Rectangle rect = 
            paths.get(oldIndex).getExtent().getRectangle(scale);
         repaint(rect);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.move_to_back");
      }
   }

   class ConvertToLine extends AbstractUndoableEdit
   {
      private JDRPathSegment oldSegment_, newSegment_;
      private int index_;
      private JDRShape path_;
      private JDRPoint oldPt_, newPt_;

      public ConvertToLine(JDRShape path, JDRPathSegment segment)
      {
         index_ = path.getIndex(segment);
         path_ = path;
         oldPt_ = getSelectedPoint();

         oldSegment_ = segment;
         newSegment_ = segment.convertToLine();
         BBox oldbox = path_.getControlBBox();

         path_.convertSegment(index_, newSegment_);

         newPt_ = oldPt_;

         if (oldSegment_ instanceof JDRBezier)
         {
            if (oldPt_ == ((JDRBezier)oldSegment_).getControl1())
            {
               newPt_ = oldSegment_.getStart();
            }
            else if (oldPt_ == ((JDRBezier)oldSegment_).getControl2())
            {
               newPt_ = oldSegment_.getEnd();
            }
         }

         editedPath.selectControl(newPt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = path_.getControlBBox();
         path_.convertSegment(index_, newSegment_);
         editedPath.selectControl(newPt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = path_.getControlBBox();
         path_.convertSegment(index_, oldSegment_);
         editedPath.selectControl(oldPt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_line");
      }
   }

   class ConvertToSegment extends AbstractUndoableEdit
   {
      private JDRPathSegment oldSegment_, newSegment_;
      private int index_;
      private JDRShape path_;
      private JDRPoint oldPt_=null, newPt_=null;

      public ConvertToSegment(JDRShape path, JDRPathSegment segment)
      {
         index_ = path.getIndex(segment);
         path_ = path;
         oldPt_ = getSelectedPoint();

         oldSegment_ = segment;
         newSegment_ = segment.convertToSegment();
         BBox oldbox = path_.getControlBBox();

         path_.convertSegment(index_, newSegment_);

         newPt_ = oldPt_;

         if (oldSegment_ instanceof JDRBezier)
         {
            if (oldPt_ == ((JDRBezier)oldSegment_).getControl1())
            {
               newPt_ = oldSegment_.getStart();
            }
            else if (oldPt_ == ((JDRBezier)oldSegment_).getControl2())
            {
               newPt_ = oldSegment_.getEnd();
            }
         }

         editedPath.selectControl(newPt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = path_.getControlBBox();
         path_.convertSegment(index_, newSegment_);
         editedPath.selectControl(newPt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = path_.getControlBBox();
         path_.convertSegment(index_, oldSegment_);
         editedPath.selectControl(oldPt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_move");
      }
   }

   class ConvertToCurve extends AbstractUndoableEdit
   {
      private JDRPathSegment oldSegment_, newSegment_;
      private int index_;
      private JDRShape path_;
      private JDRPoint pt_=null;

      public ConvertToCurve(JDRShape path, JDRPathSegment segment)
      {
         index_ = path.getIndex(segment);
         path_ = path;

         pt_ = getSelectedPoint();
         oldSegment_ = segment;
         newSegment_ = segment.convertToBezier();
         BBox oldbox = path_.getControlBBox();

         path_.convertSegment(index_, newSegment_);

         editedPath.selectControl(pt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = path_.getControlBBox();
         path_.convertSegment(index_, newSegment_);
         editedPath.selectControl(pt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = path_.getControlBBox();
         path_.convertSegment(index_, oldSegment_);
         editedPath.selectControl(pt_);
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_curve");
      }
   }

   class AddPoint extends AbstractUndoableEdit
   {
      private int index_;
      private JDRShape path_, oldPath_;
      private JDRPoint oldPt=null, newPt=null;

      public AddPoint(JDRShape path)
      {
         index_ = paths.indexOf(path);
         path_ = (JDRShape)path.clone();
         oldPath_ = path;
         oldPt = getSelectedPoint();
         int ctrIdx = editedPath.getControlIndex(oldPt);

         BBox oldbox = oldPath_.getControlBBox();

         paths.set(index_, path_);
         editedPath.setEditMode(false);
         editedPath = path_;
         editedPath.selectControl(ctrIdx);

         newPt = path_.addPoint();

         editedPath.selectControl(newPt);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, path_);
         editedPath.setEditMode(false);
         editedPath = path_;
         editedPath.selectControl(newPt);

         BBox oldbox = oldPath_.getControlBBox();
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, oldPath_);
         editedPath.setEditMode(false);
         editedPath = oldPath_;
         editedPath.selectControl(oldPt);

         BBox oldbox = path_.getControlBBox();
         BBox newbox = oldPath_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.add_point");
      }
   }

   class DeletePoint extends AbstractUndoableEdit
   {
      private int index_;
      private JDRShape path_, oldPath_;
      private boolean pathRemoved_=false;
      private int oldPtIndex_, newPtIndex_;

      public DeletePoint(JDRShape path)
      {
         index_ = paths.indexOf(path);
         oldPath_ = (JDRShape)path.clone();

         JDRPoint newPt = editedPath.getSelectedSegment().getEnd();
         oldPtIndex_ = editedPath.getSelectedControlIndex();

         JDRPathSegment lastSegment = editedPath.getLastSegment();

         if (editedPath.getSelectedSegment() == lastSegment
           && getSelectedPoint() != lastSegment.getStart())
         {
            newPt = editedPath.getSelectedSegment().getStart();
         }

         BBox oldbox = oldPath_.getControlBBox();
         path.removeSelectedSegment();

         newPtIndex_ = editedPath.getSelectedControlIndex();

         if (newPtIndex_ == -1 || (path.size()==1 && path.isClosed()))
         {
            paths.remove(index_);
            pathRemoved_ = true;
            if (editedPath == path)
            {
               editedPath = null;
               finishEditingPath();
               enableTools();
            }
         }
         else
         {
            editedPath.selectControl(newPtIndex_);
            breakPathItem.setEnabled(
               editedPath.getSelectedSegment()
             !=editedPath.getLastSegment());
         }

         path_ = path;
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         oldPath_ = (JDRPath)paths.get(index_);
         if (pathRemoved_)
         {
            finishEditingPath();
            paths.remove(index_);
         }
         else
         {
            paths.set(index_, path_);
            editedPath = path_;
            path_.selectControl(newPtIndex_);
            breakPathItem.setEnabled(
               editedPath.getSelectedSegment()
             !=editedPath.getLastSegment());
         }

         BBox oldbox = oldPath_.getControlBBox();
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         if (pathRemoved_)
         {
            paths.add(index_, oldPath_);
         }
         else
         {
            path_ = (JDRPath)paths.get(index_);
            paths.set(index_, oldPath_);
         }

         editedPath = oldPath_;
         editedPath.selectControl(oldPtIndex_);
         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());

         BBox oldbox = path_.getControlBBox();
         BBox newbox = oldPath_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.delete_point");
      }
   }

   class BreakPath extends AbstractUndoableEdit
   {
      private int index_;
      private JDRShape oldPath, newPath1, newPath2;
      private int controlIndex;

      public BreakPath(JDRShape path)
         throws InvalidPathException
      {
         index_ = paths.indexOf(path);
         oldPath = path;
         path = (JDRShape)oldPath.clone();
         controlIndex = editedPath.getSelectedControlIndex();
         editedPath = path;
         editedPath.selectControl(controlIndex);

         newPath2 = path.breakPath();

         if (newPath2.size() == 0)
         {
            editedPath = oldPath;
            throw new EmptyPathException();
         }

         newPath1 = path;

         paths.set(index_, newPath1);
         paths.add(index_+1, newPath2);

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());

         setBackgroundImage();
         repaint(oldPath.getControlBBox().getRectangle(
            getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.remove(index_+1);
         paths.set(index_, oldPath);

         editedPath = oldPath;
         editedPath.selectControl(controlIndex);

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());

         setBackgroundImage();
         repaint(oldPath.getControlBBox().getRectangle(
           getMagnification()));
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, newPath1);
         paths.add(index_+1, newPath2);

         editedPath = newPath1;
         editedPath.selectControl(controlIndex);

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());

         setBackgroundImage();
         repaint(oldPath.getControlBBox().getRectangle(
           getMagnification()));
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.break");
      }
   }

   class OpenPath extends AbstractUndoableEdit
   {
      private int index_;
      private JDRShape path_, oldPath_;
      private int oldPtIdx, newPtIdx;

      public OpenPath(JDRShape path, boolean removeLast)
      {
         index_ = paths.indexOf(path);
         oldPath_ = path;
         path_ = (JDRShape)oldPath_.clone();
         oldPtIdx = editedPath.getSelectedControlIndex();
         newPtIdx = oldPtIdx;

         BBox oldbox = oldPath_.getControlBBox();
         path_.open(removeLast);
         paths.set(index_, path_);

         editedPath.setEditMode(false);
         editedPath = path_;

         if (editedPath.selectControl(newPtIdx) == null)
         {
            newPtIdx = 0;
            editedPath.selectControl(newPtIdx);
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, path_);
         editedPath.setEditMode(false);
         editedPath = path_;
         editedPath.selectControl(newPtIdx);

         BBox oldbox = oldPath_.getControlBBox();
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, oldPath_);
         editedPath.setEditMode(false);
         editedPath = oldPath_;
         editedPath.selectControl(oldPtIdx);

         BBox oldbox = path_.getControlBBox();
         BBox newbox = oldPath_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.open");
      }
   }

   class ClosePath extends AbstractUndoableEdit
   {
      private int index_;
      private JDRShape path_, oldPath_;
      private int oldPtIdx;

      public ClosePath(JDRShape path, int closeType)
         throws EmptyPathException
      {
         index_ = paths.indexOf(path);
         oldPath_ = path;
         path_ = (JDRShape)oldPath_.clone();
         oldPtIdx = editedPath.getSelectedControlIndex();

         BBox oldbox = oldPath_.getControlBBox();
         path_.close(closeType);
         paths.set(index_, path_);

         editedPath.setEditMode(false);
         editedPath = path_;
         editedPath.selectControl(oldPtIdx);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, path_);
         editedPath.setEditMode(false);
         editedPath = path_;
         editedPath.selectControl(oldPtIdx);

         BBox oldbox = oldPath_.getControlBBox();
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, oldPath_);
         editedPath.setEditMode(false);
         editedPath = oldPath_;
         editedPath.selectControl(oldPtIdx);

         BBox oldbox = path_.getControlBBox();
         BBox newbox = oldPath_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.close");
      }
   }

   class MovePoint extends AbstractUndoableEdit
   {
      private JDRPoint dp_;
      private JDRPathSegment segment_;
      private JDRShape path_;
      private double x_, y_;
      private Rectangle rect;

      public MovePoint(JDRShape path, JDRPathSegment segment, JDRPoint dp, 
                       double x, double y)
      {
         dp_ = dp;
         path_ = path;
         x_ = x;
         y_ = y;
         segment_ = segment;

         BBox box = path_.getControlBBox();

         path_.translateControl(segment_, dp_, x_, y_);

         path_.mergeControlBBox(box);

         rect = box.getRectangle(getMagnification());
         rect.grow(2, 2);

         repaint(rect);
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         path_.translateControl(segment_, dp_, x_, y_);
         repaint(rect);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         path_.translateControl(segment_, dp_, -x_, -y_);
         repaint(rect);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.move_point");
      }
   }

   class SetSymmetryJoinAnchor extends AbstractUndoableEdit
   {
      private JDRSymmetricPath path_;
      private JDRPartialSegment oldJoin, newJoin;

      public SetSymmetryJoinAnchor(boolean anchor)
         throws InvalidClassException
      {
         if (!(editedPath instanceof JDRSymmetricPath))
         {
            throw new InvalidClassException("Not a JDRSymmetricPath");
         }

         path_ = (JDRSymmetricPath)editedPath;

         BBox oldbox = path_.getControlBBox();

         oldJoin = path_.getJoin();

         if (anchor)
         {
            newJoin = null;
         }
         else
         {
            newJoin = new JDRPartialSegment(
               path_.getLastSegment().getEnd(),
               path_.getSymmetry());
         }

         path_.setJoin(newJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         BBox oldbox = path_.getControlBBox();

         path_.setJoin(newJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         BBox oldbox = path_.getControlBBox();

         path_.setJoin(oldJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.symmetry.join_anchor");
      }
   }

   class SetSymmetryCloseAnchor extends AbstractUndoableEdit
   {
      private JDRSymmetricPath path_;
      private JDRPartialSegment oldSegment, newSegment;

      public SetSymmetryCloseAnchor(boolean anchor)
         throws InvalidClassException,
                EmptyPathException,
                IllFittingPathException
      {
         if (!(editedPath instanceof JDRSymmetricPath))
         {
            throw new InvalidClassException("Not a JDRSymmetricPath");
         }

         path_ = (JDRSymmetricPath)editedPath;

         BBox oldbox = path_.getControlBBox();

         oldSegment = path_.getClosingSegment();

         if (anchor)
         {
            newSegment = null;
         }
         else
         {
            newSegment = new JDRPartialLine();
         }

         path_.close(newSegment);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         BBox oldbox = path_.getControlBBox();

         try
         {
            path_.close(newSegment);
         }
         catch (IllFittingPathException e)
         {
         }
         catch (EmptyPathException e)
         {
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         BBox oldbox = path_.getControlBBox();

         try
         {
            path_.close(oldSegment);
         }
         catch (IllFittingPathException e)
         {
         }
         catch (EmptyPathException e)
         {
         }
         
         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.symmetry.close_anchor");
      }
   }

   class JoinToMove extends AbstractUndoableEdit
   {
      private JDRSymmetricPath path_;
      private JDRPartialSegment oldJoin, newJoin;

      public JoinToMove(JDRSymmetricPath path)
      {
         path_ = path;

         BBox oldbox = path_.getControlBBox();

         oldJoin = path_.getJoin();

         newJoin = new JDRPartialSegment(
            path_.getLastSegment().getEnd(),
            path_.getSymmetry());

         path_.setJoin(newJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         BBox oldbox = path_.getControlBBox();

         path_.setJoin(newJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         BBox oldbox = path_.getControlBBox();

         path_.setJoin(oldJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_move");
      }
   }

   class ClosingToMove extends AbstractUndoableEdit
   {
      private JDRSymmetricPath path_;
      private JDRPartialSegment oldSegment, newSegment;

      public ClosingToMove(JDRSymmetricPath path)
         throws InvalidPathException
      {
         path_ = path;

         BBox oldbox = path_.getControlBBox();

         oldSegment = path_.getClosingSegment();

         newSegment = new JDRPartialSegment();

         path_.close(newSegment);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         BBox oldbox = path_.getControlBBox();

         try
         {
            path_.close(newSegment);
         }
         catch (InvalidPathException e)
         {
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         BBox oldbox = path_.getControlBBox();

         try
         {
            path_.close(oldSegment);
         }
         catch (InvalidPathException e)
         {
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_move");
      }
   }

   class JoinToLine extends AbstractUndoableEdit
   {
      private JDRSymmetricPath path_;
      private JDRPartialSegment oldJoin, newJoin;

      public JoinToLine(JDRSymmetricPath path)
      {
         path_ = path;

         BBox oldbox = path_.getControlBBox();

         oldJoin = path_.getJoin();

         newJoin = new JDRPartialLine(
            path_.getLastSegment().getEnd(),
            path_.getSymmetry());

         path_.setJoin(newJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         BBox oldbox = path_.getControlBBox();

         path_.setJoin(newJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         BBox oldbox = path_.getControlBBox();

         path_.setJoin(oldJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_line");
      }
   }

   class ClosingToLine extends AbstractUndoableEdit
   {
      private JDRSymmetricPath path_;
      private JDRPartialSegment oldSegment, newSegment;

      public ClosingToLine(JDRSymmetricPath path)
         throws InvalidPathException
      {
         path_ = path;

         BBox oldbox = path_.getControlBBox();

         oldSegment = path_.getClosingSegment();

         newSegment = new JDRPartialLine();

         path_.close(newSegment);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         BBox oldbox = path_.getControlBBox();

         try
         {
            path_.close(newSegment);
         }
         catch (InvalidPathException e)
         {
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         BBox oldbox = path_.getControlBBox();

         try
         {
            path_.close(oldSegment);
         }
         catch (InvalidPathException e)
         {
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_line");
      }
   }

   class JoinToCurve extends AbstractUndoableEdit
   {
      private JDRSymmetricPath path_;
      private JDRPartialSegment oldJoin, newJoin;

      public JoinToCurve(JDRSymmetricPath path)
      {

         path_ = path;

         BBox oldbox = path_.getControlBBox();

         oldJoin = path_.getJoin();

         newJoin = new JDRPartialBezier(
            path_.getLastSegment().getEnd(),
            path_.getSymmetry());

         path_.setJoin(newJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         BBox oldbox = path_.getControlBBox();

         path_.setJoin(newJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         BBox oldbox = path_.getControlBBox();

         path_.setJoin(oldJoin);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_curve");
      }
   }

   class ClosingToCurve extends AbstractUndoableEdit
   {
      private JDRSymmetricPath path_;
      private JDRPartialSegment oldSegment, newSegment;

      public ClosingToCurve(JDRSymmetricPath path)
        throws InvalidPathException
      {

         path_ = path;

         BBox oldbox = path_.getControlBBox();

         oldSegment = path_.getClosingSegment();

         newSegment = (JDRPartialSegment)new JDRPartialBezier(
            path_.getFirstControl(),
            path_.getSymmetry()).reverse();

         path_.close(newSegment);

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         BBox oldbox = path_.getControlBBox();

         try
         {
            path_.close(newSegment);
         }
         catch (InvalidPathException e)
         {
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         BBox oldbox = path_.getControlBBox();

         try
         {
            path_.close(oldSegment);
         }
         catch (InvalidPathException e)
         {
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.convert_to_curve");
      }
   }

   class MakeContinuous extends AbstractUndoableEdit
   {
      private JDRPathSegment segment_;
      private JDRShape path_;
      private double x_, y_, newX_, newY_;
      private boolean isStart=true;

      public MakeContinuous() throws InvalidClassException
      {
         if (!(editedPath.getSelectedSegment() instanceof JDRBezier)
           && !(editedPath.getSelectedSegment() instanceof JDRPartialBezier))
         {
            throw new InvalidClassException("Not a Bezier segment");
         }

         path_ = editedPath;

         segment_ = editedPath.getSelectedSegment();

         if (segment_ instanceof JDRPartialBezier)
         {
            isStart = true;

            x_ = ((JDRPartialBezier)segment_).getControl1().getX();
            y_ = ((JDRPartialBezier)segment_).getControl1().getY();
         }
         else
         {
            isStart = true;

            if (path_.getSelectedControl()
                   == ((JDRBezier)segment_).getControl2())
            {
               isStart = false;
            }

            x_ = isStart
               ? ((JDRBezier)segment_).getControl1().x
               : ((JDRBezier)segment_).getControl2().x;

            y_ = isStart
               ? ((JDRBezier)segment_).getControl1().y
               : ((JDRBezier)segment_).getControl2().y;
         }

         BBox oldbox = path_.getControlBBox();
         path_.makeContinuous(isStart);

         if (segment_ instanceof JDRPartialBezier)
         {
            newX_ = ((JDRPartialBezier)segment_).getControl1().getX();
            newY_ = ((JDRPartialBezier)segment_).getControl1().getY();
         }
         else
         {
            newX_ = isStart
                  ? ((JDRBezier)segment_).getControl1().x
                  : ((JDRBezier)segment_).getControl2().x;
            newY_ = isStart
                  ? ((JDRBezier)segment_).getControl1().y
                  : ((JDRBezier)segment_).getControl2().y;
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = path_.getControlBBox();

         if (isStart)
         {
            if (segment_ instanceof JDRPartialBezier)
            {
               ((JDRPartialBezier)segment_).setControl(newX_, newY_);
            }
            else
            {
               ((JDRBezier)segment_).setControl1(newX_, newY_);
            }
         }
         else
         {
            ((JDRBezier)segment_).setControl2(newX_, newY_);
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = path_.getControlBBox();

         if (isStart)
         {
            if (segment_ instanceof JDRPartialBezier)
            {
               ((JDRPartialBezier)segment_).setControl(x_, y_);
            }
            else
            {
               ((JDRBezier)segment_).setControl1(x_, y_);
            }
         }
         else
         {
            ((JDRBezier)segment_).setControl2(x_, y_);
         }

         BBox newbox = path_.getControlBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.continuous");
      }
   }

   class SelectControl extends AbstractUndoableEdit
   {
      private int oldIndex_, newIndex_;
      private String presentation_
         = JDRResources.getString("undo.select_point");

      public SelectControl(Point2D currentPos)
         throws NullPointerException
      {
         oldIndex_ = editedPath.getSelectedControlIndex();
         newIndex_ = editedPath.getControlIndex(currentPos);

         if (newIndex_ == -1)
         {
            throw new NullPointerException();
         }

         editedPath.selectControl(newIndex_);
         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         editedPath.selectControl(newIndex_);

         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         editedPath.selectControl(oldIndex_);
         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return presentation_;
      }
   }

   class SelectNextControl extends AbstractUndoableEdit
   {
      private int oldIndex, newIndex;
      private String presentation_
         = JDRResources.getString("undo.select_point");

      public SelectNextControl()
         throws NullPointerException
      {
         oldIndex = editedPath.getSelectedControlIndex();

         JDRPoint newPt = editedPath.selectNextControl();
         if (newPt == null)
         {
            throw new NullPointerException();
         }

         newIndex = editedPath.getSelectedControlIndex();

         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()!=editedPath.getLastSegment());
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         editedPath.selectControl(newIndex);

         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         editedPath.selectControl(oldIndex);
         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return presentation_;
      }
   }

   class SelectPrevControl extends AbstractUndoableEdit
   {
      private int oldIndex, newIndex;
      private String presentation_
         = JDRResources.getString("undo.select_point");

      public SelectPrevControl()
         throws NullPointerException
      {
         oldIndex = editedPath.getSelectedControlIndex();

         JDRPoint newPt = editedPath.selectPreviousControl();

         if (newPt == null)
         {
            throw new NullPointerException();
         }

         newIndex = editedPath.getSelectedControlIndex();

         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()!=editedPath.getLastSegment());
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         editedPath.selectControl(newIndex);

         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         editedPath.selectControl(oldIndex);
         BBox box = editedPath.getControlBBox();
         repaint(box.getRectangle(getMagnification()));

         breakPathItem.setEnabled(
            editedPath.getSelectedSegment()
          !=editedPath.getLastSegment());
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return presentation_;
      }
   }

   class SelectObject extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_;
      private boolean oldselected_, selected_;
      private String string_=JDRResources.getString("undo.select");

      public SelectObject(JDRCompleteObject object, boolean selected)
      {
         object_ = object;
         oldselected_ = object.isSelected();
         selected_ = selected;
         object_.setSelected(selected_);
         enableTools();

         BBox box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));
      }

      public SelectObject(JDRCompleteObject object, boolean selected,
                        String presentationString)
      {
         this(object, selected);
         string_ = presentationString;
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         if (frame_.currentTool() != JpgfDraw.ACTION_SELECT)
         {
            frame_.setTool(JpgfDraw.ACTION_SELECT);
         }

         object_.setSelected(selected_);
         enableTools();
         BBox box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         if (frame_.currentTool() != JpgfDraw.ACTION_SELECT)
         {
            frame_.setTool(JpgfDraw.ACTION_SELECT);
         }
         
         object_.setSelected(oldselected_);
         enableTools();
         BBox box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string_;
      }
   }

   class MoveObject extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_;
      private double x_, y_;

      public MoveObject(JDRCompleteObject object, double x, double y)
      {
         object_ = object;
         x_ = x;
         y_ = y;

         BBox oldbox = object_.getExtent();
         object_.translate(-x_, -y_);
         BBox newbox = object_.getExtent();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = object_.getExtent();
         object_.translate(-x_, -y_);
         BBox newbox = object_.getExtent();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = object_.getExtent();
         object_.translate(x_, y_);
         BBox newbox = object_.getExtent();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.move");
      }
   }

   // Set symmetry
   class SetSymmetricPath extends AbstractUndoableEdit
   {
      private JDRShape oldPath_, newPath_;
      private int index_;

      public SetSymmetricPath(JDRShape path, boolean addSymmetry,
         int index)
      throws uk.ac.uea.cmp.nlct.jdr.InvalidObjectException
      {
         oldPath_ = path;
         index_ = index;

         if (addSymmetry)
         {
            // Converting regular shape to a symmetric path.
            // If the specified path is already symmetric,
            // something's gone wrong.
            if (path.hasSymmetricPath())
            {
               throw new uk.ac.uea.cmp.nlct.jdr.InvalidObjectException
                 ("Path is already symmetric");
            }

            // If the path is a pattern, then add symmetry to the
            // underlying path

            if (path instanceof JDRPattern)
            {
               newPath_ = (JDRCompoundShape)path.clone();

               ((JDRPattern)newPath_).setUnderlyingShape(
                  new JDRSymmetricPath(
                         ((JDRPattern)path).getUnderlyingShape()));
            }
            else
            {
               newPath_ = new JDRSymmetricPath(path);
            }
         }
         else
         {
            // Is this shape a symmetric path, or is the underlying
            // shape symmetric?

            if (!path.hasSymmetricPath())
            {
               throw new uk.ac.uea.cmp.nlct.jdr.InvalidObjectException
                  ("Not a symmetric path");
            }

            newPath_ = ((JDRCompoundShape)path).removeSymmetry();
         }

         newPath_.setSelected(true);
         editedPath = newPath_;

         paths.set(index_, newPath_);
         BBox oldbox = oldPath_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = newPath_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldPath_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));

         paths.set(index_, newPath_);
         editedPath = newPath_;

         BBox newbox = newPath_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldPath_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));

         paths.set(index_, oldPath_);
         editedPath = oldPath_;

         BBox newbox = newPath_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.symmetry");
      }
   }

   // edit bitmap properties
   class SetBitmapProperties extends AbstractUndoableEdit
   {
      private JDRBitmap object_;
      private String filename_, latexfilename_;
      private String oldFilename_, oldLatexfilename_;
      private String oldCommand_, command_;
      private double[] oldmatrix, newmatrix;

      public SetBitmapProperties(JDRBitmap bitmap, String newfilename,
                        String newlatexfilename, String command,
                        double[] matrix)
      {
         object_ = bitmap;
         oldFilename_      = bitmap.getFilename();
         oldLatexfilename_ = bitmap.getLaTeXLinkName();
         filename_         = newfilename;
         latexfilename_    = newlatexfilename;
         oldCommand_       = bitmap.latexCommand;
         command_          = command;
         newmatrix         = matrix;

         oldmatrix = new double[6];
         bitmap.getTransformation(oldmatrix);

         BBox oldbox = object_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));

         object_.setProperties(filename_, latexfilename_);
         object_.latexCommand = command_;
         object_.setTransformation(newmatrix);

         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = object_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));

         object_.setProperties(filename_, latexfilename_);
         object_.latexCommand = command_;
         object_.setTransformation(newmatrix);

         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = object_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));

         object_.setProperties(oldFilename_, oldLatexfilename_);
         object_.latexCommand = oldCommand_;
         object_.setTransformation(oldmatrix);

         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.bitmap_properties");
      }
   }

   class Scale extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_, oldobject_;
      private int index_;

      public Scale(JDRCompleteObject object, double factor, int index)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.scale(factor);
         paths.set(index_, object_);
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public Scale(JDRCompleteObject object, double factorX, double factorY,
                   int index, JDRPoint p)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.scale(p, factorX, factorY);
         paths.set(index_, object_);

         if (dragScaleObject == oldobject_) dragScaleObject = object_;

         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, object_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, oldobject_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.scale");
      }
   }

   class ScaleX extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_, oldobject_;
      private int index_;

      public ScaleX(JDRCompleteObject object, double factor, int index)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.scaleX(factor);
         paths.set(index_, object_);
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public ScaleX(JDRCompleteObject object, double factor, int index,
                    JDRPoint p)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.scaleX(p, factor);
         paths.set(index_, object_);

         if (dragScaleObject == oldobject_) dragScaleObject = object_;

         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, object_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, oldobject_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.scale_x");
      }
   }

   class ScaleY extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_, oldobject_;
      private int index_;

      public ScaleY(JDRCompleteObject object, double factor, int index)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.scaleY(factor);
         paths.set(index_, object_);
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public ScaleY(JDRCompleteObject object, double factor, int index,
                    JDRPoint p)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.scaleY(p, factor);

         if (dragScaleObject == oldobject_) dragScaleObject = object_;

         paths.set(index_, object_);
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, object_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, oldobject_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.scale_y");
      }
   }

   class Shear extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_, oldobject_;
      private int index_;

      public Shear(JDRCompleteObject object, double factorX, double factorY, int index)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.shear(factorX, factorY);
         paths.set(index_, object_);

         if (dragScaleObject == oldobject_) dragScaleObject = object_;

         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public Shear(JDRCompleteObject object, double factorX, double factorY, int index, JDRPoint p)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.shear(p, factorX, factorY);
         paths.set(index_, object_);

         if (dragScaleObject == oldobject_) dragScaleObject = object_;

         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, object_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, oldobject_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.shear");
      }
   }

   class Rotate extends AbstractUndoableEdit
   {
      private JDRCompleteObject object_, oldobject_;
      private int index_;

      public Rotate(JDRCompleteObject object, double angle, int index)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.rotate(angle);
         paths.set(index_, object_);
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public Rotate(JDRCompleteObject object, double angle, 
                   int index, JDRPoint p)
      {
         oldobject_ = object;
         object_ = (JDRCompleteObject)object.clone();
         index_ = index;

         object_.rotate(p, angle);
         paths.set(index_, object_);

         if (dragScaleObject == oldobject_) dragScaleObject = object_;

         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, object_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, oldobject_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.rotate");
      }
   }

   class ReversePath extends AbstractUndoableEdit
   {
      private JDRShape object_, oldobject_;
      private int index_;

      public ReversePath(JDRShape object, int index)
         throws InvalidPathException
      {
         oldobject_ = object;
         object_ = object.reverse();
         index_ = index;

         paths.set(index_, object_);
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, object_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, oldobject_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.reverse");
      }
   }

   class AlignGroup extends AbstractUndoableEdit
   {
      private JDRGroup object_, oldobject_;
      private int index_, align_;
      private String string_=JDRResources.getString("undo.justify");

      public static final int LEFT=0, CENTRE=1, RIGHT=2,
                              TOP=3, MIDDLE=4, BOTTOM=5;

      public AlignGroup(JDRGroup object, int index, int align)
      {
         oldobject_ = object;
         object_ = (JDRGroup)object.clone();
         index_ = index;
         align_ = align;

         switch (align_)
         {
            case LEFT :
               object_.leftAlign();
               string_ = JDRResources.getString("undo.justify.left");

               if (JpgfDraw.isAutoAnchorEnabled())
               {
                  for (int i = 0; i < object_.size(); i++)
                  {
                     JDRCompleteObject obj = object_.get(i);

                     if (obj instanceof JDRText)
                     {
                        JDRText text = (JDRText)obj;

                        try
                        {
                           text.setHAlign(JDRText.PGF_HALIGN_LEFT);
                        }
                        catch (InvalidHAlignException e)
                        {
                           JDRResources.internalError(null, e);
                        }
                     }
                  }
               }
            break;
            case CENTRE :
               object_.centreAlign();
               string_ = JDRResources.getString("undo.justify.centre");

               if (JpgfDraw.isAutoAnchorEnabled())
               {
                  for (int i = 0; i < object_.size(); i++)
                  {
                     JDRCompleteObject obj = object_.get(i);

                     if (obj instanceof JDRText)
                     {
                        JDRText text = (JDRText)obj;

                        try
                        {
                           text.setHAlign(JDRText.PGF_HALIGN_CENTRE);
                        }
                        catch (InvalidHAlignException e)
                        {
                           JDRResources.internalError(null, e);
                        }
                     }
                  }
               }
            break;
            case RIGHT :
               object_.rightAlign();
               string_ = JDRResources.getString("undo.justify.right");

               if (JpgfDraw.isAutoAnchorEnabled())
               {
                  for (int i = 0; i < object_.size(); i++)
                  {
                     JDRCompleteObject obj = object_.get(i);

                     if (obj instanceof JDRText)
                     {
                        JDRText text = (JDRText)obj;

                        try
                        {
                           text.setHAlign(JDRText.PGF_HALIGN_RIGHT);
                        }
                        catch (InvalidHAlignException e)
                        {
                           JDRResources.internalError(null, e);
                        }
                     }
                  }
               }
            break;
            case TOP :
               object_.topAlign();
               string_ = JDRResources.getString("undo.justify.top");

               if (JpgfDraw.isAutoAnchorEnabled())
               {
                  for (int i = 0; i < object_.size(); i++)
                  {
                     JDRCompleteObject obj = object_.get(i);

                     if (obj instanceof JDRText)
                     {
                        JDRText text = (JDRText)obj;

                        try
                        {
                           text.setVAlign(JDRText.PGF_VALIGN_TOP);
                        }
                        catch (InvalidVAlignException e)
                        {
                           JDRResources.internalError(null, e);
                        }
                     }
                  }
               }
            break;
            case MIDDLE :
               object_.middleAlign();
               string_ = JDRResources.getString("undo.justify.middle");

               if (JpgfDraw.isAutoAnchorEnabled())
               {
                  for (int i = 0; i < object_.size(); i++)
                  {
                     JDRCompleteObject obj = object_.get(i);

                     if (obj instanceof JDRText)
                     {
                        JDRText text = (JDRText)obj;

                        try
                        {
                           text.setVAlign(JDRText.PGF_VALIGN_CENTRE);
                        }
                        catch (InvalidVAlignException e)
                        {
                           JDRResources.internalError(null, e);
                        }
                     }
                  }
               }
            break;
            case BOTTOM :
               object_.bottomAlign();
               string_ = JDRResources.getString("undo.justify.bottom");

               if (JpgfDraw.isAutoAnchorEnabled())
               {
                  for (int i = 0; i < object_.size(); i++)
                  {
                     JDRCompleteObject obj = object_.get(i);

                     if (obj instanceof JDRText)
                     {
                        JDRText text = (JDRText)obj;

                        try
                        {
                           text.setVAlign(JDRText.PGF_VALIGN_BOTTOM);
                        }
                        catch (InvalidVAlignException e)
                        {
                           JDRResources.internalError(null, e);
                        }
                     }
                  }
               }
            break;
         }

         paths.set(index_, object_);

         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, object_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = oldobject_.getExtent();
         repaint(oldbox.getRectangle(getMagnification()));
         paths.set(index_, oldobject_);
         BBox newbox = object_.getExtent();
         repaint(newbox.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string_;
      }
   }

   class SetText extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private String oldtext_, newtext_;
      private String oldltxtext_, newltxtext_;

      public SetText(JDRTextual object, String newtext,
         String newlatexText)
      {
         object_     = object;
         oldtext_    = object.getText();
         newtext_    = newtext;
         oldltxtext_ = object.getLaTeXText();
         newltxtext_ = newlatexText;

         BBox oldbox = object_.getExtent();

         Graphics2D g = (Graphics2D)getGraphics();
         g.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g.scale(norm,norm);

         object_.setText(g, newtext_);
         object_.setLaTeXText(newltxtext_);
         g.dispose();

         BBox newbox = object_.getExtent();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));

         Graphics2D g = (Graphics2D)getGraphics();
         g.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g.scale(norm,norm);

         object_.setText(g, newtext_);
         object_.setLaTeXText(newltxtext_);
         g.dispose();

         box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));

         Graphics2D g = (Graphics2D)getGraphics();
         g.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g.scale(norm,norm);

         object_.setText(g, oldtext_);
         object_.setLaTeXText(oldltxtext_);
         g.dispose();

         box = object_.getExtent();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.edit_text");
      }
   }

   class FadeObject extends AbstractUndoableEdit
   {
      private JDRCompleteObject oldObject_, newObject_;
      private BBox box;
      private int index_;

      public FadeObject(JDRCompleteObject object, double value, int index)
      {
         index_ = index;
         oldObject_ = object;
         newObject_ = (JDRCompleteObject)object.clone();

         newObject_.fade(value);
         paths.set(index_, newObject_);

         BBox box = oldObject_.getBBox();
         repaint(box.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         paths.set(index_, newObject_);

         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         paths.set(index_, oldObject_);

         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.colour");
      }
   }

   class SetLinePaint extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private JDRPaint oldpaint_, newpaint_;

      public SetLinePaint(JDRShape object, JDRPaint paint)
      {
         object_ = object;
         oldpaint_ = object.getLinePaint();
         newpaint_ = paint;

         object_.setLinePaint(newpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         object_.setLinePaint(newpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         object_.setLinePaint(oldpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_colour");
      }
   }

   class SetTextPaint extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private JDRPaint oldpaint_, newpaint_;

      public SetTextPaint(JDRTextual object, JDRPaint paint)
      {
         object_ = object;
         oldpaint_ = object.getTextPaint();
         newpaint_ = paint;

         object_.setTextPaint(newpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         object_.setTextPaint(newpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         object_.setTextPaint(oldpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.colour");
      }
   }

   class SetFillPaint extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private JDRPaint oldpaint_, newpaint_;

      public SetFillPaint(JDRShape object, JDRPaint paint)
      {
         object_ = object;
         oldpaint_ = object.getFillPaint();
         newpaint_ = paint;

         object_.setFillPaint(newpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         object_.setFillPaint(newpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         object_.setFillPaint(oldpaint_);

         BBox box = object_.getBBox();
         repaint(box.getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.fill_colour");
      }
   }

   class SetLineStyle extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private JDRStroke oldstroke_, newstroke_;

      public SetLineStyle(JDRShape object, JDRStroke stroke)
      {
         object_ = object;
         oldstroke_ = object.getStroke();
         newstroke_ = stroke;

         repaintObject(object_);

         object_.setStroke(newstroke_);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         object_.setStroke(newstroke_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         object_.setStroke(oldstroke_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetLineWidth extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private double oldStyle_, newStyle_;

      public SetLineWidth(JDRShape object, double style)
         throws InvalidPenWidthException
      {
         object_ = object;
         oldStyle_ = ((JDRBasicStroke)object.getStroke()).getPenWidth();
         newStyle_ = style;

         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setPenWidth(style);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setPenWidth(newStyle_);
         }
         catch (InvalidPenWidthException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setPenWidth(oldStyle_);
         }
         catch (InvalidPenWidthException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }


         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetDashPattern extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private DashPattern oldStyle_, newStyle_;

      public SetDashPattern(JDRShape object, DashPattern style)
      {
         object_ = object;
         oldStyle_ = ((JDRBasicStroke)object.getStroke()).dashPattern;
         newStyle_ = style;

         BBox oldbox = object_.getBBox();

         ((JDRBasicStroke)object_.getStroke()).dashPattern=style;

         BBox newbox = object_.getBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = object_.getBBox();

         ((JDRBasicStroke)object_.getStroke()).dashPattern=newStyle_;

         BBox newbox = object_.getBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = object_.getBBox();

         ((JDRBasicStroke)object_.getStroke()).dashPattern=oldStyle_;

         BBox newbox = object_.getBBox();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetCapStyle extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private int oldStyle_, newStyle_;

      public SetCapStyle(JDRShape object, int style)
         throws InvalidCapStyleException
      {
         object_ = object;
         oldStyle_ = ((JDRBasicStroke)object.getStroke()).getCapStyle();
         newStyle_ = style;

         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setCapStyle(style);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setCapStyle(newStyle_);
         }
         catch (InvalidCapStyleException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setCapStyle(oldStyle_);
         }
         catch (InvalidCapStyleException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }


         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetJoinStyle extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private int oldStyle_, newStyle_;

      public SetJoinStyle(JDRShape object, int style)
         throws InvalidJoinStyleException
      {
         object_ = object;
         oldStyle_ = ((JDRBasicStroke)object.getStroke()).getJoinStyle();
         newStyle_ = style;

         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setJoinStyle(style);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setJoinStyle(newStyle_);
         }
         catch (InvalidJoinStyleException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setJoinStyle(oldStyle_);
         }
         catch (InvalidJoinStyleException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetMitreLimit extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private double oldLimit_, newLimit_;

      public SetMitreLimit(JDRShape object, double limit)
         throws InvalidMitreLimitException
      {
         object_ = object;
         oldLimit_ = ((JDRBasicStroke)object.getStroke()).getMitreLimit();
         newLimit_ = limit;

         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setMitreLimit(limit);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setMitreLimit(newLimit_);
         }
         catch (InvalidMitreLimitException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setMitreLimit(oldLimit_);
         }
         catch (InvalidMitreLimitException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }


         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetStartArrow extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private JDRMarker oldMarker_, newMarker_;

      public SetStartArrow(JDRShape object, JDRMarker marker)
         throws InvalidMarkerTypeException
      {
         object_ = object;
         JDRBasicStroke stroke = (JDRBasicStroke)object.getStroke();

         oldMarker_     = stroke.getStartArrow();
         newMarker_     = (JDRMarker)marker.clone();

         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setStartArrow(newMarker_);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setStartArrow(newMarker_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setStartArrow(oldMarker_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetMidArrow extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private JDRMarker oldMarker_, newMarker_;

      public SetMidArrow(JDRShape object, JDRMarker marker)
         throws InvalidMarkerTypeException
      {
         object_ = object;
         JDRBasicStroke stroke = (JDRBasicStroke)object.getStroke();

         oldMarker_     = stroke.getMidArrow();
         newMarker_     = (JDRMarker)marker.clone();

         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setMidArrow(newMarker_);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setMidArrow(newMarker_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setMidArrow(oldMarker_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetEndArrow extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private JDRMarker oldMarker_, newMarker_;

      public SetEndArrow(JDRShape object, JDRMarker marker)
         throws InvalidMarkerTypeException
      {
         object_ = object;
         JDRBasicStroke stroke = (JDRBasicStroke)object.getStroke();

         oldMarker_      = stroke.getEndArrow();
         newMarker_      = (JDRMarker)marker.clone();

         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setEndArrow(newMarker_);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setEndArrow(newMarker_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setEndArrow(oldMarker_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetMarkers extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private JDRMarker newMarker_;
      private JDRMarker oldStartMarker_, oldMidMarker_, oldEndMarker_;

      public SetMarkers(JDRShape object, JDRMarker marker)
         throws InvalidMarkerTypeException
      {
         object_ = object;
         JDRBasicStroke stroke = (JDRBasicStroke)object.getStroke();

         oldStartMarker_ = stroke.getStartArrow();
         oldMidMarker_   = stroke.getMidArrow();
         oldEndMarker_   = stroke.getEndArrow();
         newMarker_      = (JDRMarker)marker.clone();

         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setStartArrow(newMarker_);
         ((JDRBasicStroke)object_.getStroke()).setMidArrow(newMarker_);
         ((JDRBasicStroke)object_.getStroke()).setEndArrow(newMarker_);

         repaintObject(object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setStartArrow(newMarker_);
         ((JDRBasicStroke)object_.getStroke()).setMidArrow(newMarker_);
         ((JDRBasicStroke)object_.getStroke()).setEndArrow(newMarker_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject(object_);

         ((JDRBasicStroke)object_.getStroke()).setStartArrow(oldStartMarker_);
         ((JDRBasicStroke)object_.getStroke()).setMidArrow(oldMidMarker_);
         ((JDRBasicStroke)object_.getStroke()).setEndArrow(oldEndMarker_);

         repaintObject(object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetWindingRule extends AbstractUndoableEdit
   {
      private JDRShape object_;
      private int oldStyle_, newStyle_;

      public SetWindingRule(JDRShape object, int style)
         throws InvalidWindingRuleException
      {
         object_ = object;
         oldStyle_ = ((JDRBasicStroke)object.getStroke()).getWindingRule();
         newStyle_ = style;

         BBox oldbox = object_.getExtent();

         ((JDRBasicStroke)object_.getStroke()).setWindingRule(style);

         BBox newbox = object_.getExtent();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         BBox oldbox = object_.getExtent();

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setWindingRule(newStyle_);
         }
         catch (InvalidWindingRuleException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         BBox newbox = object_.getExtent();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         BBox oldbox = object_.getExtent();

         try
         {
            ((JDRBasicStroke)object_.getStroke()).setWindingRule(oldStyle_);
         }
         catch (InvalidWindingRuleException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         BBox newbox = object_.getExtent();
         repaint(oldbox.add(newbox).getRectangle(getMagnification()));
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.line_style");
      }
   }

   class SetHAlign extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private int oldAlign_, newAlign_;
      private BBox oldBox, newBox;

      public SetHAlign(JDRTextual object, int align)
         throws InvalidHAlignException
      {
         object_ = object;
         oldAlign_ = object.getHAlign();
         newAlign_ = align;

         oldBox = object.getExtent();
         object_.setHAlign(align);
         newBox = object_.getExtent();

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setHAlign(newAlign_);
         }
         catch (InvalidHAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setHAlign(oldAlign_);
         }
         catch (InvalidHAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.halign");
      }
   }

   class SetVAlign extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private int oldAlign_, newAlign_;
      private BBox oldBox, newBox;

      public SetVAlign(JDRTextual object, int align)
         throws InvalidVAlignException
      {
         object_ = object;
         oldAlign_ = object.getVAlign();
         newAlign_ = align;

         oldBox = object.getExtent();
         object_.setVAlign(align);
         newBox = object.getExtent();

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setVAlign(newAlign_);
         }
         catch (InvalidVAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setVAlign(oldAlign_);
         }
         catch (InvalidVAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.halign");
      }
   }

   class SetAnchor extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private int oldHAlign_, newHAlign_;
      private int oldVAlign_, newVAlign_;
      private BBox oldBox, newBox;

      public SetAnchor(JDRTextual object, int halign, int valign)
         throws InvalidVAlignException,InvalidHAlignException
      {
         object_ = object;
         oldHAlign_ = object.getHAlign();
         newHAlign_ = halign;
         oldVAlign_ = object.getVAlign();
         newVAlign_ = valign;

         oldBox = object.getExtent();

         object_.setHAlign(halign);
         object_.setVAlign(valign);

         newBox = object_.getExtent();

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setHAlign(newHAlign_);
            object_.setVAlign(newVAlign_);
         }
         catch (InvalidHAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }
         catch (InvalidVAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setHAlign(oldHAlign_);
            object_.setVAlign(oldVAlign_);
         }
         catch (InvalidHAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }
         catch (InvalidVAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         repaint(oldBox.getRectangle(getMagnification()));
         repaint(newBox.getRectangle(getMagnification()));

         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.anchor");
      }
   }

   class SetTextTransform extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private double[] oldmatrix, newmatrix;

      public SetTextTransform(JDRTextual object, double[] matrix)
      {
         object_ = object;

         oldmatrix = object.getTransformation(null);
         newmatrix = matrix;

         repaintObject((JDRCompleteObject)object_);

         object_.setTransformation(matrix);

         repaintObject((JDRCompleteObject)object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         object_.setTransformation(newmatrix);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         object_.setTransformation(oldmatrix);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.set_transform");
      }
   }

   class SetFont extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private String oldfamily, newfamily;
      private int oldseries, oldshape, oldsize, oldvalign, oldhalign;
      private int newseries, newshape, newsize, newvalign, newhalign;
      private String oldlatexfamily, oldlatexsize, oldlatexseries,
                     oldlatexshape;
      private String newlatexfamily, newlatexsize, newlatexseries,
                     newlatexshape;

      public SetFont(JDRTextual object, String family, int series,
                     int shape, int size, String latexfamily,
                     String latexsize, String latexseries,
                     String latexshape, int halign, int valign)
      throws InvalidFontWeightException,
             InvalidFontSizeException,
             InvalidFontShapeException,
             InvalidHAlignException,
             InvalidVAlignException
      {
         object_ = object;

         oldfamily = object_.getFontFamily();
         oldseries = object_.getFontSeries();
         oldshape  = object_.getFontShape();
         oldsize   = object_.getFontSize();

         oldlatexfamily = object_.getLaTeXFamily();
         oldlatexsize   = object_.getLaTeXSize();
         oldlatexseries = object_.getLaTeXSeries();
         oldlatexshape  = object_.getLaTeXShape();
         oldhalign = object_.getHAlign();
         oldvalign = object_.getVAlign();

         newfamily = family;
         newseries = series;
         newsize   = size;
         newshape  = shape;

         newlatexfamily = latexfamily;
         newlatexsize   = latexsize;
         newlatexseries = latexseries;
         newlatexshape  = latexshape;
         newhalign = halign;
         newvalign = valign;

         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         object_.setFont(g2,
                         newfamily,
                         newseries,
                         newshape,
                         newsize);

         g2.dispose();

         object_.setLaTeXFont(newlatexfamily, 
                          newlatexsize, 
                          newlatexseries,
                          newlatexshape);

         object_.setHAlign(newhalign);
         object_.setVAlign(newvalign);

         repaintObject((JDRCompleteObject)object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
            object_.setFont(g2,
                            newfamily,
                            newseries,
                            newshape,
                            newsize);

            object_.setHAlign(newhalign);
            object_.setVAlign(newvalign);

         }
         catch (InvalidFormatException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         g2.dispose();

         object_.setLaTeXFont(newlatexfamily, 
                          newlatexsize, 
                          newlatexseries,
                          newlatexshape);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
               object_.setFont(g2,
                               oldfamily,
                               oldseries,
                               oldshape,
                               oldsize);

            object_.setHAlign(oldhalign);
            object_.setVAlign(oldvalign);
         }
         catch (InvalidFormatException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         g2.dispose();

         object_.setLaTeXFont(oldlatexfamily, 
                          oldlatexsize, 
                          oldlatexseries,
                          oldlatexshape);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.font");
      }
   }

   class SetFontFamily extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private String oldfamily, newfamily;
      private String oldlatexfamily;
      private String newlatexfamily;

      public SetFontFamily(JDRTextual object, String family, 
                     String latexfamily)
      {
         object_ = object;

         oldfamily = object_.getFontFamily();

         oldlatexfamily = object_.getLaTeXFamily();

         newfamily = family;

         newlatexfamily = latexfamily;

         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         object_.setFontFamily(g2, newfamily);

         g2.dispose();

         object_.setLaTeXFamily(newlatexfamily);

         repaintObject((JDRCompleteObject)object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         object_.setFontFamily(g2, newfamily);

         g2.dispose();

         object_.setLaTeXFamily(newlatexfamily);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         object_.setFontFamily(g2, oldfamily);

         g2.dispose();

         object_.setLaTeXFamily(oldlatexfamily);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.font");
      }
   }

   class SetFontShape extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private int oldShape, newShape;
      private String oldlatexshape;
      private String newlatexshape;

      public SetFontShape(JDRTextual object, int shape, 
                     String latexshape)
      throws InvalidFontShapeException
      {
         object_ = object;

         oldShape = object_.getFontShape();

         oldlatexshape = object_.getLaTeXShape();

         newShape = shape;

         newlatexshape = latexshape;

         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         object_.setFontShape(g2, newShape);

         g2.dispose();

         object_.setLaTeXShape(newlatexshape);

         repaintObject((JDRCompleteObject)object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
            object_.setFontShape(g2, newShape);
         }
         catch (InvalidFormatException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         g2.dispose();

         object_.setLaTeXShape(newlatexshape);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
            object_.setFontShape(g2, oldShape);
         }
         catch (InvalidFontShapeException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         g2.dispose();

         object_.setLaTeXShape(oldlatexshape);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.font");
      }
   }

   class SetFontSeries extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private int oldSeries, newSeries;
      private String oldlatexseries;
      private String newlatexseries;

      public SetFontSeries(JDRTextual object, int series, 
                     String latexseries)
      throws InvalidFontWeightException
      {
         object_ = object;

         oldSeries = object_.getFontSeries();

         oldlatexseries = object_.getLaTeXSeries();

         newSeries = series;

         newlatexseries = latexseries;

         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         object_.setFontSeries(g2, newSeries);

         g2.dispose();

         object_.setLaTeXSeries(newlatexseries);

         repaintObject((JDRCompleteObject)object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
            object_.setFontSeries(g2, newSeries);
         }
         catch (InvalidFontWeightException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         g2.dispose();

         object_.setLaTeXSeries(newlatexseries);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
            object_.setFontSeries(g2, oldSeries);
         }
         catch (InvalidFontWeightException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         g2.dispose();

         object_.setLaTeXSeries(oldlatexseries);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.font");
      }
   }

   class SetFontSize extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private int oldSize, newSize;
      private String oldlatexsize;
      private String newlatexsize;

      public SetFontSize(JDRTextual object, int size, 
                     String latexsize)
      throws InvalidFontSizeException
      {
         object_ = object;

         oldSize = object_.getFontSize();

         oldlatexsize = object_.getLaTeXSize();

         newSize = size;

         newlatexsize = latexsize;

         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         object_.setFontSize(g2, newSize);

         g2.dispose();

         object_.setLaTeXSize(newlatexsize);

         repaintObject((JDRCompleteObject)object_);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
            object_.setFontSize(g2, newSize);
         }
         catch (InvalidFormatException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         g2.dispose();

         object_.setLaTeXSize(newlatexsize);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         repaintObject((JDRCompleteObject)object_);

         Graphics2D g2 = (Graphics2D)getGraphics();
         g2.setRenderingHints(frame_.getRenderingHints());
         double norm = JDRUnit.getNormalizingFactor();
         g2.scale(norm,norm);

         try
         {
            object_.setFontSize(g2, oldSize);
         }
         catch (InvalidFontSizeException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         g2.dispose();

         object_.setLaTeXSize(oldlatexsize);

         repaintObject((JDRCompleteObject)object_);
         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.font");
      }
   }

   class SetLaTeXFontSize extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private String oldlatexsize;
      private String newlatexsize;

      public SetLaTeXFontSize(JDRTextual object, String latexsize)
      {
         object_ = object;

         oldlatexsize = object_.getLaTeXSize();

         newlatexsize = latexsize;

         object_.setLaTeXSize(newlatexsize);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         object_.setLaTeXSize(newlatexsize);

         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         object_.setLaTeXSize(oldlatexsize);

         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.latexsize");
      }
   }

   class SetFontHAlign extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private int oldhalign;
      private int newhalign;

      public SetFontHAlign(JDRTextual object, int halign)
      throws InvalidHAlignException
      {
         object_ = object;

         oldhalign = object_.getHAlign();
         newhalign = halign;

         object_.setHAlign(newhalign);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setHAlign(newhalign);
         }
         catch (InvalidHAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setHAlign(oldhalign);
         }
         catch (InvalidHAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.font");
      }
   }

   class SetFontVAlign extends AbstractUndoableEdit
   {
      private JDRTextual object_;
      private int oldvalign;
      private int newvalign;

      public SetFontVAlign(JDRTextual object, int valign)
      throws InvalidVAlignException
      {
         object_ = object;

         oldvalign = object_.getVAlign();
         newvalign = valign;

         object_.setVAlign(newvalign);

         frame_.markAsModified();
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setVAlign(newvalign);
         }
         catch (InvalidVAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         frame_.markAsModified();
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();

         try
         {
            object_.setVAlign(oldvalign);
         }
         catch (InvalidVAlignException e)
         {
            // this shouldn't happen
            JDRResources.internalError(null, e);
         }

         frame_.markAsModified();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return JDRResources.getString("undo.font");
      }
   }

   class EditPath extends AbstractUndoableEdit
   {
      private JDRShape path_;
      private boolean edited_;
      private int index_;

      public EditPath(JDRShape object, boolean flag)
      {
         object.setEditMode(flag);
         edited_ = object.isEdited();
         path_ = object;

         index_ = 0;

         if (!object.isEdited())
         {
            index_ = editedPath.getSelectedControlIndex();
            finishEditingPath();
         }
         else
         {
            enableTools();
            editedPath = path_;
            editedPath.selectControl(index_);

            breakPathItem.setEnabled(
               editedPath.getSelectedSegment()
             !=editedPath.getLastSegment());
         }

         setBackgroundImage();
         BBox box = path_.getExtent();
         BBox cbox = path_.getControlBBox();

         repaint(box.add(cbox).getRectangle(getMagnification()));
      }

      public void redo() throws CannotRedoException
      {
         frame_.selectThisFrame();
         path_.setEditMode(edited_);

         if (!path_.isEdited())
         {
            index_ = editedPath.getSelectedControlIndex();
            finishEditingPath();
         }
         else
         {
            enableTools();
            editedPath = path_;
            editedPath.selectControl(index_);
            breakPathItem.setEnabled(
               editedPath.getSelectedSegment()
             !=editedPath.getLastSegment());
         }

         frame_.setEditPathButton(path_.isEdited());
         setBackgroundImage();
         BBox box = path_.getExtent();
         BBox cbox = path_.getControlBBox();

         repaint(box.add(cbox).getRectangle(getMagnification()));
      }

      public void undo() throws CannotUndoException
      {
         frame_.selectThisFrame();
         path_.setEditMode(!edited_);

         if (!path_.isEdited())
         {
            index_ = editedPath.getSelectedControlIndex();
            finishEditingPath();
         }
         else
         {
            enableTools();
            editedPath = path_;
            editedPath.selectControl(index_);
            breakPathItem.setEnabled(
               editedPath.getSelectedSegment()
             !=editedPath.getLastSegment());
         }
         frame_.setEditPathButton(path_.isEdited());
         setBackgroundImage();
         BBox box = path_.getExtent();
         BBox cbox = path_.getControlBBox();

         repaint(box.add(cbox).getRectangle(getMagnification()));
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return edited_ ? 
                JDRResources.getString("undo.edit.start") :
                JDRResources.getString("undo.edit.finish");
      }
   }

   class DisplayPageEdit extends AbstractUndoableEdit
   {
      private int _oldPage, _newPage;
      private String string_
         = JDRResources.getString("undo.displaypage");

      public DisplayPageEdit(int page)
      {
         _oldPage = displayPage;
         _newPage = page;
         displayPage = page;
         frame_.updateTitle();
         repaint();
      }

      public void redo() throws CannotRedoException
      {
         displayPage = _newPage;
         frame_.updateTitle();
         repaint();
      }

      public void undo() throws CannotUndoException
      {
         displayPage = _oldPage;
         frame_.updateTitle();
         repaint();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string_;
      }
   }

   class ShowTextField extends AbstractUndoableEdit
   {
      private String string_
         = JDRResources.getString("undo.start_new_text");
      private JDRText textarea_;
      private Point location_;
      private String text_ = "";
      private Font font_, oldFont_;
      private JDRPaint foreground_;

      public ShowTextField(Point2D currentPos)
      {
         Point2D textpos = (Point2D)currentPos.clone();
         currentText = new JDRText(textpos);
         textarea_ = (JDRText)currentText.clone();
         location_ = new Point((int)Math.round(textpos.getX()),
            (int)Math.round(textpos.getY()));
         resetTextField();
         font_ = textFieldFont;
         oldFont_ = textFieldFont;
         foreground_ = textField.getTextPaint();

         text_ = textField.getText();
         textField.setNoUndoText("");
         textField.setPositionY(location_.y);
         textField.setVisible(true);
         textField.requestFocusInWindow();
         frame_.newImage = false;
      }

      public void redo() throws CannotRedoException
      {
         currentText = textarea_;
         textField.setNoUndoText("");
         oldFont_ = textFieldFont;
         setTextFieldFont(font_);
         resetTextField(foreground_, location_);
         textField.setVisible(true);
         textField.requestFocusInWindow();
      }

      public void undo() throws CannotUndoException
      {
         font_ = textFieldFont;
         textField.setNoUndoText(text_);
         setTextFieldFont(oldFont_);
         textField.setVisible(false);
         currentText = null;
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string_;
      }
   }

   class HideTextField extends AbstractUndoableEdit
   {
      private String string_
         = JDRResources.getString("undo.finish_new_text");
      private Point location_;
      private JDRText textarea_ = null;
      private Font font_, oldFont_;
      private JDRPaint foreground_;

      public HideTextField()
      {
         font_ = textFieldFont;
         oldFont_ = textFieldFont;
         foreground_ = textField.getTextPaint();
         if (currentText != null)
         {
            currentText.setTextPaint(foreground_);
            textarea_ = (JDRText)currentText.clone();
            location_ = textarea_.getStart().getPoint();
         }
         else
         {
            location_ = (Point)textField.getPosition().clone();
         }
         textField.setVisible(false);
      }

      public void redo() throws CannotRedoException
      {
         font_ = textFieldFont;
         setTextFieldFont(oldFont_);
         textField.setVisible(false);
      }

      public void undo() throws CannotUndoException
      {
         currentText = textarea_;
         oldFont_ = textFieldFont;
         setTextFieldFont(font_);
         resetTextField(foreground_, location_);
         textField.setVisible(true);
         textField.requestFocusInWindow();
      }

      public boolean canUndo() {return true;}
      public boolean canRedo() {return true;}

      public String getPresentationName()
      {
         return string_;
      }
   }

   public void printInfo(PrintWriter out) throws IOException
   {
      out.println(paths.info());

      out.println("anchor: "+anchor);
      out.println("selected index:"+selectedIndex);
      out.println("drag scale object:"+dragScaleObject);
      out.println("drag scale anchor:"+dragScaleAnchor);
      out.println("drag scale hotspot:"+dragScaleHotspot);
      out.println("drag scale index:"+dragScaleIndex);
      out.println("scanshape:"+scanshape);
      out.println("background image:"+backgroundImage);
      out.println(symbolSelector);
      out.println(movePtDialog);
   }

   /**
    * Gets information on selected objects.
    * @return string containing information about selected objects
    */
   public String getSelectedInfo()
   {
      if (paths == null)
      {
         return "no objects available";
      }

      String str = "";

      boolean done = false;

      for (int i = 0, n = paths.size(); i < n; i++)
      {
         JDRCompleteObject object = paths.get(i);

         if (object.isSelected())
         {
            str += "Object index: "+i+"\n";
            str += object.info()+"\n\n";
            done = true;
         }
      }

      if (!done)
      {
         str = "No objects selected\n";
      }

      return str;
   }

   public void discardImage()
   {
      currentPath=null;
      editedPath = null;
      currentSegment=null;
      cpedit=null;
      anchor=null;
      editedSegmentDash=null;
      scanshape=null;
      count = 0;
      selectedIndex=-1;
      dragScaleIndex=-1;
      dragScaleHotspot=BBox.HOTSPOT_NONE;
      dragScaleAnchor=null;
      paths = null;
   }

   private Point2D mouse;
   private JDRCanvasCompoundEdit cpedit=null;
   private JDRPath currentPath=null;
   private JDRShape editedPath=null;
   private JDRSegment currentSegment=null;
   private Point2D anchor;
   private BBox dragBBox=null;
   private JDRGroup paths;
   private int selectedIndex=-1;
   private JDRCompleteObject dragScaleObject=null;
   private JDRPoint dragScaleAnchor=null;
   private int dragScaleHotspot=BBox.HOTSPOT_NONE;
   private int dragScaleIndex = -1;
   private boolean mouseDown=false;

   private BasicStroke editedSegmentDash 
      = new BasicStroke(1.0f,
                        BasicStroke.CAP_BUTT,
                        BasicStroke.JOIN_MITER,
                        1.0f,new float[]{4.0f, 4.0f},0.0f);

   private static Color currentSegmentColor      = Color.gray;
   private static Color currentPathColor         = Color.gray;
   private static Color currentSegmentPointColor = Color.red;
   private static Color currentPathPointColor    = Color.red;
   private static Color typeblockColor           = Color.lightGray;
   private static Color marginColor              = new Color(230,230,230,100);

   private CanvasTextField textField;
   private Font textFieldFont;
   private Font symbolButtonFont;
   protected JDRText currentText;

   private CharacterSelector symbolSelector;
   private MovePointDialog movePtDialog;

   private JPopupMenu popupMenu, textpopupMenu;
   private JPopupMenu selectPopupMenu, selectTextPopupMenu,
                      selectPathPopupMenu, noneSelectedPopupMenu,
                      selectBitmapPopupMenu, selectTextPathPopupMenu;

   private Parshape scanshape;

   private static int count=0;

   private JDRFrame frame_;

   private JDRMenuItem convertToLineItem, convertToCurveItem,
      convertToMoveItem, breakPathItem, openPathRemoveItem, openPathKeepItem,
      continuousItem, deletePointItem, addPointItem;

   private JMenuItem copyText, cutText;

   private JDRCheckBoxMenuItem anchorSymmetryItem, closeAnchorSymmetryItem,
      hasSymmetryItem;

   private JDRMenuItem textDescriptionItem, pathDescriptionItem,
      descriptionItem, editTextItem, editPathItem, selectAllItem,
      findByDescriptionItem, bitmapDescriptionItem, bitmapPropItem,
      bitmapCopyItem, bitmapCutItem, cutItem, copyItem, pasteItem,
      generalInsertBitmapItem, insertBitmapItem, generalBitmapPropItem,
      nonePasteItem, generalSelectAllItem, generalDeselectAllItem,
      textPathDescriptionItem, editTPTextItem, editTPPathItem,
      textGroupItem, pathGroupItem, textPathGroupItem, bitmapGroupItem,
      generalGroupItem, ungroupItem;

   private JDRRadioButtonMenuItem capButtItem, capRoundItem,
      capSquareItem, generalCapButtItem, generalCapRoundItem,
      generalCapSquareItem, windingEvenOddItem, windingNonZeroItem,
      generalWindingEvenOddItem, generalWindingNonZeroItem;

   private JMenu pathMenu, textMenu, bitmapMenu, justifyMenu,
      symmetryMenu, openPathMenu, closePathMenu;

   private BufferedImage backgroundImage=null;
   private double backgroundImageX=0, backgroundImageY=0;

   // for flowframes
   public static final int 
      PAGES_NONE=0, PAGES_ALL=-1, PAGES_ODD=-2, PAGES_EVEN=-3; 
   private int displayPage=PAGES_ALL;

   // hotspot flags (for when shown)
   private static final short hotspotFlags = BBox.SOUTH
                                           | BBox.SOUTH_EAST
                                           | BBox.EAST
                                           | BBox.NORTH_EAST
                                           | BBox.NORTH_WEST
                                           | BBox.SOUTH_WEST;

}

class CanvasTextField extends JTextField
   implements CaretListener,DocumentListener,
      MouseListener
{
   public CanvasTextField(JDRCanvas c, UndoableEditListener editList)
   {
      super(64);

      canvas = c;

      setOpaque(false);
      setHorizontalAlignment(JTextField.LEADING);

      setBorder(null);
      setMargin(new Insets(0,0,0,0));

      addCaretListener(this);
      getDocument().addDocumentListener(this);
      addMouseListener(this);

      el = editList;
      getDocument().addUndoableEditListener(el);

      updateBounds();

      Action action = new AbstractAction()
      {
         public void actionPerformed(ActionEvent evt)
         {
            moveLeft();
         }
      };

      getInputMap(JComponent.WHEN_FOCUSED).put(
         KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, 0),
         "moveLeft"
         );

      getActionMap().put("moveLeft", action);

      action = new AbstractAction()
      {
         public void actionPerformed(ActionEvent evt)
         {
            moveRight();
         }
      };

      getInputMap(JComponent.WHEN_FOCUSED).put(
         KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, 0),
         "moveRight"
         );

      getActionMap().put("moveRight", action);

      action = new AbstractAction()
      {
         public void actionPerformed(ActionEvent evt)
         {
            shiftLeft();
         }
      };

      getInputMap(JComponent.WHEN_FOCUSED).put(
         KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, InputEvent.SHIFT_MASK),
         "shiftLeft"
         );

      getActionMap().put("shiftLeft", action);

      action = new AbstractAction()
      {
         public void actionPerformed(ActionEvent evt)
         {
            shiftRight();
         }
      };

      getInputMap(JComponent.WHEN_FOCUSED).put(
         KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, InputEvent.SHIFT_MASK),
         "shiftRight"
         );

      getActionMap().put("shiftRight", action);
   }

   public void setNoUndoText(String string)
   {
      if (el != null)
      {
         getDocument().removeUndoableEditListener(el);
         setText(string);
         getDocument().addUndoableEditListener(el);
      }
      else
      {
         setText(string);
      }
   }

   public void mouseClicked(MouseEvent e)
   {
   }

   public void mouseEntered(MouseEvent e)
   {
   }

   public void mouseExited(MouseEvent e)
   {
   }

   public void mousePressed(MouseEvent e)
   {
      repaint();
   }

   public void mouseReleased(MouseEvent e)
   {
      repaint();
   }

   public void moveLeft()
   {
      String selection = getSelectedText();

      int pos = getCaretPosition()-1;

      if (pos < 0)
      {
         pos = 0;
      }

      setCaretPosition(pos);

      if (selection != null)
      {
         repaint();
      }
   }

   public void moveRight()
   {
      String selection = getSelectedText();

      int n = getText().length();

      if (n == 0) return;

      int pos = getCaretPosition()+1;

      if (pos > n)
      {
         pos = n;
      }

      setCaretPosition(pos);

      if (selection != null)
      {
         repaint();
      }
   }

   public void shiftLeft()
   {
      int pos = getCaretPosition()-1;

      if (pos < 0)
      {
         return;
      }

      moveCaretPosition(pos);
   }

   public void shiftRight()
   {
      int n = getText().length();

      if (n == 0) return;

      int pos = getCaretPosition()+1;

      if (pos > n)
      {
         return;
      }

      moveCaretPosition(pos);
   }

   public void caretUpdate(CaretEvent e)
   {
      int dot = e.getDot();
      int mark = e.getMark();

      try
      {
         Rectangle caretArea = modelToView(dot);
         Rectangle bounds = getBounds();

         double x = bounds.getX()+caretArea.getX();
         double y = bounds.getY()+caretArea.getY();
         double w = caretArea.getWidth();
         double h = bounds.getHeight();

         bounds.setBounds((int)x, (int)y, (int)w, (int)h);

         Rectangle canvasVisible = canvas.getVisibleRect();

         double minX = canvasVisible.getX();
         double minY = canvasVisible.getY();
         double maxX = minX+canvasVisible.getWidth();
         double maxY = minY+canvasVisible.getHeight();

         if (!canvasVisible.contains(bounds))
         {
            if (minY < y && y+h <= maxY)
            {
               if (maxX < x
                && x < maxX+canvasVisible.getWidth())
               {
                  canvas.blockScrollRight();
               }
               else if (minX-canvasVisible.getWidth() < x
                     && x+w < minX)
               {
                  canvas.blockScrollLeft();
               }
               else
               {
                  canvas.scrollToPixel(x, y);
               }
            }
            else if (minX < x && x+w <= maxX)
            {
               if (maxY <= y+h && y < maxY+canvasVisible.getHeight())
               {
                  canvas.blockScrollDown();
               }
               else if (minY-canvasVisible.getHeight() < y
                     && y < minY)
               {
                  canvas.blockScrollUp();
               }
               else
               {
                  canvas.scrollToPixel(x, y);
               }
            }
            else
            {
               canvas.scrollToPixel(x, y);
            }
         }
      }
      catch (BadLocationException ble)
      {
         System.out.println(ble);
      }
      catch (NullPointerException npe)
      {
      }

      repaint(getSelectedArea());
   }

   public void changedUpdate(DocumentEvent e)
   {
      canvas.markAsModified();
      updateBounds();
   }

   public void insertUpdate(DocumentEvent e)
   {
      canvas.markAsModified();
      updateBounds();
   }

   public void removeUpdate(DocumentEvent e)
   {
      canvas.markAsModified();
      updateBounds();
   }

   public void setPosition(int px, int py)
   {
      pos.x = px;
      pos.y = py;
      updateBounds();
   }

   public void setPositionY(int py)
   {
      pos.y = py;
      updateBounds();
   }

   public Point getPosition()
   {
      return pos;
   }

   public void setFont(Font f)
   {
      AffineTransform af = new AffineTransform();

      if (canvas != null)
      {
         double scale = canvas.getMagnification();

         af.scale(scale, scale);
      }

      super.setFont(f.deriveFont(af));

      updateBounds();
   }

   public void updateBounds()
   {
      Graphics2D g2 = (Graphics2D)getGraphics();

      if (g2 == null)
      {
         return;
      }

      Dimension dim = getSize();

      if (dim == null) return;

      double scale = canvas.getMagnification();
      g2.setRenderingHints(canvas.getRenderingHints());

      Font font = getFont();
      FontRenderContext frc = g2.getFontRenderContext();
      FontMetrics fm = g2.getFontMetrics(font);

      int height = fm.getHeight();
      int descent = fm.getDescent();

      dim.height = height+descent;

      try
      {
         Caret caret = getCaret();
         Rectangle caretArea = modelToView(caret.getDot());
         dim.width = (int)caretArea.getWidth();
      }
      catch (BadLocationException ble)
      {
      }
      catch (NullPointerException npe)
      {
      }

      String text = getText();
      int n = text.length();

      text += widestChar;

      if (!text.equals(""))
      {
         if (text.startsWith(" "))
         {
            text = widestChar+text.substring(1);
         }

         TextLayout layout = new TextLayout(text, font, frc);
         Rectangle2D bounds = layout.getBounds();
         dim.width += (int)Math.ceil(bounds.getWidth())+1;
      }

      g2.dispose();

      int maxAscent = fm.getMaxAscent();
      int maxDescent = fm.getMaxDescent();

      setBounds((int)Math.round((pos.x)*scale),
                (int)Math.round((pos.y*scale-maxAscent)),
                (int)Math.round(dim.getWidth()),
                (int)Math.round(maxAscent+maxDescent));
   }

   protected void paintComponent(Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;
      Rectangle bounds = getVisibleRect();
      Paint oldPaint = g2.getPaint();
      AffineTransform oldAf = g2.getTransform();

      g2.setPaint(background);
      g2.fill(bounds);

      RenderingHints oldHints = g2.getRenderingHints();

      //double norm = JDRUnit.getNormalizingFactor();
      double factor = canvas.getMagnification();
//      g2.scale(factor, factor);

      TextLayout layout = null;

      if (!getText().equals(""))
      {
         g2.setRenderingHints(canvas.getRenderingHints());

         Font font = getFont();

         FontMetrics fm = g2.getFontMetrics(font);
         int descent = fm.getMaxDescent();

         FontRenderContext frc = g2.getFontRenderContext();
         layout = new TextLayout(getText(), font, frc);
         BBox box = new BBox(layout.getBounds());

         double yoffset = fm.getMaxAscent();

         g2.translate(0, yoffset);

         g2.setPaint(foreground.getPaint(box));

         layout.draw(g2, 0, 0);
         g2.translate(0, -yoffset);
      }

      g2.setRenderingHints(oldHints);

      Caret caret = getCaret();

      g.setColor(getCaretColor());
      caret.paint(g);

      g2.setTransform(oldAf);

      if (layout != null)
      {
         int start = getSelectionStart();
         int end   = getSelectionEnd();

         if (start != end)
         {
            g2.setXORMode(foreground.getColor());

            g2.setColor(getSelectedTextColor());

            g2.fill(getSelectedArea());

            g2.setPaintMode();
         }
      }

      g2.setPaint(oldPaint);

   }

   public Rectangle getSelectedArea()
   {
      int start = getSelectionStart();
      int end   = getSelectionEnd();

      int startX = 0;

      int endX = 0;

      try
      {
         Rectangle startView = modelToView(start);

         startX = startView.x;
      }
      catch (BadLocationException e)
      {
      }

      try
      {
         Rectangle endView = modelToView(end);

         endX = endView.x+endView.width;
      }
      catch (BadLocationException e)
      {
      }

      Rectangle area 
         = new Rectangle(startX, 0, endX-startX, getHeight());

      return area;
   }

   public Color getForeground()
   {
      return null;
   }

   public void setTextPaint(JDRPaint paint)
   {
      foreground = paint;
   }

   public JDRPaint getTextPaint()
   {
      return foreground;
   }

   private Point pos = new Point(0,0);

   private JDRCanvas canvas;
   private Paint background = new Color(255,255,0,40);
   private JDRPaint foreground = new JDRColor();

   private UndoableEditListener el=null;

   // I initially used char, but using a String allows
   // extra flexibility.
   public static String widestChar="M";
}

