// File          : CanvasAction.java
// Description   : Action for path editing etc
// Date          : 2012-03-15
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jpgfdraw;

import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import uk.ac.uea.cmp.nlct.jdr.JDRShape;
import uk.ac.uea.cmp.nlct.jdrresources.*;

public class CanvasAction extends JDRAction
{
   public CanvasAction(JDRCanvas canvas)
   {
      super();
      this.canvas = canvas;
   }

   public void actionPerformed(ActionEvent evt)
   {
      int id = getActionId();

      if (id == -1 && evt instanceof JDRActionEvent)
      {
         id = ((JDRActionEvent)evt).getActionId();
      }

      if (id == -1)
      {
         return;
      }

      int tool = canvas.getCurrentTool();
      int modifiers = evt.getModifiers();

      switch (id)
      {
         case ACTION_TO_LINE:
            canvas.convertToLine();
         return;
         case ACTION_TO_CURVE:
            canvas.convertToCurve();
         return;
         case ACTION_TO_MOVE:
            canvas.convertToMove();
         return;
         case ACTION_NEXT:
            if (tool != JpgfDraw.ACTION_SELECT) return;
            canvas.skipObjOrNextPt();
         return;
         case ACTION_PREV:
            if (tool != JpgfDraw.ACTION_SELECT) return;
            canvas.addNextObjOrPrevPt();
         return;
         case ACTION_BREAK:
            canvas.breakPath();
         return;
         case ACTION_INSERT:
            canvas.insertControlOrSymbol();
         return;
         case ACTION_DELETE_POINT:
            canvas.deleteCurrentControlPoint();
         return;
         case ACTION_SYMMETRY_HAS:
            canvas.setSymmetry();
         return;
         case ACTION_SYMMETRY_ANCHOR:
            canvas.anchorSymmetry();
         return;
         case ACTION_SYMMETRY_CLOSE:
            canvas.closeAnchorSymmetry();
         return;
         case ACTION_CONTINUOUS:
            canvas.makeContinuous();
         return;
         case ACTION_OPEN_REMOVE:
            canvas.openPath(true);
         return;
         case ACTION_OPEN_KEEP:
            canvas.openPath(false);
         return;
         case ACTION_CLOSE_LINE:
            canvas.closePath(JDRShape.CLOSE_LINE);
         return;
         case ACTION_CLOSE_CONT:
            canvas.closePath(JDRShape.CLOSE_CONT);
         return;
         case ACTION_CLOSE_MERGE:
            canvas.closePath(JDRShape.CLOSE_MERGE_ENDS);
         return;
         case ACTION_MOVE:
            if (tool != JpgfDraw.ACTION_SELECT) return;

            canvas.moveControlOrObject();
         return;
         case ACTION_SNAP:
            canvas.snapToGrid();
         return;
         case ACTION_FINISH:
            canvas.finish();
         return;
         case ACTION_ABANDON:
            canvas.abandonPath();
         return;
         case ACTION_SHOW_POPUP:
            canvas.showPopup();
         return;
         case ACTION_CONSTRUCT_CLICK:
            canvas.doConstructMouseClick();
         return;
         case ACTION_MOVE_LEFT:
            canvas.moveLeft(modifiers);
         return;
         case ACTION_MOVE_RIGHT:
            canvas.moveRight(modifiers);
         return;
         case ACTION_MOVE_UP:
            canvas.moveUp(modifiers);
         return;
         case ACTION_MOVE_DOWN:
            canvas.moveDown(modifiers);
         return;
         case ACTION_SCROLL_HOME_UP:
           canvas.scrollHomeUp();
         return;
         case ACTION_SCROLL_HOME_LEFT:
           canvas.scrollHomeLeft();
         return;
         case ACTION_SCROLL_END_DOWN:
           canvas.scrollEndDown();
         return;
         case ACTION_SCROLL_END_RIGHT:
           canvas.scrollEndRight();
         return;
         case ACTION_BLOCK_SCROLL_UP:
           canvas.blockScrollUp();
         return;
         case ACTION_BLOCK_SCROLL_DOWN:
           canvas.blockScrollDown();
         return;
         case ACTION_BLOCK_SCROLL_LEFT:
           canvas.blockScrollLeft();
         return;
         case ACTION_BLOCK_SCROLL_RIGHT:
           canvas.blockScrollRight();
         return;
         case ACTION_DELETE_LAST:
           canvas.deleteLast();
         return;
      }
   }

   public static CanvasAction getFinishAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_FINISH);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getAbandonAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_ABANDON);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getNextAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_NEXT);

      KeyStroke keyStroke = KEY_NEXT;

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getPrevAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_PREV);

      KeyStroke keyStroke = KEY_PREV;

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getDeletePointAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_DELETE_POINT);

      KeyStroke keyStroke = KEY_DELETE;

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getInsertAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_INSERT);

      KeyStroke keyStroke = KEY_ADD;

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getMoveAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_MOVE);

      KeyStroke keyStroke = KEY_MOVE;

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getShowPopupAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_SHOW_POPUP);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_F3, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(KeyEvent.VK_CONTEXT_MENU, 0), id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getConstructClickAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_CONSTRUCT_CLICK);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_F4, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getMoveLeftAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_MOVE_LEFT);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getMoveRightAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_MOVE_RIGHT);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getMoveUpAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_MOVE_UP);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_UP, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getMoveDownAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_MOVE_DOWN);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getScrollHomeUpAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_SCROLL_HOME_UP);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_HOME, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getScrollHomeLeftAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_SCROLL_HOME_LEFT);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_HOME, InputEvent.CTRL_MASK);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getScrollEndDownAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_SCROLL_END_DOWN);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_END, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getScrollEndRightAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_SCROLL_END_RIGHT);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_END, InputEvent.CTRL_MASK);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getBlockScrollDownAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_BLOCK_SCROLL_DOWN);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_DOWN, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getBlockScrollRightAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_BLOCK_SCROLL_RIGHT);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_DOWN,
         InputEvent.CTRL_MASK);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getBlockScrollUpAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_BLOCK_SCROLL_UP);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_UP, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getBlockScrollLeftAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_BLOCK_SCROLL_LEFT);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_UP,
         InputEvent.CTRL_MASK);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public static CanvasAction getDeleteLastAction(JDRCanvas canvas)
   {
      CanvasAction action = new CanvasAction(canvas);

      action.setActionId(ACTION_DELETE_LAST);

      KeyStroke keyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_BACK_SPACE, 0);

      Integer id = new Integer(action.getActionId());

      canvas.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(keyStroke, id);
      canvas.getActionMap().put(id, action);

      return action;
   }

   public JDRMenuItem createNextPointItem()
   {
      JDRMenuItem item = new JDRMenuItem(
         JDRResources.getString("editpath.next"),
         JDRResources.getChar("editpath.next.mnemonic"));
      item.setToolTipText(
         JDRResources.getString("tooltip.next_control"));
      item.setAccelerator(KEY_NEXT);

      item.addActionListener(this);
      item.setActionId(ACTION_NEXT);

      return item;
   }

   public JDRMenuItem createPrevPointItem()
   {
      JDRMenuItem item = new JDRMenuItem(
         JDRResources.getString("editpath.prev"),
         JDRResources.getChar("editpath.prev.mnemonic"));
      item.setToolTipText(
         JDRResources.getString("tooltip.prev_control"));
      item.setAccelerator(KEY_PREV);

      item.addActionListener(this);
      item.setActionId(ACTION_PREV);

      return item;
   }

   public JDRMenuItem createDelPointItem()
   {
      JDRMenuItem item = new JDRMenuItem(
         JDRResources.getString("editpath.delete"),
         JDRResources.getChar("editpath.delete.mnemonic"));
      item.setToolTipText(
         JDRResources.getString("tooltip.delete_control"));
      item.setAccelerator(KEY_DELETE);

      item.addActionListener(this);
      item.setActionId(ACTION_DELETE_POINT);

      return item;
   }

   public JDRMenuItem createAddPointItem()
   {
      JDRMenuItem item = new JDRMenuItem(
         JDRResources.getString("editpath.add"),
         JDRResources.getChar("editpath.add.mnemonic"));
      item.setToolTipText(
         JDRResources.getString("tooltip.add_point"));
      item.setAccelerator(KEY_ADD);

      item.addActionListener(this);
      item.setActionId(ACTION_INSERT);

      return item;
   }

   public JDRMenuItem createConvertToLineItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.line"),
        JDRResources.getChar("editpath.line.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.convert_to_line"));

      item.addActionListener(this);

      item.setActionId(ACTION_TO_LINE);

      return item;
   }

   public JDRMenuItem createConvertToCurveItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.curve"),
        JDRResources.getChar("editpath.curve.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.convert_to_curve"));

      item.addActionListener(this);

      item.setActionId(ACTION_TO_CURVE);

      return item;
   }

   public JDRMenuItem createConvertToMoveItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.move"),
        JDRResources.getChar("editpath.move.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.convert_to_move"));

      item.addActionListener(this);

      item.setActionId(ACTION_TO_MOVE);

      return item;
   }

   public JDRCheckBoxMenuItem createHasSymmetryItem()
   {
      JDRCheckBoxMenuItem item = new JDRCheckBoxMenuItem(
        JDRResources.getString("editpath.symmetry.has"),
        JDRResources.getChar("editpath.symmetry.has.mnemonic"),
        false);

      item.setToolTipText(
        JDRResources.getString("tooltip.symmetry.has"));

      item.addActionListener(this);

      item.setActionId(ACTION_SYMMETRY_HAS);

      return item;
   }

   public JDRCheckBoxMenuItem createSymmetryAnchorItem()
   {
      JDRCheckBoxMenuItem item = new JDRCheckBoxMenuItem(
        JDRResources.getString("editpath.symmetry.join_anchor"),
        JDRResources.getChar("editpath.symmetry.join_anchor.mnemonic"),
        false);

      item.setToolTipText(
        JDRResources.getString("tooltip.symmetry.join_anchor"));

      item.addActionListener(this);

      item.setActionId(ACTION_SYMMETRY_ANCHOR);

      return item;
   }

   public JDRCheckBoxMenuItem createSymmetryCloseItem()
   {
      JDRCheckBoxMenuItem item = new JDRCheckBoxMenuItem(
        JDRResources.getString("editpath.symmetry.close_anchor"),
        JDRResources.getChar("editpath.symmetry.close_anchor.mnemonic"),
        false);

      item.setToolTipText(
        JDRResources.getString("tooltip.symmetry.close_anchor"));

      item.addActionListener(this);

      item.setActionId(ACTION_SYMMETRY_CLOSE);

      return item;
   }

   public JDRMenuItem createContinuousItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.continuous"),
        JDRResources.getChar("editpath.continuous.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.continuous"));

      item.addActionListener(this);

      item.setActionId(ACTION_CONTINUOUS);

      return item;
   }

   public JDRMenuItem createOpenRemoveItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.open.remove_last"),
        JDRResources.getChar("editpath.open.remove_last.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.open_path.remove_last"));

      item.addActionListener(this);

      item.setActionId(ACTION_OPEN_REMOVE);

      return item;
   }

   public JDRMenuItem createOpenKeepItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.open.keep_last"),
        JDRResources.getChar("editpath.open.keep_last.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.open_path.keep_last"));

      item.addActionListener(this);

      item.setActionId(ACTION_OPEN_KEEP);

      return item;
   }

   public JDRMenuItem createCloseLineItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.close.line"),
        JDRResources.getChar("editpath.close.line.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.close_path.line"));

      item.addActionListener(this);

      item.setActionId(ACTION_CLOSE_LINE);

      return item;
   }

   public JDRMenuItem createCloseContItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.close.cont"),
        JDRResources.getChar("editpath.close.cont.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.close_path.cont"));

      item.addActionListener(this);

      item.setActionId(ACTION_CLOSE_CONT);

      return item;
   }

   public JDRMenuItem createCloseMergeItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.close.merge"),
        JDRResources.getChar("editpath.close.merge.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.close_path.merge"));

      item.addActionListener(this);

      item.setActionId(ACTION_CLOSE_MERGE);

      return item;
   }

   public JDRMenuItem createMovePointItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.coordinates"),
        JDRResources.getChar("editpath.coordinates.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.coordinates"));

      item.addActionListener(this);

      item.setActionId(ACTION_MOVE);
      item.setAccelerator(KEY_MOVE);

      return item;
   }

   public JDRMenuItem createSnapItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.snap"),
        JDRResources.getChar("editpath.snap.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.snap"));

      item.addActionListener(this);

      item.setActionId(ACTION_SNAP);

      return item;
   }

   public JDRMenuItem createBreakPathItem()
   {
      JDRMenuItem item = new JDRMenuItem(
        JDRResources.getString("editpath.break"),
        JDRResources.getChar("editpath.break.mnemonic"));

      item.setToolTipText(
        JDRResources.getString("tooltip.break_path"));

      item.addActionListener(this);

      item.setActionId(ACTION_BREAK);

      return item;
   }

   private static KeyStroke KEY_NEXT = KeyStroke.getKeyStroke(KeyEvent.VK_F6, 0);
   private static KeyStroke KEY_PREV = KeyStroke.getKeyStroke(KeyEvent.VK_F6,
      InputEvent.SHIFT_MASK);
   private static KeyStroke KEY_DELETE = KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0);
   private static KeyStroke KEY_ADD = KeyStroke.getKeyStroke(KeyEvent.VK_INSERT, 0);
   private static KeyStroke KEY_MOVE = KeyStroke.getKeyStroke(KeyEvent.VK_F7, 0);

   private JDRCanvas canvas;

   public static final int
     ACTION_TO_LINE         = 0,
     ACTION_TO_CURVE        = 1,
     ACTION_TO_MOVE         = 2,
     ACTION_NEXT            = 3,
     ACTION_PREV            = 4,
     ACTION_BREAK           = 5,
     ACTION_INSERT          = 6,
     ACTION_DELETE_POINT    = 7,
     ACTION_SYMMETRY_HAS    = 8,
     ACTION_SYMMETRY_ANCHOR = 9,
     ACTION_SYMMETRY_CLOSE  = 10,
     ACTION_CONTINUOUS      = 11,
     ACTION_OPEN_REMOVE     = 12,
     ACTION_OPEN_KEEP       = 13,
     ACTION_CLOSE_LINE      = 14,
     ACTION_CLOSE_CONT      = 15,
     ACTION_CLOSE_MERGE     = 16,
     ACTION_MOVE            = 17,
     ACTION_SNAP            = 18,
     ACTION_FINISH          = 19,
     ACTION_ABANDON         = 20,
     ACTION_SHOW_POPUP      = 21,
     ACTION_CONSTRUCT_CLICK = 22,
     ACTION_MOVE_LEFT       = 23,
     ACTION_MOVE_RIGHT      = 24,
     ACTION_MOVE_UP         = 25,
     ACTION_MOVE_DOWN       = 26,
     ACTION_SCROLL_HOME_UP     = 27,
     ACTION_SCROLL_HOME_LEFT   = 28,
     ACTION_SCROLL_END_DOWN    = 29,
     ACTION_SCROLL_END_RIGHT   = 30,
     ACTION_BLOCK_SCROLL_UP    = 31,
     ACTION_BLOCK_SCROLL_DOWN  = 32,
     ACTION_BLOCK_SCROLL_LEFT  = 33,
     ACTION_BLOCK_SCROLL_RIGHT = 34,
     ACTION_DELETE_LAST        = 35;
}
