// File          : NonNegativeLengthPanel.java
// Description   : Panel for specifying a non-negative length
// Date          : 6th February 2006
// Last Modified : 9th June 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdrresources.numfield;

import java.awt.*;
import java.awt.event.*;
import java.text.DecimalFormat;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

import uk.ac.uea.cmp.nlct.jdr.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;

/**
 * Panel for specifying a non-negative length.
 * @author Nicola L C Talbot
 */

public class NonNegativeLengthPanel extends JPanel
   implements ItemListener
{
   public NonNegativeLengthPanel(String label)
   {
      super();
      setLayout(new BoxLayout(this, BoxLayout.X_AXIS));

      sizeLabel = new JLabel(label);
      text      = new NonNegativeDoubleField(0.0F);
      text.setColumns(4);

      //unitBox = new JComboBox<String>(new String[] {"pt", "in", "cm", "bp"}); // Java7
      unitBox = new JComboBox(new String[] {"pt", "in", "cm", "bp"});
      unitBox.addItemListener(this);
      currentUnit = BP;
      setUnit(JDRUnit.bp);

      add(sizeLabel);
      add(text);
      add(unitBox);

      sizeLabel.setLabelFor(text);

      Dimension dim = text.getPreferredSize();
   }

   public NonNegativeLengthPanel(String label, char mnemonic)
   {
      this(label);

      sizeLabel.setDisplayedMnemonic(mnemonic);
   }

   public NonNegativeLengthPanel()
   {
      super();
      setLayout(new BoxLayout(this, BoxLayout.X_AXIS));

      sizeLabel = null;
      text      = new NonNegativeDoubleField(0.0F);
      text.setColumns(4);

      //unitBox = new JComboBox<String>(new String[] {"pt", "in", "cm", "bp"});// Java7
      unitBox = new JComboBox(new String[] {"pt", "in", "cm", "bp"});
      unitBox.addItemListener(this);
      currentUnit = BP;
      setUnit(JDRUnit.bp);

      add(text);
      add(unitBox);
   }

   public NonNegativeLengthPanel(String label, SamplePanel panel)
   {
      this(label);

      text.getDocument().addDocumentListener(
          new TextFieldSampleListener(panel));
   }

   public NonNegativeLengthPanel(String label, char mnemonic, SamplePanel panel)
   {
      this(label, mnemonic);

      text.getDocument().addDocumentListener(
          new TextFieldSampleListener(panel));
   }

   public NonNegativeLengthPanel(SamplePanel panel)
   {
      this();

      text.getDocument().addDocumentListener(
          new TextFieldSampleListener(panel));
   }

   public Document getDocument()
   {
      return text.getDocument();
   }

   public NonNegativeDoubleField getTextField()
   {
      return text;
   }

   public double getValue()
   {
      JDRUnit unit = getUnit();
      return unit.toBp(text.getValue());
   }

   public void setBpValue(double val)
   {
      JDRUnit unit = getUnit();
      if (unit == JDRUnit.bp)
      {
         text.setValue(val);
      }
      else
      {
         text.setValue(unit.fromBp(val));
      }
      text.setCaretPosition(0);
   }

   public void setValue(double val)
   {
      JDRUnit unit = getUnit();
      if (unit == JDRUnit.in)
      {
         currentUnit = IN;
      }
      else if (unit == JDRUnit.cm)
      {
         currentUnit = CM;
      }
      else if (unit == JDRUnit.bp)
      {
         currentUnit = BP;
      }
      else
      {
         currentUnit = PT;
      }

      DecimalFormat df = new DecimalFormat("0.##");
      text.setText(df.format(unit.fromBp(val)));
      text.setCaretPosition(0);
   }

   public void addKeyListener(KeyListener kl)
   {
      text.addKeyListener(kl);
      unitBox.addKeyListener(kl);
   }

   public void setUnit(JDRUnit unit)
   {
      if (unit == JDRUnit.in)
      {
         currentUnit = IN;
      }
      else if (unit == JDRUnit.cm)
      {
         currentUnit = CM;
      }
      else if (unit == JDRUnit.bp)
      {
         currentUnit = BP;
      }
      else
      {
         currentUnit = PT;
      }

      unitBox.setSelectedIndex(currentUnit);
   }

   public JDRUnit getUnit()
   {
      switch (unitBox.getSelectedIndex())
      {
         case PT : return JDRUnit.pt;
         case IN : return JDRUnit.in;
         case CM : return JDRUnit.cm;
         case BP : return JDRUnit.bp;
      }

      return JDRUnit.pt;
   }

   public void itemStateChanged(ItemEvent evt)
   {
      Object source = evt.getSource();

      if (evt.getStateChange() == ItemEvent.SELECTED)
      {
         if (source == unitBox)
         {
            JDRUnit oldUnit = JDRUnit.pt;

            switch (currentUnit)
            {
               case IN:
                  oldUnit = JDRUnit.in;
               break;
               case CM :
                  oldUnit = JDRUnit.cm;
               break;
               case BP :
                  oldUnit = JDRUnit.bp;
               break;
            }

            currentUnit = unitBox.getSelectedIndex();
            JDRUnit unit = getUnit();

            double val = text.getValue();
            DecimalFormat df = new DecimalFormat("0.##");
            double newVal = unit.fromUnit(val, oldUnit);
            text.setText(df.format(newVal));
            text.setCaretPosition(0);
         }
      }
   }

   public void setEnabled(boolean flag)
   {
      if (sizeLabel != null) sizeLabel.setEnabled(flag);
      unitBox.setEnabled(flag);
      text.setEnabled(flag);
   }

   public Dimension getMinimumSize()
   {
      return getPreferredSize();
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "NonNegativeLengthPanel:"+eol;

      str += "   unit box:"+eol;
      str += "      value: "+unitBox.getSelectedItem()+eol;
      str += "      has focus: "+unitBox.hasFocus()+eol;
      str += "   value box:"+eol;
      str += "      value: "+text.getValue()+eol;
      str += "      has focus: "+text.hasFocus()+eol;

      return str;
   }

   private JLabel sizeLabel;
   //private JComboBox<String> unitBox; // Java 7
   private JComboBox unitBox;
   private NonNegativeDoubleField text;
   public static final int PT=0, IN=1, CM=2, BP=3;
   private int currentUnit=0;
}
