// File          : JDRResources.java
// Purpose       : Resources required by JpgfDraw and associated
//                 applications
// Date          : 4th June 2008
// Last Modified : 5th April 2011
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdrresources;

import java.io.*;
import java.net.*;  
import java.beans.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;

import javax.swing.*;
import javax.swing.filechooser.*;
import javax.swing.plaf.basic.*;
import javax.help.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdrresources.filter.*;

/**
 * Resources required by JpgfDraw and JDRView.
 */
public class JDRResources
{
   /**
    * Gets an application icon from the resource icon directory.
    * @param name the name of the icon
    * @return the icon as an ImageIcon
    */
   public static ImageIcon appIcon(String name)
   {
      java.net.URL imgURL 
         = JDRResources.class.getResource("icons/"+name);

      if (imgURL != null)
      {
         return new ImageIcon(imgURL);
      }

      error(getString("error.file_not_found")+": icons/"+name);
      return null;
   } 

   /**
    * Gets an application icon from the resource icon directory.
    * @param name the name of the icon
    * @return the icon as an Image
    */
   public static Image getImage(String name)
   {
      Toolkit tk = Toolkit.getDefaultToolkit();
      return tk.getImage("icons/" + name);
   }

   /**
    * Displays stack trace in a dialog box with the option for the
    * user to continue or quit the application.
    * @param parent the parent for the dialog box
    * @param frameTitle the title for the dialog box
    * @param e the exception with the required stack trace
    */
   public static void displayStackTrace(Component parent,
       String frameTitle, Exception e)
   {
      displayStackTrace(parent, frameTitle, ""+e, e);
   }

   /**
    * Displays stack trace in a dialog box with the option for the
    * user to continue or quit the application.
    * @param parent the parent for the dialog box
    * @param frameTitle the title for the dialog box
    * @param e the exception with the required stack trace
    */
   public static void displayStackTrace(Component parent,
       String frameTitle, String message, Exception e)
   {
      JTabbedPane pane = new JTabbedPane();
      String title = getString("stacktrace.message","Error Message");

      JTextArea messageArea = new JTextArea(10,10);
      messageArea.setEditable(false);
      messageArea.setText(message);
      pane.addTab(title, null, messageArea, title);

      JPanel p2 = new JPanel();

      JTextArea details = new JTextArea(10,20);
      StackTraceElement[] trace = e.getStackTrace();
      String stackTrace = "";
      for (int i = 0, n=trace.length; i < n; i++)
      {
         stackTrace += trace[i]+"\n";
      }
      details.setText(stackTrace);
      details.setEditable(false);

      JScrollPane scrollPane = new JScrollPane(details);
      p2.add(scrollPane, "Center");

      JButton copyButton =
         new JButton(getString("stacktrace.copy","Copy"));
      copyButton.setMnemonic(getChar("stacktrace.copy.mnemonic",'C'));
      copyButton.addActionListener(new CopyAction(details));

      p2.add(copyButton,"South");

      title = getString("stacktrace.details","Details");
      pane.addTab(title, null, p2, title);

      String okayOption = getString("label.okay", "Okay");
      String exitOption = getString("label.quit_without_saving",
         "Quit Without Saving");

      int result = JOptionPane.showOptionDialog(parent, pane,
         frameTitle,
         JOptionPane.YES_NO_OPTION,
         JOptionPane.ERROR_MESSAGE, null,
         new String[] {okayOption, exitOption}, okayOption);

      if (result == JOptionPane.NO_OPTION)
      {
         System.exit(0);
      }
   }

   /**
    * Displays stack trace in a dialog box with the option for the
    * user to continue or quit the application.
    * @param parent the parent for the dialog box
    * @param e the exception with the required stack trace
    */
   public static void internalError(Component parent, Exception e)
   {
      displayStackTrace(parent,
         getString("internal_error.title", "Internal Error"),
         e);
   }

   /**
    * Displays stack trace in a dialog box.
    * @param parent the parent for the dialog box
    * @param message the error message
    * @param e the exception with the required stack trace
    */
   public static void internalError(Component parent,
      String message, Exception e)
   {
      displayStackTrace(parent,
         getString("internal_error.title", "Internal Error"),
         message, e);
   }

   /**
    * Displays a dialog box with the given message and the option
    * for the user to continue or quit the application.
    * @param parent the parent for the dialog box
    * @param message the error message
    */
   public static void internalError(Component parent, String message)
   {
      String okayOption = getString("label.okay", "Okay");
      String exitOption = getString("label.quit_without_saving",
         "Quit Without Saving");

      int result = JOptionPane.showOptionDialog(parent, message,
         getString("internal_error.title","Internal Error"),
         JOptionPane.YES_NO_OPTION,
         JOptionPane.ERROR_MESSAGE, null,
         new String[] {okayOption, exitOption}, okayOption);

      if (result == JOptionPane.NO_OPTION)
      {
         System.exit(0);
      }
   }

   /**
    * Displays a dialog box with the given message and the option
    * for the user to continue or quit the application.
    * @param parent the parent for the dialog box
    * @param message the error message
    */
   public static void internalError(Component parent, String[] message)
   {
      String okayOption = getString("label.okay", "Okay");
      String exitOption = getString("label.quit_without_saving",
         "Quit Without Saving");

      int result = JOptionPane.showOptionDialog(parent, message,
         getString("internal_error.title","Internal Error"),
         JOptionPane.YES_NO_OPTION,
         JOptionPane.ERROR_MESSAGE, null,
         new String[] {okayOption, exitOption}, okayOption);

      if (result == JOptionPane.NO_OPTION)
      {
         System.exit(0);
      }
   }

   /**
    * Displays a dialog box with the given message and the option
    * for the user to continue or quit the application. The dialog's
    * parent is set to null.
    * @param message the error message
    */
   public static void internalError(String message)
   {
      System.err.println(message);
      internalError(null, message);
   }

   /**
    * Displays an error message in a dialog box.
    * @param parent the dialog box's parent
    * @param message the error message
    */
   public static void error(Component parent, String message)
   {
      JOptionPane.showMessageDialog(parent,
      message,
      getString("error.title", "Error"),
      JOptionPane.ERROR_MESSAGE);
   }

   /**
    * Displays an error message in a dialog box.
    * @param parent the dialog box's parent
    * @param message the error message
    */
   public static void error(Component parent, String[] message)
   {
      JOptionPane.showMessageDialog(parent,
      message,
      getString("error.title", "Error"),
      JOptionPane.ERROR_MESSAGE);
   }

   /**
    * Displays an error message in a dialog box. The parent component
    * is set to null.
    * @param message the error message
    */
   public static void error(String message)
   {
      System.err.println(message);
      error(null, message);
   }

   /**
    * Displays an error message in a dialog box. The parent component
    * is set to null.
    * @param message the error message
    */
   public static void error(String[] message)
   {
      for (int i = 0; i < message.length; i++)
         System.err.println(message[i]);

      error(null, message);
   }

   /**
    * Displays the stack trace in a dialog box where the user
    * has the option to continue or quit the application.
    * @param parent the parent component
    * @param e the exception
    */
   public static void error(Component parent, Exception e)
   {
      displayStackTrace(parent,
                        getString("error.title", "Error"),e);
   }

   /**
    * Displays the stack trace in a dialog box where the user
    * has the option to continue or quit the application.
    * @param parent the parent component
    * @param e the exception
    */
   public static void error(Component parent, String message,
      Exception e)
   {
      displayStackTrace(parent,
                        getString("error.title", "Error"),
                        message, e);
   }

   /**
    * Displays message and stack trace in a dialog box and
    * exits the application when the dialog box is dismissed.
    * @param message the error message
    * @param e the exception
    */
   public static void fatalError(String message, Exception e)
   {
      JTabbedPane pane = new JTabbedPane();

      String title = getString("stacktrace.message","Error Message");

      JTextArea messageArea = new JTextArea(10,10);
      messageArea.setEditable(false);
      messageArea.setText(message);
      pane.addTab(title, null, messageArea, title);

      JPanel p2 = new JPanel();

      JTextArea details = new JTextArea(10,20);
      StackTraceElement[] trace = e.getStackTrace();
      String stackTrace = "";
      for (int i = 0, n=trace.length; i < n; i++)
      {
         stackTrace += trace[i]+"\n";
      }
      details.setText(stackTrace);
      details.setEditable(false);

      JScrollPane scrollPane = new JScrollPane(details);
      p2.add(scrollPane, "Center");

      JButton copyButton =
         new JButton(getString("stacktrace.copy","Copy"));
      copyButton.setMnemonic(getChar("stacktrace.copy.mnemonic",'C'));
      copyButton.addActionListener(new CopyAction(details));

      p2.add(copyButton,"South");

      title = getString("stacktrace.details","Details");
      pane.addTab(title, null, p2, title);

      JOptionPane.showMessageDialog(null,
      pane,
      getString("error.fatal", "Fatal Error"),
      JOptionPane.ERROR_MESSAGE);

      System.exit(0);
   }

   /**
    * Displays a warning message in a dialog box.
    * @param parent the dialog's parent
    * @param message the warning message
    */
   public static void warning(Component parent, String[] message)
   {
      JOptionPane.showMessageDialog(parent,
      message,
      getString("warning.title"),
      JOptionPane.WARNING_MESSAGE);
   }

   /**
    * Displays a warning message in a dialog box.
    * @param parent the dialog's parent
    * @param message the warning message
    */
   public static void warning(Component parent, String message)
   {
      JOptionPane.showMessageDialog(parent,
      message,
      getString("warning.title"),
      JOptionPane.WARNING_MESSAGE);
   }

   /**
    * Displays a warning message in a dialog box.
    * @param message the warning message
    */
   public static void warning(String message)
   {
      warning(null, message);
   }

   /**
    * Gets an integer associated with the given key in the
    * resources dictionary.
    */
   public static int getInt(String key)
      throws NumberFormatException
   {
      if (dictionary == null)
      {
         return -1;
      }

      return Integer.parseInt(dictionary.getString(key));
   }

   /**
    * Gets the value associated with the given key, with all
    * occurrences of \1 replaced with value[0], all occurrences
    * of \2 replaced with value[1] etc. If the key is not in
    * the dictionary, the default value is returned (no
    * substitution is performed on the default value).
    * @param key the key identifying the string in the dictionary
    * @param values the replacement values
    * @param defaultValue the default value to use if the
    * dictionary hasn't been initialised or the key is not
    * in the dictionary
    * @return the substituted string or the default value
    */
   public static String getStringWithValue(String key,
      String[] values, String defaultValue)
   {
      if (dictionary == null)
      {
         return defaultValue;
      }

      String str = dictionary.getString(key);

      if (str == null)
      {
         return defaultValue;
      }

      for (int i = 0; i < values.length && i < 9; i++)
      {
         char c = (new String(""+(i+1))).charAt(0);

         str = getReplacementString(str, c, values[i]);
      }

      return str.replace("\\\\", "\\");
   }

   /**
    * Gets the value associated with the given key, with all
    * occurences of \1 replaced with value[0], all occurences
    * of \2 replaced with value[1] etc.
    * @param key the key identifying the string in the dictionary
    * @param values the replacement values
    * @return the substituted string
    */
   public static String getStringWithValue(String key, String[] values)
   {
      String str = getString(key);

      for (int i = 0; i < values.length && i < 9; i++)
      {
         char c = (new String(""+(i+1))).charAt(0);

         str = getReplacementString(str, c, values[i]);
      }

      return str.replace("\\\\", "\\");
   }

   /**
    * Gets the value associated with the given key, with all
    * occurences of \1 replaced with value.
    * @param key the key identifying the string in the dictionary
    * @param value the replacement value
    * @return the substituted string
    */
   public static String getStringWithValue(String key, String value)
   {
      String str = getString(key);

      str = getReplacementString(str, '1', value);

      return str.replace("\\\\", "\\");
   }

   /**
    * Gets the value associated with the given key, with all
    * occurences of \1 replaced with value.
    * @param key the key identifying the string in the dictionary
    * @param value the replacement value
    * @return the substituted string
    */
   public static String getStringWithValue(String key, int value)
   {
      return getStringWithValue(key, ""+value);
   }

   /**
    * Gets the value associated with the given key, with all
    * occurences of \1 replaced with value.
    * @param key the key identifying the string in the dictionary
    * @param value the replacement value
    * @return the substituted string
    */
   public static String getStringWithValue(String key, long value)
   {
      return getStringWithValue(key, ""+value);
   }

   /**
    * Gets the value associated with the given key, with all
    * occurences of \1 replaced with value.
    * @param key the key identifying the string in the dictionary
    * @param value the replacement value
    * @return the substituted string
    */
   public static String getStringWithValue(String key, double value)
   {
      return getStringWithValue(key, ""+value);
   }

   private static String getReplacementString(String str,
      char searchChar, String value)
   {
      String stringSoFar = "";

      while (!str.equals(""))
      {
         int i = str.indexOf('\\');

         if (i == -1 || i == str.length()-1)
         {
            return stringSoFar+str;
         }

         while (str.charAt(i+1) == '\\')
         {
            i++;

            if (i == str.length())
            {
               return stringSoFar+str;
            }
         }

         if (str.charAt(i+1) == searchChar)
         {
            stringSoFar += str.substring(0, i) + value;

            if (i+2 == str.length()) return stringSoFar;

            str = str.substring(i+2);
         }
         else
         {
            i = str.indexOf('\\', i+1);

            if (i == -1)
            {
               return stringSoFar + str;
            }

            stringSoFar += str.substring(0, i);

            str = str.substring(i);
         }
      }

      return stringSoFar;
   }

   /**
    * Gets the string associated with the given key in the
    * resource dictionary or the default value if not found.
    * @param key the key identifying the required string
    * @param defVal the default value
    * @return the required string or default value if not found or 
    * if the dictionary has not been initialised using
    * {@link #initialiseDictionary()}
    */
   public static String getString(String key, String defVal)
   {
      if (dictionary == null)
      {
         return defVal;
      }

      return dictionary.getString(key, defVal);
   }

   /**
    * Gets the string associated with the given key in the
    * resource dictionary. If not found or 
    * if the dictionary has not been initialised using
    * {@link #initialiseDictionary()},
    * the key is returned instead.
    * @param key the key identifying the required string
    * @return the required string or the key if not found
    */
   public static String getString(String key)
   {
      if (dictionary == null) return key;

      return dictionary.getString(key);
   }

   /**
    * Gets the first character of the string associated with the 
    * given key in the resource dictionary, or the default value 
    * if not found or 
    * if the dictionary has not been initialised using
    * {@link #initialiseDictionary()}.
    * @param key the key identifying the required string
    * @param defVal the default value
    * @return the first character of the required string 
    * or default value if not found
    */
   public static char getChar(String key, char defVal)
   {
      if (dictionary == null)
      {
         return defVal;
      }

      return dictionary.getChar(key, defVal);
   }

   /**
    * Gets the first character of the string associated with the 
    * given key in the resource dictionary. if not found or 
    * if the dictionary has not been initialised using
    * {@link #initialiseDictionary()}, the
    * first character of the key is used instead.
    * @param key the key identifying the required string
    * @return the first character of the required string 
    * or the first character of the key if not found
    */
   public static char getChar(String key)
   {
      return getChar(key, key.charAt(0));
   }

   public static boolean isDictInitialised()
   {
      return dictionary != null;
   }

   /**
    * Initialises dictionary using the language resources.
    * @param appClass the application class
    */
   public static void initialiseDictionary(Class<? extends Object> appClass)
   throws IOException
   {
      dictionary = new JDRDictionary(appClass);
   }

   /**
    * Initialises dictionary using the language resources.
    */
   public static void initialiseDictionary()
   throws IOException
   {
      dictionary = new JDRDictionary();
   }
   /**
    * Initialises dictionary.
    * @param dictionaryInputStream the dictionary input stream
    * @param licenceInputStream the licence input stream
    */
   public static void initialiseDictionary(
      InputStream dictionaryInputStream,
      InputStream licenceInputStream)
   throws IOException
   {
      dictionary = new JDRDictionary(dictionaryInputStream,
         licenceInputStream);
   }

   /**
    * Gets user's configuration directory.
    * If the environment variable <TT>JDRSETTINGS</TT> exists
    * and is a directory, then that value is returned. If the
    * environment variable <TT>HOME</TT> exists and contains
    * a directory called <TT>.jpgfdraw</TT> or the file system
    * is able to create the directory <TT>.jpgfdraw</TT> then that 
    * is returned, unless the OS is Windows, in which case 
    * <TT>jpgfdraw-settings</TT> is returned if it can be created.
    * If <TT>HOME</TT> exists and contains a directory
    * called <TT>jpgfdraw-settings</TT> or the file system is
    * able to create that directory, then that directory is returned.
    * If neither <TT>HOME</TT> nor <TT>JDRSETTINGS</TT> are defined
    * then a directory with the user's name is created in JpgfDraw's
    * application directory and that is used.
    * @return path name of user configuration directory or null
    * if configuration directory can't be created
    */
   public static String getUserConfigDir()
   {
      // does JDRSETTINGS exist?

      String jdrsettings = System.getenv("JDRSETTINGS");
      String usersettings=null;
      File file=null;

      if (jdrsettings != null)
      {
         file = new File(jdrsettings);

         if (file.exists() && file.isDirectory())
         {
            try
            {
               return file.getCanonicalPath();
            }
            catch (IOException ignore)
            {
            }
         }
      }

      String dirname = ".jpgfdraw";

      if (System.getProperty("os.name").contains("Win"))
      {
         dirname = "jpgfdraw-settings";
      }

      // does HOME exist?

      String home = System.getProperty("user.home");

      if (home != null)
      {
         file = new File(home);

         if (file.exists() && file.isDirectory())
         {
            file = new File(home, ".jpgfdraw");

            if (file.exists() && file.isDirectory())
            {
               try
               {
                  return file.getCanonicalPath();
               }
               catch (IOException e)
               {
               }
            }

            file = new File(home, "jpgfdraw-settings");

            if (file.exists() && file.isDirectory())
            {
               try
               {
                  return file.getCanonicalPath();
               }
               catch (IOException e)
               {
               }
            }

            file = new File(home, dirname);

            file.mkdir();

            if (file.exists() && file.isDirectory())
            {
               try
               {
                  return file.getCanonicalPath();
               }
               catch (IOException ignore)
               {
               }
            }
         }
      }

      // neither "user.home" nor JDRSETTINGS are defined, so
      // assume settings are in "settings" subdirectory of
      // JpgfDraw's installation directory.

      String dir = "settings";
      String user = System.getenv("USER");
      if (user == null) user = System.getenv("USERNAME");

      // Using URI, so forward slash required regardless of OS

      if (user != null) dir += "/"+user;

      try
      {
         URL url = JDRResources.class.getResource(dir);

         if (url != null)
         {
            String path = url.toURI().getPath();

            file = new File(path);

            if (file.isDirectory())
            {
               usersettings = path;
            }
            else if (file.exists())
            {
               if (dictionary == null)
               {
                  error("'"+file.getCanonicalPath()+"' is not a directory");
               }
               else
               {
                  error(
                  JDRResources.getStringWithValue(
                     "error.not_a_directory",
                     file.getCanonicalPath()));
               }
            }
         }
         else
         {
            url = JDRResources.class.getResource(".");

            if (url != null)
            {
               file = new File(url.toURI().getPath(), dir);

               if ((file.exists() && file.isDirectory())
                 || file.mkdirs())
               {
                  url = JDRResources.class.getResource(dir);

                  if (url != null)
                  {
                     usersettings = url.toURI().getPath();
                  }
               }
               else
               {
                  if (dictionary == null)
                  {
                     error("Can't create config directory '"
                           +file.getCanonicalPath()+"'");
                  }
                  else
                  {
                     error(
                        JDRResources.getStringWithValue(
                           "error.config_cant_create",
                           file.getCanonicalPath()));
                  }
               }
            }
         }
      }
      catch (IOException e)
      {
         JDRResources.error(null, e);
         e.printStackTrace();
      }
      catch (URISyntaxException e)
      {
         JDRResources.error(null, e);
         e.printStackTrace();
      }

      return usersettings;
   }

   // Added 2011/04/05
   public static void debugMessage(String message)
   {
      if (debugMode)
      {
         System.err.println("debug message: "+message);
      }
   }

   private static URL getHelpSetLocation(String appName)
     throws IOException
   {
      String helpsetLocation = "/resources/helpsets/";

      String hsLocation;

      URL hsURL;

      if (helpLocaleId != null)
      {
         hsLocation = helpsetLocation+appName+"/"+helpLocaleId
              +"/"+appName+"-"+helpLocaleId+".hs";

         hsURL = JDRResources.class.getResource(hsLocation);

         if (hsURL == null)
         {
            warning("Can't find helpset for language '"+helpLocaleId+"'");
            helpLocaleId = null;
         }

         return hsURL;
      }

      Locale locale = Locale.getDefault();

      String localeId = locale.getLanguage()+"-"+locale.getCountry();

      helpLocaleId = localeId;

      hsLocation = helpsetLocation+appName+"/"+localeId
        +"/"+appName+"-"+localeId+".hs";

      hsURL = JDRResources.class.getResource(hsLocation);

      if (hsURL == null)
      {
         String tried = hsLocation;

         helpLocaleId = locale.getLanguage();

         hsLocation = helpsetLocation+appName+"/"+helpLocaleId
           +"/"+appName+"-"+helpLocaleId+".hs";

         hsURL = JDRResources.class.getResource(hsLocation);

         if (hsURL == null)
         {
            tried += "\n"+hsLocation;

            if (!localeId.equals("en-GB"))
            {
               helpLocaleId = "en-GB";

               hsLocation = helpsetLocation+appName+"/"+helpLocaleId
                  +"/"+appName+"-"+helpLocaleId+".hs";

               hsURL = JDRResources.class.getResource(hsLocation);

               if (hsURL == null)
               {
                  tried += "\n"+hsLocation;
               }
            }

            if (hsURL == null)
            {
               throw new IOException("Can't find helpset. Tried:\n"+tried);
            }
         }
      }

      return hsURL;
   }

   public static String[] getAvailableDictLanguages()
   {
      URL url = JDRResources.class.getResource("/resources/dictionaries/");

      File parent;

      try
      {
         parent = new File(url.toURI());
      }
      catch (URISyntaxException e)
      {
         // this shouldn't happen!

         e.printStackTrace();
         return new String[] {"en-GB"};
      }

      File[] files = parent.listFiles(dictionaryFilter);

      if (files == null)
      {
         debugMessage("no dictionaries found");
         return new String[] {"en-GB"};
      }

      String[] lang = new String[files.length];

      for (int i = 0; i < files.length; i++)
      {
         String name = files[i].getName();

         lang[i] = name.substring(name.indexOf("-")+1, name.lastIndexOf("."));
      }

      return lang;
   }

   public static String[] getAvailableHelpLanguages(String appName)
   {
      URL url = JDRResources.class.getResource("/resources/helpsets/"+appName);

      File parent;

      try
      {
         parent = new File(url.toURI());
      }
      catch (URISyntaxException e)
      {
         // this shouldn't happen!

         e.printStackTrace();
         return new String[] {"en-GB"};
      }

      File[] files = parent.listFiles(directoryFilter);

      if (files == null)
      {
         debugMessage("no dictionaries found");
         return new String[] {"en-GB"};
      }

      String[] lang = new String[files.length];

      for (int i = 0; i < files.length; i++)
      {
         lang[i] = files[i].getName();
      }

      return lang;
   }

   public static void initialiseHelp(JFrame parent, String appName)
   {
      if (mainHelpBroker == null)
      {
         HelpSet mainHelpSet = null;

         try
         {
            URL hsURL = getHelpSetLocation(appName);

            mainHelpSet = new HelpSet(null, hsURL);
         }
         catch (Exception e)
         {
            error(parent, e);
         }

         if (mainHelpSet != null)
         {
            mainHelpBroker = mainHelpSet.createHelpBroker();
         }

         if (mainHelpBroker != null)
         {
            csh = new CSH.DisplayHelpFromSource(mainHelpBroker);
         }
      }
   }

   public static void enableHelpOnButton(AbstractButton button, String id)
   {
      if (mainHelpBroker != null)
      {
         try
         {
            mainHelpBroker.enableHelpOnButton(button, id,
               mainHelpBroker.getHelpSet());

            csh = new CSH.DisplayHelpFromSource(mainHelpBroker);

            button.registerKeyboardAction(csh,
               button.getActionCommand(), 
               KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0),
               JComponent.WHEN_IN_FOCUSED_WINDOW);
         }
         catch (BadIDException e)
         {
            internalError(null, e);
         }
      }
      else
      {
         internalError(getString("internal_error.no_helpset"));
      }
   }

   public static JMenuItem addHelpItem(JMenu helpM, String appName)
   {
      JMenuItem helpItem = new JMenuItem(
         getStringWithValue("help.handbook", appName));

      helpM.add(helpItem);
      helpItem.setAccelerator(
         KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0));

      helpItem.setMnemonic(getChar("help.handbook.mnemonic"));

      if (csh != null)
      {
         helpItem.addActionListener(csh);
      }
      else
      {
         internalError(getString("internal_error.no_helpset"));
      }

      return helpItem;
   }

   public static JDRButton createMainHelpButton()
   {
      return createMainHelpButton(getString("help.label"));
   }

   public static JDRButton createMainHelpButton(String tooltipText)
   {
      JDRButton helpButton = createAppButton("help", csh, null, 
         tooltipText);

      return helpButton;
   }

   public static JDRButton createHelpButton(String id)
   {
      return createHelpButton(id, getString("help.label"));
   }

   public static JDRButton createHelpButton(String id, String tooltipText)
   {
      JDRButton helpButton = createAppButton("help", null, null, 
         tooltipText);

      enableHelpOnButton(helpButton, id);

      return helpButton;
   }

   public static JDRButton createAppButton(String name, 
      ActionListener listener, KeyStroke keyStroke, String tooltipText)
   {
      JDRButton button = new JDRButton(
         appIcon(name+"up.png"),
         appIcon(name+".png"),
         appIcon(name+"r.png"),
         listener,
         tooltipText);

      java.net.URL imgURL = JDRResources.class.getResource(
         "icons/"+name+"dis.png");

      if (imgURL != null)
      {
         button.setDisabledIcon(new ImageIcon(imgURL));
      }

      button.setActionCommand(name);

      if (keyStroke != null)
      {
         button.registerKeyboardAction(listener, name, keyStroke,
            JComponent.WHEN_IN_FOCUSED_WINDOW);
      }

      return button;
   }

   public static JDRToggleButton createToggleButton(String name, 
      ActionListener listener, KeyStroke keyStroke, String tooltipText)
   {
      JDRToggleButton button = new JDRToggleButton(
         appIcon(name+"up.png"),
         appIcon(name+".png"),
         appIcon(name+"r.png"),
         listener,
         tooltipText);

      java.net.URL imgURL = JDRResources.class.getResource(
         "icons/"+name+"dis.png");

      if (imgURL != null)
      {
         button.setDisabledIcon(new ImageIcon(imgURL));
      }

      button.setActionCommand(name);

      if (keyStroke != null)
      {
         button.registerKeyboardAction(listener, name, keyStroke,
            JComponent.WHEN_IN_FOCUSED_WINDOW);
      }

      return button;
   }

   public static JDRToolButton createToolButton(String name, 
      ActionListener listener, KeyStroke keyStroke,
      ButtonGroup g, boolean selected, String tooltipText)
   {
      JDRToolButton button = new JDRToolButton(
         appIcon(name+"up.png"),
         appIcon(name+".png"),
         appIcon(name+"r.png"),
         listener, g, selected,
         tooltipText);

      g.add(button);

      java.net.URL imgURL = JDRResources.class.getResource(
         "icons/"+name+"dis.png");

      if (imgURL != null)
      {
         button.setDisabledIcon(new ImageIcon(imgURL));
      }

      button.setActionCommand(name);

      if (keyStroke != null)
      {
         button.registerKeyboardAction(listener, name, keyStroke,
            JComponent.WHEN_IN_FOCUSED_WINDOW);
      }

      return button;
   }

   public static JDRButton createOkayButton(ActionListener listener)
   {
      return createOkayButton(listener, getString("label.okay"));
   }

   public static JDRButton createOkayButton(ActionListener listener, 
     String tooltipText)
   {
      JDRButton button = createAppButton("okay", listener,
         KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
         tooltipText);

      button.registerKeyboardAction(listener, "okay", 
         KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.SHIFT_MASK),
         JComponent.WHEN_IN_FOCUSED_WINDOW);

      return button;
   }

   public static JDRButton createCancelButton(ActionListener listener)
   {
      return createCancelButton(listener, getString("label.cancel"));
   }

   public static JDRButton createCancelButton(ActionListener listener,
      String tooltipText)
   {
      return createAppButton("cancel", listener,
         KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
         tooltipText);
   }

   public static JDRButton createCloseButton(ActionListener listener)
   {
      return createCloseButton(listener, getString("label.close"));
   }

   public static JDRButton createCloseButton(ActionListener listener, 
      String tooltipText)
   {
      JDRButton button = new JDRButton(
         appIcon("cancelup.png"),
         appIcon("cancel.png"),
         appIcon("cancelr.png"),
         listener,
         tooltipText);

      button.setActionCommand("close");

      button.registerKeyboardAction(listener, "close", 
         KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
         JComponent.WHEN_IN_FOCUSED_WINDOW);

      return button;
   }

   public static JMenuItem createAppMenuItem(String id, 
      String actionName, KeyStroke keyStroke, ActionListener listener,
      String tooltipText)
   {
      JMenuItem item = new JMenuItem(getString(id),
         getChar(id+".mnemonic"));

      if (keyStroke != null)
      {
         item.setAccelerator(keyStroke);
      }

      if (listener != null)
      {
         item.addActionListener(listener);
      }

      if (actionName != null)
      {
         item.setActionCommand(actionName);
      }

      if (tooltipText != null)
      {
         item.setToolTipText(tooltipText);
      }

      return item;
   }

   public static JCheckBoxMenuItem createToggleMenuItem(String id, 
      String actionName, KeyStroke keyStroke, ActionListener listener,
      String tooltipText)
   {
      JCheckBoxMenuItem item = new JCheckBoxMenuItem(getString(id));

      item.setMnemonic(getChar(id+".mnemonic"));

      if (keyStroke != null)
      {
         item.setAccelerator(keyStroke);
      }

      if (listener != null)
      {
         item.addActionListener(listener);
      }

      if (actionName != null)
      {
         item.setActionCommand(actionName);
      }

      if (tooltipText != null)
      {
         item.setToolTipText(tooltipText);
      }

      return item;
   }

   public static JRadioButtonMenuItem createToolMenuItem(String id, 
      String actionName, KeyStroke keyStroke, ButtonGroup group,
      boolean selected, ActionListener listener, String tooltipText)
   {
      JRadioButtonMenuItem item = new JRadioButtonMenuItem(getString(id),
         selected);

      group.add(item);

      item.setMnemonic(getChar(id+".mnemonic"));

      if (keyStroke != null)
      {
         item.setAccelerator(keyStroke);
      }

      if (listener != null)
      {
         item.addActionListener(listener);
      }

      if (actionName != null)
      {
         item.setActionCommand(actionName);
      }

      if (tooltipText != null)
      {
         item.setToolTipText(tooltipText);
      }

      return item;
   }

   private static JDRDictionary dictionary;

   private static HelpBroker mainHelpBroker = null;
   private static CSH.DisplayHelpFromSource csh = null;

   public static String dictLocaleId, helpLocaleId;

   public static boolean debugMode = false;

   private static DirectoryFilter directoryFilter = new DirectoryFilter();
   private static DictionaryFilter dictionaryFilter = new DictionaryFilter();
}

class DirectoryFilter implements java.io.FilenameFilter
{
   public boolean accept(File dir, String name)
   {
      return (new File(dir, name)).isDirectory();
   }
}

class DictionaryFilter implements java.io.FilenameFilter
{
   public boolean accept(File dir, String name)
   {
      return name.contains("-") && name.endsWith(".prop");
   }
}
