// File          : JDRGuiMessage.java
// Purpose       : GUI message system
// Date          : 12th June 2008
// Last Modified : 12th June 2008
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdrresources;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * GUI message system. .
 * @author Nicola L C Talbot
 */

public class JDRGuiMessage extends Thread implements JDRMessage,ActionListener
{
   public JDRGuiMessage()
   {
      messageDialog = new JDialog();
      messageDialog.setTitle(
         JDRResources.getString("message.title", "Messages"));
      messageArea = new JTextArea();
      messageArea.setText("");
      messageArea.setEditable(false);
      messageArea.setBackground(Color.white);

      messageScrollPane = new JScrollPane(messageArea);


      messageDialog.getContentPane().add(messageScrollPane, "Center");

      progressBar = new JProgressBar();

      messageDialog.getContentPane().add(progressBar, "North");

      JPanel buttonPanel = new JPanel();

      buttonPanel.add(JDRResources.createCloseButton(this));

      messageArea.getInputMap(JComponent.WHEN_FOCUSED)
         .put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
              "close");

      messageDialog.getContentPane().add(buttonPanel, "South");

      int width = 400;
      int height = 200;

      messageDialog.setSize(width,height);

      Toolkit tk = Toolkit.getDefaultToolkit();
      Dimension d = tk.getScreenSize();
      int screenHeight = d.height;
      int screenWidth  = d.width;

      int x = (screenWidth-width)/2;
      int y = (screenHeight-height)/2;

      messageDialog.setLocation(x, y);

      state = IDLE;

      start();
   }

   public void run()
   {
      try
      {
         while (true)
         {
            sleep(100);

            switch (state)
            {
               case SHOW :
                  messageDialog.setVisible(true);
                  state = IDLE;
                  break;
               case HIDE :
                  messageDialog.setVisible(false);
                  state = IDLE;
                  break;
            }
         }
      }
      catch (InterruptedException e)
      {
      }
   }

   public void actionPerformed(ActionEvent evt)
   {
      String action = evt.getActionCommand();

      if (action == null) return;

      if (action.equals("close"))
      {
         state = HIDE;
      }
   }

   /**
    * Displays message dialogue if not suspended.
    */
   public void displayMessages()
   {
      state = HIDE;
   }

   /**
    * Hides message dialogue if not suspended.
    */
   public void hideMessages()
   {
      state = HIDE;
   }

/*
   public void suspend()
   {
      suspended = true;
   }

   public void resume()
   {
      suspended = false;
   }
*/

   public void setIndeterminate(boolean indeterminate)
   {
      progressBar.setIndeterminate(indeterminate);
   }

   public boolean isIndeterminate()
   {
      return progressBar.isIndeterminate();
   }

   public void incrementProgress()
   {
      setProgress(progressBar.getValue()+1);
   }

   public void setMaxProgress(int maxValue)
   {
      progressBar.setMaximum(maxValue);
   }

   public int getMaxProgress()
   {
      return progressBar.getMaximum();
   }

   public void resetProgress(int maxValue)
   {
      progressBar.setValue(0);
      progressBar.setMaximum(maxValue);
   }

   public void resetProgress()
   {
      progressBar.setValue(0);
   }

   public int getProgress()
   {
      return progressBar.getValue();
   }

   public void setProgress(int value)
   {
      progressBar.setValue(value);
   }

   public void message(String messageText)
   {
      messageArea.append(messageText);
   }

   public void message(Exception excp)
   {
      messageArea.append(excp.getMessage());
   }

   public void messageln(String messageText)
   {
      messageArea.append(messageText+"\n");
   }

   public void messageln(Exception excp)
   {
      messageArea.append(excp.getMessage()+"\n");
   }

   public void warning(String messageText)
   {
      JDRResources.warning(messageText);
   }

   public void warning(Exception excp)
   {
      JDRResources.warning(excp.getMessage());
   }

   public void error(String messageText)
   {
      JDRResources.error(null, messageText);
   }

   public void error(Exception excp)
   {
      JDRResources.error(null, excp);
   }

   public void internalerror(String messageText)
   {
      JDRResources.internalError(null, messageText);
   }

   public void internalerror(Exception excp)
   {
      JDRResources.internalError(null, excp);
   }

   public void fatalerror(String messageText)
   {
      JDRResources.error(null, messageText);
      System.exit(0);
   }

   public void fatalerror(Exception excp)
   {
      JDRResources.fatalError(excp.getMessage(), excp);
   }

   private JProgressBar progressBar;

   private JDialog messageDialog;
   private JTextArea messageArea;
   private JScrollPane messageScrollPane;

   private boolean suspended;

   private int state;

   private static final int IDLE = 0;
   private static final int SHOW = 1;
   private static final int HIDE = 2;
}
