// File          : JDRDictionary.java
// Purpose       : Dictionary containing language resources
// Date          : 4th June 2008
// Last Modified : 4th June 2008
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdrresources;

import java.io.*;
import java.beans.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;

import javax.swing.*;
import javax.swing.filechooser.*;
import javax.swing.plaf.basic.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdrresources.filter.*;

/**
 * Dictionary containing JpgfDraw's language resources.
 * @author Nicola L C Talbot
 */

public class JDRDictionary
{
   /**
    * Creates a new instance of the language set.
    * @param appClass the application class
    */
   public JDRDictionary(Class<? extends Object> appClass)
      throws IOException
   {
      InputStream dictionaryInputStream 
         = getDictionaryInputStream(appClass);;

      InputStream licenceInputStream 
         = appClass.getResourceAsStream(
            "/resources/LICENSE");

      if (licenceInputStream == null)
      {
         throw new IOException(
            "Unable to open 'resources/LICENSE'");
      }

      loadInputStreams(dictionaryInputStream,
            licenceInputStream);
   }

   private static InputStream getDictionaryInputStream(Class<? extends Object> appClass)
      throws IOException
   {
      String name = "/resources/dictionaries/jpgfdraw";

      InputStream dictionaryInputStream;
      String filename;

      if (JDRResources.dictLocaleId != null)
      {
         filename = name+"-"+JDRResources.dictLocaleId+".prop";

         dictionaryInputStream 
            = appClass.getResourceAsStream(filename);

         if (dictionaryInputStream == null)
         {
            JDRResources.warning("No dictionary available for language '"
               +JDRResources.dictLocaleId+"'");
            JDRResources.dictLocaleId = null;
         }

         return dictionaryInputStream;
      }

      Locale locale = Locale.getDefault();

      String id = locale.getLanguage()+"-"+locale.getCountry();

      filename = name+"-"+id+".prop";

      JDRResources.dictLocaleId = id;

      dictionaryInputStream 
         = appClass.getResourceAsStream(filename);

      if (dictionaryInputStream == null)
      {
         String tried = filename;

         JDRResources.dictLocaleId = locale.getLanguage();

         filename = name+"-"+JDRResources.dictLocaleId+".prop";

         dictionaryInputStream 
           = appClass.getResourceAsStream(filename);

         if (dictionaryInputStream == null)
         {
            tried += "\n"+filename;

            if (!id.equals("en-GB"))
            {
               JDRResources.dictLocaleId = "en-GB";

               filename = name+"-"+JDRResources.dictLocaleId+".prop";

               dictionaryInputStream 
                 = appClass.getResourceAsStream(filename);

               if (dictionaryInputStream == null)
               {
                  tried += "\n"+filename;
               }
            }

            if (dictionaryInputStream == null)
            {
               throw new IOException(
                 "Unable to open dictionary file. Tried: \n"+tried);
            }
         }
      }

      return dictionaryInputStream;
   }

   /**
    * Creates a new instance of the language set.
    */
   public JDRDictionary()
      throws IOException
   {
      this(JDRResources.class);
   }

   /**
    * Creates a new instance of the language set. The 
    * dictionary is loaded from the dictionary input stream.
    * The licence text is loaded from the licence input stream
    * and added to the dictionary with the key "licence".
    * @param dictionaryStream the dictionary input stream
    * @param licenceStream the licence input stream
    */
   public JDRDictionary(InputStream dictionaryStream,
      InputStream licenceStream)
      throws IOException
   {
      loadInputStreams(dictionaryStream, licenceStream);
   }

   private void loadInputStreams(InputStream dictionaryStream,
      InputStream licenceStream)
      throws IOException
   {
      BufferedReader reader;

      dictionary = null;

      reader = new BufferedReader(
         new InputStreamReader(dictionaryStream));

      initDictionary(reader);

      reader.close();

      reader = new BufferedReader(
         new InputStreamReader(licenceStream));

      initLicence(reader);

      reader.close();
   }

   private void initDictionary(BufferedReader reader)
      throws IOException
   {
      dictionary = new Hashtable<String,String>(1024);

      String s;

      while ((s=reader.readLine()) != null)
      {
         StringTokenizer t = new StringTokenizer(s, "#");
         
         if (!t.hasMoreTokens()) continue;

         s = t.nextToken();

         t = new StringTokenizer(s, "=");

         if (!t.hasMoreTokens()) continue;

         String key = t.nextToken();

         String value = "";

         if (t.hasMoreTokens())
         {
            value = t.nextToken();
         }

         int idx;
         int lastIdx=0;

         while((idx = value.indexOf("\\n", lastIdx)) != -1)
         {
            if (idx == 0)
            {
               s = value.substring(2);
            }
            else
            {
               s = value.substring(0,idx)+"\n"
                 + value.substring(idx+2);
            }

            value = s;

            lastIdx = idx;
         }

         dictionary.put(key, value);
      }
   }

   private void initLicence(BufferedReader reader)
      throws IOException
   {
      String licenceText = "";

      String s;

      while ((s=reader.readLine()) != null)
      {
         licenceText += s+"\n";
      }

      if (dictionary != null)
      {
         dictionary.put("licence", licenceText);
      }
   }

   /**
    * Gets the string from the dictionary associated with the 
    * given key. If the key is not found, the key is returned
    * instead.
    * @param key the key identifying the required string
    * @return the string identified by the key if found, or 
    * the key if the dictionary doesn't contain the key
    */
   public String getString(String key)
   {
      if (dictionary == null)
      {
         return key;
      }

      String s = (String)dictionary.get(key);

      if (s == null)
      {
         s = key;
         System.err.println("Unknown key '"+key+"'");
      }

      return s;
   }

   /**
    * Gets the string associated with the given key. If the key
    * is not found, the given default value is returned instead.
    * @param key the key identifying the required string
    * @param defVal the default value to return in the event the
    * entry is not found
    * @return the string associated with the given key, or the
    * default value if not found
    */
   public String getString(String key, String defVal)
   {
      if (dictionary == null)
      {
         return defVal;
      }

      String s = (String)dictionary.get(key);

      if (s == null)
      {
         s = defVal;
      }

      return s;
   }

   /**
    * Gets the first character of the string identified by the
    * given key.
    * @param key the key identifying the required string
    * @return the first character of the string identified by the
    * key, or the first character of the key if this dictionary
    * doesn't contain the key
    */
   public char getChar(String key)
   {
      String s = getString(key);
      return s.charAt(0);
   }

   /**
    * Gets the first character of the string identified by the
    * given key or the default value if not found.
    * @param key the key identifying the required string
    * @param defVal the default value if not found
    * @return the first character of the string identified by the
    * key, or the default value if this dictionary
    * doesn't contain the key
    */
   public char getChar(String key, char defVal)
   {
      if (dictionary == null)
      {
         return defVal;
      }

      String s = getString(key);

      if (s == null) return defVal;

      return s.charAt(0);
   }

   /**
    * Determines if this dictionary contains the given key.
    * @param key the key to check
    * @return true if this dictionary contains the given key
    */
   public boolean containsKey(String key)
   {
      if (dictionary == null)
      {
         return false;
      }

      return dictionary.containsKey(key);
   }

   private Hashtable<String,String> dictionary;
}

