// File          : JDRMarker.java
// Date          : 28th September 2006
// Last Modified : 27th April 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;

import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Generic vertex marker.
 * <p>
 * Markers may appear at the start, mid or end vertices of
 * a path. When a marker is drawn, it needs the segment to
 * which it is attached in order to correctly align the
 * marker so that its origin coincides with the vertex. 
 * The mid and end markers are always drawn at the
 * end of the given segment. The start markers are reflected
 * and drawn at the start of the segment. (See 
 * <a href="#fig1">Figure 1</a>.)
 * <p>
 * <center>
 * <table width=60%>
 * <tr align=center><td colspan=2><a name="fig1"></a>
 * <img src="../images/basicMarker.png" alt="[image illustrating basic marker shape]"></td></tr>
 * <tr><th valign=top>Figure&nbsp;1:</th>
 * <td>(a) basic marker shape (for LaTeX style arrow 
 * marker defined by {@link ArrowSingle}), (b) marker appearance
 * as mid or end marker, (c) marker appearance as a start marker.
 * </td></tr>
 * </table>
 * </center>
 * <p>
 * A marker has a basic shape which is given by 
 * {@link #getGeneralPath()}, but the shape of the marker when
 * it is drawn may be a reflection of this shape, or may 
 * repeat this shape. (See <a href="#fig2">Figure 2</a>.)
 * <p>
 * <center>
 * <table width=60%>
 * <tr align=center><td colspan=2><a name="fig2"></a>
 * <img src="../images/basicMarker2.png" alt="[image illustrating variations of marker shape]"></td></tr>
 * <tr><th valign=top>Figure&nbsp;2:</th>
 * <td>(a) basic marker shape (for LaTeX style arrow 
 * marker defined by {@link ArrowSingle}), (b) marker appearance
 * as mid or end marker where the marker has been reversed, 
 * (c) marker appearance where the marker has a repeat factor
 * of 2, (d) marker appearance where the marker has a repeat
 * factor of 2 and is reversed.</td></tr>
 *</table>
 * </center>
 * <p>
 * A marker may even be combined with a 
 * secondary marker. This secondary marker may be overlaid on top
 * of the primary marker so that both the marker origins coincide
 * with the vertex, or the secondary marker may be offset. (See
 * <a href="#fig3">Figure 3</a>.)
 * <p>
 * <center>
 * <table width=60%>
 * <tr align=center><td colspan=2><a name="fig3"></a>
 * <img src="../images/basicMarker3.png" alt="[image illustrating composite markers]"></td></tr>
 * <tr><th valign=top>Figure&nbsp;3:</th>
 * <td>(a) basic marker shape for LaTeX style arrow 
 * marker defined by {@link ArrowSingle}, (b) basic marker
 * shape for square bracket marker defined by {@link ArrowSquare}.
 * A composite marker was then formed from the LaTeX arrow marker
 * as the primary marker and the square bracket as the
 * secondary marker: (c) the secondary marker is offset from
 * the primary marker, (d) the secondary marker is overlaid on
 * top of the primary marker.</td></tr>
 *</table>
 * </center>
 * <p>
 * Sub classes should overload {@link #getGeneralPath()}
 * to construct the basic marker shape, but should not
 * overload the other <code>get&lt;X&gt;GeneralPath</code> methods
 * as these are generic methods which use 
 * <code>getGeneralPath()</code> to construct the actual
 * marker shape given its properties. Note that the origin of the
 * path given by <code>getGeneralPath()</code> is mapped onto
 * the required vertex. Any repeated shapes will be offset
 * along the marker's x axis. The origin need not be the centre
 * of the marker.
 * <p>
 * A marker will either be oriented so that its x axis lies along
 * the gradient of the path at the given vertex (auto orient
 * enabled), or it may be oriented by a fixed angle (auto orient
 * disabled.) Note that if the auto orientation property
 * is disabled, start markers will not be reflected. (See
 * <a href="#fig4">Figure 4</a>.)
 * <p>
 * <center>
 * <table width=60%>
 * <tr align=center><td colspan=2><a name="fig4"></a>
 * <img src="../images/basicMarker4.png" alt="[image illustrating marker orientation]"></td></tr>
 * <tr><th valign=top>Figure&nbsp;4:</th>
 * <td>(a) basic marker shape for LaTeX style arrow 
 * marker, (b) the marker
 * is oriented so that its x axis lies along the line's 
 * gradient (the start marker has been reflected), (c) the marker is oriented according to its
 * angle of rotation (0 radians by default.) The start marker 
 * hasn't been reflected.</td></tr>
 * </table>
 * </center>
 * <p>
 * The marker's paint may be dependent on the colour of the
 * associated path, or it may be a fixed colour independent of
 * the line colour. (See
 * <a href="#fig5">Figure 5</a>.)
 * <p>
 * <center>
 * <table width=60%>
 * <tr align=center><td colspan=2><a name="fig5"></a>
 * <img src="../images/basicMarker5.png" alt="[image illustrating colours]"></td></tr>
 * <tr><th valign=top>Figure&nbsp;5:</th>
 * <td>(a) the marker colour is dependent on the line
 * colour, (b) the marker colour is independent of the line
 * colour.</td></tr>
 * </table>
 * </center>
 * <p>
 * A marker may have an associated size, or it may scale
 * according to the line width of the associated path (or
 * both.) For this reason, you can specify both the line
 * width of the associated path, and the marker size.
 *<p>
 * The secondary (or composite) marker has its
 * own independent settings.
 * (See <a href="#fig6">Figure 6</a>.)
 * The code in fact allows for 
 * composite markers to have their own composite markers, 
 * but these are not implemented in 
 * <code>jpgfdraw</code>.
 * <p>
 * <center>
 * <table width=60%>
 * <tr align=center><td colspan=2><a name="fig6"></a>
 * <img src="../images/basicMarker6.png" alt="[image illustrating composite markers independent settings]"></td></tr>
 * <tr><th valign=top>Figure&nbsp;6:</th>
 * <td>(a) basic marker shape for LaTeX style arrow 
 * marker defined by {@link ArrowSingle}, (b) basic marker
 * shape for square bracket marker defined by {@link ArrowSquare},
 * (c) Composite marker formed with the LaTeX arrow marker as
 * the primary marker and the square bracket as the secondary marker.
 * The primary marker has a repeat factor of 2 and its fill
 * colour is set to red, while the secondary marker has a blue
 * fill colour and has a repeat factor of 1 but has been reversed.
 *</td></tr>
 *</table>
 * </center>
 * <p>
 * Each known subclass of <code>JDRMarker</code> has an
 * associated numeric value used in the JDR/AJR file formats
 * to identify that marker type. The <code>JDRMarker</code>
 * class itself has the numeric type {@link #ARROW_NONE} indicating 
 * no visible marker.
 * <p>
 * <b>Notes:</b>
 * <ul>
 * <li> The markers have become too complicated to use the
 * <code>pgf</code> LaTeX package arrow mechanism, so each 
 * marker is now set as an individual path when the image
 * is exported to a TeX file.
 * <li> Similarly, the markers are set as individual paths
 * when exporting the image to an EPS file.
 * <li> I need to update the SVG related code.
 * </ul>
 * @author Nicola L C Talbot
 */
public class JDRMarker implements Serializable,Cloneable
{
   /**
    * Create a generic marker for a path with given pen width.
    * Markers may be repeated and/or reversed. Some markers
    * may depend on the line width of the associated path.
    * @param penwidth the width of the associated path in
    * PostScript points
    * @param repeat the repeat factor
    * @param isReversed specifies if marker should be reversed
    * @throws InvalidRepeatValueException if the repeat factor is 
    * less than 1
    */
   public JDRMarker(double penwidth, int repeat, boolean isReversed)
     throws InvalidRepeatValueException
   {
      if (repeat < 1)
      {
         throw new InvalidRepeatValueException(repeat);
      }

      repeated = repeat;
      reversed = isReversed;
      penWidth = penwidth;
      offset_  = 0;
      repeatOffset = 7.0*penWidth;
   }

   /**
    * Create a generic marker of given size for a path with 
    * given pen width.
    * Markers may be repeated and/or reversed. Some markers
    * may depend on the line width of the associated path as
    * well as having a specified size.
    * @param penwidth the width of the associated path in
    * PostScript points
    * @param repeat the repeat factor
    * @param isReversed specifies if marker should be reversed
    * @param arrowSize the size of the marker
    * @throws InvalidRepeatValueException if the repeat factor is 
    * less than 1
    */
   public JDRMarker(double penwidth, int repeat, boolean isReversed,
                    double arrowSize)
      throws InvalidRepeatValueException
   {
      this(penwidth, repeat, isReversed);
      size = arrowSize;
   }

   /**
    * Create default marker. This is equivalent to
    * <code>JDRMarker(1.0, 1, false)</code>
    */
   public JDRMarker()
   {
      repeated = 1;
      reversed = false;
      penWidth = 1.0;
      offset_  = 0;
      repeatOffset = 7.0;
   }

   /**
    * Draw this marker on given graphics context for given segment.
    * If this marker is to be drawn at
    * the end of the segment it is drawn in the direction of
    * the segment, otherwise it is drawn in the opposite 
    * direction unless the auto orientation is not set in which
    * case it is drawn at the given angle of orientation
    * given by {@link #angle_}. This just sets
    * {@link #penWidth} and calls
    * {@link #draw(Graphics, JDRPathSegment, boolean)}.
    * @param g the graphics context
    * @param penwidth the line width of the given segment in
    * PostScript points
    * @param segment the segment on which to draw marker
    * @param start marker is drawn at the start of the
    * segment if this is true, otherwise it is drawn at the
    * end.
    */
   public void draw(Graphics g, double penwidth, JDRPathSegment segment, boolean start)
   {
      penWidth = penwidth;
      if (!userRepeatOffset)
      {
         repeatOffset = 7.0*penWidth;
      }
      draw(g, segment, start);
   }

   /**
    * Draw this marker on given graphics context for given segment
    * where the segment line width is given by {@link #penWidth}.
    * If this marker is to be drawn at
    * the end of the segment it is drawn in the direction of
    * the segment, otherwise it is drawn in the opposite 
    * direction unless the auto orientation is disabled in which
    * case it is drawn at the given angle of orientation
    * given by {@link #angle_}.
    * @param g the graphics context
    * @param segment the segment on which to draw marker
    * @param start marker is drawn at the start of the
    * segment if this is true, otherwise it is drawn at the
    * end.
    */
   public void draw(Graphics g, JDRPathSegment segment, boolean start)
   {
      Graphics2D g2 = (Graphics2D)g;

      Stroke oldStroke = g2.getStroke();
      g2.setStroke(new BasicStroke(1.0f));

      Shape shape = getShape(segment, start);

      Paint oldPaint = null;

      if (fillPaint != null)
      {
         oldPaint = g2.getPaint();
         BBox box = getBBox();
         g2.setPaint(fillPaint.getPaint(box));
      }

      g2.fill(shape);

      g2.setStroke(oldStroke);
      if (fillPaint != null) g2.setPaint(oldPaint);

      if (composite != null)
      {
         composite.draw(g, segment, start);
      }
   }

   /**
    * Saves this marker in Encapsulated PostScript format.
    * (Sets {@link #penWidth} and calls 
    * {@link #saveEPS(JDRPaint, BBox, JDRPathSegment, boolean, PrintWriter)}.)
    * @param penwidth line width of the associated path
    * in PostScript points
    * @param pathPaint the colour of the associated path
    * @param pathBBox the bounding box of the associated path
    * @param segment the segment on which marker is to be
    * drawn
    * @param start if this is true, draw marker at the start
    * of the segment, otherwise draw it at the end
    * @param out the output stream
    * @throws IOException if I/O error occurs
    */
   public void saveEPS(double penwidth, JDRPaint pathPaint,
      BBox pathBBox, JDRPathSegment segment, boolean start,
      PrintWriter out)
      throws IOException
   {
      penWidth = penwidth;
      if (!userRepeatOffset)
      {
         repeatOffset = 7.0*penWidth;
      }
      saveEPS(pathPaint, pathBBox, segment, start, out);
   }

   /**
    * Saves this marker in Encapsulated PostScript format where
    * the line width is given by {@link #penWidth}.
    * @param pathPaint the colour of the associated path
    * @param pathBBox the bounding box of the associated path
    * @param segment the segment on which the marker is to be
    * drawn
    * @param start if this is true, draw the marker at the start
    * of the segment, otherwise draw it at the end
    * @param out the output stream
    * @throws IOException if I/O error occurs
    */
   public void saveEPS(JDRPaint pathPaint, BBox pathBBox,
      JDRPathSegment segment, boolean start, PrintWriter out)
      throws IOException
   {
      out.println("gsave");

      Shape shape = getShape(segment, start);

      JDRPaint paint = pathPaint;

      if (fillPaint != null)
      {
         paint = fillPaint;
      }

      paint.saveEPS(out, pathBBox);

      EPS.savePath(shape, out);

      if ((paint instanceof JDRGradient)
         || (paint instanceof JDRRadial))
      {
         out.println("clip shfill");
      }
      else
      {
         out.println("fill");
      }

      out.println("grestore");

      if (composite != null)
      {
         composite.saveEPS(pathPaint, pathBBox, segment, start, out);
      }
   }


   /**
    * Gets this marker's primary shape. 
    * The marker shape depends on whether
    * it is at the start or end of the segment, and whether
    * the auto orientation is set. (If this marker is at the
    * start of the segment, the shape is reflected, 
    * unless the auto orientation is off.) This shape does not
    * include the secondary marker shape, but does include
    * rotation.
    * @see #getPrimaryGeneralPath()
    * @param segment the segment on which this marker should be
    * drawn
    * @param start indictates whether this marker should be
    * drawn at the start (<code>true</code>) or the end 
    * (<code>false</code>) of the segment
    */
   public Shape getShape(JDRPathSegment segment, boolean start)
   {
      double p0x=0, p0y=0, p1x=0, p1y=0;

      if (start)
      {
         if (segment instanceof JDRBezier)
         {
            JDRBezier curve = (JDRBezier)segment;

            p1x = segment.getStart().x;
            p1y = segment.getStart().y;

            p0x = 3*curve.getControl1().x - 2*p1x;
            p0y = 3*curve.getControl1().y - 2*p1y;
         }
         else
         {
            p0x = segment.getEnd().x;
            p0y = segment.getEnd().y;

            p1x = segment.getStart().x;
            p1y = segment.getStart().y;
         }
      }
      else
      {
         if (segment instanceof JDRBezier)
         {
            JDRBezier curve = (JDRBezier)segment;

            p1x = segment.getEnd().x;
            p1y = segment.getEnd().y;

            p0x = 3*curve.getControl2().x - 2*p1x;
            p0y = 3*curve.getControl2().y - 2*p1y;
         }
         else if (segment instanceof JDRPartialBezier)
         {
            JDRPartialBezier curve = (JDRPartialBezier)segment;

            JDRPoint end = curve.getEnd();

            p1x = end.getX();
            p1y = end.getY();

            Point2D c2 = curve.getControl2();

            p0x = 3*c2.getX() - 2*p1x;
            p0y = 3*c2.getY() - 2*p1y;
         }
         else
         {
            JDRPoint end = segment.getEnd();

            p1x = end.getX();
            p1y = end.getY();

            p0x = segment.getStart().x;
            p0y = segment.getStart().y;
         }
      }

      Shape shape = getShape(p0x, p0y, p1x, p1y,
         start && !hasXAxisSymmetry());

      return shape;
   }

   /**
    * Gets complete shape include composite markers.
    * This is only used to determine the area taken up by
    * this marker and its composites.
    * @see #getShape(JDRPathSegment,boolean)
    */
   public Shape getCompleteShape(JDRPathSegment segment, boolean start)
   {
      GeneralPath path = new GeneralPath(getShape(segment, start));

      if (composite != null)
      {
         path.append(composite.getCompleteShape(segment, start), false);
      }

      return path;
   }

   /**
    * Indicates if this marker shape is symmetric about the x-axis.
    * @return true if this marker shape is symmetric about the
    * marker's x-axis
    */
   public boolean hasXAxisSymmetry()
   {
      return true;
   }

   /**
    * Gets the shape of the marker aligned along the given 
    * end points (of the gradient vector).
    * @param p0x the x co-ordinate of the starting point
    * @param p0y the y co-ordinate of the starting point
    * @param p1x the x co-ordinate of the end point
    * @param p1y the y co-ordinate of the end point
    * @param reflect indicates whether to reflect marker
    */
   private Shape getShape(double p0x, double p0y,
                         double p1x, double p1y,
                         boolean reflect)
   {
      AffineTransform af = new AffineTransform();

      //GeneralPath marker = getCompleteGeneralPath();
      GeneralPath marker = getPrimaryGeneralPath();

      // rotate so that the x-axis lies along 
      // the line defined by p0 -> p1

      double angle=0;

      if (autoOrient_)
      {
         if (reflect)
         {
            af.scale(1, -1);
            marker.transform(af);
            af.setToIdentity();
         }

         double dx = p1x-p0x;
         double dy = p1y-p0y;

         if (dy == 0)
         {
            angle = dx < 0 ? Math.PI : 0;
         }
         else if (dx == 0)
         {
            angle = dy < 0 ? -Math.PI/2 : Math.PI/2;
         }
         else
         {
            angle = Math.atan(dy/dx);
            if (dx < 0) angle += Math.PI;
         }
      }
      else
      {
         angle = angle_;
      }

      af.rotate(angle);

      Point2D r0 = new Point2D.Double(offset_, 0);
      Point2D r1 = new Point2D.Double(offset_, 0);
      af.transform(r0, r1);

      // shift so that marker origin lies at p1
      af.preConcatenate(AffineTransform.getTranslateInstance(p1x-r1.getX(), p1y-r1.getY()));

      Shape shape;

      shape = marker.createTransformedShape(af);

      double[] coords = new double[6];

      return shape;
   }

   /**
    * Converts the complete shape of this marker to a series of PGF 
    * commands.
    * The resulting set of commands are returned as a 
    * <code>String</code>. This includes the composite marker (if
    * one exists.)
    * @param pathPaint the colour of the associated path
    * @param pathBBox the bounding box of the associated path
    * @param segment the segment on which this marker should
    * be drawn
    * @param start indicates whether this marker should be
    * drawn at the start or end of the segment
    * @param af the affine transformation to convert from 
    * JDR co-ordinates to PGF co-ordinates
    */
   public String pgfShape(JDRPaint pathPaint, BBox pathBBox,
      JDRPathSegment segment, boolean start, AffineTransform af)
   {
      if (getType() == ARROW_NONE) return "";

      String eol = System.getProperty("line.separator", "\n");

      Shape shape = getShape(segment, start);

      PathIterator pi = shape.getPathIterator(af);

      double[] coords = new double[6];

      String str = "% marker type "+getType()+eol;

      str += "{\\begin{pgfscope}"+eol;

      // not yet implemented gradient fill
      if (fillPaint != null)
      {
         //str += fillPaint.pgffillcolor(getBBox());
         str += fillPaint.pgffillcolor(null);
      }
      else
      {
         //str += pathPaint.pgffillcolor(pathBBox);
         str += pathPaint.pgffillcolor(null);
      }

      while (!pi.isDone())
      {
         switch (pi.currentSegment(coords))
         {
            case PathIterator.SEG_MOVETO:
               str += "\\pgfpathqmoveto{"+PGF.format(coords[0])+"bp}{"
                    + PGF.format(coords[1])+"bp}";
            break;
            case PathIterator.SEG_LINETO:
               str += "\\pgfpathqlineto{"+PGF.format(coords[0])+"bp}{"
                    + PGF.format(coords[1])+"bp}";
            break;
            case PathIterator.SEG_CUBICTO:
               str += "\\pgfpathqcurveto{"
                    + PGF.format(coords[0])+"bp}{"
                    + PGF.format(coords[1])+"bp}{"
                    + PGF.format(coords[2])+"bp}{"
                    + PGF.format(coords[3])+"bp}{"
                    + PGF.format(coords[4])+"bp}{"
                    + PGF.format(coords[5])+"bp}";
            break;
            case PathIterator.SEG_CLOSE:
               str += "\\pgfclosepath";
            break;
         }

         pi.next();
         str += eol;
      }

      str += "\\pgfusepathqfill"+eol;
      str += "\\end{pgfscope}}"+eol;

      if (composite != null)
      {
         str += "% composite"+eol;
         str += composite.pgfShape(pathPaint, pathBBox, segment, start, af);
      }

      return str;
   }

   /**
    * Gets the width of the complete marker.
    * This includes space taken up by repeats and composites.
    * @return width of marker (in PostScript points)
    */
   public double getWidth()
   {
      GeneralPath path = getCompleteGeneralPath();

      return path.getBounds2D().getWidth();
   }

   /**
    * Gets the bounding box of the complete marker.
    * This includes space taken up by repeats and composites.
    */
   public BBox getBBox()
   {
      GeneralPath path = getCompleteGeneralPath();

      return new BBox(path.getBounds2D());
   }

   /**
    * Gets the <code>GeneralPath</code> describing the basic
    * shape of this marker.
    * Sub classes need to override this method (using 
    * {@link #getSize()} if the marker may have a variable size,
    * and using {@link #getPenWidth()} if marker size
    * should depend on the line width of the associated path.)
    * @return empty <code>GeneralPath</code>
    */
   public GeneralPath getGeneralPath()
   {
      return new GeneralPath();
   }

   /**
    * Gets the <code>GeneralPath</code> describing the primary
    * shape of this marker. This includes repeats and reflections but
    * not composites. This does not include rotation.
    * @see #getShape(JDRPathSegment, boolean)
    * @return primary path created from the path defined by
    * {@link #getGeneralPath()}, {@link #reversed} and
    * {@link #repeated}
    */
   public GeneralPath getPrimaryGeneralPath()
   {
      GeneralPath path = getGeneralPath();
      GeneralPath marker;
      AffineTransform af;

      if (reversed)
      {
         af = new AffineTransform(-1, 0, 0, 1, 0, 0);
         marker = new GeneralPath(af.createTransformedShape(path));
      }
      else
      {
         af = new AffineTransform(1, 0, 0, 1, 0, 0);
         marker = new GeneralPath(path);
      }

      for (int i = 2; i <= repeated; i++)
      {
         af.translate(-repeatOffset,0.0);

         marker.append(af.createTransformedShape(path), false);
      }

      return marker;
   }

   /**
    * Gets the <code>GeneralPath</code> describing the complete
    * marker. This includes repeats, reflections and composite
    * markers, but not rotation. This is equivalent to 
    * <code>getCompleteGeneralPath(false)</code>.
    * @return the <code>GeneralPath</code> describing the
    * complete marker
    */
   public GeneralPath getCompleteGeneralPath()
   {
      return getCompleteGeneralPath(false);
   }

   /**
    * Gets the <code>GeneralPath</code> describing the complete
    * marker where the composite may be reversed. This includes 
    * repeats, reflections and composite markers, but not rotation.
    * @param reverseComposite indicates whether to reverse the
    * composite marker
    * @return the <code>GeneralPath</code> describing the
    * complete marker
    */
   public GeneralPath getCompleteGeneralPath(boolean reverseComposite)
   {
      GeneralPath path = getGeneralPath();
      GeneralPath markerPath;
      AffineTransform af;

      markerPath = getPrimaryGeneralPath();

      if (composite != null)
      {
         markerPath.append(composite.getCompleteGeneralPath(), false);
      }

      return markerPath;
   }

   /**
    * Determines if this marker uses {@link #size} in determining
    * its shape.
    * @return true if {@link #getGeneralPath()} uses {@link #size}
    */
   public boolean isResizable()
   {
      return false;
   }

   /**
    * Determines if this marker uses the associated path's line
    * with in determining its shape.
    * @return true if {@link #getGeneralPath()} uses the path's
    * line width
    */
   public boolean usesLineWidth()
   {
      return false;
   }

   /**
    * Saves marker definition in SVG format.
    * @param out the output stream
    * @param id a unique identifier for this marker
    * @throws IOException if I/O error occurs
    */
   public void svgDef(PrintWriter out, String id)
      throws IOException
   {
      if (penWidth==0.0f || type==ARROW_NONE) return;

      GeneralPath path = getCompleteGeneralPath(isStart_);

      AffineTransform af = new AffineTransform(
         1.0f/penWidth, 0.0f, 0.0f, 1.0f/penWidth, 0.0f, 0.0f);

      Shape shape = path.createTransformedShape(af);

      Rectangle2D bounds = shape.getBounds2D();

      double minX = bounds.getX();
      double minY = bounds.getY();
      double width = bounds.getWidth();
      double height = bounds.getHeight();

      out.println("      <marker id=\""+id+"\" ");
      out.println("         markerWidth=\""+width+"\" "
                + "         markerHeight=\""+height+"\"");
      out.println("         viewBox=\"0 0 "
         +width+" " + height
         +"\" refX=\""+(-minX)+"\" refY=\""+(-minY)+"\"");
      out.println("         markerUnits=\"strokeWidth\"");
      out.println("         orient=\""+(autoOrient_?"auto":angle_)+"rad\" >");
      out.print("        <path d=\"");

      PathIterator pi = shape.getPathIterator(null);
      double[] coords = new double[6];

      while (!pi.isDone())
      {
         switch (pi.currentSegment(coords))
         {
            case PathIterator.SEG_MOVETO:
               out.print("M "+(coords[0]-minX)
                        +" "+(coords[1]-minY)+" ");
            break;
            case PathIterator.SEG_LINETO:
               out.print("L "+(coords[0]-minX)
                         +" "+(coords[1]-minY)+" ");
            break;
            case PathIterator.SEG_QUADTO:
               out.print("S "+(coords[0]-minX)
                         +" "+(coords[1]-minY)
                         +" "+(coords[2]-minX)
                         +" "+(coords[3]-minY));
            break;
            case PathIterator.SEG_CUBICTO:
               out.print("C "+(coords[0]-minX)
                         +" "+(coords[1]-minY)
                         +" "+(coords[2]-minX)
                         +" "+(coords[3]-minY)
                         +" "+(coords[4]-minX)
                         +" "+(coords[5]-minY));
            break;
            case PathIterator.SEG_CLOSE:
               out.print("Z ");
            break;
         }

         pi.next();
      }

      out.println("\"");

      if (fillPaint != null) out.print(fillPaint.svgFill());

      out.println("/>");

      out.println("      </marker>");
   }

   /**
    * Returns PGF arrow command.
    * No longer used, as all markers are now saved as PGF paths
    * when saving as a PGF image.
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "";
   }

   /**
    * Returns PGF commands to reverse or repeat markers.
    * No longer used, as all markers are now saved as PGF paths
    * when saving as a PGF image.
    * @deprecated
    */
   private String pgfdoarrow()
   {
      String str = "";

      String arrow = pgfarrow();

      if (reversed) str += "\\pgfarrowswap{";
 
      switch (repeated)
      {
         case 1:
            str += arrow;
         break;
         case 2:
            str += ("\\pgfarrowdouble{"+arrow+"}");
         break;
         case 3:
            str += ("\\pgfarrowtriple{"+arrow+"}");
         break;
      }

      if (reversed) str += "}";

      return str;
   }

   /**
    * Returns PGF commands to define the marker as a start/end
    * arrow.
    * No longer used as the markers are now drawn as paths
    * instead of using the PGF arrow commands.
    * @deprecated
    */
   public String pgf(boolean start)
   {

      return (start ? "\\pgfsetstartarrow{}"
                    : "\\pgfsetendarrow{}");
/*
      String eol = System.getProperty("line.separator", "\n");

      if (!autoOrient_ || (composite!=null && !composite.autoOrient_))
      {
         return (start ? "\\pgfsetstartarrow{}"+eol
                       : "\\pgfsetendarrow{}"+eol);
      }

      String str = (start ? "\\pgfsetstartarrow{{" 
                          : "\\pgfsetendarrow{{");

      if (composite == null)
      {
         if (fillPaint != null)
         {
            str += fillPaint.pgffillcolor(getBBox());
         }

         str += pgfdoarrow();
      }
      else
      {
         if (!overlay)
         {
            str += "\\pgfarrowcombine"+eol;
         }

         str += "{{";
         if (fillPaint != null)
         {
            str += fillPaint.pgffillcolor(getBBox());
         }

         str += pgfdoarrow();
         str += "}}"+eol;

         str += "{{";
         if (composite.fillPaint != null)
         {
            str += composite.fillPaint.pgffillcolor(getBBox());
         }

         str += composite.pgfdoarrow();
         str += "}}";
      }
         
      str += "}}"+eol;

      return str;
*/
   }

   /**
    * Gets a known marker with default settings.
    * This assumes a line width of 1 PostScript point, a
    * repeat value of 1, not reversed and a marker size of
    * 5 PostScript points.
    * @param markerID the number uniquely identifying a 
    * known marker
    * @return the marker
    * @throws InvalidMarkerTypeException if <code>markerID</code>
    * doesn't correspond to any known marker
    */
   public static JDRMarker getPredefinedMarker(int markerID)
      throws InvalidMarkerTypeException
   {
      JDRMarker marker = null;
      try
      {
         marker = getPredefinedMarker(markerID, 1.0f, 1, false, 5.0f);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }

   /**
    * Gets a known marker with given settings.
    * This assumes a marker size of
    * 5 PostScript points.
    * @param markerID the number uniquely identifying a 
    * known marker
    * @param penwidth the line width of the associated path
    * @param repeat the number of times the marker is repeated
    * @param isReversed determines whether or not to reverse
    * the marker
    * @return the marker
    * @throws InvalidMarkerTypeException if <code>markerID</code>
    * doesn't correspond to any known marker
    */
   public static JDRMarker getPredefinedMarker(int markerID,
      double penwidth, int repeat, boolean isReversed)
      throws InvalidMarkerTypeException,
             InvalidRepeatValueException
   {
      return getPredefinedMarker(markerID, penwidth,
         repeat, isReversed, 5.0f);
   }

   /**
    * Gets a known marker with given settings and size.
    * Note that some predefined markers may only have a size
    * dependent on the line width.
    * @param markerID the number uniquely identifying a 
    * known marker
    * @param penwidth the line width of the associated path
    * @param repeat the number of times the marker is repeated
    * @param isReversed determines whether or not to reverse
    * @param arrowSize the size of the marker where the marker
    * may have a variable size
    * the marker
    * @return the marker
    * @throws InvalidMarkerTypeException if <code>markerID</code>
    * doesn't correspond to any known marker
    * @throws InvalidRepeatValueException if repeat factor is less
    * than 1.
    */
   public static JDRMarker getPredefinedMarker(int markerID,
      double penwidth, int repeat, boolean isReversed,
      double arrowSize)
      throws InvalidMarkerTypeException,
             InvalidRepeatValueException
   {
      switch (markerID)
      {
         case ARROW_NONE:
            return new JDRMarker(penwidth, repeat, isReversed,
               arrowSize);
         case ARROW_POINTED :
            return new ArrowPointed(penwidth, repeat, isReversed,
               arrowSize);
         case ARROW_TRIANGLE :
            return new ArrowTriangle(penwidth, repeat, isReversed,
               arrowSize);
         case ARROW_CIRCLE :
            return new ArrowCircle(penwidth, repeat, isReversed,
               arrowSize);
         case ARROW_DIAMOND :
            return new ArrowDiamond(penwidth, repeat, isReversed);
         case ARROW_SQUARE :
            return new ArrowSquare(penwidth, repeat, isReversed);
         case ARROW_BAR :
            return new ArrowBar(penwidth, repeat, isReversed);
         case ARROW_SINGLE :
            return new ArrowSingle(penwidth, repeat, isReversed);
         case ARROW_ROUND :
            return new ArrowRound(penwidth, repeat, isReversed);
         case ARROW_DOTFILLED :
            return new ArrowDotFilled(penwidth, repeat, isReversed,
              arrowSize);
         case ARROW_DOTOPEN :
            return new ArrowDotOpen(penwidth, repeat, isReversed,
              arrowSize);
         case ARROW_BOXFILLED :
            return new ArrowBoxFilled(penwidth, repeat, isReversed,
              arrowSize);
         case ARROW_BOXOPEN :
            return new ArrowBoxOpen(penwidth, repeat, isReversed,
              arrowSize);
         case ARROW_CROSS :
            return new ArrowCross(penwidth, repeat, isReversed,
              arrowSize);
         case ARROW_PLUS :
            return new ArrowPlus(penwidth, repeat, isReversed,
              arrowSize);
         case ARROW_STAR :
            return new ArrowStar(penwidth, repeat, isReversed,
              arrowSize);
         case ARROW_TRIANGLE_UP_FILLED :
            return new ArrowTriangleUpFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_TRIANGLE_UP_OPEN :
            return new ArrowTriangleUpOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_TRIANGLE_DOWN_FILLED :
            return new ArrowTriangleDownFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_TRIANGLE_DOWN_OPEN :
            return new ArrowTriangleDownOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_RHOMBUS_FILLED :
            return new ArrowRhombusFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_RHOMBUS_OPEN :
            return new ArrowRhombusOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_PENTAGON_FILLED:
            return new ArrowPentagonFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_PENTAGON_OPEN:
            return new ArrowPentagonOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HEXAGON_FILLED:
            return new ArrowHexagonFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HEXAGON_OPEN:
            return new ArrowHexagonOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_OCTAGON_FILLED:
            return new ArrowOctagonFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_OCTAGON_OPEN:
            return new ArrowOctagonOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_POINTED60:
            return new ArrowPointed60(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_POINTED45:
            return new ArrowPointed45(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HOOKS:
            return new ArrowHooks(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HOOK_UP:
            return new ArrowHookUp(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HOOK_DOWN:
            return new ArrowHookDown(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HALF_POINTED_UP:
            return new ArrowHalfPointedUp(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HALF_POINTED_DOWN:
            return new ArrowHalfPointedDown(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HALF_POINTED60_UP:
            return new ArrowHalfPointed60Up(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HALF_POINTED60_DOWN:
            return new ArrowHalfPointed60Down(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HALF_POINTED45_UP:
            return new ArrowHalfPointed45Up(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HALF_POINTED45_DOWN:
            return new ArrowHalfPointed45Down(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_CUSP:
            return new ArrowCusp(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HALF_CUSP_UP:
            return new ArrowHalfCuspUp(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HALF_CUSP_DOWN:
            return new ArrowHalfCuspDown(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_ALT_SINGLE:
            return new ArrowAltSingle(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_ALT_SINGLE_OPEN:
            return new ArrowAltSingleOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_TRIANGLE_OPEN:
            return new ArrowTriangleOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_CIRCLE_OPEN:
            return new ArrowCircleOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_DIAMOND_OPEN:
            return new ArrowDiamondOpen(penwidth, repeat,
              isReversed);
         case ARROW_BRACE:
            return new ArrowBrace(penwidth, repeat,
              isReversed);
         case ARROW_RECTANGLE_CAP:
            return new ArrowRectangleCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_BALL_CAP:
            return new ArrowBallCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_LEAF_CAP:
            return new ArrowLeafCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_LEAF2_CAP:
            return new ArrowLeaf2Cap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_LEAF3_CAP:
            return new ArrowLeaf3Cap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_CLUB_CAP:
            return new ArrowClubCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_LEAF3FOR_CAP:
            return new ArrowLeaf3ForCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_LEAF3BACK_CAP:
            return new ArrowLeaf3BackCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_LEAF2FOR_CAP:
            return new ArrowLeaf2ForCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_LEAF2BACK_CAP:
            return new ArrowLeaf2BackCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_BULGE_CAP:
            return new ArrowBulgeCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_CUTOUTBULGE_CAP:
            return new ArrowCutoutBulgeCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_CHEVRON_CAP:
            return new ArrowChevronCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_FAST_CAP:
            return new ArrowFastCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_ROUND_CAP:
            return new ArrowRoundCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_TRIANGLE_CAP:
            return new ArrowTriangleCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_INVERT_TRIANGLE_CAP:
            return new ArrowInvertTriangleCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_INVERT_CHEVRON_CAP:
            return new ArrowInvertChevronCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_INVERT_FAST_CAP:
            return new ArrowInvertFastCap(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_ALT_BAR:
            return new ArrowAltBar(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_ALT_ROUND:
            return new ArrowAltRound(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_ALT_SQUARE:
            return new ArrowAltSquare(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_ALT_BRACE:
            return new ArrowAltBrace(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_SEMICIRCLE_OPEN:
            return new ArrowSemiCircleOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_SEMICIRCLE_FILLED:
            return new ArrowSemiCircleFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_STAR5_OPEN:
            return new ArrowStar5Open(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_STAR5_FILLED:
            return new ArrowStar5Filled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_ASTERISK:
            return new ArrowAsterisk(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_SCISSORS_DOWN_FILLED:
            return new ArrowScissorsDownFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_SCISSORS_UP_FILLED:
            return new ArrowScissorsUpFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_SCISSORS_DOWN_OPEN:
            return new ArrowScissorsDownOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_SCISSORS_UP_OPEN:
            return new ArrowScissorsUpOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HEART_RIGHT_FILLED:
            return new ArrowHeartRightFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HEART_RIGHT_OPEN:
            return new ArrowHeartRightOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HEART_FILLED:
            return new ArrowHeartFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_HEART_OPEN:
            return new ArrowHeartOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_SNOWFLAKE:
            return new ArrowSnowflake(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_STAR_CHEVRON_OPEN:
            return new ArrowStarChevronOpen(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_STAR_CHEVRON_FILLED:
            return new ArrowStarChevronFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_STAR6_OPEN:
            return new ArrowStar6Open(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_STAR6_FILLED:
            return new ArrowStar6Filled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_EQUILATERAL_FILLED:
            return new ArrowEquilateralFilled(penwidth, repeat,
              isReversed, arrowSize);
         case ARROW_EQUILATERAL_OPEN:
            return new ArrowEquilateralOpen(penwidth, repeat,
              isReversed, arrowSize);
      }

      throw new InvalidMarkerTypeException(markerID);
   }

   /**
    * Gets a copy of this marker.
    * @return a copy of this marker
    */
   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new JDRMarker(penWidth, repeated, reversed,
                                size);

         marker.makeEqual(this);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }

   /**
    * Determines if this object is the same as another object.
    * @param o the other object
    * @return true if this object is the same as the other object
    */
   public boolean equals(Object o)
   {
      if (this == o) return true;

      if (o == null) return false;
      if (!(o instanceof JDRMarker)) return false;

      JDRMarker m = (JDRMarker)o;

      if (penWidth != m.penWidth) return false;
      if (repeated != m.repeated) return false;
      if (reversed != m.reversed) return false;
      if (size != m.size) return false;
      if (autoOrient_ != m.autoOrient_) return false;
      if (angle_ != m.angle_) return false;
      if (overlay != m.overlay) return false;
      if (offset_ != m.offset_) return false;
      if (repeatOffset != m.repeatOffset) return false;
      if (userOffset != m.userOffset) return false;
      if (userRepeatOffset != m.userRepeatOffset) return false;

      if (fillPaint == null)
      {
         if (m.fillPaint != null) return false;
      }
      else
      {
         if (!fillPaint.equals(m.fillPaint)) return false;
      }

      if (composite == null)
      {
         if (m.composite != null) return false;
      }
      else
      {
         if (!composite.equals(m.composite)) return false;
      }

      return true;
   }

   /**
    * Makes given marker identical to this marker.
    * @param marker the marker to make the same as this marker
    */
   public void makeEqual(JDRMarker marker)
   {
      if (fillPaint != null)
      {
         marker.fillPaint = (JDRPaint)fillPaint.clone();
      }

      marker.repeated = repeated;
      marker.reversed = reversed;
      marker.size     = size;
      marker.autoOrient_ = autoOrient_;
      marker.angle_   = angle_;
      marker.overlay  = overlay;
      marker.offset_  = offset_;
      marker.penWidth = penWidth;
      marker.repeatOffset  = repeatOffset;
      marker.userOffset  = userOffset;
      marker.userRepeatOffset  = userRepeatOffset;

      if (composite == null)
      {
         marker.composite = null;
      }
      else
      {
         marker.composite = (JDRMarker)composite.clone();
         marker.composite.parent = this;
      }
   }

   /**
    * Gets this marker's identifying number. (As used by
    * <code>getPredefinedMarker</code>)
    * @return the identifying number
    */
   public int getType()
   {
      return type;
   }

   /**
    * Saves this marker in given version of JDR format.
    * @see JDR#save(JDRGroup, CanvasSettings, java.io.DataOutputStream, float)
    * @param dout the output stream
    * @param version the JDR file version number
    * @throws IOException if I/O error occurs
    */
   public void save(DataOutputStream dout, float version)
      throws IOException
   {
      if (version == 1.0f)
      {
         if (type >= NUM_ARROWS1_0)
         {
            dout.writeByte((byte)ARROW_NONE);
            return;
         }

         dout.writeByte((byte)type);
         if (type != JDRMarker.ARROW_NONE)
         {
            dout.writeFloat((float)size);
            dout.writeBoolean(repeated==2);
            dout.writeBoolean(reversed);
         }
      }
      else
      {
         if (version < 1.4f && type >= NUM_ARROWS1_1)
         {
            dout.writeByte((byte)ARROW_NONE);
            return;
         }

         dout.writeByte((byte)type);

         if (type != JDRMarker.ARROW_NONE)
         {
            dout.writeFloat((float)size);
            dout.writeByte((byte)repeated);
            dout.writeBoolean(reversed);

            dout.writeBoolean(autoOrient_);
            if (!autoOrient_) dout.writeFloat((float)angle_);

            JDRPaintLoader paintLoader = JDR.getPaintLoader();

            paintLoader.saveJDR((fillPaint == null ?
                              new JDRTransparent() :
                              fillPaint),
                              dout, version);

            dout.writeBoolean(overlay);

            if (version >= 1.4f)
            {
               if (!overlay)
               {
                  dout.writeBoolean(userOffset);

                  if (userOffset)
                  {
                     dout.writeFloat((float)offset_);
                  }

                  dout.writeBoolean(userRepeatOffset);

                  if (userRepeatOffset)
                  {
                     dout.writeFloat((float)repeatOffset);
                  }
               }
            }

            // is it a composite arrow?

            if (composite == null)
            {
               dout.writeByte((byte)ARROW_NONE);
            }
            else
            {
               composite.save(dout, version);
            }
         }
      }
   }

   /**
    * Reads marker data from input stream in given version of 
    * JDR file format.
    * @see JDR#load(DataInputStream, CanvasSettings)
    * @param din the input stream
    * @param version the JDR file format version
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if marker data is incorrectly
    * formatted
    * @return the marker defined by the input stream
    */
   public static JDRMarker read(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      if (version == 1.0f)
      {
         int arrowType = (int)din.readByte();

         if (arrowType < 0 || arrowType >= NUM_ARROWS1_0)
         {
            throw new InvalidMarkerTypeException(arrowType);
         }

         double arrowSize = 5.0F;
         boolean arrowDouble=false;
         boolean arrowReversed=false;

         if (arrowType != ARROW_NONE)
         {
            arrowSize = din.readFloat();
            arrowDouble = din.readBoolean();
            arrowReversed = din.readBoolean();
         }

         return getPredefinedMarker(arrowType, 1.0f,
                  arrowDouble?2:1, arrowReversed, arrowSize);
      }
      else
      {
         int arrowType = (int)din.readByte();

         int maxType = (version < 1.4f ?
            NUM_ARROWS1_1 :
            (version < 1.6f ? NUM_ARROWS1_4 : NUM_ARROWS1_6));

         if (arrowType < 0 || arrowType >= maxType)
         {
            throw new InvalidMarkerTypeException(arrowType);
         }

         double arrowSize   = 5.0F;
         int arrowRepeated = 1;
         boolean arrowReversed=false;
         boolean autoorient=true;
         double angle=0.0f;
         boolean overlayFlag=false;
         boolean userOffsetFlag=false;
         float userOffsetValue=0.0f;
         float repeatOffsetValue=0;
         boolean repeatOffsetFlag=false;

         JDRPaint paint=null;

         JDRMarker compositeMarker=null;

         if (arrowType != ARROW_NONE)
         {
            arrowSize = din.readFloat();
            arrowRepeated = (int)din.readByte();
            arrowReversed = din.readBoolean();
            autoorient=din.readBoolean();
            if (!autoorient) angle=din.readFloat();

            JDRPaintLoader paintLoader = JDR.getPaintLoader();
            paint = paintLoader.loadJDR(din, version);

            if (paint instanceof JDRTransparent)
            {
               paint = null;
            }

            overlayFlag = din.readBoolean();

            if (version >= 1.4f)
            {
               if (!overlayFlag)
               {
                  userOffsetFlag = din.readBoolean();

                  if (userOffsetFlag)
                  {
                     userOffsetValue = din.readFloat();
                  }

                  repeatOffsetFlag = din.readBoolean();

                  if (repeatOffsetFlag)
                  {
                     repeatOffsetValue = din.readFloat();
                  }
               }
            }

            compositeMarker = read(din, version);

            if (compositeMarker.getType() == ARROW_NONE)
            {
               compositeMarker = null;
            }
         }

         JDRMarker marker = getPredefinedMarker(arrowType, 1.0f,
                  arrowRepeated, arrowReversed, arrowSize);

         marker.fillPaint=paint;
         marker.setCompositeMarker(compositeMarker);
         marker.setOrient(autoorient, angle);
         marker.setOverlay(overlayFlag);

         marker.enableUserRepeatOffset(repeatOffsetFlag);

         if (repeatOffsetFlag)
         {
            marker.setRepeatOffset(repeatOffsetValue);
         }

         marker.enableUserOffset(userOffsetFlag);

         if (userOffsetFlag)
         {
            marker.setOffset(userOffsetValue);
         }

         return marker;
      }
   }

   /**
    * Saves this marker in given version of AJR file format.
    * @see AJR#save(JDRGroup, CanvasSettings, java.io.PrintWriter, float)
    * @param out the output stream
    * @param version the JDR file version number
    * @throws IOException if I/O error occurs
    * @throws InvalidStartColourException if marker paint
    * is radial/linear paint with a starting colour other
    * than {@link JDRColor} or {@link JDRColorCMYK}
    * @throws InvalidEndColourException if marker paint
    * is radial/linear paint with a end colour other
    * than <code>JDRColor</code> or <code>JDRColorCMYK</code>
    */
   public void saveAJR(PrintWriter out, float version)
    throws IOException,
      InvalidFormatException
   {
      if (version == 1.0f)
      {
         int type1_0 = (type >= NUM_ARROWS1_0 ? ARROW_NONE : type);

         AJR.writeInt(out, type1_0);

         if (type1_0 != JDRMarker.ARROW_NONE)
         {
            AJR.writeFloat(out, (float)size);
            AJR.writeInt(out, repeated);
            AJR.writeBoolean(out, reversed);
         }

         out.println();
      }
      else
      {
         if (version < 1.4f && type >= NUM_ARROWS1_1)
         {
            AJR.writeInt(out, ARROW_NONE);
            return;
         }

         AJR.writeInt(out, type);

         if (type != JDRMarker.ARROW_NONE)
         {
            AJR.writeFloat(out, (float)size);
            AJR.writeInt(out, repeated);
            AJR.writeBoolean(out, reversed);
            AJR.writeBoolean(out, autoOrient_);

            if (!autoOrient_) AJR.writeFloat(out, (float)angle_);

            JDRPaintLoader paintLoader = AJR.getPaintLoader();

            paintLoader.saveAJR((fillPaint == null ?
                              new JDRTransparent() :
                              fillPaint),
                              out, version);

            AJR.writeBoolean(out, overlay);

            if (version >= 1.4f)
            {
               if (!overlay)
               {
                  AJR.writeBoolean(out, userOffset);

                  if (userOffset)
                  {
                     AJR.writeFloat(out, (float)offset_);
                  }

                  AJR.writeBoolean(out, userRepeatOffset);

                  if (userRepeatOffset)
                  {
                     AJR.writeFloat(out, (float)repeatOffset);
                  }
               }
            }

            // is it a composite arrow?

            if (composite == null)
            {
               AJR.writeInt(out, ARROW_NONE);
               out.println();
            }
            else
            {
               composite.saveAJR(out, version);
            }
         }
      }
   }

   /**
    * Reads marker data from text input stream in given version 
    * of AJR file format.
    * @see AJR#load(BufferedReader, CanvasSettings)
    * @param in the input stream
    * @param version the JDR file format version
    * @throws IOException if I/O error occurs
    * @throws EOFException if EOF occurs
    * @throws InvalidFormatException if marker data is incorrectly
    * @throws java.nio.BufferOverflowException if any string read
    * in exceeds {@link AJR#buffLength}
    * formatted
    * @return the marker defined by the input stream
    */
   public static JDRMarker readAJR(BufferedReader in, float version)
      throws IOException,
             InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      if (version == 1.0f)
      {
         int arrowType = AJR.readInt(in);

         if (arrowType < 0 || arrowType >= NUM_ARROWS1_0)
         {
            throw new InvalidMarkerTypeException(arrowType,
               AJR.getLineNum());
         }

         double arrowSize = 5.0F;
         boolean arrowDouble=false;
         boolean arrowReversed=false;

         if (arrowType != ARROW_NONE)
         {
            arrowSize = AJR.readFloat(in);
            arrowDouble = AJR.readBoolean(in);
            arrowReversed = AJR.readBoolean(in);
         }

         return getPredefinedMarker(arrowType, 1.0f,
                  arrowDouble?2:1, arrowReversed, arrowSize);
      }
      else
      {
         int arrowType = AJR.readInt(in);

         int maxType = (version < 1.4f ?
             NUM_ARROWS1_1 :
             (version < 1.6f ? NUM_ARROWS1_4 : NUM_ARROWS1_6));

         if (arrowType < 0 || arrowType >= maxType)
         {
            throw new InvalidMarkerTypeException(arrowType,
               AJR.getLineNum());
         }

         double arrowSize   = 5.0F;
         int arrowRepeated = 1;
         boolean arrowReversed=false;
         boolean autoorient=true;
         double angle=0.0;
         boolean overlayFlag=false;

         boolean userOffsetFlag=false;
         double userOffsetValue=0.0;
         boolean repeatOffsetFlag=false;
         double repeatOffsetValue=0.0;

         JDRPaint paint=null;

         JDRMarker compositeMarker=null;

         if (arrowType != ARROW_NONE)
         {
            arrowSize = AJR.readFloat(in);

            arrowRepeated = AJR.readInt(in);

            arrowReversed = AJR.readBoolean(in);

            autoorient = AJR.readBoolean(in);

            if (!autoorient)
            {
               angle = AJR.readFloat(in);
            }

            JDRPaintLoader paintLoader = AJR.getPaintLoader();
            paint = paintLoader.loadAJR(in, version);

            if (paint instanceof JDRTransparent)
            {
               paint = null;
            }

            overlayFlag = AJR.readBoolean(in);

            if (version >= 1.4f)
            {
               if (!overlayFlag)
               {
                  userOffsetFlag = AJR.readBoolean(in);

                  if (userOffsetFlag)
                  {
                     userOffsetValue = AJR.readFloat(in);
                  }

                  repeatOffsetFlag = AJR.readBoolean(in);

                  if (repeatOffsetFlag)
                  {
                     repeatOffsetValue = AJR.readFloat(in);
                  }
               }
            }

            compositeMarker = readAJR(in, version);

            if (compositeMarker.getType() == ARROW_NONE)
            {
               compositeMarker = null;
            }
         }

         JDRMarker marker = getPredefinedMarker(arrowType, 1.0f,
                  arrowRepeated, arrowReversed, arrowSize);

         marker.fillPaint=paint;
         marker.setCompositeMarker(compositeMarker);
         marker.setOrient(autoorient, angle);
         marker.setOverlay(overlayFlag);

         marker.enableUserRepeatOffset(repeatOffsetFlag);

         if (repeatOffsetFlag)
         {
            marker.setRepeatOffset(repeatOffsetValue);
         }

         marker.enableUserOffset(userOffsetFlag);

         if (userOffsetFlag)
         {
            marker.setOffset(userOffsetValue);
         }

         return marker;
      }
   }

   /**
    * Gets string identifying this marker. Sub classes should
    * overload this method. This is used for the SVG definition
    * of this marker.
    * @return identifying string
    */
   public String getID()
   {
      return "none";
   }

   /**
    * Gets string identifying this marker with independent colour
    * where this marker's location is given. 
    * @param start if this is true, this marker occurs at the
    * start of a segment, otherwise it is at the end of a
    * segment.
    * @return the identifying string
    */
   public String getID(boolean start)
   {
      return getID(start, fillPaint);
   }

   /**
    * Gets string identifying this marker with the given colour
    * and location. 
    * @param start if this is true, this marker occurs at the
    * start of a segment, otherwise it is at the end of a
    * segment.
    * @return the identifying string
    */
   public String getID(boolean start, JDRPaint p)
   {
      if (start) reversed=!reversed;
      String id = getID();
      if (start) reversed=!reversed;

      if (fillPaint == null)
      {
         if (p != null) id += p.getID();
      }
      else
      {
         id += fillPaint.getID();
      }

      return id;
   }

   /**
    * Gets the start marker SVG syntax for this marker.
    * @param p the line paint
    * @return the string containing the <code>marker-start</code>
    * syntax
    */
   public String svgStartMarker(JDRPaint p)
   {
      if (type==JDRMarker.ARROW_NONE)
         return "marker-start=\"none\"";

      return "marker-start=\"url(#" + getID(true, p)+")\"";
   }

   /**
    * Gets the mid marker SVG syntax for this marker.
    * @param p the line paint
    * @return the string containing the <code>marker-mid</code>
    * syntax
    */
   public String svgMidMarker(JDRPaint p)
   {
      if (type==JDRMarker.ARROW_NONE)
         return "marker-mid=\"none\"";

      return "marker-mid=\"url(#" + getID(false, p)+")\"";
   }

   /**
    * Gets the end marker SVG syntax for this marker.
    * @param p the line paint
    * @return the string containing the <code>marker-end</code>
    * syntax
    */
   public String svgEndMarker(JDRPaint p)
   {
      if (type==ARROW_NONE)
          return "marker-end=\"none\"";

      return "marker-end=\"url(#" +getID(false, p)+")\"";
   }

   /**
    * Writes all marker SVG definitions to output stream.
    * Some markers may be used more than once, but only
    * want to define them once at the start of the SVG file.
    * This iterates through all markers used in paths contained 
    * in the group, ignoring duplicates, and writes the 
    * SVG definition.
    * @param out the output stream
    * @param group the group of objects which may or may not
    * contain paths
    * @throws IOException if I/O error occurs
    */
   public static void svgDefs(PrintWriter out, JDRGroup group)
      throws IOException
   {
      Hashtable<String,JDRMarker> markers
         = new Hashtable<String,JDRMarker>();

      // get all markers used in this image
      // some markers may be used more than once, but
      // only want to define it once.

      for (int i = 0, n=group.size(); i < n; i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object instanceof JDRShape
          && ((JDRShape)object).getStroke() instanceof JDRBasicStroke)
         {
            JDRBasicStroke s = (JDRBasicStroke)((JDRShape)object).getStroke();
            JDRPaint paint = ((JDRShape)object).getLinePaint();

            // start marker

            JDRMarker start = (JDRMarker)s.getStartArrow().clone();
            if (start.fillPaint == null) start.fillPaint=paint;

            String id = start.getID(true, paint);
            start.reversed = !start.reversed;
            start.isStart_=true;

            if (!id.equals("none")) markers.put(id, start);

            // mid-point marker

            JDRMarker mid = (JDRMarker)s.getMidArrow().clone();
            if (mid.fillPaint == null) mid.fillPaint = paint;
            mid.isStart_=false;

            id = mid.getID(false, paint);
            if (!id.equals("none")) markers.put(id, mid);

            // end marker

            JDRMarker end = (JDRMarker)s.getEndArrow().clone();
            if (end.fillPaint == null) end.fillPaint = paint;
            end.isStart_=false;

            id = end.getID(false, paint);
            if (!id.equals("none")) markers.put(id, end);
         }
      }

      for (Enumeration e = markers.keys(); e.hasMoreElements(); )
      {
         String id = (String)e.nextElement();

         JDRMarker marker = (JDRMarker)markers.get(id);
         marker.svgDef(out, id);
      }
   }

   /**
    * Gets this marker's auto orientation property. 
    * If true, this marker should be oriented to lie along the 
    * segment's gradient, otherwise it should be oriented 
    * according to the value returned by {@link #getAngle()}.
    * @return the auto orientation property
    */
   public boolean getAutoOrient()
   {
      return autoOrient_;
   }

   /**
    * Gets this marker's angle to be used if the auto 
    * orientation is disabled. If auto orientation is enabled
    * this value is ignored.
    * @see #getAutoOrient()
    * @see #setAngle(double)
    * @return the angle of orientation
    */
   public double getAngle()
   {
      return angle_;
   }

   /**
    * Sets this marker's angle to be used if the auto 
    * orientation is disabled. If auto orientation is enabled
    * this value is ignored.
    * @see #getAutoOrient()
    * @see #getAngle()
    */
   public void setAngle(double angle)
   {
      angle_ = angle;
   }

   /**
    * Sets whether or not to auto orient this marker.
    * @see #getAutoOrient()
    * @param autoOrient if true, automatically orient this
    * marker, otherwise orient it according to the value
    * of {@link #getAngle()}
    */
   public void setOrient(boolean autoOrient)
   {
      autoOrient_ = autoOrient;
   }

   /**
    * Sets whether or not to auto orient this marker and
    * the angle of orientation. The angle is only used
    * if the auto orientation property is disabled
    * @see #getAutoOrient()
    * @see #setAngle(double)
    * @param autoOrient if true, automatically orient this
    * marker, otherwise orient it according to the given angle
    */
   public void setOrient(boolean autoOrient, double angle)
   {
      autoOrient_ = autoOrient;
      angle_ = angle;
   }

   /**
    * Gets the composite marker for this marker.
    * @return the composite marker or <code>null</code> if this 
    * marker doesn't have a composite
    */
   public JDRMarker getCompositeMarker()
   {
      return composite;
   }

   /**
    * Gets this marker's parent if this marker is a composite
    * marker.
    * @return parent marker or <code>null</code> if this marker
    * is a primary marker
    */
   public JDRMarker getParentMarker()
   {
      return parent;
   }

   /**
    * Sets the composite marker for this marker.
    * This will override any previous composite marker.
    * Setting the composite marker to <code>null</code>
    * will mean this marker no longer has a composite.
    * @param marker the composite marker or <code>null</code>
    * to remove current composite marker
    * @throws InvalidCompositeMarkerException if this marker
    * is of type {@link #ARROW_NONE}
    */
   public void setCompositeMarker(JDRMarker marker)
      throws InvalidCompositeMarkerException
   {
      if (marker == null)
      {
         composite = null;
         return;
      }
      else if (marker.getType() == ARROW_NONE)
      {
         composite = null;
         return;
      }

      if (getType() == ARROW_NONE)
         throw new InvalidCompositeMarkerException();

      composite = marker;
      composite.setPenWidth(penWidth);
      composite.parent = this;

      updateCompositeOffset();
   }

   /**
    * Updates this marker's knowledge of the line width of the 
    * associated path. This should be called whenever the 
    * line width of the associated path is changed.
    * @param penwidth the line width of the associated path
    */
   public void setPenWidth(double penwidth)
   {
      penWidth = penwidth;
      if (!userRepeatOffset)
      {
         repeatOffset = 7.0*penWidth;
      }
      if (composite != null) composite.setPenWidth(penwidth);
      updateCompositeOffset();
   }

   /**
    * Gets line width of this marker's associated path.
    * @return the line width of the associated path.
    */
   public double getPenWidth()
   {
      return penWidth;
   }

   /**
    * Sets the fill paint for this marker. If the fill paint
    * is transparent or <code>null</code>, the marker uses the 
    * line paint of the associated path.
    * @param paint the paint for this marker
    * @see #getFillPaint()
    */
   public void setFillPaint(JDRPaint paint)
   {
      if (paint instanceof JDRTransparent)
      {
         fillPaint = null;
      }
      else
      {
         fillPaint = paint;
      }
   }

   /**
    * Gets the paint for this marker. If <code>null</code>, this
    * marker uses the line paint of the associated path.
    * @return this marker's paint or <code>null</code> if paint 
    * dependent on associated path
    * @see #setFillPaint(JDRPaint)
    */
   public JDRPaint getFillPaint()
   {
      return fillPaint;
   }

   public void fade(double factor)
   {
      if (fillPaint != null)
      {
         fillPaint.fade(factor);
      }
   }

   /**
    * Sets the repeat factor for this marker.
    * @param factor the repeat factor
    * @see #getRepeated()
    * @throws InvalidRepeatValueException if repeat factor is less
    * than 1
    */
   public void setRepeated(int factor)
      throws InvalidRepeatValueException
   {
      if (factor < 1)
      {
         throw new InvalidRepeatValueException(factor);
      }

      repeated = factor;
      updateCompositeOffset();
   }

   /**
    * Gets the repeat factor for this marker.
    * @return the number of replicates
    * @see #setRepeated(int)
    */
   public int getRepeated()
   {
      return repeated;
   }

   /**
    * Sets whether or not to reverse this marker.
    * @param isReversed this should be <code>true</code> if this
    * marker should be reversed, otherwise <code>false</code>
    * @see #isReversed()
    */
   public void setReversed(boolean isReversed)
   {
      reversed = isReversed;
      updateCompositeOffset();
   }

   /**
    * Gets whether or not to reverse this marker.
    * @return <code>true</code> if this marker is reversed, 
    * otherwise <code>false</code>
    * @see #setReversed(boolean)
    */
   public boolean isReversed()
   {
      return reversed;
   }

   /**
    * Sets this marker's size, if the marker has a size setting.
    * (Some markers ignore the size setting.) The marker size
    * may also depend on the line width of the associated
    * path.
    * @param markerSize the size of the marker
    * @see #getSize()
    */
   public void setSize(double markerSize)
   {
      size = markerSize;
      updateCompositeOffset();
   }

   /**
    * Gets this marker's size. Some markers may ignore the
    * size setting, and some may also depend on the line
    * width of the associated path.
    * @see #setSize(double)
    * @return the size of this marker
    */
   public double getSize()
   {
      return size;
   }

   /**
    * Sets the overlay property for this marker.
    * If the overlay property is enabled, the composite
    * marker will overlap the primary marker, otherwise
    * the composite marker will be offset from the
    * primary marker.
    * @param isOverlaid if <code>true</code> enable overlay
    * property for this marker, otherwise disable it
    * @see #isOverlaid()
    */
   public void setOverlay(boolean isOverlaid)
   {
      overlay = isOverlaid;
      updateCompositeOffset();
   }

   /**
    * Gets the overlay property for this marker.
    * @see #setOverlay(boolean)
    * @return <code>true</code> if composite marker should overlap
    * primary marker, <code>false</code> otherwise
    */
   public boolean isOverlaid()
   {
      return overlay;
   }

   /**
    * Updates the composite marker offset. Required when 
    * marker settings are updated.
    */
   private void updateCompositeOffset()
   {
      if (composite != null)
      {
         if (composite.userOffset)
         {
            return;
         }

         if (overlay)
         {
            composite.offset_ = offset_;
         }
         else if (!userOffset)
         {
            composite.offset_ =
               getPrimaryGeneralPath().getBounds2D().getWidth()
               + repeatOffset - offset_
               - getGeneralPath().getBounds2D().getWidth();
         }
      }
   }

   /**
    * Enables or disables whether this marker's offset can be
    * specified by the user. If this is enabled, the user can
    * specify the offset using {@link #setOffset(double)}, otherwise
    * the offset is zero if this marker is the primary marker or
    * the offset is computed from the parent marker's bounds and
    * the line width if this marker is a composite marker.
    * @param flag if true, allow user to specify the offset
    * @see #setOffset(double)
    * @see #isUserOffsetEnabled()
    */
   public void enableUserOffset(boolean flag)
   {
      userOffset = flag;

      if (!userOffset)
      {
         if (parent == null)
         {
            offset_ = 0;
         }
         else
         {
            parent.updateCompositeOffset();
         }

         updateCompositeOffset();
      }
   }

   /**
    * Enables or disables whether this marker's repeat offset can be
    * specified by the user. If this is enabled, the user can
    * specify the offset using {@link #setRepeatOffset(double)}, 
    * the offset is given by 7 times the line width.
    * @param flag if true, allow user to specify the repeat offset
    * @see #setRepeatOffset(double)
    * @see #isUserRepeatOffsetEnabled()
    */
   public void enableUserRepeatOffset(boolean flag)
   {
      userRepeatOffset = flag;

      if (!userRepeatOffset)
      {
         repeatOffset = 7.0*penWidth;

         updateCompositeOffset();
      }
   }

   /**
    * Determines whether this marker's offset can be specified
    * using {@link #setOffset(double)}
    * @return true if user can specify this marker's offset
    * @see #setOffset(double)
    * @see #enableUserOffset(boolean)
    */
   public boolean isUserOffsetEnabled()
   {
      return userOffset;
   }

   /**
    * Determines whether this marker's offset can be specified
    * using {@link #setRepeatOffset(double)}
    * @return true if user can specify this marker's repeat offset
    * @see #setRepeatOffset(double)
    * @see #enableUserRepeatOffset(boolean)
    */
   public boolean isUserRepeatOffsetEnabled()
   {
      return userRepeatOffset;
   }

   /**
    * Sets the offset for this marker if user offset flag is
    * enabled. This method only has an effect if
    * {@link #isUserOffsetEnabled()} returns true.
    * @param offset the new value of the offset
    * @see #setRepeatOffset(double)
    */
   public void setOffset(double offset)
   {
      if (userOffset)
      {
         offset_ = offset;
      }
   }

   /**
    * Gets this marker's offset.
    * @return this marker's offset
    */
   public double getOffset()
   {
      return offset_;
   }

   /**
    * Sets the repeat offset for this marker if user offset flag is
    * enabled. This method only has an effect if
    * {@link #isUserOffsetEnabled()} returns true.
    * @param offset the new value of the repeat offset (gap between
    * repeat markers)
    * @see #setOffset(double)
    */
   public void setRepeatOffset(double offset)
   {
      if (userRepeatOffset)
      {
         repeatOffset = offset;
      }
   }

   /**
    * Gets the gap between repeat markers.
    * @return repeat offset
    */
   public double getRepeatOffset()
   {
      return repeatOffset;
   }

   /**
    * Gets the number of known markers for the latest version.
    * @return number of known markers
    */
   public static int maxMarkers()
   {
      return NUM_ARROWS1_6;
   }

   /**
    * Auto orient property. If <code>true</code>, this marker
    * is oriented to align with the associated path's gradient,
    * otherwise it is oriented by {@link #angle_}
    */
   protected boolean autoOrient_=true;
   /**
    * Angle (in radians) to rotate this marker if auto orient 
    * property is disabled.
    * @see #autoOrient_
    */
   protected double angle_=0.0f;
   /**
    * Fill paint for this marker. If <code>null</code>, the
    * line paint for the associated path is used instead.
    */
   public JDRPaint fillPaint=null;
   /**
    * Line width of the associated path. This value must
    * be updated when the line width of the associated path is
    * changed. (Some marker sizes are dependent on the line 
    * width.)
    */
   protected double penWidth=1.0f;

   /**
    * This marker's repeat factor.
    */
   protected int repeated=1;
   /**
    * This marker's directional setting.
    */
   protected boolean reversed=false;
   /**
    * This marker's size property. Some markers ignore this
    * value.
    */
   protected double size=5.0;

   /**
    * This marker's composite marker. If <code>null</code>, this
    * marker has no secondary marker.
    */
   protected JDRMarker composite=null;

   /**
    * This marker's parent. If <code>null</code>, this marker is
    * a primary marker.
    */
   protected JDRMarker parent = null;

   /**
    * Overlay property for this marker. If <code>true</code>, the
    * composite marker overlaps this marker.
    */

   protected boolean overlay=false;
   /**
    * The offset from the vertex to the marker origin.
    * This must be updated whenever the marker properties are
    * changed.
    */
   private double offset_=0;

   /**
    * Determines whether the user has specified an explicit value
    * for this marker's offset.
    */
   private boolean userOffset = false;

   /**
    * The offset between repeated markers.
    */
   private double repeatOffset=0;

   /**
    * Determines whether the user has specified an explicit value
    * for this marker's repeat offset.
    */
   private boolean userRepeatOffset = false;

   /**
    * Identifying number for this marker type, as used in 
    * the AJR and JDR file formats.
    */
   protected int type=ARROW_NONE;

   /**
    * No marker.
    */
   public static final int ARROW_NONE=0;
   /**
    * Pointed marker corresponding to \pgfarrowpointed.
    */
   public static final int ARROW_POINTED=1;
   /**
    * Triangle marker corresponding to \pgfarrowtriangle.
    */
   public static final int ARROW_TRIANGLE=2;
   /**
    * Circle marker corresponding to \pgfarrowcircle.
    */
   public static final int ARROW_CIRCLE=3;
   /**
    * Diamond marker corresponding to \pgfarrowdiamond.
    */
   public static final int ARROW_DIAMOND=4;
   /**
    * Square marker corresponding to \pgfarrowsquare.
    */
   public static final int ARROW_SQUARE=5;
   /**
    * Bar marker corresponding to \pgfarrowbar.
    */
   public static final int ARROW_BAR=6;
   /**
    * LaTeX arrow style marker corresponding to \pgfarrowsingle.
    */
   public static final int ARROW_SINGLE=7;
   /**
    * Round bracket marker corresponding to \pgfarrowround.
    */
   public static final int ARROW_ROUND=8;
   /**
    * Filled dot marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_DOTFILLED=9;
   /**
    * Open dot marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_DOTOPEN=10;
   /**
    * Filled box marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_BOXFILLED=11;
   /**
    * Open box marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_BOXOPEN=12;
   /**
    * Cross marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_CROSS=13;
   /**
    * Plus marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_PLUS=14;
   /**
    * Star marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_STAR=15;
   /**
    * Filled up triangle marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_TRIANGLE_UP_FILLED=16;
   /**
    * Open up triangle marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_TRIANGLE_UP_OPEN=17;
   /**
    * Filled down triangle marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_TRIANGLE_DOWN_FILLED=18;
   /**
    * Open down triangle marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_TRIANGLE_DOWN_OPEN=19;
   /**
    * Filled rhombus marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_RHOMBUS_FILLED=20;
   /**
    * Open rhombus marker.
    * (JDR/AJR file formats version 1.1 onwards.)
    */
   public static final int ARROW_RHOMBUS_OPEN=21;

   /**
    * Filled pentagon marker.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_PENTAGON_FILLED=22;

   /**
    * Open pentagon marker.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_PENTAGON_OPEN=23;

   /**
    * Filled hexagon marker.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HEXAGON_FILLED=24;

   /**
    * Open hexagon marker.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HEXAGON_OPEN=25;

   /**
    * Filled octagon marker.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_OCTAGON_FILLED=26;

   /**
    * Open octagon marker.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_OCTAGON_OPEN=27;

   /**
    * Pointed arrow (60 degree angle).
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_POINTED60=28;

   /**
    * Pointed arrow (45 degree angle).
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_POINTED45=29;

   /**
    * Hooks arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HOOKS=30;

   /**
    * Hook up arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HOOK_UP=31;

   /**
    * Hook down arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HOOK_DOWN=32;

   /**
    * Upper half pointed arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HALF_POINTED_UP=33;

   /**
    * Lower half pointed arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HALF_POINTED_DOWN=34;

   /**
    * Upper half pointed 60 arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HALF_POINTED60_UP=35;

   /**
    * Lower half pointed 60 arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HALF_POINTED60_DOWN=36;

   /**
    * Upper half pointed 45 arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HALF_POINTED45_UP=37;

   /**
    * Lower half pointed 45 arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HALF_POINTED45_DOWN=38;

   /**
    * Cusp arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_CUSP=39;

   /**
    * Upper half cusp.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HALF_CUSP_UP=40;

   /**
    * Lower half cusp.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_HALF_CUSP_DOWN=41;

   /**
    * Alternative LaTeX style arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_ALT_SINGLE=42;

   /**
    * Outline LaTeX style arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_ALT_SINGLE_OPEN=43;

   /**
    * Outline of right triangle arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_TRIANGLE_OPEN=44;

   /**
    * Outline of circle arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_CIRCLE_OPEN=45;

   /**
    * Outline of diamond arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_DIAMOND_OPEN=46;

   /**
    * Brace arrow.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_BRACE=47;

   /**
    * Rectangle cap.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_RECTANGLE_CAP=48;

   /**
    * Chevron cap.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_CHEVRON_CAP=49;

   /**
    * Fast cap.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_FAST_CAP=50;

   /**
    * Round cap.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_ROUND_CAP=51;

   /**
    * Triangle cap.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_TRIANGLE_CAP=52;

   /**
    * Inverted Triangle cap.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_INVERT_TRIANGLE_CAP=53;

   /**
    * Inverted Chevron cap.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_INVERT_CHEVRON_CAP=54;

   /**
    * Inverted Fast cap.
    * (JDR/AJR file formats version 1.4 onwards.)
    */
   public static final int ARROW_INVERT_FAST_CAP=55;

   /**
    * Alternative bar marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_ALT_BAR=56;

   /**
    * Alternative round bracket marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_ALT_ROUND=57;

   /**
    * Alternative square bracket marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_ALT_SQUARE=58;

   /**
    * Alternative brace bracket marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_ALT_BRACE=59;

   /**
    * Open semi-circle marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_SEMICIRCLE_OPEN=60;

   /**
    * Closed semi-circle marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_SEMICIRCLE_FILLED=61;

   /**
    * Open 5 pointed star marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_STAR5_OPEN=62;

   /**
    * Filled 5 pointed star marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_STAR5_FILLED=63;

   /**
    * Asterisk marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_ASTERISK=64;

   /**
    * Down partial filled scissor marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_SCISSORS_DOWN_FILLED=65;

   /**
    * Up partial filled scissor marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_SCISSORS_UP_FILLED=66;

   /**
    * Down partial open scissor marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_SCISSORS_DOWN_OPEN=67;

   /**
    * Up partial open scissor marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_SCISSORS_UP_OPEN=68;

   /**
    * Filled right pointing heart shaped marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_HEART_RIGHT_FILLED=69;

   /**
    * Open right pointing heart shaped marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_HEART_RIGHT_OPEN=70;

   /**
    * Filled heart shaped marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_HEART_FILLED=71;

   /**
    * Open heart shaped marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_HEART_OPEN=72;

   /**
    * Snowflake shaped marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_SNOWFLAKE=73;

   /**
    * Open star chevron marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_STAR_CHEVRON_OPEN=74;

   /**
    * Filled star chevron marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_STAR_CHEVRON_FILLED=75;

   /**
    * Filled 6 pointed star marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_STAR6_FILLED=76;

   /**
    * Open 6 pointed star marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_STAR6_OPEN=77;

   /**
    * Filled equilateral marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_EQUILATERAL_FILLED=78;

   /**
    * Open equilateral marker.
    * (JDR/AJR file formats version 1.4 onwards)
    */
   public static final int ARROW_EQUILATERAL_OPEN=79;

   /**
    * Maximum number of known markers for AJR and JDR file
    * versions 1.6 onwards
    */
   public static final int NUM_ARROWS1_6=91;

   /**
    * Maximum number of known markers for AJR and JDR file
    * versions 1.4 onwards
    */
   public static final int NUM_ARROWS1_4=80;

   /**
    * Maximum number of known markers for AJR and JDR file 
    * versions 1.1 to 1.3.
    */
   public static final int NUM_ARROWS1_1=22;
   /**
    * Maximum number of known markers for AJR/JDR file version
    * 1.0.
    */
   public static final int NUM_ARROWS1_0=8;

   /**
    * Ball cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_BALL_CAP=80;

   /**
    * Leaf cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_LEAF_CAP=81;

   /**
    * Double leaf cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_LEAF2_CAP=82;

   /**
    * Triple leaf cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_LEAF3_CAP=83;

   /**
    * Club cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_CLUB_CAP=84;

   /**
    * Triple leaf forward cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_LEAF3FOR_CAP=85;

   /**
    * Triple leaf backwards cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_LEAF3BACK_CAP=86;

   /**
    * Double leaf forward cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_LEAF2FOR_CAP=87;

   /**
    * Double leaf backwards cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_LEAF2BACK_CAP=88;

   /**
    * Bulge cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_BULGE_CAP=89;

   /**
    * Cutout Bulge cap.
    * (JDR/AJR file formats version 1.6 onwards.)
    */
   public static final int ARROW_CUTOUTBULGE_CAP=90;

   /**
    * Indicates whether or not this marker is at the start of
    * a segment.
    */
   private boolean isStart_=false;

   public static final double HALF_PI = 0.5*Math.PI;

   public static final double QUARTER_PI = 0.25*Math.PI;

   public static final double THREEQUARTER_PI = 0.75*Math.PI;

   public static final double HALF_ROOT_3 = 0.866025404;

   public static final double ROOT_3 = 1.732050808;

   public static final double ONE_OVER_ROOT_3 = 0.577350269;
}
