// File          : ArrowTriangleDownOpen.java
// Date          : 1st February 2006
// Last Modified : 28th July 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Open downward facing triangle marker.
 * The basic open downward facing triangle marker shape looks like:
 * <img src="../images/triangleDownOpenMarker.png" alt="[open downward facing triangle]">
 * (the origin is at the centre of the shape.)
 * This marker's shape does not depend on the associated path's
 * line width. Instead it depends on the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 * @author Nicola L C Talbot
 */
public class ArrowTriangleDownOpen extends JDRMarker
{
   /**
    * Creates open downward facing triangle marker of given size.
    * The marker may be repeated and/or reversed. Since the 
    * marker is symmetrical, the reversed setting only has an
    * effect if the marker is repeated.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowTriangleDownOpen(double penwidth, int repeat,
                      boolean isReversed, double markerSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, markerSize);

      type = ARROW_TRIANGLE_DOWN_OPEN;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"triangledownopen":
           "arrow-"+repeated+"triangledownopen";
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      GeneralPath path = new GeneralPath();

      double length = size-1;

      double x = length*0.866025;
      double y = length*0.5;

      path.moveTo(0.0f, (float)length);
      path.lineTo((float)-x, (float)-y);
      path.lineTo((float)x, (float)-y);
      path.closePath();

      BasicStroke stroke = new BasicStroke(2.0f);

      Shape shape = stroke.createStrokedShape(path);

      return new GeneralPath(shape);
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return false;
   }

   /**
    * @deprecated
    */
   protected String pgfarrow()
   {
      double x = size*0.866025;
      double y = size*0.5;

      return "\\pgfsetdash{}{0pt}\\pgfsetlinewidth{2bp}"
           + "\\pgfpathqmoveto{0bp}{"+PGF.format(-size)+"bp}"
           + "\\pgfpathqlineto{"+PGF.format(-x)+"bp}{"
           + PGF.format(y)+"bp}"
           + "\\pgfpathqlineto{"+PGF.format(x)+"bp}{"
           + PGF.format(y)+"bp}"
           + "\\pgfclosepath"
           + "\\pgfusepathqstroke";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowTriangleDownOpen(penWidth, repeated,
                                         reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }

   public boolean hasXAxisSymmetry()
   {
      return false;
   }
}
