// File          : ArrowTriangle.java
// Date          : 1st February 2006
// Last Modified : 28th July 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Right facing triangle marker.
 * The basic right facing triangle marker shape looks like:
 * <img src="../images/triangleMarker.png" alt="[filled right facing triangle]">
 * (the origin is to the right of the centre of the shape.)
 * This corresponds to the PGF <code>\pgfarrowtriangle</code>
 * arrow style.
 * This marker's shape depends on the associated path's
 * line width as well as the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 * @author Nicola L C Talbot
 */
public class ArrowTriangle extends JDRMarker
{
   /**
    * Creates right facing triangle marker of given size.
    * The marker may be repeated and/or reversed.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowTriangle(double penwidth, int repeat,
                       boolean isReversed, double arrowSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, arrowSize);

      type = ARROW_TRIANGLE;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"triangle-"+size+"-"+penWidth:
           "arrow-"+repeated+"triangle-"+size+"-"+penWidth;
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      GeneralPath path = new GeneralPath();

      double xa = size + 0.7*penWidth;
      double ya = 0.577*size + 0.7*penWidth;

      path.moveTo((float)-xa, (float)-ya);
      path.lineTo((float)(size*penWidth*0.65), 0.0f);
      path.lineTo((float)-xa, (float)ya);
      path.closePath();

      return path;
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return true;
   }

   /**
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "\\pgfarrowtriangle{"+PGF.format(size)+"bp}";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowTriangle(penWidth, repeated,
                                    reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }
}
