// File          : ArrowStar6Filled.java
// Date          : 9th July 2008
// Last Modified : 9th July 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Filled 6-pointed star marker. The basic marker shape looks like:
 * <img src="../images/star6FilledMarker.png" alt="[filled 6 pointed star]">
 * (the origin is at the centre of the shape.)
 * This marker's shape does not depend on the associated path's
 * line width. Instead it depends on the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 * @author Nicola L C Talbot
 */
public class ArrowStar6Filled extends JDRMarker
{
   /**
    * Creates filled 6 pointed star marker of given size.
    * The marker may be repeated and/or reversed.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowStar6Filled(double penwidth, int repeat,
                      boolean isReversed, double markerSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, markerSize);

      type = ARROW_STAR6_FILLED;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"star6filled-"+size:
           "arrow-"+repeated+"star6filled-"+size;
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      GeneralPath path = new GeneralPath();

      double root3over2 = 0.866025404;

      double shortradius = 0.5*size;

      double x0 = size*root3over2;
      double y0 = size*0.5;

      double x1 = shortradius*0.5;
      double y1 = shortradius*root3over2;

      path.moveTo((float)shortradius, 0.0f);
      path.lineTo((float)x0, (float)y0);
      path.lineTo((float)x1, (float)y1);
      path.lineTo((float)0, (float)size);
      path.lineTo((float)-x1, (float)y1);
      path.lineTo((float)-x0, (float)y0);
      path.lineTo((float)-shortradius, 0.0f);
      path.lineTo((float)-x0, (float)-y0);
      path.lineTo((float)-x1, (float)-y1);
      path.lineTo((float)0, (float)-size);
      path.lineTo((float)x1, (float)-y1);
      path.lineTo((float)x0, (float)-y0);
      path.closePath();

      return path;
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return false;
   }

   /**
    * Not implemented (returns empty string).
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowStar6Filled(penWidth, repeated,
                                     reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }

   public boolean hasXAxisSymmetry()
   {
      return true;
   }
}
