// File          : ArrowStar5Open.java
// Date          : 9th May 2008
// Last Modified : 9th May 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Open 5-pointed star marker. The basic marker shape looks like:
 * <img src="../images/star5OpenMarker.png" alt="[open 5 pointed star]">
 * (the origin is at the centre of the shape.)
 * This marker's shape does not depend on the associated path's
 * line width. Instead it depends on the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 * @author Nicola L C Talbot
 */
public class ArrowStar5Open extends JDRMarker
{
   /**
    * Creates open 5 pointed star marker of given size.
    * The marker may be repeated and/or reversed.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowStar5Open(double penwidth, int repeat,
                      boolean isReversed, double markerSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, markerSize);

      type = ARROW_STAR5_OPEN;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"star5open-"+size:
           "arrow-"+repeated+"star5open-"+size;
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      GeneralPath path = new GeneralPath();

      double length = size-1;

      double x0 = length*ArrowPentagonOpen.sinAngle;
      double y0 = length*ArrowPentagonOpen.cosAngle;
      double x1 = length*ArrowPentagonOpen.sinHalfAngle;
      double y1 = length*ArrowPentagonOpen.cosHalfAngle;

      double halfx0 = 0.5*x0;
      double halfy0 = 0.5*y0;
      double halfx1 = 0.5*x1;
      double halfy1 = 0.5*y1;

      path.moveTo(0.0f, -(float)length);
      path.lineTo(-(float)halfx1, -(float)halfy1);
      path.lineTo(-(float)x0, -(float)y0);
      path.lineTo(-(float)halfx0, (float)halfy0);
      path.lineTo(-(float)x1, (float)y1);
      path.lineTo(0.0f, (float)(length*0.5));
      path.lineTo((float)x1, (float)y1);
      path.lineTo((float)halfx0, (float)halfy0);
      path.lineTo((float)x0, -(float)y0);
      path.lineTo((float)halfx1, -(float)halfy1);
      path.closePath();

      BasicStroke stroke = new BasicStroke(2.0f);

      return new GeneralPath(stroke.createStrokedShape(path));
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return false;
   }

   /**
    * Not implemented (returns empty string).
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowStar5Open(penWidth, repeated,
                                     reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }

   public boolean hasXAxisSymmetry()
   {
      return false;
   }
}
