// File          : ArrowStar.java
// Date          : 1st February 2006
// Last Modified : 28th July 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Star marker. The basic star marker shape looks like:
 * <img src="../images/starMarker.png" alt="[star shaped]">
 * (the origin is at the centre of the shape.)
 * This marker's shape does not depend on the associated path's
 * line width. Instead it depends on the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 * @author Nicola L C Talbot
 */
public class ArrowStar extends JDRMarker
{
   /**
    * Creates star marker of given size.
    * The marker may be repeated and/or reversed. Since the 
    * marker is symmetrical, the reversed setting only has an
    * effect if the marker is repeated.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowStar(double penwidth, int repeat,
                      boolean isReversed, double markerSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, markerSize);

      type = ARROW_STAR;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"star":
           "arrow-"+repeated+"star";
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      GeneralPath path = new GeneralPath();

      path.moveTo((float)-size, 0.0f);
      path.lineTo((float)size, 0.0f);
      path.moveTo(0.0f, (float)size);
      path.lineTo(0.0f, (float)-size);
      // as above but rotated 45 degress
      double a = size*0.707107;
      path.moveTo((float)-a, (float)-a);
      path.lineTo((float)a, (float)a);
      path.moveTo((float)a, (float)-a);
      path.lineTo((float)-a, (float)a);

      BasicStroke stroke = new BasicStroke(2.0f);

      Shape shape = stroke.createStrokedShape(path);

      return new GeneralPath(shape);
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return false;
   }

   /**
    * @deprecated
    */
   protected String pgfarrow()
   {
      double a = size*0.707107;

      return "\\pgfsetdash{}{0pt}\\pgfsetlinewidth{2bp}"
           + "\\pgfpathqmoveto{"+PGF.format(-size)+"bp}{0bp}"
           + "\\pgfpathqlineto{"+PGF.format(size)+"bp}{0bp}"
           + "\\pgfpathqmoveto{0bp}{"+PGF.format(size)+"bp}"
           + "\\pgfpathqlineto{0bp}{"+PGF.format(-size)+"bp}"
           + "\\pgfpathqmoveto{"+PGF.format(-a)+"bp}{"
           + PGF.format(-a)+"bp}"
           + "\\pgfpathqlineto{"+PGF.format(a)+"bp}{"
           + PGF.format(a)+"bp}"
           + "\\pgfpathqmoveto{"+PGF.format(a)+"bp}{"
           + PGF.format(-a)+"bp}"
           + "\\pgfpathqlineto{"+PGF.format(-a)+"bp}{"
           + PGF.format(a)+"bp}"
           + "\\pgfusepathqstroke";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowStar(penWidth, repeated,
                                reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }
}
