// File          : ArrowHalfCuspUp.java
// Date          : 28th April 2008
// Last Modified : 28th April 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Half cusp up marker. The basic marker shape looks like:
 * <img src="../images/halfCuspUpMarker.png" alt="[ upper half cusp arrow shape]">
 * (the origin is at the bottom right of the shape.)
 * This marker's shape depends on the associated path's
 * line width as well as the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 * @author Nicola L C Talbot
 */
public class ArrowHalfCuspUp extends JDRMarker
{
   /**
    * Creates upper half of cusp marker of given size.
    * The marker may be repeated and/or reversed.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowHalfCuspUp(double penwidth, int repeat,
                       boolean isReversed, double arrowSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, arrowSize);

      type = ARROW_HALF_CUSP_UP;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"halfcuspup-"+size+"-"+penWidth:
           "arrow-"+repeated+"halfcuspup-"+size+"-"+penWidth;
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      GeneralPath path = new GeneralPath();

      path.moveTo(0, 0);
      path.curveTo((float)(-size*0.5), 0.0f,
                   (float)(-size), (float)(-0.25*size),
                   (float)-size, (float)(-0.5*size));

      BasicStroke stroke = new BasicStroke((float)penWidth);
      Shape shape = stroke.createStrokedShape(path);

      return new GeneralPath(shape);
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return true;
   }

   public boolean hasXAxisSymmetry()
   {
      return false;
   }

   /**
    * Not implemented (returns empty string).
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowHalfCuspUp(penWidth, repeated,
                                   reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }
}
