// File          : ArrowCross.java
// Date          : 1st February 2006
// Last Modified : 28 July 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Cross marker. The basic cross marker shape looks like:
 * <img src="../images/crossMarker.png" alt="[X shaped]">
 * (the origin is at the centre of the shape.)
 * This marker's shape does not depend on the associated path's
 * line width. Instead it depends on the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 * @author Nicola L C Talbot
 */
public class ArrowCross extends JDRMarker
{
   /**
    * Creates cross marker of given size.
    * The marker may be repeated and/or reversed. Since the 
    * marker is symmetrical, the reversed setting only has an
    * effect if the marker is repeated.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowCross(double penwidth, int repeat,
                      boolean isReversed, double markerSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, markerSize);

      type = ARROW_CROSS;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"cross":
           "arrow-"+repeated+"cross";
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      GeneralPath path = new GeneralPath();

      path.moveTo((float)-size, (float)-size);
      path.lineTo((float)size, (float)size);
      path.moveTo((float)-size, (float)size);
      path.lineTo((float)size, (float)-size);

      BasicStroke stroke = new BasicStroke(2.0f);

      Shape shape = stroke.createStrokedShape(path);

      return new GeneralPath(shape);
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return false;
   }

   /**
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "\\pgfsetdash{}{0pt}\\pgfsetlinewidth{2bp}"
           + "\\pgfpathqmoveto{"+PGF.format(-size)+"bp}{"
           + PGF.format(-size)+"bp}"
           + "\\pgfpathqlineto{"+PGF.format(size)+"bp}{"
           + PGF.format(size)+"bp}"
           + "\\pgfpathqmoveto{"+PGF.format(-size)+"bp}{"
           + PGF.format(size)+"bp}"
           + "\\pgfpathqlineto{"+PGF.format(size)+"bp}{"
           + PGF.format(-size)+"bp}"
           + "\\pgfusepathqstroke";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowCross(penWidth, repeated,
                                 reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }
}
