// File          : ArrowCircle.java
// Date          : 1st February 2006
// Last Modified : 28th April 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Circle marker. The basic circle marker shape looks like:
 * <img src="../images/circleMarker.png" alt="[filled circle]">
 * (the origin is off the centre of the shape.)
 * This corresponds to the PGF <code>\pgfarrowcircle</code> arrow style.
 * This marker's shape depends on the associated path's
 * line width as well as the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 * @author Nicola L C Talbot
 */
public class ArrowCircle extends JDRMarker
{
   /**
    * Creates filled circle marker of given size.
    * The marker may be repeated and/or reversed.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowCircle(double penwidth, int repeat,
                      boolean isReversed, double arrowSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, arrowSize);

      type = ARROW_CIRCLE;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"circle-"+size+"-"+penWidth:
           "arrow-"+repeated+"circle-"+size+"-"+penWidth;
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      double radius = size + 0.6*penWidth;

      // centre of circle
      double x = 0.5*penWidth - size;
      double y = 0.0;

      Ellipse2D circle = new Ellipse2D.Float(
         (float)(x-radius), (float)(y-radius),
         (float)(2.0*radius), (float)(2.0*radius));

      return new GeneralPath(circle);
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return true;
   }

   /**
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "\\pgfarrowcircle{"+PGF.format(size)+"bp}";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowCircle(penWidth, repeated,
                                  reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }
}
