// File          : ArrowBoxOpen.java
// Date          : 1st February 2006
// Last Modified : 10th July 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Open box marker. The basic open box marker shape looks like:
 * <img src="../images/boxOpenMarker.png" alt="[open square]">
 * (the origin is at the centre of the shape.)
 * This marker's shape does not depend on the associated path's
 * line width. Instead it depends on the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 */
public class ArrowBoxOpen extends JDRMarker
{
   /**
    * Creates open box marker of given size.
    * The marker may be repeated and/or reversed. Since the box
    * marker is symmetrical, the reversed setting only has an
    * effect if the marker is repeated.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowBoxOpen(double penwidth, int repeat,
                      boolean isReversed, double markerSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, markerSize);

      type = ARROW_BOXOPEN;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"boxopen":
           "arrow-"+repeated+"boxopen";
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      double length = size-1;

      GeneralPath path = new GeneralPath(
         new Rectangle2D.Float((float)-length, (float)-length,
                             (float)(2.0*length), (float)(2.0*length)));

      BasicStroke stroke = new BasicStroke(2.0f);

      Shape shape = stroke.createStrokedShape(path);

      return new GeneralPath(shape);
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return false;
   }

   /**
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "\\pgfsetdash{}{0pt}\\pgfsetlinewidth{2bp}"
           + "\\pgfpathrectangle{\\pgfpoint{"+PGF.format(-size)+"bp}{"
           + PGF.format(-size) + "bp}}{\\pgfpoint{"
           + PGF.format(2*size)+"bp}{"
           + PGF.format(2*size) + "bp}}"
           + "\\pgfusepathqstroke";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowBoxOpen(penWidth, repeated,
                                   reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }
}
