// File          : ArrowBoxFilled.java
// Date          : 1st February 2006
// Last Modified : 13th January 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.marker;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Filled box marker. The basic filled box marker shape looks like:
 * <img src="../images/boxFilledMarker.png" alt="[filled square]">
 * (the origin is at the centre of the shape.)
 * This marker's shape does not depend on the associated path's
 * line width. Instead it depends on the given marker size.
 * See {@link JDRMarker} for a description of markers.
 *
 */
public class ArrowBoxFilled extends JDRMarker
{
   /**
    * Creates filled box marker of given size.
    * The marker may be repeated and/or reversed. Since the box
    * marker is symmetrical, the reversed setting only has an
    * effect if the marker is repeated.
    * @throws InvalidRepeatValueException if repeat value is less 
    * than 1
    */
   public ArrowBoxFilled(double penwidth, int repeat,
                      boolean isReversed, double markerSize)
   throws InvalidRepeatValueException
   {
      super(penwidth, repeat, isReversed, markerSize);

      type = ARROW_BOXFILLED;
   }

   public String getID()
   {
      return reversed ?
           "arrow-r"+repeated+"boxfilled":
           "arrow-"+repeated+"boxfilled";
   }

   /**
    * Gets the path describing the basic shape of this marker.
    */
   public GeneralPath getGeneralPath()
   {
      GeneralPath path = new GeneralPath(
         new Rectangle2D.Float((float)-size, (float)-size,
                             (float)(2*size), (float)(2*size)));

      return path;
   }

   public boolean isResizable()
   {
      return true;
   }

   public boolean usesLineWidth()
   {
      return false;
   }

   /**
    * @deprecated
    */
   protected String pgfarrow()
   {
      return "\\pgfsetdash{}{0pt}\\pgfsetlinewidth{2bp}"
           + "\\pgfpathrectangle{\\pgfpoint{"+PGF.format(-size)+"bp}{"
           + PGF.format(-size) + "bp}}{\\pgfpoint{"
           + PGF.format(2*size)+"bp}{"
           + PGF.format(2*size) + "bp}}"
           + "\\pgfusepathqfillstroke";
   }

   public Object clone()
   {
      JDRMarker marker = null;

      try
      {
         marker = new ArrowBoxFilled(penWidth, repeated,
                                     reversed, size);
         makeEqual(marker);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      return marker;
   }
}
