// File          : PGF.java
// Purpose       : functions to save JDRGroup as a pgfpicture environment
// Date          : 1st February 2006
// Last Modified : 13th March 2008
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr.io;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;
import java.text.*;
import uk.ac.uea.cmp.nlct.jdr.*;

/**
 * Functions to save JDRGroup as a pgfpicture environment for use
 * with the pgf package. Note that this library can only export
 * to LaTeX files and can't load them.
 * @author Nicola L C Talbot
 */

public class PGF
{
   /**
    * Saves the image as a pgfpicture environment.
    * @param allObjects all the objects that constitute the image
    * @param out output stream
    * @param creator name of application that calls this method
    * @throws IOException if I/O error occurs
    */
   public static void save(JDRGroup allObjects, PrintWriter out,
      String creator)
      throws IOException
   {
      BBox box = allObjects.getBBox();

      double shiftX = -box.getMinX();
      double shiftY = -box.getMaxY();

      // transformation to convert from left handed
      // co-ordinate system to right-hand co-ordinate system
      AffineTransform af = new AffineTransform(
         1, 0, 0, -1, shiftX, -shiftY);

      out.println("% "+creator);
      out.println("% "+dateFormat.format(new Date()));
      out.println("\\begin{pgfpicture}{0bp}{0bp}{"+
                  format(box.getWidth())+"bp}{"+
                  format(box.getHeight())+"bp}");
      out.print(allObjects.pgf(af));
      out.println("\\end{pgfpicture}");
   }

   /**
    * Saves the image as a pgfpicture environment in a complete
    * document.
    * @param allObjects all the objects that constitute the image
    * @param paper required paper size
    * @param latexBase LaTeX font base
    * @param preamble additional information for preamble
    * @param out output stream
    * @param creator name of application that calls this method
    * @throws IOException if I/O error occurs
    */
   public static void saveDoc(JDRGroup allObjects,
      JDRPaper paper, LaTeXFontBase latexBase, String preamble,
      PrintWriter out, String creator)
      throws IOException
   {
      BBox box = allObjects.getBBox();

      int normalsize = (int)latexBase.getNormalSize();

      double shiftX = -box.getMinX();
      double shiftY = -box.getMaxY();

      // transformation to convert from left handed
      // co-ordinate system to right-hand co-ordinate system
      AffineTransform af = new AffineTransform(
         1, 0, 0, -1, shiftX, -shiftY);

      out.println("% "+creator);
      out.println("% "+dateFormat.format(new Date()));

      if (normalsize >= 10 || normalsize <= 12)
      {
         out.println("\\documentclass["
           +normalsize+"pt]{article}");
      }
      else if (normalsize == 25)
      {
         out.println("\\documentclass{a0poster}");
      }
      else
      {
         out.println("\\documentclass["
           +normalsize+"pt]{extarticle}");
      }

      out.println("\\usepackage["+paper.tex()
       +",top="+format(box.getMinY())
       +"bp,left="+format(box.getMinX())
       +"bp,width="+format(box.getWidth())
       +"bp,height="+format(box.getHeight())
       +"bp]{geometry}");

      out.println("\\usepackage{pgf}");

      if (allObjects.anyTextPaths())
      {
         out.println("\\usepgflibrary{decorations.text}");
      }

      out.print(preamble);

      out.println("\\pagestyle{empty}");
      out.println("\\begin{document}\\noindent");

      out.println("\\begin{pgfpicture}{0bp}{0bp}{"+
                  format(box.getWidth())+"bp}{"+
                  format(box.getHeight())+"bp}");
      out.print(allObjects.pgf(af));
      out.println("\\end{pgfpicture}");

      out.println("\\end{document}");
   }

   /**
    * Formats a value in decimal notation. (TeX can't handle 
    * exponents.)
    * @param value value to be formatted
    * @return formatted value
    */
   public static String format(double value)
   {
      return df.format(value);
   }

   /**
    * Writes the transformed point in pgf format. This passes
    * the transformed co-ordinates to {@link #point(double,double)}.
    * That is, the transformation is done by Java not by pgf
    * package.
    * @param af transform to apply before printing
    * @param x x co-ordinate
    * @param y y co-ordinate
    * @return string containing pgf command to specify the 
    * transformed point
    * @see #point(double,double)
    */
   public static String point(AffineTransform af, double x, double y)
   {
      Point2D.Double org = new Point2D.Double(x, y);
      Point2D trans = af.transform(org, null);

      return point(trans.getX(), trans.getY());
   }

   /**
    * Gets the given co-ordinates in pgf format.
    * @param x x co-ordinate
    * @param y y co-ordinate
    * @return string containing pgf command to specify the point
    * @see #point(AffineTransform,double,double)
    */
   public static String point(double x, double y)
   {
      return new String("\\pgfpoint{"
         +format(x)+"bp}{" +format(y)+"bp}");
   }

   /**
    * Gets the pgf commands to set the transformation matrix.
    * @param matrix transformation matrix in flat format
    * @return string containing pgf commands to set the transformation
    * matrix
    * @see #transform(AffineTransform)
    */
   public static String transform(double[] matrix)
   {
      return "\\pgftransformcm{"
           + format(matrix[0])+"}{"
           + format(matrix[1])+"}{"
           + format(matrix[2])+"}{"
           + format(matrix[3])+"}{"
           + point(matrix[4], matrix[5])+"}";
   }

   /**
    * Gets the pgf commands to set the transformation matrix.
    * @param af transformation
    * @return string containing pgf commands to set the transformation
    * matrix
    * @see #transform(double[])
    */
   public static String transform(AffineTransform af)
   {
      double[] matrix = new double[6];
      af.getMatrix(matrix);
      transform(matrix);
      return transform(matrix);
   }

   /**
    * Number format to use to print decimal numbers in LaTeX file.
    * @see #format(double)
    */
   public static NumberFormat df
      = new DecimalFormat("#####0.0#####",
        new DecimalFormatSymbols(Locale.ENGLISH));

   /**
    * Gets date format. (The creation date is written to the 
    * LaTeX file.)
    */
   protected static DateFormat dateFormat
      = DateFormat.getDateTimeInstance();
}
