// File          : JDRMessage.java
// Purpose       : Interface for messages
// Date          : 12th June 2008
// Last Modified : 12th June 2008
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr.io;

import java.io.*;

import uk.ac.uea.cmp.nlct.jdr.*;

/**
 * Interface for messages.
 * @author Nicola L C Talbot
 */

public interface JDRMessage
{
   /**
    * Sets whether the progress is indeterminate.
    * @param indeterminate <code>true</code> if progress is
    * indeterminate
    */
   public void setIndeterminate(boolean indeterminate);

   /**
    * Gets whether progress is indeterminate.
    */
   public boolean isIndeterminate();

   /**
    * Sets the maximum progress value.
    * @param maxValue value that indicates progress has completed
    */
   public void setMaxProgress(int maxValue);

   /**
    * Gets the maximum progress value.
    */
   public int getMaxProgress();

   /**
    * Increment progress monitor.
    */
   public void incrementProgress();

   /**
    * Resets progress monitor.
    * @param maxValue value that indicates progress has completed
    */
   public void resetProgress(int maxValue);

   /**
    * Resets progress monitor.
    */
   public void resetProgress();

   /**
    * Gets current value of progress monitor.
    */
   public int getProgress();

   /**
    * Sets the value of the progress monitor.
    */
   public void setProgress(int value);

   /**
    * Switch on message system.
    * Has no effect if message system is suspended.
    */
   public void displayMessages();

   /**
    * Switch off message system (except for warnings and errors).
    * Has no effect if message system is suspended.
    */
   public void hideMessages();

   /**
    * Suspends message system. Prevents 
    * {@link #displayMessages()} and {@link #hideMessages()}
    * from having an effect.
    */
   public void suspend();

   /**
    * Resumes message system. Prevents 
    * {@link #displayMessages()} and {@link #hideMessages()}
    * from having an effect.
    */
   public void resume();

   /**
    * Displays the given message text.
    * @param messageText the message text
    * @see #messageln(String)
    */
   public void message(String messageText);

   /**
    * Displays the message corresponding to the given exception.
    * @param excp the exception with the relevant message
    * @see #messageln(Exception)
    */
   public void message(Exception excp);

   /**
    * Displays the given message text followed by new line
    * character.
    * @param messageText the message text
    * @see #message(String)
    */
   public void messageln(String messageText);

   /**
    * Displays the message corresponding to the given exception
    * followed by new line character.
    * @param excp the exception with the relevant message
    * @see #message(Exception)
    */
   public void messageln(Exception excp);

   /**
    * Displays the given warning message.
    * @param messageText the message text
    */
   public void warning(String messageText);

   /**
    * Displays the warning corresponding to the given exception.
    * @param excp the exception with the relevant message
    */
   public void warning(Exception excp);

   /**
    * Displays the given error message.
    * @param messageText the message text
    */
   public void error(String messageText);

   /**
    * Displays the error corresponding to the given exception.
    * @param excp the exception with the relevant message
    */
   public void error(Exception excp);

   /**
    * Displays the given error message.
    * @param messageText the message text
    */
   public void internalerror(String messageText);

   /**
    * Displays the error corresponding to the given exception.
    * @param excp the exception with the relevant message
    */
   public void internalerror(Exception excp);

   /**
    * Displays the given error message.
    * @param messageText the message text
    */
   public void fatalerror(String messageText);

   /**
    * Displays the error corresponding to the given exception.
    * @param excp the exception with the relevant message
    */
   public void fatalerror(Exception excp);

}
