// File          : JDRDefaultMessage.java
// Purpose       : Default message system
// Date          : 12th June 2008
// Last Modified : 12th June 2008
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr.io;

/**
 * Default message system. This just prints messages to STDOUT and
 * warnings/errors to STDERR.
 * @author Nicola L C Talbot
 */

public class JDRDefaultMessage implements JDRMessage
{
   public JDRDefaultMessage()
   {
   }

   public void setIndeterminate(boolean indeterminate)
   {
      indeterminate_ = indeterminate;
   }

   public boolean isIndeterminate()
   {
      return indeterminate_;
   }

   public void incrementProgress()
   {
      setProgress(progressValue+1);
   }

   public int getProgress()
   {
      return progressValue;
   }

   public void setProgress(int value)
   {
      progressValue = value;

      if (showMessages)
      {
         if (isIndeterminate())
         {
            System.out.print(".");
         }
         else
         {
            int n = (int)Math.round(
               ((double)value/(double)progressMax)*100);

            if (n < 10)
            {
               System.out.print("  "+n+"%");
            }
            else if (n < 99)
            { 
              System.out.print(" "+n+"%");
            }
            else
            {
               System.out.print(""+n+"%");
            }

            System.out.print("\b\b\b\b");
         }
      }
   }

   public void setMaxProgress(int maxValue)
   {
      progressMax = maxValue;
   }

   public int getMaxProgress()
   {
      return progressMax;
   }

   public void resetProgress(int maxValue)
   {
      progressMax = maxValue;
      progressValue = 0;
   }

   public void resetProgress()
   {
      resetProgress(progressMax);
   }

   public void hideMessages()
   {
      if (!suspended)
      {
         showMessages = false;
      }
   }

   public void displayMessages()
   {
      if (!suspended)
      {
         showMessages = true;
      }
   }

   public void suspend()
   {
      suspended = true;
   }

   public void resume()
   {
      suspended = false;
   }

   public void message(String messageText)
   {
      if (showMessages && !suspended)
      {
         System.out.print(messageText);
      }
   }

   public void message(Exception excp)
   {
      if (showMessages && !suspended)
      {
         System.out.print(excp.getMessage());
      }
   }

   public void messageln(String messageText)
   {
      if (showMessages && !suspended)
      {
         System.out.println(messageText);
      }
   }

   public void messageln(Exception excp)
   {
      if (showMessages && !suspended)
      {
         System.out.println(excp.getMessage());
      }
   }

   public void warning(String messageText)
   {
      System.err.println("Warning: "+messageText);
   }

   public void warning(Exception excp)
   {
      System.err.println("Warning: "+excp.getMessage());
   }

   public void error(String messageText)
   {
      System.err.println("Error: "+messageText);
   }

   public void error(Exception excp)
   {
      System.err.println("Error: "+excp.getMessage());
   }

   public void internalerror(String messageText)
   {
      System.err.println("Internal error: "+messageText);
   }

   public void internalerror(Exception excp)
   {
      System.err.println("Internal error: "+excp.getMessage());
      excp.printStackTrace();
   }

   public void fatalerror(String messageText)
   {
      System.err.println("Fatal error: "+messageText);
   }

   public void fatalerror(Exception excp)
   {
      System.err.println("Fatal error: "+excp.getMessage());
   }

   private int progressMax=100;
   private int progressValue=0;
   private boolean indeterminate_=true;

   private boolean showMessages=false;
   private boolean suspended=false;
}
