// File          : JDR.java
// Purpose       : functions to save and load JDR files
// Date          : 1st February 2006
// Last Modified : 29th August 2010
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.io;

import java.io.*;
import uk.ac.uea.cmp.nlct.jdr.*;

/**
 * Functions to save and load JDR files.
 * All the methods in this class are static.
 *
 * @version 0.3.0b 28 Jul 2007
 * @author Nicola L C Talbot
 */

public class JDR
{
   /**
    * Constructor not needed as all methods static.
    */
   protected JDR()
   {
   }

   /**
    * Saves all objects in latest JDR format without settings.
    * <p>
    * None of the current JDR formats allow any linear or radial
    * gradient paint to have a start or end colour that isn't
    * either {@link JDRColor} or {@link JDRColorCMYK}.
    * @param allObjects all the objects constituting the image
    * @param out the output stream
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if any gradient/radial
    * start/end colour isn't either <code>JDRColor</code> or
    * <code>JDRColorCMYK</code>
    */
   public static void save(JDRGroup allObjects, DataOutputStream out)
      throws IOException,
       InvalidFormatException
   {
      try
      {
        save(allObjects, out, currentVersion);
      }
      catch (UnknownJDRVersionException e)
      {
         /* this shouldn't happen unless there is a mistake in
            the code! */
         messageSystem.internalerror(e);
      }
   }

   /**
    * Saves all objects in latest JDR format with settings.
    * <p>
    * None of the current JDR formats allow any linear or radial
    * gradient paint to have a start or end colour that isn't
    * either {@link JDRColor} or {@link JDRColorCMYK}.
    * @param allObjects all the objects constituting the image
    * @param settings the canvas settings
    * @param out the output stream
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if any gradient/radial
    * start/end colour isn't either <code>JDRColor</code> or
    * <code>JDRColorCMYK</code>
    */
   public static void save(JDRGroup allObjects,
      CanvasSettings settings, DataOutputStream out)
   throws IOException,
     InvalidFormatException
   {
      try
      {
         save(allObjects, settings, out, currentVersion);
      }
      catch (UnknownJDRVersionException e)
      {
         /* this shouldn't happen unless there is a mistake in
            the code! */
         messageSystem.internalerror(e);
      }
   }

   /**
    * Saves all objects in latest JDR format.
    * The way in which the settings are
    * saved depends on the settings flag which may be one of:
    * {@link #NO_SETTINGS} (don't
    * save the settings), {@link #ALL_SETTINGS} (save all
    * settings) or {@link #PAPER_ONLY} (only save the
    * paper size.)
    * <p>
    * None of the current JDR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or
    * {@link JDRColorCMYK}.
    * @param allObjects all objects constituting the image
    * @param settings canvas settings
    * @param out the output stream
    * @param settingsFlag indicate whether to save settings
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if any gradient/radial
    * start/end colour isn't either <code>JDRColor</code> or
    * <code>JDRColorCMYK</code>
    * @throws InvalidSettingsIDException if the settings flag isn't
    * one of: <code>JDR.NO_SETTINGS</code>, <code>JDR.ALL_SETTINGS</code>
    * or <code>JDR.PAPER_ONLY</code
    */
   public static void save(JDRGroup allObjects,
      CanvasSettings settings, DataOutputStream out, int settingsFlag)
      throws IOException,
         InvalidFormatException,
         InvalidSettingsIDException
   {
      try
      {
         save(allObjects, settings, out, currentVersion, settingsFlag);
      }
      catch (UnknownJDRVersionException e)
      {
         /* this shouldn't happen unless there is a mistake in
            the code! */
         messageSystem.internalerror(e);
      }
   }

   /**
    * Saves all objects in given JDR format.
    *
    * The way in which the settings are
    * saved depends on the settings flag which may be one of:
    * {@link #NO_SETTINGS} (don't
    * save the settings), {@link #ALL_SETTINGS} (save all
    * settings) or {@link #PAPER_ONLY} (only save the
    * paper size.)
    * <p>
    * None of the current JDR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or
    * {@link JDRColorCMYK}.
    * @param allObjects all objects constituting the image
    * @param settings canvas settings
    * @param out the output stream
    * @param settingsFlag indicate whether to save settings
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if any gradient/radial
    * start/end colour isn't either <code>JDRColor</code> or
    * <code>JDRColorCMYK</code>
    * @throws InvalidSettingsIDException if the settings flag isn't
    * one of: <code>JDR.NO_SETTINGS</code>, <code>JDR.ALL_SETTINGS</code>
    * or <code>JDR.PAPER_ONLY</code>
    * @throws UnknownJDRVersionException if specified version is not
    * a valid version number
    */
   public static void save(JDRGroup allObjects, CanvasSettings settings, DataOutputStream out, float version, int settingsFlag)
      throws IOException,
      InvalidFormatException,
      InvalidSettingsIDException,
      UnknownJDRVersionException
   {
      switch (settingsFlag)
      {
         case NO_SETTINGS:
            save(allObjects, out, version);
         break;
         case ALL_SETTINGS :
            save(allObjects, settings, out, version);
         break;
         case PAPER_ONLY :
            save(allObjects, settings.paper, out, version);
         break;
         default :
            throw new InvalidSettingsIDException(settingsFlag);
      }
   }

   /**
    * Saves all objects in given JDR format without settings.
    * <p>
    * None of the current JDR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or
    * {@link JDRColorCMYK}.
    * @param allObjects all objects constituting the image
    * @param out the output stream
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if any gradient/radial
    * start/end colour isn't either <code>JDRColor</code> or
    * <code>JDRColorCMYK</code>
    * @throws UnknownJDRVersionException if specified version is not
    * a valid version number
    */
   public static void save(JDRGroup allObjects, DataOutputStream out,
      float version)
      throws IOException,
        InvalidFormatException,
        UnknownJDRVersionException
   {
      // add listeners if not already done so
      addAllListeners();

      out.writeChars("JDR");
      String vers = "";
      if (version == 1.0f)
      {
         vers="1.0";
      }
      else if (version == 1.1f)
      {
         vers="1.1";
      }
      else if (version == 1.2f)
      {
         vers="1.2";
      }
      else if (version == 1.3f)
      {
         vers="1.3";
      }
      else if (version == 1.4f)
      {
         vers="1.4";
      }
      else if (version == 1.5f)
      {
         vers="1.5";
      }
      else if (version == 1.6f)
      {
         vers="1.6";
      }
      else
      {
         throw new UnknownJDRVersionException(""+version);
      }
      out.writeInt(vers.length());
      out.writeChars(vers);
      // don't save settings
      if (version < 1.3f)
      {
         out.writeBoolean(false);
      }
      else
      {
         out.writeByte((byte)NO_SETTINGS);
      }
      //out.writeChar('G');
      //allObjects.save(out, version);
      objectLoader.saveJDR(allObjects, out, version);
   }

   /**
    * Saves all objects in given JDR format with all settings.
    * <p>
    * None of the current JDR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or
    * {@link JDRColorCMYK}.
    * @param allObjects all objects constituting the image
    * @param settings canvas settings
    * @param out the output stream
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if any gradient/radial
    * start/end colour isn't either <code>JDRColor</code> or
    * <code>JDRColorCMYK</code>
    * @throws UnknownJDRVersionException if specified version is not
    * a valid version number
    */
   public static void save(JDRGroup allObjects,
      CanvasSettings settings, DataOutputStream out, float version)
      throws IOException,
      InvalidFormatException,
      UnknownJDRVersionException
   {
      // add listeners if not already done so
      addAllListeners();

      out.writeChars("JDR");
      String vers = "";
      if (version == 1.0f)
      {
         vers="1.0";
      }
      else if (version == 1.1f)
      {
         vers="1.1";
      }
      else if (version == 1.2f)
      {
         vers="1.2";
      }
      else if (version == 1.3f)
      {
         vers="1.3";
      }
      else if (version == 1.4f)
      {
         vers="1.4";
      }
      else if (version == 1.5f)
      {
         vers="1.5";
      }
      else if (version == 1.6f)
      {
         vers="1.6";
      }
      else
      {
         throw new UnknownJDRVersionException(""+version);
      }
      out.writeInt(vers.length());
      out.writeChars(vers);
      if (version < 1.3f)
      {
         out.writeBoolean(true);
      }
      else
      {
         out.writeByte((byte)ALL_SETTINGS);
      }
      settings.save(out, version);
      //out.writeChar('G');
      //allObjects.save(out, version);
      objectLoader.saveJDR(allObjects, out, version);
   }

   /**
    * Saves all objects in given JDR format with given paper size.
    * The only canvas setting to be saved is the paper size.
    * Note: only JDR versions 1.3 onwards supports saving only the paper 
    * size, so if the given version number is less than 1.3, no
    * settings will be saved.
    * <p>
    * None of the current JDR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or
    * {@link JDRColorCMYK}.
    * @param allObjects all objects constituting the image
    * @param paper the required paper
    * @param out the output stream
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if any gradient/radial
    * start/end colour isn't either <code>JDRColor</code> or
    * <code>JDRColorCMYK</code>
    * @throws UnknownJDRVersionException if specified version is not
    * a valid version number
    */
   public static void save(JDRGroup allObjects,
      JDRPaper paper, DataOutputStream out, float version)
      throws IOException,
      InvalidFormatException,
      UnknownJDRVersionException
   {
      // add listeners if not already done so
      addAllListeners();

      out.writeChars("JDR");
      String vers = "";
      if (version == 1.0f)
      {
         vers="1.0";
      }
      else if (version == 1.1f)
      {
         vers="1.1";
      }
      else if (version == 1.2f)
      {
         vers="1.2";
      }
      else if (version == 1.3f)
      {
         vers="1.3";
      }
      else if (version == 1.4f)
      {
         vers="1.4";
      }
      else if (version == 1.5f)
      {
         vers="1.5";
      }
      else if (version == 1.6f)
      {
         vers="1.6";
      }
      else
      {
         throw new UnknownJDRVersionException(""+version);
      }
      out.writeInt(vers.length());
      out.writeChars(vers);
      if (version < 1.3f)
      {
         out.writeBoolean(false);
      }
      else
      {
         out.writeByte((byte)PAPER_ONLY);
         paper.save(out, version);
      }
      //out.writeChar('G');
      //allObjects.save(out, version);
      objectLoader.saveJDR(allObjects, out, version);
   }

   /**
    * Reads image from JDR formatted file. The
    * image is returned as a {@link JDRGroup}.
    * <p>
    * Any settings found in the file are put in <code>settings</code>
    * which should be initialised prior to calling this method.
    * The settings flag can afterwards be retrieved using 
    * {@link #getLastLoadedSettingsID()} .
    * @param in the input stream
    * @param settings the canvas settings
    * @return the image as a <code>JDRGroup</code>.
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if file is incorrectly 
    * formatted
    */
   public static JDRGroup load(DataInputStream in, 
      CanvasSettings settings)
      throws IOException,InvalidFormatException
   {
      // add listeners if not already done so
      addAllListeners();

      bitmapReplaced = false;

      char[] str = new char[3];
      for (int i = 0; i < 3; i++) str[i] = in.readChar();
      if (!(new String(str)).equals("JDR"))
         throw new NotJDRException();

      int n = in.readInt();
      char[] version = new char[n];
      for (int i = 0; i < n; i++) version[i] = in.readChar();

      String thisFileVersion = new String(version);

      if (thisFileVersion.equals("1.0"))
      {
         _lastLoadedVersion = 1.0f;
      }
      else if (thisFileVersion.equals("1.1"))
      {
         _lastLoadedVersion = 1.1f;
      }
      else if (thisFileVersion.equals("1.2"))
      {
         _lastLoadedVersion = 1.2f;
      }
      else if (thisFileVersion.equals("1.3"))
      {
         _lastLoadedVersion = 1.3f;
      }
      else if (thisFileVersion.equals("1.4"))
      {
         _lastLoadedVersion = 1.4f;
      }
      else if (thisFileVersion.equals("1.5"))
      {
         _lastLoadedVersion = 1.5f;
      }
      else if (thisFileVersion.equals("1.6"))
      {
         _lastLoadedVersion = 1.6f;
      }
      else
      {
         throw new UnknownJDRVersionException(thisFileVersion);
      }

      int settingsFlag;

      if (_lastLoadedVersion < 1.3f)
      {
         settingsFlag = (in.readBoolean()?ALL_SETTINGS:NO_SETTINGS);
      }
      else
      {
         settingsFlag = (int)in.readByte();
      }

      if (settingsFlag == ALL_SETTINGS)
      {
         CanvasSettings.read(settings, in, _lastLoadedVersion);
      }
      else if (settingsFlag == PAPER_ONLY)
      {
         settings.paper = JDRPaper.read(in, _lastLoadedVersion);
      }
      else if (settingsFlag != NO_SETTINGS)
      {
         throw new InvalidSettingsIDException(settingsFlag);
      }

      lastLoadedSettings_ = settingsFlag;

/*
      char c = in.readChar();

      if (c != 'G')
      {
         throw new JDRMissingTopLevelException();
      }
*/

      //return JDRGroup.read(in, _lastLoadedVersion);
      JDRObject allObjects = objectLoader.loadJDR(in, _lastLoadedVersion);

      if (!(allObjects instanceof JDRGroup))
      {
        throw new JDRMissingTopLevelException();
      }

      return (JDRGroup)allObjects;
   }

   /**
    * Adds all the supported listeners if not already done.
    */
    private static void addAllListeners()
    {
       if (paintLoader.getListeners().isEmpty())
       {
          paintLoader.addListener(new JDRColorListener());
          paintLoader.addListener(new JDRTransparentListener());
          paintLoader.addListener(new JDRColorCMYKListener());
          paintLoader.addListener(new JDRGradientListener());
          paintLoader.addListener(new JDRRadialListener());
          paintLoader.addListener(new JDRGrayListener());
          paintLoader.addListener(new JDRColorHSBListener());
       }

       if (objectLoader.getListeners().isEmpty())
       {
          objectLoader.addListener(new JDRGroupListener());
          objectLoader.addListener(new JDRPathListener());
          objectLoader.addListener(new JDRTextListener());
          objectLoader.addListener(new JDRBitmapListener());
          objectLoader.addListener(new JDRTextPathListener());
          objectLoader.addListener(new JDRSymmetricPathListener());
          objectLoader.addListener(new JDRRotationalPatternListener());
          objectLoader.addListener(new JDRScaledPatternListener());
          objectLoader.addListener(new JDRSpiralPatternListener());
       }

       if (segmentLoader.getListeners().isEmpty())
       {
          segmentLoader.addListener(new JDRSegmentLoaderListener());
          segmentLoader.addListener(new JDRLineLoaderListener());
          segmentLoader.addListener(new JDRBezierLoaderListener());
          segmentLoader.addListener(new JDRPartialSegmentLoaderListener());
          segmentLoader.addListener(new JDRPartialLineLoaderListener());
          segmentLoader.addListener(new JDRPartialBezierLoaderListener());
       }

       if (pathStyleLoader.getListeners().isEmpty())
       {
          pathStyleLoader.addListener(new JDRBasicPathStyleListener());
          pathStyleLoader.addListener(new JDRTextPathStyleListener());
       }

       if (gridLoader.getListeners().isEmpty())
       {
          gridLoader.addListener(new JDRRectangularGridListener());
          gridLoader.addListener(new JDRRadialGridListener());
       }

    }

   /**
    * Gets the version number of last JDR file to be loaded.
    *
    * The version number is set by 
    * {@link #load(DataInputStream, CanvasSettings)}.
    * @return the version number of 0 if no JDR file has been loaded
    */
   public static float getLastLoadedVersion()
   {
      return _lastLoadedVersion;
   }

   /**
    * Gets the settings flag of the last JDR file to be loaded.
    *
    * The settings flag is set by 
    * {@link #load(DataInputStream, CanvasSettings)}.
    * @return the settings flag (one of {@link #NO_SETTINGS},
    * {@link #ALL_SETTINGS}, {@link #PAPER_ONLY}) or -1 if no 
    * JDR file has been loaded
    */
   public static int getLastLoadedSettingsID()
   {
      return lastLoadedSettings_;
   }

   public static JDRPaintLoader getPaintLoader()
   {
      addAllListeners();
      return paintLoader;
   }

   public static JDRObjectLoader getObjectLoader()
   {
      addAllListeners();
      return objectLoader;
   }

   public static JDRSegmentLoader getSegmentLoader()
   {
      addAllListeners();
      return segmentLoader;
   }

   public static JDRPathStyleLoader getPathStyleLoader()
   {
      addAllListeners();
      return pathStyleLoader;
   }

   public static JDRGridLoader getGridLoader()
   {
      addAllListeners();
      return gridLoader;
   }

   /**
    * Returns either {@link #OPTIMIZE_SPEED} (speed is more important
    * than memory restrictions) or {@link #OPTIMIZE_MEMORY}
    * (reducing memory requirements is more important that speed).
    * @return either {@link #OPTIMIZE_SPEED} or 
    * {@link #OPTIMIZE_MEMORY}
    */
   public static int getOptimize()
   {
      return optimize;
   }

   /**
    * Sets whether speed is more important than memory restrictions
    * or whether reducing memory requirements is more important
    * than speed.
    * @param setting may be either {@link #OPTIMIZE_SPEED} or
    * {@link #OPTIMIZE_MEMORY}
    * @throws IllegalArgumentException if argument is invalid
    */
   public static void setOptimize(int setting)
   throws IllegalArgumentException
   {
      if (!(setting == OPTIMIZE_SPEED || setting == OPTIMIZE_MEMORY))
      {
         throw new IllegalArgumentException("Setting "+setting
            +" is invalid");
      }

      optimize = setting;
   }

   /**
    * Newest JDR version number.
    */
   public static final float currentVersion = 1.6f;

   /**
    * No canvas settings saved in JDR file.
    */
   public static final int NO_SETTINGS=0;
   /**
    * All canvas settings saved in JDR file.
    *
    * @see CanvasSettings
    */
   public static final int ALL_SETTINGS=1;
   /**
    * Only required paper saved in JDR file.
    */
   public static final int PAPER_ONLY=2;

   /**
    * Stores the version number of the last JDR file to be loaded.
    *
    * @see #getLastLoadedVersion()
    */
   private static float _lastLoadedVersion=0.0f;

   /**
    * Stores the value of the settings flag for the last JDR file
    * to be loaded.
    *
    * @see #getLastLoadedSettingsID()
    */
   private static int lastLoadedSettings_=-1;

   /**
    * Indicates if a bitmap has been replaced or discard. Should
    * be reset before loading an image.
    */
   public static boolean bitmapReplaced=false;

   private static JDRPaintLoader paintLoader = new JDRPaintLoader();

   private static JDRObjectLoader objectLoader = new JDRObjectLoader();

   private static JDRSegmentLoader segmentLoader = new JDRSegmentLoader();

   private static JDRPathStyleLoader pathStyleLoader = new JDRPathStyleLoader();

   private static JDRGridLoader gridLoader = new JDRGridLoader();

   /**
    * Speed is more important than memory requirements.
    */
   public static final int OPTIMIZE_SPEED=0;
   /**
    * Conserving memory is more important that speed.
    */
   public static final int OPTIMIZE_MEMORY=1;

   /**
    * Determines whether speed or conserving memory are more important.
    */
   private static int optimize=OPTIMIZE_SPEED;

   /**
    * Message system to use for all the jdr classes.
    * This defaults to {@link JDRDefaultMessage} but can be 
    * set to something else, for example, if messages need to
    * be displayed in a dialog box or written to a log file.
    */
   public static JDRMessage messageSystem = new JDRDefaultMessage();
}
