// File          : FLF.java
// Purpose       : functions to save JDRGroup as a LaTeX2e package 
//                 based on the flowfram package
// Date          : 27th February 2006
// Last Modified : 11th August 2009
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr.io;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;
import java.text.*;
import uk.ac.uea.cmp.nlct.jdr.*;

/**
 * Functions to save JDRGroup as a LaTeX2e package 
 * based on the flowfram package. Only objects that have 
 * {@link JDRCompleteObject#flowframe} set will be saved. Note that
 * this library can't load LaTeX packages, but can only create them.
 * The typeblock for the document should be specified by the
 * flowframe information for the entire image.
 * @author Nicola L C Talbot
 */
public class FLF
{
   /**
    * Exports flowframe information to the named file.
    * The typeblock for the document should be specified by the
    * flowframe information for the image
    * (<code>group.flowframe</code>).
    * @param group the image containing flow frame information
    * @param filename name of file to save the information
    * @param paper required paper size
    * @param latexBase LaTeX font base identifying the normal font
    * size and baselineskip
    * @param creator name of application calling this method
    * @param comment comment to appear in file
    * @throws IOException if I/O error occurs
    * @throws MissingTypeBlockException if the typeblock is missing
    * (i.e. <code>group.flowframe==null</code>)
    * @throws InvalidShapeException if any of the frames use
    * <code>\parshape</code> or <code>\shapepar</code> but the
    * parameters can be computed from the frame's shape
    * @throws MissplacedTypeBlockException if any of the objects
    * within the image contain typeblock flow frame data (only
    * <code>group.flowframe</code> should have the type
    * {@link FlowFrame#TYPEBLOCK})
    */
   public static void save(JDRGroup group, String filename,
      JDRPaper paper, LaTeXFontBase latexBase,
      String creator, String comment)
   throws IOException,
          MissingTypeBlockException,
          InvalidShapeException,
          MissplacedTypeBlockException
   {
      if (group.flowframe == null)
      {
         throw new MissingTypeBlockException("typeblock missing");
      }

      PrintWriter out = new PrintWriter(new FileWriter(filename));

      BBox box = group.getBBox();
      out.println("% "+creator);
      out.println("% "+PGF.dateFormat.format(new Date()));
      out.println("% "+comment);
      out.println("% This package is designed for use with normal font = "+latexBase.getNormalSize()+"pt");
      out.println("\\NeedsTeXFormat{LaTeX2e}");
      String name = (new File(filename)).getName();
      int idx = name.lastIndexOf(".sty");
      out.println("\\ProvidesPackage{"
                  +(idx==-1?name:name.substring(0,idx))+"}");
      out.println("\\DeclareOption{draft}{\\PassOptionsToPackage{draft}{flowfram}}");
      out.println("\\DeclareOption{final}{\\PassOptionsToPackage{final}{flowfram}}");
      out.println("\\DeclareOption{rotate}{\\PassOptionsToPackage{rotate}{flowfram}}");
      out.println("\\DeclareOption{norotate}{\\PassOptionsToPackage{norotate}{flowfram}}");
      out.println("\\DeclareOption{ttbtitle}{\\PassOptionsToPackage{ttbtitle}{flowfram}}");
      out.println("\\DeclareOption{ttbnotitle}{\\PassOptionsToPackage{ttbnotitle}{flowfram}}");
      out.println("\\DeclareOption{ttbnum}{\\PassOptionsToPackage{ttbnum}{flowfram}}");
      out.println("\\DeclareOption{ttbnonum}{\\PassOptionsToPackage{ttbnonum}{flowfram}}");
      out.println("\\DeclareOption{color}{\\PassOptionsToPackage{color}{flowfram}}");
      out.println("\\DeclareOption{nocolor}{\\PassOptionsToPackage{nocolor}{flowfram}}");
      out.println("\\ProcessOptions");

      out.println("\\RequirePackage{flowfram}");
      out.println("\\RequirePackage{pgf}");
      out.println("\\RequirePackage["+paper.tex()+"]{geometry}");

      double left = group.flowframe.left;
      double right = group.flowframe.right;
      double top = group.flowframe.top;
      double bottom = group.flowframe.bottom;
      double width = paper.getWidth()-(left+right);
      double height = paper.getHeight()-(top+bottom);

      Rectangle2D typeblock = new Rectangle2D.Double(left,right,width,height);

      double baselineskip = latexBase.getBaselineskip(LaTeXFontBase.NORMALSIZE);
      group.flowframe.tex(group, typeblock, out, baselineskip);

      for (int i = 0; i < group.size(); i++)
      {
         JDRCompleteObject object = group.get(i);

         if (object.flowframe != null && 
             object.flowframe.getType() == FlowFrame.TYPEBLOCK)
         {
            throw new MissplacedTypeBlockException();
         }

         object.saveFlowframe(typeblock, out, baselineskip);
      }

      if (group.isFlowframeDefined(FlowFrame.DYNAMIC, "header"))
      {
         out.println("\\renewcommand{\\@dothehead}{}%");
         out.println("\\renewcommand{\\@dodynamicthehead}{%");
         out.println("\\@dynamicframeid{header}%");
         out.println("\\expandafter");
         out.println("\\def\\csname @dynamicframe@\\romannumeral\\ff@id\\endcsname{%");
         out.println("\\vfill\\@thehead\\vfill}%");
         out.println("}%");
      }

      if (group.isFlowframeDefined(FlowFrame.DYNAMIC, "footer"))
      {
         out.println("\\renewcommand{\\@dothefoot}{}%");
         out.println("\\renewcommand{\\@dodynamicthefoot}{%");
         out.println("\\@dynamicframeid{footer}%");
         out.println("\\expandafter");
         out.println("\\def\\csname @dynamicframe@\\romannumeral\\ff@id\\endcsname{%");
         out.println("\\vfill\\@thefoot\\vfill}%");
         out.println("}%");
      }

      out.println("\\endinput");

      out.close();
   }

}
