// File          : AJR.java
// Purpose       : functions to save and load AJR files
// Date          : 1st February 2006
// Last Modified : 29th August 2010
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr.io;

import java.io.*;
import uk.ac.uea.cmp.nlct.jdr.*;

/**
 * Functions to save and load AJR files.
 * All the methods in this class are static.
 * @author Nicola L C Talbot
*/

public class AJR
{
   /**
    * Constructor not needed as all methods static.
    */
   protected AJR()
   {
   }

   /**
    * Saves all objects in latest AJR format without settings.
    * The settings flag in the AJR file is set to 
    * {@link JDR#NO_SETTINGS}. 
    * <p>
    * None of the current AJR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or 
    * {@link JDRColorCMYK}.
    * @param allObjects all objects constituting the image
    * @param out the print writer to the required file
    * @throws IOException if an I/O error occurs
    * @throws InvalidStartColourException if any gradient/radial 
    * start colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidEndColourException if any gradient/radial 
    * end colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
   */
   public static void save(JDRGroup allObjects, PrintWriter out)
      throws IOException,
        InvalidFormatException
   {
      try
      {
         save(allObjects, out, currentVersion);
      }
      catch (UnknownAJRVersionException e)
      {
         /* this shouldn't happen unless there is a mistake in 
            the code! */
         JDR.messageSystem.internalerror(e);
      }
   }

   /**
    * Saves all objects in latest AJR format with settings.
    *
    * All the given settings are also saved.
    * <p>
    * None of the current AJR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or 
    * {@link JDRColorCMYK}.
    * @param allObjects all objects constituting the image
    * @param settings canvas settings
    * @param out the print writer to the required file
    * @throws IOException if an I/O error occurs
    * @throws InvalidStartColourException if any gradient/radial 
    * start colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidEndColourException if any gradient/radial 
    * end colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
   */
   public static void save(JDRGroup allObjects,
      CanvasSettings settings, PrintWriter out)
      throws IOException,
       InvalidFormatException
   {
      try
      {
         save(allObjects, settings, out, currentVersion);
      }
      catch (UnknownAJRVersionException e)
      {
         /* this shouldn't happen unless there is a mistake in 
            the code! */
         JDR.messageSystem.internalerror(e);
      }
   }

   /**
    * Saves all objects in latest AJR format.
    *
    * The way in which the settings are
    * saved depends on the settings flag which may be one of:
    * {@link JDR#NO_SETTINGS} (don't
    * save the settings), {@link JDR#ALL_SETTINGS} (save all
    * settings) or {@link JDR#PAPER_ONLY} (only save the
    * paper size.)
    * <p>
    * None of the current AJR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or 
    * {@link JDRColorCMYK}.
    * @param allObjects all objects constituting the image
    * @param settings canvas settings
    * @param out the print writer to the required file
    * @param settingsFlag indicate whether to save settings
    * @throws IOException if an I/O error occurs
    * @throws InvalidStartColourException if any gradient/radial 
    * start colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidEndColourException if any gradient/radial end 
    * colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidSettingsIDException if the settings flag isn't 
    * one of: <code>JDR.NO_SETTINGS</code>, <code>JDR.ALL_SETTINGS</code> 
    * or <code>JDR.PAPER_ONLY</code>
   */
   public static void save(JDRGroup allObjects,
      CanvasSettings settings, PrintWriter out, int settingsFlag)
      throws IOException,
        InvalidFormatException,
        InvalidSettingsIDException
   {
      try
      {
        save(allObjects, settings, out, currentVersion, settingsFlag);
      }
      catch (UnknownAJRVersionException e)
      {
         /* this shouldn't happen unless there is a mistake in 
            the code! */
         JDR.messageSystem.internalerror(e);
      }
   }

   /**
    * Saves all objects in given AJR format.
    *
    * The way in which the settings are
    * saved depends on the settings flag which may be one of:
    * {@link JDR#NO_SETTINGS} (don't
    * save the settings), {@link JDR#ALL_SETTINGS} (save all
    * settings) or {@link JDR#PAPER_ONLY} (only save the
    * paper size.)
    * <p>
    * None of the current AJR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or 
    * {@link JDRColorCMYK}.
    * <p>
    * Allowed version numbers are 1.0, 1.1, 1.2, 1.3, 1.4 and 1.5.
    * @param allObjects all objects constituting the image
    * @param settings canvas settings
    * @param out the print writer to the required file
    * @param version AJR version
    * @param settingsFlag indicate whether to save settings
    * @throws IOException if an I/O error occurs
    * @throws InvalidStartColourException if any gradient/radial 
    * start colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidEndColourException if any gradient/radial end 
    * colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidSettingsIDException if the settings flag isn't 
    * one of: <code>JDR.NO_SETTINGS</code>, <code>JDR.ALL_SETTINGS</code> 
    * or <code>JDR.PAPER_ONLY</code>
    * @throws UnknownAJRVersionException if the specified version 
    * isn't a recognised version number
   */
   public static void save(JDRGroup allObjects,
      CanvasSettings settings, PrintWriter out, float version,
      int settingsFlag)
      throws IOException,
        InvalidFormatException,
        InvalidSettingsIDException,
        UnknownAJRVersionException
   {
      switch (settingsFlag)
      {
         case JDR.NO_SETTINGS:
            save(allObjects, out, version);
         break;
         case JDR.ALL_SETTINGS :
            save(allObjects, settings, out, version);
         break;
         case JDR.PAPER_ONLY :
            save(allObjects, settings.paper, out, version);
         break;
         default :
            throw new InvalidSettingsIDException(settingsFlag);
      }
   }

   /**
    * Saves all objects in given AJR format.
    *
    * The settings flag in the file is set to {@link JDR#NO_SETTINGS}.
    * <p>
    * None of the current AJR formats allow
    * any linear or radial gradient paint to have a start or
    * end colour that isn't either {@link JDRColor} or 
    * {@link JDRColorCMYK}.
    * <p>
    * Allowed version numbers are 1.0, 1.1, 1.2, 1.3, 1.4 and 1.5.
    * @param allObjects all objects constituting the image
    * @param out the print writer to the required file
    * @param version AJR version
    * @throws IOException if an I/O error occurs
    * @throws InvalidStartColourException if any gradient/radial 
    * start colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidEndColourException if any gradient/radial end 
    * colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws UnknownAJRVersionException if the specified version 
    * isn't a recognised version number
   */
   public static void save(JDRGroup allObjects, PrintWriter out,
      float version)
      throws IOException,
        InvalidFormatException,
        UnknownAJRVersionException
   {
      // add listeners if not already done so
      addAllListeners();

      String vers = "";
      if (version == 1.0f)
      {
         vers="1.0";
      }
      else if (version == 1.1f)
      {
         vers="1.1";
      }
      else if (version == 1.2f)
      {
         vers="1.2";
      }
      else if (version == 1.3f)
      {
         vers="1.3";
      }
      else if (version == 1.4f)
      {
         vers="1.4";
      }
      else if (version == 1.5f)
      {
         vers="1.5";
      }
      else if (version == 1.6f)
      {
         vers="1.6";
      }
      else
      {
         throw new UnknownAJRVersionException(""+version);
      }

      out.println("AJR "+vers);
      // don't save settings
      //out.print(""+JDR.NO_SETTINGS+" G ");
      out.println(""+JDR.NO_SETTINGS);
      //allObjects.saveAJR(out, version);
      objectLoader.saveAJR(allObjects, out, version);
   }

   /**
    * Saves all objects in given AJR format.
    *
    * The settings flag in the file is set to 
    * {@link JDR#ALL_SETTINGS}.
    * <p>
    * None of the current AJR formats allow any linear or radial 
    * gradient paint to have a start or end colour that isn't either 
    * {@link JDRColor} or {@link JDRColorCMYK}.
    * <p>
    * Allowed version numbers are 1.0, 1.1, 1.2, 1.3, 1.4 and 1.5.
    * @param allObjects all objects constituting the image
    * @param settings Canvas settings
    * @param out the print writer to the required file
    * @param version AJR version
    * @throws IOException if an I/O error occurs
    * @throws InvalidStartColourException if any gradient/radial 
    * start colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidEndColourException if any gradient/radial end 
    * colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws UnknownAJRVersionException if specified version isn't a 
    * recognised version number
   */
   public static void save(JDRGroup allObjects,
      CanvasSettings settings, PrintWriter out, float version)
      throws IOException,
         InvalidFormatException,
         UnknownAJRVersionException
   {
      // add listeners if not already done so
      addAllListeners();

      String vers = "";
      if (version == 1.0f)
      {
         vers="1.0";
      }
      else if (version == 1.1f)
      {
         vers="1.1";
      }
      else if (version == 1.2f)
      {
         vers="1.2";
      }
      else if (version == 1.3f)
      {
         vers="1.3";
      }
      else if (version == 1.4f)
      {
         vers="1.4";
      }
      else if (version == 1.5f)
      {
         vers="1.5";
      }
      else if (version == 1.6f)
      {
         vers="1.6";
      }
      else
      {
         throw new UnknownAJRVersionException(""+version);
      }
      out.println("AJR "+vers);
      out.print(""+JDR.ALL_SETTINGS+" ");
      settings.saveAJR(out, version);
      out.println();
      //out.print("G ");
      //allObjects.saveAJR(out, version);
      objectLoader.saveAJR(allObjects, out, version);
   }

   /**
    * Saves all objects in given AJR format.
    *
    * The required paper is also saved to the AJR file.
    * <p>
    * None of the current AJR formats allow any linear or radial 
    * gradient paint to have a start or end colour that isn't 
    * either {@link JDRColor} or {@link JDRColorCMYK}.
    * <p>
    * Allowed version numbers are 1.0, 1.1, 1.2, 1.3, 1.4 and 1.5.
    * <p>
    * Note that the {@link JDR#PAPER_ONLY} setting is only
    * available for versions from 1.3, so if a lower version
    * number is specified, the settings flag will be set to
    * {@link JDR#NO_SETTINGS}.
    * @param allObjects all objects constituting the image
    * @param paper paper settings
    * @param out the print writer to the required file
    * @param version AJR version
    * @throws IOException if an I/O error occurs
    * @throws InvalidStartColourException if any gradient/radial 
    * start colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws InvalidEndColourException if any gradient/radial start 
    * colour isn't either <code>JDRColor</code> or 
    * <code>JDRColorCMYK</code>
    * @throws UnknownAJRVersionException if version isn't a 
    * recognised version number
   */
   public static void save(JDRGroup allObjects,
      JDRPaper paper, PrintWriter out, float version)
      throws IOException,
        InvalidFormatException,
        UnknownAJRVersionException
   {
      // add listeners if not already done so
      addAllListeners();

      String vers = "";
      if (version == 1.0f)
      {
         vers="1.0";
      }
      else if (version == 1.1f)
      {
         vers="1.1";
      }
      else if (version == 1.2f)
      {
         vers="1.2";
      }
      else if (version == 1.3f)
      {
         vers="1.3";
      }
      else if (version == 1.4f)
      {
         vers="1.4";
      }
      else if (version == 1.5f)
      {
         vers="1.5";
      }
      else if (version == 1.6f)
      {
         vers="1.6";
      }
      else
      {
         throw new UnknownAJRVersionException(""+version);
      }
      out.println("AJR "+vers);
      if (version >= 1.3f)
      {
         out.print(""+JDR.PAPER_ONLY+" ");
         paper.saveAJR(out, version);
      }
      else
      {
         out.print(""+JDR.NO_SETTINGS+" ");
      }
      out.println();
      //out.print("G ");
      //allObjects.saveAJR(out, version);
      objectLoader.saveAJR(allObjects, out, version);
   }

   /**
   * Reads image from AJR format.
   *
   * The image is stored as a {@link JDRGroup}.
   * The settings flag can afterwards be retrieved using
   * {@link #getLastLoadedSettingsID()}.
   * @param in BufferedReader to input file
   * @param settings Any settings found in file will be put here
   * @return the image as a <code>JDRGroup</code>
   * @throws IOException if any I/O errors
   * @throws InvalidFormatException if the file is incorrectly
   * formatted
   */
   public static JDRGroup load(BufferedReader in,
      CanvasSettings settings)
      throws IOException,InvalidFormatException
   {
      // add listeners if not already done so
      addAllListeners();

      JDR.bitmapReplaced = false;

      lineNum_ = 1;

      String word = readWord(in);

      if (!word.equals("AJR"))
      {
         throw new NotAJRException(
           "Not an AJR file. ('"+word+"' found)");
      }

      String thisFileVersion = readWord(in);

      if (thisFileVersion.equals("1.0"))
      {
         _lastLoadedVersion = 1.0f;
      }
      else if (thisFileVersion.equals("1.1"))
      {
         _lastLoadedVersion = 1.1f;
      }
      else if (thisFileVersion.equals("1.2"))
      {
         _lastLoadedVersion = 1.2f;
      }
      else if (thisFileVersion.equals("1.3"))
      {
         _lastLoadedVersion = 1.3f;
      }
      else if (thisFileVersion.equals("1.4"))
      {
         _lastLoadedVersion = 1.4f;
      }
      else if (thisFileVersion.equals("1.5"))
      {
         _lastLoadedVersion = 1.5f;
      }
      else if (thisFileVersion.equals("1.6"))
      {
         _lastLoadedVersion = 1.6f;
      }
      else
      {
         throw new UnknownAJRVersionException(thisFileVersion);
      }

      int settingsFlag;

      if (_lastLoadedVersion < 1.3f)
      {
         settingsFlag = (readBoolean(in)
             ? JDR.ALL_SETTINGS : JDR.NO_SETTINGS);
      }
      else
      {
         settingsFlag = readInt(in);
      }

      if (settingsFlag == JDR.ALL_SETTINGS)
      {
         CanvasSettings.readAJR(settings, in, _lastLoadedVersion);
      }
      else if (settingsFlag == JDR.PAPER_ONLY)
      {
         settings.paper = JDRPaper.readAJR(in, _lastLoadedVersion);
      }
      else if (settingsFlag != JDR.NO_SETTINGS)
      {
         throw new InvalidSettingsIDException(settingsFlag);
      }

      _lastLoadedSettings = settingsFlag;

/*
      char c = readChar(in);

      if (c != 'G')
      {
         throw new JDRMissingTopLevelException();
      }

      return JDRGroup.readAJR(in, _lastLoadedVersion);
*/
      JDRGroup allObjects = null;

      try
      {
         allObjects = (JDRGroup)objectLoader.loadAJR(in, _lastLoadedVersion);
      }
      catch (ClassCastException e)
      {
        throw new JDRMissingTopLevelException();
      }

      return allObjects;
   }

   /**
   * Reads sequence of characters terminated by white space
   * from the input stream. The result is returned as a string. Skips
   * any following white space.
   * <p>
   * The word can not contain more than {@link #buffLength} 
   * characters.
   * @param in input stream
   * @return the word as a string
   * @throws IOException if I/O error occurs
   * @throws EOFException if EOF encountered
   * @throws BufferOverflowException if word exceeds buffer length
   */
   public static String readWord(BufferedReader in)
      throws IOException,EOFException,java.nio.BufferOverflowException
   {
      int c = in.read();

      if (c == '\n') lineNum_++;

      while (c == ' ' || c == '\t' || c == '\n' || c == '\r' || c == '\f')
      {
         c = in.read();

         if (c == '\n') lineNum_++;
      }

      char[] buffer = new char[buffLength];
      int i = 0;

      while (!(c == ' ' || c == '\t' || c == '\n' || c == '\r' || c == '\f'))
      {
         if (c == -1)
         {
            throw new EOFException("EOF found while reading word");
         }

         if (i == buffLength)
         {
            throw new java.nio.BufferOverflowException();
         }

         buffer[i++] = (char)c;

         c = in.read();
      }

      if (c == '\n') lineNum_++;

      return (new String(buffer)).substring(0, i);
   }

   /**
   * Reads an integer terminated by white space from input 
   * stream. Skips through any following white space.
   * @param in input stream
   * @return the integer
   * @throws IOException if I/O error occurs
   * @throws EOFException if EOF encountered
   * @throws InvalidFormatException if invalid number format
   */
   public static int readInt(BufferedReader in)
   throws IOException, EOFException, InvalidFormatException
   {
      String word = readWord(in);

      try
      {
         return Integer.parseInt(word);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidFormatException(
            "Integer expected, '"+word +"' found (line "+lineNum_+")");
      }
   }

   /**
   * Reads a real number terminated by white space from input 
   * stream. Skips through any following white space. The
   * value is returned as a float.
   * @param in input stream
   * @return the number as a float
   * @throws IOException if I/O error occurs
   * @throws EOFException if EOF encountered
   * @throws InvalidFormatException if invalid number format
   */
   public static float readFloat(BufferedReader in)
   throws IOException, EOFException, InvalidFormatException
   {
      String word = readWord(in);

      try
      {
         return Float.parseFloat(word);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidFormatException(
            "Number expected, '"+word +"' found (line "+lineNum_+")");
      }
   }

   /**
   * Reads a real number terminated by white space from input 
   * stream. Skips through any following white space. The
   * value is returned as a double.
   * @param in input stream
   * @return the value as a double
   * @throws IOException if I/O error occurs
   * @throws EOFException if EOF encountered
   * @throws InvalidFormatException if invalid number format
   */
   public static double readDouble(BufferedReader in)
   throws IOException, EOFException, InvalidFormatException
   {
      String word = readWord(in);

      try
      {
         return Double.parseDouble(word);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidFormatException(
            "Number expected, '"+word +"' found (line "+lineNum_+")");
      }
   }

   /**
   * Reads a boolean value (that is 0 or 1) from the input stream.
   * Skips through any following white space.
   * @param in input stream
   * @return the value as a boolean
   * @throws IOException if I/O error occurs
   * @throws EOFException if EOF encountered
   * @throws InvalidFormatException if next word is not a 0 or 1
   */
   public static boolean readBoolean(BufferedReader in)
   throws IOException, EOFException,
      InvalidFormatException
   {
      String word = readWord(in);

      try
      {
         int value = Integer.parseInt(word);

         if (value == 0)
         {
            return false;
         }
         else if (value == 1)
         {
            return true;
         }
         else
         {
            throw new NumberFormatException();
         }
      }
      catch (NumberFormatException e)
      {
         throw new InvalidFormatException(
            "'1' or '0' expected, '"+word +"' found (line "+lineNum_+")");
      }
   }

   /**
   * Reads given number of characters (including white space)
   * from the input stream. The result is stored as a 
   * <code>String</code>.
   * <p>
   * Note: if the end of line terminator consists of two characters
   * it will be counted as two characters.
   * @param in input stream
   * @param stringLength number of characters
   * @return the string
   * @throws IOException if I/O error occurs
   * @throws EOFException if EOF encountered
   * @throws InvalidFormatException if <code>stringLength</code>
   * isn't positive
   */
   public static String readString(BufferedReader in,
      int stringLength)
   throws IOException,EOFException,InvalidFormatException
   {
      if (stringLength <= 0)
      {
         throw new InvalidFormatException(
           "positive value required for string length (line "
           +lineNum_+")");
      }

      int c = in.read();

      if (c == '\n') lineNum_++;

      char[] buffer = new char[stringLength];

      for (int i = 0; i < stringLength; i++)
      {
         if (c == -1)
         {
            throw new EOFException("EOF found while reading word");
         }

         buffer[i] = (char)c;

         c = in.read();

         if (c == '\n') lineNum_++;
      }

      return new String(buffer);
   }

   /**
    * Reads a character from input stream. Skips through any
    * following white space.
    * @param in input stream
    * @return the character
    * @throws IOException if I/O error occurs
    * @throws EOFException if EOF encountered
    */
   public static char readChar(BufferedReader in)
      throws IOException,EOFException
   {
      int c = in.read();

      if (c == -1)
      {
         throw new EOFException();
      }

      if (c == '\n') lineNum_++;

      while (c == ' ' || c == '\t' || c == '\n' || c == '\r')
      {
         c = in.read();

         if (c == -1)
         {
            throw new EOFException();
         }

         if (c == '\n') lineNum_++;
      }

      return (char)c;
   }

   /**
    * Writes boolean value as 0 (if false) or 1 (if true) to output 
    * stream terminated by white space.
    * @param out output stream
    * @param value boolean value
    */
   public static void writeBoolean(PrintWriter out, boolean value)
   {
      out.print((value?"1 ":"0 "));
   }

   /**
    * Writes the character to output 
    * stream terminated by white space.
    * @param out output stream
    * @param c character
    */
   public static void writeChar(PrintWriter out, char c)
   {
      out.print(""+c+" ");
   }

   /**
    * Writes the value of an integer to output 
    * stream terminated by white space.
    * @param out output stream
    * @param value number
    */
   public static void writeInt(PrintWriter out, int value)
   {
      out.print(""+value+" ");
   }

   /**
    * Writes the value of a float to output 
    * stream terminated by white space.
    * @param out output stream
    * @param value number
    */
   public static void writeFloat(PrintWriter out, float value)
   {
      out.print(""+value+" ");
   }

   /**
    * Writes the value of a double to output 
    * stream terminated by white space.
    * @param out output stream
    * @param value number
    */
   public static void writeDouble(PrintWriter out, double value)
   {
      out.print(""+value+" ");
   }

   /**
    * Gets current line number.
    * The current line number is set to 1 at the start of
    * {@link #load(BufferedReader, CanvasSettings)} and 
    * incremented each time <code>\n</code> is encountered.
    * @return current line number
    */
   public static int getLineNum()
   {
      return lineNum_;
   }

   /**
    * Gets the version number of last AJR file to be loaded.
    * The version number will be one of: 1.0, 1.1, 1.2, 1.3, 1.4 or 1.5.
    * If no AJR file has been loaded using 
    * {@link #load(BufferedReader, CanvasSettings)}, 0 is returned.
    * @return version number or 0 if no AJR file has been loaded
    */
   public static float getLastLoadedVersion()
   {
      return _lastLoadedVersion;
   }

   /**
    * Gets the settings ID from last AJR file to be loaded.
    * The settings ID will be one of: {@link JDR#NO_SETTINGS},
    * {@link JDR#ALL_SETTINGS}, {@link JDR#PAPER_ONLY} or
    * -1 if no AJR file has been loaded using
    * {@link #load(BufferedReader, CanvasSettings)}.
    * @return settings ID or -1 if no AJR file has been loaded
    */
   public static int getLastLoadedSettingsID()
   {
      return _lastLoadedSettings;
   }

   /**
    * Adds all the listeners if not already done.
    */
    private static void addAllListeners()
    {
       if (paintLoader.getListeners().isEmpty())
       {
          paintLoader.addListener(new JDRColorListener());
          paintLoader.addListener(new JDRTransparentListener());
          paintLoader.addListener(new JDRColorCMYKListener());
          paintLoader.addListener(new JDRGradientListener());
          paintLoader.addListener(new JDRRadialListener());
          paintLoader.addListener(new JDRGrayListener());
          paintLoader.addListener(new JDRColorHSBListener());
       }

       if (objectLoader.getListeners().isEmpty())
       {
          objectLoader.addListener(new JDRGroupListener());
          objectLoader.addListener(new JDRPathListener());
          objectLoader.addListener(new JDRTextListener());
          objectLoader.addListener(new JDRBitmapListener());
          objectLoader.addListener(new JDRTextPathListener());
          objectLoader.addListener(new JDRSymmetricPathListener());
          objectLoader.addListener(new JDRRotationalPatternListener());
          objectLoader.addListener(new JDRScaledPatternListener());
          objectLoader.addListener(new JDRSpiralPatternListener());
       }

       if (segmentLoader.getListeners().isEmpty())
       {
          segmentLoader.addListener(new JDRSegmentLoaderListener());
          segmentLoader.addListener(new JDRLineLoaderListener());
          segmentLoader.addListener(new JDRBezierLoaderListener());
          segmentLoader.addListener(new JDRPartialSegmentLoaderListener());
          segmentLoader.addListener(new JDRPartialLineLoaderListener());
          segmentLoader.addListener(new JDRPartialBezierLoaderListener());
       }

       if (pathStyleLoader.getListeners().isEmpty())
       {
          pathStyleLoader.addListener(new JDRBasicPathStyleListener());
          pathStyleLoader.addListener(new JDRTextPathStyleListener());
       }

       if (gridLoader.getListeners().isEmpty())
       {
          gridLoader.addListener(new JDRRectangularGridListener());
          gridLoader.addListener(new JDRRadialGridListener());
       }
    }

    public static JDRPaintLoader getPaintLoader()
    {
       addAllListeners();
       return paintLoader;
    }

    public static JDRObjectLoader getObjectLoader()
    {
       addAllListeners();
       return objectLoader;
    }

   public static JDRSegmentLoader getSegmentLoader()
   {
      addAllListeners();
      return segmentLoader;
   }

   public static JDRPathStyleLoader getPathStyleLoader()
   {
      addAllListeners();
      return pathStyleLoader;
   }

    public static JDRGridLoader getGridLoader()
    {
       addAllListeners();
       return gridLoader;
    }

   /**
    * Newest AJR version number.
    */
   public static final float currentVersion = 1.6f;

   /** 
    * Maximum buffer length. 
    *
    * @see #readWord(BufferedReader)
    */
   public static int buffLength=255;

   /**
    * Stores the version number of the last AJR file to be loaded.
    *
    * @see #getLastLoadedVersion()
    */
   private static float _lastLoadedVersion = 0.0f;

   /**
    * Stores the value of the settings flag for the last AJR
    * file to be loaded.
    *
    * @see #getLastLoadedSettingsID()
    */
   private static int _lastLoadedSettings = -1;

   /**
    * Stores the value of the current line number.
    *
    * @see #getLineNum()
    */
   private static int lineNum_=0;

   private static JDRPaintLoader paintLoader = new JDRPaintLoader();

   private static JDRObjectLoader objectLoader = new JDRObjectLoader();

   private static JDRSegmentLoader segmentLoader = new JDRSegmentLoader();

   private static JDRPathStyleLoader pathStyleLoader 
      = new JDRPathStyleLoader();

   private static JDRGridLoader gridLoader = new JDRGridLoader();

}
