//File          : JDRUnit.java
//Description   : Provide conversion between TeX length units
//Author        : Nicola L.C. Talbot
//Date          : 17th March 2006
//Last Modified : 17th August 2010
//              http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr;

import java.awt.*;
import java.awt.geom.*;

/**
 * Class representing units of length. A normalizing factor is required
 * to convert to/from screen pixels. Since this library was written
 * primarily to create LaTeX files, the principle units are pt (TeX
 * points) and bp (Postscript points), where 1in = 72.27pt = 72bp.
 * @author Nicola L C Talbot
 */
public class JDRUnit
{
   /**
    * Creates a new unit.
    * @param name the name of this unit
    * @param conversionFactor the factor to convert from this unit
    * to TeX points (<code>pt = conversionFactor * unit</code>)
    */
   public JDRUnit (String name, double conversionFactor)
   {
      label = name;
      factor = conversionFactor;
   }

   private JDRUnit (String name, double conversionFactor, int identNum)
   {
      label = name;
      factor = conversionFactor;
      id = identNum;
   }

   /**
    * Converts value (in this unit) to TeX points.
    * @param val value in terms of this unit
    * @return value in terms of TeX points
    * @see #fromPt(double)
    * @see #toBp(double)
    * @see #fromBp(double)
    */
   public double toPt(double val)
   {
      return val*factor;
   }

   /**
    * Converts value in TeX points to value in terms of this unit.
    * @param val value in terms of TeX points
    * @return value in terms of this unit
    * @see #toPt(double)
    * @see #toBp(double)
    * @see #fromBp(double)
    */
   public double fromPt(double val)
   {
      return val/factor;
   }

   /**
    * Converts value (in this unit) to PostScript points.
    * @param val value in terms of this unit
    * @return value in terms of PostScript points
    * @see #fromBp(double)
    * @see #toPt(double)
    * @see #fromPt(double)
    */
   public double toBp(double val)
   {
      return toUnit(val, JDRUnit.bp);
   }

   /**
    * Converts value in PostScript points to value in terms of 
    * this unit.
    * @param val value in terms of PostScript points
    * @return value in terms of this unit
    * @see #toBp(double)
    * @see #toPt(double)
    * @see #fromPt(double)
    */
   public double fromBp(double val)
   {
      return fromUnit(val, JDRUnit.bp);
   }

   /**
    * Converts value (in this unit) to pixels. The normalizing factor
    * must be set first. The conversion is given by
    * <code>toBp(val)*normalizingFactor</code>
    * @param val value in terms of this unit
    * @return value in terms of pixels
    * @see #fromPixel(double)
    * @see #setNormalizingFactor(double)
    * @see #setNormalizingFactor(GraphicsConfiguration)
    */
   public double toPixel(double val)
   {
      return toBp(val)*normalizingFactor;
   }

   /**
    * Converts value in pixels to value in terms of this unit.
    * The normalizing factor
    * must be set first. The conversion is given by
    * <code>fromBp(val/normalizingFactor)</code>
    * @param val value in terms of pixels
    * @return value in terms of this unit
    * @see #toPixel(double)
    * @see #setNormalizingFactor(double)
    * @see #setNormalizingFactor(GraphicsConfiguration)
    */
   public double fromPixel(double val)
   {
      return fromBp(val/normalizingFactor);
   }

   /**
    * Converts value (in this unit) to value in terms of another unit.
    * @param val value in terms of this unit
    * @param unit the other unit
    * @return value in terms of the other unit
    * @see #fromUnit(double,JDRUnit)
    * @see #fromBp(double)
    * @see #toBp(double)
    * @see #toPt(double)
    * @see #fromPt(double)
    */
   public double toUnit(double val, JDRUnit unit)
   {
      return unit.fromPt(toPt(val));
   }

   /**
    * Converts value in another unit to value in terms of 
    * this unit.
    * @param val value in terms of another unit
    * @param unit the other unit
    * @return value in terms of this unit
    * @see #toUnit(double,JDRUnit)
    * @see #toBp(double)
    * @see #fromBp(double)
    * @see #toPt(double)
    * @see #fromPt(double)
    */
   public double fromUnit(double val, JDRUnit unit)
   {
      return fromPt(unit.toPt(val));
   }

   /**
    * Gets the normalizing factor. That is, the factor that converts
    * from PostScript points to pixels.
    * @return normalizing factor
    * @see #setNormalizingFactor(double)
    * @see #setNormalizingFactor(GraphicsConfiguration)
    * @see #toPixel(double)
    * @see #fromPixel(double)
    */
   public static double getNormalizingFactor()
   {
      return normalizingFactor;
   }

   /**
    * Sets the normalizing factor. That is, the factor that converts
    * from PostScript points to pixels.
    * @param normFactor normalizing factor
    * @see #setNormalizingFactor(GraphicsConfiguration)
    * @see #getNormalizingFactor()
    * @see #toPixel(double)
    * @see #fromPixel(double)
    */
   public static void setNormalizingFactor(double normFactor)
   {
      normalizingFactor = normFactor;
   }

   /**
    * Sets the normalizing factor. That is, the factor that converts
    * from PostScript points to pixels. This factor is obtained from
    * the x scaling factor of the transformation obtained 
    * by concatenating the default 
    * transform ({@link GraphicsConfiguration#getDefaultTransform()})
    * with the normalizing transform
    * ({@link GraphicsConfiguration#getNormalizingTransform()})
    * @param gc graphics configuration
    * @see #setNormalizingFactor(double)
    * @see #getNormalizingFactor()
    * @see #toPixel(double)
    * @see #fromPixel(double)
    */
   public static void setNormalizingFactor(GraphicsConfiguration gc)
   {
      AffineTransform af = gc.getDefaultTransform();
      af.concatenate(gc.getNormalizingTransform());
      setNormalizingFactor(af.getScaleX());
   }

   /**
    * Parses a string containing a length in one of the known units
    * and returns the value in terms of PostScript points. If the
    * unit is omitted, PostScript points are assumed. The string
    * must be in the form: number optionally followed by a unit.
    * Examples: "34.5cm", "2in",  "56". The last example assumes
    * PostScript units.
    * @param string the string containing the length
    * @return the value from the string converted to PostScript points
    * @throws InvalidUnitException if the unit is not recognised
    * @throws InvalidDimensionException if the string is incorrectly
    * formatted
    */
   public static double parseUnit(String string)
      throws InvalidDimensionException,InvalidUnitException
   {
      JDRUnit unit = bp;

      String[] split = string.split("[-+0-9.]*",2);

      if (split[1].equals("pt"))
      {
         unit = pt;
      }
      else if (split[1].equals("bp"))
      {
         unit = bp;
      }
      else if (split[1].equals("in"))
      {
         unit = in;
      }
      else if (split[1].equals("mm"))
      {
         unit = mm;
      }
      else if (split[1].equals("cm"))
      {
         unit = cm;
      }
      else if (split[1].equals("pc"))
      {
         unit = pc;
      }
      else if (split[1].equals("dd"))
      {
         unit = dd;
      }
      else if (split[1].equals("cc"))
      {
         unit = cc;
      }
      else if (split[1].equals(""))
      {
         unit = bp;
      }
      else
      {
         throw new InvalidUnitException(split[1]);
      }

      int idx = string.lastIndexOf(split[1]);

      try
      {
         double d = Double.parseDouble(string.substring(0,idx));
         return unit.toBp(d);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidDimensionException(string);
      }
   }

   /**
    * Gets the label identifying this unit.
    * @return this unit's label
    */
   public String getLabel()
   {
      return label;
   }

   /**
    * Gets a string representation of this unit.
    * @return string representation of this unit.
    */
   public String toString()
   {
      return "JDRUnit@"+label+"@"+factor;
   }

   public int getID()
   {
      return id;
   }

   /**
    * Gets the predefined unit given by the identifying number.
    * @param identNum the identifying number
    * @return the predefined unit associated with the given ID or
    * null
    */
   public static JDRUnit getUnit(int identNum)
   {
      switch (identNum)
      {
         case PT: return pt;
         case IN: return in;
         case CM: return cm;
         case BP: return bp;
         case MM: return mm;
         case PC: return pc;
         case DD: return dd;
         case CC: return cc;
      }

      return null;
   }

   /**
    * This unit's label.
    */
   private String label;
   private double factor;
   /**
    * Normalising factor to convert from PostScript points to
    * pixels.
    */
   private static double normalizingFactor=1;

   private int id = -1;

   /**
    * JDR/AJR unit identifier.
    */
   public static final byte PT=0, IN=1, CM=2, BP=3, MM=4, PC=5, DD=6,
      CC=7;

   // predefined units

   /** LaTeX point. */
   public static final JDRUnit pt = new JDRUnit("pt", 1.0, PT);
   /** Postscript point. */
   public static final JDRUnit bp = new JDRUnit("bp", 72.27/72.0, BP);
   /** Inch. */
   public static final JDRUnit in = new JDRUnit("in", 72.27, IN);
   /** Centimetre. */
   public static final JDRUnit cm = new JDRUnit("cm", 28.4528, CM);
   /** Millimetre. */
   public static final JDRUnit mm = new JDRUnit("mm", 2.84528, MM);
   /** Pica. */
   public static final JDRUnit pc = new JDRUnit("pc", 12.0, PC);
   /** Didot. */
   public static final JDRUnit dd = new JDRUnit("dd", 1.07001, DD);
   /** Cicero. */
   public static final JDRUnit cc = new JDRUnit("cc", 12.8401, CC);
}
