// File          : JDRTextPathStroke.java
// Date          : 10 July 2009
// Last Modified : 29th August 2010
// Author        : Nicola L.C. Talbot
//                 (with bits adapted from Jerry Huxtable's TextStroke.java)

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing stroke made of text.
 * This code was adapted from Jerry Huxtable's TextStroke.java
 * code (available from http://www.jhlabs.com/java/java2d/strokes/
 * Licensed under the Apache License, Version 2.0
 * http://www.apache.org/licenses/LICENSE-2.0)
 */

public class JDRTextPathStroke implements JDRStroke
{
   /**
     * Creates a new stroke using given text and font.
     * @param text the text to follow the required path
     * @param font the font
    */
   public JDRTextPathStroke(String text, Font font)
   {
      this.text = text;
      latexText = text;

      halign = LEFT;
      valign = BASE;

      jdrFont.setFamily(font.getName());

      try
      {
         jdrFont.setWeight(font.isBold() ? 
            JDRFont.SERIES_BOLD : JDRFont.SERIES_MEDIUM);
      }
      catch (InvalidFontWeightException ignore)
      {
      }

      try
      {
        jdrFont.setShape(font.isItalic() ?
           JDRFont.SHAPE_EM : JDRFont.SHAPE_UPRIGHT);
      }
      catch (InvalidFontShapeException ignore)
      {
      }

      try
      {
         jdrFont.setSize(font.getSize());
      }
      catch (InvalidFontSizeException ignore)
      {
      }

      this.font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());

      matrix = new double[6];

      matrix[0] = 1;
      matrix[1] = 0;
      matrix[2] = 0;
      matrix[3] = 1;
      matrix[4] = 0;
      matrix[5] = 0;
   }

   /**
    * Creates a new stroke using the given text attributes.
    * (The transformation matrix isn't used.)
    * @param jdrtext the text attributes
    */
   public JDRTextPathStroke(JDRText jdrtext)
   {
      text = jdrtext.getText();
      latexText = jdrtext.getLaTeXText();
      jdrFont = (JDRFont)jdrtext.getJDRFont().clone();
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
      latexFont = (LaTeXFont)jdrtext.latexFont.clone();
      halign = jdrtext.pgfHalign;
      valign = jdrtext.pgfValign;

      matrix = jdrtext.getTransformation(null);

      matrix[4]=0;
      matrix[5]=0;
   }

   public JDRTextPathStroke()
   {
      this("", "", new JDRFont(), JDRText.PGF_HALIGN_LEFT, 
           JDRText.PGF_VALIGN_BASE, new LaTeXFont());
   }

   protected JDRTextPathStroke(String text, String ltxtext, 
      JDRFont jdrfont, int hAlign, int vAlign, LaTeXFont ltxfont)
   {
      this.text = text;
      latexText = ltxtext;
      jdrFont = jdrfont;
      latexFont = ltxfont;
      halign = hAlign;
      valign = vAlign;

      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());

      matrix = new double[6];

      matrix[0] = 1;
      matrix[1] = 0;
      matrix[2] = 0;
      matrix[3] = 1;
      matrix[4] = 0;
      matrix[5] = 0;
   }

   public Object clone()
   {
      JDRTextPathStroke stroke = new JDRTextPathStroke(text, font);

      stroke.latexText = latexText;
      stroke.latexFont = (LaTeXFont)latexFont.clone();
      stroke.halign = halign;
      stroke.valign = valign;

      for (int i = 0; i < 6; i++)
      {
         stroke.matrix[i] = matrix[i];
      }

      return stroke;
   }

   /**
    * Gets the text attributes.
    * @param g the graphics device
    * @return the text attributes
    */
   public JDRText getJDRText(Graphics g)
   {
      JDRText jdrtext = new JDRText();

      jdrtext.jdrFont = (JDRFont)jdrFont.clone();
      jdrtext.font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
      jdrtext.latexFont = (LaTeXFont)latexFont.clone();

      jdrtext.setText(g, text, latexText);

      jdrtext.pgfHalign = halign;
      jdrtext.pgfValign = valign;

      jdrtext.setTransformation(matrix);

      return jdrtext;
   }

   /**
    * Gets the text attributes. (Doesn't set the bounds)
    * @return the text attributes
    */
   public JDRText getJDRText()
   {
      return getJDRText(null);
   }

   public void setFont(String name, int series, int shape, int size)
   throws InvalidFontWeightException,
          InvalidFontShapeException,
          InvalidFontSizeException
   {
      setFontFamily(name);
      setFontSeries(series);
      setFontShape(shape);
      setFontSize(size);
   }

   /**
    * Sets the font family.
    * @param name the Java font family name
    */
   public void setFontFamily(String name)
   {
      jdrFont.setFamily(name);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
   }

   /**
    * Sets the font series.
    * @param series the font series
    * @throws InvalidFontWeightException if the font weight is not one
    * of: {@link JDRFont#SERIES_MEDIUM} or {@link JDRFont#SERIES_BOLD}
    */
   public void setFontSeries(int series)
      throws InvalidFontWeightException
   {
      jdrFont.setWeight(series);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
   }

   /**
    * Sets the font shape.
    * @param shape the font shape
    * @throws InvalidFontShapeException if the font shape is not one
    * of:  {@link JDRFont#SHAPE_UPRIGHT}, {@link JDRFont#SHAPE_EM},
    * {@link JDRFont#SHAPE_ITALIC}, {@link JDRFont#SHAPE_SLANTED} or
    * {@link JDRFont#SHAPE_SC}
    */
   public void setFontShape(int shape)
      throws InvalidFontShapeException
   {
      jdrFont.setShape(shape);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
   }

   /**
    * Sets the font size.
    * @param size the font size (in PostScript points)
    * @throws InvalidFontSizeException if the font size is negative
    */
   public void setFontSize(int size)
      throws InvalidFontSizeException
   {
      jdrFont.setSize(size);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
   }

   /**
    * Gets the Java font family name.
    * @return font family name
    */
   public String getFontFamily()
   {
      return jdrFont.getFamily();
   }

   /**
    * Gets the font series.
    * @return font series
    */
   public int getFontSeries()
   {
      return jdrFont.getWeight();
   }

   /**
    * Gets the font shape.
    * @return font shape
    */
   public int getFontShape()
   {
      return jdrFont.getShape();
   }

   /**
    * Gets the Java font weight (required by {@link Font}).
    * @return {@link Font} weight
    */
   private int getFontWeight()
   {
      int weight = 0;
      weight += (getFontSeries() == JDRFont.SERIES_MEDIUM?
                 Font.PLAIN : Font.BOLD);
      weight += (getFontShape() == JDRFont.SHAPE_UPRIGHT?
                 0 : Font.ITALIC);

      return weight;
   }

   /**
    * Gets the font size.
    * @return font size (in PostScript points)
    */
   public int getFontSize()
   {
      return jdrFont.getSize();
   }

   /**
    * Gets the Java font.
    * @return font
    */
   public Font getFont()
   {
      return font;
   }

   /**
    * Gets the JDR font.
    * @return font
    */
   public JDRFont getJDRFont()
   {
      return jdrFont;
   }

   /**
    * Gets the text.
    * @return the text
    */
   public String getText()
   {
      return text;
   }

   /**
    * Gets the alternative LaTeX text.
    * @return the alternative LaTeX text
    */
   public String getLaTeXText()
   {
      return latexText;
   }

   public void setLaTeXFont(LaTeXFont ltxFont)
   {
      latexFont = ltxFont;
   }

   public LaTeXFont getLaTeXFont()
   {
      return latexFont;
   }

   public void setLaTeXFont(String family, String size, 
      String series, String shape)
   {
      latexFont.setFamily(family);
      latexFont.setWeight(series);
      latexFont.setShape(shape);
      latexFont.setSize(size);
   }

   public void setLaTeXFamily(String name)
   {
      latexFont.setFamily(name);
   }

   public void setLaTeXSeries(String series)
   {
      latexFont.setWeight(series);
   }

   public void setLaTeXShape(String shape)
   {
      latexFont.setShape(shape);
   }

   public void setLaTeXSize(String size)
   {
      latexFont.setSize(size);
   }

   public String getLaTeXFamily()
   {
      return latexFont.getFamily();
   }

   public String getLaTeXSeries()
   {
      return latexFont.getWeight();
   }

   public String getLaTeXSize()
   {
      return latexFont.getSize();
   }

   public String getLaTeXShape()
   {
      return latexFont.getShape();
   }

   /**
    * Sets the text.
    * @param newText the new text
    */
   public void setText(String newText)
   {
      text = newText.replaceAll("[\t\r\n]", " ");
      latexText = text;
   }

   /**
    * Sets the text and LaTeX alternative text.
    * @param newText the text
    * @param newLaTeXText the alternative text
    */
   public void setText(String newText, String newLaTeXText)
   {
      text = newText.replaceAll("[\t\r\n]", " ");
      latexText = newLaTeXText.replaceAll("[\t\r\n]", " ");
   }

   /**
    * Sets LaTeX alternative text.
    * @param newLaTeXText the alternative text
    */
   public void setLaTeXText(String newLaTeXText)
   {
      latexText = newLaTeXText.replaceAll("[\t\r\n]", " ");
   }

   /**
    * Sets the horizontal alignment
    * @param hAlign the horizontal alignment
    * @throws InvalidAHalignException if hAlign is not one of
    * {@link #LEFT}, {@link #CENTER} or {@link #RIGHT}
    * @see #getHAlign()
    * @see #setVAlign(int)
    */
   public void setHAlign(int hAlign)
      throws InvalidHAlignException
   {
      if (hAlign < 0 || hAlign > 2)
      {
         throw new InvalidHAlignException(hAlign);
      }

      halign = hAlign;
   }

   /**
    * Gets the horizontal alignment
    * @return the horizontal alignment
    * @see #setHAlign(int)
    * @see #getVAlign()
    */
   public int getHAlign()
   {
      return halign;
   }

   /**
    * Sets the vertical alignment
    * @param vAlign the vertical alignment
    * @throws InvalidVAlignException if vAlign is not one of
    * {@link #TOP}, {@link #MIDDLE}, {@link #BASE} or 
    * {@link #BOTTOM}
    * @see #getVAlign()
    * @see #setHAlign(int)
    */
   public void setVAlign(int vAlign)
      throws InvalidVAlignException
   {
      if (vAlign < 0 || vAlign > 3)
      {
         throw new InvalidVAlignException(vAlign);
      }

      valign = vAlign;
   }

   /**
    * Gets the vertical alignment
    * @return the vertical alignment
    * @see #setVAlign(int)
    * @see #getHAlign()
    */
   public int getVAlign()
   {
      return valign;
   }

   public void saveEPS(JDRShape path, PrintWriter out)
   {
   }

   public static JDRTextPathStroke read(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      // font specs
      JDRFont jf = JDRFont.read(din, version);

      // transformation matrix

      double[] mtx = new double[6];

      for (int i = 0; i < 6; i++)
      {
         mtx[i] = din.readDouble();
      }

      // LaTeX font specs

      LaTeXFont ltxfont;

      String ltxText = null;
      int hAlign = LEFT;
      int vAlign = RIGHT;

      if (din.readBoolean())
      {
         ltxfont = LaTeXFont.read(din, version);

         hAlign = (int)din.readByte();

         if (hAlign < 0 || hAlign > RIGHT)
         {
            throw new InvalidHAlignException(hAlign);
         }

         vAlign = (int)din.readByte();

         if (vAlign < 0 || vAlign > BOTTOM)
         {
            throw new InvalidVAlignException(vAlign);
         }

         int m = din.readInt();

         if (m > 0)
         {
            char[] lstr = new char[m];

            for (int i = 0; i < m; i++)
            {
               lstr[i] = din.readChar();
            }
            ltxText = new String(lstr);
         }
      }
      else
      {
         ltxfont = new LaTeXFont();
      }

      // text

      int n = din.readInt();

      String string = "";

      if (n <= 0)
      {
         throw new InvalidTextLengthException(n);
      }
      else
      {
         char[] strChars = new char[n];
         for (int i = 0; i < n; i++)
         {
            strChars[i] = din.readChar();
         }

         string = new String(strChars);
      }

      JDRTextPathStroke tps = new JDRTextPathStroke(string, ltxText, 
         jf, hAlign, vAlign, ltxfont);

      tps.setTransformation(mtx);

      return tps;
   }

   public void save(DataOutputStream dout, float version)
      throws IOException
   {
      // font specs
      jdrFont.save(dout,version);

      // transformation matrix

      for (int i = 0; i < 6; i++)
      {
         dout.writeDouble(matrix[i]);
      }

      // LaTeX stuff
      dout.writeBoolean(true);
      latexFont.save(dout, version);

      dout.writeByte((byte)halign);
      dout.writeByte((byte)valign);

      if (latexText == null || latexText.equals(text))
      {
         dout.writeInt(0);
      }
      else
      {
         dout.writeInt(latexText.length());
         dout.writeChars(latexText);
      }

      // text
      dout.writeInt(text.length());
      dout.writeChars(text);
   }

   public static JDRTextPathStroke readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      // font specs
      JDRFont jf = JDRFont.readAJR(in, version);

      // transformation matrix

      double[] mtx = new double[6];

      for (int i = 0; i < 6; i++)
      {
         mtx[i] = AJR.readDouble(in);
      }

      // LaTeX font specs

      LaTeXFont ltxfont;

      String ltxText = null;
      int hAlign = LEFT;
      int vAlign = RIGHT;

      if (AJR.readBoolean(in))
      {
         ltxfont = LaTeXFont.readAJR(in, version);

         hAlign = AJR.readInt(in);

         if (hAlign < 0 || hAlign > RIGHT)
         {
            throw new InvalidHAlignException(hAlign);
         }

         vAlign = AJR.readInt(in);

         if (vAlign < 0 || vAlign > BOTTOM)
         {
            throw new InvalidVAlignException(vAlign);
         }

         int m = AJR.readInt(in);

         if (m > 0)
         {
            ltxText = AJR.readString(in, m);
         }
      }
      else
      {
         ltxfont = new LaTeXFont();
      }

      // text

      int n = AJR.readInt(in);

      String string = "";

      if (n <= 0)
      {
         throw new InvalidTextLengthException(n);
      }
      else
      {
         string = AJR.readString(in, n);
      }

      JDRTextPathStroke tps = new JDRTextPathStroke(string, ltxText, 
         jf, hAlign, vAlign, ltxfont);

      tps.setTransformation(mtx);

      return tps;
   }

   public void saveAJR(PrintWriter out, float version)
      throws IOException, InvalidFormatException
   {
      // font specs
      jdrFont.saveAJR(out, version);

      // transformation matrix

      for (int i = 0; i < 6; i++)
      {
         AJR.writeDouble(out, matrix[i]);
      }

      // LaTeX stuff
      AJR.writeBoolean(out, true);
      latexFont.saveAJR(out, version);

      AJR.writeInt(out, halign);
      AJR.writeInt(out, valign);

      if (latexText == null || latexText.equals(text))
      {
         AJR.writeInt(out, 0);
      }
      else
      {
         AJR.writeInt(out, latexText.length());
         out.println(latexText);
      }

      // text
      AJR.writeInt(out, text.length());
      out.println(text);
   }

   public Area getStrokedArea(JDRShape path)
   {
      return new Area(getStrokedPath(path));
   }

   public Shape getStrokedPath(JDRShape path)
   {
      return createStrokedShape(path.getGeneralPath());
   }

   public void drawPath(JDRShape path, Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;
      //g2.draw(path.getGeneralPath());

      g2.fill(path.getStrokedPath());
   }

   public void drawPath(JDRShape shape, GeneralPath path, Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;

      g2.fill(createStrokedShape(path));
   }

   public void setWindingRule(int rule)
      throws InvalidWindingRuleException
   {
   }

   public int getWindingRule()
   {
      return GeneralPath.WIND_NON_ZERO;
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "TextPathStroke:"+eol;
      str += "text: "+text+eol;
      str += "LaTeX equivalent: "+latexText+eol;
      str += "font: "+jdrFont.info()+eol;
      str += "LaTeX font: "+latexFont.info()+eol;
      str += "HAlign: "+halign+eol;
      str += "VAlign: "+valign+eol;

      str += "Matrix: [";

      for (int i = 0; i < 6; i++)
      {
         if (i != 0) str += ",";

         str += matrix[i];
      }

      str += "]";

      return str;
   }

   /**
    * Creates outline of text along the path.
    * This code was adapted from Jerry Huxtable's TextStroke.java
    * code.
    * @param shape the shape along which the text should go
    * @return outline of text along the given shape
    */
   public Shape createStrokedShape(Shape shape)
   {
      FontRenderContext frc = new FontRenderContext(null, true, true);

      Font transformedFont = font.deriveFont(
         new AffineTransform(matrix));
      GlyphVector glyphVector
         = transformedFont.createGlyphVector(frc, text);

      TextLayout layout = new TextLayout(text, transformedFont, frc);

      float descent = layout.getDescent();
      float ascent = layout.getAscent();
      Rectangle2D bounds = layout.getBounds();

      float xoffset = (float)matrix[4];
      float yoffset = (float)matrix[5];

      switch (valign)
      {
         case TOP:
            yoffset = -ascent;
         break;

         case MIDDLE:
            yoffset = descent-(float)bounds.getHeight()*0.5f;
         break;

         case BOTTOM:
           yoffset = descent;
         break;
      }

      if (halign != LEFT)
      {
         double pathLength = measurePathLength(shape);

         if (halign == CENTER)
         {
            xoffset = (float)(pathLength-bounds.getWidth())*0.5f;
         }
         else
         {
            xoffset = (float)(pathLength-bounds.getWidth());
         }
      }

      GeneralPath result = new GeneralPath();
      PathIterator it = new FlatteningPathIterator(
        shape.getPathIterator(null), FLATNESS);

      float points[] = new float[6];
      float moveX = 0;
      float moveY = 0;
      float lastX = 0;
      float lastY = 0;
      float thisX = 0;
      float thisY = 0;

      int type = 0;
      boolean first = false;
      float next = 0;
      int currentChar = 0;
      int length = glyphVector.getNumGlyphs();

      if (length == 0) return result;

      float nextAdvance = 0;

      while (currentChar < length && !it.isDone())
      {
         type = it.currentSegment(points);

         switch (type)
         {
            case PathIterator.SEG_MOVETO:
               moveX = lastX = points[0];
               moveY = lastY = points[1];
               result.moveTo(moveX, moveY);
               first = true;
               nextAdvance = glyphVector.getGlyphMetrics(currentChar)
                 .getAdvance() * 0.5f;
               next = nextAdvance+xoffset;
            break;

            case PathIterator.SEG_CLOSE:
               points[0] = moveX;
               points[1] = moveY;
            case PathIterator.SEG_LINETO:
               thisX = points[0];
               thisY = points[1];
               float dx = thisX-lastX;
               float dy = thisY-lastY;
               double distance = Math.sqrt(dx*dx+dy*dy);

               if (distance >= next)
               {
                  double r = 1.0/distance;
                  double angle = Math.atan2(dy,dx);

                  while (currentChar < length && distance >= next)
                  {
                     Shape glyph = glyphVector.getGlyphOutline(currentChar);
                     Point2D p = glyphVector.getGlyphPosition(currentChar);
                     double px = p.getX();
                     double py = p.getY()+yoffset;
                     double x = lastX + next*dx*r;
                     double y = lastY + next*dy*r;

                     double advance = nextAdvance;
                     nextAdvance = currentChar < length-1 ?
                        glyphVector.getGlyphMetrics(currentChar+1).getAdvance() * 0.5f :
                        0.0f;

                     af.setToTranslation(x, y);
                     af.rotate(angle);
                     af.translate(-px-advance, -py);

                     result.append(af.createTransformedShape(glyph),false);

                     currentChar++;

                     next += advance+nextAdvance;
                  }
               }

               next -= distance;
               first = false;
               lastX = thisX;
               lastY = thisY;
            break;
         }

         it.next();
      }

      return result;
   }

   /**
    * Measure the approximate length of the given shape.
    * This code was adapted from Jerry Huxtable's TextStroke.java
    * code.
   */
   public static float measurePathLength(Shape shape)
   {
      PathIterator it = new FlatteningPathIterator(
        shape.getPathIterator(null), FLATNESS);

      float points[] = new float[6];
      float moveX = 0;
      float moveY = 0;
      float lastX = 0;
      float lastY = 0;
      float thisX = 0;
      float thisY = 0;

      int type = 0;
      double total = 0;

      while (!it.isDone())
      {
         type = it.currentSegment(points);

         switch (type)
         {
            case PathIterator.SEG_MOVETO:
               moveX = lastX = points[0];
               moveY = lastY = points[1];
            break;

            case PathIterator.SEG_CLOSE:
               points[0] = moveX;
               points[1] = moveY;
            case PathIterator.SEG_LINETO:
               thisX = points[0];
               thisY = points[1];
               double dx = thisX-lastX;
               double dy = thisY-lastY;
               total += Math.sqrt(dx*dx+dy*dy);
               lastX = thisX;
               lastY = thisY;
            break;
         }

         it.next();
      }

      return (float)total;
   }

   public JDRGroup split(Graphics2D g2, JDRTextPath textPath)
   {
      Shape shape = textPath.getGeneralPath();

      JDRGroup group = new JDRGroup();

      double[] mtx = new double[6];

      FontRenderContext frc = new FontRenderContext(null, true, true);

      Font transformedFont = getFont().deriveFont(
         new AffineTransform(matrix));
      GlyphVector glyphVector
         = transformedFont.createGlyphVector(frc, text);

      TextLayout layout = new TextLayout(text, transformedFont, frc);

      float descent = layout.getDescent();
      float ascent = layout.getAscent();
      Rectangle2D bounds = layout.getBounds();

      float yoffset = (float)matrix[5];
      float xoffset = (float)matrix[4];

      switch (valign)
      {
         case TOP:
            yoffset = -ascent;
         break;

         case MIDDLE:
            yoffset = descent-(float)bounds.getHeight()*0.5f;
         break;

         case BOTTOM:
           yoffset = descent;
         break;
      }

      if (halign != LEFT)
      {
         double pathLength = measurePathLength(shape);

         if (halign == CENTER)
         {
            xoffset = (float)(pathLength-bounds.getWidth())*0.5f;
         }
         else
         {
            xoffset = (float)(pathLength-bounds.getWidth());
         }
      }

      PathIterator it = new FlatteningPathIterator(
        shape.getPathIterator(null), FLATNESS);


      float points[] = new float[6];
      float moveX = 0;
      float moveY = 0;
      float lastX = 0;
      float lastY = 0;
      float thisX = 0;
      float thisY = 0;

      int type = 0;
      boolean first = false;
      float next = 0;
      int currentChar = 0;
      int length = glyphVector.getNumGlyphs();

      if (length == 0) return group;

      float nextAdvance = 0;

      while (currentChar < length && !it.isDone())
      {
         type = it.currentSegment(points);

         switch (type)
         {
            case PathIterator.SEG_MOVETO:
               moveX = lastX = points[0];
               moveY = lastY = points[1];
               first = true;
               nextAdvance = glyphVector.getGlyphMetrics(currentChar)
                 .getAdvance() * 0.5f;
               next = nextAdvance+xoffset;
            break;

            case PathIterator.SEG_CLOSE:
               points[0] = moveX;
               points[1] = moveY;
            case PathIterator.SEG_LINETO:
               thisX = points[0];
               thisY = points[1];
               float dx = thisX-lastX;
               float dy = thisY-lastY;
               double distance = Math.sqrt(dx*dx+dy*dy);

               if (distance >= next)
               {
                  double r = 1.0/distance;
                  double angle = Math.atan2(dy,dx);

                  while (currentChar < length && distance >= next)                  {
                     Shape glyph = glyphVector.getGlyphOutline(currentChar);
                     Point2D p = glyphVector.getGlyphPosition(currentChar);
                     double x = lastX + next*dx*r;
                     double y = lastY + next*dy*r;

                     double advance = nextAdvance;
                     nextAdvance = currentChar < length-1 ?
                        glyphVector.getGlyphMetrics(currentChar+1).getAdvance() * 0.5f :
                        0.0f;

                     char thisChar = text.charAt(currentChar);

                     if (!Character.isWhitespace(thisChar)
                      && glyphVector.getGlyphMetrics(currentChar)
                                    .isStandard())
                     {
                        JDRText textArea
                           = new JDRText(g2, ""+thisChar);

                        af.setToIdentity();
                        af.translate(x, y-yoffset);
                        af.rotate(angle);
                        af.translate(-advance, 0);
                        af.concatenate(
                           new AffineTransform(matrix[0], matrix[1],
                              matrix[2], matrix[3], 0, 0));

                        af.getMatrix(mtx);
                        textArea.setTransformation(mtx);

                        textArea.setLaTeXFamily(getLaTeXFamily());
                        textArea.setLaTeXSize(getLaTeXSize());
                        textArea.setLaTeXSeries(getLaTeXSeries());
                        textArea.setLaTeXShape(getLaTeXShape());
                        textArea.setTextPaint(textPath.getLinePaint());

                        try
                        {
                           textArea.setFont(g2, getFontFamily(),
                                            getFontSeries(),
                                            getFontShape(),
                                            getFontSize());
                           textArea.setVAlign(getVAlign());
                           textArea.setHAlign(getHAlign());
                        }
                        catch (InvalidFormatException e)
                        {
                           // this shouldn't happen
                        }

                        group.add(textArea);
                     }

                     currentChar++;

                     next += advance+nextAdvance;
                  }
               }

               next -= distance;
               first = false;
               lastX = thisX;
               lastY = thisY;
            break;
         }
          it.next();
      }

      group.setSelected(textPath.isSelected());
      return group;
   }

   public double[] getTransformation(double[] mtx)
   {
      if (mtx == null)
      {
         mtx = new double[6];
      }

      for (int i = 0; i < 6; i++)
      {
         mtx[i] = matrix[i];
      }

      return mtx;
   }

   public void setTransformation(double[] mtx)
   {
      matrix = mtx;
   }

   public JDRPathStyleListener getPathStyleListener()
   {
      return pathStyleListener;
   }

   private String text;
   private Font font;
   private JDRFont jdrFont = new JDRFont();
   private int halign, valign;

   /**
    * Associated LaTeX font
    */
   public LaTeXFont latexFont = new LaTeXFont();

   /**
    * Associated LaTeX text
    */
   public String latexText;

   private AffineTransform af = new AffineTransform();

   /**
    * Horizontal alignment. (Before transformation)
    */
   public static final int LEFT=0, CENTER=1, RIGHT=2;

   /**
    * Vertical alignment. (Before transformation)
    */
   public static final int TOP=0, MIDDLE=1, BASE=2, BOTTOM=3;

   private static final float FLATNESS = 1.0f;

   private double[] matrix;

   private static JDRPathStyleListener pathStyleListener
      = new JDRTextPathStyleListener();
}
