// File          : JDRTextPathListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for text paths. This is for JDR/AJR versions
 * 1.5 and above. If the version number is less than 1.5, only the
 * path information is save.
 * @author Nicola L C Talbot
 */

public class JDRTextPathListener extends JDRPathListener
{
   public char getJDRid(float version)
   {
      return (version < 1.5f ? 'G' : 'X');
   }

   public void writeJDR(JDRObject object, DataOutputStream dout,
      float version)
   throws IOException,InvalidFormatException
   {
      JDRTextPath path = (JDRTextPath)object;

      if (version < 1.5f)
      {
         // Text paths not supported before version 1.5

         JDRGroup group = path.separate();

         JDRObjectLoaderListener listener = group.getListener();

         listener.writeJDR(group, dout, version);
      }
      else if (version < 1.6f)
      {
         JDRPaintLoader paintLoader = JDR.getPaintLoader();
         paintLoader.saveJDR(path.getTextPaint(), dout, version);
         path.getStroke().save(dout, version);

         writeJDRPathSpecs(path, dout, version);
      }
      else
      {
         JDRObjectLoader loader = JDR.getObjectLoader();

         loader.saveJDR(path.getUnderlyingShape(), dout, version);
      }
   }

   public JDRObject readJDR(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      JDRShape shape;

      if (version < 1.5f)
      {
         throw new InvalidFormatException
           ("Text-path not supported in JDR version "+version);
      }
      else if (version < 1.6f)
      {
         shape = new JDRPath();

         JDRPaintLoader paintLoader = JDR.getPaintLoader();

         shape.setLinePaint(paintLoader.loadJDR(din, version));
         shape.setFillPaint(JDRTransparent.Transparent);
         shape.setStroke(JDRTextPathStroke.read(din, version));

         readJDRPathSpecs(shape, din, version);
      }
      else
      {
         JDRObjectLoader loader = JDR.getObjectLoader();
         JDRObject object = loader.loadJDR(din, version);

         if (!(object instanceof JDRShape))
         {
            throw new InvalidFormatException
              ("Text-path object must have a shape as the underlying object");
         }

         shape = (JDRShape)object;
      }

      return new JDRTextPath(shape);
   }

   public char getAJRid(float version)
   {
      return (version < 1.5f ? 'G' : 'X');
   }


   public void writeAJR(JDRObject object, PrintWriter out,
      float version)
   throws IOException,InvalidFormatException
   {
      JDRTextPath path = (JDRTextPath)object;

      if (version < 1.5f)
      {
         // Text paths not supported before version 1.5

         JDRGroup group = path.separate();

         JDRObjectLoaderListener listener = group.getListener();

         listener.writeAJR(group, out, version);
      }
      else if (version < 1.6f)
      {
         JDRPaintLoader paintLoader = AJR.getPaintLoader();
         paintLoader.saveAJR(path.getTextPaint(), out, version);
         path.getStroke().saveAJR(out, version);

         writeAJRPathSpecs(path, out, version);
      }
      else
      {
         JDRObjectLoader loader = AJR.getObjectLoader();

         loader.saveAJR(path.getUnderlyingShape(), out, version);
      }
   }

   public JDRObject readAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      JDRShape shape;

      if (version < 1.5f)
      {
         throw new InvalidFormatException
           ("Text-path not supported in JDR version "+version);
      }
      else if (version < 1.6f)
      {
         shape = new JDRPath();

         JDRPaintLoader paintLoader = AJR.getPaintLoader();

         shape.setLinePaint(paintLoader.loadAJR(in, version));
         shape.setFillPaint(JDRTransparent.Transparent);
         shape.setStroke(JDRTextPathStroke.readAJR(in, version));

         readAJRPathSpecs(shape, in, version);
      }
      else
      {
         JDRObjectLoader loader = AJR.getObjectLoader();
         JDRObject object = loader.loadAJR(in, version);

         if (!(object instanceof JDRShape))
         {
            throw new InvalidFormatException
              ("Text-path object must have a shape as the underlying object");
         }

         shape = (JDRShape)object;
      }

      return new JDRTextPath(shape);
   }

}
