// File          : JDRTextPath.java
// Date          : 7th July 2009
// Last Modified : 7th July 2009
// Author        : Nicola L C Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing text flowing along a path.
 * @author Nicola L C Talbot
 */

public class JDRTextPath extends JDRCompoundShape implements JDRTextual
{
   /**
    * Creates a text path.
    * @param path the path
    * @param text the text to go along the path
    */
   public JDRTextPath(JDRShape path, JDRText text)
   {
      super();

      path_ = path;

      setStroke(new JDRTextPathStroke(text));
      setTextPaint(text.getTextPaint());
   }

   public JDRTextPath(JDRShape path, JDRTextPathStroke stroke)
   {
      super();

      path_ = path;

      setStroke(stroke);
      setTextPaint(path.getLinePaint());
   }

   public JDRTextPath(int n, JDRPaint paint, JDRTextPathStroke stroke)
   {
      super();

      path_ = new JDRPath(n);

      setStroke(stroke);
      setTextPaint(paint);
   }

   public JDRTextPath(JDRShape shape)
   {
      super();

      path_ = shape;

      if (!(shape.getStroke() instanceof JDRTextPathStroke))
      {
         setStroke(new JDRTextPathStroke());
      }
   }

   public JDRStroke getStroke()
   {
      return path_.getStroke();
   }

   public void setStroke(JDRStroke stroke)
   {
      path_.setStroke(stroke);
   }

   public JDRPaint getTextPaint()
   {
      return path_.getLinePaint();
   }

   public void setTextPaint(JDRPaint paint)
   {
      path_.setLinePaint(paint);
   }

   public JDRPaint getFillPaint()
   {
      return JDRTransparent.Transparent;
   }

   public void setFillPaint(JDRPaint paint)
   {
   }

   public JDRPaint getLinePaint()
   {
      return path_.getLinePaint();
   }

   public void setLinePaint(JDRPaint paint)
   {
      path_.setLinePaint(paint);
   }

   /**
    * Gets the path for this.
    * @return the path
    */
   public JDRShape getJDRShape()
   {
      JDRShape path = (JDRShape)path_.clone();

      if (!(path_ instanceof JDRTextual))
      {
         path.setStroke(new JDRBasicStroke());
      }

      return path;
   }

   /**
    * Gets the text for this as a text area.
    * @return the text
    */
   public JDRText getJDRText(Graphics g)
   {
      JDRText text = ((JDRTextPathStroke)getStroke()).getJDRText(g);

      JDRPoint p = getFirstSegment().getStart();

      text.setPosition(p.x, p.y);

      text.setTextPaint(getLinePaint());

      return text;
   }

   /**
    * Gets the text for this as a text area.
    * @return the text
    */
   public JDRText getJDRText()
   {
      return getJDRText(null);
   }

   /**
    * Splits this into a group containing path and text area.
    * @return the group containing the underlying path and text
    */
   public JDRGroup separate(Graphics g)
   {
      JDRShape path = getJDRShape();
      JDRText text = getJDRText(g);
      text.updateBounds(g);

      JDRGroup group = new JDRGroup();

      group.add(path);
      group.add(text);

      return group;
   }

   /**
    * Splits this into a group containing path and text area.
    * @return the group containing the underlying path and text
    */
   public JDRGroup separate()
   {
      return separate(null);
   }

   public JDRGroup split(Graphics2D g2)
   {
      return ((JDRTextPathStroke)getStroke()).split(g2, this);
   }

   /**
    * Sets horizontal alignment.
    * @param align the alignment
    * @see JDRTextPathStroke#setHAlign(int)
    */
   public void setHAlign(int align)
   throws InvalidHAlignException
   {
      ((JDRTextPathStroke)getStroke()).setHAlign(align);
   }

   /**
    * Sets vertical alignment.
    * @param align the alignment
    * @see JDRTextPathStroke#setVAlign(int)
    */
   public void setVAlign(int align)
   throws InvalidVAlignException
   {
      ((JDRTextPathStroke)getStroke()).setVAlign(align);
   }

   /**
    * Sets horizontal and vertical alignment.
    * @param halign the horizontal alignment
    * @param valign the vertical alignment
    * @see #setVAlign(int)
    * @see #setHAlign(int)
    * @see JDRTextPathStroke#setVAlign(int)
    */
   public void setAlign(int halign, int valign)
   throws InvalidHAlignException,InvalidVAlignException
   {
      setHAlign(halign);
      setVAlign(valign);
   }

   /**
    * Gets horizontal alignment.
    * @return the horizontal alignment
    * @see #setHAlign(int)
    * @see #getVAlign()
    * @see JDRTextPathStroke#getHAlign()
    */
   public int getHAlign()
   {
      return ((JDRTextPathStroke)getStroke()).getHAlign();
   }

   /**
    * Gets vertical alignment.
    * @return the vertical alignment
    * @see #setVAlign(int)
    * @see #getHAlign()
    * @see JDRTextPathStroke#getVAlign()
    */
   public int getVAlign()
   {
      return ((JDRTextPathStroke)getStroke()).getVAlign();
   }

   public void setText(Graphics g, String text, String latexText)
   {
      ((JDRTextPathStroke)getStroke()).setText(text, latexText);
   }

   public void setText(Graphics g, String text)
   {
      ((JDRTextPathStroke)getStroke()).setText(text);
   }

   public void setLaTeXText(String latexText)
   {
      ((JDRTextPathStroke)getStroke()).setLaTeXText(latexText);
   }

   public String getText()
   {
      return ((JDRTextPathStroke)getStroke()).getText();
   }

   public String getLaTeXText()
   {
      return ((JDRTextPathStroke)getStroke()).getLaTeXText();
   }

   public void setFont(Graphics g, String name, int series, 
     int shape, int size)
   throws InvalidFontWeightException,
          InvalidFontShapeException,
          InvalidFontSizeException
   {
      ((JDRTextPathStroke)getStroke()).setFont(name, series,
        shape, size);
   }

   public void setFontFamily(Graphics g, String name)
   {
      ((JDRTextPathStroke)getStroke()).setFontFamily(name);
   }

   public void setFontSeries(Graphics g, int series)
   throws InvalidFontWeightException
   {
      ((JDRTextPathStroke)getStroke()).setFontSeries(series);
   }

   public void setFontShape(Graphics g, int shape)
   throws InvalidFontShapeException
   {
      ((JDRTextPathStroke)getStroke()).setFontShape(shape);
   }

   public void setFontSize(Graphics g, int size)
   throws InvalidFontSizeException
   {
      ((JDRTextPathStroke)getStroke()).setFontSize(size);
   }

   public String getFontFamily()
   {
      return ((JDRTextPathStroke)getStroke()).getFontFamily();
   }

   public int getFontSeries()
   {
      return ((JDRTextPathStroke)getStroke()).getFontSeries();
   }

   public int getFontShape()
   {
      return ((JDRTextPathStroke)getStroke()).getFontShape();
   }

   public int getFontSize()
   {
      return ((JDRTextPathStroke)getStroke()).getFontSize();
   }

   public Font getFont()
   {
      return ((JDRTextPathStroke)getStroke()).getFont();
   }

   public JDRFont getJDRFont()
   {
      return ((JDRTextPathStroke)getStroke()).getJDRFont();
   }

   public void setLaTeXFamily(String name)
   {
      ((JDRTextPathStroke)getStroke()).setLaTeXFamily(name);
   }

   public void setLaTeXSize(String size)
   {
      ((JDRTextPathStroke)getStroke()).setLaTeXSize(size);
   }

   public void setLaTeXSeries(String series)
   {
      ((JDRTextPathStroke)getStroke()).setLaTeXSeries(series);
   }

   public void setLaTeXShape(String shape)
   {
      ((JDRTextPathStroke)getStroke()).setLaTeXShape(shape);
   }

   public void setLaTeXFont(String family, String size, 
      String series, String shape)
   {
      ((JDRTextPathStroke)getStroke()).setLaTeXFont(family, size,
         series, shape);
   }

   public void setLaTeXFont(LaTeXFont ltxFont)
   {
      ((JDRTextPathStroke)getStroke()).setLaTeXFont(ltxFont);
   }

   public String getLaTeXFamily()
   {
      return ((JDRTextPathStroke)getStroke()).getLaTeXFamily();
   }

   public String getLaTeXShape()
   {
      return ((JDRTextPathStroke)getStroke()).getLaTeXShape();
   }

   public String getLaTeXSeries()
   {
      return ((JDRTextPathStroke)getStroke()).getLaTeXSeries();
   }

   public String getLaTeXSize()
   {
      return ((JDRTextPathStroke)getStroke()).getLaTeXSize();
   }

   public void makeEqual(JDRObject object)
   {
      JDRTextPath textPath = (JDRTextPath)object;

      super.makeEqual(textPath);

      path_.makeEqual(textPath.getUnderlyingShape());
   }

   public Object clone()
   {
      JDRTextPath textPath = new JDRTextPath((JDRShape)path_.clone());

      textPath.setSelected(isSelected());
      textPath.description = description;

      if (flowframe != null)
      {
         textPath.flowframe = (FlowFrame)flowframe.clone();
      }

      return textPath;
   }

   public JDRShape breakPath()
      throws InvalidPathException
   {
      return new JDRTextPath(path_.breakPath(), 
        (JDRTextPathStroke)getStroke());
   }

   public void draw(Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;
      Paint oldPaint = g2.getPaint();

      JDRPaint paint = getTextPaint();

      Shape shape= getStrokedArea();

      BBox box = null;

      if (paint instanceof JDRShading)
      {
         box = new BBox(shape.getBounds2D());
      }

      g2.setPaint(paint.getPaint(box));
      g2.fill(shape);
      g2.setPaint(oldPaint);
   }

   public void draw(Graphics g, double scale)
   {
      Graphics2D g2 = (Graphics2D)g;
      Paint oldPaint = g2.getPaint();

      AffineTransform trans = new AffineTransform();
      trans.scale(scale,scale);

      Shape shape = trans.createTransformedShape(getStrokedPath());

      g2.setPaint(getTextPaint().getPaint(getBBox()));
      g2.fill(shape);
      g2.setPaint(oldPaint);
   }

   public void drawDraft(Graphics g, double scale)
   {
      Graphics2D g2 = (Graphics2D)g;
      Paint oldPaint = g2.getPaint();

      path_.drawDraft(g, scale);

      AffineTransform trans = new AffineTransform();
      trans.scale(scale,scale);

      Shape shape = trans.createTransformedShape(getStrokedPath());

      g2.setPaint(getTextPaint().getPaint(getBBox()));
      g2.fill(shape);
      g2.setPaint(oldPaint);
   }

   public BBox getControlBBox()
   {
      return path_.getControlBBox();
   }

   public String pgf(AffineTransform af)
   {
      BBox pathBBox = getBBox();

      if (pathBBox == null) return "";

      String eol = System.getProperty("line.separator");

      String path = "";

      if (!description.equals(""))
      {
         path += "% "+description+eol;
      }

      JDRTextPathStroke s = (JDRTextPathStroke)getStroke();

      path += "\\begin{pgfscope}"+s.getLaTeXFont().tex()+eol;

      String latexText = getLaTeXText();

      if (latexText == null)
      {
         latexText = getText();
      }

      path += "\\pgfset{decoration/text={|"

           + getLinePaint().pgf(pathBBox)+"|"

           + latexText+"}}"+eol;

      path += "\\pgfdecoratepath{text along path}{";

      path += path_.pgfPath(af);

      path += "}"+eol;

      path += "\\end{pgfscope}";

      return path;
   }

   public void saveSVG(PrintWriter out)
   throws IOException
   {
      try
      {
         JDRShape outline = outlineToPath();

         outline.saveSVG(out);
      }
      catch (InvalidPathException e)
      {
      }
   }

   public void saveEPS(PrintWriter out)
   throws IOException
   {
      try
      {
         JDRShape outline = outlineToPath();

         outline.saveEPS(out);
      }
      catch (InvalidPathException e)
      {
      }
   }

   /**
   * Gets a new text path object with a full path as the underlying
   * shape.
   */
   public JDRShape getFullPath()
      throws EmptyPathException,IllFittingPathException
   {
      JDRTextPath newShape = new JDRTextPath(path_.getFullPath(),
        (JDRTextPathStroke)getStroke().clone());

      return newShape;
   }

   public JDRObjectLoaderListener getListener()
   {
      return textPathListener;
   }

   public double[] getTransformation(double[] mtx)
   {
      return ((JDRTextPathStroke)getStroke()).getTransformation(mtx);
   }

   public void setTransformation(double[] mtx)
   {
      ((JDRTextPathStroke)getStroke()).setTransformation(mtx);
   }

   public JDRShape reverse()
      throws InvalidPathException
   {
      JDRTextPath path = new JDRTextPath(path_.reverse(),
         (JDRTextPathStroke)getStroke().clone());

      return path;
   }

   public JDRShape exclusiveOr(JDRShape path)
   throws InvalidPathException
   {
      return new JDRTextPath(path_.exclusiveOr(path),
         (JDRTextPathStroke)getStroke().clone());
   }

   public JDRShape pathUnion(JDRShape path)
   throws InvalidPathException
   {
      return new JDRTextPath(path_.pathUnion(path),
         (JDRTextPathStroke)getStroke().clone());
   }

   public JDRShape intersect(JDRShape path)
   throws InvalidPathException
   {
      return new JDRTextPath(path_.intersect(path),
         (JDRTextPathStroke)getStroke().clone());
   }

   public JDRShape subtract(JDRShape path)
   throws InvalidPathException
   {
      return new JDRTextPath(path_.subtract(path),
         (JDRTextPathStroke)getStroke().clone());
   }

   /**
    * Gets string representation of this textpath.
    * @return string representation of this textpath
    */
   public String toString()
   {
      String str = "TextPath: text="+getText();

      str += ", size="+size()+", segments=[";

      for (int i = 0; i < size(); i++)
      {
         str += get(i)+",";
      }

      str += "]";

      return str;
   }

   public JDRTextual getTextual()
   {
      return this;
   }

   public boolean hasTextual()
   {
      return true;
   }

   public JDRShape getUnderlyingShape()
   {
      return path_;
   }

   public void setUnderlyingShape(JDRShape shape)
   {
      path_ = shape;
   }

   public JDRPathIterator getIterator()
   {
      return path_.getIterator();
   }

   public JDRPointIterator getPointIterator()
   {
      return path_.getPointIterator();
   }

   public int size()
   {
      return path_.size();
   }

   public int getCapacity()
   {
      return path_.getCapacity();
   }

   public void setCapacity(int capacity)
      throws IllegalArgumentException
   {
      path_.setCapacity(capacity);
   }

   public void open()
   {
      path_.open();
   }

   public void open(boolean removeLastSegment)
   {
      path_.open(removeLastSegment);
   }

   public void close(JDRPathSegment segment)
      throws EmptyPathException,IllFittingPathException
   {
      path_.close(segment);
   }

   public boolean isClosed()
   {
      return path_.isClosed();
   }

   public boolean segmentHasEnd(JDRPathSegment segment)
   {
      return path_.segmentHasEnd(segment);
   }

   public int getIndex(JDRPathSegment segment)
   {
      return path_.getIndex(segment);
   }

   public int getLastIndex(JDRPathSegment segment)
   {
      return path_.getLastIndex(segment);
   }

   public JDRPathSegment get(int index)
      throws ArrayIndexOutOfBoundsException
   {
      return path_.get(index);
   }

   public JDRPathSegment getLastSegment()
   {
      return path_.getLastSegment();
   }

   public JDRPathSegment getFirstSegment()
   {
      return path_.getLastSegment();
   }

   public JDRPoint getFirstControl()
   {
      return path_.getFirstControl();
   }

   public JDRPoint getLastControl()
   {
      return path_.getLastControl();
   }

   public void stopEditing()
   {
      path_.stopEditing();
   }

   public int getSelectedControlIndex()
   {
      return path_.getSelectedControlIndex();
   }

   protected void selectControl(JDRPoint p, int pointIndex, 
      int segmentIndex)
   {
      path_.selectControl(p, pointIndex, segmentIndex);
   }

   public JDRPathSegment getSelectedSegment()
   {
      return path_.getSelectedSegment();
   }

   public JDRPoint getSelectedControl()
   {
      return path_.getSelectedControl();
   }

   public int getSelectedIndex()
   {
      return path_.getSelectedIndex();
   }

   public GeneralPath getGeneralPath()
   {
      return path_.getGeneralPath();
   }

   public JDRPathSegment setSegment(int index, JDRPathSegment segment)
      throws ArrayIndexOutOfBoundsException
   {
      return path_.setSegment(index, segment);
   }

   public void add(JDRSegment s)
   {
      path_.add(s);
   }

   public JDRPoint addPoint()
   {
      return path_.addPoint();
   }

   public void makeContinuous(boolean atStart)
   {
      path_.makeContinuous(atStart);
   }

   public void convertSegment(int idx, JDRPathSegment segment)
   {
      path_.convertSegment(idx, segment);
   }

   public JDRPathSegment remove(int i)
      throws ArrayIndexOutOfBoundsException
   {
      return path_.remove(i);
   }

   public JDRPathSegment remove(JDRPathSegment segment)
      throws ArrayIndexOutOfBoundsException
   {
      return path_.remove(segment);
   }

   public JDRPathSegment removeSelectedSegment()
      throws ArrayIndexOutOfBoundsException
   {
      return path_.removeSelectedSegment();
   }

   public JDRSegment removeSegment(int i)
      throws ArrayIndexOutOfBoundsException
   {
      return path_.removeSegment(i);
   }

   public void translateControl(JDRPathSegment segment, JDRPoint p,
      double x, double y)
   {
      path_.translateControl(segment, p, x, y);
   }

   public void translateParams(double shiftX, double shiftY)
   {
   }

   public void scaleParams(Point2D p, double factorX, double factorY)
   {
   }

   public void shearParams(Point2D p, double factorX, double factorY)
   {
   }

   public void rotateParams(Point2D p, double angle)
   {
   }

   public boolean showPath()
   {
      return false;
   }

   public boolean hasSymmetricPath()
   {
      return path_.hasSymmetricPath();
   }

   public void fade(double value)
   {
      getTextPaint().fade(value);
   }

   public JDRSymmetricPath getSymmetricPath()
   {
      return path_.getSymmetricPath();
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "TextPath:"+eol;

      str += "Underlying shape: "+path_.info();

      return str;
   }

   public String[] getDescriptionInfo()
   {
      return new String[] {getText(), getLaTeXText()};
   }

   public int getTotalPathSegments()
   {
      if (path_ instanceof JDRCompoundShape)
      {
         return ((JDRCompoundShape)path_).getTotalPathSegments();
      }

      return path_.size();
   }

   protected void setSelectedElements(int segmentIndex, int controlIndex,
      JDRPathSegment segment, JDRPoint control)
   {
      path_.setSelectedElements(segmentIndex, controlIndex, segment, control);
   }

   private static JDRTextPathListener textPathListener = new JDRTextPathListener();

   private JDRShape path_;
}
