// File          : JDRTextListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;
import java.awt.*;
import java.awt.image.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for text areas.
 * @author Nicola L C Talbot
 */

public class JDRTextListener implements JDRObjectLoaderListener
{
   public char getJDRid(float version)
   {
      return 'T';
   }

   public void writeJDR(JDRObject object,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException
   {
      JDRText text = (JDRText)object;

      // font specs
      text.getJDRFont().save(dout, version);

      // transformation

      text.getTransform().save(dout, version);

      // LaTeX stuff
      dout.writeBoolean(true);
      text.getLaTeXFont().save(dout, version);

      dout.writeByte((byte)text.getHAlign());
      dout.writeByte((byte)text.getVAlign());

      String latexText = text.getLaTeXText();

      if (latexText.equals(text.getText()))
      {
         dout.writeInt(0);
      }
      else
      {
         dout.writeInt(latexText.length());
         dout.writeChars(latexText);
      }

      // text colour

      JDRPaint paint = text.getTextPaint();

      JDRPaintLoader paintLoader = JDR.getPaintLoader();
      paintLoader.saveJDR(paint, dout, version);

      // text
      dout.writeInt(text.getText().length());
      dout.writeChars(text.getText());
   }

   public JDRObject readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      // font specs
      JDRFont jdrfont = JDRFont.read(din, version);

      // transformation matrix

      double[] matrix = JDRTransform.read(din, version);

      // LaTeX font specs

      LaTeXFont ltxfont = new LaTeXFont();

      String ltxText   = null;
      int halign = JDRText.PGF_HALIGN_LEFT;
      int valign = JDRText.PGF_VALIGN_BASE;

      if (din.readBoolean())
      {
         ltxfont = LaTeXFont.read(din, version);

         halign = (int)din.readByte();

         if (halign < 0 || halign > JDRText.PGF_HALIGN_RIGHT)
         {
            throw new InvalidHAlignException(halign);
         }

         valign = (int)din.readByte();

         if (valign < 0 || valign > JDRText.PGF_VALIGN_BOTTOM)
         {
            throw new InvalidVAlignException(valign);
         }

         int m = din.readInt();
         if (m > 0)
         {
            char[] lstr = new char[m];
            for (int i = 0; i < m; i++)
            {
               lstr[i] = din.readChar();
            }
            ltxText = new String(lstr);
         }
      }

      // JDRText colour

      JDRPaintLoader paintLoader = JDR.getPaintLoader();

      JDRPaint foreground = paintLoader.loadJDR(din, version);

      // text

      int n = din.readInt();

      String string = "";

      if (n <= 0)
      {
         throw new InvalidTextLengthException(n);
      }
      else
      {
         char[] strChars = new char[n];
         for (int i = 0; i < n; i++)
         {
            strChars[i] = din.readChar();
         }

         string = new String(strChars);
      }

      JDRTransform trans = new JDRTransform(matrix);

      JDRText textsegment = new JDRText(trans,
         jdrfont.getFamily(), jdrfont.getWeight(),
         jdrfont.getShape(), jdrfont.getSize(), string);

      textsegment.setTextPaint(foreground);

      textsegment.setLaTeXFont(ltxfont);

      textsegment.pgfHalign   = halign;
      textsegment.pgfValign   = valign;

      if (ltxText != null) textsegment.setLaTeXText(ltxText);

      // update bounds (not needed for JpgfDraw, as it
      // can call updateBounds, but required for jdr2ajr)
      BufferedImage buffImage = new BufferedImage(1,1,
         BufferedImage.TYPE_INT_RGB);

      Graphics2D g2 = buffImage.createGraphics();
      textsegment.updateBounds(g2);
      g2.dispose();

      return textsegment;
   }

   public char getAJRid(float version)
   {
      return 'T';
   }

   public void writeAJR(JDRObject object,
                        PrintWriter out,
                        float version)
   throws IOException,InvalidFormatException
   {
      JDRText text = (JDRText)object;

      // font specs
      text.getJDRFont().saveAJR(out, version);

      // transformation

      text.getTransform().saveAJR(out, version);

      // LaTeX stuff
      AJR.writeBoolean(out, true);
      text.getLaTeXFont().saveAJR(out, version);

      AJR.writeInt(out, text.getHAlign());
      AJR.writeInt(out, text.getVAlign());

      String latexText = text.getLaTeXText();

      if (latexText.equals(text.getText()))
      {
         AJR.writeInt(out, 0);
      }
      else
      {
         AJR.writeInt(out, latexText.length());
         out.println(latexText);
      }

      // text colour

      JDRPaint paint = text.getTextPaint();

      JDRPaintLoader paintLoader = JDR.getPaintLoader();
      paintLoader.saveAJR(paint, out, version);

      // text
      AJR.writeInt(out, text.getText().length());
      out.println(text.getText());
   }

   public JDRObject readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      // font specs
      JDRFont jdrfont = JDRFont.readAJR(in, version);

      // transformation matrix

      double[] matrix = JDRTransform.readAJR(in, version);

      // LaTeX font specs

      LaTeXFont ltxfont = new LaTeXFont();

      String ltxText   = null;
      int halign = JDRText.PGF_HALIGN_LEFT;
      int valign = JDRText.PGF_VALIGN_BASE;

      if (AJR.readBoolean(in))
      {
         ltxfont = LaTeXFont.readAJR(in, version);

         halign = AJR.readInt(in);

         if (halign < 0 || halign > JDRText.PGF_HALIGN_RIGHT)
         {
            throw new InvalidHAlignException(halign);
         }

         valign = AJR.readInt(in);

         if (valign < 0 || valign > JDRText.PGF_VALIGN_BOTTOM)
         {
            throw new InvalidVAlignException(valign);
         }

         int m = AJR.readInt(in);

         if (m > 0)
         {
            ltxText = AJR.readString(in, m);
         }
      }

      // JDRText colour

      JDRPaintLoader paintLoader = AJR.getPaintLoader();

      JDRPaint foreground = paintLoader.loadAJR(in, version);

      // text

      int n = AJR.readInt(in);

      String string = "";

      if (n <= 0)
      {
         throw new InvalidTextLengthException(n);
      }
      else
      {
         string = AJR.readString(in, n);
      }

      JDRTransform trans = new JDRTransform(matrix);

      JDRText textsegment = new JDRText(trans,
         jdrfont.getFamily(), jdrfont.getWeight(),
         jdrfont.getShape(), jdrfont.getSize(), string);

      textsegment.setTextPaint(foreground);

      textsegment.setLaTeXFont(ltxfont);

      textsegment.pgfHalign   = halign;
      textsegment.pgfValign   = valign;

      if (ltxText != null) textsegment.setLaTeXText(ltxText);

      // update bounds (not needed for JpgfDraw, as it
      // can call updateBounds, but required for jdr2ajr)
      BufferedImage buffImage = new BufferedImage(1,1,
         BufferedImage.TYPE_INT_RGB);

      Graphics2D g2 = buffImage.createGraphics();
      textsegment.updateBounds(g2);
      g2.dispose();

      return textsegment;
   }

}
