// File          : JDRText.java
// Date          : 1st February 2006
// Last Modified : 18th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing a single line text area.
 * A text area has the following attributes:
 * <ul>
 * <li> The text that should appear in the text area (no line breaks
 * allowed).
 * <li> The alternative text to be used when exporting to a LaTeX
 * file.
 * <li> The Java font to use to display on screen or when converting
 * to PNG.
 * <li> The LaTeX font changing commands used when exporting to
 * a LaTeX file.
 * <li> The vertical and horizontal alignment settings to be passed
 * to <code>\pgftext</code> when exporting to a LaTeX file.
 * <li> The transformation matrix to apply to the text.
 * </ul>
 */

public class JDRText extends JDRCompleteObject implements JDRTextual
{
   /**
    * Creates an empty text area at the origin.
    * The font is set to SansSerif family,
    * medium weight, upright shape and 10bp.
    * @see #JDRText(Point2D)
    */
   public JDRText()
   {
      try
      {
         init("SansSerif",
              JDRFont.SERIES_MEDIUM,
              JDRFont.SHAPE_UPRIGHT, 10);
      }
      catch (InvalidFormatException e)
      {
      }
   }

   /**
    * Creates an empty text area at the given location.
    * The font is set to SansSerif family,
    * medium weight, upright shape and 10bp.
    * @param p the location of the new text area
    * @see #JDRText()
    */
   public JDRText(Point2D p)
   {
      try
      {
         init("SansSerif",
              JDRFont.SERIES_MEDIUM,
              JDRFont.SHAPE_UPRIGHT,10);
      }
      catch (Exception excp)
      {
      }
      setPosition(p.getX(),p.getY());
   }

   /**
    * Creates a new text area.
    * The font is set to SansSerif family,
    * medium weight, upright shape and 10bp.
    * The graphics device is needed to set up the bounds correctly.
    * @param g the graphics device
    * @param p the location of the new text area
    * @param str the text string to appear in the text area
    */
   public JDRText(Graphics g, Point2D p, String str)
   {
      try
      {
         init(g, "SansSerif",
              JDRFont.SERIES_MEDIUM,
              JDRFont.SHAPE_UPRIGHT,10,str);
      }
      catch (InvalidFormatException excp)
      {
      }
      setPosition(p.getX(),p.getY());
   }

   /**
    * Creates a new text area.
    * The font is set to SansSerif family,
    * medium weight, upright shape and 10bp.
    * The graphics device is needed to set up the bounds correctly.
    * @param g the graphics device
    * @param str the text string to appear in the text area
    */
   public JDRText(Graphics g, String str)
   {
      try
      {
         init(g, "SansSerif",
              JDRFont.SERIES_MEDIUM,
              JDRFont.SHAPE_UPRIGHT,10,str);
      }
      catch (InvalidFormatException excp)
      {
      }
   }

   /**
    * Creates a new text area at the origin.
    * The graphics device is needed to set up the bounds correctly.
    * @param g the graphics device
    * @param family the Java font family name
    * @param series the font series
    * @param shape the font shape
    * @param size the font size (in PostScript points)
    * @param str the text string to appear in the text area
    * @throws InvalidFontWeightException if the font weight is not one
    * of: {@link JDRFont#SERIES_MEDIUM} or {@link JDRFont#SERIES_BOLD}
    * @throws InvalidFontShapeException if the font shape is not one 
    * of:  {@link JDRFont#SHAPE_UPRIGHT}, {@link JDRFont#SHAPE_EM},
    * {@link JDRFont#SHAPE_ITALIC}, {@link JDRFont#SHAPE_SLANTED} or
    * {@link JDRFont#SHAPE_SC}
    * @throws InvalidFontSizeException if the font size is negative
    */
   public JDRText(Graphics g, String family,
               int series, int shape, int size, String str)
      throws InvalidFontWeightException,
             InvalidFontShapeException,
             InvalidFontSizeException
   {
      init(g, family, series, shape,size,str);
      setPosition(0, 0);
   }

   /**
    * Creates a new text area at the origin.
    * The graphics device is needed to set up the bounds correctly.
    * @param g the graphics device
    * @param javaFont the font 
    * @param str the text string to appear in the text area
   */
   public JDRText(Graphics g, Font javaFont, String str)
   {
      String family = javaFont.getName();
      int series = (javaFont.isBold() ?
                    JDRFont.SERIES_BOLD :
                    JDRFont.SERIES_MEDIUM);
      int shape = (javaFont.isItalic() ?
                   JDRFont.SHAPE_EM :
                   JDRFont.SHAPE_UPRIGHT);
      int size = javaFont.getSize();

      try
      {
         init(g, family, series, shape,size,str);
      }
      catch (InvalidFontShapeException e)
      {
      }
      catch (InvalidFontWeightException e)
      {
      }
      catch (InvalidFontSizeException e)
      {
      }
      setPosition(0, 0);
   }

   /**
    * Creates a new text area at the given location.
    * The graphics device is needed to set up the bounds correctly.
    * @param g the graphics device
    * @param p the location of the new text area
    * @param family the Java font family name
    * @param series the font series
    * @param shape the font shape
    * @param size the font size (in PostScript points)
    * @param str the text string to appear in the text area
    * @throws InvalidFontWeightException if the font weight is not one
    * of: {@link JDRFont#SERIES_MEDIUM} or {@link JDRFont#SERIES_BOLD}
    * @throws InvalidFontShapeException if the font shape is not one 
    * of:  {@link JDRFont#SHAPE_UPRIGHT}, {@link JDRFont#SHAPE_EM},
    * {@link JDRFont#SHAPE_ITALIC}, {@link JDRFont#SHAPE_SLANTED} or
    * {@link JDRFont#SHAPE_SC}
    * @throws InvalidFontSizeException if the font size is negative
    */
   public JDRText(Graphics g, Point2D p, String family,
               int series, int shape, int size, String str)
      throws InvalidFontWeightException,
             InvalidFontShapeException,
             InvalidFontSizeException
   {
      init(g, family, series, shape,size,str);
      setPosition(p.getX(),p.getY());
   }

   /**
    * Creates a new text area with the given transformation.
    * (The text area bounds should already be set by the
    * transformation.)
    * @param trans the transformation to apply to this text area
    * @param family the Java font family name
    * @param series the font series
    * @param shape the font shape
    * @param size the font size (in PostScript points)
    * @param str the text string to appear in the text area
    * @throws InvalidFontWeightException if the font weight is not one
    * of: {@link JDRFont#SERIES_MEDIUM} or {@link JDRFont#SERIES_BOLD}
    * @throws InvalidFontShapeException if the font shape is not one 
    * of:  {@link JDRFont#SHAPE_UPRIGHT}, {@link JDRFont#SHAPE_EM},
    * {@link JDRFont#SHAPE_ITALIC}, {@link JDRFont#SHAPE_SLANTED} or
    * {@link JDRFont#SHAPE_SC}
    * @throws InvalidFontSizeException if the font size is negative
    */
   public JDRText(JDRTransform trans, String family,
      int series, int shape, int size, String str)
      throws InvalidFontWeightException,
             InvalidFontShapeException,
             InvalidFontSizeException
   {
      jdrtransform = (JDRTransform)trans.clone();
      init(family, series, shape,size,str);
      setFont(family, series, shape, size);
   }

   private void init(String family,
                    int series,
                    int shape,
                    int size)
      throws InvalidFontWeightException,
             InvalidFontShapeException,
             InvalidFontSizeException
   {
      jdrtransform = new JDRTransform();
      init(family,series,shape,size,"");
      setFont(family, series, shape, size);
   }

   private void init(Graphics g,
                    String family,
                    int series,
                    int shape,
                    int size,
                    String str)
      throws InvalidFontWeightException,
             InvalidFontShapeException,
             InvalidFontSizeException
   {
      jdrtransform = new JDRTransform();
      init(family,series,shape,size,str);
      setFont((Graphics2D)g, family, series, shape, size);
   }

   private void init(String family,
                    int series,
                    int shape,
                    int size,
                    String str)
   {
      //start = new JDRPoint(0,0);
      //end   = new JDRPoint(0,0);

      text = str.replaceAll("[\t\r\n]", " ");

      latexFont   = new LaTeXFont();
      latexText   = text;

      pgfValign = PGF_VALIGN_BASE;
      pgfHalign = PGF_HALIGN_LEFT;

      try
      {
         setTextPaint(new JDRColor(0,0,0));
      }
      catch (InvalidFormatException e)
      {
      }
   }

   public void setTextPaint(JDRPaint paint)
   {
      textPaint = paint;
   }

   public JDRPaint getTextPaint()
   {
      return textPaint;
   }

   public void fade(double value)
   {
      textPaint.fade(value);
   }

   /**
    * Resets the transformation.
    */
   public void reset()
   {
      jdrtransform.reset();
   }

   /**
    * Gets the width of this text area.
    * @param g graphics device
    * @return width of this text area
    */
   public double getWidth(Graphics g)
   {
      updateBounds(g);
      return getBBox().getWidth();
   }

   /**
    * Updates the bounding box of this text area.
    * @param g graphics device
    */
   public void updateBounds(Graphics g)
   {
      if (g == null)
      {
         return;
      }

      Graphics2D g2 = (Graphics2D)g;

      if (text.equals(""))
      {
         FontMetrics fm = g2.getFontMetrics(font);
         double h = fm.getHeight();
         double d = fm.getDescent();

         jdrtransform.updateOriginalBounds(new BBox(0,d-h,0,d));
      }
      else
      {
         String str = text;

         if (str.startsWith(" "))
         {
            str = "_"+str.substring(1);
         }

         if (str.endsWith(" "))
         {
            str = str.substring(0, str.length()-1)+"_";
         }

         FontRenderContext frc = g2.getFontRenderContext();
         TextLayout layout = new TextLayout(str, font, frc);

         jdrtransform.updateOriginalBounds(layout.getBounds());
      }
   }

   /**
    * Sets the font for this text area.
    * @param g graphics device
    * @param name the Java font family name
    * @param series the font series
    * @param shape the font shape
    * @param size the font size (in PostScript points)
    * @throws InvalidFontWeightException if the font weight is not one
    * of: {@link JDRFont#SERIES_MEDIUM} or {@link JDRFont#SERIES_BOLD}
    * @throws InvalidFontShapeException if the font shape is not one 
    * of:  {@link JDRFont#SHAPE_UPRIGHT}, {@link JDRFont#SHAPE_EM},
    * {@link JDRFont#SHAPE_ITALIC}, {@link JDRFont#SHAPE_SLANTED} or
    * {@link JDRFont#SHAPE_SC}
    * @throws InvalidFontSizeException if the font size is negative
    */
   public void setFont(Graphics g, String name, int series, 
                       int shape, int size)
      throws InvalidFontWeightException,
             InvalidFontShapeException,
             InvalidFontSizeException
   {
      setFont(name, series, shape, size);
      updateBounds(g);
   }

   private void setFont(String name, int series, 
                       int shape, int size)
      throws InvalidFontWeightException,
             InvalidFontShapeException,
             InvalidFontSizeException
   {
      jdrFont.setFamily(name);
      jdrFont.setWeight(series);
      jdrFont.setShape(shape);
      jdrFont.setSize(size);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
   }

   /**
    * Sets the font family for this text area.
    * @param g graphics device
    * @param name the Java font family name
    */
   public void setFontFamily(Graphics g, String name)
   {
      jdrFont.setFamily(name);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
      updateBounds(g);
   }

   /**
    * Sets the font series for this text area.
    * @param g graphics device
    * @param series the font series
    * @throws InvalidFontWeightException if the font weight is not one
    * of: {@link JDRFont#SERIES_MEDIUM} or {@link JDRFont#SERIES_BOLD}
    */
   public void setFontSeries(Graphics g, int series)
      throws InvalidFontWeightException
   {
      jdrFont.setWeight(series);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
      updateBounds(g);
   }

   /**
    * Sets the font shape for this text area.
    * @param g graphics device
    * @param shape the font shape
    * @throws InvalidFontShapeException if the font shape is not one 
    * of:  {@link JDRFont#SHAPE_UPRIGHT}, {@link JDRFont#SHAPE_EM},
    * {@link JDRFont#SHAPE_ITALIC}, {@link JDRFont#SHAPE_SLANTED} or
    * {@link JDRFont#SHAPE_SC}
    */
   public void setFontShape(Graphics g, int shape)
      throws InvalidFontShapeException
   {
      jdrFont.setShape(shape);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
      updateBounds(g);
   }

   /**
    * Sets the font for this text area.
    * @param g graphics device
    * @param size the font size (in PostScript points)
    * @throws InvalidFontSizeException if the font size is negative
    */
   public void setFontSize(Graphics g, int size)
      throws InvalidFontSizeException
   {
      jdrFont.setSize(size);
      font = new Font(jdrFont.getFamily(), getFontWeight(),
         jdrFont.getSize());
      updateBounds(g);
   }

   /**
    * Gets the name of the Java font family used by this text area.
    * @return font family name
    */
   public String getFontFamily()
   {
      return jdrFont.getFamily();
   }

   /**
    * Gets the font series for this text area.
    * @return font series
    */
   public int getFontSeries()
   {
      return jdrFont.getWeight();
   }

   /**
    * Gets the font shape for this text area.
    * @return font shape
    */
   public int getFontShape()
   {
      return jdrFont.getShape();
   }

   /**
    * Gets the Java font weight (required by {@link Font}).
    * @return {@link Font} weight
    */
   private int getFontWeight()
   {
      int weight = 0;
      weight += (getFontSeries() == JDRFont.SERIES_MEDIUM?
                 Font.PLAIN : Font.BOLD);
      weight += (getFontShape() == JDRFont.SHAPE_UPRIGHT?
                 0 : Font.ITALIC);

      return weight;
   }

   /**
    * Gets the font size for this text field.
    * @return font size (in PostScript points)
    */
   public int getFontSize()
   {
      return jdrFont.getSize();
   }

   /**
    * Gets the Java font used to display this text area.
    * @return font
    */
   public Font getFont()
   {
      return font;
   }

   /**
    * Gets the JDR font associated with this text area.
    * @return font
    */
   public JDRFont getJDRFont()
   {
      return jdrFont;
   }

   /**
    * Splits this text area into a group of text areas each 
    * consisting of a single character from this text area.
    * @param g2 graphics device
    * @return group containing new text areas
    */
   public JDRGroup split(Graphics2D g2) 
   {
      FontRenderContext frc = g2.getFontRenderContext();

      BBox bbox = getBBox();

      JDRGroup group = new JDRGroup();

      int n = text.length();

      for (int i = 0; i < n; i++)
      {
         char c = text.charAt(i);
         if (c == ' ') continue;

         FontMetrics fm = g2.getFontMetrics(font);
      
         Rectangle2D rect = fm.getStringBounds(text,0,i+1,g2);
         int cw = fm.charWidth(c);

         Point2D p = new Point2D.Double(rect.getWidth()-cw,0);

         JDRText newText = new JDRText(g2, p, ""+c);
         newText.setTextPaint(getTextPaint());
         newText.jdrFont.makeEqual(jdrFont);
         newText.font = getFont();
         newText.latexFont.makeEqual(latexFont);
         newText.pgfValign = pgfValign;
         newText.pgfHalign = pgfHalign;
         newText.transform(jdrtransform);
         newText.updateBounds(g2);

         group.add(newText);
      }

      group.setSelected(isSelected());

      return group;
   }

   /**
    * Gets the outline of this text area.
    * @param frc the font render context
    * @return shape describing the outline of this text area
    */
   public Shape getOutline(FontRenderContext frc)
   {
      TextLayout tl = new TextLayout(text, font, frc);

      return tl.getOutline(jdrtransform.getAffineTransform());
   }

   /**
    * Converts this text area into a group containing paths
    * that approximate the outline of the text.
    * @param g2 graphics device
    * @return group containing new paths
    */
   public JDRGroup convertToPath(Graphics2D g2) 
      throws MissingMoveException,EmptyGroupException
   {
      return convertToPath(g2, new JDRBasicStroke());
   }

   public JDRGroup convertToPath(Graphics2D g2, JDRStroke stroke) 
      throws MissingMoveException,EmptyGroupException
   {
      FontRenderContext frc = g2.getFontRenderContext();

      BBox bbox = getBBox();

      JDRGroup group = new JDRGroup();

      int n = text.length();

      JDRPaint paint = getTextPaint();

      for (int i = 0; i < n; i++)
      {
         char c = text.charAt(i);
         if (c == ' ') continue;

         FontMetrics fm = g2.getFontMetrics(font);
      
         Rectangle2D rect = fm.getStringBounds(text,0,i+1,g2);
         int cw = fm.charWidth(c);

         AffineTransform af = AffineTransform.getTranslateInstance(
                             rect.getWidth()-cw, 0);

         TextLayout tl = new TextLayout(""+c, font, frc);
         Shape outline = tl.getOutline(af);

         PathIterator pi = outline.getPathIterator(null);

         JDRPath path = null;

         double[] coords = new double[6];
         double oldX=0, oldY=0;

         boolean startFlag=true;

         while (!pi.isDone())
         {
            int type = pi.currentSegment(coords);

            switch (type)
            {
               case PathIterator.SEG_MOVETO :
                  if (startFlag)
                  {
                     startFlag = false;
                     oldX = coords[0];
                     oldY = coords[1];
                     path = new JDRPath(new JDRTransparent(), paint, (JDRStroke)stroke.clone());
                  }
                  else
                  {
                     JDRSegment segment = new JDRSegment(oldX, oldY,
                                                   coords[0],coords[1]);
                     oldX = coords[0];
                     oldY = coords[1];
                     if (path == null)
                     {
                        throw new MissingMoveException();
                     }
                     path.add(segment);
                  }
               break;
               case PathIterator.SEG_LINETO :
                  JDRLine line = new JDRLine(oldX, oldY, coords[0],coords[1]);
                  oldX = coords[0];
                  oldY = coords[1];
                  path.add(line);
                  if (path == null)
                  {
                     throw new MissingMoveException();
                  }
               break;
               case PathIterator.SEG_QUADTO :
                  JDRBezier curve = JDRBezier.quadToCubic(oldX, oldY, 
                                            coords[0],coords[1],
                                            coords[2],coords[3]);
                  oldX = coords[2];
                  oldY = coords[3];
                  path.add(curve);
                  if (path == null)
                  {
                     throw new MissingMoveException();
                  }
               break;
               case PathIterator.SEG_CUBICTO :
                  curve = new JDRBezier(oldX, oldY, 
                                     coords[0],coords[1],
                                     coords[2],coords[3],
                                     coords[4],coords[5]);
                  oldX = coords[4];
                  oldY = coords[5];
                  path.add(curve);
                  if (path == null)
                  {
                     throw new MissingMoveException();
                  }
               break;
            }

            pi.next();
         }

         if (path != null) group.add(path);
      }

      if (group.size() == 0)
      {
         throw new EmptyGroupException();
      }

      group.setSelected(isSelected());

      jdrtransform.transform(group);
      
/*
      BBox grpBox = group.getBBox();
      double shiftx = bbox.getMinX()-grpBox.getMinX();
      double shifty = bbox.getMinY()-grpBox.getMinY();
      group.translate(-shiftx, -shifty);
*/

      return group;
   }

   /**
    * Preconcatenates a transformation with this text areas
    * transformation.
    * @param trans affine transformation to preconcatenate
    */
   public void preConcatenate(AffineTransform trans)
   {
      jdrtransform.preConcatenate(trans);
   }

   /**
    * Transforms this text area by concatenation a transformation
    * with this text area's transformation matrix.
    */
   public void transform(double[] matrix)
   {
      jdrtransform.concat(matrix);
   }

   /**
    * Transforms this text area by concatenation a transformation
    * with this text area's transformation matrix.
    * @param jdrt transformation to apply
    */
   public void transform(JDRTransform jdrt)
   {
      jdrtransform.concat(jdrt);
   }

   /**
    * Transforms this text area by concatenation a transformation
    * with this text area's transformation matrix.
    * @param af transformation to apply
    */
   public void transform(AffineTransform af)
   {
      jdrtransform.concat(af);
   }

   public void rotate(double angle)
   {
      jdrtransform.rotate(angle);
   }

   public void rotate(Point2D p, double angle)
   {
      jdrtransform.rotate(p, angle);
   }

   public void scaleX(double factor)
   {
      scale(factor, 1.0);
   }

   public void scaleX(Point2D p, double factor)
   {
      scale(p,factor,1.0);
   }

   public void scaleY(double factor)
   {
      scale(1.0, factor);
   }

   public void scaleY(Point2D p, double factor)
   {
      scale(p,1.0,factor);
   }

   public void scale(double factorX, double factorY)
   {
      jdrtransform.scale(factorX, factorY);
   }

   public void scale(Point2D p, double factorX, double factorY)
   {
      jdrtransform.scale(p, factorX, factorY);
   }

   public void shearX(double factor)
   {
      shear(factor, 0.0);
   }

   public void shearX(Point2D p, double factor)
   {
      shear(p,factor,0.0);
   }

   public void shearY(double factor)
   {
      shear(0.0, factor);
   }

   public void shearY(Point2D p, double factor)
   {
      shear(p,0.0,factor);
   }

   public void shear(Point2D p, double factor)
   {
      shear(p,factor,factor);
   }

   public void shear(double factor)
   {
      shear(factor, factor);
   }

   public void shear(double factorX, double factorY)
   {
      jdrtransform.shear(factorX, factorY);
   }

   public void shear(Point2D p, double factorX, double factorY)
   {
      jdrtransform.shear(p, factorX, factorY);
   }

   public void translate(double x, double y)
   {
      jdrtransform.translate(x, y);
   }

   /**
    * Sets this text area's position.
    * @param x the x co-ordinate
    * @param y the y co-ordinate
    */
   public void setPosition(double x, double y)
   {
      jdrtransform.setPosition(x, y);
   }

   public BBox getBBox()
   {
      return jdrtransform.getBBox();
   }

   /**
    * Gets the starting position of this text area.
    * This is given by {@link JDRTransform#getAnchor()}.
    * @return starting position
    */
   public JDRPoint getStart()
   {
      return jdrtransform.getAnchor();
   }

   /**
    * Gets the centre point.
    * This is given by {@link JDRTransform#getCentre()}
    * @return the centre of the text area
    */
   public JDRPoint getCentre()
   {
      return jdrtransform.getCentre();
   }

   /**
    * Sets the text to appear in this text area. The LaTeX alternative
    * is set to the same text.
    * @param g graphics device
    * @param str the new text
    * @see #setText(Graphics,String,String)
    */
   public void setText(Graphics g, String str)
   {
      text = str.replaceAll("[\t\r\n]", " ");
      latexText = text;

      updateBounds(g);
   }

   /**
    * Sets the text to appear in this text area and the alternative
    * LaTeX text.
    * @param g graphics device
    * @param str the new text
    * @param latexStr the alternative LaTeX text
    * @see #setText(Graphics,String)
    */
   public void setText(Graphics g, String str, String latexStr)
   {
      setText(str, latexStr);
      updateBounds(g);
   }

   /**
    * Sets the text to appear in this text area and the alternative
    * LaTeX text. (Doesn't update bounds.)
    * @param str the new text
    * @param latexStr the alternative LaTeX text
    * @see #setText(Graphics,String,String)
    */
   public void setText(String str, String latexStr)
   {
      text = str.replaceAll("[\t\r\n]", " ");

      if (latexStr == null)
      {
         latexText = text;
      }
      else
      {
         latexText = latexStr.replaceAll("[\t\r\n]", " ");
      }
   }

   /**
    * Gets the text to appear in this text area.
    * @return the text to appear in this text area
    */
   public String getText()
   {
      return text;
   }

   /**
    * Sets the alternative LaTeX text.
    * @param str the alternative LaTeX text
    * @see #setText(Graphics,String,String)
    */
   public void setLaTeXText(String str)
   {
      latexText = str.replaceAll("[\t\r\n]", " ");
   }

   /**
    * Gets the alternative LaTeX text.
    * @return alternative LaTeX text
    */
   public String getLaTeXText()
   {
      return latexText;
   }

   public void draw(Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;

      // this code doesn't work with Java6
/*
      AffineTransform af = jdrtransform.getAffineTransform();

      AffineTransform oldAf = g2.getTransform();

      g2.setTransform(new AffineTransform());

      af.preConcatenate(oldAf);

      Font f = font.deriveFont(af);

      FontRenderContext frc = g2.getFontRenderContext();
      TextLayout layout = new TextLayout(text, f, frc);

      if (getTextPaint() instanceof JDRShading)
      {
         Rectangle2D bounds = layout.getBounds();

         BBox box = new BBox(bounds);

         g2.setPaint(getTextPaint().getPaint(box));
      }

      layout.draw(g2, 0.0f, 0.0f);
      g2.setTransform(oldAf);
*/
      
      FontRenderContext frc = g2.getFontRenderContext();
      TextLayout layout = new TextLayout(text, font, frc);
      AffineTransform af = jdrtransform.getAffineTransform();
      Shape outline = layout.getOutline(af);

      BBox box = null;

      if (getTextPaint() instanceof JDRShading)
      {
         Rectangle2D bounds = outline.getBounds();

         box = new BBox(bounds);
      }

      g2.setPaint(getTextPaint().getPaint(box));
      g2.fill(outline);
   }

   /**
    * Draws text area with the PGF anchors.
    * @param g the graphics device
    * @param anchorPaint the anchor paint
    */
   public void drawWithAnchors(Graphics g, Paint anchorPaint)
   {
      Graphics2D g2 = (Graphics2D)g;

      AffineTransform af = jdrtransform.getAffineTransform();

      AffineTransform oldAf = g2.getTransform();

      g2.setTransform(new AffineTransform());

      FontRenderContext frc = g2.getFontRenderContext();
      TextLayout layout = new TextLayout(text, font, frc);

      af.preConcatenate(oldAf);
      g2.setTransform(af);

      if (getTextPaint() instanceof JDRShading)
      {
         Rectangle2D bounds = layout.getBounds();

         BBox box = new BBox(bounds);

         g2.setPaint(getTextPaint().getPaint(box));
      }
      else
      {
         g2.setPaint(getTextPaint().getColor());
      }

      layout.draw(g2, 0.0f, 0.0f);

      g2.setPaint(anchorPaint);

      Point2D p = getPGFAnchor();

      double radius = 1;

      g2.fill(new Ellipse2D.Double(p.getX()-radius, p.getY()-radius,
              2*radius, 2*radius));

      g2.setTransform(oldAf);
   }

   public void fill(Graphics g)
   {
   }

   public Object clone()
   {
      JDRText dt = null;

      try
      {
         dt = new JDRText(jdrtransform,
                            getFontFamily(), getFontSeries(),
                            getFontShape(), getFontSize(), text);

         dt.makeEqual(this);

      }
      catch (InvalidFormatException e)
      {
      }
      return dt;
   }

   public boolean equals(Object obj)
   {
      if (!super.equals(obj)) return false;

      if (!(obj instanceof JDRText)) return false;

      JDRText textObj = (JDRText)obj;

      if (!getTextPaint().equals(textObj.getTextPaint())) return false;

      if (!jdrFont.equals(textObj.jdrFont)) return false;

      if (!text.equals(textObj.getText())) return false;

      if (latexText == null || textObj.latexText == null)
      {
         if (latexText != textObj.latexText) return false;
      }
      else
      {
         if (!latexText.equals(textObj.latexText)) return false;
      }

      if (!jdrtransform.equals(textObj.jdrtransform)) return false;

      if (!latexFont.equals(textObj.latexFont)) return false;

      if (pgfValign != textObj.pgfValign) return false;

      if (pgfHalign != textObj.pgfHalign) return false;

      return true;
   }

   /**
    * Makes this text area equivalent to another text area.
    * @param t the other text area
    */
   public void makeEqual(JDRText t)
   {
      super.makeEqual(t);

      setTextPaint((JDRPaint)t.getTextPaint().clone());

      text = t.getText();
      jdrFont.makeEqual(t.jdrFont);
      font = t.getFont();
      jdrtransform.makeEqual(t.getTransform());
      latexFont.makeEqual(t.latexFont);
      pgfValign = t.pgfValign;
      pgfHalign = t.pgfHalign;
      latexText = t.latexText;
   }

   /**
    * Gets the transformation to apply to this text area.
    * @return text area's transformation
    */
   public JDRTransform getTransform()
   {
      return jdrtransform;
   }

   /**
    * Gets this text area's transformation matrix.
    * @param matrix flat matrix in which to store transformation
    * or null if new array should be created
    * @return transformation matrix
    */
   public double[] getTransformation(double[] matrix)
   {
      if (matrix == null)
      {
         matrix = new double[6];
      }

      jdrtransform.getTransformation(matrix);

      return matrix;
   }

   /**
    * Sets this text area's transformation matrix.
    * @param matrix flat matrix storing transformation
    */
   public void setTransformation(double[] matrix)
   {
      jdrtransform.setTransformation(matrix);
   }

   /**
    * Sets the LaTeX font family declaration for this text area.
    * @param family LaTeX font family declaration
    * @see LaTeXFont#setFamily(String)
    */
   public void setLaTeXFamily(String family)
   {
      latexFont.setFamily(family);
   }

   /**
    * Sets the LaTeX font size declaration for this text area.
    * @param size LaTeX font size declaration
    * @see LaTeXFont#setSize(String)
    */
   public void setLaTeXSize(String size)
   {
      latexFont.setSize(size);
   }

   /**
    * Sets the LaTeX font series declaration for this text area.
    * @param series LaTeX font series declaration
    * @see LaTeXFont#setWeight(String)
    */
   public void setLaTeXSeries(String series)
   {
      latexFont.setWeight(series);
   }

   /**
    * Sets the LaTeX font shape declaration for this text area.
    * @param shape LaTeX font shape declaration
    * @see LaTeXFont#setShape(String)
    */
   public void setLaTeXShape(String shape)
   {
      latexFont.setShape(shape);
   }

   /**
    * Sets the LaTeX font declarations for this text area.
    * @param family LaTeX font family declaration
    * @param shape LaTeX font shape declaration
    * @param size LaTeX font size declaration
    * @param series LaTeX font series declaration
    * @see #setLaTeXFont(LaTeXFont)
    */
   public void setLaTeXFont(String family, String size, String series,
                        String shape)
   {
      latexFont.setFamily(family);
      latexFont.setSize(size);
      latexFont.setWeight(series);
      latexFont.setShape(shape);
   }

   /**
    * Sets this text area's LaTeX font.
    * @param ltxFont LaTeX font
    * @see #setLaTeXFont(String,String,String,String)
    */
   public void setLaTeXFont(LaTeXFont ltxFont)
   {
      latexFont = ltxFont;
   }

   /**
    * Sets the vertical alignment for <code>\pgftext</code>.
    * @param valign the vertical alignment
    * @throws InvalidVAlignException if valign is not one of:
    * {@link #PGF_VALIGN_TOP}, {@link #PGF_VALIGN_CENTRE},
    * {@link #PGF_VALIGN_BASE} or {@link #PGF_VALIGN_BOTTOM}
    */
   public void setVAlign(int valign)
      throws InvalidVAlignException
   {
      if (valign < 0 || valign > 3)
      {
         throw new InvalidVAlignException(valign);
      }

      pgfValign = valign;
   }

   /**
    * Sets the horizontal alignment for <code>\pgftext</code>.
    * @param halign the horizontal alignment
    * @throws InvalidHAlignException if halign is not one of:
    * {@link #PGF_HALIGN_LEFT}, {@link #PGF_HALIGN_CENTRE} or
    * {@link #PGF_HALIGN_RIGHT}
    */
   public void setHAlign(int halign)
      throws InvalidHAlignException
   {
      if (halign < 0 || halign > 2)
      {
         throw new InvalidHAlignException(halign);
      }

      pgfHalign = halign;
   }

   public void setAlign(int halign, int valign)
      throws InvalidHAlignException,InvalidVAlignException
   {
      setHAlign(halign);
      setVAlign(valign);
   }

   /**
    * Gets the horizontal alignment for <code>\pgftext</code>.
    * @return horizontal alignment
    */
   public int getHAlign()
   {
      return pgfHalign;
   }

   /**
    * Gets the vertical alignment for <code>\pgftext</code>.
    * @return vertical alignment
    */
   public int getVAlign()
   {
      return pgfValign;
   }

   /**
    * Gets the LaTeX font family declaration for this text area.
    * @return LaTeX font family declaration
    * @see LaTeXFont#getFamily()
    */
   public String getLaTeXFamily()
   {
      return latexFont.getFamily();
   }

   /**
    * Gets the LaTeX font series declaration for this text area.
    * @return LaTeX font series declaration
    * @see LaTeXFont#getWeight()
    */
   public String getLaTeXSeries()
   {
      return latexFont.getWeight();
   }

   /**
    * Gets the LaTeX font shape declaration for this text area.
    * @return LaTeX font shape declaration
    * @see LaTeXFont#getShape()
    */
   public String getLaTeXShape()
   {
      return latexFont.getShape();
   }

   /**
    * Gets the LaTeX font size declaration for this text area.
    * @return LaTeX font size declaration
    * @see LaTeXFont#getSize()
    */
   public String getLaTeXSize()
   {
      return latexFont.getSize();
   }

   /**
    * Gets the LaTeX font associated with this text area.
    * @return LaTeX font for this text area
    */
   public LaTeXFont getLaTeXFont()
   {
      return latexFont;
   }

   private Point2D getPGFAnchor()
   {
      double x = 0;
      double y = 0;

      BBox box = jdrtransform.getOriginalBBox();

      switch (pgfValign)
      {
         case PGF_VALIGN_TOP:
            y = box.getMinY();
         break;
         case PGF_VALIGN_CENTRE:
            y = box.getMidY();
         break;
         case PGF_VALIGN_BASE:
            y = 0;
         break;
         case PGF_VALIGN_BOTTOM:
            y = box.getMaxY();
         break;
      }

      switch (pgfHalign)
      {
         case PGF_HALIGN_LEFT:
            x = 0;
         break;
         case PGF_HALIGN_CENTRE:
            x = box.getMidX();
         break;
         case PGF_HALIGN_RIGHT:
            x = box.getMinX()+box.getMaxX();
         break;
      }

      return new Point2D.Double(x, y);
   }

   public String pgf(AffineTransform af)
   {
      String eol = System.getProperty("line.separator", "\n");

      String valign="";
      String halign="";

      double x = 0;
      double y = 0;

      BBox box = jdrtransform.getOriginalBBox();

      switch (pgfValign)
      {
         case PGF_VALIGN_TOP:
            valign= "top";
            y = box.getMaxY()-box.getMinY();
         break;
         case PGF_VALIGN_CENTRE:
            valign= "center";
            y = box.getMidY()-box.getMinY();
         break;
         case PGF_VALIGN_BASE:
            valign= "base";
            y = box.getMaxY();
         break;
         case PGF_VALIGN_BOTTOM:
            valign= "bottom";
            y = 0;
         break;
      }

      switch (pgfHalign)
      {
         case PGF_HALIGN_LEFT:
            halign= "left";
            x = 0;
         break;
         case PGF_HALIGN_CENTRE:
            halign= "center";
            x = box.getMidX();
         break;
         case PGF_HALIGN_RIGHT:
            halign= "right";
            x = box.getMinX()+box.getMaxX();
         break;
      }

      String str = "";

      if (!description.equals(""))
      {
         str += "% "+description+eol;
      }

      JDRPaint p = getTextPaint();

      if (p instanceof JDRTransparent)
      {
         str += "\\pgftext["+halign+","+valign+"]{"
              + latexFont.tex() + eol 
              + "\\phantom{"+ latexText + "}}";
      }
      else
      {
         str += "\\pgftext["+halign+","+valign+"]{"
              + latexFont.tex() + eol 
              + p.pgf(box) + latexText + "}";
      }

      return jdrtransform.pgf(af, x, y, str);
   }

   public void saveEPS(PrintWriter out)
      throws IOException
   {
      JDRPaint paint = getTextPaint();

      if (paint instanceof JDRTransparent)
      {
         return;
      }

      out.println("gsave");
      String psname = getFont().getPSName();

      // substitute generic names
      if (psname.equals("SansSerif.plain"))
      {
         psname = "Helvetica";
      }
      else if (psname.equals("SansSerif.bold"))
      {
         psname = "Helvetica-Bold";
      }
      else if (psname.equals("SansSerif.italic"))
      {
         psname = "Helvetica-Oblique";
      }
      else if (psname.equals("SansSerif.bolditalic"))
      {
         psname = "Helvetica-BoldOblique";
      }
      else if (psname.equals("Serif.plain"))
      {
         psname = "Times-Roman";
      }
      else if (psname.equals("Serif.bold"))
      {
         psname = "Times-Bold";
      }
      else if (psname.equals("Serif.italic"))
      {
         psname = "Times-Italic";
      }
      else if (psname.equals("Serif.bolditalic"))
      {
         psname = "Times-BoldItalic";
      }
      else if (psname.equals("Monospaced.plain"))
      {
         psname = "Courier";
      }
      else if (psname.equals("Monospaced.bold"))
      {
         psname = "Courier-Bold";
      }
      else if (psname.equals("Monospaced.italic"))
      {
         psname = "Courier-Oblique";
      }
      else if (psname.equals("Monospaced.bolditalic"))
      {
         psname = "Courier-BoldOblique";
      }

      out.println("/"+psname+" findfont");
      out.println(""+jdrFont.getSize()+" scalefont");
      out.println("setfont");
      jdrtransform.saveEPS(out);
      out.println("0 0 moveto");

      String psString = "";

      char[] charArray = text.toCharArray();

      for (int i = 0; i < charArray.length; i++)
      {
         if (charArray[i] == '\\')
         {
            psString += "\\\\";
         }
         else if (charArray[i] == '(')
         {
            psString += "\\(";
         }
         else if (charArray[i] == ')')
         {
            psString += "\\)";
         }
         else
         {
            psString += charArray[i];
         }
      }

      out.println("("+psString+")");

      if ((paint instanceof JDRGradient)
        || (paint instanceof JDRRadial))
      {
         out.println("true charpath");

         // need original bounding box
         BBox box = jdrtransform.getOriginalBBox();
         // flip
         double minY = -box.getMaxY();
         double maxY = -box.getMinY();
         double minX = box.getMinX();
         double maxX = box.getMaxX();
         box.reset(minX, minY, maxX, maxY);

         paint.saveEPS(out, box);

         out.println("clip shfill");
      }
      else
      {
         paint.saveEPS(out, null);
         out.println("show");
      }

      out.println("grestore");
   } 

   public void saveSVG(PrintWriter out)
      throws IOException
   {
      out.println("   <text x=\"0\" y=\"0\" ");
      out.println("   "+getTextPaint().svgFill());
      out.println("       "+jdrFont.svg()
                  +" "+jdrtransform.svg()+">");
      out.println(text);
      out.println("   </text>");
   }

   /**
    * Gets string representation of this text area.
    * @return string representation of this text area
    */
   public String toString()
   {
      return "JDRText:"+text+"@"+jdrtransform;
   }

   public JDRObjectLoaderListener getListener()
   {
      return textListener;
   }

   /**
    * If the contents of this text area contain TeX special 
    * characters, set the LaTeX equivalent with special characters 
    * replaced with control sequences.
    */
   public void escapeTeXChars()
   {
      boolean replaced=false;

      String newString="";

      for (int i = 0, n=text.length(); i < n; i++)
      {
         char c = text.charAt(i);

         switch (c)
         {
            case '\\' :
               newString += "\\textbackslash ";
               replaced = true;
            break;
            case '^' :
               newString += "\\textasciicirum ";
               replaced = true;
            break;
            case '~' :
               newString += "\\textasciitilde ";
               replaced = true;
            break;
            case '$' :
               newString += "\\$";
               replaced = true;
            break;
            case '&' :
               newString += "\\&";
               replaced = true;
            break;
            case '#' :
               newString += "\\#";
               replaced = true;
            break;
            case '%' :
               newString += "\\%";
               replaced = true;
            break;
            case '_' :
               newString += "\\_";
               replaced = true;
            break;
            case '{' :
               newString += "\\{";
               replaced = true;
            break;
            case '}' :
               newString += "\\}";
               replaced = true;
            break;
            default :
               newString += c;
         }
      }

      if (replaced)
      {
         latexText = newString;
      }
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "Text:"+eol;
      str += "contents: "+text+eol;
      str += "LaTeX equivalent: "+latexText+eol;
      str += "Paint: "+getTextPaint()+eol;
      str += "font: "+jdrFont.info()+eol;
      str += "LaTeX font: "+latexFont.info()+eol;
      str += "start: "+getStart()+eol;
      str += "pgfHalign: " +pgfHalign+eol;
      str += "pgfValign: " +pgfValign+eol;
      str += "pgfanchor: "+getPGFAnchor()+eol;
      str += "transformation: "+jdrtransform.info()+eol;
      str += "original bounds: "+jdrtransform.getOriginalBBox().info()+eol;

      return str+super.info();
   }

   public JDRTextual getTextual()
   {
      return this;
   }

   public boolean hasTextual()
   {
      return true;
   }

   public boolean hasShape()
   {
      return false;
   }

   public boolean hasSymmetricPath()
   {
      return false;
   }

   public JDRSymmetricPath getSymmetricPath()
   {
      return null;
   }

   public boolean hasPattern()
   {
      return false;
   }

   public JDRPattern getPattern()
   {
      return null;
   }
   public String[] getDescriptionInfo()
   {
      return new String[] {getText(), latexText};
   }

   private String text;
   protected JDRFont jdrFont = new JDRFont();
   protected Font font;

   private JDRTransform jdrtransform;

   // LaTeX stuff

   /**
    * Associated LaTeX font.
    */
   public LaTeXFont latexFont = new LaTeXFont();
   /**
    * Alternative LaTeX text.
    */
   public String latexText;

   /**
    * Vertical alignment for <code>\pgftext</code>.
    */
   protected int pgfValign;
   /**
    * Horizontal alignment for <code>\pgftext</code>.
    */
   protected int pgfHalign;

   /**
    * <code>\pgftext</code> horizontal alignment.
    */
   public static final int PGF_HALIGN_LEFT=0,
                           PGF_HALIGN_CENTRE=1,
                           PGF_HALIGN_RIGHT=2;
   /**
    * <code>\pgftext</code> vertical alignment.
    */
   public static final int PGF_VALIGN_TOP=0,
                           PGF_VALIGN_CENTRE=1,
                           PGF_VALIGN_BASE=2,
                           PGF_VALIGN_BOTTOM=3;

   private static JDRTextListener textListener = new JDRTextListener();

   private JDRPaint textPaint;
}
