// File          : JDRSymmetricPathListener.java
// Date          : 25th July 2010
// Last Modified : 25th July 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for symmetric paths. This is for JDR/AJR versions
 * 1.6 and above. If the version number is less than 1.6, saved as
 * a group of paths.
 * @author Nicola L C Talbot
 */

public class JDRSymmetricPathListener extends JDRPathListener
{
   public char getJDRid(float version)
   {
      return (version < 1.6 ? 'G' : 'S');
   }

   public char getAJRid(float version)
   {
      return (version < 1.6 ? 'G' : 'S');
   }

   public void writeJDR(JDRObject object,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException
   {
      JDRSymmetricPath path = (JDRSymmetricPath)object;

      JDRObjectLoader objectLoader = JDR.getObjectLoader();

      if (version < 1.6)
      {
         JDRGroup group = path.separate(null);

         JDRObjectLoaderListener listener = group.getListener();

         listener.writeJDR(group, dout, version);
      }
      else
      {
         objectLoader.saveJDR(path.getUnderlyingShape(), dout, version);

         dout.writeBoolean(path.isAnchored());

         JDRSegmentLoader loader = JDR.getSegmentLoader();

         if (!path.isAnchored())
         {
            // write join segment

            loader.saveJDR(path.getJoin(), dout, version);
         }

         JDRLine line = path.getSymmetry();
         dout.writeDouble(line.start.x);
         dout.writeDouble(line.start.y);
         dout.writeDouble(line.end.x);
         dout.writeDouble(line.end.y);

         dout.writeBoolean(path.isClosed());

         if (path.isClosed())
         {
            JDRPartialSegment segment = path.getClosingSegment();

            dout.writeBoolean(segment == null);

            if (segment != null)
            {
               loader.saveJDR(segment, dout, version);
            }
         }
      }
   }

   public JDRObject readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      JDRObjectLoader objectLoader = JDR.getObjectLoader();

      JDRObject object = objectLoader.loadJDR(din, version);

      if (!(object instanceof JDRShape))
      {
         throw new InvalidFormatException
            ("Shape specs expected for underlying symmetric path object.");
      }

      JDRSymmetricPath sPath = new JDRSymmetricPath((JDRShape)object);

      boolean anchored = din.readBoolean();

      JDRPartialSegment joinSegment = null;

      JDRSegmentLoader loader = JDR.getSegmentLoader();

      if (!anchored)
      {
         // read join information

         joinSegment = (JDRPartialSegment)loader.loadJDR(din, version);
      }

      double x0 = din.readDouble();
      double y0 = din.readDouble();
      double x1 = din.readDouble();
      double y1 = din.readDouble();

      sPath.setSymmetry(x0, y0, x1, y1);
      sPath.setJoin(joinSegment);

      boolean isClosed = din.readBoolean();

      if (isClosed)
      {
         boolean isClosedAnchored = din.readBoolean();

         if (!isClosedAnchored)
         {
            JDRPartialSegment segment 
               = (JDRPartialSegment)loader.loadJDR(din, version);

            sPath.close(segment);
         }
         else
         {
            sPath.close();
         }
      }

      return sPath;
   }

   public void writeAJR(JDRObject object,
                        PrintWriter out,
                        float version)
   throws IOException,InvalidFormatException
   {
      JDRSymmetricPath path = (JDRSymmetricPath)object;

      JDRObjectLoader objectLoader = AJR.getObjectLoader();

      if (version < 1.6)
      {
         JDRGroup group = path.separate(null);

         JDRObjectLoaderListener listener = group.getListener();

         listener.writeAJR(group, out, version);
      }
      else
      {
         objectLoader.saveAJR(path.getUnderlyingShape(), out, version);

         AJR.writeBoolean(out, path.isAnchored());

         JDRSegmentLoader loader = AJR.getSegmentLoader();

         if (!path.isAnchored())
         {
            // Write join information

            loader.saveAJR(path.getJoin(), out, version);
         }

         JDRLine line = path.getSymmetry();
         AJR.writeDouble(out, line.start.x);
         AJR.writeDouble(out, line.start.y);
         AJR.writeDouble(out, line.end.x);
         AJR.writeDouble(out, line.end.y);

         AJR.writeBoolean(out, path.isClosed());

         if (path.isClosed())
         {
            JDRPartialSegment segment = path.getClosingSegment();

            AJR.writeBoolean(out, segment == null);

            if (segment != null)
            {
               loader.saveAJR(segment, out, version);
            }
         }

         out.println();
      }
   }

   public JDRObject readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      JDRObjectLoader objectLoader = AJR.getObjectLoader();

      JDRObject object = objectLoader.loadAJR(in, version);

      if (!(object instanceof JDRShape))
      {
         throw new InvalidFormatException
            ("Shape specs expected for underlying text path object",
             AJR.getLineNum());
      }

      JDRSymmetricPath sPath = new JDRSymmetricPath((JDRShape)object);

      boolean anchored = AJR.readBoolean(in);

      JDRPartialSegment joinSegment = null;

      JDRSegmentLoader loader = AJR.getSegmentLoader();

      if (!anchored)
      {
         // read join information

         joinSegment = (JDRPartialSegment)loader.loadAJR(in, version);
      }

      double x0 = AJR.readDouble(in);
      double y0 = AJR.readDouble(in);
      double x1 = AJR.readDouble(in);
      double y1 = AJR.readDouble(in);

      sPath.setSymmetry(x0, y0, x1, y1);
      sPath.setJoin(joinSegment);

      boolean isClosed = AJR.readBoolean(in);

      if (isClosed)
      {
         boolean isCloseAnchored = AJR.readBoolean(in);

         if (!isCloseAnchored)
         {
            JDRPartialSegment segment 
               = (JDRPartialSegment)loader.loadAJR(in, version);

            sPath.close(segment);
         }
         else
         {
            sPath.close();
         }
      }

      return sPath;
   }
}
