// File          : JDRSymmetricPath.java
// Date          : 25th July 2010
// Last Modified : 22nd Aug 2010
// Author        : Nicola L. C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;

/**
 *  Class representing a symmetric path.
 *  @author Nicola L C Talbot
 */

public class JDRSymmetricPath extends JDRCompoundShape
{
   /**
    * Creates a symmetric path from the given path. This creates a
    * symmetric path from the given path. If endAnchor is true, the
    * final point in the path is anchored to the line of symmetry,
    * so the complete path will have an even number of segments. If
    * endAnchor is false, the final point isn't anchored to the line
    * of symmetry, so the complete path will have an odd number of
    * segments.
    * @param path the path
    * @param endAnchor if true anchor end point to the line of symmetry
    * @param line line of symmetry
    */
    public JDRSymmetricPath(JDRShape path, boolean endAnchor, JDRLine line)
    {
       path_ = path;

       selected = path_.isSelected();

       setSymmetry(line);

       join = (endAnchor ? null 
            : new JDRPartialSegment(path_.getLastControl(), line_));

       initIterators();

       if (path_.isClosed())
       {
          path_.open(false);
          try
          {
             close(CLOSE_MERGE_ENDS);
          }
          catch (EmptyPathException e)
          {
          }
       }
    }

    protected JDRSymmetricPath(JDRShape path, JDRPartialSegment joinSegment, JDRLine line)
    {
       path_ = path;

       selected = path_.isSelected();

       line_ = line;

       join = joinSegment;

       if (join != null)
       {
          join.setSymmetryLine(line_);
          join.setStart(path_.getLastControl());
       }

       initIterators();

       if (path_.isClosed())
       {
          path_.open(false);
          try
          {
             close(CLOSE_MERGE_ENDS);
          }
          catch (EmptyPathException e)
          {
          }
       }
    }

    public JDRSymmetricPath(JDRShape path)
    {
       path_ = path;

       selected = path_.isSelected();

       join = null;

       // Get the last point

       JDRPoint p = path_.getLastControl();

       // Construct vertical line through this point

       BBox box = path.getBBox();

       line_ = new JDRLine(new JDRSymmetryLinePoint(p.x, box.getMinY()),
                           new JDRSymmetryLinePoint(p.x, box.getMaxY()));

       initIterators();

       if (path_.isClosed())
       {
          path_.open(false);

          try
          {
             close(CLOSE_MERGE_ENDS);
          }
          catch (EmptyPathException e)
          {
          }
       }
    }

    /**
     * Creates an empty path with the given line of symmetry and
     * path attributes.
     * @param line the line of symmetry
     * @param capacity initial path capacity
     * @param linePaint the path outline paint
     * @param fillPaint the fill paint
     * @param stroke the path stroke
     */
    public JDRSymmetricPath(JDRLine line,
       int capacity, JDRPaint linePaint, JDRPaint fillPaint,
       JDRStroke stroke)
    {
       path_ = new JDRPath(capacity, linePaint, fillPaint, stroke);

       setSymmetry(line);

       join = null;

       initIterators();
    }

    /**
     * Creates an empty path with a line of symmetry passing through
     * (0,1) and (0,-1).
     */
    public JDRSymmetricPath()
    {
       path_ = new JDRPath();
       join  = null;
       line_ = new JDRLine(new JDRSymmetryLinePoint(0.0, 1.0),
                           new JDRSymmetryLinePoint(0.0, -1.0));

       initIterators();
    }

    public JDRSymmetricPath(int capacity, JDRPaint lineColor,
                            JDRPaint fillColor, JDRStroke s)
    {
       path_ = new JDRPath(capacity, lineColor, fillColor, s);
       join = null;

       initIterators();
    }

    protected void initIterators()
    {
       iterator = new JDRSymmetricPathIterator(this);
       pointIterator = new JDRPointIterator(this);
    }

    public JDRPathIterator getIterator()
    {
       iterator.reset();
       return iterator;
    }

    public JDRPointIterator getPointIterator()
    {
       pointIterator.reset();
       return pointIterator;
    }

    public JDRPathSegment getFirstSegment()
    {
       return path_.getFirstSegment();
    }

    public JDRPathSegment getLastSegment()
    {
       return line_;
    }

    public JDRPoint getFirstControl()
    {
       return path_.getFirstControl();
    }

    public JDRPoint getLastControl()
    {
       return line_.getEnd();
    }

    public JDRShape getUnderlyingShape()
    {
       return path_;
    }

    public void setUnderlyingShape(JDRShape shape)
    {
       path_ = shape;

       if (join == null)
       {
          moveToLine(path_.getLastSegment().getEnd());
       }
       else
       {
          join.setStart(path_.getLastSegment().getEnd());
       }

       if (isClosed())
       {
          if (closingSegment == null)
          {
             moveToLine(path_.getFirstControl());
          }
          else
          {
             closingSegment.setEnd(path_.getFirstControl());
          }
       }
    }

    public void add(JDRSegment s)
    {
       path_.add(s);
    }

    public JDRShape getFullPath()
      throws EmptyPathException,IllFittingPathException
    {
       return (JDRShape)getFullObject();
    }

    public JDRCompleteObject getFullObject()
      throws EmptyPathException,IllFittingPathException
    {
       int n = path_.size();

       JDRShape shape = (JDRShape)path_.getFullPath();

       // add in joining segment

       if (join != null)
       {
          shape.add(join.getFullSegment());
       }

       // add in reflected half

       for (int i = n-1; i >= 0; i--)
       {
          shape.add((JDRSegment)getReflected(i).reverse());
       }

       if (closingSegment != null)
       {
          shape.close(closingSegment.getFullSegment());
       }

       return shape;
    }

    public Object clone()
    {
       JDRSymmetricPath p 
          = new JDRSymmetricPath((JDRShape)path_.clone(), true, 
                                 (JDRLine)line_.clone());

       p.makeEqual(this);

       return p;
    }

    public void makeEqual(JDRObject object)
    {
       super.makeEqual(object);

       JDRSymmetricPath symPath = (JDRSymmetricPath)object;

       path_.makeEqual(symPath.path_);

       line_.makeEqual(symPath.line_);

       JDRPartialSegment newJoin;

       if (join == null)
       {
          newJoin = (symPath.join == null ? null
                   : (JDRPartialSegment)symPath.join.clone());
       }
       else if (symPath.join == null)
       {
          newJoin = null;
       }
       else
       {
          newJoin = (JDRPartialSegment)symPath.join.clone();
       }

       setJoin(newJoin);

       if (!symPath.isClosed())
       {
          open();
          return;
       }

       if (symPath.closingSegment != null)
       {
          try
          {
             close((JDRPathSegment)symPath.closingSegment.clone());
          }
          catch (InvalidPathException e)
          {
          }
       }
       else
       {
          try
          {
             close(CLOSE_MERGE_ENDS);
          }
          catch (EmptyPathException e)
          {
          }
       }
    }

    /**
     * Moves the given point to the nearest point on the line of
     * symmetry.
     */
    protected void moveToLine(JDRPoint p)
    {
       p.moveToLine(line_);
    }

    public void translateControl(JDRPathSegment segment, JDRPoint p, 
      double x, double y)
    {
       path_.translateControl(segment, p, x, y);

       // Is this control point anchored to the line of symmetry?
       // Or is the control point defining the line of symmetry?

       if (join == null)
       {
         JDRPoint lastPt = path_.getLastControl();

         if (line_ == segment)
         {
            p = lastPt;
         }

         if (p == lastPt)
         {
            // Move it to the nearest point on the line

            moveToLine(p);
         }
       }

       if (isClosed())
       {
          if (closingSegment == null)
          {
             JDRPoint firstPt = path_.getFirstControl();

             if (line_ == segment)
             {
                p = firstPt;
             }

             if (p == firstPt)
             {
                moveToLine(p);
             }
          }
          else if (segment == closingSegment &&
                   p == closingSegment.getStart())
          {
             Point2D reflected = p.getReflection(line_);

             path_.getFirstSegment().setStart(reflected);
          }
          else if (p == path_.getFirstControl())
          {
             Point2D reflected = p.getReflection(line_);

             closingSegment.setStart(reflected);
          }
          else if (segment == line_)
          {
             closingSegment.setEnd(getFirstSegment().getStart());
          }
       }
    }

   public void setCapacity(int capacity)
      throws IllegalArgumentException
   {
      path_.setCapacity(capacity);
   }

   public int getCapacity()
   {
      return path_.getCapacity();
   }

   public JDRPoint addPoint()
   {
      return path_.addPoint();
   }

   public void makeContinuous(boolean atStart)
   {
      int n = path_.size();

      JDRPathSegment selectedSegment = getSelectedSegment();
      JDRPoint selectedControl = getSelectedControl();
      int selectedSegmentIndex = getSelectedIndex();

      if (selectedSegmentIndex == 0 && isClosed() && atStart)
      {
         if (!(selectedSegment instanceof JDRBezier))
         {
            return;
         }

         JDRBezier curve = (JDRBezier)selectedSegment;

         Point2D dP;

         if (closingSegment == null)
         {
            // Make gradient perpendicular to line of symmetry

            dP = line_.getdP();

            dP.setLocation(dP.getY(), -dP.getX());
         }
         else
         {
            dP = closingSegment.getdP1();
         }

         curve.setStartGradient(dP);

         return;
      }

      if (selectedSegmentIndex < n-1)
      {
         path_.makeContinuous(atStart);
         return;
      }

      if (selectedSegment == null)
      {
         return;
      }

      if (selectedSegmentIndex == n-1)
      {
         if (!(selectedSegment instanceof JDRBezier))
         {
            // Not a Bezier curve
            return;
         }

         JDRBezier curve = (JDRBezier)selectedSegment;

         if (selectedControl == curve.getControl1())
         {
            path_.makeContinuous(atStart);
            return;
         }

         if (selectedControl != curve.getControl2())
         {
            return;
         }

         // The selected point is the second curvature control
         // on the last segment of the underlying path

         if (join == null)
         {
            // No joining segment, so make the gradient
            // perpendicular to the line of symmetry

            Point2D dP = line_.getdP();

            dP.setLocation(dP.getY(), -dP.getX());

            curve.setEndGradient(dP);

            return;
         }

         curve.setEndGradient(join.getdP0());

         return;
      }

      if (!(selectedSegment instanceof JDRPartialBezier))
      {
         return;
      }

      JDRPartialBezier curve = (JDRPartialBezier)selectedSegment;

      if (curve == join)
      {
         if (selectedControl != curve.getControl1())
         {
            return;
         }

         JDRSegment segment = (JDRSegment)path_.getLastSegment();

         Point2D dP = segment.getdP1();

         curve.setGradient(dP);

         return;
      }

      if (curve == closingSegment)
      {
         if (selectedControl != curve.getControl1())
         {
            return;
         }

         Point2D dP = getReflected(0).reverse().getdP1();

         curve.setGradient(dP);
      }
   }

   protected void stopEditing()
   {
      path_.stopEditing();
/*
      JDRPathSegment segment = getSelectedSegment();

      selectedControl = null;
      selectedSegmentIndex = -1;
      selectedControlIndex = -1;

      if (segment != null)
      {
         segment.setSelected(false);
      }
*/

      editMode = false;
   }

   public void setEditMode(boolean mode)
   {
      if (mode)
      {
         selectNextControl();
      }
      else
      {
         stopEditing();
      }
   }

   public int getSelectedControlIndex()
   {
      return path_.getSelectedControlIndex();
   }

   public JDRPathSegment removeSelectedSegment()
   {
      int selectedSegmentIndex = getSelectedIndex();

      if (selectedSegmentIndex < path_.size())
      {
         return remove(selectedSegmentIndex);
      }

      return null;
   }

   public JDRPathSegment remove(JDRPathSegment segment)
   {
      for (int i = 0, n = path_.size(); i < n; i++)
      {
         if (get(i) == segment)
         {
            return remove(i);
         }
      }

      return null;
   }

   public JDRSegment removeSegment(int index)
     throws ArrayIndexOutOfBoundsException
   {
      return (JDRSegment)path_.remove(index);
   }

   public JDRPathSegment setSegment(int index, JDRPathSegment segment)
     throws ArrayIndexOutOfBoundsException
   {
      int n = path_.size();

      if (index < n)
      {
         return path_.setSegment(index, segment);
      }

      int i = 0;

      if (join != null)
      {
         if (index == n)
         {
            JDRPathSegment oldJoin = join;
            setJoin((JDRPartialSegment)segment);

            return oldJoin;
         }

         i++;
      }

      if (closingSegment != null)
      {
         if (index == n+i)
         {
            JDRPathSegment oldSegment = closingSegment;
            closingSegment = (JDRPartialSegment)segment;

            closingSegment.setSymmetryLine(line_);

            return oldSegment;
         }
      }

      throw new ArrayIndexOutOfBoundsException(index);
   }

   public JDRSegment remove(int i)
   {
      JDRSegment segment = (JDRSegment)path_.get(i);
      JDRPoint dp = segment.getEnd();

      JDRPoint selectedControl = getSelectedControl();

      int index = getSelectedControlIndex();

      if (dp == selectedControl ||
          (segment instanceof JDRBezier
           && selectedControl == ((JDRBezier)segment).control2))
      {
         dp = segment.getStart();
      }

      if (i == 0)
      {
         if (isClosed())
         {
            closingSegment.setStart(dp);
         }
      }
      else
      {
         JDRSegment prev = (JDRSegment)path_.get(i-1);

         if (i == path_.size()-1 && isAnchored())
         {
            moveToLine(prev.getEnd());
         }
         else
         {
            prev.setEnd(dp);
         }
      }

      JDRSegment oldSegment = path_.removeSegment(i);

      stopEditing();

      dp = selectControl(index);

      if (dp == null)
      {
         selectControl(0);
      }

      return oldSegment;
   }

   public int getSelectedIndex()
   {
      return path_.getSelectedIndex();
   }

   public JDRPoint getSelectedControl()
   {
      return path_.getSelectedControl();
   }

   public JDRPathSegment getSelectedSegment()
   {
/*
      if (selectedSegmentIndex < 0) return null;

      return get(selectedSegmentIndex);
*/
      return path_.getSelectedSegment();
   }

   public boolean segmentHasEnd(JDRPathSegment segment)
   {
      if (segment instanceof JDRPartialSegment)
      {
         return false;
      }

      if (segment == line_) return true;

      if (isAnchored() && segment == path_.getLastSegment())
      {
         return true;
      }

      return false;
   }

   public int size()
   {
        return getTotalBaseSegments();
   }

   /**
    * Gets total number of base segments including the join segment (if
    * any), the line of symmetry and the closing segment (if any).
    */

   public int getTotalBaseSegments()
   {
      int n = path_.size()+1;

      if (join != null) n++;

      if (closingSegment != null) n++;

      return n;
   }

   /**
    * Gets total number of segments that make up the path. Doesn't
    * include the line of symmetry, does include the join segment
    * (if non-null), the reflected segments and the closing segment 
    * (if non-null).
    */
   public int getTotalPathSegments()
   {
      int n = 2*path_.size();

      if (join != null) n++;

      if (closingSegment != null) n++;

      return n;
   }

   /**
    * Gets the segment given by index.
    */
   public JDRPathSegment get(int index)
     throws ArrayIndexOutOfBoundsException
   {
      int n = path_.size();

      if (index < n) return path_.get(index);

      int i = 0;

      if (join != null)
      {
         if (index == n) return join;
         i++;
      }

      if (closingSegment != null)
      {
         if (index == n+i) return closingSegment;
         i++;
      }

      if (index == n+i) return line_;

      throw new ArrayIndexOutOfBoundsException(index);
   }

   public int getIndex(JDRPathSegment segment)
   {
      int index = 0;

      for (int n = path_.size(); index < n; index++)
      {
         JDRPathSegment s = path_.get(index);

         if (s == segment) return index;
      }

      if (join != null)
      {
         if (join == segment) return index;
         index++;
      }

      if (closingSegment != null)
      {
         if (closingSegment == segment) return index;
         index++;
      }

      if (line_ == segment) return index;

      return -1;
   }

   public int getLastIndex(JDRPathSegment segment)
   {
      int n = path_.size();

      int i = 0;

      if (join != null)
      {
         if (segment == join) return n;
         i++;
      }

      if (closingSegment != null)
      {
         if (segment == closingSegment) return n+i;
         i++;
      }

      if (segment == line_)
      {
         return n+i;
      }

      return path_.getLastIndex(segment);
   }

    /**
     * Gets the segment that's the reflection of the given segment.
     */
    public JDRPathSegment getReflected(int index)
    {
       return get(index).getReflection(line_);
    }

    public void draw(Graphics g)
    {
       if (showPath())
       {
          super.draw(g);
       }
       else
       {
          Graphics2D g2 = (Graphics2D)g;

          AffineTransform oldAf = g2.getTransform();

          path_.draw(g);

          AffineTransform af = line_.getReflectionTransform(null);

          af.preConcatenate(oldAf);

          g2.setTransform(af);

          path_.draw(g);

          g2.setTransform(oldAf);
       }
    }

    public void drawDraft(Graphics g, double scale)
    {
       Graphics2D g2 = (Graphics2D)g;

       path_.drawDraft(g, scale);

       if (join != null)
       {
          join.drawDraft(g, scale, false);
       }

       g2.setPaint(draftColor);

       AffineTransform af = AffineTransform.getScaleInstance(scale,scale);

       af.concatenate(line_.getReflectionTransform(null));

       GeneralPath p = path_.getGeneralPath();

       p.transform(af);

       g2.draw(p);

       if (closingSegment != null)
       {
          closingSegment.drawDraft(g, scale, false);
          closingSegment.start.draw(g, scale);
       }

       line_.drawDraft(g, scale, true);
    }

    public BBox getControlBBox()
    {
       BBox bbox = path_.getControlBBox();

       if (bbox == null) return null;

       for (int i = 0; i < path_.size(); i++)
       {
          JDRSegment segment = (JDRSegment)path_.get(i);
          segment.mergeReflectedBBox(line_, bbox);
       }

       if (join != null)
       {
          join.mergeControlBBox(bbox);
       }

       if (closingSegment != null)
       {
          closingSegment.mergeControlBBox(bbox);
       }

       line_.mergeControlBBox(bbox);

       return bbox;
    }

    public void mergeControlBBox(BBox box)
    {
       path_.mergeControlBBox(box);

       for (int i = 0; i < path_.size(); i++)
       {
          JDRSegment segment = (JDRSegment)path_.get(i);
          segment.mergeReflectedBBox(line_, box);
       }

       if (join != null)
       {
          join.mergeControlBBox(box);
       }

       if (closingSegment != null)
       {
          closingSegment.mergeControlBBox(box);
       }

       line_.mergeControlBBox(box);
    }


    public void convertSegment(int idx, JDRPathSegment segment)
    {
       JDRPathSegment orgSegment = get(idx);

       JDRPoint selectedControl = getSelectedControl();
       int selectedControlIndex = getSelectedControlIndex();

       if (idx < path_.size())
       {
          path_.convertSegment(idx, segment);

          if (join != null && orgSegment.getEnd() == join.getStart())
          {
             join.setStart(segment.getEnd());
          }

          return;
       }

       setSegment(idx, segment);

       try
       {
          int i = orgSegment.getControlIndex(selectedControl);

          selectedControl = segment.getStart();

          if (i > 0)
          {
             selectedControlIndex -= i;
          }

          setSelectedElements(idx, selectedControlIndex, segment, 
             selectedControl);
       }
       catch (NoSuchElementException e)
       {
       }
    }

    public void shearParams(Point2D p, double factorX, double factorY)
    {
       if (join != null)
       {
          join.shear(p, factorX, factorY);
       }

       line_.shear(p, factorX, factorY);

       if (closingSegment != null)
       {
          closingSegment.shear(p, factorX, factorY);

          closingSegment.setEnd(getFirstControl());
       }
    }

    public void scaleParams(Point2D p, double factorX, double factorY)
    {
       if (join != null)
       {
          join.scale(p, factorX, factorY);
       }

       line_.scale(p, factorX, factorY);

       if (closingSegment != null)
       {
          closingSegment.scale(p, factorX, factorY);

          closingSegment.setEnd(getFirstControl());
       }
    }

    public void rotateParams(Point2D p, double angle)
    {
       if (join != null)
       {
          join.rotate(p, angle);
       }

       line_.rotate(p, angle);

       if (closingSegment != null)
       {
          closingSegment.rotate(p, angle);

          closingSegment.setEnd(getFirstControl());
       }
    }

    public void translateParams(double shiftX, double shiftY)
    {
       if (join != null)
       {
          join.translate(shiftX, shiftY);
       }

       line_.translate(shiftX, shiftY);

       if (closingSegment != null)
       {
          closingSegment.translate(shiftX, shiftY);

          closingSegment.setEnd(getFirstControl());
       }
    }

    public boolean isEmpty()
    {
       return size() == 0;
    }

    public boolean isClosed()
    {
       return closed;
    }

    public void open()
    {
       closed = false;
       closingSegment = null;
    }

    public void open(boolean removeLastSegment)
    {
       open();
    }

    public void close(int type)
      throws EmptyPathException
    {
       if (isEmpty())
       {
          throw new EmptyPathException();
       }

       JDRSegment firstSeg = (JDRSegment)path_.getFirstSegment();

       try
       {
          switch (type)
          {
             case CLOSE_LINE :
                close(new JDRPartialLine());
             return;
             case CLOSE_CONT :
                JDRPartialBezier seg = new JDRPartialBezier();
                close(seg);

                JDRSegment lastSeg = ((JDRSegment)firstSeg.getReflection(line_)).reverse();

                Point2D dp = lastSeg.getdP1();

                seg.setGradient(dp);

             return;
             case CLOSE_MERGE_ENDS :
                moveToLine(firstSeg.getStart());
                closed = true;
                closingSegment = null;
             return;
          }
       }
       catch (IllFittingPathException e)
       {
       }
    }

    /**
     * Closes this symmetric path with the given segment, which must
     * be a partial segment. If the given segment is null, this
     * method just calls {@link #close(CLOSE_MERGE_ENDS)}
     * @param segment the closing segment (must be an instance of 
     * {@link JDRPartialSegment}).
     * @see #close(int)
     */
    public void close(JDRPathSegment segment)
      throws EmptyPathException,IllFittingPathException
    {
       if (segment == null)
       {
          close(CLOSE_MERGE_ENDS);
          return;
       }

       if (!(segment instanceof JDRPartialSegment))
       {
          throw new IllFittingPathException
             ("Can't close a symmetric path with a full segment");
       }

       closed = true;

       closingSegment = (JDRPartialSegment)segment;

       closingSegment.setSymmetryLine(line_);

       closingSegment.setEnd(path_.getFirstSegment().getStart());
    }

    public JDRShape breakPath()
       throws InvalidPathException
    {
       // Make new path have the same join as this path had before
       // it gets broken

       JDRLine line = (JDRLine)line_.clone();

       JDRPartialSegment joinSegment = null;

       if (getJoin() != null)
       {
          joinSegment = (JDRPartialSegment)getJoin().clone();
       }

       // break this path

       JDRShape path = path_.breakPath();

       // make the new path symmetric

       JDRSymmetricPath newPath = new JDRSymmetricPath(path,
          joinSegment,
          line);

       // make the join for this path a gap

       setJoin(new JDRPartialSegment(path_.getLastSegment().getEnd(), line_));

      return newPath;
    }

   public Shape getStrokedPath()
   {
      Shape shape;

      if (showPath())
      {
         shape = getStroke().getStrokedPath(this);
      }
      else
      {
         shape = path_.getStrokedPath();

         AffineTransform af = line_.getReflectionTransform(null);

         GeneralPath reflectedShape = new GeneralPath(shape);

         reflectedShape.transform(af);

         reflectedShape.append(shape, false);

         shape = reflectedShape;
      }

      return shape;
   }

   public Area getStrokedArea()
   {
      Area shape;

      if (showPath())
      {
         shape = getStroke().getStrokedArea(this);
      }
      else
      {
         shape = path_.getStrokedArea();

         AffineTransform af = line_.getReflectionTransform(null);

         Area reflectedShape = new Area(shape);

         reflectedShape.transform(af);

         shape.add(reflectedShape);
      }

      return shape;
   }

   public JDRStroke getStroke()
   {
      return path_.getStroke();
   }

   public void setStroke(JDRStroke stroke)
   {
      path_.setStroke(stroke);
   }

   public void fade(double value)
   {
      path_.fade(value);
   }

   public JDRPaint getLinePaint()
   {
      return path_.getLinePaint();
   }

   public void setLinePaint(JDRPaint paint)
   {
      path_.setLinePaint(paint);
   }

   public JDRPaint getFillPaint()
   {
      return path_.getFillPaint();
   }

   public void setFillPaint(JDRPaint paint)
   {
      path_.setFillPaint(paint);
   }

   /**
    * Separates this symmetric path into a group containing its
    * constituent parts. The groups consists of: the underlying
    * shape, a path containing the join segment if it exists, a path
    * containing the line of symmetry, the reflected shape, and a
    * path containing the closing segment if it exists.
    * @return a group containing between 3 and 5 objects
    */
   public JDRGroup separate(Graphics g)
      throws InvalidPathException
   {
      JDRGroup group = new JDRGroup();

      JDRShape shape = getUnderlyingShape();

      if (shape instanceof JDRTextPath)
      {
         return ((JDRTextPath)shape).separate(g);
      }

      group.add(shape);

      JDRPath path;

      if (join != null)
      {
         path = new JDRPath();
         path.setLinePaint(getLinePaint());
         path.setFillPaint(getFillPaint());
         path.setStroke(getStroke() instanceof JDRBasicStroke ?
            (JDRStroke)getStroke().clone() : new JDRBasicStroke());

         if (join.isGap())
         {
            path.add(new JDRLine(join.getStart(), join.getEnd()));
         }
         else
         {
            path.add(join.getFullSegment());
         }

         group.add(path);
      }

      JDRShape reflectedShape = shape.reflection(line_);

      group.add(reflectedShape);

      if (closingSegment != null)
      {
         path = new JDRPath();
         path.setLinePaint(getLinePaint());
         path.setFillPaint(getFillPaint());
         path.setStroke(getStroke() instanceof JDRBasicStroke ?
            (JDRStroke)getStroke().clone() : new JDRBasicStroke());

         path.add(closingSegment.getFullSegment());

         group.add(path);
      }

      path = new JDRPath();
      path.setLinePaint(getLinePaint());
      path.setFillPaint(getFillPaint());
      path.setStroke(new JDRBasicStroke());
      path.add(line_);

      group.add(path);

      return group;
   }

   /**
    * Gets this path as a GeneralPath.
    * @return this path as a GeneralPath
    */
   public GeneralPath getGeneralPath()
   {
      if (isEmpty()) return null;

      JDRStroke stroke = getStroke();

      GeneralPath reflectedPath = new GeneralPath();

      GeneralPath path = new GeneralPath();

      if (stroke != null)
      {
         path.setWindingRule(stroke.getWindingRule());
         reflectedPath.setWindingRule(stroke.getWindingRule());
      }

      boolean closeSubPath = isClosed() && 
               ((join != null && join.isGap()) || 
                (closingSegment != null && closingSegment.isGap()));

      int n = path_.size();

      JDRPathSegment segment = path_.getFirstSegment();

      path.moveTo((float)segment.getStart().x,
                  (float)segment.getStart().y);
      segment.appendToGeneralPath(path);

      segment = getReflected(n-1).reverse();

      reflectedPath.moveTo((float)segment.getStart().x,
                           (float)segment.getStart().y);
      segment.appendToGeneralPath(reflectedPath);

      boolean done = false;

      for (int i = 1; i < n; i++)
      {
         segment = path_.get(i);
         segment.appendToGeneralPath(path);

         segment = getReflected(n-i-1).reverse();

         if (segment.isGap() && closeSubPath && !done)
         {
            reflectedPath.closePath();
            done = true;
         }

         segment.appendToGeneralPath(reflectedPath);
      }

      if (closeSubPath)
      {
         path.closePath();

         JDRPoint p = path_.getLastSegment().getEnd();
         path.moveTo((float)p.getX(), (float)p.getY());
      }

      if (join != null)
      {
         join.appendToGeneralPath(path);
      }

      if (closeSubPath && !done)
      {
         reflectedPath.closePath();
      }

      path.append(reflectedPath, true);

      if (closingSegment != null)
      {
         closingSegment.appendToGeneralPath(path);
      }

      if (isClosed() && !closeSubPath)
      {
         path.closePath();
      }

/*
      JDRStroke stroke = getStroke();
      GeneralPath path = new GeneralPath(stroke.getWindingRule());

      JDRPathIterator pi = getIterator();

      JDRPathSegment segment = pi.next();

      path.moveTo((float)segment.getStart().x,
                  (float)segment.getStart().y);
      segment.appendToGeneralPath(path);

      while (pi.hasNext())
      {
         segment = pi.next();
         segment.appendToGeneralPath(path);
      }

      segment = null;

      if (isClosed()) path.closePath();
*/

      return path;
   }


    public JDRShape reverse()
       throws InvalidPathException
    {
       int n = path_.size();

       if (n == 0)
       {
          throw new EmptyPathException();
       }

       JDRSymmetricPath path = new JDRSymmetricPath(line_,
         n, (JDRPaint)getLinePaint().clone(), 
            (JDRPaint)getFillPaint().clone(),
            (JDRStroke)getStroke().clone());

       path.flowframe = (flowframe==null?null
          : (FlowFrame)flowframe.clone());

       path.description = description;
       path.setSelected(isSelected());

       JDRPathSegment segment = null;

       for (int i = 0; i < n; i++)
       {
          segment = getReflected(i);

          path.add((JDRSegment)segment);
       }

       if (join != null)
       {
          if (isJoinLine())
          {
             path.setJoin(new JDRPartialLine(segment.getEnd(),
               path.getSymmetry()));
          }
          else if (isJoinBezier())
          {
             path.setJoin(new JDRPartialBezier(segment.getEnd(),
               new JDRPoint(((JDRPartialBezier)join).getControl2()),
               path.getSymmetry()));
          }
          else
          {
             path.setJoin(new JDRPartialSegment(segment.getEnd(),
               path.getSymmetry()));
          }
       }

       return path;
    }

    public JDRShape intersect(JDRShape shape)
       throws InvalidPathException
    {
       return path_.intersect(shape);
    }

    public JDRShape pathUnion(JDRShape shape)
       throws InvalidPathException
    {
       return path_.pathUnion(shape);
    }

    public JDRShape exclusiveOr(JDRShape shape)
       throws InvalidPathException
    {
       return path_.exclusiveOr(shape);
    }

    public JDRShape subtract(JDRShape shape)
       throws InvalidPathException
    {
       return path_.subtract(shape);
    }

    public JDRObjectLoaderListener getListener()
    {
       return symmetricPathListener;
    }

   /**
    * Gets string representation of this symmetric path.
    * @return string representation of this symmetric path
    */
   public String toString()
   {
      String str = "SymmetricPath: join="+join+", symmetry: " +line_
                 + ", marker symmetry="+markerSymmetry
                 + ", size="+size()+", segments=[";

      for (int i = 0; i < size(); i++)
      {
         str += get(i)+",";
      }

      str += "]";

      return str;
    }

    /**
     * Gets if the end point is anchored to the line of symmetry.
     * @return true if end point anchored otherwise false
     */
    public boolean isAnchored()
    {
       return join == null;
    }

    /**
     * Sets the join.
     * @param segment the join segment (may be null)
     */
    public void setJoin(JDRPartialSegment segment)
    {
       boolean flag = (isJoinBezier() && join.isEdited());

       join = segment;

       if (join == null)
       {
          moveToLine(path_.getLastSegment().getEnd());
       }
       else
       {
          join.setStart(path_.getLastSegment().getEnd());
          join.setSymmetryLine(line_);
       }

       if (flag)
       {
          selectControl(path_.getLastSegment().getEnd());
       }
    }

    public JDRPartialSegment getJoin()
    {
       return join;
    }

    /**
     * Gets the closing segment.
     * @return the closing segment or null if this path is open or if
     * this path is closed and the start point is anchored to the line 
     * of symmetry
     */
    public JDRPartialSegment getClosingSegment()
    {
       return closingSegment;
    }

    /**
     *  Checks if join is an instance of JDRPartialLine and is
     *  non-null.
     *  @return true if join not null and an instance of
     *  JDRPartialLine
     */
    public boolean isJoinLine()
    {
       if (join == null) return false;

       return (join instanceof JDRPartialLine);
    }

    /**
     *  Checks if the closing segment is an instance of JDRPartialLine 
     *  and is non-null.
     *  @return true if closing segment not null and an instance of
     *  JDRPartialLine
     */
    public boolean isClosingSegmentLine()
    {
       if (closingSegment == null) return false;

       return (closingSegment instanceof JDRPartialLine);
    }

    /**
     *  Checks if join is an instance of JDRPartialBezier and is
     *  non-null.
     *  @return true if join not null and an instance of
     *  JDRPartialBezier
     */
    public boolean isJoinBezier()
    {
       if (join == null) return false;

       return (join instanceof JDRPartialBezier);
    }

    /**
     *  Checks if the closing segment is an instance of
     *  JDRPartialBezier and is non-null.
     *  @return true if closing segment not null and an instance of
     *  JDRPartialBezier
     */
    public boolean isClosingSegmentBezier()
    {
       if (closingSegment == null) return false;

       return (closingSegment instanceof JDRPartialBezier);
    }

    /**
     * Sets the line of symmetry.
     * @param x0 x co-ordinate of first point
     * @param y0 y co-ordinate of first point
     * @param x1 x co-ordinate of second point
     * @param y1 y co-ordinate of second point
     */
    public void setSymmetry(double x0, double y0, double x1, double y1)
    {
       if (line_ == null)
       {
          line_ = new JDRLine(new JDRSymmetryLinePoint(x0, y0),
                              new JDRSymmetryLinePoint(x1, y1));
       }
       else
       {
          line_.setStart(x0, y0);
          line_.setEnd(x1, y1);
       }

       if (join != null) join.setSymmetryLine(line_);
    }

    public void setSymmetry(JDRLine line)
    {
       line_ = line;

       if (line == null)
       {
          return;
       }

       JDRPoint p = line_.getStart();

       if (!(p instanceof JDRSymmetryLinePoint))
       {
          line_.setStart(new JDRSymmetryLinePoint(p.x, p.y));
       }

       p = line_.getEnd();

       if (!(p instanceof JDRSymmetryLinePoint))
       {
          line_.setEnd(new JDRSymmetryLinePoint(p.x, p.y));
       }
    }

    /**
     * Gets the line of symmetry.
     * @return line of symmetry
     */ 
    public JDRLine getSymmetry()
    {
       return line_;
    }

    /**
     * Sets the marker symmetry flag. If true, the markers are
     * reflected with the path, otherwise the markers follow the
     * entire path.
     * @param flag if true, markers are reflected with the path
     */
    public void setMarkerSymmetry(boolean flag)
    {
       markerSymmetry = flag;
    }

    /**
     * Gets the marker symmetry.
     * @return true if the markers are reflected with the path,
     * otherwise the markers follow the entire path.
     */
    public boolean hasMarkerSymmetry()
    {
       return markerSymmetry;
    }

    public void selectControl(JDRPoint p, int pointIndex, int segmentIndex)
    {
       stopEditing();

       selected = true;

/*
       selectedControl = p;
       selectedControlIndex = pointIndex;
       selectedSegmentIndex = segmentIndex;
*/

       if (segmentIndex < path_.size())
       {
          path_.selectControl(p, pointIndex, segmentIndex);
       }
       else
       {
          JDRPathSegment selectedSegment = get(segmentIndex);

          stopEditing();

          setSelectedElements(segmentIndex, pointIndex, selectedSegment, p);

          p.setSelected(true);
          selectedSegment.setSelected(true);
       }

       editMode = true;
    }

    public int psLevel()
    {
       return path_.psLevel();
    }

    public void saveEPS(PrintWriter out) throws IOException
    {
       JDRPaint paint = getFillPaint();

       GeneralPath path = getGeneralPath();

       out.println("gsave");
       EPS.fillPath(path, paint, out);
       out.println("grestore");

       out.println("gsave");
       getStroke().saveEPS(this, out);
       out.println("grestore");
    }

   public void saveSVG(PrintWriter out)
      throws IOException
   {
      if (isEmpty()) return;

      JDRPathSegment segment = get(0);

      out.print("   <path d=\"M ");

      segment.getStart().saveSVG(out);
      segment.saveSVG(out);

      JDRPathIterator pi = getIterator();

      while (pi.hasNext())
      {
         segment = pi.next();
         segment.saveSVG(out);
      }

      if (isClosed()) out.print("z");

      out.println("\"");
      out.println("      "+getLinePaint().svgLine());
      out.println("      "+getFillPaint().svgFill());

      if (getStroke() instanceof JDRBasicStroke)
      {
         out.println("      "
            +((JDRBasicStroke)getStroke()).svg(getLinePaint()));
      }

      out.println("   />");
   } 

   public String pgf(AffineTransform af)
   {
      BBox pathBBox = getBBox();

      if (pathBBox == null)
      {
         return "";
      }

      String eol = System.getProperty("line.separator", "\n");

      String path = new String("");

      if (!description.equals(""))
      {
         path  += "% "+description+eol;
      }

      path += pgfPath(af);

      String str = new String();

      JDRPaint paint = getFillPaint();
      JDRPaint linePaint = getLinePaint();

      str += "\\begin{pgfscope}"+eol;

      if (getStroke() instanceof JDRBasicStroke)
      {
         str += ((JDRBasicStroke)getStroke()).pgf();
      }

      if (paint instanceof JDRTransparent)
      {
         str += path;

         if (!(linePaint instanceof JDRTransparent))
         {
            str += linePaint.pgfstrokecolor(pathBBox);
            str += eol;
            str += "\\pgfusepath{stroke}"+eol;
         }
      }
      else
      {
         str += path;
         str += paint.pgffillcolor(pathBBox);

         if (getStroke() instanceof JDRBasicStroke)
         {
            str += (((JDRBasicStroke)getStroke()).windingRule
                       == GeneralPath.WIND_EVEN_ODD ? 
                   "\\pgfseteorule" :
                   "\\pgfsetnonzerorule");
         }

         if (paint instanceof JDRGradient
               || paint instanceof JDRRadial)
         {
            str += linePaint.pgfstrokecolor(pathBBox);
            str += eol;
            if (linePaint instanceof JDRTransparent)
            {
               str += "\\pgfsetstrokeopacity{0.0}"+eol;
            }
            str += "\\pgfusepath{stroke}"+eol;
         }
         else if (linePaint instanceof JDRTransparent)
         {
            str += "\\pgfusepath{fill}"+eol;
         }
         else
         {
            // \pgfsetstrokecolor and \pgfsetfillcolor
            // don't seem to have a cmyk option
            // so redo path and just use \color
            str += "\\pgfusepath{fill}"+eol;
            str += path;
            str += linePaint.pgfstrokecolor(pathBBox);
            str += eol;
            str += "\\pgfusepath{stroke}"+eol;
         }
      }

      str += "\\end{pgfscope}"+eol;

      str += pgfMarkers(af, pathBBox);

      return str;
   }

   /** 
    * Gets just the path construction commands.
    */
   public String pgfPath(AffineTransform af)
   {
      String eol = System.getProperty("line.separator", "\n");
      String path = "";

      JDRPathSegment segment = get(0);
      path += "\\pgfpathmoveto{"+segment.getStart().pgf(af)+"}"+eol;

      JDRPathIterator pi = getIterator();

      while (pi.hasNext())
      {
         segment = pi.next();

         path += segment.pgf(af)+eol;
      }

      if (isClosed()) path += "\\pgfclosepath"+eol;

      return path;
   }

   public String pgfMarkers(AffineTransform af, BBox pathBBox)
   {
      String str = "";

      if (getStroke() instanceof JDRBasicStroke)
      {
         JDRBasicStroke stroke = (JDRBasicStroke)getStroke();

         if (stroke.getStartArrow().getType() == JDRMarker.ARROW_NONE
          && stroke.getEndArrow().getType() == JDRMarker.ARROW_NONE
          && stroke.getMidArrow().getType() == JDRMarker.ARROW_NONE)
         {
            return str;
         }

         JDRPaint linePaint = getLinePaint();

         JDRPathIterator pi = getIterator();

         while (pi.hasNext())
         {
            JDRPathSegment segment = pi.next();

            JDRMarker marker = segment.getStartMarker();

            if (marker != null && marker.getType() != JDRMarker.ARROW_NONE)
            {
               str += marker.pgfShape(linePaint, pathBBox,
                         segment, true, af);
            }

            marker = segment.getEndMarker();

            if (marker != null && marker.getType() != JDRMarker.ARROW_NONE)
            {
               str += marker.pgfShape(linePaint, pathBBox,
                         segment, false, af);
            }
         }
      }

      return str;
   }

   public JDRTextual getTextual()
   {
      return path_.getTextual();
   }

   public boolean hasTextual()
   {
      return path_.hasTextual();
   }

   public boolean showPath()
   {
      return path_.showPath();
   }

   public boolean hasSymmetricPath()
   {
      return true;
   }

   public JDRSymmetricPath getSymmetricPath()
   {
      return this;
   }

   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "SymmetricPath:"+eol;

      str += "line of symmetry: "+line_.info()+eol;

      str += "anchored: "+isAnchored()+eol;

      if (join != null)
      {
         str += "join segment: "+join.info();
      }

      if (closingSegment != null)
      {
         str += "closing segment: "+closingSegment.info();
      }

      str += "Underlying shape:"+path_.info();

      return str;
   }

   protected void setSelectedElements(int segmentIndex, int controlIndex,
      JDRPathSegment segment, JDRPoint control)
   {
      path_.setSelectedElements(segmentIndex, controlIndex, segment, control);
   }

   public String[] getDescriptionInfo()
   {
      return getUnderlyingShape().getDescriptionInfo();
   }

   private boolean markerSymmetry=true;

   private boolean closed=false;

   private JDRShape path_;
   private JDRPartialSegment join;
   private JDRLine line_;
   private JDRPartialSegment closingSegment=null;

   private static JDRPathListener symmetricPathListener = new JDRSymmetricPathListener();

   protected JDRSymmetricPathIterator iterator;
   protected JDRPointIterator pointIterator;

/*
   private JDRPoint selectedControl;
   private int selectedControlIndex;
   private int selectedSegmentIndex;
*/
}

