// File          : JDRSpiralPattern.java
// Date          : 9th Sept 2010
// Last Modified : 6th April 2011
// Author        : Nicola L. C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;

/**
 *  Class representing a spiral pattern. This has an underlying
 *  shape that gets replicated by spiralling around a given point. The
 *  overall shape is governed by the following parameters:
 *  <ul>
 *  <li> Point of rotation.
 *  <li> Angle increment.
 *  <li> Distance parameter
 *  <li> Number of replicas.
 *  <li> Mode.
 *  </ul>
 *  The mode determines whether the underlying path and replicas are
 *  drawn in one go (single mode) or whether they are drawn independently 
 *  of each other (multi mode).
 *  @author Nicola L C Talbot
 */

public class JDRSpiralPattern extends JDRPattern
{
   /**
    * Creates a spiral pattern from the given path.
    * @param path the path
    * @param point the point of rotation
    * @param adjust adjust control
    * @param angle the angle increment (Radians)
    * @param distance the distance parameter of Archimedean spiral
    * @param singleMode true if single path mode
    * @param replicas the number of times the underlying path is
    * replicated
    */
    public JDRSpiralPattern(JDRShape path, JDRPoint point, JDRPoint adjust,
       double angle, double distance,
       int replicas, boolean singleMode)
       throws InvalidReplicaException
    {
       setNumReplicas(replicas);

       path_ = path;

       setPatternAnchor(point);
       setPatternAdjust(adjust);

       setDistance(distance);

       angle_    = angle;
       singlemode_ = singleMode;

       initIterators();
    }

   /**
    * Creates a spiral pattern from the given path.
    * @param path the path
    * @param point the point of rotation
    * @param adjust adjust control
    * @param angle the angle increment (Radians)
    * @param distance the distance parameter of Archimedean spiral
    * @param singleMode true if single path mode
    * @param showOriginalShape true if original path is to be drawn
    * @param replicas the number of times the underlying path is
    * replicated
    */
    public JDRSpiralPattern(JDRShape path, JDRPoint point, JDRPoint adjust,
       double angle, double distance,
       int replicas, boolean singleMode, boolean showOriginalShape)
       throws InvalidReplicaException
    {
       setNumReplicas(replicas);

       path_ = path;

       setPatternAnchor(point);
       setPatternAdjust(adjust);

       setDistance(distance);

       angle_    = angle;
       singlemode_ = singleMode;
       showoriginal_ = showOriginalShape;

       initIterators();
    }

    public JDRSpiralPattern(JDRShape path, JDRPoint point,
       double angle, double distance,
       int replicas, boolean singleMode)
       throws InvalidReplicaException
    {
       path_ = path;

       setNumReplicas(replicas);
       setPatternAnchor(point);
       setDefaultPatternAdjust();

       setDistance(distance);

       angle_    = angle;
       singlemode_ = singleMode;

       initIterators();
    }

    /**
     * Creates an empty path with one replica, angle of PI radians
     * and point at the origin.
     */
    public JDRSpiralPattern()
    {
       path_     = new JDRPath();
       point_    = new JDRPatternAnchorPoint();
       adjust_   = new JDRPatternAdjustPoint();
       angle_    = Math.PI*0.25;
       replicas_ = 1;
       distance_ = 1.0;
       singlemode_ = true;

       initIterators();
    }

    public JDRSpiralPattern(int capacity, JDRPaint lineColor,
                            JDRPaint fillColor, JDRStroke s)
    {
       path_ = new JDRPath(capacity, lineColor, fillColor, s);

       point_    = new JDRPatternAnchorPoint();
       adjust_   = new JDRPatternAdjustPoint();
       angle_    = Math.PI*0.25;
       replicas_ = 4;
       distance_ = 1.0;
       singlemode_ = true;

       initIterators();
    }

    public boolean hasAdjust()
    {
       return true;
    }

    public JDRPattern createTemplate()
    {
       try
       {
         return new JDRSpiralPattern(null, null, null,
          angle_, distance_, replicas_, singlemode_);
       }
       catch (InvalidFormatException e)
       {
          // this shouldn't happen
       }

       return null;
    }

    /**
     * Makes the angle of rotation for this pattern the same as the
     * angle of rotation for the given pattern. (Which must also be
     * an instance of JDRSpiralPattern.)
     */
    public void makeParametersEqual(JDRPattern object)
    {
       JDRSpiralPattern pattern = (JDRSpiralPattern)object;

       angle_ = pattern.angle_;
       distance_ = pattern.distance_;
    }

   public void getReplicaTransform(double[] matrix, int replicaIndex)
   {
      double dx = 0;
      double dy = 0;

      double a = 0.0;

      double repAngle = replicaIndex*angle_;

      if (adjust_ != null)
      {
         dx = point_.x - adjust_.x;
         dy = point_.y - adjust_.y;

         a = Math.sqrt(dx*dx + dy*dy);

         repAngle += Math.atan2(dy, dx);
      }

      double cosAngle = Math.cos(repAngle);
      double sinAngle = Math.sin(repAngle);

      double b = distance_ / (2.0*Math.PI);

      double radius = a + b * repAngle;

      matrix[0] = cosAngle;
      matrix[1] = sinAngle;
      matrix[2] = -sinAngle;
      matrix[3] = cosAngle;
      matrix[4] = cosAngle*(radius-point_.x) + point_.y*sinAngle+point_.x;
      matrix[5] = sinAngle*(radius-point_.x) + point_.y*(1-cosAngle);
   }


   public JDRObjectLoaderListener getListener()
   {
      return shapeListener;
   }

   public String toString()
   {
      String str = "SpiralPattern: point="+point_+", angle: " +angle_
                 + ", distance: " + distance_
                 + ", replicas: "+replicas_
                 + ", size="+size()+", segments=[";

      for (int i = 0; i < size(); i++)
      {
         str += get(i)+",";
      }

      str += "]";

      return str;
    }


   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "SpiralPattern:"+eol;

      str += "anchor: "+point_.info()+eol;

      str += "adjust: "+adjust_.info()+eol;

      str += "angle: "+angle_+eol;

      str += "distance: "+distance_+eol;

      str += "replicas: "+replicas_;

      str += "Underlying shape:"+path_.info();

      return str;
   }

   public double getRotationAngle()
   {
      return angle_;
   }

   public void setRotationAngle(double angle)
   {
      angle_ = angle;
   }

   public void setDistance(double distance)
   {
      distance_ = distance;
   }

   public double getDistance()
   {
      return distance_;
   }

   public String[] getDescriptionInfo()
   {
      return new String[] {""+replicas_, 
                           "("+angle_+":"+distance_+")"};
   }

   private double angle_, distance_;

   private static JDRPathListener shapeListener = new JDRSpiralPatternListener();

}

